"""Setup cuQuantum as external dependency"""
_CUQUANTUM_ROOT = "CUQUANTUM_ROOT"


def _tpl(repository_ctx, tpl, substitutions = {}, out = None):
    if not out:
        out = tpl
    repository_ctx.template(
        out,
        Label("//third_party/cuquantum:%s.tpl" % tpl),
        substitutions,
    )


def _fail(msg):
    """Output failure message when auto configuration fails."""
    red = "\033[0;31m"
    no_color = "\033[0m"
    fail("%sPython Configuration Error:%s %s\n" % (red, no_color, msg))


def _execute(
        repository_ctx,
        cmdline,
        error_msg = None,
        error_details = None,
        empty_stdout_fine = False):
    """Executes an arbitrary shell command.

    Args:
      repository_ctx: the repository_ctx object
      cmdline: list of strings, the command to execute
      error_msg: string, a summary of the error if the command fails
      error_details: string, details about the error or steps to fix it
      empty_stdout_fine: bool, if True, an empty stdout result is fine, otherwise
        it's an error
    Return:
      the result of repository_ctx.execute(cmdline)
    """
    result = repository_ctx.execute(cmdline)
    if result.stderr or not (empty_stdout_fine or result.stdout):
        _fail("\n".join([
            error_msg.strip() if error_msg else "Repository command failed",
            result.stderr.strip(),
            error_details if error_details else "",
        ]))
    return result


def _read_dir(repository_ctx, src_dir):
    """Returns a string with all files in a directory.

    Finds all files inside a directory, traversing subfolders and following
    symlinks. The returned string contains the full path of all files
    separated by line breaks.
    """
    find_result = _execute(
        repository_ctx,
        ["find", src_dir, "-follow", "-type", "f"],
        empty_stdout_fine = True,
    )
    result = find_result.stdout
    return result


def _find_file(repository_ctx, filename):
    """Returns a string with a directory path including the filename.

    The returned string contains the parent path of the filename.
    """
    result = repository_ctx.execute(
        ["timeout", "5", "find", "/", "-name", filename, "-print", "-quit", "-not", "-path", "'*/.*'", "-quit"]).stdout
    result = result[:result.find(filename)+len(filename)]
    return result


def _genrule(genrule_name, command, outs):
    """Returns a string with a genrule.

    Genrule executes the given command and produces the given outputs.

    Args:
        genrule_name: A unique name for genrule target.
        command: The command to run.
        outs: A list of files generated by this rule.

    Returns:
        A genrule target.
    """
    return (
        "genrule(\n" +
        '    name = "' +
        genrule_name + '",\n' +
        "    outs = [\n" +
        outs +
        "\n    ],\n" +
        '    cmd = """\n' +
        command +
        '\n   """,\n' +
        ")\n"
    )

def _norm_path(path):
    """Returns a path with '/' and remove the trailing slash."""
    path = path.replace("\\", "/")
    if path[-1] == "/":
        path = path[:-1]
    return path


def _symlink_genrule_for_dir(
        repository_ctx,
        src_dir,
        dest_dir,
        genrule_name,
        src_files = [],
        dest_files = [],
        is_empty_genrule = False):
    """Returns a genrule to symlink(or copy if on Windows) a set of files.

    If src_dir is passed, files will be read from the given directory; otherwise
    we assume files are in src_files and dest_files. Here are the examples:

    ```
    genrule(
        name = "cuquantum_header_include",
        outs = [
            "include/custatevec.h",
            "include/cutensornet.h",
            "include/cutensornet/types.h",
            "include/cutensornet/typesDistributed.h",
        ],
        cmd = [some copy command lines based on users' local environment],
    )

    genrule(
        name = "libcustatevec.so",
        outs = [
            "libcustatevec.so",
        ],
        cmd = [some copy command lines based on users' local environment],
    )
    ```

    Args:
        repository_ctx: the repository_ctx object.
        src_dir: source directory.
        dest_dir: directory to create symlink in.
        genrule_name: genrule name.
        src_files: list of source files instead of src_dir.
        dest_files: list of corresonding destination files.
        is_empty_genrule: True if CUQUANTUM_ROOT is not set.

    Returns:
        genrule target that creates the symlinks.
    """
    if is_empty_genrule:
        if dest_dir != "":
          target_path = "%s/%s.h" % (dest_dir, genrule_name)
        else:
          target_path = genrule_name
        genrule = _genrule(
            genrule_name,
            "touch $(OUTS)",
            "'%s'" % (target_path),
        )
        return genrule

    if src_dir != None:
        src_dir = _norm_path(src_dir)
        dest_dir = _norm_path(dest_dir)
        files = "\n".join(sorted(_read_dir(repository_ctx, src_dir).splitlines()))

        dest_files = files.replace(src_dir, "").splitlines()
        src_files = files.splitlines()
    command = []
    outs = []
    
    for i in range(len(dest_files)):
        if dest_files[i] != "":
            # If we have only one file to link we do not want to use the dest_dir, as
            # $(@D) will include the full path to the file.
            dest = "$(@D)/" + dest_dir + dest_files[i] if len(dest_files) != 1 else "$(@D)/" + dest_files[i]

            # Copy the headers to create a sandboxable setup.
            cmd = "cp -f"
            command.append(cmd + ' "%s" "%s"' % (src_files[i], dest))
            outs.append('        "' + dest_dir + dest_files[i] + '",')

    genrule = _genrule(
        genrule_name,
        " && ".join(command),
        "\n".join(outs),
    )
    return genrule


def _cuquantum_pip_impl(repository_ctx):
    if _CUQUANTUM_ROOT in repository_ctx.os.environ:
      cuquantum_root = repository_ctx.os.environ[_CUQUANTUM_ROOT]
    else:
      repository_ctx.os.environ[_CUQUANTUM_ROOT] = ""
      cuquantum_root = ""

    if cuquantum_root == "":
      cuquantum_header_path = _find_file(repository_ctx, "custatevec.h")
      cuquantum_header_path = cuquantum_header_path[:cuquantum_header_path.find("/custatevec.h")]
      custatevec_shared_library_path = _find_file(repository_ctx, "libcustatevec.so")
    else:
      cuquantum_header_path = "%s/include" % cuquantum_root
      custatevec_shared_library_path = "%s/lib/libcustatevec.so" % (cuquantum_root)

    is_empty_genrule = cuquantum_header_path == "" or custatevec_shared_library_path == ""

    cuquantum_header_rule = _symlink_genrule_for_dir(
        repository_ctx,
        cuquantum_header_path,
        "include",
        "cuquantum_header_include",
        is_empty_genrule=is_empty_genrule,
    )

    custatevec_shared_library_rule = _symlink_genrule_for_dir(
        repository_ctx,
        None,
        "",
        "libcustatevec.so",
        [custatevec_shared_library_path],
        ["libcustatevec.so"],
        is_empty_genrule=is_empty_genrule,
    )

    _tpl(repository_ctx, "BUILD", {
        "%{CUQUANTUM_HEADER_GENRULE}": cuquantum_header_rule,
        "%{CUSTATEVEC_SHARED_LIBRARY_GENRULE}": custatevec_shared_library_rule,
    })
        

cuquantum_configure = repository_rule(
    implementation = _cuquantum_pip_impl,
    environ = [
        _CUQUANTUM_ROOT,
    ],
)
