/*
    FreeRTOS V8.2.0 - Copyright (C) 2015 Real Time Engineers Ltd.
    All rights reserved

    VISIT http://www.FreeRTOS.org TO ENSURE YOU ARE USING THE LATEST VERSION.

    This file is part of the FreeRTOS distribution.

    FreeRTOS is free software; you can redistribute it and/or modify it under
    the terms of the GNU General Public License (version 2) as published by the
    Free Software Foundation >>!AND MODIFIED BY!<< the FreeRTOS exception.

    ***************************************************************************
    >>!   NOTE: The modification to the GPL is included to allow you to     !<<
    >>!   distribute a combined work that includes FreeRTOS without being   !<<
    >>!   obliged to provide the source code for proprietary components     !<<
    >>!   outside of the FreeRTOS kernel.                                   !<<
    ***************************************************************************

    FreeRTOS is distributed in the hope that it will be useful, but WITHOUT ANY
    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
    FOR A PARTICULAR PURPOSE.  Full license text is available on the following
    link: http://www.freertos.org/a00114.html

    ***************************************************************************
     *                                                                       *
     *    FreeRTOS provides completely free yet professionally developed,    *
     *    robust, strictly quality controlled, supported, and cross          *
     *    platform software that is more than just the market leader, it     *
     *    is the industry's de facto standard.                               *
     *                                                                       *
     *    Help yourself get started quickly while simultaneously helping     *
     *    to support the FreeRTOS project by purchasing a FreeRTOS           *
     *    tutorial book, reference manual, or both:                          *
     *    http://www.FreeRTOS.org/Documentation                              *
     *                                                                       *
    ***************************************************************************

    http://www.FreeRTOS.org/FAQHelp.html - Having a problem?  Start by reading
    the FAQ page "My application does not run, what could be wrong?".  Have you
    defined configASSERT()?

    http://www.FreeRTOS.org/support - In return for receiving this top quality
    embedded software for free we request you assist our global community by
    participating in the support forum.

    http://www.FreeRTOS.org/training - Investing in training allows your team to
    be as productive as possible as early as possible.  Now you can receive
    FreeRTOS training directly from Richard Barry, CEO of Real Time Engineers
    Ltd, and the world's leading authority on the world's leading RTOS.

    http://www.FreeRTOS.org/plus - A selection of FreeRTOS ecosystem products,
    including FreeRTOS+Trace - an indispensable productivity tool, a DOS
    compatible FAT file system, and our tiny thread aware UDP/IP stack.

    http://www.FreeRTOS.org/labs - Where new FreeRTOS products go to incubate.
    Come and try FreeRTOS+TCP, our new open source TCP/IP stack for FreeRTOS.

    http://www.OpenRTOS.com - Real Time Engineers ltd. license FreeRTOS to High
    Integrity Systems ltd. to sell under the OpenRTOS brand.  Low cost OpenRTOS
    licenses offer ticketed support, indemnification and commercial middleware.

    http://www.SafeRTOS.com - High Integrity Systems also provide a safety
    engineered and independently SIL3 certified version for use in safety and
    mission critical applications that require provable dependability.

    1 tab == 4 spaces!
*/

/*
 * This is a heap allocator that can allocate memory out of several tagged memory regions,
 * with the regions having differing capabilities. In the ESP32, this is used to
 * allocate memory for the various applications within the space the MMU allows them
 * to work with. It can also be used to e.g. allocate memory in DMA-capable regions.
 *
 * Usage notes:
 *
 * vPortDefineHeapRegions() ***must*** be called before pvPortMalloc().
 * pvPortMalloc() will be called if any task objects (tasks, queues, event
 * groups, etc.) are created, therefore vPortDefineHeapRegions() ***must*** be
 * called before any other objects are defined.
 *
 * vPortDefineHeapRegions() takes a single parameter.  The parameter is an array
 * of HeapRegionTagged_t structures.  HeapRegion_t is defined in portable.h as
 *
 * typedef struct HeapRegion
 * {
 *  uint8_t *pucStartAddress; << Start address of a block of memory that will be part of the heap.
 *  size_t xSizeInBytes;      << Size of the block of memory.
 *  BaseType_t xTag;          << Tag
 * } HeapRegionTagged_t;
 *
 * 'Tag' allows you to allocate memory of a certain type. Tag -1 is special;
 * it basically tells the allocator to ignore this region as if it is not
 * in the array at all. This facilitates disabling memory regions.
 *
 * The array is terminated using a NULL zero sized region definition, and the
 * memory regions defined in the array ***must*** appear in address order from
 * low address to high address.  So the following is a valid example of how
 * to use the function.
 *
 * HeapRegionTagged_t xHeapRegions[] =
 * {
 *  { ( uint8_t * ) 0x80000000UL, 0x10000, 1 }, << Defines a block of 0x10000 bytes starting at address 0x80000000, tag 1
 *  { ( uint8_t * ) 0x90000000UL, 0xa0000, 2 }, << Defines a block of 0xa0000 bytes starting at address of 0x90000000, tag 2
 *  { NULL, 0, 0 }                << Terminates the array.
 * };
 *
 * vPortDefineHeapRegions( xHeapRegions ); << Pass the array into vPortDefineHeapRegions().
 *
 * Note 0x80000000 is the lower address so appears in the array first.
 *
 * pvPortMallocTagged can be used to get memory in a tagged region.
 *
 */

/*

ToDo:
- This malloc implementation can be somewhat slow, especially when it is called multiple times with multiple tags
when having low memory issues. ToDo: Make it quicker.
 -JD
 */


#include <stdlib.h>

/* Defining MPU_WRAPPERS_INCLUDED_FROM_API_FILE prevents task.h from redefining
all the API functions to use the MPU wrappers.  That should only be done when
task.h is included from an application file. */
#define MPU_WRAPPERS_INCLUDED_FROM_API_FILE

#include "FreeRTOS.h"
#include "task.h"
#include "heap_regions_debug.h"

#undef MPU_WRAPPERS_INCLUDED_FROM_API_FILE

#include "heap_regions.h"

#include "rom/ets_sys.h"

/* Block sizes must not get too small. */
#define heapMINIMUM_BLOCK_SIZE  ( ( size_t ) ( uxHeapStructSize << 1 ) )

/* Assumes 8bit bytes! */
#define heapBITS_PER_BYTE       ( ( size_t ) 8 )

/* Define the linked list structure.  This is used to link free blocks in order
   of their memory address. This is optimized for size of the linked list struct
   and assumes a region is never larger than 16MiB. */
#define HEAPREGIONS_MAX_REGIONSIZE (16*1024*1024)
typedef struct A_BLOCK_LINK
{
    struct A_BLOCK_LINK *pxNextFreeBlock;   /*<< The next free block in the list. */
    int xBlockSize: 24;                     /*<< The size of the free block. */
    int xTag: 7;                            /*<< Tag of this region */
    int xAllocated: 1;                      /*<< 1 if allocated */
} BlockLink_t;

//Mux to protect the memory status data
static portMUX_TYPE xMallocMutex = portMUX_INITIALIZER_UNLOCKED;

/*-----------------------------------------------------------*/

/*
 * Inserts a block of memory that is being freed into the correct position in
 * the list of free memory blocks.  The block being freed will be merged with
 * the block in front it and/or the block behind it if the memory blocks are
 * adjacent to each other.
 */
static void prvInsertBlockIntoFreeList( BlockLink_t *pxBlockToInsert );

/*-----------------------------------------------------------*/

/* The size of the structure placed at the beginning of each allocated memory
block must be correctly byte aligned. */
static const uint32_t uxHeapStructSize  = ( ( sizeof ( BlockLink_t ) + BLOCK_HEAD_LEN + BLOCK_TAIL_LEN + ( portBYTE_ALIGNMENT - 1 ) ) & ~portBYTE_ALIGNMENT_MASK );

/* Create a couple of list links to mark the start and end of the list. */
static BlockLink_t xStart, *pxEnd = NULL;

/* Keeps track of the number of free bytes remaining, but says nothing about
fragmentation. */
static size_t xFreeBytesRemaining[HEAPREGIONS_MAX_TAGCOUNT] = {0};
static size_t xMinimumEverFreeBytesRemaining[HEAPREGIONS_MAX_TAGCOUNT] = {0};


/*-----------------------------------------------------------*/

void *pvPortMallocTagged( size_t xWantedSize, BaseType_t tag )
{
BlockLink_t *pxBlock, *pxPreviousBlock, *pxNewBlockLink;
void *pvReturn = NULL;

    /* The heap must be initialised before the first call to
    prvPortMalloc(). */
    configASSERT( pxEnd );

    taskENTER_CRITICAL(&xMallocMutex);
    {
        /* The wanted size is increased so it can contain a BlockLink_t
        structure in addition to the requested amount of bytes. */
        if( xWantedSize > 0 )
        {
            xWantedSize += uxHeapStructSize;

            /* Ensure that blocks are always aligned to the required number
            of bytes. */
            if( ( xWantedSize & portBYTE_ALIGNMENT_MASK ) != 0x00 )
            {
                /* Byte alignment required. */
               xWantedSize += ( portBYTE_ALIGNMENT - ( xWantedSize & portBYTE_ALIGNMENT_MASK ) );
            }
            else
            {
               mtCOVERAGE_TEST_MARKER();
            }
        }
        else
        {
           mtCOVERAGE_TEST_MARKER();
        }

        if( ( xWantedSize > 0 ) && ( xWantedSize <= xFreeBytesRemaining[ tag ] ) )
        {
            /* Traverse the list from the start (lowest address) block until
            one of adequate size is found. */
            pxPreviousBlock = &xStart;
            pxBlock = xStart.pxNextFreeBlock;
            while( ( ( pxBlock->xTag != tag ) ||  ( pxBlock->xBlockSize < xWantedSize ) ) && ( pxBlock->pxNextFreeBlock != NULL ) )
            {
//             ets_printf("Block %x -> %x\n", (uint32_t)pxBlock, (uint32_t)pxBlock->pxNextFreeBlock);

                                    #if (configENABLE_MEMORY_DEBUG == 1)
                                    {
                                        mem_check_block(pxBlock);
                                    }
                                    #endif

               pxPreviousBlock = pxBlock;
               pxBlock = pxBlock->pxNextFreeBlock;
            }

            /* If the end marker was not reached then a block of adequate size
            was found. */
            if( pxBlock != pxEnd )
            {
                /* Return the memory space pointed to - jumping over the
                BlockLink_t structure at its start. */
                pvReturn = ( void * ) ( ( ( uint8_t * ) pxPreviousBlock->pxNextFreeBlock ) + uxHeapStructSize - BLOCK_TAIL_LEN - BLOCK_HEAD_LEN);

                /* This block is being returned for use so must be taken out
                of the list of free blocks. */
                pxPreviousBlock->pxNextFreeBlock = pxBlock->pxNextFreeBlock;

                /* If the block is larger than required it can be split into
                two. */

                if( ( pxBlock->xBlockSize - xWantedSize ) > heapMINIMUM_BLOCK_SIZE )
                {
                    /* This block is to be split into two.  Create a new
                    block following the number of bytes requested. The void
                    cast is used to prevent byte alignment warnings from the
                    compiler. */
                    pxNewBlockLink = ( void * ) ( ( ( uint8_t * ) pxBlock ) + xWantedSize);

                    /* Calculate the sizes of two blocks split from the
                    single block. */
                    pxNewBlockLink->xBlockSize = pxBlock->xBlockSize - xWantedSize;
                    pxNewBlockLink->xTag = tag;
                    pxBlock->xBlockSize = xWantedSize;

                                            #if (configENABLE_MEMORY_DEBUG == 1)
                                            {
                                                mem_init_dog(pxNewBlockLink);
                                            }
                                            #endif


                    /* Insert the new block into the list of free blocks. */
                    prvInsertBlockIntoFreeList( ( pxNewBlockLink ) );
                }
                else
                {
                    mtCOVERAGE_TEST_MARKER();
                }

                xFreeBytesRemaining[ tag ] -= pxBlock->xBlockSize;

                if( xFreeBytesRemaining[ tag ] < xMinimumEverFreeBytesRemaining[ tag ] )
                {
                    xMinimumEverFreeBytesRemaining[ tag ] = xFreeBytesRemaining[ tag ];
                }
                else
                {
                   mtCOVERAGE_TEST_MARKER();
                }

                /* The block is being returned - it is allocated and owned
                by the application and has no "next" block. */
                pxBlock->xAllocated = 1;
                pxBlock->pxNextFreeBlock = NULL;

                                    #if (configENABLE_MEMORY_DEBUG == 1)
                                    {
                                        mem_init_dog(pxBlock);
                                        mem_malloc_block(pxBlock);
                                    }
                                    #endif
            }
            else
            {
                mtCOVERAGE_TEST_MARKER();
            }
        }
        else
        {
            mtCOVERAGE_TEST_MARKER();
        }

        traceMALLOC( pvReturn, xWantedSize );
    }
    taskEXIT_CRITICAL(&xMallocMutex);

    #if( configUSE_MALLOC_FAILED_HOOK == 1 )
    {
        if( pvReturn == NULL )
        {
            extern void vApplicationMallocFailedHook( void );
            vApplicationMallocFailedHook();
        }
        else
        {
            mtCOVERAGE_TEST_MARKER();
        }
    }
    #endif

    return pvReturn;
}
/*-----------------------------------------------------------*/

void vPortFreeTagged( void *pv )
{
uint8_t *puc = ( uint8_t * ) pv;
BlockLink_t *pxLink;

    if( pv != NULL )
    {
        /* The memory being freed will have an BlockLink_t structure immediately
        before it. */
        puc -= (uxHeapStructSize - BLOCK_TAIL_LEN - BLOCK_HEAD_LEN) ;

        /* This casting is to keep the compiler from issuing warnings. */
        pxLink = ( void * ) puc;

                #if (configENABLE_MEMORY_DEBUG == 1)
                {
                    taskENTER_CRITICAL(&xMallocMutex);
                    mem_check_block(pxLink);
                    mem_free_block(pxLink);
                    taskEXIT_CRITICAL(&xMallocMutex);
                }
                #endif

        /* Check the block is actually allocated. */
        configASSERT( ( pxLink->xAllocated ) != 0 );
        configASSERT( pxLink->pxNextFreeBlock == NULL );

        if( pxLink->xAllocated != 0 )
        {
            if( pxLink->pxNextFreeBlock == NULL )
            {
                /* The block is being returned to the heap - it is no longer
                allocated. */
                pxLink->xAllocated = 0;

                taskENTER_CRITICAL(&xMallocMutex);
                {
                    /* Add this block to the list of free blocks. */
                    xFreeBytesRemaining[ pxLink->xTag ] += pxLink->xBlockSize;
                    traceFREE( pv, pxLink->xBlockSize );
                    prvInsertBlockIntoFreeList( ( ( BlockLink_t * ) pxLink ) );
                }
                taskEXIT_CRITICAL(&xMallocMutex);
            }
            else
            {
                mtCOVERAGE_TEST_MARKER();
            }
        }
        else
        {
            mtCOVERAGE_TEST_MARKER();
        }
    }
}
/*-----------------------------------------------------------*/

size_t xPortGetFreeHeapSizeTagged( BaseType_t tag )
{
    return xFreeBytesRemaining[ tag ];
}
/*-----------------------------------------------------------*/

size_t xPortGetMinimumEverFreeHeapSizeTagged( BaseType_t tag )
{
    return xMinimumEverFreeBytesRemaining[ tag ];
}
/*-----------------------------------------------------------*/

static void prvInsertBlockIntoFreeList( BlockLink_t *pxBlockToInsert )
{
BlockLink_t *pxIterator;
uint8_t *puc;

    /* Iterate through the list until a block is found that has a higher address
    than the block being inserted. */
    for( pxIterator = &xStart; pxIterator->pxNextFreeBlock < pxBlockToInsert; pxIterator = pxIterator->pxNextFreeBlock )
    {
        /* Nothing to do here, just iterate to the right position. */
    }

    /* Do the block being inserted, and the block it is being inserted after
    make a contiguous block of memory, and are the tags the same? */
    puc = ( uint8_t * ) pxIterator;
    if( ( puc + pxIterator->xBlockSize ) == ( uint8_t * ) pxBlockToInsert && pxBlockToInsert->xTag==pxIterator->xTag)
    {
        pxIterator->xBlockSize += pxBlockToInsert->xBlockSize;
        pxBlockToInsert = pxIterator;
    }
    else
    {
        mtCOVERAGE_TEST_MARKER();
    }

    /* Do the block being inserted, and the block it is being inserted before
    make a contiguous block of memory, and are the tags the same */
    puc = ( uint8_t * ) pxBlockToInsert;
    if( ( puc + pxBlockToInsert->xBlockSize ) == ( uint8_t * ) pxIterator->pxNextFreeBlock && pxBlockToInsert->xTag==pxIterator->pxNextFreeBlock->xTag )
    {
        if( pxIterator->pxNextFreeBlock != pxEnd )
        {
            /* Form one big block from the two blocks. */
            pxBlockToInsert->xBlockSize += pxIterator->pxNextFreeBlock->xBlockSize;
            pxBlockToInsert->pxNextFreeBlock = pxIterator->pxNextFreeBlock->pxNextFreeBlock;
        }
        else
        {
            pxBlockToInsert->pxNextFreeBlock = pxEnd;
        }
    }
    else
    {
        pxBlockToInsert->pxNextFreeBlock = pxIterator->pxNextFreeBlock;
    }

    /* If the block being inserted plugged a gap, so was merged with the block
    before and the block after, then it's pxNextFreeBlock pointer will have
    already been set, and should not be set here as that would make it point
    to itself. */
    if( pxIterator != pxBlockToInsert )
    {
        pxIterator->pxNextFreeBlock = pxBlockToInsert;
    }
    else
    {
        mtCOVERAGE_TEST_MARKER();
    }
}
/*-----------------------------------------------------------*/

void vPortDefineHeapRegionsTagged( const HeapRegionTagged_t * const pxHeapRegions )
{
BlockLink_t *pxFirstFreeBlockInRegion = NULL, *pxPreviousFreeBlock;
uint8_t *pucAlignedHeap;
size_t xTotalRegionSize, xTotalHeapSize = 0;
BaseType_t xDefinedRegions = 0, xRegIdx = 0;
uint32_t ulAddress;
const HeapRegionTagged_t *pxHeapRegion;

    /* Can only call once! */
    configASSERT( pxEnd == NULL );

    vPortCPUInitializeMutex(&xMallocMutex);

    pxHeapRegion = &( pxHeapRegions[ xRegIdx ] );

    while( pxHeapRegion->xSizeInBytes > 0 )
    {
        if ( pxHeapRegion->xTag == -1 ) {
            /* Move onto the next HeapRegionTagged_t structure. */
            xRegIdx++;
            pxHeapRegion = &( pxHeapRegions[ xRegIdx ] );
            continue;
        }

        configASSERT(pxHeapRegion->xTag < HEAPREGIONS_MAX_TAGCOUNT);
        configASSERT(pxHeapRegion->xSizeInBytes < HEAPREGIONS_MAX_REGIONSIZE);
        xTotalRegionSize = pxHeapRegion->xSizeInBytes;

        /* Ensure the heap region starts on a correctly aligned boundary. */
        ulAddress = ( uint32_t ) pxHeapRegion->pucStartAddress;
        if( ( ulAddress & portBYTE_ALIGNMENT_MASK ) != 0 )
        {
            ulAddress += ( portBYTE_ALIGNMENT - 1 );
            ulAddress &= ~portBYTE_ALIGNMENT_MASK;

            /* Adjust the size for the bytes lost to alignment. */
            xTotalRegionSize -= ulAddress - ( uint32_t ) pxHeapRegion->pucStartAddress;
        }

        pucAlignedHeap = ( uint8_t * ) ulAddress;

        /* Set xStart if it has not already been set. */
        if( xDefinedRegions == 0 )
        {
            /* xStart is used to hold a pointer to the first item in the list of
            free blocks.  The void cast is used to prevent compiler warnings. */
            xStart.pxNextFreeBlock = ( BlockLink_t * ) (pucAlignedHeap + BLOCK_HEAD_LEN);
            xStart.xBlockSize = ( size_t ) 0;
        }
        else
        {
            /* Should only get here if one region has already been added to the
            heap. */
            configASSERT( pxEnd != NULL );

            /* Check blocks are passed in with increasing start addresses. */
            configASSERT( ulAddress > ( uint32_t ) pxEnd );
        }

        /* Remember the location of the end marker in the previous region, if
        any. */
        pxPreviousFreeBlock = pxEnd;

        /* pxEnd is used to mark the end of the list of free blocks and is
        inserted at the end of the region space. */
        ulAddress = ( ( uint32_t ) pucAlignedHeap ) + xTotalRegionSize;
        ulAddress -= uxHeapStructSize;
        ulAddress &= ~portBYTE_ALIGNMENT_MASK;
        pxEnd = ( BlockLink_t * ) (ulAddress + BLOCK_HEAD_LEN);
        pxEnd->xBlockSize = 0;
        pxEnd->pxNextFreeBlock = NULL;
        pxEnd->xTag = -1;

        /* To start with there is a single free block in this region that is
        sized to take up the entire heap region minus the space taken by the
        free block structure. */
        pxFirstFreeBlockInRegion = ( BlockLink_t * ) (pucAlignedHeap + BLOCK_HEAD_LEN);
        pxFirstFreeBlockInRegion->xBlockSize = ulAddress - ( uint32_t ) pxFirstFreeBlockInRegion + BLOCK_HEAD_LEN;
        pxFirstFreeBlockInRegion->pxNextFreeBlock = pxEnd;
        pxFirstFreeBlockInRegion->xTag=pxHeapRegion->xTag;

        /* If this is not the first region that makes up the entire heap space
        then link the previous region to this region. */
        if( pxPreviousFreeBlock != NULL )
        {
            pxPreviousFreeBlock->pxNextFreeBlock = pxFirstFreeBlockInRegion;
        }

        xTotalHeapSize += pxFirstFreeBlockInRegion->xBlockSize;
        xMinimumEverFreeBytesRemaining[ pxHeapRegion->xTag ] += pxFirstFreeBlockInRegion->xBlockSize;
        xFreeBytesRemaining[ pxHeapRegion->xTag ] += pxFirstFreeBlockInRegion->xBlockSize;

        /* Move onto the next HeapRegionTagged_t structure. */
        xDefinedRegions++;
        xRegIdx++;
        pxHeapRegion = &( pxHeapRegions[ xRegIdx ] );

                #if (configENABLE_MEMORY_DEBUG == 1)
                {
                    mem_init_dog(pxFirstFreeBlockInRegion);
                    mem_init_dog(pxEnd);
                }
                #endif
    }

    /* Check something was actually defined before it is accessed. */
    configASSERT( xTotalHeapSize );


        #if (configENABLE_MEMORY_DEBUG == 1)
        {
            mem_debug_init(uxHeapStructSize, &xStart, pxEnd, &xMallocMutex);
            mem_check_all(0);
        }
        #endif
}

