#!/usr/bin/env python3

###
# Generates build files for the project.
# This file also includes the project configuration,
# such as compiler flags and the object matching status.
#
# Usage:
#   python3 configure.py
#   ninja
#
# Append --help to see available options.
###

import argparse
import sys
from pathlib import Path

from tools.project import (
    Object,
    ProjectConfig,
    calculate_progress,
    generate_build,
    is_windows,
)

# Game versions
DEFAULT_VERSION = 4
VERSIONS = [
    "GPVE01_D17",  # 0
    "GPVE01_D18",  # 1
    "GPVJ01",      # 2
    "GPVP01",      # 3
    "GPVE01",      # 4
]

parser = argparse.ArgumentParser()
parser.add_argument(
    "mode",
    choices=["configure", "progress"],
    default="configure",
    help="script mode (default: configure)",
    nargs="?",
)
parser.add_argument(
    "-v",
    "--version",
    choices=VERSIONS,
    type=str.upper,
    default=VERSIONS[DEFAULT_VERSION],
    help="version to build",
)
parser.add_argument(
    "--build-dir",
    metavar="DIR",
    type=Path,
    default=Path("build"),
    help="base build directory (default: build)",
)
parser.add_argument(
    "--binutils",
    metavar="BINARY",
    type=Path,
    help="path to binutils (optional)",
)
parser.add_argument(
    "--compilers",
    metavar="DIR",
    type=Path,
    help="path to compilers (optional)",
)
parser.add_argument(
    "--map",
    action="store_true",
    help="generate map file(s)",
)
parser.add_argument(
    "--debug",
    action="store_true",
    help="build with debug info (non-matching)",
)
if not is_windows():
    parser.add_argument(
        "--wrapper",
        metavar="BINARY",
        type=Path,
        help="path to wibo or wine (optional)",
    )
parser.add_argument(
    "--dtk",
    metavar="BINARY | DIR",
    type=Path,
    help="path to decomp-toolkit binary or source (optional)",
)
parser.add_argument(
    "--objdiff",
    metavar="BINARY | DIR",
    type=Path,
    help="path to objdiff-cli binary or source (optional)",
)
parser.add_argument(
    "--sjiswrap",
    metavar="EXE",
    type=Path,
    help="path to sjiswrap.exe (optional)",
)
parser.add_argument(
    "--verbose",
    action="store_true",
    help="print verbose output",
)
parser.add_argument(
    "--non-matching",
    dest="non_matching",
    action="store_true",
    help="builds equivalent (but non-matching) or modded objects",
)
args = parser.parse_args()

config = ProjectConfig()
config.version = str(args.version)
version_num = VERSIONS.index(config.version)

# Apply arguments
config.build_dir = args.build_dir
config.dtk_path = args.dtk
config.objdiff_path = args.objdiff
config.binutils_path = args.binutils
config.compilers_path = args.compilers
config.debug = args.debug
config.generate_map = args.map
config.non_matching = args.non_matching
config.sjiswrap_path = args.sjiswrap
if not is_windows():
    config.wrapper = args.wrapper
# Don't build asm unless we're --non-matching
if not config.non_matching:
    config.asm_dir = None

# Tool versions
config.binutils_tag = "2.42-1"
config.compilers_tag = "20240706"
config.dtk_tag = "v0.9.4"
config.objdiff_tag = "v2.0.0-beta.3"
config.sjiswrap_tag = "v1.1.1"
config.wibo_tag = "0.6.11"

# Project
config.config_path = Path("config") / config.version / "config.yml"
config.check_sha_path = Path("config") / config.version / "build.sha1"
config.asflags = [
    "-mgekko",
    "--strip-local-absolute",
    "-I include",
    f"-I build/{config.version}/include",
    f"--defsym version={version_num}",
    "--no-warn",
]
config.ldflags = [
    "-fp hardware",
    "-nodefaults",
    "-warn off",
]
# Use for any additional files that should cause a re-configure when modified
config.reconfig_deps = []

# Progress configuration
config.progress_all = False
config.progress_use_fancy = True
config.progress_code_fancy_frac = 10_000
config.progress_code_fancy_item = "Pokos"
config.progress_data_fancy_frac = 201
config.progress_data_fancy_item = "treasures"

# Base flags, common to most GC/Wii games.
# Generally leave untouched, with overrides added below.
cflags_base = [
    "-nodefaults",
    "-proc gekko",
    "-align powerpc",
    "-enum int",
    "-fp hardware",
    "-Cpp_exceptions off",
    # "-W all",
    "-O4,p",
    "-inline auto",
    '-pragma "cats off"',
    '-pragma "warn_notinlined off"',
    "-maxerrors 1",
    "-nosyspath",
    "-RTTI off",
    "-fp_contract on",
    "-str reuse",
    "-multibyte",
    "-i include",
    "-i include/stl",
    f"-i build/{config.version}/include",
    f"-DVERNUM={version_num}",
]

# Debug flags
if config.debug:
    cflags_base.extend(["-sym on", "-D_DEBUG=1"])
else:
    cflags_base.extend(["-DNDEBUG=1", "-w off"])

# Metrowerks library flags
cflags_runtime = [
    *cflags_base,
    "-use_lmw_stmw on",
    "-str reuse,readonly",
    "-common off",
    "-inline auto",
]

# Game code flags
cflags_pikmin = [
    *cflags_base,
    "-use_lmw_stmw on",
    "-str reuse,readonly",
    "-common on",
]

config.linker_version = "GC/2.6"

Matching = True                   # Object matches and should be linked
NonMatching = False               # Object does not match and should not be linked
Equivalent = config.non_matching  # Object should be linked when configured with --non-matching

config.warn_missing_config = True
config.warn_missing_source = False
config.libs = [
    {
        "lib": "JStudio_JParticle",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(NonMatching, "JSystem/JStudio_JParticle/object-particle.cpp"),
            Object(Matching, "JSystem/JStudio_JParticle/control.cpp"),
        ],
    },
    {
        "lib": "JMessage",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(NonMatching, "JSystem/JMessage/resource.cpp"),
            Object(Matching, "JSystem/JMessage/data.cpp"),
            Object(NonMatching, "JSystem/JMessage/processor.cpp"),
            Object(Matching, "JSystem/JMessage/control.cpp"),
        ],
    },
    {
        "lib": "JStudio",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(Matching, "JSystem/JStudio/stb-data-parse.cpp"),
            Object(NonMatching, "JSystem/JStudio/functionvalue.cpp"),
            Object(NonMatching, "JSystem/JStudio/fvb.cpp"),
            Object(Matching, "JSystem/JStudio/fvb-data.cpp"),
            Object(Matching, "JSystem/JStudio/fvb-data-parse.cpp"),
            Object(Matching, "JSystem/JStudio/jstudio-control.cpp"),
            Object(Matching, "JSystem/JStudio/jstudio-data.cpp"),
            Object(NonMatching, "JSystem/JStudio/jstudio-object.cpp"),
            Object(Matching, "JSystem/JStudio/object-id.cpp"),
            Object(Matching, "JSystem/JStudio/stb.cpp"),
            Object(Matching, "JSystem/JStudio/stb-data.cpp"),
            Object(Matching, "JSystem/JStudio/jstudio-math.cpp"),
        ],
    },
    {
        "lib": "JStudio_JStage",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(Matching, "JSystem/JStudio_JStage/object-light.cpp"),
            Object(Matching, "JSystem/JStudio_JStage/object.cpp"),
            Object(Matching, "JSystem/JStudio_JStage/object-actor.cpp"),
            Object(Matching, "JSystem/JStudio_JStage/object-ambientlight.cpp"),
            Object(Matching, "JSystem/JStudio_JStage/object-camera.cpp"),
            Object(Matching, "JSystem/JStudio_JStage/object-fog.cpp"),
            Object(Matching, "JSystem/JStudio_JStage/control.cpp"),
        ],
    },
    {
        "lib": "JStudio_JMessage",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(Matching, "JSystem/JStudio_JMessage/object-message.cpp"),
            Object(Matching, "JSystem/JStudio_JMessage/control.cpp"),
        ],
    },
    {
        "lib": "JStudio_JAudio",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(Matching, "JSystem/JStudio_JAudio/object-sound.cpp"),
            Object(Matching, "JSystem/JStudio_JAudio/control.cpp"),
        ],
    },
    {
        "lib": "J3DU",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(Matching, "JSystem/J3DU/J3DUDL.cpp"),
            Object(Matching, "JSystem/J3DU/J3DUMotion.cpp"),
            Object(Matching, "JSystem/J3DU/J3DUShadow.cpp"),
            Object(Matching, "JSystem/J3DU/J3DUMtxCache.cpp"),
        ],
    },
    {
        "lib": "JKernel",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(Matching, "JSystem/JKernel/JKRAram.cpp"),
            Object(Matching, "JSystem/JKernel/JKRAramArchive.cpp"),
            Object(Matching, "JSystem/JKernel/JKRAramBlock.cpp"),
            Object(Matching, "JSystem/JKernel/JKRAramHeap.cpp"),
            Object(Matching, "JSystem/JKernel/JKRAramPiece.cpp"),
            Object(Matching, "JSystem/JKernel/JKRAramStream.cpp"),
            Object(Matching, "JSystem/JKernel/JKRArchivePri.cpp"),
            Object(Matching, "JSystem/JKernel/JKRArchivePub.cpp"),
            Object(Matching, "JSystem/JKernel/JKRCompArchive.cpp"),
            Object(Matching, "JSystem/JKernel/JKRDecomp.cpp"),
            Object(Matching, "JSystem/JKernel/JKRDisposer.cpp"),
            Object(Matching, "JSystem/JKernel/JKRDvdFile.cpp"),
            Object(Matching, "JSystem/JKernel/JKRDvdAramRipper.cpp"),
            Object(Matching, "JSystem/JKernel/JKRDvdArchive.cpp"),
            Object(Matching, "JSystem/JKernel/JKRDvdRipper.cpp"),
            Object(Matching, "JSystem/JKernel/JKRExpHeap.cpp"),
            Object(Matching, "JSystem/JKernel/JKRFileCache.cpp"),
            Object(Matching, "JSystem/JKernel/JKRFileFinder.cpp"),
            Object(Matching, "JSystem/JKernel/JKRFile.cpp"),
            Object(Matching, "JSystem/JKernel/JKRFileLoader.cpp"),
            Object(Matching, "JSystem/JKernel/JKRHeap.cpp"),
            Object(Matching, "JSystem/JKernel/JKRMemArchive.cpp"),
            Object(Matching, "JSystem/JKernel/JKRSolidHeap.cpp"),
            Object(Matching, "JSystem/JKernel/JKRThread.cpp"),
        ],
    },
    {
        "lib": "JSupport",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(Matching, "JSystem/JSupport/JSUOutputStream.cpp"),
            Object(Matching, "JSystem/JSupport/JSUInputStream.cpp"),
            Object(Matching, "JSystem/JSupport/JSUList.cpp"),
            Object(Matching, "JSystem/JSupport/JSUMemoryStream.cpp"),
            Object(Matching, "JSystem/JSupport/JSUFileStream.cpp"),
        ],
    },
    {
        "lib": "JGadget",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(Matching, "JSystem/JGadget/binary.cpp"),
            Object(Matching, "JSystem/JGadget/linklist.cpp"),
            Object(Matching, "JSystem/JGadget/std-list.cpp"),
            Object(Matching, "JSystem/JGadget/std-vector.cpp"),
        ],
    },
    {
        "lib": "JUtility",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(Matching, "JSystem/JUtility/JUTAssert.cpp"),
            Object(Matching, "JSystem/JUtility/JUTConsole.cpp"),
            Object(Matching, "JSystem/JUtility/JUTDbPrint.cpp"),
            Object(Matching, "JSystem/JUtility/JUTDirectFile.cpp"),
            Object(Matching, "JSystem/JUtility/JUTDirectPrint.cpp"),
            Object(Matching, "JSystem/JUtility/JUTException.cpp"),
            Object(Matching, "JSystem/JUtility/JUTFader.cpp"),
            Object(Matching, "JSystem/JUtility/JUTFont.cpp"),
            Object(Matching, "JSystem/JUtility/JUTGamePad.cpp"),
            Object(Matching, "JSystem/JUtility/JUTGraphFifo.cpp"),
            Object(Matching, "JSystem/JUtility/JUTNameTab.cpp"),
            Object(Matching, "JSystem/JUtility/JUTPalette.cpp"),
            Object(Matching, "JSystem/JUtility/JUTProcBar.cpp"),
            Object(Matching, "JSystem/JUtility/JUTResFont.cpp"),
            Object(Matching, "JSystem/JUtility/JUTResource.cpp"),
            Object(Matching, "JSystem/JUtility/JUTRomFont.cpp"),
            Object(Matching, "JSystem/JUtility/JUTFontData_Ascfont_fix12.cpp"),
            Object(Matching, "JSystem/JUtility/JUTTexture.cpp"),
            Object(Matching, "JSystem/JUtility/JUTVideo.cpp"),
            Object(Matching, "JSystem/JUtility/JUTXfb.cpp"),
            Object(Matching, "JSystem/JUtility/JUTCacheFont.cpp"),
        ],
    },
    {
        "lib": "JMath",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(Matching, "JSystem/JMath/JMath.cpp"),
            Object(Matching, "JSystem/JMath/random.cpp"),
            Object(Matching, "JSystem/JMath/JMATrigonometric.cpp"),
        ],
    },
    {
        "lib": "J2D",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(Matching, "JSystem/J2D/J2DOrthoGraph.cpp"),
            Object(Matching, "JSystem/J2D/J2DPerspGraph.cpp"),
            Object(Matching, "JSystem/J2D/J2DGrafContext.cpp"),
            Object(Matching, "JSystem/J2D/J2DPane.cpp"),
            Object(Matching, "JSystem/J2D/J2DPicture.cpp"),
            Object(Matching, "JSystem/J2D/J2DPrint.cpp"),
            Object(NonMatching, "JSystem/J2D/J2DScreen.cpp"),
            Object(Matching, "JSystem/J2D/J2DTextBox.cpp"),
            Object(NonMatching, "JSystem/J2D/J2DWindow.cpp"),
            Object(NonMatching, "JSystem/J2D/J2DWindowEx.cpp"),
            Object(Matching, "JSystem/J2D/J2DAnmLoader.cpp"),
            Object(NonMatching, "JSystem/J2D/J2DBloSaver.cpp"),
            Object(Matching, "JSystem/J2D/J2DManage.cpp"),
            Object(NonMatching, "JSystem/J2D/J2DMatBlock.cpp"),
            Object(NonMatching, "JSystem/J2D/J2DMaterial.cpp"),
            Object(NonMatching, "JSystem/J2D/J2DMaterialFactory.cpp"),
            Object(NonMatching, "JSystem/J2D/J2DPictureEx.cpp"),
            Object(Matching, "JSystem/J2D/J2DTevs.cpp"),
            Object(NonMatching, "JSystem/J2D/J2DTextBoxEx.cpp"),
            Object(Matching, "JSystem/J2D/J2DAnimation.cpp"),
        ],
    },
    {
        "lib": "J3D",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(Matching, "JSystem/J3D/J3DSys.cpp"),
            Object(Matching, "JSystem/J3D/J3DVertex.cpp"),
            Object(Matching, "JSystem/J3D/J3DTransform.cpp"),
            Object(Matching, "JSystem/J3D/J3DPacket.cpp"),
            Object(NonMatching, "JSystem/J3D/J3DShape.cpp"),
            Object(NonMatching, "JSystem/J3D/J3DMaterial.cpp"),
            Object(NonMatching, "JSystem/J3D/J3DTevs.cpp"),
            Object(Matching, "JSystem/J3D/J3DDrawBuffer.cpp"),
            Object(NonMatching, "JSystem/J3D/J3DModel.cpp"),
            Object(Matching, "JSystem/J3D/J3DAnimation.cpp"),
            Object(Matching, "JSystem/J3D/J3DMaterialAnm.cpp"),
            Object(NonMatching, "JSystem/J3D/J3DCluster.cpp"),
            Object(NonMatching, "JSystem/J3D/J3DJoint.cpp"),
            Object(NonMatching, "JSystem/J3D/J3DMaterialFactory.cpp"),
            Object(Matching, "JSystem/J3D/J3DBinaryFormat.cpp"),
            Object(Matching, "JSystem/J3D/J3DModelLoader.cpp"),
            Object(Matching, "JSystem/J3D/J3DJointFactory.cpp"),
            Object(Matching, "JSystem/J3D/J3DShapeFactory.cpp"),
            Object(Matching, "JSystem/J3D/J3DAnmLoader.cpp"),
            Object(Matching, "JSystem/J3D/J3DModelSaver.cpp"),
            Object(NonMatching, "JSystem/J3D/J3DGD.cpp"),
            Object(NonMatching, "JSystem/J3D/J3DMatBlock.cpp"),
            Object(Matching, "JSystem/J3D/J3DModelData.cpp"),
            Object(NonMatching, "JSystem/J3D/J3DMaterialAttach.cpp"),
            Object(NonMatching, "JSystem/J3D/J3DMaterialFactory_v21.cpp"),
            Object(Matching, "JSystem/J3D/J3DShapeMtx.cpp"),
            Object(NonMatching, "JSystem/J3D/J3DModelLoaderCalcSize.cpp"),
            Object(Matching, "JSystem/J3D/J3DJointTree.cpp"),
            Object(Matching, "JSystem/J3D/J3DSkinDeform.cpp"),
            Object(Matching, "JSystem/J3D/J3DShapeDraw.cpp"),
            Object(NonMatching, "JSystem/J3D/J3DMtxBuffer.cpp"),
            Object(Matching, "JSystem/J3D/J3DShapeTable.cpp"),
        ],
    },
    {
        "lib": "JFramework",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(Matching, "JSystem/JFramework/JFWSystem.cpp"),
            Object(Matching, "JSystem/JFramework/JFWDisplay.cpp"),
        ],
    },
    {
        "lib": "JParticle",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(NonMatching, "JSystem/JParticle/JPABaseShape.cpp"),
            Object(Matching, "JSystem/JParticle/JPAChildShape.cpp"),
            Object(NonMatching, "JSystem/JParticle/JPADynamicsBlock.cpp"),
            Object(Matching, "JSystem/JParticle/JPAEmitter.cpp"),
            Object(Matching, "JSystem/JParticle/JPAEmitterManager.cpp"),
            Object(Matching, "JSystem/JParticle/JPAExTexShape.cpp"),
            Object(Matching, "JSystem/JParticle/JPAExtraShape.cpp"),
            Object(Matching, "JSystem/JParticle/JPAFieldBlock.cpp"),
            Object(Matching, "JSystem/JParticle/JPAKeyBlock.cpp"),
            Object(NonMatching, "JSystem/JParticle/JPAMath.cpp"),
            Object(Matching, "JSystem/JParticle/JPAParticle.cpp"),
            Object(NonMatching, "JSystem/JParticle/JPAResource.cpp"),
            Object(NonMatching, "JSystem/JParticle/JPAResourceLoader.cpp"),
            Object(Matching, "JSystem/JParticle/JPAResourceManager.cpp"),
            Object(Matching, "JSystem/JParticle/JPATexture.cpp"),
        ],
    },
    {
        "lib": "JStage",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(Matching, "JSystem/JStage/JSGSystem.cpp"),
            Object(Matching, "JSystem/JStage/JSGCamera.cpp"),
            Object(Matching, "JSystem/JStage/JSGObject.cpp"),
            Object(Matching, "JSystem/JStage/JSGActor.cpp"),
        ],
    },
    {
        "lib": "JAudio_JAS",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(NonMatching, "JSystem/JAudio/JAS/JASWSParser.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASBankMgr.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASBasicBank.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASBasicInst.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASBasicWaveBank.cpp"),
            Object(NonMatching, "JSystem/JAudio/JAS/JASBNKParser.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASDrumSet.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASInstEffect.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASInstRand.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASInstSense.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASSimpleWaveBank.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASWaveArcLoader.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASWaveBank.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASWaveBankMgr.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASBank.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASTrackPort.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASOuterParam.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASPlayer_impl.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASRegisterParam.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASSeqCtrl.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASSeqParser.cpp"),
            Object(NonMatching, "JSystem/JAudio/JAS/JASTrack.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASTrackInterrupt.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASOscillator.cpp"),
            Object(NonMatching, "JSystem/JAudio/JAS/JASChannel.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASDriverIF.cpp"),
            Object(NonMatching, "JSystem/JAudio/JAS/JASDSPChannel.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASDSPInterface.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASAudioThread.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASCalc.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASCallback.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASCmdStack.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASDvdThread.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASHeapCtrl.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASProbe.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASResArcLoader.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASReport.cpp"),
            Object(NonMatching, "JSystem/JAudio/JAS/JASAiCtrl.cpp"),
            Object(Matching, "JSystem/JAudio/JAS/JASChannelUpdater.cpp"),
            Object(NonMatching, "JSystem/JAudio/JAS/JASTaskThread.cpp"),
            Object(NonMatching, "JSystem/JAudio/JAS/JASAramStream.cpp"),
        ],
    },
    {
        "lib": "JAudio_DSP",
        "cflags": [
            *cflags_pikmin,
            "-lang=c++",
            "-func_align 32",
            "-inline noauto",
            "-common off",
            "-use_lmw_stmw off",
            "-O4,s",
        ],
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(Matching, "JSystem/JAudio/dsp/dspproc.c"),
            Object(Matching, "JSystem/JAudio/dsp/dsptask.c"),
            Object(Matching, "JSystem/JAudio/dsp/osdsp.c"),
            Object(Matching, "JSystem/JAudio/dsp/osdsp_task.c"),
        ],
    },
    {
        "lib": "JAudio_JAI",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(NonMatching, "JSystem/JAudio/JAI/JAIAnimation.cpp"),
            Object(Matching, "JSystem/JAudio/JAI/JAIBankWave.cpp"),
            Object(NonMatching, "JSystem/JAudio/JAI/JAIBasic.cpp"),
            Object(Matching, "JSystem/JAudio/JAI/JAIConst.cpp"),
            Object(Matching, "JSystem/JAudio/JAI/JAIDummyObject.cpp"),
            Object(NonMatching, "JSystem/JAudio/JAI/JAIFx.cpp"),
            Object(Matching, "JSystem/JAudio/JAI/JAIGlobalParameter.cpp"),
            Object(NonMatching, "JSystem/JAudio/JAI/JAIInitData.cpp"),
            Object(NonMatching, "JSystem/JAudio/JAI/JAISeMgr.cpp"),
            Object(NonMatching, "JSystem/JAudio/JAI/JAISequenceHeap.cpp"),
            Object(NonMatching, "JSystem/JAudio/JAI/JAISequenceMgr.cpp"),
            Object(NonMatching, "JSystem/JAudio/JAI/JAISound.cpp"),
            Object(NonMatching, "JSystem/JAudio/JAI/JAISoundTable.cpp"),
            Object(NonMatching, "JSystem/JAudio/JAI/JAIStreamMgr.cpp"),
            Object(Matching, "JSystem/JAudio/JAI/JAISystemInterface.cpp"),
            Object(Matching, "JSystem/JAudio/JAI/JAIObject.cpp"),
        ],
    },
    {
        "lib": "JAudio_JAD",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [Object(Matching, "JSystem/JAudio/JAD/JADHioNode.cpp")],
    },
    {
        "lib": "JAudio_JAL",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [Object(Matching, "JSystem/JAudio/JAL/JALCalc.cpp")],
    },
    {
        "lib": "JAudio_JAU",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(Matching, "JSystem/JAudio/JAU/JAUData.cpp"),
            Object(Matching, "JSystem/JAudio/JAU/JAUDataMgr.cpp"),
        ],
    },
    {
        "lib": "TRK_MINNOW_DOLPHIN",
        "cflags": [*cflags_runtime, "-inline deferred", "-sdata 0", "-sdata2 0"],
        "mw_version": "GC/2.6",
        "host": False,
        "objects": [
            Object(Matching, "Dolphin/TRK_MINNOW_DOLPHIN/mainloop.c"),
            Object(Matching, "Dolphin/TRK_MINNOW_DOLPHIN/nubevent.c"),
            Object(Matching, "Dolphin/TRK_MINNOW_DOLPHIN/nubinit.c"),
            Object(Matching, "Dolphin/TRK_MINNOW_DOLPHIN/msg.c"),
            Object(Matching, "Dolphin/TRK_MINNOW_DOLPHIN/msgbuf.c"),
            Object(
                Matching,
                "Dolphin/TRK_MINNOW_DOLPHIN/serpoll.c",
                extra_cflags=["-sdata 8"],
            ),
            Object(Matching, "Dolphin/TRK_MINNOW_DOLPHIN/usr_put.c"),
            Object(Matching, "Dolphin/TRK_MINNOW_DOLPHIN/dispatch.c"),
            Object(Matching, "Dolphin/TRK_MINNOW_DOLPHIN/msghndlr.c"),
            Object(Matching, "Dolphin/TRK_MINNOW_DOLPHIN/support.c"),
            Object(Matching, "Dolphin/TRK_MINNOW_DOLPHIN/mutex_TRK.c"),
            Object(Matching, "Dolphin/TRK_MINNOW_DOLPHIN/notify.c"),
            Object(Matching, "Dolphin/TRK_MINNOW_DOLPHIN/flush_cache.c"),
            Object(Matching, "Dolphin/TRK_MINNOW_DOLPHIN/mem_TRK.c"),
            Object(Matching, "Dolphin/TRK_MINNOW_DOLPHIN/targimpl.c"),
            Object(
                Matching,
                "Dolphin/TRK_MINNOW_DOLPHIN/targsupp.c",
                extra_cflags=["-func_align 32"],
            ),
            Object(Matching, "Dolphin/TRK_MINNOW_DOLPHIN/mpc_7xx_603e.c"),
            Object(Matching, "Dolphin/TRK_MINNOW_DOLPHIN/__exception.s"),
            Object(Matching, "Dolphin/TRK_MINNOW_DOLPHIN/dolphin_trk.c"),
            Object(Matching, "Dolphin/TRK_MINNOW_DOLPHIN/main_TRK.c"),
            Object(Matching, "Dolphin/TRK_MINNOW_DOLPHIN/dolphin_trk_glue.c"),
            Object(Matching, "Dolphin/TRK_MINNOW_DOLPHIN/targcont.c"),
            Object(Matching, "Dolphin/TRK_MINNOW_DOLPHIN/target_options.c"),
            Object(Matching, "Dolphin/TRK_MINNOW_DOLPHIN/mslsupp.c"),
            Object(Matching, "Dolphin/TRK_MINNOW_DOLPHIN/UDP_Stubs.c"),
            Object(
                Matching,
                "Dolphin/TRK_MINNOW_DOLPHIN/ddh/main.c",
                extra_cflags=["-sdata 8"],
            ),
            Object(Matching, "Dolphin/TRK_MINNOW_DOLPHIN/CircleBuffer.c"),
            Object(
                Matching,
                "Dolphin/TRK_MINNOW_DOLPHIN/gdev/main.c",
                extra_cflags=["-sdata 8"],
            ),
            Object(Matching, "Dolphin/TRK_MINNOW_DOLPHIN/MWTrace.c"),
            Object(Matching, "Dolphin/TRK_MINNOW_DOLPHIN/MWCriticalSection_gc.cpp"),
        ],
    },
    {
        "lib": "Runtime",
        "cflags": [*cflags_runtime, "-inline deferred"],
        "mw_version": "GC/2.6",
        "host": False,
        "objects": [
            Object(Matching, "Dolphin/Runtime/__va_arg.c"),
            Object(Matching, "Dolphin/Runtime/global_destructor_chain.c"),
            Object(Matching, "Dolphin/Runtime/CPlusLibPPC.cp"),
            Object(
                Matching,
                "Dolphin/Runtime/NMWException.cp",
                extra_cflags=["-Cpp_exceptions on"],
            ),
            Object(Matching, "Dolphin/Runtime/ptmf.c"),
            Object(Matching, "Dolphin/Runtime/runtime.c"),
            Object(Matching, "Dolphin/Runtime/__init_cpp_exceptions.cpp"),
            Object(
                Matching,
                "Dolphin/Runtime/Gecko_ExceptionPPC.cp",
                extra_cflags=["-Cpp_exceptions on"],
            ),
            Object(Matching, "Dolphin/Runtime/GCN_mem_alloc.c"),
            Object(Matching, "Dolphin/Runtime/__mem.c"),
        ],
    },
    {
        "lib": "MSL_C",
        "cflags": [*cflags_runtime, "-inline deferred"],
        "mw_version": "GC/2.6",
        "host": False,
        "objects": [
            Object(Matching, "Dolphin/MSL_C/PPC_EABI/abort_exit.c"),
            Object(Matching, "Dolphin/MSL_C/MSL_Common/alloc.c"),
            Object(Matching, "Dolphin/MSL_C/MSL_Common/ansi_files.c"),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/ansi_fp.c",
                extra_cflags=["-str pool"],
            ),
            Object(Matching, "Dolphin/MSL_C/MSL_Common/arith.c"),
            Object(Matching, "Dolphin/MSL_C/MSL_Common/buffer_io.c"),
            Object(Matching, "Dolphin/MSL_C/PPC_EABI/critical_regions.gamecube.c"),
            Object(Matching, "Dolphin/MSL_C/MSL_Common/ctype.c"),
            Object(Matching, "Dolphin/MSL_C/MSL_Common/direct_io.c"),
            Object(Matching, "Dolphin/MSL_C/MSL_Common/errno.c"),
            Object(Matching, "Dolphin/MSL_C/MSL_Common/file_io.c"),
            Object(Matching, "Dolphin/MSL_C/MSL_Common/FILE_POS.C"),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common/locale.c",
                extra_cflags=["-str pool"],
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common/mbstring.c",
                extra_cflags=["-inline noauto,deferred"],
            ),
            Object(Matching, "Dolphin/MSL_C/MSL_Common/mem.c"),
            Object(Matching, "Dolphin/MSL_C/MSL_Common/mem_funcs.c"),
            Object(Matching, "Dolphin/MSL_C/MSL_Common/misc_io.c"),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common/printf.c",
                extra_cflags=["-str pool"],
            ),
            Object(Matching, "Dolphin/MSL_C/MSL_Common/rand.c"),
            Object(Matching, "Dolphin/MSL_C/MSL_Common/float.c"),
            Object(Matching, "Dolphin/MSL_C/MSL_Common/scanf.c"),
            Object(Matching, "Dolphin/MSL_C/MSL_Common/string.c"),
            Object(Matching, "Dolphin/MSL_C/MSL_Common/strtold.c"),
            Object(Matching, "Dolphin/MSL_C/MSL_Common/strtoul.c"),
            Object(Matching, "Dolphin/MSL_C/MSL_Common/wchar_io.c"),
            Object(Matching, "Dolphin/MSL_C/PPC_EABI/uart_console_io_gcn.c"),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/e_asin.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/e_atan2.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/e_exp.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/e_fmod.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/e_log.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/e_log10.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/e_pow.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/e_rem_pio2.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/k_cos.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/k_rem_pio2.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/k_sin.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/k_tan.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/s_atan.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/s_ceil.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/s_copysign.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/s_cos.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/s_floor.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/s_frexp.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/s_ldexp.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/s_modf.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/s_sin.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/s_tan.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/w_asin.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/w_atan2.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/w_exp.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/w_fmod.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/w_log10.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/w_pow.c",
            ),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/e_sqrt.c",
            ),
            Object(Matching, "Dolphin/MSL_C/PPC_EABI/math_ppc.c"),
            Object(
                Matching,
                "Dolphin/MSL_C/MSL_Common_Embedded/Math/Double_precision/w_sqrt.c",
            ),
            Object(Matching, "Dolphin/MSL_C/MSL_Common/extras.c"),
        ],
    },
    {
        "lib": "OdemuExi2",
        "cflags": [*cflags_runtime, "-inline deferred"],
        "mw_version": "GC/1.2.5n",
        "host": False,
        "objects": [Object(Matching, "Dolphin/OdemuExi2/DebuggerDriver.c")],
    },
    {
        "lib": "vi",
        "cflags": [*cflags_runtime, "-str noreadonly"],
        "mw_version": "GC/1.2.5n",
        "host": False,
        "objects": [Object(Matching, "Dolphin/vi/vi.c")],
    },
    {
        "lib": "amcstubs",
        "cflags": cflags_runtime,
        "mw_version": "GC/1.2.5n",
        "host": False,
        "objects": [Object(Matching, "Dolphin/amcstubs/AmcExi2Stubs.c")],
    },
    {
        "lib": "ar",
        "cflags": [*cflags_runtime, "-str noreadonly"],
        "mw_version": "GC/1.2.5n",
        "host": False,
        "objects": [
            Object(Matching, "Dolphin/ar/ar.c"),
            Object(Matching, "Dolphin/ar/arq.c"),
        ],
    },
    {
        "lib": "base",
        "cflags": cflags_runtime,
        "mw_version": "GC/1.2.5n",
        "host": False,
        "objects": [Object(Matching, "Dolphin/base/PPCArch.c")],
    },
    {
        "lib": "card",
        "cflags": [*cflags_runtime, "-str noreadonly"],
        "mw_version": "GC/1.2.5n",
        "host": False,
        "objects": [
            Object(Matching, "Dolphin/card/CARDBios.c"),
            Object(Matching, "Dolphin/card/CARDUnlock.c"),
            Object(Matching, "Dolphin/card/CARDRdwr.c"),
            Object(Matching, "Dolphin/card/CARDBlock.c"),
            Object(Matching, "Dolphin/card/CARDDir.c"),
            Object(Matching, "Dolphin/card/CARDCheck.c"),
            Object(Matching, "Dolphin/card/CARDMount.c"),
            Object(Matching, "Dolphin/card/CARDFormat.c"),
            Object(Matching, "Dolphin/card/CARDOpen.c"),
            Object(Matching, "Dolphin/card/CARDCreate.c"),
            Object(Matching, "Dolphin/card/CARDRead.c"),
            Object(Matching, "Dolphin/card/CARDWrite.c"),
            Object(Matching, "Dolphin/card/CARDStat.c"),
            Object(Matching, "Dolphin/card/CARDNet.c"),
        ],
    },
    {
        "lib": "db",
        "cflags": [*cflags_runtime, "-str noreadonly"],
        "mw_version": "GC/1.2.5n",
        "host": False,
        "objects": [Object(Matching, "Dolphin/db/db.c")],
    },
    {
        "lib": "dsp",
        "cflags": [*cflags_runtime, "-str noreadonly"],
        "mw_version": "GC/1.2.5n",
        "host": False,
        "objects": [
            Object(Matching, "Dolphin/dsp/dsp.c"),
            Object(Matching, "Dolphin/dsp/dsp_debug.c"),
            Object(Matching, "Dolphin/dsp/dsp_task.c"),
        ],
    },
    {
        "lib": "dvd",
        "cflags": [*cflags_runtime, "-str noreadonly"],
        "mw_version": "GC/1.2.5n",
        "host": False,
        "objects": [
            Object(Matching, "Dolphin/dvd/dvdlow.c"),
            Object(Matching, "Dolphin/dvd/dvdfs.c"),
            Object(Matching, "Dolphin/dvd/dvd.c"),
            Object(Matching, "Dolphin/dvd/dvdqueue.c"),
            Object(Matching, "Dolphin/dvd/dvderror.c"),
            Object(Matching, "Dolphin/dvd/dvdidutils.c"),
            Object(Matching, "Dolphin/dvd/dvdFatal.c"),
            Object(Matching, "Dolphin/dvd/fstload.c"),
        ],
    },
    {
        "lib": "exi",
        "cflags": [*cflags_runtime, "-str noreadonly"],
        "mw_version": "GC/1.2.5n",
        "host": False,
        "objects": [
            Object(Matching, "Dolphin/exi/EXIBios.c"),
            Object(Matching, "Dolphin/exi/EXIUart.c"),
        ],
    },
    {
        "lib": "gd",
        "cflags": cflags_runtime,
        "mw_version": "GC/1.2.5n",
        "host": False,
        "objects": [
            Object(Matching, "Dolphin/gd/GDBase.c"),
            Object(Matching, "Dolphin/gd/GDGeometry.c"),
        ],
    },
    {
        "lib": "gx",
        "cflags": [*cflags_runtime, "-str noreadonly", "-fp_contract off"],
        "mw_version": "GC/1.2.5n",
        "host": False,
        "objects": [
            Object(Matching, "Dolphin/gx/GXInit.c"),
            Object(Matching, "Dolphin/gx/GXFifo.c"),
            Object(Matching, "Dolphin/gx/GXAttr.c"),
            Object(Matching, "Dolphin/gx/GXMisc.c"),
            Object(Matching, "Dolphin/gx/GXGeometry.c"),
            Object(Matching, "Dolphin/gx/GXFrameBuf.c"),
            Object(Matching, "Dolphin/gx/GXLight.c"),
            Object(Matching, "Dolphin/gx/GXTexture.c"),
            Object(Matching, "Dolphin/gx/GXBump.c"),
            Object(Matching, "Dolphin/gx/GXTev.c"),
            Object(Matching, "Dolphin/gx/GXPixel.c"),
            Object(Matching, "Dolphin/gx/GXDisplayList.c"),
            Object(Matching, "Dolphin/gx/GXTransform.c"),
            Object(Matching, "Dolphin/gx/GXPerf.c"),
        ],
    },
    {
        "lib": "mtx",
        "cflags": cflags_runtime,
        "mw_version": "GC/1.2.5n",
        "host": False,
        "objects": [
            Object(Matching, "Dolphin/mtx/mtx.c"),
            Object(Matching, "Dolphin/mtx/mtxvec.c"),
            Object(Matching, "Dolphin/mtx/mtx44.c"),
            Object(Matching, "Dolphin/mtx/vec.c"),
        ],
    },
    {
        "lib": "odenotstub",
        "cflags": cflags_runtime,
        "mw_version": "GC/1.2.5n",
        "host": False,
        "objects": [Object(Matching, "Dolphin/odenotstub/odenotstub.c")],
    },
    {
        "lib": "os",
        "cflags": [*cflags_runtime, "-str noreadonly"],
        "mw_version": "GC/1.2.5n",
        "host": False,
        "objects": [
            Object(Matching, "Dolphin/os/OS.c"),
            Object(Matching, "Dolphin/os/OSAlarm.c"),
            Object(Matching, "Dolphin/os/OSAlloc.c"),
            Object(Matching, "Dolphin/os/OSArena.c"),
            Object(Matching, "Dolphin/os/OSAudioSystem.c"),
            Object(Matching, "Dolphin/os/OSCache.c"),
            Object(Matching, "Dolphin/os/OSContext.c"),
            Object(Matching, "Dolphin/os/OSError.c"),
            Object(Matching, "Dolphin/os/OSFont.c"),
            Object(Matching, "Dolphin/os/OSInterrupt.c"),
            Object(Matching, "Dolphin/os/OSLink.c"),
            Object(Matching, "Dolphin/os/OSMessage.c"),
            Object(Matching, "Dolphin/os/OSMemory.c"),
            Object(Matching, "Dolphin/os/OSMutex.c"),
            Object(Matching, "Dolphin/os/OSReboot.c"),
            Object(Matching, "Dolphin/os/OSReset.c"),
            Object(Matching, "Dolphin/os/OSResetSW.c"),
            Object(Matching, "Dolphin/os/OSRtc.c"),
            Object(Matching, "Dolphin/os/OSSync.c"),
            Object(Matching, "Dolphin/os/OSThread.c"),
            Object(Matching, "Dolphin/os/OSTime.c"),
            Object(Matching, "Dolphin/os/__start.c"),
            Object(Matching, "Dolphin/os/__ppc_eabi_init.cpp"),
        ],
    },
    {
        "lib": "pad",
        "cflags": [*cflags_runtime, "-fp_contract off", "-str noreadonly"],
        "mw_version": "GC/1.2.5n",
        "host": False,
        "objects": [
            Object(Matching, "Dolphin/pad/Padclamp.c"),
            Object(Matching, "Dolphin/pad/Pad.c"),
        ],
    },
    {
        "lib": "si",
        "cflags": [*cflags_runtime, "-str noreadonly"],
        "mw_version": "GC/1.2.5n",
        "host": False,
        "objects": [
            Object(Matching, "Dolphin/si/SIBios.c"),
            Object(Matching, "Dolphin/si/SISamplingRate.c"),
        ],
    },
    {
        "lib": "ai",
        "cflags": [*cflags_runtime, "-str noreadonly"],
        "mw_version": "GC/1.2.5n",
        "host": False,
        "objects": [Object(Matching, "Dolphin/ai/ai.c")],
    },
    {
        "lib": "thp",
        "cflags": [*cflags_runtime, "-str noreadonly"],
        "mw_version": "GC/1.2.5n",
        "host": False,
        "objects": [
            Object(Matching, "Dolphin/thp/THPDec.c"),
            Object(Matching, "Dolphin/thp/THPAudio.c"),
        ],
    },
    {
        "lib": "gba",
        "cflags": [*cflags_runtime, "-str noreadonly"],
        "mw_version": "GC/1.2.5n",
        "host": False,
        "objects": [
            Object(Matching, "Dolphin/gba/GBA.c"),
            Object(Matching, "Dolphin/gba/GBARead.c"),
            Object(Matching, "Dolphin/gba/GBAWrite.c"),
            Object(Matching, "Dolphin/gba/GBAXfer.c"),
        ],
    },
    {
        "lib": "plugProjectYamashitaU",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(Matching, "plugProjectYamashitaU/enemyBase.cpp"),
            Object(Matching, "plugProjectYamashitaU/pelplant.cpp"),
            Object(Matching, "plugProjectYamashitaU/pelplantState.cpp"),
            Object(Matching, "plugProjectYamashitaU/enemyInteractBattle.cpp"),
            Object(Matching, "plugProjectYamashitaU/generalEnemyMgr.cpp"),
            Object(Matching, "plugProjectYamashitaU/kochappyAnimator.cpp"),
            Object(Matching, "plugProjectYamashitaU/kochappy.cpp"),
            Object(NonMatching, "plugProjectYamashitaU/kochappyState.cpp"),
            Object(Matching, "plugProjectYamashitaU/kochappyMgr.cpp"),
            Object(NonMatching, "plugProjectYamashitaU/enemyAction.cpp"),
            Object(Matching, "plugProjectYamashitaU/chappy.cpp"),
            Object(NonMatching, "plugProjectYamashitaU/chappyState.cpp"),
            Object(Matching, "plugProjectYamashitaU/chappyAnimator.cpp"),
            Object(Matching, "plugProjectYamashitaU/chappyMgr.cpp"),
            Object(Matching, "plugProjectYamashitaU/lifeGaugeMgr.cpp"),
            Object(Matching, "plugProjectYamashitaU/carryInfoMgr.cpp"),
            Object(NonMatching, "plugProjectYamashitaU/gameLightMgr.cpp"),
            Object(NonMatching, "plugProjectYamashitaU/vtxAnm.cpp"),
            Object(Matching, "plugProjectYamashitaU/enemyInfo.cpp"),
            Object(Matching, "plugProjectYamashitaU/farm.cpp"),
            Object(Matching, "plugProjectYamashitaU/farmMgr.cpp"),
            Object(Matching, "plugProjectYamashitaU/genEnemy.cpp"),
            Object(Matching, "plugProjectYamashitaU/timeMgr.cpp"),
            Object(Matching, "plugProjectYamashitaU/pelplantGenerator.cpp"),
            Object(Matching, "plugProjectYamashitaU/enemyInteractActions.cpp"),
            Object(Matching, "plugProjectYamashitaU/enemyAnimatorBase.cpp"),
            Object(Matching, "plugProjectYamashitaU/enemyStoneMgr.cpp"),
            Object(Matching, "plugProjectYamashitaU/enemyStoenInfo.cpp"),
            Object(NonMatching, "plugProjectYamashitaU/enemyStoneDrawInfo.cpp"),
            Object(Matching, "plugProjectYamashitaU/enemyStoneObj.cpp"),
            Object(Matching, "plugProjectYamashitaU/enemyParmsBase.cpp"),
            Object(Matching, "plugProjectYamashitaU/walkSmokeEffect.cpp"),
            Object(Matching, "plugProjectYamashitaU/ChappyBaseMgr.cpp"),
            Object(Matching, "plugProjectYamashitaU/ChappyBase.cpp"),
            Object(Matching, "plugProjectYamashitaU/BlueChappyMgr.cpp"),
            Object(Matching, "plugProjectYamashitaU/BlueChappy.cpp"),
            Object(Matching, "plugProjectYamashitaU/YellowChappyMgr.cpp"),
            Object(Matching, "plugProjectYamashitaU/YellowChappy.cpp"),
            Object(Matching, "plugProjectYamashitaU/BlueKochappyMgr.cpp"),
            Object(Matching, "plugProjectYamashitaU/BlueKochappy.cpp"),
            Object(Matching, "plugProjectYamashitaU/YellowKochappy.cpp"),
            Object(Matching, "plugProjectYamashitaU/YellowKochappyMgr.cpp"),
            Object(Matching, "plugProjectYamashitaU/KochappyBaseMgr.cpp"),
            Object(Matching, "plugProjectYamashitaU/KochappyBase.cpp"),
            Object(Matching, "plugProjectYamashitaU/enemyBlendAnimatorBase.cpp"),
            Object(Matching, "plugProjectYamashitaU/enemyPelletInfo.cpp"),
            Object(Matching, "plugProjectYamashitaU/enemyEffectNode.cpp"),
            Object(Matching, "plugProjectYamashitaU/enemyMgrBase.cpp"),
            Object(Matching, "plugProjectYamashitaU/enemyFSM.cpp"),
            Object(NonMatching, "plugProjectYamashitaU/singleGS_ZukanParms.cpp"),
            Object(Matching, "plugProjectYamashitaU/treasureLightMgr.cpp"),
            Object(Matching, "plugProjectYamashitaU/effectAnimator.cpp"),
        ],
    },
    {
        "lib": "plugProjectKandoU",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(Matching, "plugProjectKandoU/pikiAnimator.cpp"),
            Object(NonMatching, "plugProjectKandoU/collinfo.cpp"),
            Object(NonMatching, "plugProjectKandoU/gameDynamics.cpp"),
            Object(Matching, "plugProjectKandoU/creature.cpp"),
            Object(NonMatching, "plugProjectKandoU/fakePiki.cpp"),
            Object(NonMatching, "plugProjectKandoU/navi.cpp"),
            Object(NonMatching, "plugProjectKandoU/piki.cpp"),
            Object(NonMatching, "plugProjectKandoU/baseGameSection.cpp"),
            Object(Matching, "plugProjectKandoU/singleGameSection.cpp"),
            Object(NonMatching, "plugProjectKandoU/cellPyramid.cpp"),
            Object(Matching, "plugProjectKandoU/naviMgr.cpp"),
            Object(Matching, "plugProjectKandoU/pikiMgr.cpp"),
            Object(Matching, "plugProjectKandoU/mapMgr.cpp"),
            Object(Matching, "plugProjectKandoU/baseHIOSection.cpp"),
            Object(NonMatching, "plugProjectKandoU/naviWhistle.cpp"),
            Object(NonMatching, "plugProjectKandoU/pelletMgr.cpp"),
            Object(NonMatching, "plugProjectKandoU/routeMgr.cpp"),
            Object(Matching, "plugProjectKandoU/onyonMgr.cpp"),
            Object(Matching, "plugProjectKandoU/objectTypes.cpp"),
            Object(NonMatching, "plugProjectKandoU/naviState.cpp"),
            Object(NonMatching, "plugProjectKandoU/pikiState.cpp"),
            Object(Matching, "plugProjectKandoU/interactPiki.cpp"),
            Object(NonMatching, "plugProjectKandoU/gameCPlate.cpp"),
            Object(Matching, "plugProjectKandoU/updateMgr.cpp"),
            Object(Matching, "plugProjectKandoU/aiAction.cpp"),
            Object(NonMatching, "plugProjectKandoU/aiPrimitives.cpp"),
            Object(NonMatching, "plugProjectKandoU/aiFormation.cpp"),
            Object(Matching, "plugProjectKandoU/creatureStick.cpp"),
            Object(Matching, "plugProjectKandoU/interactBattle.cpp"),
            Object(Matching, "plugProjectKandoU/aiFree.cpp"),
            Object(Matching, "plugProjectKandoU/aiAttack.cpp"),
            Object(Matching, "plugProjectKandoU/aiTransport.cpp"),
            Object(NonMatching, "plugProjectKandoU/aiEnter.cpp"),
            Object(NonMatching, "plugProjectKandoU/pathfinder.cpp"),
            Object(NonMatching, "plugProjectKandoU/pelletState.cpp"),
            Object(NonMatching, "plugProjectKandoU/dynCreature.cpp"),
            Object(Matching, "plugProjectKandoU/gameGenerator.cpp"),
            Object(Matching, "plugProjectKandoU/genPiki.cpp"),
            Object(Matching, "plugProjectKandoU/genNavi.cpp"),
            Object(Matching, "plugProjectKandoU/genItem.cpp"),
            Object(Matching, "plugProjectKandoU/gameStages.cpp"),
            Object(Matching, "plugProjectKandoU/gameSeaMgr.cpp"),
            Object(Matching, "plugProjectKandoU/pikiAI.cpp"),
            Object(Matching, "plugProjectKandoU/pelletConfig.cpp"),
            Object(Matching, "plugProjectKandoU/gameFootmark.cpp"),
            Object(Matching, "plugProjectKandoU/gameSystem.cpp"),
            Object(Matching, "plugProjectKandoU/aiConstants.cpp"),
            Object(NonMatching, "plugProjectKandoU/gameMapParts.cpp"),
            Object(Matching, "plugProjectKandoU/vsGameSection.cpp"),
            Object(Matching, "plugProjectKandoU/gamePlatMgr.cpp"),
            Object(NonMatching, "plugProjectKandoU/itemGate.cpp"),
            Object(Matching, "plugProjectKandoU/itemMgr.cpp"),
            Object(NonMatching, "plugProjectKandoU/aiBreakGate.cpp"),
            Object(Matching, "plugProjectKandoU/gameStat.cpp"),
            Object(Matching, "plugProjectKandoU/itemHole.cpp"),
            Object(NonMatching, "plugProjectKandoU/itemHoney.cpp"),
            Object(NonMatching, "plugProjectKandoU/gameCaveInfo.cpp"),
            Object(NonMatching, "plugProjectKandoU/creatureLOD.cpp"),
            Object(Matching, "plugProjectKandoU/interactNavi.cpp"),
            Object(NonMatching, "plugProjectKandoU/itemPikihead.cpp"),
            Object(NonMatching, "plugProjectKandoU/itemPlant.cpp"),
            Object(Matching, "plugProjectKandoU/itemRock.cpp"),
            Object(Matching, "plugProjectKandoU/aiBreakRock.cpp"),
            Object(Matching, "plugProjectKandoU/aiCrop.cpp"),
            Object(Matching, "plugProjectKandoU/registItem.cpp"),
            Object(NonMatching, "plugProjectKandoU/gamePlayData.cpp"),
            Object(NonMatching, "plugProjectKandoU/itemCave.cpp"),
            Object(Matching, "plugProjectKandoU/itemBigFountain.cpp"),
            Object(NonMatching, "plugProjectKandoU/itemBridge.cpp"),
            Object(Matching, "plugProjectKandoU/pikiContainer.cpp"),
            Object(Matching, "plugProjectKandoU/gameGeneratorCache.cpp"),
            Object(Matching, "plugProjectKandoU/itemTreasure.cpp"),
            Object(Matching, "plugProjectKandoU/itemDownFloor.cpp"),
            Object(Matching, "plugProjectKandoU/kandoLib.cpp"),
            Object(Matching, "plugProjectKandoU/itemBarrel.cpp"),
            Object(Matching, "plugProjectKandoU/pelletNumber.cpp"),
            Object(Matching, "plugProjectKandoU/pelletCarcass.cpp"),
            Object(Matching, "plugProjectKandoU/pelletFruit.cpp"),
            Object(Matching, "plugProjectKandoU/pelletOtakara.cpp"),
            Object(Matching, "plugProjectKandoU/genPellet.cpp"),
            Object(Matching, "plugProjectKandoU/pelletItem.cpp"),
            Object(Matching, "plugProjectKandoU/mapMgrTraceMove.cpp"),
            Object(Matching, "plugProjectKandoU/efxModelObjects.cpp"),
            Object(NonMatching, "plugProjectKandoU/itemUjamushi.cpp"),
            Object(Matching, "plugProjectKandoU/aiWeed.cpp"),
            Object(NonMatching, "plugProjectKandoU/flockMgr.cpp"),
            Object(NonMatching, "plugProjectKandoU/itemWeed.cpp"),
            Object(Matching, "plugProjectKandoU/aiBridge.cpp"),
            Object(Matching, "plugProjectKandoU/aiTeki.cpp"),
            Object(Matching, "plugProjectKandoU/singleGS_MainGame.cpp"),
            Object(Matching, "plugProjectKandoU/singleGS_CaveGame.cpp"),
            Object(Matching, "plugProjectKandoU/singleGS_MainResult.cpp"),
            Object(NonMatching, "plugProjectKandoU/singleGS_CaveResult.cpp"),
            Object(Matching, "plugProjectKandoU/singleGS_WorldMap.cpp"),
            Object(Matching, "plugProjectKandoU/singleGS_FileSelect.cpp"),
            Object(NonMatching, "plugProjectKandoU/gamePlayDataMemCard.cpp"),
            Object(Matching, "plugProjectKandoU/baseHIOparms.cpp"),
            Object(Matching, "plugProjectKandoU/radarInfo.cpp"),
            Object(Matching, "plugProjectKandoU/singleGS_Movie.cpp"),
            Object(Matching, "plugProjectKandoU/navi_demoCheck.cpp"),
            Object(NonMatching, "plugProjectKandoU/singleGS_Zukan.cpp"),
            Object(Matching, "plugProjectKandoU/gameResultTexMgr.cpp"),
            Object(Matching, "plugProjectKandoU/gamePelletList.cpp"),
            Object(Matching, "plugProjectKandoU/vsGS_Title.cpp"),
            Object(Matching, "plugProjectKandoU/vsGS_Game.cpp"),
            Object(Matching, "plugProjectKandoU/vsGS_Result.cpp"),
            Object(Matching, "plugProjectKandoU/vsGS_Load.cpp"),
            Object(Matching, "plugProjectKandoU/vsStageData.cpp"),
            Object(Matching, "plugProjectKandoU/cellMgrParms.cpp"),
            Object(NonMatching, "plugProjectKandoU/cellIterator.cpp"),
            Object(Matching, "plugProjectKandoU/vsGS_VSGame.cpp"),
            Object(Matching, "plugProjectKandoU/gameSoundEvent.cpp"),
            Object(Matching, "plugProjectKandoU/aiBattle.cpp"),
            Object(Matching, "plugProjectKandoU/gameDeathCount.cpp"),
            Object(Matching, "plugProjectKandoU/aiBore.cpp"),
            Object(Matching, "plugProjectKandoU/mapPartsView.cpp"),
            Object(Matching, "plugProjectKandoU/singleGS_Ending.cpp"),
            Object(Matching, "plugProjectKandoU/gameIconTexture.cpp"),
            Object(Matching, "plugProjectKandoU/gameTekiStat.cpp"),
            Object(Matching, "plugProjectKandoU/gameHighscore.cpp"),
            Object(Matching, "plugProjectKandoU/gamePlayCommonData.cpp"),
            Object(Matching, "plugProjectKandoU/pelletCarry.cpp"),
            Object(Matching, "plugProjectKandoU/gameChallenge2D.cpp"),
            Object(Matching, "plugProjectKandoU/vsTekiMgr.cpp"),
            Object(NonMatching, "plugProjectKandoU/vsCardMgr.cpp"),
            Object(Matching, "plugProjectKandoU/aiRescue.cpp"),
            Object(Matching, "plugProjectKandoU/baseGameSectionDraw.cpp"),
            Object(Matching, "plugProjectKandoU/singleGS_Load.cpp"),
            Object(Matching, "plugProjectKandoU/singleGS_DayEnd.cpp"),
            Object(Matching, "plugProjectKandoU/baseGameSectionKantei.cpp"),
            Object(Matching, "plugProjectKandoU/sweepPrune.cpp"),
            Object(NonMatching, "plugProjectKandoU/texCaster.cpp"),
            Object(Matching, "plugProjectKandoU/vsCoinOtakaraName.cpp"),
            Object(Matching, "plugProjectKandoU/pelletBirthBuffer.cpp"),
            Object(Matching, "plugProjectKandoU/vsFifo.cpp"),
        ],
    },
    {
        "lib": "plugProjectNishimuraU",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(NonMatching, "plugProjectNishimuraU/nslibmath.cpp"),
            Object(Matching, "plugProjectNishimuraU/ShadowCylinder.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/playCamera.cpp"),
            Object(Matching, "plugProjectNishimuraU/shadowMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/MapUnit.cpp"),
            Object(Matching, "plugProjectNishimuraU/MapNode.cpp"),
            Object(Matching, "plugProjectNishimuraU/EnemyUnit.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/RandMapMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/RandMapDraw.cpp"),
            Object(Matching, "plugProjectNishimuraU/RandMapChecker.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/RandMapUnit.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/RandEnemyUnit.cpp"),
            Object(Matching, "plugProjectNishimuraU/DoorNode.cpp"),
            Object(Matching, "plugProjectNishimuraU/MapUnitGenerator.cpp"),
            Object(Matching, "plugProjectNishimuraU/MapCreator.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/RandMapScore.cpp"),
            Object(Matching, "plugProjectNishimuraU/ItemUnit.cpp"),
            Object(Matching, "plugProjectNishimuraU/RandItemUnit.cpp"),
            Object(Matching, "plugProjectNishimuraU/GateUnit.cpp"),
            Object(Matching, "plugProjectNishimuraU/RandGateUnit.cpp"),
            Object(Matching, "plugProjectNishimuraU/ObjectLayout.cpp"),
            Object(Matching, "plugProjectNishimuraU/CameraMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/ContRumble.cpp"),
            Object(Matching, "plugProjectNishimuraU/RumbleMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/PomAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/PomMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/Pom.cpp"),
            Object(Matching, "plugProjectNishimuraU/PomState.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/FrogState.cpp"),
            Object(Matching, "plugProjectNishimuraU/FrogAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/FrogMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/Frog.cpp"),
            Object(Matching, "plugProjectNishimuraU/UjibMgr.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/Ujib.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/UjibState.cpp"),
            Object(Matching, "plugProjectNishimuraU/UjibAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/KoganeState.cpp"),
            Object(Matching, "plugProjectNishimuraU/KoganeMgr.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/Kogane.cpp"),
            Object(Matching, "plugProjectNishimuraU/KoganeAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/QurioneState.cpp"),
            Object(Matching, "plugProjectNishimuraU/QurioneMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/Qurione.cpp"),
            Object(Matching, "plugProjectNishimuraU/QurioneAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/RumbleData.cpp"),
            Object(Matching, "plugProjectNishimuraU/MaroFrogMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/MaroFrog.cpp"),
            Object(Matching, "plugProjectNishimuraU/RockState.cpp"),
            Object(Matching, "plugProjectNishimuraU/RockAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/RockMgr.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/Rock.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/UjiaState.cpp"),
            Object(Matching, "plugProjectNishimuraU/UjiaAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/UjiaMgr.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/Ujia.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/TobiState.cpp"),
            Object(Matching, "plugProjectNishimuraU/TobiAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/TobiMgr.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/Tobi.cpp"),
            Object(Matching, "plugProjectNishimuraU/HibaState.cpp"),
            Object(Matching, "plugProjectNishimuraU/HibaAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/HibaMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/Hiba.cpp"),
            Object(Matching, "plugProjectNishimuraU/GasHibaState.cpp"),
            Object(Matching, "plugProjectNishimuraU/GasHibaAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/GasHibaMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/GasHiba.cpp"),
            Object(Matching, "plugProjectNishimuraU/ElecHibaState.cpp"),
            Object(Matching, "plugProjectNishimuraU/ElecHibaAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/ElecHibaMgr.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/ElecHiba.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/SaraiState.cpp"),
            Object(Matching, "plugProjectNishimuraU/SaraiAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/SaraiMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/Sarai.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/TankState.cpp"),
            Object(Matching, "plugProjectNishimuraU/TankAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/TankMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/Tank.cpp"),
            Object(Matching, "plugProjectNishimuraU/CatfishMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/Catfish.cpp"),
            Object(Matching, "plugProjectNishimuraU/TadpoleState.cpp"),
            Object(Matching, "plugProjectNishimuraU/TadpoleAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/TadpoleMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/Tadpole.cpp"),
            Object(Matching, "plugProjectNishimuraU/ElecBugState.cpp"),
            Object(Matching, "plugProjectNishimuraU/ElecBugAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/ElecBugMgr.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/ElecBug.cpp"),
            Object(Matching, "plugProjectNishimuraU/WtankMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/Wtank.cpp"),
            Object(Matching, "plugProjectNishimuraU/ArmorMgr.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/Armor.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/Mar.cpp"),
            Object(Matching, "plugProjectNishimuraU/MarAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/MarMgr.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/MarState.cpp"),
            Object(Matching, "plugProjectNishimuraU/WealthyMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/Wealthy.cpp"),
            Object(Matching, "plugProjectNishimuraU/FartMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/Fart.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/ArmorState.cpp"),
            Object(Matching, "plugProjectNishimuraU/ArmorAnimator.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/QueenState.cpp"),
            Object(Matching, "plugProjectNishimuraU/QueenAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/QueenMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/Queen.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/BabyState.cpp"),
            Object(Matching, "plugProjectNishimuraU/BabyAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/BabyMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/Baby.cpp"),
            Object(Matching, "plugProjectNishimuraU/DemonMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/Demon.cpp"),
            Object(Matching, "plugProjectNishimuraU/QueenShadow.cpp"),
            Object(Matching, "plugProjectNishimuraU/FireChappyMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/FireChappy.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/SnakeCrowState.cpp"),
            Object(Matching, "plugProjectNishimuraU/SnakeCrowAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/SnakeCrowMgr.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/SnakeCrow.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/KumaChappyState.cpp"),
            Object(Matching, "plugProjectNishimuraU/KumaChappyAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/KumaChappyMgr.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/KumaChappy.cpp"),
            Object(Matching, "plugProjectNishimuraU/FuefukiState.cpp"),
            Object(Matching, "plugProjectNishimuraU/FuefukiAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/FuefukiMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/Fuefuki.cpp"),
            Object(Matching, "plugProjectNishimuraU/KoganemushiMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/Koganemushi.cpp"),
            Object(Matching, "plugProjectNishimuraU/FtankMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/Ftank.cpp"),
            Object(Matching, "plugProjectNishimuraU/RandPlantUnit.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/HanachirashiState.cpp"),
            Object(Matching, "plugProjectNishimuraU/HanachirashiAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/HanachirashiMgr.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/Hanachirashi.cpp"),
            Object(Matching, "plugProjectNishimuraU/DamagumoState.cpp"),
            Object(Matching, "plugProjectNishimuraU/DamagumoAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/DamagumoMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/Damagumo.cpp"),
            Object(Matching, "plugProjectNishimuraU/IKSystemMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/IKSystemBase.cpp"),
            Object(Matching, "plugProjectNishimuraU/DamagumoShadow.cpp"),
            Object(Matching, "plugProjectNishimuraU/KurageState.cpp"),
            Object(Matching, "plugProjectNishimuraU/KurageAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/KurageMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/Kurage.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/BombSaraiState.cpp"),
            Object(Matching, "plugProjectNishimuraU/BombSaraiAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/BombSaraiMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/BombSarai.cpp"),
            Object(Matching, "plugProjectNishimuraU/OtakaraBaseState.cpp"),
            Object(Matching, "plugProjectNishimuraU/OtakaraBaseAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/OtakaraBaseMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/OtakaraBase.cpp"),
            Object(Matching, "plugProjectNishimuraU/FireOtakaraMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/FireOtakara.cpp"),
            Object(Matching, "plugProjectNishimuraU/WaterOtakaraMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/WaterOtakara.cpp"),
            Object(Matching, "plugProjectNishimuraU/GasOtakaraMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/GasOtakara.cpp"),
            Object(Matching, "plugProjectNishimuraU/ElecOtakaraMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/ElecOtakara.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/ImomushiState.cpp"),
            Object(Matching, "plugProjectNishimuraU/ImomushiAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/ImomushiMgr.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/Imomushi.cpp"),
            Object(Matching, "plugProjectNishimuraU/HoudaiState.cpp"),
            Object(Matching, "plugProjectNishimuraU/HoudaiAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/HoudaiMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/HoudaiShadow.cpp"),
            Object(Matching, "plugProjectNishimuraU/Houdai.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/HoudaiShotGun.cpp"),
            Object(Matching, "plugProjectNishimuraU/LeafChappyMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/LeafChappy.cpp"),
            Object(Matching, "plugProjectNishimuraU/BigFootState.cpp"),
            Object(Matching, "plugProjectNishimuraU/BigFootAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/BigFootMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/BigFootShadow.cpp"),
            Object(Matching, "plugProjectNishimuraU/BigFoot.cpp"),
            Object(Matching, "plugProjectNishimuraU/SnakeWholeState.cpp"),
            Object(Matching, "plugProjectNishimuraU/SnakeWholeAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/SnakeWholeMgr.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/SnakeWhole.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/SnakeJointMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/OniKurageState.cpp"),
            Object(Matching, "plugProjectNishimuraU/OniKurageAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/OniKurageMgr.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/OniKurage.cpp"),
            Object(Matching, "plugProjectNishimuraU/BigTreasureState.cpp"),
            Object(Matching, "plugProjectNishimuraU/BigTreasureMgr.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/BigTreasureShadow.cpp"),
            Object(Matching, "plugProjectNishimuraU/BigTreasure.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/KabutoState.cpp"),
            Object(Matching, "plugProjectNishimuraU/KabutoAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/KabutoMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/Kabuto.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/KumaKochappyState.cpp"),
            Object(Matching, "plugProjectNishimuraU/KumaKochappyAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/KumaKochappyMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/KumaKochappy.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/MiniHoudaiState.cpp"),
            Object(Matching, "plugProjectNishimuraU/MiniHoudaiAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/MiniHoudaiMgr.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/MiniHoudai.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/MiniHoudaiShotGun.cpp"),
            Object(Matching, "plugProjectNishimuraU/SokkuriState.cpp"),
            Object(Matching, "plugProjectNishimuraU/SokkuriAnimator.cpp"),
            Object(Matching, "plugProjectNishimuraU/SokkuriMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/Sokkuri.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/JointShadowBase.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/UmimushiShadow.cpp"),
            Object(Matching, "plugProjectNishimuraU/HanaMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/Hana.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/BigTreasureAttack.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/SnakeCrowShadow.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/SnakeWholeShadow.cpp"),
            Object(Matching, "plugProjectNishimuraU/BombOtakaraMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/BombOtakara.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/DangoMushiState.cpp"),
            Object(Matching, "plugProjectNishimuraU/DangoMushiMgr.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/DangoMushi.cpp"),
            Object(Matching, "plugProjectNishimuraU/GreenKabutoMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/GreenKabuto.cpp"),
            Object(Matching, "plugProjectNishimuraU/RedKabutoMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/RedKabuto.cpp"),
            Object(Matching, "plugProjectNishimuraU/FixKabutoMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/FixKabuto.cpp"),
            Object(Matching, "plugProjectNishimuraU/RandCapEnemyUnit.cpp"),
            Object(Matching, "plugProjectNishimuraU/NormMiniHoudaiMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/NormMiniHoudai.cpp"),
            Object(Matching, "plugProjectNishimuraU/FixMiniHoudaiMgr.cpp"),
            Object(Matching, "plugProjectNishimuraU/FixMiniHoudai.cpp"),
            Object(NonMatching, "plugProjectNishimuraU/TyreShadow.cpp"),
        ],
    },
    {
        "lib": "plugProjectOgawaU",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(Matching, "plugProjectOgawaU/ogScreen.cpp"),
            Object(Matching, "plugProjectOgawaU/ogDopingScreen.cpp"),
            Object(Matching, "plugProjectOgawaU/ogPikminCounter.cpp"),
            Object(Matching, "plugProjectOgawaU/ogAnime.cpp"),
            Object(Matching, "plugProjectOgawaU/ogCounter.cpp"),
            Object(Matching, "plugProjectOgawaU/ogLifeGauge.cpp"),
            Object(Matching, "plugProjectOgawaU/ogSunMeter.cpp"),
            Object(Matching, "plugProjectOgawaU/ogCatchPiki.cpp"),
            Object(Matching, "plugProjectOgawaU/ogContenaCounter.cpp"),
            Object(Matching, "plugProjectOgawaU/ogMapCounter.cpp"),
            Object(Matching, "plugProjectOgawaU/ogAnimTextScreen.cpp"),
            Object(Matching, "plugProjectOgawaU/ogCallBackMessage.cpp"),
            Object(NonMatching, "plugProjectOgawaU/ogMenuMgr.cpp"),
            Object(Matching, "plugProjectOgawaU/ogCallBackScreen.cpp"),
            Object(Matching, "plugProjectOgawaU/ogCounterRV.cpp"),
            Object(Matching, "plugProjectOgawaU/ogSE.cpp"),
            Object(Matching, "plugProjectOgawaU/ogCounterDay.cpp"),
            Object(Matching, "plugProjectOgawaU/testScene.cpp"),
            Object(Matching, "plugProjectOgawaU/testObj.cpp"),
            Object(Matching, "plugProjectOgawaU/ogSceneGround.cpp"),
            Object(Matching, "plugProjectOgawaU/ogObjGround.cpp"),
            Object(NonMatching, "plugProjectOgawaU/ogCopyPane.cpp"),
            Object(Matching, "plugProjectOgawaU/ogBloGroup.cpp"),
            Object(Matching, "plugProjectOgawaU/DispMemberBase.cpp"),
            Object(Matching, "plugProjectOgawaU/ogLib2D.cpp"),
            Object(NonMatching, "plugProjectOgawaU/ogObjSMenuMap.cpp"),
            Object(Matching, "plugProjectOgawaU/ogSceneSMenuMap.cpp"),
            Object(Matching, "plugProjectOgawaU/ogObjSMenuItem.cpp"),
            Object(Matching, "plugProjectOgawaU/ogSceneSMenuItem.cpp"),
            Object(Matching, "plugProjectOgawaU/ogSceneSMenuPause.cpp"),
            Object(Matching, "plugProjectOgawaU/ogObjSMenuPause.cpp"),
            Object(Matching, "plugProjectOgawaU/ogObjSMenuBase.cpp"),
            Object(Matching, "plugProjectOgawaU/ogObjCourseName.cpp"),
            Object(Matching, "plugProjectOgawaU/ogUtil.cpp"),
            Object(Matching, "plugProjectOgawaU/ogSceneCourseName.cpp"),
            Object(Matching, "plugProjectOgawaU/ogObjKantei.cpp"),
            Object(Matching, "plugProjectOgawaU/ogSceneKantei.cpp"),
            Object(Matching, "plugProjectOgawaU/ogObjSpecialItem.cpp"),
            Object(Matching, "plugProjectOgawaU/ogSceneSpecialItem.cpp"),
            Object(Matching, "plugProjectOgawaU/ogObjFloor.cpp"),
            Object(Matching, "plugProjectOgawaU/ogSceneFloor.cpp"),
            Object(Matching, "plugProjectOgawaU/ogObjCave.cpp"),
            Object(Matching, "plugProjectOgawaU/ogSceneCave.cpp"),
            Object(Matching, "plugProjectOgawaU/ogObjAnaDemo.cpp"),
            Object(Matching, "plugProjectOgawaU/ogSceneAnaDemo.cpp"),
            Object(Matching, "plugProjectOgawaU/ogObjChallenge2P.cpp"),
            Object(Matching, "plugProjectOgawaU/ogSceneChallenge2P.cpp"),
            Object(NonMatching, "plugProjectOgawaU/ogObjContena.cpp"),
            Object(Matching, "plugProjectOgawaU/ogSceneContena.cpp"),
            Object(Matching, "plugProjectOgawaU/ogObjSMenuPauseDoukutu.cpp"),
            Object(Matching, "plugProjectOgawaU/ogSceneSMenuPauseDoukutu.cpp"),
            Object(Matching, "plugProjectOgawaU/ogObjUfo.cpp"),
            Object(Matching, "plugProjectOgawaU/ogSceneUfo.cpp"),
            Object(Matching, "plugProjectOgawaU/ogObjSave.cpp"),
            Object(Matching, "plugProjectOgawaU/ogSceneSave.cpp"),
            Object(Matching, "plugProjectOgawaU/ogObjFinalMsg.cpp"),
            Object(Matching, "plugProjectOgawaU/ogSceneFinalMsg.cpp"),
            Object(NonMatching, "plugProjectOgawaU/ogObjVs.cpp"),
            Object(Matching, "plugProjectOgawaU/ogSceneVs.cpp"),
            Object(Matching, "plugProjectOgawaU/ogObjChallenge1P.cpp"),
            Object(Matching, "plugProjectOgawaU/ogSceneChallenge1P.cpp"),
            Object(Matching, "plugProjectOgawaU/ogScaleMgr.cpp"),
            Object(Matching, "plugProjectOgawaU/ogObjSMenuPauseVs.cpp"),
            Object(Matching, "plugProjectOgawaU/ogSceneSMenuPauseVs.cpp"),
            Object(Matching, "plugProjectOgawaU/ogFuriko.cpp"),
            Object(NonMatching, "plugProjectOgawaU/ogCounterSlot.cpp"),
            Object(Matching, "plugProjectOgawaU/ogObjWorldMapInfoWindow0.cpp"),
            Object(Matching, "plugProjectOgawaU/ogSceneWorldMapInfoWindow0.cpp"),
            Object(Matching, "plugProjectOgawaU/ogObjWorldMapInfoWindow1.cpp"),
            Object(Matching, "plugProjectOgawaU/ogSceneWorldMapInfoWindow1.cpp"),
            Object(Matching, "plugProjectOgawaU/ogTotalPokoScreen.cpp"),
            Object(Matching, "plugProjectOgawaU/ogObjChallengeBase.cpp"),
            Object(Matching, "plugProjectOgawaU/ogSceneChallengeBase.cpp"),
            Object(Matching, "plugProjectOgawaU/ogCallBackPicture.cpp"),
            Object(Matching, "plugProjectOgawaU/ogDrawAfter.cpp"),
            Object(Matching, "plugProjectOgawaU/ogOtakaraSensor.cpp"),
            Object(Matching, "plugProjectOgawaU/ogTitleMsg.cpp"),
            Object(Matching, "plugProjectOgawaU/ogAngleMgr.cpp"),
            Object(Matching, "plugProjectOgawaU/ogSceneSMenuCont.cpp"),
            Object(Matching, "plugProjectOgawaU/ogObjSMenuCont.cpp"),
        ],
    },
    {
        "lib": "plugProjectHikinoU",
        "cflags": [*cflags_pikmin, "-sym on"],
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(NonMatching, "plugProjectHikinoU/PSSeq.cpp"),
            Object(NonMatching, "plugProjectHikinoU/PSGame.cpp"),
            Object(Matching, "plugProjectHikinoU/PSSystemIF.cpp"),
            Object(NonMatching, "plugProjectHikinoU/PSAutoBgm.cpp"),
            Object(NonMatching, "plugProjectHikinoU/PSBnkMgr.cpp"),
            Object(Matching, "plugProjectHikinoU/PSTaskBase.cpp"),
            Object(Matching, "plugProjectHikinoU/PSBgmTask.cpp"),
            Object(Matching, "plugProjectHikinoU/PSCreatureMgr.cpp"),
            Object(Matching, "plugProjectHikinoU/PSAutoBgm_MeloArranger.cpp"),
            Object(NonMatching, "plugProjectHikinoU/PSSe.cpp"),
            Object(Matching, "plugProjectHikinoU/PSSeBase.cpp"),
            Object(Matching, "plugProjectHikinoU/PSScene.cpp"),
            Object(Matching, "plugProjectHikinoU/PSBgmTrack.cpp"),
            Object(Matching, "plugProjectHikinoU/PSDirector.cpp"),
        ],
    },
    {
        "lib": "plugProjectMorimuraU",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(NonMatching, "plugProjectMorimuraU/dayEndCount.cpp"),
            Object(NonMatching, "plugProjectMorimuraU/hurryUp2D.cpp"),
            Object(Matching, "plugProjectMorimuraU/gameOver2D.cpp"),
            Object(Matching, "plugProjectMorimuraU/testBase.cpp"),
            Object(Matching, "plugProjectMorimuraU/bombState.cpp"),
            Object(Matching, "plugProjectMorimuraU/bombAnimator.cpp"),
            Object(Matching, "plugProjectMorimuraU/bombMgr.cpp"),
            Object(Matching, "plugProjectMorimuraU/bomb.cpp"),
            Object(Matching, "plugProjectMorimuraU/eggState.cpp"),
            Object(Matching, "plugProjectMorimuraU/eggAnimator.cpp"),
            Object(Matching, "plugProjectMorimuraU/eggMgr.cpp"),
            Object(Matching, "plugProjectMorimuraU/egg.cpp"),
            Object(Matching, "plugProjectMorimuraU/panModokiState.cpp"),
            Object(Matching, "plugProjectMorimuraU/panModokiAnimator.cpp"),
            Object(Matching, "plugProjectMorimuraU/panModokiMgr.cpp"),
            Object(NonMatching, "plugProjectMorimuraU/panModoki.cpp"),
            Object(Matching, "plugProjectMorimuraU/plantsMgr.cpp"),
            Object(Matching, "plugProjectMorimuraU/plantsAnimator.cpp"),
            Object(Matching, "plugProjectMorimuraU/plants.cpp"),
            Object(NonMatching, "plugProjectMorimuraU/kingChappyState.cpp"),
            Object(Matching, "plugProjectMorimuraU/kingChappyMgr.cpp"),
            Object(NonMatching, "plugProjectMorimuraU/kingChappy.cpp"),
            Object(Matching, "plugProjectMorimuraU/miulinState.cpp"),
            Object(Matching, "plugProjectMorimuraU/miulinAnimator.cpp"),
            Object(Matching, "plugProjectMorimuraU/miulinMgr.cpp"),
            Object(NonMatching, "plugProjectMorimuraU/miulin.cpp"),
            Object(Matching, "plugProjectMorimuraU/jigumoState.cpp"),
            Object(Matching, "plugProjectMorimuraU/jigumoAnimator.cpp"),
            Object(Matching, "plugProjectMorimuraU/jigumoMgr.cpp"),
            Object(NonMatching, "plugProjectMorimuraU/jigumo.cpp"),
            Object(Matching, "plugProjectMorimuraU/enemyNestMgr.cpp"),
            Object(Matching, "plugProjectMorimuraU/enemyNest.cpp"),
            Object(Matching, "plugProjectMorimuraU/tamagoMushiState.cpp"),
            Object(Matching, "plugProjectMorimuraU/tamagoMushiAnimator.cpp"),
            Object(Matching, "plugProjectMorimuraU/tamagoMushiMgr.cpp"),
            Object(Matching, "plugProjectMorimuraU/tamagoMushi.cpp"),
            Object(NonMatching, "plugProjectMorimuraU/zukan2D.cpp"),
            Object(NonMatching, "plugProjectMorimuraU/hiScore2D.cpp"),
            Object(Matching, "plugProjectMorimuraU/umiMushiState.cpp"),
            Object(Matching, "plugProjectMorimuraU/umiMushiAnimator.cpp"),
            Object(Matching, "plugProjectMorimuraU/umiMushiMgr.cpp"),
            Object(NonMatching, "plugProjectMorimuraU/umiMushi.cpp"),
            Object(Matching, "plugProjectMorimuraU/shijimiChouState.cpp"),
            Object(Matching, "plugProjectMorimuraU/shijimiChouAnimator.cpp"),
            Object(Matching, "plugProjectMorimuraU/shijimiChouMgr.cpp"),
            Object(NonMatching, "plugProjectMorimuraU/shijimiChou.cpp"),
            Object(NonMatching, "plugProjectMorimuraU/challengeSelect2D.cpp"),
            Object(NonMatching, "plugProjectMorimuraU/challengeResult2D.cpp"),
            Object(NonMatching, "plugProjectMorimuraU/vsSelect2D.cpp"),
            Object(Matching, "plugProjectMorimuraU/mrUtil.cpp"),
            Object(NonMatching, "plugProjectMorimuraU/scrollList.cpp"),
            Object(Matching, "plugProjectMorimuraU/mrWindow.cpp"),
            Object(Matching, "plugProjectMorimuraU/blackManState.cpp"),
            Object(Matching, "plugProjectMorimuraU/blackManAnimator.cpp"),
            Object(Matching, "plugProjectMorimuraU/blackManMgr.cpp"),
            Object(NonMatching, "plugProjectMorimuraU/blackMan.cpp"),
            Object(Matching, "plugProjectMorimuraU/tyreState.cpp"),
            Object(Matching, "plugProjectMorimuraU/tyreAnimator.cpp"),
            Object(Matching, "plugProjectMorimuraU/tyreMgr.cpp"),
            Object(Matching, "plugProjectMorimuraU/tyre.cpp"),
        ],
    },
    {
        "lib": "plugProjectEbisawaU",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(NonMatching, "plugProjectEbisawaU/efxBase.cpp"),
            Object(NonMatching, "plugProjectEbisawaU/efxEnemy.cpp"),
            Object(Matching, "plugProjectEbisawaU/efxObject.cpp"),
            Object(NonMatching, "plugProjectEbisawaU/efxPikmin.cpp"),
            Object(Matching, "plugProjectEbisawaU/particle2dMgr.cpp"),
            Object(Matching, "plugProjectEbisawaU/efx2dBase.cpp"),
            Object(Matching, "plugProjectEbisawaU/efx2dEffect.cpp"),
            Object(NonMatching, "plugProjectEbisawaU/particleMgr.cpp"),
            Object(NonMatching, "plugProjectEbisawaU/ebiP2Title.cpp"),
            Object(Matching, "plugProjectEbisawaU/ebiP2TitleUnit.cpp"),
            Object(Matching, "plugProjectEbisawaU/ebiUtility.cpp"),
            Object(Matching, "plugProjectEbisawaU/ebiP2TitleCamera.cpp"),
            Object(Matching, "plugProjectEbisawaU/ebiP2TitleLight.cpp"),
            Object(Matching, "plugProjectEbisawaU/ebiScreenMemoryCard.cpp"),
            Object(Matching, "plugProjectEbisawaU/efxTPkEffectMgr.cpp"),
            Object(Matching, "plugProjectEbisawaU/efxEnemyGeneral.cpp"),
            Object(Matching, "plugProjectEbisawaU/ebi3DGraph.cpp"),
            Object(Matching, "plugProjectEbisawaU/ebiGeometry.cpp"),
            Object(Matching, "plugProjectEbisawaU/ebi2DGraph.cpp"),
            Object(Matching, "plugProjectEbisawaU/ebiScreenOption.cpp"),
            Object(Matching, "plugProjectEbisawaU/ebiScreenProgre.cpp"),
            Object(Matching, "plugProjectEbisawaU/ebiOptionMgr.cpp"),
            Object(NonMatching, "plugProjectEbisawaU/ebi2DCallBack.cpp"),
            Object(Matching, "plugProjectEbisawaU/ebiCardMgr.cpp"),
            Object(Matching, "plugProjectEbisawaU/ebiScreenFramework.cpp"),
            Object(Matching, "plugProjectEbisawaU/ebiScreenPushStart.cpp"),
            Object(
                NonMatching,
                "plugProjectEbisawaU/ebiScreenFileSelect.cpp",
                extra_cflags=["-sym on"],
            ),
            Object(Matching, "plugProjectEbisawaU/ebiScreenTitleMenu.cpp"),
            Object(Matching, "plugProjectEbisawaU/ebiSaveMgr.cpp"),
            Object(Matching, "plugProjectEbisawaU/ebiScreenSaveMenu.cpp"),
            Object(
                Matching,
                "plugProjectEbisawaU/ebiScreenFileSelect_Mgr.cpp",
                extra_cflags=["-sym on"],
            ),
            Object(Matching, "plugProjectEbisawaU/ebiFileSelectMgr.cpp"),
            Object(Matching, "plugProjectEbisawaU/ebiCardMgr_Load.cpp"),
            Object(Matching, "plugProjectEbisawaU/ebiP2TitleCoordMgr.cpp"),
            Object(
                NonMatching,
                "plugProjectEbisawaU/ebiP2TitlePikmin.cpp",
                extra_cflags=["-sym on"],
            ),
            Object(
                Matching,
                "plugProjectEbisawaU/ebiP2TitleKogane.cpp",
                extra_cflags=["-sym on"],
            ),
            Object(
                Matching,
                "plugProjectEbisawaU/ebiP2TitleChappy.cpp",
                extra_cflags=["-sym on"],
            ),
            Object(
                Matching,
                "plugProjectEbisawaU/ebiScreenTMBack.cpp",
                extra_cflags=["-sym on"],
            ),
            Object(
                NonMatching,
                "plugProjectEbisawaU/ebiMainTitleMgr.cpp",
                extra_cflags=["-sym on"],
            ),
            Object(Matching, "plugProjectEbisawaU/ebiP2TitleFog.cpp"),
            Object(
                NonMatching,
                "plugProjectEbisawaU/efxEnemyBoss.cpp",
                extra_cflags=["-sym on"],
            ),
            Object(Matching, "plugProjectEbisawaU/ebiCardEReader.cpp"),
            Object(
                NonMatching,
                "plugProjectEbisawaU/ebiScreenOmake.cpp",
                extra_cflags=["-sym on"],
            ),
            Object(
                NonMatching,
                "plugProjectEbisawaU/ebiOmakeMgr.cpp",
                extra_cflags=["-sym on"],
            ),
            Object(
                Matching,
                "plugProjectEbisawaU/ebiScreenOmakeCardE.cpp",
                extra_cflags=["-sym on"],
            ),
            Object(
                Matching,
                "plugProjectEbisawaU/ebiScreenOmakeGame.cpp",
                extra_cflags=["-sym on"],
            ),
            Object(
                Matching,
                "plugProjectEbisawaU/ebiScreenInfoWindow.cpp",
                extra_cflags=["-sym on"],
            ),
        ],
    },
    {
        "lib": "plugProjectKonoU",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(NonMatching, "plugProjectKonoU/khWorldMap.cpp"),
            Object(Matching, "plugProjectKonoU/khCaveResult.cpp"),
            Object(Matching, "plugProjectKonoU/khSceneLoader.cpp"),
            Object(Matching, "plugProjectKonoU/newGame2DMgr.cpp"),
            Object(Matching, "plugProjectKonoU/newScreenMgr.cpp"),
            Object(Matching, "plugProjectKonoU/khReadyGo.cpp"),
            Object(Matching, "plugProjectKonoU/khFinalFloor.cpp"),
            Object(NonMatching, "plugProjectKonoU/khDayEndResult.cpp"),
            Object(Matching, "plugProjectKonoU/khUtil.cpp"),
            Object(NonMatching, "plugProjectKonoU/khFinalResult.cpp"),
            Object(Matching, "plugProjectKonoU/khPayDept.cpp"),
            Object(Matching, "plugProjectKonoU/khWinLose.cpp"),
            Object(Matching, "plugProjectKonoU/khWinLoseReason.cpp"),
            Object(Matching, "plugProjectKonoU/khMailSaveData.cpp"),
        ],
    },
    {
        "lib": "sysBootupU",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [Object(Matching, "sysBootupU/sysBootup.cpp")],
    },
    {
        "lib": "sysCommonU",
        "cflags": cflags_pikmin,
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(Matching, "sysCommonU/node.cpp"),
            Object(NonMatching, "sysCommonU/sysMath.cpp"),
            Object(Matching, "sysCommonU/id32.cpp"),
            Object(Matching, "sysCommonU/parameters.cpp"),
            Object(Matching, "sysCommonU/stream.cpp"),
            Object(NonMatching, "sysCommonU/geometry.cpp"),
            Object(Matching, "sysCommonU/mapCollision.cpp"),
            Object(NonMatching, "sysCommonU/camera.cpp"),
            Object(Matching, "sysCommonU/tagparams.cpp"),
            Object(Matching, "sysCommonU/sysTemplates.cpp"),
            Object(Matching, "sysCommonU/mapCode.cpp"),
            Object(NonMatching, "sysCommonU/geomIntersection.cpp"),
            Object(NonMatching, "sysCommonU/geomOBBTree.cpp"),
            Object(NonMatching, "sysCommonU/geomTraceMove.cpp"),
            Object(Matching, "sysCommonU/geomCylinder.cpp"),
            Object(Matching, "sysCommonU/geomClone.cpp"),
        ],
    },
    {
        "lib": "sysGCU",
        "cflags": [*cflags_pikmin, "-lang=c++"],
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(NonMatching, "sysGCU/system.cpp"),
            Object(NonMatching, "sysGCU/section.cpp"),
            Object(Matching, "sysGCU/gameflow.cpp"),
            Object(Matching, "sysGCU/menuSection.cpp"),
            Object(Matching, "sysGCU/dvdThread.cpp"),
            Object(Matching, "sysGCU/appThread.cpp"),
            Object(Matching, "sysGCU/controller.cpp"),
            Object(NonMatching, "sysGCU/graphics.cpp"),
            Object(NonMatching, "sysGCU/matMath.cpp"),
            Object(Matching, "sysGCU/sysShape.cpp"),
            Object(Matching, "sysGCU/reset.cpp"),
            Object(Matching, "sysGCU/dvdStatus.cpp"),
            Object(Matching, "sysGCU/sysTimers.cpp"),
            Object(Matching, "sysGCU/modelMgr.cpp"),
            Object(Matching, "sysGCU/heapStatus.cpp"),
            Object(Matching, "sysGCU/light.cpp"),
            Object(Matching, "sysGCU/wipe.cpp"),
            Object(NonMatching, "sysGCU/moviePlayer.cpp"),
            Object(NonMatching, "sysGCU/JSTObjectActor.cpp"),
            Object(NonMatching, "sysGCU/JSTObjectCamera.cpp"),
            Object(NonMatching, "sysGCU/JSTObjectGameActor.cpp"),
            Object(NonMatching, "sysGCU/JSTObjectSystem.cpp"),
            Object(Matching, "sysGCU/JSTFindCreature.cpp"),
            Object(Matching, "sysGCU/movieConfig.cpp"),
            Object(Matching, "sysGCU/gameConfig.cpp"),
            Object(Matching, "sysGCU/fogMgr.cpp"),
            Object(NonMatching, "sysGCU/aramMgr.cpp"),
            Object(Matching, "sysGCU/resourceMgr.cpp"),
            Object(Matching, "sysGCU/resourceMgr2D.cpp"),
            Object(Matching, "sysGCU/sysMaterialAnim.cpp"),
            Object(Matching, "sysGCU/P2DScreen.cpp"),
            Object(NonMatching, "sysGCU/movieMessage.cpp"),
            Object(Matching, "sysGCU/moviePlayerPauseAndDraw.cpp"),
            Object(Matching, "sysGCU/JSTObjectSpecialActor.cpp"),
            Object(Matching, "sysGCU/messageSequence.cpp"),
            Object(Matching, "sysGCU/messageMgr.cpp"),
            Object(Matching, "sysGCU/messageObj.cpp"),
            Object(NonMatching, "sysGCU/messageRendering.cpp"),
            Object(Matching, "sysGCU/message.cpp"),
            Object(Matching, "sysGCU/modelEffect.cpp"),
            Object(Matching, "sysGCU/messageReference.cpp"),
            Object(Matching, "sysGCU/simpleMessage.cpp"),
            Object(Matching, "sysGCU/sysShapeAnimation.cpp"),
            Object(NonMatching, "sysGCU/sysShapeModel.cpp"),
            Object(NonMatching, "sysGCU/windowMessage.cpp"),
            Object(NonMatching, "sysGCU/memoryCard.cpp"),
            Object(NonMatching, "sysGCU/pikmin2MemoryCardMgr.cpp"),
            Object(Matching, "sysGCU/commonSaveData.cpp"),
            Object(NonMatching, "sysGCU/bootSection.cpp"),
            Object(NonMatching, "sysGCU/titleSection.cpp"),
            Object(Matching, "sysGCU/loadResource.cpp"),
            Object(Matching, "sysGCU/rootMenuSection.cpp"),
            Object(Matching, "sysGCU/demoSection.cpp"),
            Object(Matching, "sysGCU/THPAudioDecode.c"),
            Object(Matching, "sysGCU/THPDraw.c"),
            Object(Matching, "sysGCU/THPPlayer.c"),
            Object(Matching, "sysGCU/THPRead.c"),
            Object(Matching, "sysGCU/THPVideoDecode.c"),
            Object(Matching, "sysGCU/pikmin2THPPlayer.cpp"),
            Object(Matching, "sysGCU/captionMgr.cpp"),
            Object(Matching, "sysGCU/captionMessage.cpp"),
            Object(Matching, "sysGCU/screenScene.cpp"),
            Object(Matching, "sysGCU/screenMgr.cpp"),
            Object(Matching, "sysGCU/screenObj.cpp"),
            Object(Matching, "sysGCU/JSTObjectParticleActor.cpp"),
            Object(Matching, "sysGCU/moviePlayerAudio.cpp"),
            Object(Matching, "sysGCU/illustratedBookMessage.cpp"),
            Object(Matching, "sysGCU/sysDrawBuffer.cpp"),
            Object(Matching, "sysGCU/dvdErrorMessage.cpp"),
            Object(Matching, "sysGCU/pikmin2AramMgr.cpp"),
            Object(Matching, "sysGCU/messageAnalyzer.cpp"),
        ],
    },
    {
        "lib": "utilityU",
        "cflags": [*cflags_pikmin, "-sym on"],
        "mw_version": "GC/2.6",
        "host": True,
        "objects": [
            Object(Matching, "utilityU/menu.cpp"),
            Object(NonMatching, "utilityU/PSMainSide_Director.cpp"),
            Object(NonMatching, "utilityU/PSMainSide_Factory.cpp"),
            Object(NonMatching, "utilityU/PSMainSide_ObjSound.cpp"),
            Object(Matching, "utilityU/PSMainSide_Demo.cpp"),
            Object(NonMatching, "utilityU/PSMainSide_Scene.cpp"),
            Object(Matching, "utilityU/PSMainSide_BossMgr.cpp"),
            Object(NonMatching, "utilityU/PSMainSide_Se.cpp"),
            Object(Matching, "utilityU/PSMainSide_DirectorMgr.cpp"),
            Object(NonMatching, "utilityU/PSMainSide_Sound.cpp"),
            Object(Matching, "utilityU/PSMainSide_TrackMap.cpp"),
            Object(NonMatching, "utilityU/PSMainSide_CreaturePrm.cpp"),
            Object(NonMatching, "utilityU/PSMainSide_ObjCalc.cpp"),
        ],
    },
]

if args.mode == "configure":
    # Write build.ninja and objdiff.json
    generate_build(config)
elif args.mode == "progress":
    # Print progress and write progress.json
    calculate_progress(config)
else:
    sys.exit("Unknown mode: " + args.mode)
