#!/usr/bin/env python3

import argparse
import os
import shutil
import sys
import subprocess
from pathlib import Path

# assuming that this util is in tools/
script_dir = os.path.dirname(os.path.realpath(__file__))
root_dir = os.path.abspath(os.path.join(script_dir, ".."))

# Project-specific
CPP_FLAGS = [
    "-undef",
    "-D__sgi",
    "-DVERSION_US",
    "-DTARGET_N64",
    "-D_LANGUAGE_C",
    "-DF3DEX_GBI_2",
    "-D_MIPS_SZLONG=32",
    "-ffreestanding",
    "-nostdinc",
    "-Iinclude/ido",
    "-Iinclude",
    "-Ibuild",
    "-Iinclude/PR",
    "-Isrc",
    "-Isrc/libnaudio",
    "-I.",
    "-DM2C",
    "-Ilib/ultralib/include",
    "-Ilib/ultralib/include/PR",
    "-Ilib/ultralib/include/ido",
]

import os
import subprocess
import shutil

def import_c_file(in_file) -> str:

    Message = '/**** Context file for Pokemon Stadium *****/\n'

    in_file = os.path.relpath(in_file, root_dir)
    # Prefer clang as C preprocessor if installed on the system
    if shutil.which('clang') is not None:
        cpp = ['clang', '-E', '-P', '-x', 'c', '-Wno-trigraphs']
    else:
        cpp = ['gcc', '-E', '-P']

    cpp_getdefines = [*cpp, "-dM", *CPP_FLAGS, in_file]
    cpp_procfile = [*cpp, *CPP_FLAGS, in_file]

    out_text = ""
    try:
        out_text += subprocess.check_output(cpp_getdefines, cwd=root_dir, encoding="utf-8")
        out_text += subprocess.check_output(cpp_procfile, cwd=root_dir, encoding="utf-8")
    except subprocess.CalledProcessError:
        print(
            "Failed to preprocess input file, when running command:\n",
            cpp_getdefines,
            file=sys.stderr,
            )
        sys.exit(1)

    if not out_text:
        print("Output is empty - aborting")
        sys.exit(1)
    return Message + out_text

def main():
    parser = argparse.ArgumentParser(
        description="""Create a context file which can be used for mips_to_c"""
    )
    parser.add_argument(
        "c_file",
        help="""File from which to create context""",
    )
    args = parser.parse_args()

    output = import_c_file(args.c_file)

    with open(os.path.join(root_dir, "ctx.c"), "w", encoding="UTF-8") as f:
        f.write(output)


if __name__ == "__main__":
    main()
