//===-- DeclTypeRecordNodes.def - Serialization Metaprogramming -*- C++ -*-===//
//
// This source file is part of the Swift.org open source project
//
// Copyright (c) 2014 - 2017 Apple Inc. and the Swift project authors
// Licensed under Apache License v2.0 with Runtime Library Exception
//
// See https://swift.org/LICENSE.txt for license information
// See https://swift.org/CONTRIBUTORS.txt for the list of Swift project authors
//
//===----------------------------------------------------------------------===//
//
/// \file
/// Defines macros used for metaprogramming with the serialized records
/// in the "decls-and-types" block of serialized ASTs.
///
/// You should define one of the following sets of macros:
/// - RECORD(Id)
/// - RECORD(Id) and RECORD_VAL(Id)
/// - Any of TYPE(Id), DECL(Id), PATTERN(Id)
//
//===----------------------------------------------------------------------===//

#ifndef RECORD
#define RECORD(Id)
#endif

#ifdef RECORD_VAL
#define DECLTYPERECORDNODES_HAS_RECORD_VAL 1
#else
#define DECLTYPERECORDNODES_HAS_RECORD_VAL 0
#define RECORD_VAL(Id, Value) RECORD(Id)
#endif

#ifndef TYPE
#define TYPE(Id) RECORD(Id##_TYPE)
#endif

#ifndef FIRST_TYPE
# if DECLTYPERECORDNODES_HAS_RECORD_VAL
#  define FIRST_TYPE(Id, Value) RECORD_VAL(Id##_TYPE, Value)
# else
#  define FIRST_TYPE(Id, Value) TYPE(Id)
# endif
#endif

#ifndef DECL
#define DECL(Id) RECORD(Id##_DECL)
#endif

#ifndef FIRST_DECL
# if DECLTYPERECORDNODES_HAS_RECORD_VAL
#  define FIRST_DECL(Id, Value) RECORD_VAL(Id##_DECL, Value)
# else
#  define FIRST_DECL(Id, Value) DECL(Id)
# endif
#endif

#ifndef PATTERN
#define PATTERN(Id) RECORD(Id##_PATTERN)
#endif

#ifndef FIRST_PATTERN
# if DECLTYPERECORDNODES_HAS_RECORD_VAL
#  define FIRST_PATTERN(Id, Value) RECORD_VAL(Id##_PATTERN, Value)
# else
#  define FIRST_PATTERN(Id, Value) PATTERN(Id)
# endif
#endif

#ifndef TRAILING_INFO
#define TRAILING_INFO(Id) RECORD(Id)
#endif

#ifndef OTHER
#define OTHER(Id, Value) RECORD_VAL(Id, Value)
#endif

// These IDs must \em not be renumbered or reordered without incrementing
// the version in ModuleFormat.h. Names, however, may change.
FIRST_TYPE(BUILTIN_ALIAS, 1)
TYPE(GENERIC_TYPE_PARAM)
TYPE(DEPENDENT_MEMBER)
TYPE(NOMINAL)
TYPE(PAREN)
TYPE(TUPLE)
TRAILING_INFO(TUPLE_TYPE_ELT)
TYPE(FUNCTION)
TRAILING_INFO(FUNCTION_PARAM)
TYPE(METATYPE)
TYPE(PRIMARY_ARCHETYPE)
TYPE(OPENED_ARCHETYPE)
TYPE(OPAQUE_ARCHETYPE)
TYPE(NESTED_ARCHETYPE)
TYPE(INTERFACE_COMPOSITION)
TYPE(BOUND_GENERIC)

TYPE(GENERIC_FUNCTION)
TYPE(ARRAY_SLICE)
TYPE(DICTIONARY)
TYPE(REFERENCE_STORAGE)
TYPE(UNBOUND_GENERIC)
TYPE(OPTIONAL)
TYPE(SIL_FUNCTION)
TYPE(DYNAMIC_SELF)
TYPE(OPENED_EXISTENTIAL)
TYPE(EXISTENTIAL_METATYPE)
TYPE(SIL_BLOCK_STORAGE)
TYPE(SIL_BOX)
TYPE(NAME_ALIAS)

FIRST_DECL(TYPE_ALIAS, 60)
DECL(GENERIC_TYPE_PARAM)
DECL(ASSOCIATED_TYPE)
DECL(STRUCT)
DECL(CONSTRUCTOR)
DECL(VAR)
DECL(PARAM)
DECL(FUNC)
DECL(OPAQUE_TYPE)
DECL(PATTERN_BINDING)
DECL(INTERFACE)
TRAILING_INFO(DEFAULT_WITNESS_TABLE)
DECL(PREFIX_OPERATOR)
DECL(POSTFIX_OPERATOR)
DECL(INFIX_OPERATOR)
DECL(CLASS)
DECL(ENUM)
DECL(ENUM_ELEMENT)
DECL(SUBSCRIPT)
DECL(EXTENSION)
DECL(DESTRUCTOR)
DECL(PRECEDENCE_GROUP)
DECL(ACCESSOR)

#ifndef DECL_ATTR
#define DECL_ATTR(NAME, CLASS, OPTIONS, CODE) RECORD_VAL(CLASS##_DECL_ATTR, 100+CODE)
#endif
#include "swift/AST/Attr.def"

FIRST_PATTERN(PAREN, 200)
PATTERN(TUPLE)
TRAILING_INFO(TUPLE_PATTERN_ELT)
PATTERN(NAMED)
PATTERN(ANY)
PATTERN(TYPED)
PATTERN(VAR)

OTHER(PARAMETERLIST, 210)
// 211 is unused
OTHER(FOREIGN_ERROR_CONVENTION, 212)
// 213 is unused
OTHER(XREF_TYPE_PATH_PIECE, 214)
OTHER(XREF_VALUE_PATH_PIECE, 215)
OTHER(XREF_EXTENSION_PATH_PIECE, 216)
OTHER(XREF_OPERATOR_OR_ACCESSOR_PATH_PIECE, 217)
OTHER(XREF_GENERIC_PARAM_PATH_PIECE, 218)
OTHER(XREF_INITIALIZER_PATH_PIECE, 219)

OTHER(ABSTRACT_CLOSURE_EXPR_CONTEXT, 220)
OTHER(PATTERN_BINDING_INITIALIZER_CONTEXT, 221)
OTHER(DEFAULT_ARGUMENT_INITIALIZER_CONTEXT, 222)
OTHER(TOP_LEVEL_CODE_DECL_CONTEXT, 223)

OTHER(GENERIC_PARAM_LIST, 230)
OTHER(GENERIC_SIGNATURE, 231)
TRAILING_INFO(GENERIC_REQUIREMENT)
TRAILING_INFO(LAYOUT_REQUIREMENT)
// 234 is unused
OTHER(SIL_GENERIC_SIGNATURE, 235)
OTHER(SUBSTITUTION_MAP, 236)

OTHER(LOCAL_DISCRIMINATOR, 237)
OTHER(PRIVATE_DISCRIMINATOR, 238)
OTHER(FILENAME_FOR_PRIVATE, 239)

OTHER(ABSTRACT_INTERFACE_CONFORMANCE, 240)
OTHER(NORMAL_INTERFACE_CONFORMANCE, 241)
OTHER(SPECIALIZED_INTERFACE_CONFORMANCE, 242)
OTHER(INHERITED_INTERFACE_CONFORMANCE, 243)
OTHER(INVALID_INTERFACE_CONFORMANCE, 244)

OTHER(SIL_LAYOUT, 245)
OTHER(NORMAL_INTERFACE_CONFORMANCE_ID, 246)
OTHER(INTERFACE_CONFORMANCE_XREF, 247)
OTHER(MEMBERS, 248)
OTHER(XREF, 249)
OTHER(INLINABLE_BODY_TEXT, 250)
OTHER(SELF_INTERFACE_CONFORMANCE, 251)

OTHER(XREF_OPAQUE_RETURN_TYPE_PATH_PIECE, 252)

#undef RECORD
#undef DECLTYPERECORDNODES_HAS_RECORD_VAL
#undef RECORD_VAL
#undef TYPE
#undef FIRST_TYPE
#undef DECL
#undef FIRST_DECL
#undef PATTERN
#undef FIRST_PATTERN
#undef TRAILING_INFO
#undef OTHER
