//===--- Enum.h - Runtime declarations for enums ----------------*- C++ -*-===//
//
// This source file is part of the Swift.org open source project
//
// Copyright (c) 2014 - 2017 Apple Inc. and the Swift project authors
// Licensed under Apache License v2.0 with Runtime Library Exception
//
// See https://swift.org/LICENSE.txt for license information
// See https://swift.org/CONTRIBUTORS.txt for the list of Swift project authors
//
//===----------------------------------------------------------------------===//
//
// Swift runtime functions in support of enums.
//
//===----------------------------------------------------------------------===//

#ifndef POLARPHP_RUNTIME_ENUM_H
#define POLARPHP_RUNTIME_ENUM_H

#include "polarphp/runtime/Config.h"

namespace polar::runtime {

struct OpaqueValue;
struct InProcess;

template <typename Runtime> struct TargetValueWitnessTable;
using ValueWitnessTable = TargetValueWitnessTable<InProcess>;

template <typename Runtime> struct TargetMetadata;
using Metadata = TargetMetadata<InProcess>;

template <typename Runtime> struct TargetEnumMetadata;
using EnumMetadata = TargetEnumMetadata<InProcess>;
struct TypeLayout;

/// Initialize the type metadata for a single-case enum type.
///
/// \param enumType - pointer to the instantiated but uninitialized metadata
///                   for the enum.
/// \param flags - flags controlling the layout
/// \param payload - type metadata for the payload of the enum.
POLAR_RUNTIME_EXPORT
void polarphp_initEnumMetadataSingleCase(EnumMetadata *enumType,
                                         EnumLayoutFlags flags,
                                         const TypeLayout *payload);

/// Initialize the type metadata for a single-payload enum type.
///
/// \param enumType - pointer to the instantiated but uninitialized metadata
///                   for the enum.
/// \param flags - flags controlling the layout
/// \param payload - type metadata for the payload case of the enum.
/// \param emptyCases - the number of empty cases in the enum.
POLAR_RUNTIME_EXPORT
void polarphp_initEnumMetadataSinglePayload(EnumMetadata *enumType,
                                            EnumLayoutFlags flags,
                                            const TypeLayout *payload,
                                            unsigned emptyCases);

using getExtraInhabitantTag_t =
SWIFT_CC(swift) unsigned (const OpaqueValue *vaue,
unsigned numExtraInhabitants,
const Metadata *payloadType);

/// Implement getEnumTagSinglePayload generically in terms of a
/// payload type with a getExtraInhabitantIndex function.
///
/// \param value - pointer to the enum value.
/// \param payloadType - type metadata for the payload case of the enum.
/// \param emptyCases - the number of empty cases in the enum.
///
/// \returns 0 if the payload case is inhabited. If an empty case is inhabited,
///          returns a value greater than or equal to one and less than or equal
///          emptyCases.
POLAR_RUNTIME_EXPORT SWIFT_CC(swift)
unsigned polarphp_getEnumTagSinglePayloadGeneric(const OpaqueValue *value,
                                                 unsigned emptyCases,
                                                 const Metadata *payloadType,
                                                 getExtraInhabitantTag_t *getTag);

using storeExtraInhabitantTag_t =
SWIFT_CC(swift) void (OpaqueValue *value,
unsigned whichCase,
unsigned numExtraInhabitants,
const Metadata *payloadType);

/// Implement storeEnumTagSinglePayload generically in terms of a
/// payload type with a storeExtraInhabitant function.
///
/// \param value - pointer to the enum value. If the case being initialized is
///                the payload case (0), then the payload should be
///                initialized.
/// \param payloadType - type metadata for the payload case of the enum.
/// \param whichCase - unique value identifying the case. 0 for the payload
///                    case, or a value greater than or equal to one and less
///                    than or equal emptyCases for an empty case.
/// \param emptyCases - the number of empty cases in the enum.
POLAR_RUNTIME_EXPORT SWIFT_CC(swift)
void polarphp_storeEnumTagSinglePayloadGeneric(OpaqueValue *value,
                                               unsigned whichCase,
unsigned emptyCases,
const Metadata *payloadType,
   storeExtraInhabitantTag_t *storeTag);

/// Initialize the type metadata for a generic, multi-payload
///        enum instance.
POLAR_RUNTIME_EXPORT
void polarphp_initEnumMetadataMultiPayload(EnumMetadata *enumType,
                                           EnumLayoutFlags flags,
                                           unsigned numPayloads,
                                           const TypeLayout * const *payloadTypes);

/// Return an integer value representing which case of a multi-payload
///        enum is inhabited.
///
/// \param value - pointer to the enum value.
/// \param enumType - type metadata for the enum.
///
/// \returns The index of the enum case.
POLAR_RUNTIME_EXPORT
unsigned polarphp_getEnumCaseMultiPayload(const OpaqueValue *value,
                                          const EnumMetadata *enumType);

/// Store the tag value for the given case into a multi-payload enum,
///        whose associated payload (if any) has already been initialized.
POLAR_RUNTIME_EXPORT
void polarphp_storeEnumTagMultiPayload(OpaqueValue *value, const EnumMetadata *enumType, unsigned whichCase);

} // polar::runtime

#endif
