// Copyright 2024 PingCAP, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package rule

import (
	"github.com/pingcap/tidb/pkg/planner/cascades/pattern"
	"github.com/pingcap/tidb/pkg/planner/cascades/util"
	"github.com/pingcap/tidb/pkg/planner/core/base"
)

var _ Rule = &BaseRule{}

// Rule regular the common rule interface for the memo optimization.
type Rule interface {
	// ID return the id of this rule.
	ID() uint

	// String implements the fmt.Stringer interface, used for rule tracing process.
	String(writer util.StrBufferWriter)

	// Pattern return the initialized pattern of a specific rule when it created.
	Pattern() *pattern.Pattern

	// PreCheck check the admission of the group expression output from binder.
	PreCheck(base.LogicalPlan) bool

	// Match checks whether the GroupExpr satisfies all the requirements of the specific rule.
	// The pattern only identifies the operator type, some transformation rules also need
	// detailed information for certain plan operators to decide whether it is applicable.
	Match(base.LogicalPlan) bool

	// XForm does the real work of the optimization rule. The returned group expressions list
	// indicates the new GroupExprs generated by the transformation rule.
	XForm(_ base.LogicalPlan) ([]base.LogicalPlan, error)
}

// BaseRule is the abstract parent class of rule.
type BaseRule struct {
	tp      ruleType
	pattern *pattern.Pattern
}

// NewBaseRule creates a new BaseRule.
func NewBaseRule(tp ruleType, pattern *pattern.Pattern) *BaseRule {
	return &BaseRule{
		tp:      tp,
		pattern: pattern,
	}
}

// ID returns the basic ordinal this rule enum type.
func (*BaseRule) ID() uint {
	// impl it in your own rule.
	return 0
}

// String implements Rule interface
func (r *BaseRule) String(writer util.StrBufferWriter) {
	writer.WriteString(r.tp.String())
}

// Pattern implements Rule interface.
func (r *BaseRule) Pattern() *pattern.Pattern {
	return r.pattern
}

// PreCheck implements the Rule interface.
func (*BaseRule) PreCheck(_ base.LogicalPlan) bool {
	return true
}

// Match implements Rule interface.
func (*BaseRule) Match(_ base.LogicalPlan) bool {
	return true
}

// XForm implements Rule interface.
func (*BaseRule) XForm(_ base.LogicalPlan) ([]base.LogicalPlan, error) {
	return nil, nil
}
