// Copyright 2020 PingCAP, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package importer

import (
	"strings"

	"github.com/jedib0t/go-pretty/v6/table"
	"github.com/jedib0t/go-pretty/v6/text"
	"github.com/pingcap/tidb/lightning/pkg/precheck"
)

// Template is the interface for lightning check.
type Template interface {
	// Collect mainly collect performance related checks' results and critical level checks' results.
	// If the performance is not as expect or one of critical check not passed. it will stop import task.
	Collect(t precheck.CheckType, passed bool, msg string)

	// Success represents the whole check has passed or not.
	Success() bool

	// FailedCount represents (the warn check failed count, the critical check failed count)
	FailedCount(t precheck.CheckType) int

	// Output print all checks results.
	Output() string

	// FailedMsg represents the error msg for the failed check.
	FailedMsg() string
}

// SimpleTemplate is a simple template for lightning check.
type SimpleTemplate struct {
	count int
	// export them for test
	warnFailedCount     int
	criticalFailedCount int
	normalMsgs          []string // only used in unit test now
	criticalMsgs        []string
	t                   table.Writer
}

// NewSimpleTemplate returns a simple template.
func NewSimpleTemplate() Template {
	t := table.NewWriter()
	t.AppendHeader(table.Row{"#", "Check Item", "Type", "Passed"})
	t.SetColumnConfigs([]table.ColumnConfig{
		{Name: "#", WidthMax: 6},
		{Name: "Check Item", WidthMax: 130},
		{Name: "Type", WidthMax: 20},
		{Name: "Passed", WidthMax: 6},
	})
	return &SimpleTemplate{
		t: t,
	}
}

// FailedMsg returns the error msg for the failed check.
func (c *SimpleTemplate) FailedMsg() string {
	return strings.Join(c.criticalMsgs, ";\n")
}

// Collect mainly collect performance related checks' results and critical level checks' results.
func (c *SimpleTemplate) Collect(t precheck.CheckType, passed bool, msg string) {
	c.count++
	if !passed {
		switch t {
		case precheck.Critical:
			c.criticalFailedCount++
		case precheck.Warn:
			c.warnFailedCount++
		}
	}
	if !passed && t == precheck.Critical {
		c.criticalMsgs = append(c.criticalMsgs, msg)
	} else {
		c.normalMsgs = append(c.normalMsgs, msg)
	}
	c.t.AppendRow(table.Row{c.count, msg, t, passed})
	c.t.AppendSeparator()
}

// Success represents the whole check has passed or not.
func (c *SimpleTemplate) Success() bool {
	return c.criticalFailedCount == 0
}

// FailedCount represents (the warn check failed count, the critical check failed count)
func (c *SimpleTemplate) FailedCount(t precheck.CheckType) int {
	if t == precheck.Warn {
		return c.warnFailedCount
	}
	if t == precheck.Critical {
		return c.criticalFailedCount
	}
	return 0
}

// Output print all checks results.
func (c *SimpleTemplate) Output() string {
	c.t.SetAllowedRowLength(170)
	c.t.SetRowPainter(func(row table.Row) text.Colors {
		if passed, ok := row[3].(bool); ok {
			if !passed {
				if typ, ok := row[2].(precheck.CheckType); ok {
					if typ == precheck.Warn {
						return text.Colors{text.FgYellow}
					}
					if typ == precheck.Critical {
						return text.Colors{text.FgRed}
					}
				}
			}
		}
		return nil
	})
	return c.t.Render() + "\n"
}
