// Copyright 2023 PingCAP, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package lll

import (
	"bufio"
	"fmt"
	"go/token"
	"os"
	"strings"
	"unicode/utf8"

	"github.com/pingcap/tidb/build/linter/util"
	"golang.org/x/tools/go/analysis"
)

const lllName = "lll"

const goCommentDirectivePrefix = "//go:"

type settings struct {
	// LineLength is the maximum line length.
	LineLength int
	// TabWidth is the width of a tab character.
	TabWidth int
}

// Analyzer is the analyzer struct of lll.
var Analyzer = &analysis.Analyzer{
	Name: lllName,
	Doc:  "Reports long lines",
	Run: func(pass *analysis.Pass) (any, error) {
		err := runLll(pass, &settings{
			LineLength: 120,
			TabWidth:   1,
		})
		if err != nil {
			return nil, err
		}

		return nil, nil
	},
}

type result struct {
	Filename string
	Line     int
	Text     string
}

func runLll(pass *analysis.Pass, settings *settings) error {
	fileNames := make([]string, 0, len(pass.Files))
	for _, f := range pass.Files {
		pos := pass.Fset.PositionFor(f.Pos(), false)
		if pos.Filename != "" && !strings.HasSuffix(pos.Filename, "failpoint_binding__.go") {
			fileNames = append(fileNames, pos.Filename)
		}
	}

	spaces := strings.Repeat(" ", settings.TabWidth)

	for _, f := range fileNames {
		lintIssues, err := getLLLIssuesForFile(f, settings.LineLength, spaces)
		if err != nil {
			return err
		}

		for _, i := range lintIssues {
			fileContent, tf, err := util.ReadFile(pass.Fset, i.Filename)
			if err != nil {
				return fmt.Errorf("can't get file %s contents: %s", i.Filename, err)
			}
			pass.Reportf(token.Pos(tf.Base()+util.FindOffset(string(fileContent), i.Line, 1)), "too long")
		}
	}

	return nil
}

func getLLLIssuesForFile(filename string, maxLineLen int, tabSpaces string) ([]result, error) {
	var res []result

	//nolint: gosec
	f, err := os.Open(filename)
	if err != nil {
		return nil, fmt.Errorf("can't open file %s: %s", filename, err)
	}
	defer f.Close()

	lineNumber := 0
	multiImportEnabled := false

	scanner := bufio.NewScanner(f)
	for scanner.Scan() {
		lineNumber++

		line := scanner.Text()
		line = strings.ReplaceAll(line, "\t", tabSpaces)

		if strings.HasPrefix(line, goCommentDirectivePrefix) {
			continue
		}

		if strings.HasPrefix(line, "import") {
			multiImportEnabled = strings.HasSuffix(line, "(")
			continue
		}

		if multiImportEnabled {
			if line == ")" {
				multiImportEnabled = false
			}

			continue
		}

		lineLen := utf8.RuneCountInString(line)
		if lineLen > maxLineLen {
			res = append(res, result{
				Filename: filename,
				Line:     lineNumber,
				Text:     fmt.Sprintf("line is %d characters", lineLen),
			})
		}
	}

	if err := scanner.Err(); err != nil {
		if !(err == bufio.ErrTooLong && maxLineLen < bufio.MaxScanTokenSize) {
			return nil, fmt.Errorf("can't scan file %s: %s", filename, err)
		}
		// scanner.Scan() might fail if the line is longer than bufio.MaxScanTokenSize
		// In the case where the specified maxLineLen is smaller than bufio.MaxScanTokenSize
		// we can return this line as a long line instead of returning an error.
		// The reason for this change is that this case might happen with autogenerated files
		// The go-bindata tool for instance might generate a file with a very long line.
		// In this case, as it's an auto generated file, the warning returned by lll will
		// be ignored.
		// But if we return a linter error here, and this error happens for an autogenerated
		// file the error will be discarded (fine), but all the subsequent errors for lll will
		// be discarded for other files, and we'll miss legit error.
		res = append(res, result{
			Filename: filename,
			Line:     lineNumber,
			Text:     fmt.Sprintf("line is more than %d characters", bufio.MaxScanTokenSize),
		})
	}

	return res, nil
}

func init() {
	util.SkipAnalyzerByConfig(Analyzer)
	util.SkipAnalyzer(Analyzer)
}
