// Copyright 2024 PingCAP, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package metautil

import (
	"bytes"
	"context"
	"crypto/sha256"
	"encoding/json"
	"fmt"

	"github.com/gogo/protobuf/proto"
	"github.com/pingcap/errors"
	backuppb "github.com/pingcap/kvproto/pkg/brpb"
	berrors "github.com/pingcap/tidb/br/pkg/errors"
	"github.com/pingcap/tidb/br/pkg/storage"
	"github.com/pingcap/tidb/br/pkg/utils"
	"github.com/pingcap/tidb/pkg/meta/model"
	"github.com/pingcap/tidb/pkg/statistics/handle"
	statstypes "github.com/pingcap/tidb/pkg/statistics/handle/types"
	statsutil "github.com/pingcap/tidb/pkg/statistics/util"
	"github.com/pingcap/tidb/pkg/util"
	"golang.org/x/sync/errgroup"
)

var maxStatsJsonTableSize = 32 * 1024 * 1024 // 32 MiB
var inlineSize = 8 * 1024                    // 8 KiB

func getStatsFileName(physicalID int64) string {
	return fmt.Sprintf("backupmeta.schema.stats.%09d", physicalID)
}

// A lightweight function wrapper to dump the statistic
type StatsWriter struct {
	storage storage.ExternalStorage
	cipher  *backuppb.CipherInfo

	// final stats file indexes
	statsFileIndexes []*backuppb.StatsFileIndex

	// temporary variables, clear after each flush
	totalSize int
	statsFile *backuppb.StatsFile
}

func newStatsWriter(
	storage storage.ExternalStorage,
	cipher *backuppb.CipherInfo,
) *StatsWriter {
	return &StatsWriter{
		storage: storage,
		cipher:  cipher,

		statsFileIndexes: make([]*backuppb.StatsFileIndex, 0),

		totalSize: 0,
		statsFile: &backuppb.StatsFile{
			Blocks: make([]*backuppb.StatsBlock, 0, 8),
		},
	}
}

// flush temporary and clear []byte to make it garbage collected as soon as possible
func (s *StatsWriter) flushTemporary() ([]byte, error) {
	defer s.clearTemporary()
	return proto.Marshal(s.statsFile)
}

func (s *StatsWriter) clearTemporary() {
	// clear the temporary variables
	s.totalSize = 0
	s.statsFile = &backuppb.StatsFile{
		Blocks: make([]*backuppb.StatsBlock, 0, 8),
	}
}

func (s *StatsWriter) writeStatsFileAndClear(ctx context.Context, physicalID int64) error {
	fileName := getStatsFileName(physicalID)
	content, err := s.flushTemporary()
	if err != nil {
		return errors.Trace(err)
	}

	if len(s.statsFileIndexes) == 0 && len(content) < inlineSize {
		s.statsFileIndexes = append(s.statsFileIndexes, &backuppb.StatsFileIndex{InlineData: content})
		return nil
	}

	checksum := sha256.Sum256(content)
	sizeOri := uint64(len(content))
	encryptedContent, iv, err := Encrypt(content, s.cipher)
	if err != nil {
		return errors.Trace(err)
	}

	if err := s.storage.WriteFile(ctx, fileName, encryptedContent); err != nil {
		return errors.Trace(err)
	}

	s.statsFileIndexes = append(s.statsFileIndexes, &backuppb.StatsFileIndex{
		Name:     fileName,
		Sha256:   checksum[:],
		SizeEnc:  uint64(len(encryptedContent)),
		SizeOri:  sizeOri,
		CipherIv: iv,
	})
	return nil
}

func (s *StatsWriter) BackupStats(ctx context.Context, jsonTable *statsutil.JSONTable, physicalID int64) error {
	if jsonTable == nil {
		return nil
	}

	statsBytes, err := json.Marshal(jsonTable)
	if err != nil {
		return errors.Trace(err)
	}

	s.totalSize += len(statsBytes)
	s.statsFile.Blocks = append(s.statsFile.Blocks, &backuppb.StatsBlock{
		PhysicalId: physicalID,
		JsonTable:  statsBytes,
	})

	// check whether need to flush
	if s.totalSize > maxStatsJsonTableSize {
		if err := s.writeStatsFileAndClear(ctx, physicalID); err != nil {
			return errors.Trace(err)
		}
	}
	return nil
}

func (s *StatsWriter) BackupStatsDone(ctx context.Context) ([]*backuppb.StatsFileIndex, error) {
	if s.totalSize == 0 || len(s.statsFile.Blocks) == 0 {
		return s.statsFileIndexes, nil
	}

	if err := s.writeStatsFileAndClear(ctx, s.statsFile.Blocks[0].PhysicalId); err != nil {
		return nil, errors.Trace(err)
	}
	return s.statsFileIndexes, nil
}

func RestoreStats(
	ctx context.Context,
	storage storage.ExternalStorage,
	cipher *backuppb.CipherInfo,
	statsHandler *handle.Handle,
	newTableInfo *model.TableInfo,
	statsFileIndexes []*backuppb.StatsFileIndex,
	rewriteIDMap map[int64]int64,
) error {
	eg, ectx := errgroup.WithContext(ctx)
	taskCh := make(chan *statstypes.PartitionStatisticLoadTask, 8)
	eg.Go(func() error {
		return downloadStats(ectx, storage, cipher, statsFileIndexes, rewriteIDMap, taskCh)
	})
	eg.Go(func() error {
		// NOTICE: skip updating cache after load stats from json
		return statsHandler.LoadStatsFromJSONConcurrently(ectx, newTableInfo, taskCh, 0)
	})
	return eg.Wait()
}

func downloadStats(
	ctx context.Context,
	storage storage.ExternalStorage,
	cipher *backuppb.CipherInfo,
	statsFileIndexes []*backuppb.StatsFileIndex,
	rewriteIDMap map[int64]int64,
	taskCh chan<- *statstypes.PartitionStatisticLoadTask,
) error {
	defer close(taskCh)
	eg, ectx := errgroup.WithContext(ctx)
	downloadWorkerpool := util.NewWorkerPool(4, "download stats for each partition")
	for _, statsFileIndex := range statsFileIndexes {
		if ectx.Err() != nil {
			break
		}
		statsFile := statsFileIndex
		downloadWorkerpool.ApplyOnErrorGroup(eg, func() error {
			var statsContent []byte
			if len(statsFile.InlineData) > 0 {
				statsContent = statsFile.InlineData
			} else {
				content, err := storage.ReadFile(ectx, statsFile.Name)
				if err != nil {
					return errors.Trace(err)
				}

				decryptContent, err := utils.Decrypt(content, cipher, statsFile.CipherIv)
				if err != nil {
					return errors.Trace(err)
				}

				checksum := sha256.Sum256(decryptContent)
				if !bytes.Equal(statsFile.Sha256, checksum[:]) {
					return berrors.ErrInvalidMetaFile.GenWithStackByArgs(fmt.Sprintf(
						"checksum mismatch expect %x, got %x", statsFile.Sha256, checksum[:]))
				}
				statsContent = decryptContent
			}

			statsFileBlocks := &backuppb.StatsFile{}
			if err := proto.Unmarshal(statsContent, statsFileBlocks); err != nil {
				return errors.Trace(err)
			}

			for _, block := range statsFileBlocks.Blocks {
				physicalId, ok := rewriteIDMap[block.PhysicalId]
				if !ok {
					return berrors.ErrRestoreInvalidRewrite.GenWithStackByArgs(fmt.Sprintf(
						"not rewrite rule matched, old physical id: %d", block.PhysicalId))
				}
				jsonTable := &statsutil.JSONTable{}
				if err := json.Unmarshal(block.JsonTable, jsonTable); err != nil {
					return errors.Trace(err)
				}
				// reset the block.JsonTable to nil to make it garbage collected as soon as possible
				block.JsonTable = nil

				select {
				case <-ectx.Done():
					return nil
				case taskCh <- &statstypes.PartitionStatisticLoadTask{
					PhysicalID: physicalId,
					JSONTable:  jsonTable,
				}:
				}
			}

			return nil
		})
	}

	return eg.Wait()
}
