/*******************************************************************************
 * Eclipse Public License - v 1.0
 *
 * THE ACCOMPANYING PROGRAM IS PROVIDED UNDER THE TERMS OF THIS ECLIPSE PUBLIC
 * LICENSE ("AGREEMENT"). ANY USE, REPRODUCTION OR DISTRIBUTION OF THE PROGRAM
 * CONSTITUTES RECIPIENT'S ACCEPTANCE OF THIS AGREEMENT.
 *
 * 1. DEFINITIONS
 *
 * "Contribution" means:
 * (a) in the case of the initial Contributor, the initial code and
 * documentation distributed under this Agreement, and
 * (b) in the case of each subsequent Contributor:
 * (i) changes to the Program, and
 * (ii) additions to the Program;
 *
 * where such changes and/or additions to the Program originate from and are
 * distributed by that particular Contributor. A Contribution 'originates' from
 * a Contributor if it was added to the Program by such Contributor itself or
 * anyone acting on such Contributor's behalf. Contributions do not include
 * additions to the Program which: (i) are separate modules of software
 * distributed in conjunction with the Program under their own license
 * agreement, and (ii) are not derivative works of the Program.
 * 
 * "Contributor" means any person or entity that distributes the Program.
 * 
 * "Licensed Patents " mean patent claims licensable by a Contributor which are
 * necessarily infringed by the use or sale of its Contribution alone or when
 * combined with the Program.
 * 
 * "Program" means the Contributions distributed in accordance with this
 * Agreement.
 * 
 * "Recipient" means anyone who receives the Program under this Agreement,
 * including all Contributors.
 * 
 * 2. GRANT OF RIGHTS
 * 
 * (a) Subject to the terms of this Agreement, each Contributor hereby grants
 * Recipient a non-exclusive, worldwide, royalty-free copyright license to
 * reproduce, prepare derivative works of, publicly display, publicly perform,
 * distribute and sublicense the Contribution of such Contributor, if any, and
 * such derivative works, in source code and object code form.
 * 
 * (b) Subject to the terms of this Agreement, each Contributor hereby grants
 * Recipient a non-exclusive, worldwide, royalty-free patent license under
 * Licensed Patents to make, use, sell, offer to sell, import and otherwise
 * transfer the Contribution of such Contributor, if any, in source code and
 * object code form. This patent license shall apply to the combination of the
 * Contribution and the Program if, at the time the Contribution is added by the
 * Contributor, such addition of the Contribution causes such combination to be
 * covered by the Licensed Patents. The patent license shall not apply to any
 * other combinations which include the Contribution. No hardware per se is
 * licensed hereunder.
 *
 * (c) Recipient understands that although each Contributor grants the licenses
 * to its Contributions set forth herein, no assurances are provided by any
 * Contributor that the Program does not infringe the patent or other
 * intellectual property rights of any other entity. Each Contributor disclaims
 * any liability to Recipient for claims brought by any other entity based on
 * infringement of intellectual property rights or otherwise. As a condition to
 * exercising the rights and licenses granted hereunder, each Recipient hereby
 * assumes sole responsibility to secure any other intellectual property rights
 * needed, if any. For example, if a third party patent license is required to
 * allow Recipient to distribute the Program, it is Recipient's responsibility
 * to acquire that license before distributing the Program.
 *
 * (d) Each Contributor represents that to its knowledge it has sufficient
 * copyright rights in its Contribution, if any, to grant the copyright license
 * set forth in this Agreement.
 * 
 * 3. REQUIREMENTS
 * 
 * A Contributor may choose to distribute the Program in object code form under
 * its own license agreement, provided that:
 * 
 * (a) it complies with the terms and conditions of this Agreement; and
 * (b) its license agreement:
 * (i) effectively disclaims on behalf of all Contributors all warranties and
 * conditions, express and implied, including warranties or conditions of title
 * and non-infringement, and implied warranties or conditions of merchantability
 * and fitness for a particular purpose;
 * (ii) effectively excludes on behalf of all Contributors all liability for
 * damages, including direct, indirect, special, incidental and consequential
 * damages, such as lost profits;
 * (iii)  states that any provisions which differ from this Agreement are offered
 * by that Contributor alone and not by any other party; and 
 * (iv) states that source code for the Program is available from such Contributor, 
 * and informs licensees how to obtain it in a reasonable manner on or through a 
 * medium customarily used for software exchange.
 * 
 * When the Program is made available in source code form:
 *
 * (a) it must be made available under this Agreement; and
 * (b) a copy of this Agreement must be included with each copy of the Program.
 * 
 * Contributors may not remove or alter any copyright notices contained within
 * the Program.
 * 
 * Each Contributor must identify itself as the originator of its Contribution,
 * if any, in a manner that reasonably allows subsequent Recipients to identify
 * the originator of the Contribution.
 * 
 * 4. COMMERCIAL DISTRIBUTION
 * 
 * Commercial distributors of software may accept certain responsibilities with
 * respect to end users, business partners and the like. While this license is
 * intended to facilitate the commercial use of the Program, the Contributor who
 * includes the Program in a commercial product offering should do so in a
 * manner which does not create potential liability for other Contributors.
 * Therefore, if a Contributor includes the Program in a commercial product
 * offering, such Contributor ("Commercial Contributor") hereby agrees to defend
 * and indemnify every other Contributor ("Indemnified Contributor") against any
 * losses, damages and costs (collectively "Losses") arising from claims,
 * lawsuits and other legal actions brought by a third party against the
 * Indemnified Contributor to the extent caused by the acts or omissions of such
 * Commercial Contributor in connection with its distribution of the Program in
 * a commercial product offering. The obligations in this section do not apply
 * to any claims or Losses relating to any actual or alleged intellectual
 * property infringement. In order to qualify, an Indemnified Contributor must:
 * a) promptly notify the Commercial Contributor in writing of such claim, and
 * b) allow the Commercial Contributor to control, and cooperate with the
 * Commercial Contributor in, the defense and any related settlement
 * negotiations. The Indemnified Contributor may participate in any such claim
 * at its own expense.
 * 
 * For example, a Contributor might include the Program in a commercial product
 * offering, Product X. That Contributor is then a Commercial Contributor. If
 * that Commercial Contributor then makes performance claims, or offers
 * warranties related to Product X, those performance claims and warranties are
 * such Commercial Contributor's responsibility alone. Under this section, the
 * Commercial Contributor would have to defend claims against the other
 * Contributors related to those performance claims and warranties, and if a
 * court requires any other Contributor to pay any damages as a result, the
 * Commercial Contributor must pay those damages.
 * 
 * 5. NO WARRANTY
 * 
 * EXCEPT AS EXPRESSLY SET FORTH IN THIS AGREEMENT, THE PROGRAM IS PROVIDED ON
 * AN "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED INCLUDING, WITHOUT LIMITATION, ANY WARRANTIES OR
 * CONDITIONS OF TITLE, NON-INFRINGEMENT, MERCHANTABILITY OR FITNESS FOR A
 * PARTICULAR PURPOSE. Each Recipient is solely responsible for determining the
 * appropriateness of using and distributing the Program and assumes all risks
 * associated with its exercise of rights under this Agreement , including but
 * not limited to the risks and costs of program errors, compliance with
 * applicable laws, damage to or loss of data, programs or equipment, and
 * unavailability or interruption of operations.
 * 
 * 6. DISCLAIMER OF LIABILITY
 * 
 * EXCEPT AS EXPRESSLY SET FORTH IN THIS AGREEMENT, NEITHER RECIPIENT NOR ANY
 * CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING WITHOUT LIMITATION
 * LOST PROFITS), HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OR DISTRIBUTION OF THE PROGRAM OR THE
 * EXERCISE OF ANY RIGHTS GRANTED HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGES.
 * 
 * 7. GENERAL
 * 
 * If any provision of this Agreement is invalid or unenforceable under
 * applicable law, it shall not affect the validity or enforceability of the
 * remainder of the terms of this Agreement, and without further action by the
 * parties hereto, such provision shall be reformed to the minimum extent
 * necessary to make such provision valid and enforceable. If Recipient
 * institutes patent litigation against any entity (including a cross-claim or
 * counterclaim in a lawsuit) alleging that the Program itself (excluding
 * combinations of the Program with other software or hardware) infringes such
 * Recipient's patent(s), then such Recipient's rights granted under Section
 * 2(b) shall terminate as of the date such litigation is filed. All Recipient's
 * rights under this Agreement shall terminate if it fails to comply with any of
 * the material terms or conditions of this Agreement and does not cure such
 * failure in a reasonable period of time after becoming aware of such
 * noncompliance. If all Recipient's rights under this Agreement terminate,
 * Recipient agrees to cease use and distribution of the Program as soon as
 * reasonably practicable. However, Recipient's obligations under this Agreement
 * and any licenses granted by Recipient relating to the Program shall continue
 * and survive. Everyone is permitted to copy and distribute copies of this
 * Agreement, but in order to avoid inconsistency the Agreement is copyrighted
 * and may only be modified in the following manner. The Agreement Steward
 * reserves the right to publish new versions (including revisions) of this
 * Agreement from time to time. No one other than the Agreement Steward has the
 * right to modify this Agreement. The Eclipse Foundation is the initial
 * Agreement Steward. The Eclipse Foundation may assign the responsibility to
 * serve as the Agreement Steward to a suitable separate entity. Each new
 * version of the Agreement will be given a distinguishing version number. The
 * Program (including Contributions) may always be distributed subject to the
 * version of the Agreement under which it was received. In addition, after a
 * new version of the Agreement is published, Contributor may elect to
 * distribute the Program (including its Contributions) under the new version.
 * Except as expressly stated in Sections 2(a) and 2(b) above, Recipient
 * receives no rights or licenses to the intellectual property of any
 * Contributor under this Agreement, whether expressly, by implication, estoppel
 * or otherwise. All rights in the Program not expressly granted under this
 * Agreement are reserved. This Agreement is governed by the laws of the State
 * of New York and the intellectual property laws of the United States of
 * America. No party to this Agreement will bring a legal action under this
 * Agreement more than one year after the cause of action arose. Each party
 * waives its rights to a jury trial in any resulting litigation.
 * 
 *******************************************************************************/

package edu.mit.jwi;

import java.util.Vector;

/**
 * A string pattern matcher, suppporting "*" and "?" wildcards.
 */
public class StringMatcher {

	protected String fPattern;

	protected String fPatternRoot;

	protected int fLength; // pattern length

	protected boolean fIgnoreWildCards;

	protected boolean fIgnoreCase;

	protected boolean fHasLeadingStar;

	protected boolean fHasTrailingStar;

	protected String fSegments[]; // the given pattern is split into *
	// separated segments

	/* boundary value beyond which we don't need to search in the text */
	protected int fBound = 0;

	protected static final char fSingleWildCard = '\u0000';

	/**
	 * StringMatcher constructor takes in a String object that is a simple
	 * pattern which may contain '*' for 0 and many characters and '?' for
	 * exactly one character. Literal '*' and '?' characters must be escaped in
	 * the pattern e.g., "\*" means literal "*", etc. Escaping any other
	 * character (including the escape character itself), just results in that
	 * character in the pattern. e.g., "\a" means "a" and "\\" means "\" If
	 * invoking the StringMatcher with string literals in Java, don't forget
	 * escape characters are represented by "\\".
	 * 
	 * @param pattern
	 *            the pattern to match text against
	 * @param ignoreCase
	 *            if true, case is ignored
	 * @param ignoreWildCards
	 *            if true, wild cards and their escape sequences are ignored
	 *            (everything is taken literally).
	 */
	public StringMatcher(String pattern, boolean ignoreCase, boolean ignoreWildCards) {
		if (pattern == null) throw new IllegalArgumentException();
		fIgnoreCase = ignoreCase;
		fIgnoreWildCards = ignoreWildCards;
		fPattern = pattern;
		fPatternRoot = Dictionary.getPatternRoot(pattern, ignoreWildCards);
		fLength = pattern.length();

		if (fIgnoreWildCards) {
			parseNoWildCards();
		}
		else {
			parseWildCards();
		}
	}

	/**
	 * Returns the initial part of the pattern that <b>must</b> match, i.e.,
	 * the part of the string up to but not including the first wildcard.
	 */
	public String getPatternRoot() {
		return fPatternRoot;
	}

	/**
	 * Match the specified text with the pattern
	 * 
	 * @param text
	 *            text to match
	 * @return <code>true</code> if matched, otherwise <code>false</code>
	 */
	public boolean match(String text) {
		if (text == null) return false;
		return match(text, 0, text.length());
	}

	/**
	 * Given the starting (inclusive) and the ending (exclusive) positions in
	 * the <code>text</code>, determine if the given substring matches with
	 * aPattern
	 * 
	 * @param text
	 *            the substring to match
	 * @param start
	 *            marks the starting position (inclusive) of the substring
	 * @param end
	 *            marks the ending index (exclusive) of the substring
	 * @return <code>true</code> if the specified portion of the text matches
	 *         the pattern
	 */
	public boolean match(String text, int start, int end) {
		if (null == text) throw new IllegalArgumentException();

		if (start > end) return false;

		if (fIgnoreWildCards) return (end - start == fLength)
				&& fPattern.regionMatches(fIgnoreCase, 0, text, start, fLength);
		int segCount = fSegments.length;
		if (segCount == 0 && (fHasLeadingStar || fHasTrailingStar)) // pattern
		// contains
		// only
		// '*'(s)
		return true;
		if (start == end) return fLength == 0;
		if (fLength == 0) return start == end;

		int tlen = text.length();
		if (start < 0) start = 0;
		if (end > tlen) end = tlen;

		int tCurPos = start;
		int bound = end - fBound;
		if (bound < 0) return false;
		int i = 0;
		String current = fSegments[i];
		int segLength = current.length();

		/* process first segment */
		if (!fHasLeadingStar) {
			if (!regExpRegionMatches(text, start, current, 0, segLength)) {
				return false;
			}
			else {
				++i;
				tCurPos = tCurPos + segLength;
			}
		}
		if ((fSegments.length == 1) && (!fHasLeadingStar) && (!fHasTrailingStar)) {
			// only one segment to match, no wildcards specified
			return tCurPos == end;
		}
		/* process middle segments */
		while (i < segCount) {
			current = fSegments[i];
			int currentMatch;
			int k = current.indexOf(fSingleWildCard);
			if (k < 0) {
				currentMatch = textPosIn(text, tCurPos, end, current);
				if (currentMatch < 0) return false;
			}
			else {
				currentMatch = regExpPosIn(text, tCurPos, end, current);
				if (currentMatch < 0) return false;
			}
			tCurPos = currentMatch + current.length();
			i++;
		}

		/* process final segment */
		if (!fHasTrailingStar && tCurPos != end) {
			int clen = current.length();
			return regExpRegionMatches(text, end - clen, current, 0, clen);
		}
		return i == segCount;
	}

	/**
	 * This method parses the given pattern into segments seperated by wildcard
	 * '*' characters. Since wildcards are not being used in this case, the
	 * pattern consists of a single segment.
	 */
	private void parseNoWildCards() {
		fSegments = new String[1];
		fSegments[0] = fPattern;
		fBound = fLength;
	}

	/**
	 * Parses the given pattern into segments seperated by wildcard '*'
	 * characters.
	 * 
	 * @param p,
	 *            a String object that is a simple regular expression with '*'
	 *            and/or '?'
	 */
	private void parseWildCards() {
		if (fPattern.startsWith("*")) //$NON-NLS-1$
		fHasLeadingStar = true;
		if (fPattern.endsWith("*")) {//$NON-NLS-1$
			/* make sure it's not an escaped wildcard */
			if (fLength > 1 && fPattern.charAt(fLength - 2) != '\\') {
				fHasTrailingStar = true;
			}
		}

		Vector<String> temp = new Vector<String>();

		int pos = 0;
		StringBuffer buf = new StringBuffer();
		while (pos < fLength) {
			char c = fPattern.charAt(pos++);
			switch (c) {
			case '\\':
				if (pos >= fLength) {
					buf.append(c);
				}
				else {
					char next = fPattern.charAt(pos++);
					/* if it's an escape sequence */
					if (next == '*' || next == '?' || next == '\\') {
						buf.append(next);
					}
					else {
						/* not an escape sequence, just insert literally */
						buf.append(c);
						buf.append(next);
					}
				}
				break;
			case '*':
				if (buf.length() > 0) {
					/* new segment */
					temp.addElement(buf.toString());
					fBound += buf.length();
					buf.setLength(0);
				}
				break;
			case '?':
				/*
				 * append special character representing single match wildcard
				 */
				buf.append(fSingleWildCard);
				break;
			default:
				buf.append(c);
			}
		}

		/* add last buffer to segment list */
		if (buf.length() > 0) {
			temp.addElement(buf.toString());
			fBound += buf.length();
		}

		fSegments = new String[temp.size()];
		temp.copyInto(fSegments);
	}

	/**
	 * @param text
	 *            a string that contains no wildcard
	 * @param start
	 *            the starting index in the text for search, inclusive
	 * @param end
	 *            the stopping point of search, exclusive
	 * @return the starting index in the text of the pattern , or -1 if not
	 *         found
	 */
	protected int posIn(String text, int start, int end) {// no wild card
		// in pattern
		int max = end - fLength;

		if (!fIgnoreCase) {
			int i = text.indexOf(fPattern, start);
			if (i == -1 || i > max) return -1;
			return i;
		}

		for (int i = start; i <= max; ++i) {
			if (text.regionMatches(true, i, fPattern, 0, fLength)) return i;
		}

		return -1;
	}

	/**
	 * @param text
	 *            a simple regular expression that may only contain '?'(s)
	 * @param start
	 *            the starting index in the text for search, inclusive
	 * @param end
	 *            the stopping point of search, exclusive
	 * @param p
	 *            a simple regular expression that may contains '?'
	 * @return the starting index in the text of the pattern , or -1 if not
	 *         found
	 */
	protected int regExpPosIn(String text, int start, int end, String p) {
		int plen = p.length();

		int max = end - plen;
		for (int i = start; i <= max; ++i) {
			if (regExpRegionMatches(text, i, p, 0, plen)) return i;
		}
		return -1;
	}

	protected boolean regExpRegionMatches(String text, int tStart, String p, int pStart, int plen) {
		while (plen-- > 0) {
			char tchar = text.charAt(tStart++);
			char pchar = p.charAt(pStart++);

			/* process wild cards */
			if (!fIgnoreWildCards) {
				/* skip single wild cards */
				if (pchar == fSingleWildCard) {
					continue;
				}
			}
			if (pchar == tchar) continue;
			if (fIgnoreCase) {
				if (Character.toUpperCase(tchar) == Character.toUpperCase(pchar)) continue;
				// comparing after converting to upper case doesn't handle
				// all cases;
				// also compare after converting to lower case
				if (Character.toLowerCase(tchar) == Character.toLowerCase(pchar)) continue;
			}
			return false;
		}
		return true;
	}

	/**
	 * @param text
	 *            the string to match
	 * @param start
	 *            the starting index in the text for search, inclusive
	 * @param end
	 *            the stopping point of search, exclusive
	 * @param p
	 *            a string that has no wildcard
	 * @return the starting index in the text of the pattern , or -1 if not
	 *         found
	 */
	protected int textPosIn(String text, int start, int end, String p) {

		int plen = p.length();
		int max = end - plen;

		if (!fIgnoreCase) {
			int i = text.indexOf(p, start);
			if (i == -1 || i > max) return -1;
			return i;
		}

		for (int i = start; i <= max; ++i) {
			if (text.regionMatches(true, i, p, 0, plen)) return i;
		}

		return -1;
	}
}