/**
 * Copyright (c) 2014-present, The osquery authors
 *
 * This source code is licensed as defined by the LICENSE file found in the
 * root directory of this source tree.
 *
 * SPDX-License-Identifier: (Apache-2.0 OR GPL-2.0-only)
 */

#include <algorithm>
#include <fstream>
#include <random>

#include <stdio.h>
#include <sys/stat.h>

#include <gtest/gtest.h>

#include <boost/filesystem.hpp>
#include <boost/uuid/uuid.hpp>
#include <boost/uuid/uuid_generators.hpp>
#include <boost/uuid/uuid_io.hpp>

#include <osquery/filesystem/filesystem.h>

#include <osquery/core/flags.h>
#include <osquery/core/system.h>
#include <osquery/filesystem/mock_file_structure.h>
#include <osquery/logger/logger.h>
#include <osquery/process/process.h>
#include <osquery/utils/info/platform_type.h>

#ifdef WIN32
#include "winbase.h"
#include <osquery/utils/conversions/windows/strings.h>
#endif

namespace fs = boost::filesystem;

namespace osquery {

namespace {

const std::vector<std::string> kFileNameList{
    "辞書.txt",
    "test_file.txt",
};

}

DECLARE_uint64(read_max);

extern inline Status listInAbsoluteDirectory(const fs::path& path,
                                             std::vector<std::string>& results,
                                             GlobLimits limits);

class FilesystemTests : public testing::Test {
 protected:
  fs::path test_working_dir_;
  fs::path fake_directory_;

  void SetUp() override {
    initializeFilesystemAPILocale();

    fake_directory_ = fs::canonical(createMockFileStructure());
    test_working_dir_ = fs::temp_directory_path() /
                        fs::unique_path("osquery.test_working_dir.%%%%.%%%%");
    fs::create_directories(test_working_dir_);

    if (isPlatform(PlatformType::TYPE_WINDOWS)) {
      etc_hosts_path_ = "C:\\Windows\\System32\\drivers\\etc\\hosts";
      etc_path_ = "C:\\Windows\\System32\\drivers\\etc";
      tmp_path_ = fs::temp_directory_path().string();
      line_ending_ = "\r\n";

      auto raw_drive = getEnvVar("SystemDrive");
      system_root_ = (raw_drive.is_initialized() ? *raw_drive : "") + "\\";
    } else {
      etc_hosts_path_ = "/etc/hosts";
      etc_path_ = "/etc";
      tmp_path_ = "/tmp";
      line_ending_ = "\n";

      system_root_ = "/";
    }
  }

  void TearDown() override {
    fs::remove_all(fake_directory_);
    fs::remove_all(test_working_dir_);
  }

  /// Helper method to check if a path was included in results.
  bool contains(const std::vector<std::string>& all, const std::string& n) {
    return !(std::find(all.begin(), all.end(), n) == all.end());
  }

  /// Helper method to generate a random names
  std::string genRandomName() {
    return boost::uuids::to_string(boost::uuids::random_generator()());
  }

  /// Helper method to create nested directory structures
  bool createNestedDirectories(const fs::path& parent,
                               unsigned int subdir_count,
                               unsigned int& created_dirs) {
    if (!fs::create_directory(parent)) {
      return false;
    }

    created_dirs++;
    for (unsigned int it = 0; it < subdir_count; ++it) {
      if (fs::create_directory(parent / genRandomName())) {
        created_dirs++;
      }
    }

    return true;
  }

  /// Helper method to create directory symlinks
  bool createDirectorySymlink(const fs::path& target,
                              const fs::path& link,
                              unsigned int& created_dir_symlink) {
    bool ret = false;

    try {
      fs::create_directory_symlink(target, link);
      created_dir_symlink++;
      ret = true;
    } catch (const fs::filesystem_error& e) {
      std::cerr << "Error creating symlink: " << e.what() << std::endl;
    }

    return ret;
  }

  /// Helper method to delete directory and its content
  bool deleteDirectoryContent(const fs::path& dir_path) {
    bool ret = false;

    try {
      fs::remove_all(dir_path);
      ret = true;
    } catch (const fs::filesystem_error& e) {
      std::cerr << "Error deleting directory content: " << e.what()
                << std::endl;
    }

    return ret;
  }

  /// Helper method to get a random number
  unsigned int getRandomNumber() {
    const unsigned int MAX_DIST_NUMBER = 30;
    static std::default_random_engine engine(
        (unsigned int)std::chrono::system_clock::now()
            .time_since_epoch()
            .count());
    static std::uniform_int_distribution<unsigned int> dist(1, MAX_DIST_NUMBER);

    return dist(engine);
  }

  // legacy listDirectoriesInDirectory logic
  Status legacyListDirectoriesInDirectory(const fs::path& path,
                                          std::vector<std::string>& results,
                                          bool recursive) {
    return listInAbsoluteDirectory(
        (path / ((recursive) ? "**" : "*")), results, GLOB_FOLDERS);
  }

 protected:
  std::string etc_hosts_path_;
  std::string etc_path_;
  std::string tmp_path_;
  std::string system_root_;
  std::string line_ending_;
};

TEST_F(FilesystemTests, test_read_file) {
  for (const auto& file_name : kFileNameList) {
    auto file_path = test_working_dir_ / file_name;

    std::ofstream test_file(file_path.string());
    test_file.write("test123\n", sizeof("test123"));
    test_file.close();

    std::string content;
    auto s = readFile(file_path, content);

    EXPECT_TRUE(s.ok());
    EXPECT_EQ(s.toString(), "OK");
    EXPECT_EQ(content, "test123" + line_ending_);

    removePath(file_path);
  }
}

TEST_F(FilesystemTests, test_remove_path) {
  for (const auto& file_name : kFileNameList) {
    auto test_dir = test_working_dir_ / file_name;
    fs::create_directories(test_dir);

    auto test_file = test_working_dir_ / file_name / "rmfile";
    writeTextFile(test_file, "testcontent");

    ASSERT_TRUE(pathExists(test_file).ok());

    // Try to remove the directory.
    EXPECT_TRUE(removePath(test_dir));
    EXPECT_FALSE(pathExists(test_file).ok());
    EXPECT_FALSE(pathExists(test_dir).ok());
  }
}

TEST_F(FilesystemTests, test_write_file) {
  for (const auto& file_name : kFileNameList) {
    auto test_file = test_working_dir_ / file_name;
    std::string content(2048, 'A');

    EXPECT_TRUE(writeTextFile(test_file, content).ok());
    ASSERT_TRUE(pathExists(test_file).ok());
    ASSERT_TRUE(isWritable(test_file).ok());
    ASSERT_TRUE(removePath(test_file).ok());

    EXPECT_TRUE(writeTextFile(test_file, content, 0400));
    ASSERT_TRUE(pathExists(test_file).ok());

    // On POSIX systems, root can still read/write.
    EXPECT_FALSE(isWritable(test_file).ok());
    EXPECT_TRUE(isReadable(test_file).ok());
    ASSERT_TRUE(removePath(test_file).ok());

    EXPECT_TRUE(writeTextFile(test_file, content, 0000));
    ASSERT_TRUE(pathExists(test_file).ok());

    // On POSIX systems, root can still read/write.
    EXPECT_FALSE(isWritable(test_file).ok());
    EXPECT_FALSE(isReadable(test_file).ok());
    ASSERT_TRUE(removePath(test_file).ok());
  }
}

TEST_F(FilesystemTests, test_readwrite_file) {
  for (const auto& file_name : kFileNameList) {
    auto test_file = test_working_dir_ / file_name;
    size_t filesize = 4096 * 10;

    std::string in_content(filesize, 'A');
    EXPECT_TRUE(writeTextFile(test_file, in_content).ok());
    ASSERT_TRUE(pathExists(test_file).ok());
    ASSERT_TRUE(isReadable(test_file).ok());

    // Now read the content back.
    std::string out_content;
    EXPECT_TRUE(readFile(test_file, out_content).ok());
    EXPECT_EQ(filesize, out_content.size());
    EXPECT_EQ(in_content, out_content);
    removePath(test_file);

    // Now try to write outside of a 4k chunk size.
    in_content = std::string(filesize + 1, 'A');
    writeTextFile(test_file, in_content);
    out_content.clear();
    readFile(test_file, out_content);
    EXPECT_EQ(in_content, out_content);
    removePath(test_file);
  }
}

TEST_F(FilesystemTests, test_read_limit) {
  auto max = FLAGS_read_max;
  FLAGS_read_max = 3;
  std::string content;
  auto status = readFile(fake_directory_ / "root.txt", content);
  EXPECT_FALSE(status.ok());
  FLAGS_read_max = max;

  // Make sure non-link files are still readable.
  content.erase();
  status = readFile(fake_directory_ / "root.txt", content);
  EXPECT_TRUE(status.ok());

  // Any the links are readable too.
  status = readFile(fake_directory_ / "root2.txt", content);
  EXPECT_TRUE(status.ok());
}

TEST_F(FilesystemTests, test_list_files_missing_directory) {
  std::vector<std::string> results;
  auto status = listFilesInDirectory("/foo/bar", results);
  EXPECT_FALSE(status.ok());
}

TEST_F(FilesystemTests, test_list_files_invalid_directory) {
  std::vector<std::string> results;
  auto status = listFilesInDirectory("/etc/hosts", results);
  EXPECT_FALSE(status.ok());
}

TEST_F(FilesystemTests, test_list_files_valid_directory) {
  std::vector<std::string> results;

  auto s = listFilesInDirectory(etc_path_, results);
  // This directory may be different on OS X or Linux.

  replaceGlobWildcards(etc_hosts_path_);
  EXPECT_TRUE(s.ok());
  EXPECT_EQ(s.toString(), "OK");
  EXPECT_TRUE(contains(results, etc_hosts_path_));
}

TEST_F(FilesystemTests, test_intermediate_globbing_directories) {
  fs::path thirdLevelDir =
      fs::path(fake_directory_) / kTopLevelMockFolderName / "%/thirdlevel1";
  std::vector<std::string> results;
  resolveFilePattern(thirdLevelDir, results);
  EXPECT_EQ(results.size(), 1U);
}

TEST_F(FilesystemTests, test_canonicalization) {
  std::string complex_path = (fs::path(fake_directory_) / "deep1/../deep1/..")
                                 .make_preferred()
                                 .string();
  std::string simple_path = fake_directory_.make_preferred().string();

  if (isPlatform(PlatformType::TYPE_WINDOWS)) {
    simple_path += "\\";
  } else {
    simple_path += "/";
  }

  // Use the inline wildcard and canonicalization replacement.
  // The 'simple_path' path contains a trailing '/', the replacement method will
  // distinguish between file and directory paths.
  replaceGlobWildcards(complex_path);
  EXPECT_EQ(simple_path, complex_path);

  // Now apply the same inline replacement on the simple_path directory and
  // expect no change to the comparison.
  replaceGlobWildcards(simple_path);
  EXPECT_EQ(simple_path, complex_path);

  // Now add a wildcard within the complex_path pattern. The replacement method
  // will not canonicalize past a '*' as the proceeding paths are limiters.
  complex_path = (fs::path(fake_directory_) / "*/deep2/../deep2/")
                     .make_preferred()
                     .string();
  replaceGlobWildcards(complex_path);
  EXPECT_EQ(complex_path,
            (fs::path(fake_directory_) / "*/deep2/../deep2/")
                .make_preferred()
                .string());
}

TEST_F(FilesystemTests, test_simple_globs) {
  std::vector<std::string> results;

  // Test the shell '*', we will support SQL's '%' too.
  auto status = resolveFilePattern(fake_directory_ / "*", results);
  EXPECT_TRUE(status.ok());
  EXPECT_EQ(results.size(), 7U);

  // Test the csh-style bracket syntax: {}.
  results.clear();
  resolveFilePattern(fake_directory_ / "{root,door}*", results);
  EXPECT_EQ(results.size(), 3U);

  // Test a tilde, home directory expansion, make no asserts about contents.
  results.clear();
  resolveFilePattern("~", results);
  if (results.size() == 0U) {
    LOG(WARNING) << "Tilde expansion failed";
  }
}

TEST_F(FilesystemTests, test_wildcard_single_all) {
  // Use '%' as a wild card to glob files within the temporarily-created dir.
  std::vector<std::string> results;
  auto status = resolveFilePattern(fake_directory_ / "%", results, GLOB_ALL);
  EXPECT_TRUE(status.ok());
  EXPECT_EQ(results.size(), 7U);
  EXPECT_TRUE(contains(
      results,
      fs::path(fake_directory_ / "roto.txt").make_preferred().string()));
  EXPECT_TRUE(contains(
      results,
      fs::path(fake_directory_ / "deep11/").make_preferred().string()));
}

TEST_F(FilesystemTests, test_wildcard_single_files) {
  // Now list again with a restriction to only files.
  std::vector<std::string> results;
  resolveFilePattern(fake_directory_ / "%", results, GLOB_FILES);
  EXPECT_EQ(results.size(), 4U);
  EXPECT_TRUE(contains(
      results,
      fs::path(fake_directory_ / "roto.txt").make_preferred().string()));
}

TEST_F(FilesystemTests, test_wildcard_single_folders) {
  std::vector<std::string> results;
  resolveFilePattern(fake_directory_ / "%", results, GLOB_FOLDERS);
  EXPECT_EQ(results.size(), 3U);
  EXPECT_TRUE(contains(
      results,
      fs::path(fake_directory_ / "deep11/").make_preferred().string()));
}

TEST_F(FilesystemTests, test_wildcard_dual) {
  // Now test two directories deep with a single wildcard for each.
  std::vector<std::string> results;
  auto status = resolveFilePattern(fake_directory_ / "%/%", results);
  EXPECT_TRUE(status.ok());
  EXPECT_TRUE(contains(results,
                       fs::path(fake_directory_ / "deep1/level1.txt")
                           .make_preferred()
                           .string()));
}

TEST_F(FilesystemTests, test_wildcard_double) {
  // TODO: this will fail.
  std::vector<std::string> results;
  auto status = resolveFilePattern(fake_directory_ / "%%", results);
  EXPECT_TRUE(status.ok());
  EXPECT_EQ(results.size(), 20U);
  EXPECT_TRUE(contains(results,
                       fs::path(fake_directory_ / "deep1/deep2/level2.txt")
                           .make_preferred()
                           .string()));
}

TEST_F(FilesystemTests, test_wildcard_double_folders) {
  std::vector<std::string> results;
  resolveFilePattern(fake_directory_ / "%%", results, GLOB_FOLDERS);
  EXPECT_EQ(results.size(), 10U);
  EXPECT_TRUE(contains(results,
                       fs::path(fake_directory_ / "deep11/deep2/deep3/")
                           .make_preferred()
                           .string()));
}

TEST_F(FilesystemTests, test_wildcard_end_last_component) {
  std::vector<std::string> results;
  auto status = resolveFilePattern(fake_directory_ / "%11/%sh", results);
  EXPECT_TRUE(status.ok());
  EXPECT_TRUE(contains(
      results,
      fs::path(fake_directory_ / "deep11/not_bash").make_preferred().string()));
}

TEST_F(FilesystemTests, test_wildcard_middle_component) {
  std::vector<std::string> results;

  auto status = resolveFilePattern(fake_directory_ / "deep1%/%", results);

  EXPECT_TRUE(status.ok());
  EXPECT_EQ(results.size(), 5U);
  EXPECT_TRUE(contains(results,
                       fs::path(fake_directory_ / "deep1/level1.txt")
                           .make_preferred()
                           .string()));
  EXPECT_TRUE(contains(results,
                       fs::path(fake_directory_ / "deep11/level1.txt")
                           .make_preferred()
                           .string()));
}

TEST_F(FilesystemTests, test_wildcard_all_types) {
  std::vector<std::string> results;

  auto status = resolveFilePattern(fake_directory_ / "%p11/%/%%", results);
  EXPECT_TRUE(status.ok());
  EXPECT_TRUE(
      contains(results,
               fs::path(fake_directory_ / "deep11/deep2/deep3/level3.txt")
                   .make_preferred()
                   .string()));
}

TEST_F(FilesystemTests, test_wildcard_invalid_path) {
  std::vector<std::string> results;
  auto status = resolveFilePattern("/not_there_abcdefz/%%", results);
  EXPECT_TRUE(status.ok());
  EXPECT_EQ(results.size(), 0U);
}

TEST_F(FilesystemTests, test_wildcard_dotdot_files) {
  std::vector<std::string> results;
  auto status = resolveFilePattern(
      fake_directory_ / "deep11/deep2/../../%", results, GLOB_FILES);
  EXPECT_TRUE(status.ok());
  EXPECT_EQ(results.size(), 4U);

  // The response list will contain canonicalized versions: /tmp/<tests>/...
  std::string door_path =
      fs::path(fake_directory_ / "deep11/deep2/../../door.txt")
          .make_preferred()
          .string();
  replaceGlobWildcards(door_path);
  EXPECT_TRUE(contains(results, door_path));
}

TEST_F(FilesystemTests, test_no_wild) {
  std::vector<std::string> results;
  auto status =
      resolveFilePattern(fake_directory_ / "roto.txt", results, GLOB_FILES);
  EXPECT_TRUE(status.ok());
  EXPECT_EQ(results.size(), 1U);
  EXPECT_TRUE(contains(
      results,
      fs::path(fake_directory_ / "roto.txt").make_preferred().string()));
}

TEST_F(FilesystemTests, test_safe_permissions) {
  fs::path path_1(fake_directory_ / "door.txt");
  fs::path path_2(fake_directory_ / "deep11");

  // For testing we can request a different directory path.
  EXPECT_TRUE(safePermissions(system_root_, path_1));

  // A file with a directory.mode & 0x1000 fails.
  EXPECT_FALSE(safePermissions(tmp_path_, path_1));

  // A directory for a file will fail.
  EXPECT_FALSE(safePermissions(system_root_, path_2));

  // A root-owned file is appropriate
  if (!isPlatform(PlatformType::TYPE_WINDOWS)) {
    EXPECT_TRUE(safePermissions("/", "/dev/zero"));
  }
}

TEST_F(FilesystemTests, test_read_proc) {
  std::string content;

  if (isPlatform(PlatformType::TYPE_LINUX)) {
    fs::path stat_path("/proc/" + std::to_string(platformGetPid()) + "/stat");
    EXPECT_TRUE(readFile(stat_path, content).ok());
    EXPECT_GT(content.size(), 0U);
  }
}

TEST_F(FilesystemTests, test_read_symlink) {
  std::string content;

  if (!isPlatform(PlatformType::TYPE_WINDOWS)) {
    auto status = readFile(fake_directory_ / "root2.txt", content);
    EXPECT_TRUE(status.ok());
    EXPECT_EQ(content, "root");
  }
}

TEST_F(FilesystemTests, create_directory) {
  auto const recursive = false;
  auto const ignore_existence = false;
  const auto tmp_path =
      fs::temp_directory_path() /
      fs::unique_path("osquery.tests.create_directory.%%%%.%%%%");
  ASSERT_FALSE(fs::exists(tmp_path));
  ASSERT_TRUE(createDirectory(tmp_path, recursive, ignore_existence).ok());
  ASSERT_TRUE(fs::exists(tmp_path));
  ASSERT_TRUE(fs::is_directory(tmp_path));
  ASSERT_FALSE(createDirectory(tmp_path).ok());
  fs::remove(tmp_path);
}

TEST_F(FilesystemTests, create_directory_without_parent) {
  auto const recursive = false;
  auto const ignore_existence = false;
  const auto tmp_root_path =
      fs::temp_directory_path() /
      fs::unique_path(
          "osquery.tests.create_directory_without_parent.%%%%.%%%%");
  ASSERT_FALSE(fs::exists(tmp_root_path));
  auto const tmp_path = tmp_root_path / "one_more";
  ASSERT_FALSE(fs::exists(tmp_path));
  ASSERT_FALSE(createDirectory(tmp_path, recursive, ignore_existence).ok());
  ASSERT_FALSE(fs::exists(tmp_path));
  ASSERT_FALSE(fs::is_directory(tmp_path));
  fs::remove_all(tmp_root_path);
}

TEST_F(FilesystemTests, create_directory_recursive) {
  auto const recursive = true;
  auto const ignore_existence = false;
  const auto tmp_root_path =
      fs::temp_directory_path() /
      fs::unique_path("osquery.tests.create_directory_recursive.%%%%.%%%%");
  ASSERT_FALSE(fs::exists(tmp_root_path));
  auto const tmp_path = tmp_root_path / "one_more";
  ASSERT_FALSE(fs::exists(tmp_path));
  ASSERT_TRUE(createDirectory(tmp_path, recursive, ignore_existence).ok());
  ASSERT_TRUE(fs::exists(tmp_path));
  ASSERT_TRUE(fs::is_directory(tmp_path));
  fs::remove_all(tmp_root_path);
}

TEST_F(FilesystemTests, create_directory_recursive_on_existing_dir) {
  auto const recursive = true;
  auto const ignore_existence = false;
  const auto tmp_root_path =
      fs::temp_directory_path() /
      fs::unique_path(
          "osquery.tests.create_directory_recursive_on_existing_dir.%%%%.%%%%");
  auto const tmp_path = tmp_root_path / "one_more";
  fs::create_directories(tmp_path);

  ASSERT_TRUE(fs::exists(tmp_path));
  ASSERT_TRUE(fs::is_directory(tmp_path));
  ASSERT_FALSE(createDirectory(tmp_path, recursive, ignore_existence).ok());
  ASSERT_TRUE(fs::exists(tmp_path));
  ASSERT_TRUE(fs::is_directory(tmp_path));
  fs::remove_all(tmp_root_path);
}

TEST_F(FilesystemTests, create_dir_recursive_ignore_existence) {
  auto const recursive = true;
  auto const ignore_existence = true;
  const auto tmp_root_path =
      fs::temp_directory_path() /
      fs::unique_path(
          "osquery.tests.create_dir_recursive_ignore_existence.%%%%.%%%%");
  auto const tmp_path = tmp_root_path / "one_more";
  fs::create_directories(tmp_path);

  ASSERT_TRUE(fs::exists(tmp_path));
  ASSERT_TRUE(fs::is_directory(tmp_path));
  ASSERT_TRUE(createDirectory(tmp_path, recursive, ignore_existence).ok());
  ASSERT_TRUE(fs::exists(tmp_path));
  ASSERT_TRUE(fs::is_directory(tmp_path));
  fs::remove_all(tmp_root_path);
}

TEST_F(FilesystemTests, test_read_empty_file) {
  auto test_file = test_working_dir_ / "fstests-empty";

  ASSERT_TRUE(writeTextFile(test_file, "").ok());
  ASSERT_TRUE(fs::is_empty(test_file));

  std::string content;
  ASSERT_TRUE(readFile(test_file, content));
  ASSERT_TRUE(content.empty());
}

TEST_F(FilesystemTests, test_directory_listing_with_no_nested_dirs) {
  // This test verifies that directories are properly listed. No nested dir
  // structure is generated. Recursive directory listing flag in
  // listDirectoriesInDirectory() is set to false.
  const unsigned int MAX_DIRS = 450;
  const fs::path test_root_dir = fs::temp_directory_path() / genRandomName();

  ASSERT_TRUE(fs::create_directory(test_root_dir));

  unsigned int created_directories = 0;
  for (unsigned int i = 0; i < MAX_DIRS; ++i) {
    const fs::path work_dir = test_root_dir / genRandomName();
    ASSERT_TRUE(createNestedDirectories(work_dir, 0, created_directories));
  }

  std::vector<std::string> found_directories;
  ASSERT_TRUE(
      listDirectoriesInDirectory(test_root_dir, found_directories, false));

  ASSERT_TRUE(!found_directories.empty());

  ASSERT_TRUE(found_directories.size() == (size_t)created_directories);

  deleteDirectoryContent(test_root_dir);
}

TEST_F(FilesystemTests, test_directory_listing_with_nested_dirs) {
  // This test verifies that directories are properly listed. Nested dir
  // structure is generated. Recursive directory listing flag in
  // listDirectoriesInDirectory() is set to true.
  const unsigned int MAX_DIRS = 450;
  const fs::path test_root_dir = fs::temp_directory_path() / genRandomName();

  ASSERT_TRUE(fs::create_directory(test_root_dir));

  unsigned int created_directories = 0;
  for (unsigned int i = 0; i < MAX_DIRS; ++i) {
    const fs::path work_dir = test_root_dir / genRandomName();
    ASSERT_TRUE(createNestedDirectories(
        work_dir, getRandomNumber(), created_directories));
  }

  std::vector<std::string> found_directories;
  ASSERT_TRUE(
      listDirectoriesInDirectory(test_root_dir, found_directories, true));

  ASSERT_TRUE(!found_directories.empty());

  ASSERT_TRUE(found_directories.size() == (size_t)created_directories);

  deleteDirectoryContent(test_root_dir);
}

TEST_F(FilesystemTests, test_directory_listing_with_dir_symlink) {
  // This test verifies that symlinks are properly listed. Nested dir
  // structure is not generated. Recursive directory listing flag in
  // listDirectoriesInDirectory() is set to false.
  const unsigned int MAX_DIRS = 450;
  const fs::path test_root_raw_dirs =
      fs::temp_directory_path() / genRandomName();
  const fs::path test_root_symlink_dirs =
      fs::temp_directory_path() / genRandomName();

  ASSERT_TRUE(fs::create_directory(test_root_raw_dirs));
  ASSERT_TRUE(fs::create_directory(test_root_symlink_dirs));

  unsigned int created_raw_directories = 0;
  unsigned int created_symlink_directories = 0;
  for (unsigned int i = 0; i < MAX_DIRS; ++i) {
    const fs::path raw_dir = test_root_raw_dirs / genRandomName();
    const fs::path symlink_dir = test_root_symlink_dirs / genRandomName();
    ASSERT_TRUE(createNestedDirectories(raw_dir, 0, created_raw_directories));

    ASSERT_TRUE(createDirectorySymlink(
        raw_dir, symlink_dir, created_symlink_directories));
  }

  std::vector<std::string> found_directories;
  ASSERT_TRUE(listDirectoriesInDirectory(
      test_root_symlink_dirs, found_directories, false));
  ASSERT_TRUE(!found_directories.empty());

  ASSERT_TRUE(found_directories.size() == (size_t)created_symlink_directories);

  deleteDirectoryContent(test_root_raw_dirs);
  deleteDirectoryContent(test_root_symlink_dirs);
}

TEST_F(FilesystemTests, test_directory_listing_with_nested_dirs_and_symlinks) {
  // This test verifies that symlinks and nested directories are properly
  // listed. Nested dir structure is generated. Recursive directory listing flag
  // in listDirectoriesInDirectory() is set to false.
  const unsigned int MAX_DIRS = 450;
  const fs::path test_root_raw_dirs =
      fs::temp_directory_path() / genRandomName();
  const fs::path test_root_work_dirs =
      fs::temp_directory_path() / genRandomName();

  ASSERT_TRUE(fs::create_directory(test_root_raw_dirs));
  ASSERT_TRUE(fs::create_directory(test_root_work_dirs));

  unsigned int created_target_symlink_directories = 0;
  unsigned int created_symlink_directories = 0;
  unsigned int created_raw_directories = 0;

  for (unsigned int i = 0; i < MAX_DIRS; ++i) {
    const fs::path raw_dir = test_root_raw_dirs / genRandomName();
    const fs::path symlink_dir = test_root_work_dirs / genRandomName();
    const fs::path work_dir = test_root_work_dirs / genRandomName();

    ASSERT_TRUE(createNestedDirectories(
        raw_dir, 0, created_target_symlink_directories));

    ASSERT_TRUE(createDirectorySymlink(
        raw_dir, symlink_dir, created_symlink_directories));

    ASSERT_TRUE(createNestedDirectories(
        work_dir, getRandomNumber(), created_raw_directories));
  }

  std::vector<std::string> found_directories;
  ASSERT_TRUE(
      listDirectoriesInDirectory(test_root_work_dirs, found_directories, true));

  ASSERT_TRUE(!found_directories.empty());

  ASSERT_TRUE(found_directories.size() ==
              (size_t)(created_symlink_directories + created_raw_directories));

  deleteDirectoryContent(test_root_raw_dirs);
  deleteDirectoryContent(test_root_work_dirs);
}

#ifdef OSQUERY_WINDOWS
TEST_F(FilesystemTests, test_directory_listing_with_recursive_junction) {
  // This test verifies that a recursive directory junction can be handled by
  // listDirectoriesInDirectory logic

  const fs::path test_root_raw = fs::temp_directory_path() / genRandomName();
  ASSERT_TRUE(fs::create_directory(test_root_raw));
  const fs::path junction_dir = test_root_raw / genRandomName();

  // Creating a junction directory that points to itself
  std::string junction_dir_str = junction_dir.string();
  std::string target_cmdline = "mklink /J ";
  target_cmdline.append(junction_dir_str);
  target_cmdline.append(" ");
  target_cmdline.append(junction_dir_str);
  target_cmdline.append(" > NUL");
  system(target_cmdline.c_str());

  std::vector<std::string> found_directories;
  ASSERT_TRUE(
      listDirectoriesInDirectory(test_root_raw, found_directories, false));
  ASSERT_TRUE(found_directories.empty());

  deleteDirectoryContent(test_root_raw);
}
#endif

TEST_F(FilesystemTests, test_directory_listing_with_legacy_logic) {
  // This test verifies that symlinks and nested directories are properly
  // listed with current and legacy logic.

  const unsigned int MAX_DIRS = 450;
  const fs::path test_root_raw_dirs =
      fs::temp_directory_path() / genRandomName();
  const fs::path test_root_work_dirs =
      fs::temp_directory_path() / genRandomName();

  ASSERT_TRUE(fs::create_directory(test_root_raw_dirs));
  ASSERT_TRUE(fs::create_directory(test_root_work_dirs));

  unsigned int created_target_symlink_directories = 0;
  unsigned int created_symlink_directories = 0;
  unsigned int created_raw_directories = 0;

  for (unsigned int i = 0; i < MAX_DIRS; ++i) {
    const fs::path raw_dir = test_root_raw_dirs / genRandomName();
    const fs::path symlink_dir = test_root_work_dirs / genRandomName();
    const fs::path work_dir = test_root_work_dirs / genRandomName();

    ASSERT_TRUE(createNestedDirectories(
        raw_dir, 0, created_target_symlink_directories));

    ASSERT_TRUE(createDirectorySymlink(
        raw_dir, symlink_dir, created_symlink_directories));

    ASSERT_TRUE(createNestedDirectories(
        work_dir, getRandomNumber(), created_raw_directories));
  }

  std::vector<std::string> found_directories;
  ASSERT_TRUE(
      listDirectoriesInDirectory(test_root_work_dirs, found_directories, true));

  ASSERT_TRUE(!found_directories.empty());

  ASSERT_TRUE(found_directories.size() ==
              (size_t)(created_symlink_directories + created_raw_directories));

  std::vector<std::string> found_directories_legacy_logic;
  ASSERT_TRUE(legacyListDirectoriesInDirectory(
      test_root_work_dirs, found_directories_legacy_logic, true));
  ASSERT_TRUE(found_directories.size() ==
              found_directories_legacy_logic.size());

  deleteDirectoryContent(test_root_raw_dirs);
  deleteDirectoryContent(test_root_work_dirs);
}

TEST_F(FilesystemTests, test_directory_listing_with_file_symlink) {
  // This test verifies that a file symlink is not mistaken for a directory.
  const fs::path test_root_dir = fs::temp_directory_path() / genRandomName();
  ASSERT_TRUE(fs::create_directory(test_root_dir));

  std::ofstream test_file((test_root_dir / "test_file.txt").string());
  test_file.close();

  // Create symlink
  try {
    fs::create_symlink(test_root_dir / "test_file.txt", test_root_dir / "link");
  } catch (const fs::filesystem_error& e) {
    FAIL() << "Error creating symlink: " << e.what();
  }

  std::vector<std::string> found_directories;
  ASSERT_TRUE(
      listDirectoriesInDirectory(test_root_dir, found_directories, false));
  ASSERT_TRUE(found_directories.empty());

  // Test with recursive=true
  ASSERT_TRUE(
      listDirectoriesInDirectory(test_root_dir, found_directories, true));
  ASSERT_TRUE(found_directories.empty());

  deleteDirectoryContent(test_root_dir);
}

TEST_F(FilesystemTests, test_directory_listing_with_bad_symlinks) {
  // This test verifies that bad symlinks are not mistaken for a directory.
  const fs::path test_root_dir = fs::temp_directory_path() / genRandomName();
  ASSERT_TRUE(fs::create_directory(test_root_dir));

  // Create symlink that points to itself.
  try {
    fs::create_symlink(test_root_dir / "link", test_root_dir / "link");
  } catch (const fs::filesystem_error& e) {
    FAIL() << "Error creating symlink: " << e.what();
  }

  // Create symlink that points to non-existent file.
  try {
    fs::create_symlink(test_root_dir / "not_exists.txt",
                       test_root_dir / "link2");
  } catch (const fs::filesystem_error& e) {
    FAIL() << "Error creating symlink: " << e.what();
  }

  std::vector<std::string> found_directories;
  ASSERT_TRUE(
      listDirectoriesInDirectory(test_root_dir, found_directories, false));
  ASSERT_TRUE(found_directories.empty());

  // Test with recursive=true
  ASSERT_TRUE(
      listDirectoriesInDirectory(test_root_dir, found_directories, true));
  ASSERT_TRUE(found_directories.empty());

  deleteDirectoryContent(test_root_dir);
}

} // namespace osquery
