/*
 * Copyright The OpenTelemetry Authors
 * SPDX-License-Identifier: Apache-2.0
 */

package io.opentelemetry.javaagent.instrumentation.cassandra.v4_4;

import static net.bytebuddy.matcher.ElementMatchers.isMethod;
import static net.bytebuddy.matcher.ElementMatchers.isPublic;
import static net.bytebuddy.matcher.ElementMatchers.named;
import static net.bytebuddy.matcher.ElementMatchers.takesArguments;

import io.opentelemetry.javaagent.extension.instrumentation.TypeInstrumentation;
import io.opentelemetry.javaagent.extension.instrumentation.TypeTransformer;
import java.util.concurrent.CompletionStage;
import net.bytebuddy.asm.Advice;
import net.bytebuddy.description.type.TypeDescription;
import net.bytebuddy.matcher.ElementMatcher;

public class SessionBuilderInstrumentation implements TypeInstrumentation {

  @Override
  public ElementMatcher<TypeDescription> typeMatcher() {
    // Note: Cassandra has a large driver and we instrument single class in it.
    // The rest is ignored in AdditionalLibraryIgnoresMatcher
    return named("com.datastax.oss.driver.api.core.session.SessionBuilder");
  }

  @Override
  public void transform(TypeTransformer transformer) {
    transformer.applyAdviceToMethod(
        isMethod().and(isPublic()).and(named("buildAsync")).and(takesArguments(0)),
        SessionBuilderInstrumentation.class.getName() + "$BuildAdvice");
  }

  @SuppressWarnings("unused")
  public static class BuildAdvice {

    /**
     * Strategy: each time we build a connection to a Cassandra cluster, the
     * com.datastax.oss.driver.api.core.session.SessionBuilder.buildAsync() method is called. The
     * opentracing contribution is a simple wrapper, so we just have to wrap the new session.
     *
     * @param stage The fresh CompletionStage to patch. This stage produces session which is
     *     replaced with new session
     */
    @Advice.OnMethodExit(suppress = Throwable.class)
    public static void injectTracingSession(
        @Advice.Return(readOnly = false) CompletionStage<?> stage) {
      stage = stage.thenApply(new CompletionStageFunction());
    }
  }
}
