import json
import numpy as np
import onnxruntime
import sys
import os
from azureml.core.model import Model
import time

def init():
    global session, input_name, output_name
    model = "vit_nsfw.onnx"
    
    # Load the model in onnx runtime to start the session    
    session = onnxruntime.InferenceSession(model, providers=["CPUExecutionProvider"])
    input_name = session.get_inputs()[0].name
    output_name = session.get_outputs()[0].name 
    
def run(input_data):
    '''Purpose: evaluate test input in Azure Cloud using onnxruntime.
        We will call the run function later from our Jupyter Notebook 
        so our azure service can evaluate our model input in the cloud. '''

    try:
        # load in our data, convert to readable format
        data = np.array([np.array(json.loads(input_data)['data']).astype('float32')])
        
        # pass input data to do model inference with ONNX Runtime
        start = time.time()
        r = session.run([output_name], {input_name : data})
        end = time.time()
        
        result = label_map(postprocess(r[0]))
        
        result_dict = {"result": result,
                      "time_in_sec": [end - start]}
    except Exception as e:
        result_dict = {"error": str(e)}
    
    return json.dumps(result_dict)



def label_map(probs, threshold=.5):
    """Take the most probable labels (output of postprocess) and returns the 
    probs of each label."""
    # labels and dictionary to 
    label_dic = {"drawings": 0, "hentai": 1, "neutral": 2, "porn": 3, "sexy": 4}
    output_probs = {}
    image_preds = {}
    for prob, key in zip(probs, range(0, len(probs))):
        label = get_key(label_dic, key)
        output_probs[label] = float(prob)

    prob_nsfw = output_probs["sexy"] + output_probs["hentai"] + output_probs["porn"]
    if prob_nsfw > threshold:
        image_preds["nsfw"] = 1
    else:
        image_preds["nsfw"] = 0

    image_preds["class_probabilities"] = output_probs
    return image_preds

def get_key(dict, value):
    """
    return key given a value. From a dictionary
    """
    for key, val in dict.items():
        if val == value:
            return key
    return "Value not found"

def postprocess(scores):
    """This function takes the scores generated by the network and 
    returns the class IDs in decreasing order of probability."""
    logits = np.array(scores)
    probabilities = np.exp(logits) / np.sum(np.exp(logits), axis=1, keepdims=True)
    probs = probabilities[0]
    
    return probs
