import json
import numpy as np
import onnxruntime
from transformers import VivitImageProcessor
import sys
import os
from azureml.core.model import Model
import time
import av
import base64
import io


def init():
    global session, input_name, output_name, image_processor, label_dic
    model = "./vivit.onnx"
    # Load the model in onnx runtime to start the session    
    session = onnxruntime.InferenceSession(model, providers=["CPUExecutionProvider"])
    input_name = session.get_inputs()[0].name
    output_name = session.get_outputs()[0].name 
    image_processor = VivitImageProcessor.from_pretrained("google/vivit-b-16x2-kinetics400")
    label_dic = {'ApplyEyeMakeup':0, 'ApplyLipstick':1, 'Archery':2, 'BabyCrawling':3, 'BalanceBeam':4, 'BandMarching':5, 'BaseballPitch':6, 'Basketball':7,'BasketballDunk':8, 'BenchPress':9}
    
def run(input_data):
    '''Purpose: evaluate test input in Azure Cloud using onnxruntime.
        We will call the run function later from our Jupyter Notebook 
        so our azure service can evaluate our model input in the cloud. '''

    try:
        # We expect a video im base64 format in the attribute data. Load the data and decode the base64
        requests_json = json.loads(input_data.encode())
        v = base64.b64decode(requests_json.get("data").encode("utf-8"))
        #create container from bytes request
        container= av.open(io.BytesIO(v))
        #Sample Frames
        indices = sample_frame_indices(clip_len=10, frame_sample_rate=1,seg_len=container.streams.video[0].frames)
        # create video with sampled frames
        video = read_video_pyav(container=container, indices=indices)
        # tokenize video
        data = np.array(image_processor(list(video), return_tensors="pt")['pixel_values'])
        #### INFERENCE ONNX #####
        # pass input data to do model inference with ONNX Runtime
        start = time.time()
        r = session.run([output_name], {input_name : data})
        end = time.time()
        probabilities, predicted_class = postprocess(r[0])
        # predicted class and label
        class_label = predicted_class[0]
        label = get_key(label_dic, class_label)
        
        result = label_map(probabilities)
        result['predicted_label'] = label
        result['predicted_calss'] = int(class_label)
        
        result_dict = {"result": result,
                      "time_in_sec": [end - start]}
    except Exception as e:
        result_dict = {"error": str(e)}
    
    return json.dumps(result_dict)



def label_map(probs, threshold=.5):
    """Take the most probable labels (output of postprocess) and returns the 
    probs of each label."""
    # labels and dictionary to 
    label_dic = {'ApplyEyeMakeup':0, 'ApplyLipstick':1, 'Archery':2, 'BabyCrawling':3, 'BalanceBeam':4, 'BandMarching':5, 
             'BaseballPitch':6, 'Basketball':7,'BasketballDunk':8, 'BenchPress':9}
    output_probs = {}
    image_preds = {}
    for prob, key in zip(probs[0], range(0, len(probs[0]))):
        label = get_key(label_dic, key)
        output_probs[label] = float(prob)

    image_preds["class_probabilities"] = output_probs
    return image_preds

def get_key(dict, value):
    """
    return key given a value. From a dictionary
    """
    for key, val in dict.items():
        if val == value:
            return key
    return "Value not found"

def postprocess(scores):
    """This function takes the scores generated by the network and 
    returns the class IDs in decreasing order of probability."""
    logits = np.array(scores)
    probabilities = np.exp(logits) / np.sum(np.exp(logits), axis=1, keepdims=True)
    predicted_class = np.argmax(probabilities, axis=1)
    
    return probabilities, predicted_class


def read_video_pyav(container, indices):
    '''
    Decode the video with PyAV decoder.
    Args:
        container (`av.container.input.InputContainer`): PyAV container.
        indices (`List[int]`): List of frame indices to decode.
    Returns:
        result (np.ndarray): np array of decoded frames of shape (num_frames, height, width, 3).
    '''
    frames = []
    container.seek(0)
    start_index = indices[0]
    end_index = indices[-1]
    for i, frame in enumerate(container.decode(video=0)):
        if i > end_index:
            break
        if i >= start_index and i in indices:
            reformatted_frame = frame.reformat(width=224,height=224)
            frames.append(reformatted_frame)
    new=np.stack([x.to_ndarray(format="rgb24") for x in frames])

    return new


def sample_frame_indices(clip_len, frame_sample_rate, seg_len):
    '''
    Sample a given number of frame indices from the video.
    Args:
        clip_len (`int`): Total number of frames to sample.
        frame_sample_rate (`int`): Sample every n-th frame.
        seg_len (`int`): Maximum allowed index of sample's last frame.
    Returns:
        indices (`List[int]`): List of sampled frame indices
    '''
    converted_len = int(clip_len * frame_sample_rate)
    end_idx = np.random.randint(converted_len, seg_len)
    start_idx = end_idx - converted_len
    indices = np.linspace(start_idx, end_idx, num=clip_len)
    indices = np.clip(indices, start_idx, end_idx - 1).astype(np.int64)
    return indices
