import {Errors, IMAPErrors} from 'isomorphic-core'
import {APIError, NylasAPI, NylasAPIRequest, N1CloudAPI} from 'nylas-exports'

async function ensureGmailAccessToken({account, logger, forceRefresh = false, expiryBufferInSecs = 0} = {}) {
  try {
    const credentials = account.decryptedCredentials()
    if (!credentials) {
      throw new Error("ensureGmailAccessToken: There are no IMAP connection credentials for this account.");
    }

    const currentUnixDate = Math.floor(Date.now() / 1000);
    if (forceRefresh && (credentials.expiry_date > currentUnixDate)) {
      console.warn("ensureGmailAccessToken: refreshing token, but token is not expired");
    }
    const bufferedExpiryDate = credentials.expiry_date - expiryBufferInSecs;
    if (forceRefresh || (currentUnixDate > bufferedExpiryDate)) {
      const req = new NylasAPIRequest({
        api: N1CloudAPI,
        options: {
          path: `/auth/gmail/refresh`,
          method: 'POST',
          accountId: account.emailAddress,
        },
      });

      const newCredentials = await req.run()
      account.setCredentials(newCredentials);
      await account.save();
      return newCredentials;
    }
    return null
  } catch (err) {
    logger.warn(`🔃  Unable to refresh access token.`, err);
    if (err instanceof APIError) {
      const {statusCode} = err
      logger.error(`🔃  Unable to refresh access token. Got status code: ${statusCode}`, err);

      const isNonPermanentError = (
        // If we got a 5xx error from the server, that means that something is wrong
        // on the Nylas API side. It could be a bad deploy, or a bug on Google's side.
        // In both cases, we've probably been alerted and are working on the issue,
        // so it makes sense to have the client retry.
        statusCode >= 500 ||
        !NylasAPI.PermanentErrorCodes.includes(statusCode)
      )
      if (isNonPermanentError) {
        throw new IMAPErrors.IMAPTransientAuthenticationError(`Server error when trying to refresh token.`);
      } else {
        throw new IMAPErrors.IMAPAuthenticationError(`Unable to refresh access token`);
      }
    }
    err.message = `Unknown error when refreshing access token: ${err.message}`
    const fingerprint = ["{{ default }}", "access token refresh", err.message];
    NylasEnv.reportError(err, {fingerprint,
      rateLimit: {
        ratePerHour: 30,
        key: `SyncError:RefreshToken:${err.message}`,
      },
    })
    throw err
  }
}


// Awaiting this function will only wait for the first run. It returns after
// retryable errors, but sets a timeout to run the task again later.
// To await until the end result, use the public runWithRetryLogic() instead.
async function _runWithRetryLogic({run, retryScheduler, onSuccess, onError, onRetryableError}) {
  try {
    const result = await run()
    onSuccess(result)
  } catch (error) {
    if (error instanceof Errors.RetryableError) {
      let delay = 0
      if (retryScheduler) {
        retryScheduler.nextDelay()
        delay = retryScheduler.currentDelay()
      }
      try {
        await onRetryableError(error, delay) // this may re-throw the error to halt the retry process
      } catch (err) {
        onError(err)
        return
      }
      setTimeout(_runWithRetryLogic, delay, {run, retryScheduler, onSuccess, onError, onRetryableError})
    } else {
      onError(error)
    }
  }
}

async function runWithRetryLogic({run, retryScheduler, onRetryableError}) {
  return new Promise((resolve, reject) => _runWithRetryLogic({
    run,
    retryScheduler,
    onRetryableError,
    onSuccess: resolve,
    onError: reject,
  }))
}


module.exports = {
  ensureGmailAccessToken,
  runWithRetryLogic,
  // This folder list was generated by aggregating examples of user folders
  // that were not properly labeled as trash, sent, or spam.
  // This list was constructed semi automatically, and manuallly verified.
  // Should we encounter problems with account folders in the future, add them
  // below to test for them.
  // Make sure these are lower case! (for comparison purposes)
  localizedCategoryNames: {
    trash: new Set([
      'gel\xc3\xb6scht', 'papierkorb',
      '\xd0\x9a\xd0\xbe\xd1\x80\xd0\xb7\xd0\xb8\xd0\xbd\xd0\xb0',
      '[imap]/trash', 'papelera', 'borradores',
      '[imap]/\xd0\x9a\xd0\xbe\xd1\x80',
      '\xd0\xb7\xd0\xb8\xd0\xbd\xd0\xb0', 'deleted items',
      '\xd0\xa1\xd0\xbc\xd1\x96\xd1\x82\xd1\x82\xd1\x8f',
      'papierkorb/trash', 'gel\xc3\xb6schte elemente',
      'deleted messages', '[gmail]/trash', 'inbox/trash', 'trash',
      'mail/trash', 'inbox.trash']),
    spam: new Set([
      'roskaposti', 'inbox.spam', 'inbox.spam', 'skr\xc3\xa4ppost',
      'spamverdacht', 'spam', 'spam', '[gmail]/spam', '[imap]/spam',
      '\xe5\x9e\x83\xe5\x9c\xbe\xe9\x82\xae\xe4\xbb\xb6', 'junk',
      'junk mail', 'junk e-mail']),
    inbox: new Set([
      'inbox',
    ]),
    sent: new Set([
      'postausgang', 'inbox.gesendet', '[gmail]/sent mail',
      '\xeb\xb3\xb4\xeb\x82\xbc\xed\x8e\xb8\xec\xa7\x80\xed\x95\xa8',
      'elementos enviados', 'sent', 'sent items', 'sent messages',
      'inbox.papierkorb', 'odeslan\xc3\xa9', 'mail/sent-mail',
      'ko\xc5\xa1', 'inbox.sentmail', 'gesendet',
      'ko\xc5\xa1/sent items', 'gesendete elemente']),
    archive: new Set([
      'archive',
    ]),
    drafts: new Set([
      'drafts', 'draft', 'brouillons',
    ]),
  },
}
