/*
 * Copyright 2013 Google Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */
/*
 * Author: ncardwell@google.com (Neal Cardwell)
 *
 * Helpers for portably safe access to unaligned multi-byte values.
 */

#ifndef __UNALIGNED_H__
#define __UNALIGNED_H__

#include "types.h"

static inline u32 __get_unaligned_be32(const u8 *p)
{
	return (u32)p[0] << 24 |
	       (u32)p[1] << 16 |
	       (u32)p[2] <<  8 |
	       (u32)p[3];
}

static inline void __put_unaligned_be32(u32 val, u8 *p)
{
	*p++ = val >> 24;
	*p++ = val >> 16;
	*p++ = val >> 8;
	*p++ = val;
}

static inline u32 __get_unaligned_be24(const u8 *p)
{
	return (u32)p[0] << 16 |
	       (u32)p[1] <<  8 |
	       (u32)p[2];
}

static inline void __put_unaligned_be24(u32 val, u8 *p)
{
	*p++ = val >> 16;
	*p++ = val >> 8;
	*p++ = val;
}

static inline u16 __get_unaligned_be16(const u8 *p)
{
	return (u32)p[0] <<  8 |
	       (u32)p[1];
}

static inline void __put_unaligned_be16(u16 val, u8 *p)
{
	*p++ = val >> 8;
	*p++ = val;
}

static inline u32 get_unaligned_be32(const void *p)
{
	return __get_unaligned_be32((const u8 *)p);
}

static inline void put_unaligned_be32(u32 val, void *p)
{
	__put_unaligned_be32(val, p);
}

static inline u32 get_unaligned_be24(const void *p)
{
	return __get_unaligned_be24((const u8 *)p);
}

static inline void put_unaligned_be24(u32 val, void *p)
{
	__put_unaligned_be24(val, p);
}

static inline u16 get_unaligned_be16(const void *p)
{
	return __get_unaligned_be16((const u8 *)p);
}

static inline void put_unaligned_be16(u16 val, void *p)
{
	__put_unaligned_be16(val, p);
}

#endif /* __UNALIGNED_H__ */
