import {AiChat} from '@nlux-dev/react/src';
import {render, waitFor} from '@testing-library/react';
import userEvent from '@testing-library/user-event';
import {act} from 'react';
import {afterEach, beforeEach, describe, expect, it, vi} from 'vitest';
import {adapterBuilder} from '../../../../utils/adapterBuilder';
import {AdapterController} from '../../../../utils/adapters';
import {waitForMdStreamToComplete, waitForReactRenderCycle} from '../../../../utils/wait';

describe('<AiChat /> + batch adapter + events + messageRendered', () => {
    let adapterController: AdapterController | undefined;

    beforeEach(() => {
        adapterController = adapterBuilder()
            .withBatchText(true)
            .withStreamText(false)
            .create();
    });

    afterEach(() => adapterController = undefined);

    describe('When a message is rendered by the markdown renderer', () => {
        it('It should trigger the messageRendered event once rendering is complete', async () => {
            // Arrange
            const messageRenderedCallback = vi.fn();
            const aiChat = (
                <AiChat
                    adapter={adapterController!.adapter}
                    events={{messageRendered: messageRenderedCallback}}
                />
            );

            const {container} = render(aiChat);
            await waitForReactRenderCycle();

            const textArea: HTMLTextAreaElement = container.querySelector('.nlux-comp-composer > textarea')!;
            await userEvent.type(textArea, 'Hello{enter}');
            await waitForReactRenderCycle();

            // Act
            await act(async () => {
                adapterController!.resolve('Yo!');
                await waitForMdStreamToComplete();
            });

            // Assert
            await waitFor(() => expect(messageRenderedCallback).toHaveBeenCalledOnce());
        });
    });

    describe('When messageRendered and messageReceived events are both registered', () => {
        it('It should trigger message rendered after message received', async () => {
            // Arrange
            const messageReceivedCallback = vi.fn();
            const messageRenderedCallback = vi.fn();
            const aiChat = (
                <AiChat
                    adapter={adapterController!.adapter}
                    events={{
                        messageReceived: messageReceivedCallback,
                        messageRendered: messageRenderedCallback,
                    }}
                />
            );

            const {container} = render(aiChat);
            await waitForReactRenderCycle();

            const textArea: HTMLTextAreaElement = container.querySelector('.nlux-comp-composer > textarea')!;
            await userEvent.type(textArea, 'Hello{enter}');
            await waitForReactRenderCycle();

            // Act
            await act(async () => {
                adapterController!.resolve('Yo!');
                await waitForMdStreamToComplete();
            });

            // Assert
            await waitFor(() => {
                expect(messageReceivedCallback).toHaveBeenCalledOnce();
                expect(messageRenderedCallback).toHaveBeenCalledOnce();
                expect(messageReceivedCallback.mock.invocationCallOrder[0])
                    .toBeLessThan(messageRenderedCallback.mock.invocationCallOrder[0]);
            });
        });
    });
});
