#include "client_updater.h"
#include "core_crypto.h"
#include "core_hash.h"
#include "core_socket.h"
#include <assert.h>

namespace core {

WCHAR const updater_t::update_path[] = L"client.dll";


// NOTE: You NEED to replace this with your own public key generated from the signing tool.
// Embedding this into the DLL is the critical to verifying authenticity.
// TODO: Seriously, replace this with your own!

BYTE const updater_t::public_key[] = {
	0x06, 0x02, 0x00, 0x00, 0x00, 0xA4, 0x00, 0x00, 0x52, 0x53, 0x41, 0x31,
	0x00, 0x10, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x39, 0x69, 0x75, 0x41,
	0xC8, 0x1F, 0xB4, 0x1F, 0xD4, 0x38, 0x2D, 0x3A, 0xB0, 0x0D, 0x57, 0xD3,
	0x49, 0x55, 0xD3, 0xDA, 0x29, 0x64, 0xEF, 0xBD, 0xDF, 0xCD, 0x88, 0x43,
	0x8E, 0xBA, 0x73, 0xB7, 0x9C, 0x65, 0xB0, 0x26, 0x8D, 0xE4, 0x50, 0xCD,
	0x73, 0xB4, 0xD1, 0x26, 0x45, 0x70, 0x75, 0x1C, 0x67, 0x9B, 0x3F, 0xE7,
	0xCF, 0xE2, 0x7E, 0x1A, 0x6F, 0xA6, 0xC3, 0xEC, 0xFA, 0xC0, 0xE3, 0x9F,
	0x4F, 0xBC, 0x7E, 0xAE, 0xD5, 0x94, 0x6B, 0x58, 0x56, 0x92, 0xC8, 0x4B,
	0xCB, 0x8C, 0xD9, 0x86, 0x54, 0x14, 0x12, 0xD3, 0x8F, 0x67, 0xEA, 0x72,
	0xC7, 0x8A, 0x81, 0x16, 0xFF, 0xEB, 0xC2, 0x43, 0xA3, 0x31, 0xB2, 0x1F,
	0x6F, 0x81, 0xE0, 0x61, 0xA0, 0xAB, 0xD3, 0x66, 0x46, 0xFA, 0x61, 0x22,
	0xE3, 0xAB, 0x4B, 0xC7, 0x36, 0x39, 0xFA, 0xDD, 0xC7, 0xC7, 0xE7, 0x3A,
	0x30, 0x41, 0xF9, 0xD1, 0xF1, 0x28, 0xF2, 0xA9, 0x74, 0xE0, 0x01, 0xDD,
	0x9F, 0x58, 0xE7, 0x82, 0x4D, 0x78, 0x03, 0x9A, 0x42, 0x6A, 0xCD, 0x77,
	0x97, 0xFB, 0x26, 0x1B, 0x91, 0xF9, 0x5D, 0x64, 0x66, 0x57, 0x19, 0xA8,
	0xB5, 0x7F, 0x3A, 0xA8, 0x2C, 0x4D, 0xE5, 0x79, 0xB1, 0x9B, 0x9E, 0xFF,
	0xBD, 0x21, 0x15, 0xE1, 0xC1, 0x12, 0x1C, 0xB4, 0xF4, 0x37, 0xDD, 0x2F,
	0x21, 0x35, 0xFD, 0xCC, 0x47, 0x6E, 0x01, 0x3E, 0x24, 0x65, 0x63, 0xCA,
	0x4B, 0x8A, 0x83, 0xAB, 0x73, 0x7A, 0x8E, 0x7C, 0xDC, 0x4B, 0x42, 0xA4,
	0xD6, 0x97, 0x3E, 0xCB, 0x36, 0xD7, 0x7B, 0x9A, 0xE8, 0x63, 0xE0, 0x2E,
	0x4E, 0xB2, 0x45, 0x62, 0xEF, 0x9F, 0x1C, 0x4B, 0x5F, 0x87, 0x64, 0x4A,
	0x1F, 0x8D, 0xB6, 0xDA, 0x2C, 0xC6, 0x9C, 0xC0, 0x81, 0x67, 0x42, 0xC3,
	0x90, 0x62, 0x5C, 0x1D, 0x39, 0xBF, 0x7C, 0x44, 0x79, 0xE4, 0x7B, 0x49,
	0x5D, 0xAB, 0xA5, 0xEC, 0x7B, 0xD9, 0xFF, 0x75, 0x55, 0x7C, 0xDF, 0xAD,
	0xF7, 0x25, 0x8E, 0x21, 0xAC, 0x56, 0xF3, 0x27, 0x2C, 0x1F, 0xA6, 0x4D,
	0xB5, 0xC9, 0x15, 0x76, 0x31, 0x30, 0x0A, 0xCF, 0x86, 0x9A, 0x60, 0x50,
	0x63, 0x86, 0xB5, 0xDE, 0x8D, 0x82, 0x77, 0x21, 0xEF, 0x99, 0x9F, 0x27,
	0xB3, 0xAA, 0x66, 0x2E, 0x1C, 0x68, 0x23, 0x31, 0xBA, 0x84, 0x13, 0x95,
	0x81, 0x7A, 0xE7, 0x1E, 0x3F, 0xEA, 0x74, 0x56, 0x55, 0x8B, 0x34, 0x99,
	0x9F, 0x9C, 0xE9, 0x92, 0xE3, 0xBF, 0x9F, 0xB8, 0x2B, 0xA3, 0xB3, 0x9C,
	0xFF, 0xA0, 0x54, 0xD7, 0x84, 0xB9, 0x4A, 0xE0, 0xA5, 0x4F, 0xCD, 0xE6,
	0xB3, 0xC4, 0x3D, 0xE6, 0x43, 0x2A, 0x20, 0xAD, 0xB2, 0xC5, 0x2F, 0x1B,
	0x75, 0x3F, 0xAF, 0xBE, 0x69, 0x8C, 0x2E, 0x54, 0x93, 0x21, 0x23, 0x6D,
	0x0D, 0x28, 0x57, 0x53, 0xCF, 0x76, 0xCB, 0x1E, 0x3B, 0x47, 0x0B, 0x2E,
	0x49, 0x75, 0xEA, 0xC0, 0x34, 0xDA, 0x98, 0x0D, 0x6E, 0x5D, 0xA6, 0x99,
	0xB7, 0x31, 0xA8, 0x31, 0x2A, 0x0D, 0x77, 0x9C, 0x7D, 0xD3, 0x2B, 0x0F,
	0x66, 0xD1, 0xB4, 0xF1, 0x19, 0x7B, 0x5F, 0x6A, 0x4A, 0x88, 0xE1, 0x37,
	0x44, 0x47, 0xF1, 0x28, 0x23, 0x9C, 0x69, 0x56, 0x38, 0x15, 0xEE, 0x19,
	0x37, 0xB4, 0x54, 0x0D, 0xD3, 0x19, 0x14, 0xBE, 0x55, 0x6F, 0xC8, 0x0B,
	0xEA, 0xCE, 0xF3, 0x4B, 0x62, 0x02, 0x95, 0x57, 0x84, 0x6E, 0xF9, 0xDD,
	0xC6, 0x6D, 0x96, 0x21, 0x3B, 0x6A, 0xE1, 0xBF, 0xB7, 0x7C, 0x54, 0xF1,
	0x60, 0x48, 0x39, 0x69, 0x5D, 0x1B, 0x4B, 0x84, 0xDC, 0xB2, 0x28, 0x1C,
	0x07, 0x54, 0xE1, 0x99, 0x8E, 0x08, 0x18, 0xFF, 0xB6, 0x5F, 0xB6, 0x2C,
	0x06, 0x95, 0x0E, 0x12, 0x82, 0xDC, 0x83, 0xDE, 0xA0, 0x08, 0x3D, 0xC9,
	0x4D, 0x9D, 0x10, 0xD7
};

updater_t::updater_t(crypto_t* crypto, socket_t* socket) {
	this->crypto = crypto;
	this->socket = socket;

	last_update_ms = 0;
	last_piece_ms = 0;
	status = {};

	client_version = 0;

	// In your real application, you'd want to get the version here. For demo purposes, I don't.
	#if 1
	filesystem::buffer_t self;
	if (filesystem::read(&self, update_path)) {
		client_version = hash::crc32(self.data, self.size);
	}
	#endif

	reset();
}

updater_t::~updater_t() {
}

updater_t::status_t updater_t::get_status() {
	return status;
}

void updater_t::handle_msg_token(BYTE const* buf, u32 len) {
	assert(buf);

	if (len != sizeof(msg::server_update_token_t)) {
		return;
	}

	auto msg = (msg::server_update_token_t*)buf;
	if (msg->client_token != client_token) {
		return;
	}

	server_token = msg->server_token;
}

void updater_t::handle_msg_begin(BYTE const* buf, u32 len) {
	assert(buf);

	// Wait until the application restarts before trying to do another update.
	if (is_finished) {
		return;
	}

	if (len != sizeof(msg::server_update_begin_t)) {
		return;
	}

	auto msg = (msg::server_update_begin_t*)buf;

	// If the version is the same we don't need to update.
	if (msg->server_version == client_version) {
		return;
	}

	// Verify that the update size is reasonable (<100MB).
	if (msg->update_size == 0 || msg->update_size >= 100000000) {
		return;
	}

	// Allocate space for update data.
	if (update_data.alloc(msg->update_size) == false) {
		return;
	}

	// Allocate space for the piece tracker table.
	if (update_pieces.alloc((msg->update_size + msg::PIECE_BYTES - 1) / msg::PIECE_BYTES) == false) {
		return;
	}

	server_version = msg->server_version;
	memcpy(&update_sig, &msg->update_sig, sizeof(update_sig));

	status.bytes = 0;
	status.total = update_data.size;

	is_updating = true;
}

void updater_t::handle_msg_piece(BYTE const* buf, u32 len) {
	assert(buf);

	// Make sure we're in the middle of an update.
	if (is_updating == false) {
		return;
	}

	if (len < sizeof(msg::server_update_piece_t)) {
		return;
	}

	auto msg = (msg::server_update_piece_t*)buf;

	// Check that the version matches.
	if (msg->server_version != server_version) {
		return;
	}

	// Verify that the tokens match.
	if (msg->client_token != client_token || msg->server_token != server_token) {
		return;
	}

	// Verify that the message piece size is valid.
	if (msg->piece_size > msg::PIECE_BYTES) {
		return;
	}

	// Verify that the message contains the full piece data.
	if (len != (sizeof(msg::server_update_piece_t) + msg->piece_size)) {
		return;
	}


	// Verify that the position is on a piece boundry and something we actually requested.
	if ((msg->piece_pos % msg::PIECE_BYTES) != 0) {
		return;
	}

	// Verify that the data doesn't write outside the buffer.
	if (msg->piece_pos + msg->piece_size > update_data.size) {
		return;
	}

	// Verify the piece position.
	u32 idx = msg->piece_pos / msg::PIECE_BYTES;
	if (idx >= update_pieces.size) {
		return;
	}

	// Verify that we need the piece.
	if (update_pieces.ptr[idx]) {
		return;
	}

	// Validate that the piece size aligns with how we request data.
	if (idx < update_pieces.size - 1) {
		if (msg->piece_size != msg::PIECE_BYTES) {
			return;
		}
	}

	memcpy(update_data.ptr + msg->piece_pos, buf + sizeof(msg::server_update_piece_t), msg->piece_size);
	update_pieces.ptr[idx] = 1;
	status.bytes += msg->piece_size;
}

bool updater_t::needs_restart() {
	return is_finished;
}

void updater_t::reset() {
	update_data.free();
	update_pieces.free();

	is_finished = false;
	is_updating = false;

	client_token = 0;
	server_token = 0;
	server_version = 0;

	update_idx = 0;
}

void updater_t::update(i64 now_ms, addr_t host) {
	// Check if we're done updating.
	if (is_finished) {
		return;
	}

	// Check if we need to start the update process.
	if (is_updating == false) {
		if (now_ms - last_update_ms >= msg::freq_client_update_begin_ms) {
			last_update_ms = now_ms;

			client_token = crypto->gen_token();

			msg::client_update_begin_t msg = {};
			msg.header.version = msg::VERSION;
			msg.header.type = msg::TYPE_CLIENT_UPDATE_BEGIN;
			msg.client_version = client_version;
			msg.client_token = client_token;
			msg.server_token = server_token;

			socket->send(&msg, sizeof(msg), host);

			status.code = STATUS_NONE;
		}

		return;
	}

	// We're updating, so check if we're finished.
	if (update_idx >= update_pieces.size) {
		u32 crc = hash::crc32(update_data.ptr, update_data.size);
		if (crc == server_version) {
			if (crypto->test_sig(update_sig.data, sizeof(update_sig.data), update_data.ptr, update_data.size, public_key, sizeof(public_key))) {
				if (filesystem::write(update_data.ptr, update_data.size, update_path)) {
					is_finished = true;
					return;
				} else {
					status.code = STATUS_BAD_WRITE;
				}
			} else {
				status.code = STATUS_BAD_SIG;
			}
		} else {
			status.code = STATUS_BAD_CRC;
		}

		reset();
		return;
	}

	// Update in progress, request pieces.
	if (now_ms - last_piece_ms >= msg::freq_client_update_piece_ms) {
		last_piece_ms = now_ms;

		msg::client_update_piece_t msg = {};
		msg.header.version = msg::VERSION;
		msg.header.type = msg::TYPE_CLIENT_UPDATE_PIECE;
		msg.client_token = client_token;
		msg.server_token = server_token;
		msg.server_version = server_version;

		bool found_missing = false;
		u32 end_idx = 0;
		u32 num_pieces = 0;

		// Request missing piecees.
		for (u32 idx = update_idx, end = update_pieces.size; idx < end; ++idx) {
			if (update_pieces.ptr[idx] == FALSE) {
				// Keep track of the last valid idx.
				if (found_missing == false) {
					found_missing = true;
					end_idx = idx;
				}

				msg.piece_pos = idx * msg::PIECE_BYTES;
				socket->send(&msg, sizeof(msg), host);

				num_pieces += 1;
				if (num_pieces > max_requests) {
					// Don't send too many.
					break;
				}
			}
		}

		if (end_idx > update_idx) {
			update_idx = end_idx;
		}

		if (found_missing == false) {
			update_idx = update_pieces.size;
		}
	}
}


inline updater_t::buffer_t::buffer_t() : ptr(nullptr), size(0) {
}

inline updater_t::buffer_t::~buffer_t() {
	free();
}

inline bool updater_t::buffer_t::alloc(u32 bytes) {
	if (ptr == nullptr || bytes > size) {
		free();

		ptr = (BYTE*)heap::raw_calloc(bytes);
		if (ptr) {
			size = bytes;
			return true;
		}
	}

	return false;
}

inline void updater_t::buffer_t::free() {
	if (ptr) {
		memset(ptr, 0, size);
		heap::raw_free(ptr);
		ptr = nullptr;
	}

	size = 0;
}

}
