import cl from "clsx";
import React, { forwardRef, useContext } from "react";
import { ChevronDownIcon } from "@navikt/aksel-icons";
import { useThemeInternal } from "../theme/Theme";
import { Heading } from "../typography";
import { composeEventHandlers } from "../util/composeEventHandlers";
import { AccordionContext } from "./AccordionContext";
import { AccordionItemContext } from "./AccordionItem";

export interface AccordionHeaderProps
  extends React.ButtonHTMLAttributes<HTMLButtonElement> {
  /**
   * Text inside Accordion.Header
   */
  children: React.ReactNode;
}

const AccordionHeader = forwardRef<HTMLButtonElement, AccordionHeaderProps>(
  ({ children, className, onClick, ...rest }, ref) => {
    const itemContext = useContext(AccordionItemContext);
    const accordionContext = useContext(AccordionContext);

    const themeContext = useThemeInternal(false);

    if (itemContext === null) {
      console.error(
        "<Accordion.Header> has to be used within an <Accordion.Item>, which in turn must be within an <Accordion>",
      );
      return null;
    }

    let headingSize = accordionContext?.headingSize ?? "small";

    if (themeContext) {
      /* Fallback to "medium" Accordion-size if any other sizes are used */
      headingSize = accordionContext?.size === "small" ? "xsmall" : "small";
    }

    return (
      <button
        ref={ref}
        {...rest}
        className={cl("navds-accordion__header", className)}
        onClick={composeEventHandlers(onClick, itemContext.toggleOpen)}
        aria-expanded={itemContext.open}
        type="button"
      >
        <span className="navds-accordion__icon-wrapper">
          <ChevronDownIcon
            className="navds-accordion__header-chevron"
            aria-hidden
          />
        </span>
        <Heading
          size={headingSize}
          as="span"
          className="navds-accordion__header-content"
        >
          {children}
        </Heading>
      </button>
    );
  },
);

export default AccordionHeader;
