import * as React from 'react';
import { styled, useTheme } from '@mui/material/styles';
import clsx from 'clsx';

const Continent = styled('g')(({ theme }) => ({
  cursor: 'pointer',
  opacity: 0.9,
  '&:hover ': { opacity: 1 },
  '&.selected': {
    fill: theme.palette.primary.main,
  },
}));

export type ContinentClickHandler = (event: React.MouseEvent<SVGElement>, timezone: string) => void;

type WorldMapSvgProps = {
  onClickContinent: ContinentClickHandler;
  selectedTimezone: string | null;
};

export default function WorldMapSvg({ onClickContinent, selectedTimezone }: WorldMapSvgProps) {
  const brandingTheme = useTheme();

  const getMapBaseColor = (lightness: number, opacity: number = 1) => {
    return brandingTheme.palette.mode === 'light'
      ? `hsla(210,20%,${lightness}%, ${opacity * 100}%)`
      : `hsla(210,20%,${lightness}%, ${opacity * 100}%)`;
  };

  const timezones: { [key: string]: string } = {
    europe: 'Europe/Paris',
    asia: 'Asia/Hong_kong',
    southAmerica: 'America/Sao_Paulo',
    northAmerica: 'America/Chicago',
    africa: 'Africa/Casablanca',
    australia: 'Australia/Brisbane',
  };
  // TODO: simplify SVG
  return (
    <svg viewBox="0 0 512 269" fill="none" xmlns="http://www.w3.org/2000/svg">
      <Continent
        className={clsx('northAmerica', selectedTimezone === timezones.northAmerica && 'selected')}
        fill={getMapBaseColor(56)}
        onClick={(event) => onClickContinent(event, timezones.northAmerica)}
      >
        <path d="M211.655 0.117841C209.929 0.211148 207.083 0.351108 205.357 0.421089C203.631 0.491069 201.904 0.631029 201.508 0.724336C201.135 0.794317 200.551 0.84097 200.225 0.817643C200.052 0.796916 199.958 1.01355 200.091 1.12544L200.225 1.23752C200.691 1.61075 200.738 1.70406 200.411 1.72739C200.202 1.72739 199.945 1.61075 199.852 1.47079C199.782 1.33083 199.502 1.26085 199.268 1.33083C199.012 1.40081 198.662 1.3075 198.499 1.12089C198.102 0.747663 197.799 0.677683 198.009 1.02758C198.102 1.16754 197.799 1.26085 197.309 1.26085C196.609 1.26085 196.469 1.33083 196.586 1.63408C196.703 1.93733 196.633 1.98398 196.213 1.84402C194.906 1.47079 194.067 1.47079 193.763 1.86735C193.483 2.2639 193.46 2.2639 193.46 1.84402C193.46 1.44747 193.274 1.42414 190.824 1.5641C189.378 1.65741 187.512 1.77404 186.695 1.84402C185.856 1.914 185.319 1.914 185.482 1.82069C186.042 1.54077 185.739 1.26085 184.876 1.26085C184.386 1.26085 183.873 1.14422 183.71 0.98093C183.523 0.794317 181.144 0.70101 175.452 0.677683C167.311 0.631029 164.675 0.77099 163.019 1.33083C162.366 1.5641 161.829 1.58743 161.106 1.44747C160.453 1.3075 159.683 1.3075 158.867 1.49412C158.167 1.63408 156.767 1.79737 155.764 1.84402C153.082 1.98398 151.472 2.21725 151.472 2.45052C151.472 2.56715 151.682 2.66046 151.939 2.66046C152.195 2.66046 152.405 2.80042 152.405 2.98703C152.405 3.15032 152.755 3.40691 153.175 3.54687C153.688 3.71016 153.781 3.78014 153.455 3.80347C153.175 3.82679 152.825 3.96675 152.662 4.10671C152.428 4.36331 152.382 4.33998 152.452 4.08339C152.498 3.89677 152.288 3.73349 151.962 3.68683C151.635 3.61685 151.192 3.38358 150.959 3.12699C150.632 2.77709 150.212 2.66046 149.373 2.66046C148.743 2.66046 148.16 2.75376 148.09 2.89372C148.02 3.01036 147.576 3.12699 147.133 3.12699C146.69 3.12699 146.34 3.2203 146.34 3.36026C146.34 3.47689 146.107 3.59353 145.827 3.61685C145.524 3.61685 144.824 3.80347 144.241 4.03673C143.424 4.38663 143.214 4.57325 143.214 5.01646C143.191 5.45966 143.354 5.59963 143.891 5.66961C144.404 5.71626 144.474 5.78624 144.194 5.90287C143.821 6.04283 143.821 6.08949 144.241 6.626C144.661 7.18584 144.754 7.20917 147.786 7.2325C150.935 7.25582 152.545 7.55907 151.612 7.9323C151.309 8.04893 151.122 8.00228 151.052 7.83899C150.959 7.55907 149.233 7.44244 148.603 7.69903C148.393 7.76901 148.3 8.00228 148.393 8.30553C148.486 8.72541 148.439 8.77206 148.02 8.65543C147.74 8.58545 146.923 8.67875 146.223 8.84204C146.175 8.85337 146.127 8.86426 146.079 8.87468C145.269 9.05254 143.774 8.61857 143.774 9.44854C143.774 9.5885 143.891 9.61182 144.054 9.51852C144.217 9.40188 144.241 9.44854 144.147 9.63515C144.007 9.84509 143.867 9.82176 143.541 9.54184C143.214 9.2386 142.864 9.19194 142.071 9.30858C141.395 9.40188 140.975 9.37856 140.882 9.21527C140.788 9.07531 140.462 8.95868 140.159 8.95868C139.855 8.95868 139.552 8.86537 139.482 8.74874C139.342 8.51547 137.196 8.70208 137.009 8.95868C136.939 9.05198 136.729 9.19194 136.543 9.26192C136.263 9.35523 136.263 9.40188 136.59 9.40188C137.657 9.46469 136.652 10.8718 135.584 10.935C135.546 10.9372 135.508 10.9394 135.47 10.9414L134.766 10.9924C134.471 11.0138 134.316 10.6492 134.537 10.4516C135.143 9.89175 135.143 9.79844 134.817 9.44854C134.513 9.14529 134.35 9.14529 133.977 9.37856C133.417 9.72846 133.207 9.72846 133.394 9.42521C133.604 9.07531 131.994 9.14529 130.571 9.56517C129.778 9.79844 129.312 10.055 129.312 10.265C129.312 10.4516 129.195 10.5449 129.078 10.4749C128.938 10.3816 128.892 10.2416 128.962 10.125C129.148 9.82177 127.982 9.19194 127.539 9.35523C126.979 9.56517 126.209 10.3583 126.559 10.3583C126.743 10.3583 126.559 10.6245 126.676 10.7668C126.879 11.0142 127.595 11.2741 128.285 11.2914C129.078 11.2914 129.265 11.3613 129.218 11.6879C129.195 12.0611 129.405 12.1078 130.921 12.0378C132.204 11.9912 132.717 11.8745 132.927 11.5946C133.114 11.338 133.347 11.268 133.65 11.408C133.954 11.5246 134 11.6179 133.79 11.6879C133.557 11.7812 133.604 11.8978 133.907 12.1311C134.169 12.3213 133.646 12.4518 133.77 12.7503C133.81 12.8441 133.92 12.9242 134.094 12.9242C134.28 12.9242 134.443 12.8309 134.443 12.7143C134.443 12.621 134.957 12.5743 135.586 12.6443C136.613 12.7376 136.823 12.6676 137.383 12.1078C137.896 11.5946 137.942 11.4313 137.686 11.2447C137.429 11.0814 137.429 10.9881 137.663 10.8948C137.803 10.8481 137.942 10.6382 137.942 10.4516C137.942 9.96173 139.949 10.0084 140.135 10.4982C140.252 10.7615 140.093 10.9831 139.83 11.1C139.744 11.1384 139.649 11.1799 139.552 11.2214C138.992 11.4546 138.456 12.411 138.829 12.481C139.552 12.621 140.718 12.691 144.311 12.8309C147.296 12.9475 148.533 12.9009 148.766 12.7143C149.023 12.5043 149.139 12.5043 149.256 12.7143C149.443 12.9942 151.985 12.8542 153.338 12.4577C154.085 12.2477 154.738 11.7112 154.738 11.3147C154.738 10.5915 151.519 10.4283 148.346 10.9648C147.32 11.1281 146.41 11.2214 146.293 11.1514C146.177 11.0814 145.617 10.9881 145.034 10.9648C143.984 10.8948 142.958 10.3583 143.914 10.3583C144.194 10.3583 144.474 10.2883 144.567 10.1717C144.661 10.0784 146.084 9.98505 147.693 9.98505C149.326 9.96173 151.285 9.9384 152.055 9.91507C152.825 9.89175 153.665 9.86842 153.921 9.89175C154.388 9.91507 154.388 9.91507 153.921 10.1483C153.595 10.2883 153.875 10.335 154.854 10.2416C155.881 10.1717 156.114 10.195 155.741 10.3583C155.041 10.6382 155.064 10.8248 155.811 10.8248C156.207 10.8248 156.464 10.6849 156.534 10.3583C156.604 10.0784 156.86 9.89175 157.187 9.89175C158.05 9.89175 159.17 9.40188 159.17 9.02866C159.17 8.77206 159.007 8.70208 158.563 8.81872C158.167 8.93535 158.027 8.8887 158.12 8.72541C158.19 8.60878 158.493 8.49214 158.773 8.49214C159.053 8.49214 159.613 8.32885 159.986 8.11891C160.383 7.90897 161.223 7.69903 161.852 7.65238C163.485 7.51242 163.998 7.3258 163.579 7.06921C163.369 6.92925 163.509 6.90592 163.975 6.99923C164.418 7.09254 164.768 7.02256 164.862 6.8826C164.955 6.74264 165.211 6.626 165.445 6.626C165.678 6.626 166.098 6.36941 166.378 6.06616C166.914 5.50632 167.521 5.34303 170.11 5.08644C170.81 5.03978 171.72 4.80652 172.139 4.6199C172.559 4.40996 173.026 4.22335 173.213 4.20002C178.088 3.3136 179.417 3.01036 179.161 2.84707C179.021 2.75376 179.277 2.73044 179.744 2.82374C180.234 2.89372 180.747 2.84707 180.91 2.70711C181.237 2.47384 184.596 1.86735 184.759 2.05396C184.806 2.10061 184.689 2.31055 184.479 2.5205C184.223 2.77709 183.663 2.89372 182.776 2.89372C181.004 2.91705 176.152 3.87345 176.012 4.24667C175.965 4.43329 176.245 4.5266 176.991 4.5266C177.575 4.5266 178.064 4.6199 178.064 4.75986C178.064 5.20307 176.968 5.52964 174.799 5.71626C171.393 5.99618 167.731 6.55602 167.031 6.92925L166.986 6.95269C166.753 7.07329 166.766 7.41033 167.008 7.51242C167.334 7.67571 167.777 7.79234 168.034 7.79234C168.436 7.79234 168.435 8.09275 168.034 8.11471C167.859 8.12426 167.653 8.12644 167.427 8.11891C166.728 8.09559 166.098 8.16557 166.028 8.2822C165.958 8.39883 166.168 8.49214 166.494 8.49214C166.891 8.49214 167.054 8.58545 166.961 8.79539C166.751 9.3319 168.594 10.4049 169.947 10.5449C170.763 10.6149 171.09 10.5682 170.973 10.4049C170.88 10.2416 171.043 10.2183 171.556 10.3583C171.976 10.4749 172.373 10.5216 172.489 10.4516C173.049 10.1017 176.455 10.195 178.158 10.6149C180.001 11.0581 180.584 11.2914 179.861 11.2914C179.324 11.2914 179.604 11.9212 180.234 12.0845C180.724 12.2244 180.724 12.2477 180.327 12.551C179.907 12.8776 179.907 12.8776 180.327 12.9942C180.654 13.0875 180.63 13.1108 180.211 13.1342C179.907 13.1342 179.721 13.2508 179.814 13.3908C179.884 13.5074 180.117 13.554 180.35 13.4841C180.607 13.3908 180.56 13.5307 180.164 13.8806L179.997 14.0275C179.893 14.1192 179.983 14.2889 180.117 14.2538C180.654 14.1372 180.654 14.1372 180.257 14.5804C179.884 14.977 179.884 15.0936 180.164 15.2569C180.444 15.4435 180.397 15.4902 180.001 15.4902C179.417 15.4902 179.277 15.8401 179.767 16.05C179.954 16.12 179.837 16.3533 179.394 16.7032C178.834 17.193 178.788 17.3096 179.068 17.7062C179.324 18.0328 179.581 18.1028 180.234 17.9861C180.7 17.9161 181.144 17.9395 181.214 18.0794C181.284 18.1961 181.19 18.2194 181.004 18.1494C180.794 18.0794 180.63 18.1261 180.63 18.266C180.63 18.406 180.84 18.5226 181.12 18.5226C181.447 18.5226 181.563 18.3827 181.517 18.1028C181.493 17.8928 181.563 17.7062 181.68 17.7062C181.797 17.7062 181.89 17.8462 181.843 17.9861C181.797 18.1727 182.007 18.2894 182.31 18.2894C182.706 18.2894 182.87 18.4293 182.893 18.7559C182.94 19.2458 182.637 19.3857 182.38 18.9892C182.233 18.7507 180.558 18.6998 179.769 18.8249C179.486 18.8699 179.254 19.1875 179.511 19.3158C179.674 19.3857 179.511 19.4091 179.161 19.3391C178.788 19.2691 178.554 19.2924 178.624 19.4091C178.694 19.5257 178.554 19.8989 178.298 20.2022C178.063 20.5104 178.155 20.9546 178.493 21.1444L178.601 21.2052C179.277 21.5551 179.487 21.5551 180.607 21.2752C182.053 20.9253 182.31 20.5987 181.424 20.1788C181.355 20.1513 181.291 20.121 181.234 20.0896C180.957 19.9363 181.153 19.6514 181.377 19.8756C181.54 20.0389 181.89 20.1555 182.17 20.1555C182.823 20.1555 182.846 20.902 182.217 21.835C181.82 22.4182 181.68 22.4415 180.677 22.3249C179.627 22.2083 179.511 22.2549 178.951 22.9314C178.601 23.3279 178.274 23.6545 178.181 23.6545C177.925 23.6545 176.898 24.8908 176.898 25.1941C176.898 25.3807 177.085 25.5207 177.318 25.544C177.621 25.544 177.598 25.5906 177.201 25.7539C176.618 25.9872 176.082 26.827 176.408 27.0136C176.525 27.0836 176.572 27.3868 176.478 27.6667C176.385 28.0633 176.432 28.1333 176.781 28.0166C177.085 27.9 177.155 27.9466 177.038 28.2032C176.968 28.3899 176.991 28.5531 177.108 28.5531C177.225 28.5531 177.178 28.9497 177.015 29.4162C176.758 30.186 176.758 30.326 177.061 30.5126C177.318 30.6525 177.341 30.7459 177.155 30.8625C176.805 31.0724 177.225 32.2854 177.621 32.2854C177.785 32.2854 177.831 32.542 177.761 32.9852C177.668 33.3584 177.691 33.685 177.808 33.685C177.901 33.685 178.111 34.1049 178.274 34.5948C178.414 35.0846 178.694 35.5745 178.881 35.6445C179.068 35.7145 179.231 35.9944 179.231 36.251C179.231 36.7642 179.511 36.8341 180.117 36.5076C180.677 36.2276 182.73 36.8575 182.73 37.324C182.73 37.5106 182.916 37.7206 183.15 37.7905C183.36 37.8605 183.78 38.0238 184.059 38.1638C184.713 38.4903 185.809 38.1171 186.112 37.464C186.229 37.184 186.462 36.9508 186.625 36.9508C186.765 36.9508 186.952 36.7408 187.045 36.4842C187.115 36.251 187.535 35.6911 187.955 35.2712C188.375 34.8514 188.795 34.1982 188.888 33.8483C188.981 33.4518 189.285 33.1019 189.635 32.9852C189.938 32.8686 190.498 32.4954 190.848 32.1688C191.197 31.8422 191.664 31.5156 191.897 31.4457C192.107 31.3757 192.294 31.1657 192.294 31.0024C192.294 30.8158 192.527 30.4659 192.784 30.2093C193.11 29.9061 193.25 29.5795 193.157 29.2763C193.087 29.0197 193.11 28.8331 193.204 28.8564C193.507 28.9497 194.86 28.2499 194.72 28.0633C194.65 27.9466 195.093 27.8533 195.676 27.8533C196.469 27.8533 196.703 27.7834 196.586 27.5501C196.446 27.3401 196.493 27.3402 196.749 27.5268C197.239 27.9 198.359 28.1333 198.359 27.8533C198.359 27.7367 198.522 27.6201 198.709 27.6201C198.895 27.6201 199.059 27.5034 199.059 27.3635C199.059 27.2468 199.175 27.2002 199.292 27.2702C199.432 27.3635 199.665 27.3401 199.828 27.2235C199.992 27.1069 200.691 26.827 201.391 26.5937C202.161 26.3371 203.141 25.7773 203.841 25.1474C204.494 24.5876 205.287 24.0744 205.637 23.9811C206.01 23.9111 206.383 23.6778 206.523 23.4679C206.71 23.1413 206.78 23.118 206.966 23.3979C207.106 23.6312 207.456 23.6779 208.203 23.5612C211.562 23.048 215.037 22.2782 216.04 21.7884C216.577 21.5318 217.09 21.3218 217.183 21.3218C217.3 21.3218 218.093 21.0186 218.956 20.6454L219.282 20.5111C219.621 20.3716 219.553 19.8728 219.19 19.8289C217.3 19.619 215.621 19.2924 216.274 19.2458C216.694 19.2224 216.81 19.1058 216.74 18.6859C216.694 18.266 216.787 18.1727 217.347 18.1727C217.837 18.1727 218.047 18.2894 218.093 18.5926C218.131 19.0109 218.488 19.2856 218.902 19.3568C219.706 19.4951 221.04 19.5672 221.242 19.4324C221.382 19.3391 221.546 18.8259 221.569 18.266C221.615 17.3097 221.569 17.2397 220.776 16.8431C220.076 16.5165 220.029 16.4699 220.589 16.5865C221.336 16.7265 222.129 16.2366 221.639 15.9334C221.406 15.7701 221.452 15.7234 221.755 15.7234C222.059 15.7234 222.152 15.6068 222.035 15.2802C221.895 14.8137 222.409 14.5571 223.575 14.5571C224.531 14.5571 225.115 13.764 224.321 13.554C224.018 13.4841 224.065 13.4374 224.485 13.3908C224.811 13.3674 225.161 13.3441 225.278 13.3674C225.628 13.4374 227.377 12.621 227.657 12.2477C227.797 12.0611 228.077 11.9912 228.287 12.0611C228.497 12.1311 228.73 12.1078 228.8 11.9912C228.893 11.8512 228.777 11.7579 228.567 11.7579C228.357 11.7579 228.264 11.6413 228.334 11.5246C228.427 11.3847 228.194 11.2914 227.797 11.2914C227.121 11.2914 226.677 11.8045 226.957 12.2477C227.039 12.3834 226.718 12.4638 226.389 12.4569C225.983 12.4485 225.775 12.0459 226.118 11.8279C226.771 11.3613 226.654 10.195 225.931 9.70513C225.254 9.26192 225.581 9.02866 226.677 9.14529C227.284 9.21527 227.447 9.3319 227.354 9.61182C227.261 9.82177 227.307 9.91507 227.447 9.82177C227.61 9.70513 227.657 9.82177 227.587 10.125C227.401 10.8481 227.75 10.6615 228.007 9.89175C228.147 9.49519 228.427 9.19194 228.614 9.19194C228.8 9.19194 228.917 9.02866 228.87 8.77206C228.823 8.49214 228.544 8.37551 227.914 8.32885C227.424 8.32885 226.957 8.23555 226.864 8.14224C226.771 8.02561 226.934 8.00228 227.191 8.07226C227.471 8.14224 227.844 8.11891 228.007 8.00228C228.24 7.86232 228.147 7.76901 227.704 7.5824C227.377 7.46576 227.027 7.2325 226.934 7.09254C226.817 6.90592 226.677 6.90592 226.444 7.11586C225.884 7.5824 225.768 7.11586 226.327 6.64933C226.747 6.2761 227.074 6.20612 227.82 6.32275C228.614 6.41606 228.73 6.39273 228.614 6.06616C228.52 5.80957 228.59 5.69293 228.87 5.66961C229.197 5.66961 229.197 5.62295 228.8 5.45966C228.217 5.2264 228.544 5.08644 230.06 4.92315C230.62 4.85317 231.156 4.68988 231.226 4.57325C231.296 4.43329 231.763 4.27 232.229 4.17669C232.719 4.08339 233.046 3.9201 232.976 3.80347C232.906 3.68683 233.232 3.59353 233.675 3.59353C234.119 3.59353 234.539 3.50022 234.609 3.40691C234.678 3.29028 235.472 3.08034 236.381 2.91705C238.737 2.49717 238.177 2.17059 234.888 2.00731C232.952 1.93733 232.089 1.98398 231.856 2.17059C231.459 2.49717 230.456 2.28723 230.666 1.93733C230.76 1.82069 230.643 1.72739 230.456 1.72739C230.246 1.72739 230.106 1.84402 230.153 2.00731C230.176 2.14727 230.013 2.33388 229.78 2.38053C229.547 2.42719 229.407 2.33388 229.43 2.17059C229.477 2.00731 229.29 1.82069 229.033 1.77404C228.777 1.72739 228.637 1.75071 228.753 1.86735C228.847 1.96065 228.427 2.24057 227.82 2.47384C226.491 2.98703 226.351 2.98703 226.934 2.59048C227.167 2.40386 227.284 2.14727 227.191 1.98398C227.097 1.84402 227.191 1.68073 227.377 1.61075C227.587 1.54077 227.914 1.44747 228.124 1.37749C228.753 1.16754 228.427 1.07424 226.724 0.91095C225.838 0.817643 224.718 0.607703 224.251 0.444415C223.132 0.0478608 215.83 -0.138753 211.655 0.117841ZM202.488 1.84402C202.488 2.24057 202.488 2.24057 201.951 2.03063C201.578 1.89067 201.578 1.86735 201.998 1.70406C202.052 1.68222 202.106 1.66037 202.156 1.63973C202.325 1.56963 202.488 1.6605 202.488 1.84402ZM224.951 1.96065C224.765 2.07729 224.391 2.17059 224.135 2.17059C223.762 2.14727 223.762 2.12394 224.251 1.96065C225.068 1.70406 225.371 1.70406 224.951 1.96065ZM154.131 2.21725C154.406 2.19227 154.3 2.45652 154.037 2.3741C154.022 2.36923 154.006 2.36361 153.991 2.35721C153.805 2.28723 153.875 2.21725 154.131 2.21725ZM155.601 4.01341C155.041 4.06006 154.621 4.24667 154.434 4.54992C154.061 5.15642 153.851 5.10976 153.968 4.43329C154.085 3.80347 154.784 3.59353 155.904 3.82679C156.371 3.9201 156.324 3.96675 155.601 4.01341ZM159.286 3.82679C159.356 3.94343 159.24 4.06006 159.03 4.06006C158.657 4.06006 158.657 4.08339 159.077 4.40996C159.496 4.73654 159.473 4.73654 158.89 4.50327C158.377 4.27 158.307 4.15337 158.493 3.89677C158.82 3.52355 159.1 3.50022 159.286 3.82679ZM155.438 5.38968C155.438 5.59963 155.368 5.66961 155.274 5.55297C155.158 5.43634 154.994 5.27305 154.854 5.17974C154.714 5.08644 154.808 4.99313 155.041 4.99313C155.251 4.99313 155.438 5.15642 155.438 5.38968ZM154.971 5.78624C154.971 5.85622 154.878 5.97285 154.738 6.04283C154.621 6.11281 154.504 6.06616 154.504 5.94953C154.504 5.80957 154.621 5.69293 154.738 5.69293C154.878 5.69293 154.971 5.73959 154.971 5.78624ZM150.539 6.74264C150.376 6.92925 150.142 7.09254 150.002 7.09254C149.862 7.09254 149.956 6.92925 150.212 6.74264C150.796 6.29943 150.912 6.29943 150.539 6.74264ZM172.933 8.07226C173.239 8.07226 173.083 8.25206 172.778 8.21493C172.73 8.2091 172.687 8.20042 172.653 8.18889C172.489 8.11891 172.606 8.07226 172.933 8.07226ZM169.434 8.30553C169.74 8.30553 169.584 8.48532 169.279 8.4482C169.231 8.44237 169.188 8.43369 169.154 8.42216C168.99 8.35218 169.107 8.30553 169.434 8.30553ZM131.294 9.91507C131.551 10.3116 131.271 10.4049 130.944 10.0084C130.758 9.79844 130.758 9.65848 130.898 9.65848C131.038 9.65848 131.224 9.77511 131.294 9.91507ZM127.305 10.1483C127.58 10.1234 127.475 10.3876 127.211 10.3052C127.196 10.3003 127.18 10.2947 127.166 10.2883C126.979 10.2183 127.049 10.1483 127.305 10.1483ZM219.936 14.3238C220.006 14.4405 219.959 14.5571 219.843 14.5571C219.703 14.5571 219.586 14.4405 219.586 14.3238C219.586 14.1839 219.633 14.0906 219.679 14.0906C219.749 14.0906 219.866 14.1839 219.936 14.3238ZM219.423 15.0236C218.746 15.3035 218.723 15.8867 219.376 16.0733C220.006 16.2833 219.843 16.4466 219.166 16.2599C218.863 16.19 218.653 15.9567 218.653 15.7001C218.653 15.4668 218.466 15.1636 218.256 15.047C217.93 14.8603 218.07 14.8137 218.886 14.8137C219.749 14.8137 219.843 14.8603 219.423 15.0236ZM221.102 14.837C221.473 14.837 221.337 15.0172 220.968 14.977C220.916 14.9713 220.867 14.9635 220.822 14.9536C220.589 14.8837 220.729 14.837 221.102 14.837ZM221.312 15.2802C221.587 15.2552 221.481 15.5195 221.218 15.4371C221.203 15.4322 221.187 15.4266 221.172 15.4202C220.986 15.3502 221.056 15.2802 221.312 15.2802ZM177.715 30.1394C177.645 30.2093 177.458 30.1394 177.295 29.9761C177.085 29.7428 177.108 29.7195 177.435 29.8594C177.668 29.9294 177.785 30.0694 177.715 30.1394Z" />
        <path d="M132.577 5.43636C131.994 5.50634 131.528 5.69296 131.528 5.85624C131.528 6.01953 131.318 6.29945 131.061 6.48606C130.804 6.67268 130.641 6.88262 130.711 6.9526C130.781 7.02258 131.621 7.11589 132.554 7.18587C133.487 7.23252 134.373 7.39581 134.513 7.53577C134.653 7.67573 135.167 7.79236 135.633 7.79236C136.38 7.79236 136.566 7.67573 136.846 7.09256C137.196 6.41608 137.196 6.39276 136.473 6.06618C136.053 5.87957 135.47 5.69296 135.143 5.62298C134.817 5.553 134.35 5.45969 134.094 5.38971C133.837 5.34305 133.16 5.34305 132.577 5.43636Z" />
        <path d="M122.85 6.48589C121.031 6.76581 120.377 7.09239 121.637 7.09239C122.08 7.09239 122.547 7.18569 122.687 7.279C122.85 7.37231 122.244 7.44229 121.357 7.44229C120.167 7.44229 119.678 7.53559 119.514 7.79219C119.188 8.3287 119.888 8.56197 121.427 8.39868C122.85 8.25872 124.016 7.72221 123.433 7.48894C123.27 7.41896 123.48 7.41896 123.923 7.48894C124.483 7.58225 124.67 7.55892 124.553 7.37231C124.436 7.18569 124.576 7.09239 124.926 7.09239C125.229 7.09239 125.626 6.97575 125.813 6.85912C126.116 6.64918 126.116 6.60252 125.813 6.39258C125.393 6.13599 125.066 6.13599 122.85 6.48589Z" />
        <path d="M112.983 8.23551C111.443 8.37547 110.324 8.63207 107.711 9.4485C107.314 9.56514 106.731 9.65845 106.405 9.65845C106.078 9.65845 105.495 9.77508 105.075 9.91504C104.189 10.2416 104.072 10.8015 104.958 10.6382C105.332 10.5682 105.635 10.6615 105.752 10.8481C105.915 11.1047 106.055 11.0814 106.428 10.7315C106.755 10.4282 106.918 10.3816 106.918 10.5915C106.918 10.8015 107.128 10.8481 107.664 10.7315C108.364 10.5915 108.387 10.6148 107.968 10.9181C107.524 11.268 107.524 11.268 108.131 11.2913C108.481 11.2913 109.017 11.198 109.344 11.0581C109.67 10.9414 109.95 10.8948 109.95 10.9414C109.95 11.0114 109.67 11.1747 109.321 11.3146C108.761 11.5246 108.714 11.5946 109.064 11.7812C109.274 11.8978 109.904 12.0378 110.464 12.0844C111.07 12.1311 111.443 12.2477 111.397 12.4343C111.257 13.0875 112.563 13.1808 115.409 12.7142C116.972 12.4577 119.188 12.2011 120.331 12.1311C122.034 12.0378 122.687 11.8978 123.666 11.3846C125.113 10.6382 124.973 10.2416 123.34 10.3349L122.916 10.353C122.636 10.3649 122.462 10.0545 122.617 9.82173C122.967 9.26189 122.92 8.95864 122.5 8.95864C122.384 8.95864 122.36 9.07528 122.454 9.23856C122.547 9.40185 122.523 9.42518 122.337 9.33187C122.197 9.23856 121.684 9.28522 121.194 9.42518C119.514 9.93837 119.608 9.89171 119.981 10.125C120.261 10.3116 120.214 10.3582 119.818 10.3582C119.188 10.3582 119.094 10.7548 119.701 10.8948C119.911 10.9647 119.491 11.0114 118.745 11.0347C117.998 11.0347 117.438 10.9647 117.508 10.8481C117.788 10.4049 116.738 9.89171 115.292 9.72843C113.823 9.58847 113.799 9.56514 114.499 9.28522C114.896 9.12193 115.106 8.95864 114.966 8.93532C114.849 8.91199 115.036 8.84201 115.409 8.7487C115.805 8.67872 116.109 8.49211 116.109 8.37547C116.132 8.07223 115.222 8.0489 112.983 8.23551ZM112.493 10.3816L113.566 10.5682L112.47 10.4982C111.886 10.4516 111.257 10.4982 111.07 10.6148C110.837 10.7781 110.883 10.8248 111.233 10.8481C111.487 10.8481 111.317 11.015 111.068 10.9717C110.926 10.9471 110.769 10.9027 110.627 10.8481C110.23 10.6848 110.184 10.6148 110.487 10.6148C110.697 10.5915 110.883 10.4749 110.883 10.3349C110.883 10.1716 110.604 10.1483 110.137 10.2416C109.367 10.3816 109.367 10.3816 109.95 10.055C110.347 9.84506 110.674 9.79841 110.977 9.96169C111.21 10.0783 111.886 10.2649 112.493 10.3816ZM113.333 9.89171C113.263 10.0083 112.983 10.125 112.726 10.125C112.47 10.125 112.33 10.0083 112.4 9.89171C112.47 9.75175 112.75 9.65845 113.006 9.65845C113.263 9.65845 113.403 9.75175 113.333 9.89171ZM113.099 11.9911C113.193 12.1311 112.936 12.2244 112.47 12.2011C111.747 12.1777 111.723 12.1777 112.283 11.9911C112.61 11.8978 112.89 11.8045 112.913 11.7812C112.936 11.7579 113.03 11.8512 113.099 11.9911Z" />
        <path d="M98.9168 12.8542C98.5203 12.9242 98.1704 13.1808 98.0071 13.5307C97.8671 13.834 97.6105 14.0906 97.4706 14.0906C97.3073 14.0906 96.7008 14.3471 96.0943 14.6737C95.4878 15.0003 94.8813 15.2569 94.7413 15.2569C94.6014 15.2569 94.2981 15.3969 94.1115 15.5835C93.9016 15.7468 93.2484 16.0733 92.6419 16.3066C92.0354 16.5399 91.5922 16.7965 91.6622 16.9131C91.7089 17.0064 91.8955 17.0531 92.0588 17.0064C92.502 16.8431 93.2251 17.5196 93.0618 17.9395C92.8519 18.476 93.4117 18.5926 94.578 18.2194C95.1379 18.0328 96.0243 17.8695 96.5142 17.8462C97.0274 17.8228 97.7038 17.6129 98.0304 17.3796C98.8235 16.8198 101.506 15.9334 101.669 16.19C101.739 16.2833 101.436 16.4466 101.016 16.5165C100.62 16.6098 100.06 16.8198 99.8266 17.0064C99.3834 17.3563 99.3834 17.3563 99.8032 17.6595C100.2 17.9628 100.176 18.0094 99.5 18.2427C98.6369 18.546 98.6136 18.5926 99.1268 19.0825C99.43 19.4091 99.8499 19.4324 101.623 19.3157C104.025 19.1291 103.979 19.1291 104.912 19.1758C105.332 19.2021 105.245 19.4191 104.825 19.4559C104.308 19.5014 103.522 19.5449 102.672 19.5723C99.1734 19.6423 98.0537 19.8523 98.0537 20.4121C98.0537 21.1819 98.7069 21.5551 100.106 21.5551C101.11 21.5551 101.343 21.6251 101.133 21.835C100.97 21.9983 100.853 22.2782 100.853 22.4882C100.853 22.8381 101.133 22.8614 102.906 22.8148C104.025 22.7681 105.472 22.6281 106.101 22.4882C106.755 22.3482 107.734 22.1849 108.317 22.1383L109.367 22.045L108.434 22.2316C106.521 22.6281 106.078 22.9314 106.218 23.8645C106.241 24.0977 106.055 24.1444 105.355 24.0044C105.089 23.9591 104.881 23.9173 104.724 23.8762C104.312 23.7685 104.631 23.4272 105.005 23.6312C105.565 23.9344 105.845 23.7012 105.402 23.2813C105.145 23.0247 104.935 23.0014 104.422 23.2113C103.116 23.7245 100.176 23.9344 99.36 23.5612C99.0335 23.3979 98.9635 23.4213 99.0801 23.6079C99.1967 23.8178 98.9401 23.8878 98.1704 23.8411C96.934 23.7945 97.1906 23.5146 98.7069 23.2813C99.9199 23.0947 100.153 22.7681 99.4767 22.2316C99.1034 21.9283 98.5902 21.8117 97.6339 21.835C96.9107 21.8584 95.791 21.6951 95.1845 21.4851C94.578 21.2752 93.9249 21.0886 93.7616 21.0886C93.575 21.0886 93.3184 20.9253 93.1551 20.7387C92.7586 20.2721 90.8691 20.2721 89.936 20.762C89.5161 20.9486 88.9796 21.0653 88.723 20.9719C88.2332 20.8087 88.0699 20.3888 88.4898 20.3888C88.6297 20.3888 88.723 20.4821 88.723 20.622C88.723 20.7387 88.8397 20.8553 88.9796 20.8553C89.0963 20.8553 89.1662 20.762 89.0963 20.6454C89.0263 20.5287 89.1196 20.3888 89.3062 20.2955C89.4928 20.2255 89.6561 20.0622 89.6561 19.9222C89.6561 19.6423 88.5364 19.8523 88.0232 20.2488C87.6967 20.5287 86.4837 21.0886 86.2504 21.0886C86.2038 21.0886 86.2271 20.6687 86.2971 20.1555C86.4137 19.4091 86.367 19.1991 86.0405 19.0825C85.8305 18.9892 85.5739 19.0358 85.4573 19.1991C85.224 19.5957 84.1043 20.1555 83.5678 20.1555C83.2646 20.1555 83.1713 20.0389 83.2879 19.759C83.4045 19.4557 83.3579 19.4091 82.9847 19.549C82.7514 19.6656 82.3315 19.7823 82.0749 19.8056C80.4887 20.0389 77.6662 20.6454 77.3396 20.8553C77.0363 21.0419 77.1063 21.0653 77.6428 20.9719C78.2493 20.8553 78.2727 20.8553 77.8994 21.0886C77.5729 21.3218 77.3396 21.2985 76.8497 21.0653C76.1966 20.7387 74.0272 20.622 74.0272 20.902C74.0272 21.0186 73.8173 21.0886 73.584 21.0886C73.3274 21.0886 72.9542 21.2752 72.7442 21.4851C72.0677 22.1616 69.4551 21.7651 69.7351 21.0186C69.805 20.8087 69.5018 20.7153 68.6154 20.6687C67.0058 20.5987 66.0961 20.4121 66.0961 20.1089C66.0961 19.9922 65.3963 19.8989 64.4166 19.9222C63.4835 19.9222 62.0372 19.8289 61.1975 19.689C60.3577 19.549 58.7948 19.3624 57.6985 19.2691C56.2289 19.1525 55.739 19.0358 55.7857 18.8026C55.8323 18.546 55.5524 18.4993 54.5494 18.5693C53.6396 18.6159 53.2431 18.546 53.1498 18.3127C53.0798 18.1494 52.8698 18.0561 52.6832 18.1494C52.4966 18.2194 52.2633 18.1261 52.1467 17.9161C51.9834 17.6362 51.7735 17.6595 50.5838 18.0794C49.814 18.336 48.5077 18.5926 47.668 18.6393C46.8282 18.6859 45.5219 18.8725 44.7521 19.0825C44.0057 19.2924 43.0493 19.4557 42.6294 19.4557C41.813 19.4557 40.0401 20.0389 38.0807 20.9486C37.2876 21.3218 35.9579 21.6717 34.8149 21.8117C33.7419 21.9517 32.5522 22.2083 32.2023 22.3949C31.8291 22.5815 31.3393 22.7214 31.0827 22.7214C30.4995 22.7214 30.5228 23.118 31.106 23.3046C31.7358 23.5146 32.179 24.3077 31.8291 24.6576C31.4326 25.0541 31.5259 25.2874 32.0391 25.2874C32.3656 25.2874 32.4589 25.404 32.3423 25.6839C32.1324 26.2671 31.3393 26.9203 30.896 26.8969C30.2429 26.8503 29.0999 26.5004 29.4964 26.4771C29.7064 26.4771 30.0096 26.2904 30.1729 26.1038C30.5461 25.6373 30.2895 25.6373 28.1901 26.1272C26.184 26.5704 25.0644 26.827 23.4082 27.1535C21.9852 27.4568 21.4954 27.76 22.0785 27.9933C22.4284 28.1333 22.4284 28.2032 22.0086 28.7864C21.3787 29.6495 21.6353 29.7895 23.7114 29.7661C24.6212 29.7428 25.4842 29.8361 25.6242 29.9527C25.8108 30.116 26.184 30.0694 26.7439 29.8361C27.7236 29.4396 28.4001 29.3696 27.8402 29.7195C27.5603 29.9061 27.5603 29.9527 27.8169 29.9527C28.3068 29.9527 27.0005 31.0957 26.324 31.259C25.9974 31.329 25.3676 31.329 24.9244 31.259C24.4579 31.1657 24.108 31.1891 24.108 31.329C24.108 31.7022 22.6617 32.0755 21.8219 31.9122C21.0288 31.7489 20.4223 31.9588 19.1627 32.7986C18.8361 33.0319 18.3929 33.2185 18.183 33.2185C17.8097 33.2185 16.1535 33.9183 15.9436 34.1749C15.8736 34.2449 15.5237 34.3848 15.1738 34.4781C14.5207 34.6414 13.9375 35.5978 14.2174 36.041C14.2874 36.1577 14.0775 36.3443 13.7742 36.4609C13.2844 36.6475 13.191 36.8341 13.2144 37.5106C13.2377 38.2571 13.3077 38.3504 13.8209 38.3504C14.1708 38.3504 14.8239 38.2571 15.2905 38.1404C16.0836 37.9538 16.1535 37.9771 15.9436 38.3504C15.8036 38.5836 15.5004 38.8402 15.2671 38.9335C15.0105 39.0035 14.6606 39.3301 14.4507 39.6333C14.2408 39.9599 13.8909 40.2165 13.6809 40.2165C13.401 40.2165 13.3543 40.2865 13.5409 40.4731C13.7276 40.6597 13.9375 40.6597 14.3107 40.4498C14.6606 40.2632 14.7773 40.2632 14.684 40.4264C14.4507 40.7763 14.684 40.753 15.4771 40.3331C16.2935 39.9133 17.4365 40.0999 17.2499 40.5897C17.0866 41.0329 17.8097 41.0096 18.1363 40.5431C18.3463 40.2865 18.5562 40.2398 18.9761 40.4031C19.5359 40.6131 19.5359 40.6131 18.5095 41.3828C17.9264 41.8027 16.9933 42.3392 16.4101 42.5958C15.827 42.8524 15.1505 43.1324 14.8939 43.249C14.6373 43.3656 13.6343 43.7155 12.6779 44.0188C10.5551 44.7186 9.64538 45.1151 9.64538 45.395C9.64538 45.5117 9.43544 45.4883 9.20218 45.3717C8.75897 45.1151 8.01251 45.3251 5.32994 46.3514C3.76704 46.9579 3.48712 47.0279 2.60071 47.1212C1.48102 47.2379 -0.268481 48.1243 0.0347663 48.4275C0.174727 48.5675 0.431321 48.5675 0.734569 48.4042C0.991163 48.2642 1.50435 48.1476 1.87758 48.1476C2.22748 48.1476 2.83398 47.9843 3.23053 47.7744C3.60376 47.5878 4.14027 47.4711 4.39687 47.5411C4.65346 47.6111 5.07334 47.4945 5.30661 47.3312C5.38939 47.2649 5.47805 47.2105 5.56321 47.1698C5.7959 47.0588 6.11998 47.1958 6.26301 46.9813C6.5196 46.5381 7.5693 46.7247 7.70926 47.2145C7.84923 47.751 8.03584 47.7977 8.43239 47.4011C8.61901 47.2145 8.7823 47.2145 9.03889 47.4245C9.31881 47.6577 9.43544 47.6111 9.64538 47.2379C9.85532 46.8413 9.832 46.748 9.52875 46.748C9.34214 46.748 9.17885 46.5847 9.17885 46.3748C9.17885 46.0948 9.31881 46.0249 9.76202 46.1415C10.1352 46.2348 10.5551 46.1415 10.9284 45.9082C11.41 45.6118 12.0132 45.7504 12.5436 45.5543C13.0888 45.3527 13.7421 45.0073 14.0308 44.7186C14.2174 44.5553 14.4507 44.462 14.5673 44.532C14.684 44.6253 14.7773 44.5553 14.7773 44.4153C14.7773 44.2754 15.0572 44.2287 15.5004 44.2987C15.9203 44.3687 16.1535 44.3453 16.0836 44.2054C16.0136 44.0887 16.3635 43.9255 16.8767 43.8322C17.3899 43.7388 18.043 43.5056 18.3229 43.2723C18.7661 42.9457 19.396 42.6891 21.0755 42.1993C21.1455 42.176 21.4954 42.036 21.8453 41.896C22.2185 41.7327 22.685 41.6161 22.895 41.6161C23.1049 41.6161 23.7114 41.3595 24.2246 41.0329C24.7378 40.7297 25.3909 40.4498 25.6475 40.4498C26.3473 40.4498 26.8838 39.7966 26.4173 39.4934C26.1607 39.3301 26.184 39.2834 26.5573 39.2601C26.8139 39.2368 27.1404 39.2368 27.2804 39.2601C27.4437 39.2834 27.607 39.1435 27.7003 38.9335C27.7703 38.7469 28.0968 38.5836 28.4234 38.5836C28.75 38.5836 29.0066 38.467 29.0066 38.3504C29.0066 38.2104 29.1932 38.1171 29.4265 38.1171C29.6597 38.1171 30.033 37.9771 30.2895 37.8139C30.9427 37.3706 31.5725 37.1607 31.5725 37.394C31.5725 37.4873 31.2926 37.6972 30.966 37.8372C30.6395 38.0005 30.4528 38.2337 30.5228 38.3504C30.6161 38.4903 30.3595 38.7003 29.963 38.8402C29.5664 38.9802 29.2398 39.1901 29.2398 39.3068C29.2398 39.68 30.4762 39.5167 32.109 38.9569C32.9721 38.6536 33.6719 38.4437 33.6719 38.5137C33.6719 38.5836 33.8119 38.537 33.9752 38.397C34.3251 38.0938 35.748 38.0005 36.2845 38.2337C36.5411 38.327 37.101 38.1871 37.7541 37.8838C38.7105 37.4173 39.1304 36.9508 38.5705 36.9508C38.4539 36.9508 38.3373 37.0441 38.3373 37.184C38.3373 37.3007 38.244 37.4173 38.1273 37.4173C38.0107 37.4173 37.9407 37.2307 37.9874 36.9974C38.0107 36.7408 38.2206 36.6242 38.5472 36.6475C38.8038 36.6942 39.0371 36.6009 39.0371 36.4842C39.0371 36.3443 39.2937 36.251 39.6202 36.251C40.2967 36.251 40.32 36.7641 39.6436 36.9508C39.387 37.0207 39.1537 37.2074 39.1071 37.3473C39.0371 37.5573 39.2237 37.5573 39.8069 37.3706C40.3667 37.2074 40.8099 37.2074 41.2764 37.3706C41.9296 37.6039 41.9296 37.6272 41.4397 37.7672C41.1365 37.8605 40.9032 38.0471 40.9032 38.1638C40.9032 38.3037 41.0665 38.327 41.3231 38.2104C41.533 38.1171 42.4661 38.0005 43.3759 37.9305C44.6355 37.8605 45.2187 37.9305 45.8018 38.2337C46.2684 38.467 46.7816 38.5603 47.1781 38.4903C47.5747 38.3737 47.6913 38.397 47.458 38.5603C47.2014 38.7236 47.2481 38.8636 47.738 39.2834C48.0645 39.5634 48.3911 40.0532 48.4611 40.3565C48.5544 40.6597 48.7177 40.9163 48.8343 40.9163C49.2075 40.9163 48.7877 42.6425 48.2745 43.2257C47.9479 43.5989 47.9012 43.7388 48.1345 43.6689C48.3911 43.5756 48.4844 43.6922 48.4844 44.0654C48.4844 44.1144 48.4826 44.1604 48.4791 44.2028C48.4585 44.4548 48.1345 44.489 48.1345 44.7419C48.1345 45.2784 48.5077 45.2084 48.9276 44.6019C49.1376 44.322 49.3475 44.1587 49.4175 44.2287C49.4875 44.2987 49.2309 44.6253 48.8343 44.9518C47.9945 45.675 47.9712 45.8149 48.671 45.8383C48.7817 45.8383 48.8117 45.9907 48.7092 46.0326L48.671 46.0482C48.2278 46.2348 48.1578 46.4214 48.2278 47.0279C48.2978 47.6811 48.3911 47.7977 48.9976 47.8444C49.4641 47.891 49.8607 47.751 50.1173 47.4478C50.5138 47.0046 50.5605 47.0046 50.8404 47.4011C51.1203 47.7744 51.1437 47.751 51.4003 47.2145C51.4282 47.1448 51.5328 47.1751 51.5191 47.249L51.5169 47.2612C51.4469 47.6111 51.4936 47.961 51.6335 48.031C51.7502 48.101 51.8668 48.3109 51.8668 48.4509C51.8668 48.6375 51.7968 48.6141 51.6568 48.4042C51.4936 48.1476 51.4003 48.1709 51.2136 48.5208C51.0286 48.8292 50.6043 48.6895 50.3432 48.9367C50.1566 49.1134 50.0179 49.3749 50.1173 49.5239C50.1873 49.6405 50.1406 49.8038 50.0006 49.8971C49.6507 50.1304 49.7207 50.4103 50.2806 50.9468C50.5605 51.2034 50.6538 51.4133 50.4905 51.4133C50.3272 51.4133 50.2339 51.6233 50.2806 51.9265C50.3272 52.2298 50.4905 52.4397 50.6771 52.3931C50.8871 52.3464 50.8404 52.5797 50.5138 53.1395C50.1873 53.676 50.1173 54.0259 50.2806 54.1193C50.4439 54.2126 50.4205 54.3292 50.2339 54.4458C50.0006 54.5858 50.1173 54.7957 50.6071 55.2856C51.097 55.7521 51.167 55.8688 50.8171 55.7288C50.2106 55.4955 49.7907 55.4955 49.1376 55.7055C48.811 55.7988 48.601 55.7755 48.601 55.6355C48.601 55.5189 48.8343 55.3556 49.1376 55.3089C49.5808 55.2389 49.5808 55.2156 49.0676 55.2389C48.5777 55.2623 48.4844 55.3789 48.5311 55.8921C48.5544 56.2653 48.4611 56.5452 48.2978 56.5452C48.1345 56.5452 48.2745 56.6852 48.601 56.8485C48.8198 56.9579 49.0177 57.1091 49.1244 57.2462C49.209 57.3548 49.2189 57.514 49.2368 57.6506C49.2617 57.8405 49.2936 58.0305 49.449 58.1426C49.5366 58.2059 49.6495 58.2626 49.7674 58.2947C50.024 58.388 50.1873 58.5513 50.1173 58.668C50.024 58.7846 50.1406 58.9479 50.3272 59.0179C50.5372 59.0878 50.6305 59.2511 50.5605 59.3678C50.4905 59.4844 50.5838 59.5777 50.7471 59.5777C50.9104 59.5777 51.237 59.741 51.4936 59.9276C51.8901 60.2309 51.8901 60.2542 51.2836 60.2075C50.5838 60.1609 50.4905 60.2775 49.9073 62.097C49.5108 63.31 48.1812 65.7826 47.598 66.3891C47.4347 66.5524 47.0382 67.0656 46.7349 67.5088C46.4317 67.952 45.7552 68.7684 45.242 69.3283C43.959 70.7046 43.5391 71.4044 43.1193 72.7573C42.9326 73.3871 42.3728 74.4368 41.8596 75.0433C41.0665 75.9764 40.9732 76.233 41.1831 76.6062C41.3931 76.9328 41.3464 77.2127 41.0432 77.8892C40.6933 78.589 40.6699 78.8689 40.8799 79.3355C41.0432 79.662 41.0898 80.1052 41.0199 80.3385C40.9499 80.5718 40.9965 80.8284 41.1132 80.8984C41.2298 80.9917 41.2764 81.4115 41.2065 81.8781C41.1132 82.4612 41.1598 82.7878 41.3931 82.9278C41.673 83.1144 41.673 83.161 41.3931 83.3477C41.0898 83.5343 41.0898 83.7675 41.3698 84.9339C41.5564 85.6803 41.6963 86.6601 41.6963 87.1033C41.7197 87.873 41.7663 87.9197 42.6061 87.9897C43.1659 88.0363 43.4692 88.1763 43.4692 88.4096C43.4692 88.6895 43.9124 88.8528 44.4955 88.7595C44.6122 88.7361 44.6355 88.8761 44.5655 89.086C44.4722 89.296 44.5422 89.4359 44.7288 89.4359C45.312 89.4359 45.7785 90.2524 45.7785 91.3021C45.7785 92.3285 45.9651 95.151 46.0351 95.151C46.0584 95.151 46.1051 95.7108 46.1517 96.4106C46.2217 97.3204 46.3617 97.7403 46.7116 97.9735C46.9682 98.1601 47.3881 98.6733 47.668 99.1165C48.1578 99.8863 48.1578 99.9796 47.8079 100.516C47.4347 101.099 47.2248 101.193 46.455 101.123C46.2217 101.099 46.0351 101.146 46.0351 101.263C46.0351 101.659 47.9479 103.665 48.1812 103.525C48.4611 103.339 50.0006 104.948 50.0006 105.391C50.0006 105.578 49.8374 106.021 49.6507 106.371C49.4641 106.744 49.3008 107.211 49.3008 107.398C49.3008 107.864 49.7441 108.331 50.1873 108.331C50.7238 108.331 52.3333 110.197 52.3333 110.827C52.3333 111.2 52.4266 111.363 52.6366 111.27C52.7765 111.2 53.1031 111.06 53.3597 110.99C53.7096 110.85 53.7563 110.663 53.6396 109.987C53.5696 109.52 53.4997 108.96 53.4997 108.727C53.4997 108.517 53.383 108.331 53.2431 108.331C53.1264 108.331 53.0798 108.471 53.1498 108.611C53.2431 108.774 53.1031 108.657 52.8465 108.354C52.4033 107.817 52.38 107.817 52.2633 108.237C52.1934 108.564 52.1467 108.517 52.1234 108.051C52.1234 107.701 52.2167 107.351 52.3333 107.281C52.6832 107.071 52.6132 106.698 52.24 106.698C51.8901 106.698 51.5869 105.345 51.9134 105.158C52.0067 105.088 52.1001 104.925 52.1001 104.808C52.1001 104.668 52.0067 104.645 51.8668 104.715C51.7268 104.785 51.6335 104.598 51.6335 104.295C51.6335 103.712 50.7471 101.752 50.4205 101.613C50.1406 101.473 50.2106 100.283 50.5138 99.9796C50.6538 99.8397 50.7005 99.6297 50.6071 99.5364C50.5138 99.4431 50.3506 99.5131 50.2339 99.6997C50.0473 99.9796 50.0006 99.933 50.0006 99.5364C50.0006 99.2565 49.9073 98.9532 49.7674 98.8833C49.6507 98.8133 49.5341 98.6267 49.5341 98.4634C49.5341 98.3234 49.6974 98.4167 49.9073 98.6733C50.2572 99.0932 50.2572 99.0932 50.1173 98.4167C49.9307 97.5536 49.7907 97.4137 49.5574 97.8102C49.4175 98.0202 49.2542 97.9035 48.9043 97.367C48.4844 96.7139 48.4844 96.5273 48.741 95.2443C49.0676 93.6114 49.1376 93.5181 49.6974 93.7514C49.9307 93.868 50.2106 93.9147 50.3039 93.8913C50.6305 93.8214 51.4702 94.7778 51.3536 95.081C51.2836 95.2443 51.3069 95.9674 51.4003 96.6672C51.5169 97.577 51.4936 98.1368 51.2603 98.5567C50.9804 99.1399 51.027 99.4664 51.4236 99.4664C51.5169 99.4664 51.5402 99.2332 51.4702 98.9299C51.4003 98.6267 51.5402 98.7433 51.7735 99.2332C52.0067 99.6764 52.3333 100.19 52.4966 100.353C52.6599 100.516 52.7999 100.749 52.7999 100.866C52.7999 100.983 52.9865 101.123 53.2431 101.216C53.5463 101.309 53.6396 101.496 53.5463 101.939C53.4763 102.406 53.5463 102.546 53.9429 102.662C54.1995 102.732 54.4327 102.965 54.4327 103.199C54.4327 103.432 54.666 103.665 54.9459 103.735C55.4125 103.875 55.4124 103.899 55.0625 104.458C54.8293 104.785 54.666 105.181 54.666 105.321C54.666 105.625 55.5524 106.464 55.879 106.464C55.9956 106.464 56.2056 106.744 56.3689 107.094C56.5088 107.468 56.9054 108.004 57.2553 108.331C57.5818 108.657 58.0484 109.287 58.2816 109.73C58.4916 110.173 58.9348 110.873 59.2147 111.27C59.7279 111.923 59.8212 112.436 59.8212 114.396C59.8212 115.842 59.7746 116.028 59.4247 116.028C58.9348 116.028 59.0281 117.055 59.6346 117.965C59.9145 118.408 60.521 118.968 60.9642 119.201C61.4074 119.434 61.874 119.877 61.9906 120.157C62.2005 120.624 63.5535 121.23 64.3699 121.23C64.5799 121.23 65.0697 121.557 65.4896 121.977C66.1894 122.7 68.5921 123.96 69.2452 123.96C69.4085 123.96 69.7117 124.099 69.875 124.239C70.3649 124.706 71.6012 125.359 71.9978 125.359C72.1844 125.359 72.6276 125.499 72.9775 125.686C73.654 126.036 74.3538 125.919 75.7067 125.266C76.1033 125.056 76.6165 124.893 76.8264 124.893C77.7828 124.893 78.9491 125.639 79.8355 126.829C81.2818 128.765 81.9816 129.325 83.2646 129.535C83.8944 129.628 84.6175 129.815 84.8741 129.954C86.2038 130.654 87.1135 130.911 88.1165 130.841C88.9796 130.794 89.1429 130.841 88.863 131.051C88.5364 131.261 88.6064 131.447 89.3762 132.287C91.4989 134.713 91.9188 135.506 91.4523 136.276C91.149 136.743 91.4989 137.442 92.2221 137.816C92.8519 138.119 92.9452 138.119 93.1551 137.769C93.3651 137.396 93.3651 137.396 93.3884 137.769C93.3884 137.979 93.7383 138.329 94.1815 138.539C94.9979 138.935 95.3945 139.472 95.0213 139.705C94.7413 139.868 94.9979 140.522 95.3245 140.522C95.4645 140.522 95.8144 140.755 96.1176 141.035C96.6075 141.478 96.6541 141.501 96.6541 141.128C96.6541 140.848 96.7708 140.755 97.0274 140.848C97.2373 140.918 97.5872 140.988 97.8205 140.988C98.3103 140.988 98.6369 141.665 98.287 142.014C97.9604 142.341 98.0071 142.621 98.4036 142.621C98.6136 142.621 98.6835 142.481 98.6136 142.271C98.4036 141.711 99.1268 142.084 99.5466 142.761C99.9199 143.297 99.9432 143.297 100.736 142.924C101.623 142.504 101.716 142.248 101.203 141.548C100.736 140.941 100.76 140.918 101.553 140.568C101.949 140.382 102.253 140.102 102.253 139.938C102.253 139.775 102.416 139.588 102.602 139.495C102.929 139.378 103.116 137.956 102.812 137.956C102.742 137.956 101.996 138.329 101.18 138.772C99.5 139.705 99.3134 139.728 98.7069 139.192C98.4036 138.935 98.287 138.912 98.287 139.122C98.287 139.355 98.217 139.355 97.9604 139.122C97.7738 138.982 97.6572 138.725 97.7272 138.562C97.7971 138.399 97.5639 138.142 97.2373 138.002C96.9107 137.862 96.6541 137.582 96.6541 137.372C96.6541 137.186 96.5375 137.022 96.4209 137.022C96.2809 137.022 96.071 136.743 95.9543 136.369C95.8144 136.019 95.6044 135.506 95.4645 135.25C95.2779 134.876 95.2779 134.62 95.4645 134.27C95.6044 134.013 95.7211 133.523 95.7211 133.197C95.7211 132.87 95.8377 132.59 95.9543 132.59C96.0943 132.59 96.1876 132.031 96.1876 131.354C96.1876 130.584 96.3509 129.815 96.6308 129.301C96.8874 128.835 97.0274 128.228 96.9807 127.972C96.9107 127.715 96.9574 127.435 97.1207 127.342C97.2839 127.249 97.144 126.992 96.7241 126.619C96.3742 126.316 95.9077 125.896 95.6977 125.709C95.2312 125.266 92.4553 124.869 92.0588 125.173C91.7322 125.429 89.7961 125.476 89.4228 125.242C89.2829 125.149 89.0263 125.219 88.8397 125.359C88.5598 125.592 88.4198 125.592 88.1399 125.359C87.9533 125.219 87.6733 125.173 87.51 125.266C87.3468 125.359 87.51 125.126 87.8599 124.729C88.2332 124.333 88.5598 123.656 88.6531 123.12C88.723 122.63 88.8863 122.07 89.0263 121.907C89.1429 121.767 89.1896 121.627 89.1196 121.627C89.0496 121.627 89.2362 121.37 89.5395 121.044C89.8661 120.74 90.1226 120.321 90.1226 120.134C90.1226 119.947 90.2393 119.504 90.4026 119.131C90.5425 118.781 90.6825 118.291 90.6825 118.081C90.7058 117.871 90.7758 117.615 90.8924 117.521C90.9857 117.428 91.0557 117.171 91.0557 116.962C91.0557 116.728 91.219 116.425 91.4056 116.262C91.7089 116.005 91.7322 116.028 91.6156 116.355C91.4056 116.892 91.8255 116.822 92.1754 116.262C92.362 115.958 92.362 115.795 92.2221 115.795C92.0588 115.795 92.0588 115.632 92.2221 115.305C92.3853 115.049 92.432 114.582 92.362 114.279C92.2221 113.766 92.1287 113.742 90.7758 113.812C89.9594 113.836 88.6064 114.046 87.7666 114.256C86.5536 114.582 86.1571 114.769 85.9472 115.235C85.8072 115.562 85.6906 115.935 85.6906 116.052C85.6906 116.192 85.434 116.845 85.1307 117.498C84.5709 118.758 83.5445 119.504 82.1916 119.667C80.9319 119.807 79.1591 120.204 78.6925 120.461C78.2727 120.694 78.1094 120.647 77.6195 120.157C77.3163 119.831 76.7564 119.527 76.4065 119.457C75.7767 119.317 75.4035 118.758 75.0769 117.311C75.0069 116.915 74.7503 116.425 74.5404 116.192C74.3304 115.958 74.1438 115.329 74.1205 114.792C74.0972 114.256 73.9805 113.556 73.8639 113.229C73.7239 112.903 73.654 112.436 73.7006 112.18C74.0738 109.87 74.9603 107.211 75.6601 106.231C75.9866 105.765 76.1733 105.088 76.2432 104.108C76.3832 101.986 76.8964 101.356 79.9522 99.5364C83.3579 97.4837 84.5709 97.0405 85.154 97.6236C85.2707 97.7403 85.9238 97.8802 86.6003 97.8802C87.4167 97.9035 87.79 98.0202 87.79 98.2301C87.79 98.3934 87.9299 98.5567 88.0932 98.58C89.0729 98.6967 89.7961 98.65 89.936 98.4401C90.0993 98.1835 90.8691 98.0202 90.7291 98.2534C90.6825 98.3468 90.7058 98.6033 90.7991 98.8366C90.9391 99.2098 90.9857 99.2098 91.4056 98.8833C91.8722 98.5334 91.8721 98.5334 91.3356 98.1135C90.8224 97.7169 90.8224 97.6936 91.2657 97.3204C91.4989 97.1338 91.7322 96.8305 91.8022 96.6672C91.9421 96.224 96.8407 96.154 97.5405 96.5739C97.8205 96.7605 98.0537 97.0638 98.0537 97.2271C98.0537 97.8102 98.8235 97.8802 99.5 97.367C100.293 96.7839 100.853 96.8305 101.226 97.5536C101.366 97.8336 101.646 98.2534 101.856 98.4867C102.346 99.0232 102.346 99.5364 101.903 100.376C101.739 100.726 101.599 101.286 101.646 101.589C101.809 103.012 102.089 104.132 102.276 104.132C102.393 104.132 102.486 104.295 102.486 104.482C102.486 104.668 102.696 105.018 102.952 105.228C103.209 105.461 103.419 105.928 103.419 106.278C103.419 106.768 103.512 106.884 103.792 106.768C104.002 106.698 104.259 106.768 104.375 106.954C104.562 107.234 104.632 107.234 104.795 106.931C105.02 106.57 105.086 106.131 105.305 105.766C105.329 105.726 105.354 105.687 105.378 105.648C106.008 104.622 106.218 103.339 105.961 101.846C105.845 101.029 105.798 100.283 105.868 100.166C105.938 100.073 105.822 99.3032 105.612 98.51C104.865 95.5709 106.148 93.0049 109.134 91.5353C109.507 91.3487 110.254 90.7422 110.79 90.1824C111.42 89.5292 111.933 89.2027 112.353 89.2027C112.68 89.2027 113.03 89.086 113.099 88.9694C113.403 88.5029 115.082 87.5698 115.642 87.5698C116.342 87.5698 116.948 86.9166 116.505 86.6601C116.295 86.5201 116.505 86.3102 117.158 86.0069C118.045 85.5637 118.091 85.5637 117.858 85.9603C117.742 86.2169 117.462 86.4035 117.228 86.4268C117.018 86.4268 116.948 86.4968 117.112 86.5434C117.275 86.6134 117.602 86.5434 117.858 86.4035C118.278 86.1935 118.348 85.9369 118.348 84.584C118.348 83.6976 118.231 82.8811 118.115 82.7645C117.928 82.5779 117.951 82.298 118.138 81.8314C118.278 81.4582 118.441 80.945 118.488 80.7117C118.628 79.9186 118.978 79.8253 118.931 80.5718C118.908 80.9683 118.768 81.4115 118.628 81.5515C118.465 81.7148 118.348 82.0414 118.348 82.298C118.348 82.7412 118.418 82.6945 118.908 82.088C119.211 81.7148 119.421 81.2949 119.351 81.1783C119.258 81.0616 119.328 81.0383 119.468 81.1316C119.631 81.2249 120.004 81.015 120.354 80.6418C120.821 80.1286 120.937 79.8253 120.844 79.2422C120.774 78.8223 120.821 78.4257 120.937 78.3557C121.054 78.2624 121.147 78.3791 121.147 78.5657C121.147 79.1022 121.287 79.0322 122.407 78.1225C122.943 77.6793 123.527 76.9561 123.69 76.5129C123.993 75.7898 124.156 75.6965 125.323 75.4632C126.956 75.1133 128.262 74.5535 127.562 74.5068C127.305 74.4835 127.585 74.3202 128.215 74.1569C128.822 73.9936 129.522 73.9003 129.755 73.9703C129.988 74.0403 130.268 73.9703 130.338 73.8304C130.431 73.6904 130.688 73.5738 130.944 73.5738C131.434 73.5738 131.761 72.8973 131.411 72.5474C131.248 72.3841 131.178 72.4307 131.178 72.7107C131.178 73.2705 130.734 73.2005 130.501 72.5707C130.338 72.2208 130.385 71.9875 130.618 71.8009C130.804 71.661 130.944 71.381 130.944 71.1944C130.944 70.4946 132.461 69.4682 134.443 68.7918C135.54 68.4185 136.659 67.9753 136.939 67.812C137.219 67.6254 137.756 67.4622 138.152 67.4622C138.829 67.4388 139.202 67.0656 138.596 67.0189C138.176 66.9956 139.132 66.6457 139.949 66.5058C140.322 66.4358 140.905 66.2725 141.255 66.1325C141.628 65.9926 141.955 65.9692 142.025 66.1092C142.095 66.2258 141.441 66.669 140.602 67.0889C139.249 67.7421 139.015 67.952 138.969 68.5352C138.922 68.9084 139.039 69.3516 139.225 69.5149C139.599 69.8881 140.275 69.9581 140.275 69.6082C140.275 69.4682 140.485 69.3749 140.742 69.3749C140.998 69.3749 141.208 69.2816 141.208 69.165C141.208 68.8384 143.028 67.6954 143.284 67.8587C143.424 67.9287 143.658 67.9287 143.821 67.8354C144.213 67.5859 146.347 66.86 147.385 66.5727C147.756 66.4701 148.143 66.4202 148.37 66.1092C148.556 65.8526 148.859 65.6427 149.069 65.6427C149.583 65.6427 150.772 64.8962 150.772 64.5463C150.772 64.383 150.609 64.2431 150.422 64.2431C150.002 64.2431 149.979 64.1497 150.352 63.5433C150.586 63.1934 150.586 63.0301 150.352 62.7968C150.142 62.5869 149.769 62.8435 148.673 63.9631C147.903 64.7562 147.273 65.3161 147.273 65.1761C147.273 65.0362 147.063 65.0362 146.76 65.1528C146.13 65.3627 144.171 65.0128 144.357 64.7096C144.521 64.453 145.64 64.4997 145.64 64.7562C145.64 64.8496 145.85 64.9429 146.107 64.9429C146.363 64.9429 146.573 64.8496 146.573 64.7096C146.573 64.593 146.853 64.3364 147.227 64.1497C147.856 63.7999 147.833 63.7999 146.573 63.7765C145.08 63.7765 144.637 63.5666 144.847 62.9601C144.964 62.6568 144.917 62.5869 144.684 62.6568C144.654 62.6682 144.625 62.682 144.599 62.6979C144.366 62.8375 144.378 63.2653 144.124 63.17C143.681 62.9834 143.681 63.4033 144.124 63.8698C144.591 64.3364 144.567 64.453 144.007 64.313C143.518 64.1731 143.261 63.5199 143.378 62.7501C143.424 62.4935 143.821 61.9337 144.264 61.5138L144.601 61.1769C144.734 61.044 144.611 60.8199 144.427 60.8607C143.471 61.0939 143.634 60.7907 144.707 60.3708C146.573 59.6244 146.06 58.2714 143.937 58.3414C142.514 58.388 139.435 59.3444 138.036 60.1609C136.729 60.9307 136.59 60.9073 137.523 60.0676C138.432 59.2978 140.158 58.4114 140.812 58.4114C141.068 58.4114 141.558 58.1081 141.908 57.7349C142.235 57.3383 142.794 56.9185 143.168 56.7785C143.564 56.6385 145.757 56.5452 148.719 56.5686C151.659 56.5919 153.688 56.4986 153.781 56.3586C153.851 56.242 154.155 56.0787 154.458 55.9854C154.738 55.8921 155.251 55.5655 155.624 55.2856C156.394 54.6558 157.584 54.2126 158.563 54.2126C158.828 54.2126 158.958 54.5334 158.769 54.7181L158.237 55.2389C156.557 56.8485 155.438 58.1081 155.438 58.388C155.438 58.528 155.274 58.6446 155.088 58.6446C154.878 58.6446 154.668 58.8079 154.598 58.9945C154.504 59.1812 154.364 59.2745 154.248 59.2045C154.038 59.0878 153.338 59.6244 153.338 59.9276C153.338 59.9976 153.571 60.0209 153.875 59.9276C154.155 59.8576 153.945 60.0676 153.385 60.4175C151.752 61.4205 152.102 61.6771 155.321 61.7938C161.269 62.0503 160.803 61.9804 160.429 62.4002C160.033 62.8201 160.01 63.31 160.359 63.31C160.476 63.31 160.546 63.2167 160.453 63.1C160.383 62.9834 160.429 62.7968 160.569 62.7268C160.709 62.6335 160.803 62.6568 160.779 62.7735C160.663 63.4733 160.826 63.7065 161.386 63.5899C162.272 63.4266 163.672 61.2106 162.902 61.2106C162.785 61.2106 162.669 61.3039 162.669 61.4438C162.669 61.5605 162.576 61.6771 162.482 61.6771C162.366 61.6771 162.272 61.5138 162.272 61.3272C162.272 61.0892 162.594 61.1352 162.777 60.983C162.82 60.9475 162.862 60.9062 162.902 60.8607C163.135 60.5808 163.112 60.5108 162.809 60.5108C162.599 60.5108 162.249 60.7207 162.039 60.9773C161.806 61.2339 161.526 61.4438 161.409 61.4438C161.293 61.4438 161.432 61.1873 161.712 60.884C162.109 60.4641 162.156 60.3008 161.922 60.2075C161.759 60.1376 161.922 60.1142 162.319 60.1142C162.785 60.1609 163.019 60.0442 163.089 59.7643C163.135 59.4844 163.042 59.4144 162.692 59.5077C162.109 59.6477 161.992 59.0179 162.552 58.7846C163.065 58.598 162.972 58.2014 162.389 57.9448C162.375 57.9393 162.361 57.9339 162.347 57.9285C162.091 57.825 162.433 57.6719 162.296 57.4316C162.179 57.2684 162.039 57.315 161.806 57.5949C161.642 57.8515 161.502 57.8982 161.502 57.7582C161.502 57.5949 161.339 57.4783 161.153 57.4783C160.966 57.4783 160.803 57.5483 160.803 57.6649C160.803 57.7582 160.499 57.8515 160.126 57.8515C159.636 57.8749 159.473 57.8049 159.566 57.5716C159.636 57.385 159.823 57.245 160.01 57.245C160.639 57.245 160.336 56.8252 159.59 56.6852C159.275 56.6185 159.174 56.2199 159.419 56.0112L159.613 55.8454C160.15 55.4022 160.476 55.2389 160.593 55.4022C160.803 55.7754 161.083 55.2856 160.896 54.8191C160.826 54.6091 160.849 54.4458 160.989 54.4458C161.106 54.4458 161.269 54.2825 161.363 54.0959C161.456 53.8393 161.316 53.746 160.849 53.746C160.406 53.746 160.313 53.816 160.523 53.956C160.733 54.0726 160.593 54.1426 160.126 54.1193C159.73 54.0959 159.403 54.0726 159.403 54.0493C159.403 54.0026 159.87 53.7694 160.453 53.4894C161.432 53.0229 161.596 52.8363 161.876 51.6C162.249 50.1071 162.179 49.8038 161.456 49.6638C160.663 49.4772 160.196 48.6141 160.709 48.3109C161.246 48.0076 160.406 47.3312 159.59 47.4245C159.17 47.4478 158.937 47.3778 158.983 47.1912C159.053 46.8646 157.91 46.4914 157.35 46.678C157.117 46.748 157.047 46.7013 157.187 46.5847C157.63 46.2348 157.56 45.7683 156.977 45.2318C156.417 44.7186 156.417 44.6719 156.814 44.532C157.257 44.392 157.42 43.3656 157.047 43.109C156.93 43.039 157 42.7358 157.21 42.4325C157.537 41.9194 157.537 41.8727 157.117 41.6394C156.79 41.4762 156.65 41.1496 156.65 40.6131C156.65 40.1699 156.487 39.5167 156.301 39.1435C156.114 38.7469 155.997 38.0471 156.067 37.464C156.114 36.9274 156.091 36.4842 155.997 36.4842C155.717 36.4842 155.181 37.184 155.321 37.4173C155.391 37.5339 155.321 37.5806 155.134 37.5106C154.948 37.4406 154.225 37.9538 153.455 38.7003C152.708 39.4001 151.962 39.9832 151.822 39.9832C151.682 39.9832 151.122 40.1932 150.562 40.4731C149.722 40.8697 149.513 40.893 149.186 40.6364C148.346 39.9599 147.95 39.4701 147.903 39.0735C147.88 38.607 148.579 37.0441 148.906 36.8341C149.046 36.7408 149.093 36.5775 148.999 36.4609C148.91 36.3123 149.235 36.3533 149.255 36.181C149.255 36.1735 149.256 36.1657 149.256 36.1577C149.256 36.0177 148.953 35.9011 148.556 35.8544C148.183 35.8311 147.6 35.7378 147.32 35.6678C146.853 35.5512 146.807 35.4812 147.063 35.1779C147.296 34.8747 147.296 34.7814 147.017 34.6181C146.34 34.1982 145.547 33.5451 145.197 33.1485C144.987 32.9386 144.707 32.7519 144.544 32.7519C144.147 32.7519 144.147 33.2185 144.544 33.2185C144.684 33.2185 144.637 33.3351 144.404 33.4751C143.984 33.7317 143.611 33.685 140.835 33.1718C139.482 32.9386 139.389 32.9386 138.899 33.4284C138.619 33.7083 138.339 34.1982 138.316 34.5015C138.246 35.2479 137.756 35.7844 136.846 36.181C135.913 36.5542 135.843 36.7875 136.683 36.5775C137.243 36.4376 137.289 36.4609 137.126 36.9741C137.009 37.2773 136.939 37.6739 136.939 37.8139C136.939 37.9771 136.869 38.1171 136.776 38.1171C136.683 38.1171 136.146 38.4903 135.586 38.9335C135.027 39.3767 134.467 39.75 134.373 39.75C134.14 39.75 133.51 40.3565 133.51 40.5431C133.51 40.6131 133.604 40.683 133.744 40.683C134.164 40.683 135.143 41.6861 135.143 42.1293C135.143 42.7125 134.28 44.5086 133.79 44.9518C132.904 45.7449 130.175 47.0279 128.845 47.2612L128.464 47.3246C127.893 47.4198 127.497 47.9455 127.562 48.5208C127.632 49.1273 127.585 49.5472 127.445 49.5472C127.329 49.5472 127.212 49.8038 127.212 50.1071C127.212 50.599 126.864 51.0118 126.808 51.5005C126.801 51.5631 126.796 51.6277 126.792 51.6933C126.746 52.2298 126.512 52.6263 126.046 52.9996C125.649 53.3028 125.393 53.6527 125.463 53.7694C125.556 53.886 125.509 53.9793 125.369 53.9793C125.253 53.9793 125.113 53.8393 125.043 53.676C124.973 53.4894 124.763 53.5594 124.343 53.956C123.713 54.5391 123.247 54.4458 123.247 53.746C123.247 53.5128 123.06 53.1629 122.85 52.9996C122.617 52.8363 122.407 52.533 122.407 52.3464C122.384 52.1598 122.29 51.8099 122.197 51.6C122.057 51.2501 122.127 51.2267 122.71 51.39C124.04 51.7866 124.063 51.7866 124.063 51.1801C124.063 50.6669 123.97 50.5969 123.34 50.6202C122.733 50.6436 122.617 50.5969 122.71 50.2237C122.78 49.9671 122.92 49.7105 123.037 49.6405C123.153 49.5705 123.247 49.2673 123.247 48.9407C123.247 48.6375 123.34 48.3809 123.433 48.3809C123.736 48.3809 124.436 47.0279 124.203 46.888C124.086 46.818 123.177 46.7013 122.174 46.6314C121.101 46.5614 120.214 46.3748 120.004 46.1882C119.794 46.0015 119.188 45.7216 118.651 45.535C118.091 45.3717 117.648 45.0918 117.648 44.9518C117.648 44.532 116.855 43.7855 116.202 43.5989C115.922 43.5056 115.316 43.2956 114.896 43.109C114.312 42.8291 113.963 42.8058 113.263 42.9924C112.796 43.1324 112.33 43.249 112.26 43.249C112.073 43.249 112.33 42.036 112.726 41.0796C113.076 40.2165 112.89 39.9832 111.91 39.9832C111.467 39.9832 111.117 39.9133 111.117 39.8433C111.117 39.6333 113.239 36.9508 113.403 36.9508C113.496 36.9508 114.196 36.4842 114.942 35.9011C115.712 35.3412 116.622 34.8047 116.972 34.7347C117.322 34.6414 117.672 34.4548 117.742 34.2682C117.811 34.0816 118.231 33.8716 118.675 33.7783C119.141 33.7083 119.514 33.5451 119.514 33.4284C119.514 33.1698 120.076 33.2925 120.034 33.0374C120.032 33.028 120.03 33.0184 120.028 33.0085C119.958 32.7053 120.167 32.7053 121.264 33.1018C121.59 33.2185 121.637 33.1952 121.404 33.0319C121.171 32.8686 121.287 32.7519 121.894 32.542C122.314 32.402 122.827 32.0755 123.013 31.8189C123.2 31.5623 123.527 31.3523 123.713 31.3523C123.923 31.3523 124.203 31.2357 124.343 31.0958C124.506 30.9325 125.113 30.7925 125.743 30.7458C127.002 30.6292 129.452 29.5095 130.338 28.6231C130.851 28.1099 132.367 27.1535 132.671 27.1535C132.741 27.1535 132.157 27.76 131.388 28.4832C130.641 29.2296 130.011 29.9294 130.011 30.0694C130.011 30.2093 129.545 30.4659 128.962 30.6292C127.702 31.0024 127.679 31.4923 128.938 31.399C129.731 31.329 129.801 31.3523 129.661 31.7956C129.475 32.4487 130.058 32.4487 131.248 31.8189C131.714 31.5623 132.274 31.3523 132.461 31.3523C132.647 31.3523 132.974 31.1424 133.207 30.8858C133.697 30.3959 134.723 30.4659 134.537 31.0024C134.35 31.469 136.123 31.6789 137.103 31.329C137.779 31.0958 137.872 30.9558 137.686 30.6292C137.569 30.3959 137.336 30.256 137.149 30.326C136.963 30.3959 136.776 30.2327 136.706 29.9061C136.566 29.4162 135.703 28.7397 134.607 28.2732C134.303 28.1333 134.234 27.9466 134.35 27.4568C134.443 27.0835 134.56 26.757 134.607 26.687C134.747 26.5237 135.12 27.2235 135.003 27.4335C134.887 27.5967 135.027 27.6434 135.913 27.7134C136.053 27.7367 136.193 27.5268 136.216 27.2702C136.286 25.8706 136.286 25.8939 136.753 26.4771C137.173 26.9902 137.219 26.9902 137.359 26.617C137.453 26.4071 137.709 26.2205 137.919 26.2205C138.129 26.2205 138.596 26.0572 138.969 25.8706C139.319 25.6839 139.762 25.5207 139.902 25.5207C140.065 25.5207 140.438 25.2641 140.742 24.9375C141.162 24.4943 141.208 24.3077 141.022 24.0511C140.695 23.6545 141.232 23.0947 142.281 22.7214C142.654 22.5815 143.074 22.2083 143.238 21.905C143.471 21.3918 143.448 21.3218 143.074 21.3218C142.258 21.3218 142.351 20.8786 143.214 20.6454C143.867 20.4588 144.217 20.4821 144.731 20.7387C145.104 20.9486 145.664 21.0186 146.06 20.9486C146.923 20.7853 147.296 21.2519 146.48 21.4618C145.78 21.6251 145.174 22.2549 145.664 22.2549C145.874 22.2782 145.78 22.4415 145.43 22.7448C145.127 23.0014 144.964 23.2813 145.057 23.3746C145.238 23.5551 145.382 23.0806 145.62 22.9869C145.671 22.9666 145.726 22.9547 145.78 22.9547C146.016 22.9547 146.139 22.7852 146.149 22.4792C146.156 22.2828 146.144 22.0216 146.34 22.0216C146.713 22.0216 148.439 20.832 148.439 20.5754C148.439 20.4354 148.253 20.4121 148.043 20.5054C147.18 20.8553 146.807 20.902 146.807 20.6454C146.807 20.5054 147.063 20.3888 147.39 20.3888C147.786 20.3888 148.066 20.2255 148.206 19.8523C148.317 19.5866 148.421 19.8442 148.701 19.9117C148.729 19.9185 148.759 19.9222 148.789 19.9222C148.999 19.9222 149.069 20.0622 148.976 20.3188C148.883 20.5754 148.906 20.6454 149.093 20.5287C149.256 20.4354 149.373 20.4588 149.373 20.5521C149.373 20.6687 149.652 20.9253 150.026 21.1352L150.102 21.1836C150.27 21.2892 150.2 21.548 150.002 21.5551C149.574 21.5551 149.613 20.9396 149.185 20.9372C149.05 20.9364 148.897 20.9826 148.766 21.0886C148.533 21.3218 148.509 21.4618 148.696 21.6951C148.836 21.8584 148.859 22.115 148.766 22.2782C148.649 22.4882 148.813 22.5115 149.419 22.3949C150.049 22.2782 150.236 22.3249 150.236 22.6048C150.236 22.8614 149.979 22.9547 149.256 22.9547C147.693 22.9547 145.99 24.0744 146.457 24.8209C146.807 25.3807 149.839 24.6576 149.839 24.0044C149.839 23.8411 149.979 23.5845 150.166 23.4446C150.422 23.2346 150.492 23.258 150.446 23.5845C150.422 23.9111 150.632 24.0277 151.309 24.0744C151.939 24.1211 152.172 24.0511 152.172 23.7945C152.172 23.6079 151.939 23.4213 151.659 23.3746C151.379 23.3279 151.099 23.1413 151.052 22.9547C150.959 22.6515 150.982 22.6515 151.285 22.9081C151.589 23.1413 151.682 23.1413 151.775 22.9081C151.869 22.6745 151.932 23.0533 152.166 23.1445C152.237 23.172 152.318 23.188 152.405 23.188C152.755 23.188 152.848 23.3046 152.778 23.7012C152.708 24.0044 152.662 24.4243 152.662 24.6342C152.638 24.8675 152.452 25.0541 152.242 25.0541C152.032 25.0541 151.682 25.1941 151.495 25.3574C151.285 25.544 150.609 25.8706 149.956 26.1038C148.509 26.617 148.253 26.827 148.626 27.2002C149.023 27.5967 148.976 27.83 148.509 27.83C148.276 27.83 147.693 27.9233 147.227 28.0399C146.69 28.1799 145.92 28.1799 145.15 28.0399C144.147 27.8533 143.797 27.9 143.098 28.2499C141.861 28.8797 141.348 29.4396 141.651 29.8128C142.048 30.2793 143.004 30.3959 144.077 30.0927C144.614 29.9294 145.43 29.8361 145.874 29.8594C146.317 29.8828 146.923 29.9061 147.227 29.9294C147.53 29.9294 147.693 30.046 147.623 30.186C147.553 30.3026 147.646 30.4193 147.856 30.4193C148.09 30.4193 148.206 30.5826 148.16 30.8158C148.113 31.0491 148.323 31.4923 148.603 31.8189C148.906 32.1455 149.139 32.5653 149.139 32.7986C149.139 33.3584 149.489 33.4984 150.166 33.1952C150.656 32.9619 150.842 33.0085 151.379 33.4284C151.729 33.685 152.242 33.9183 152.545 33.9183C152.848 33.9183 153.618 34.1282 154.271 34.3848C154.901 34.6414 155.438 34.8047 155.438 34.7347C155.438 34.6647 155.647 34.828 155.927 35.1079C156.487 35.6678 157.094 35.3645 156.907 34.6181C156.767 34.1282 156.137 33.9649 156.137 34.4081C156.137 34.6621 155.693 34.5067 155.631 34.2604C155.612 34.1816 155.607 34.0951 155.624 34.0116C155.647 33.755 155.438 33.3351 155.088 33.0319C154.761 32.7519 154.504 32.4254 154.504 32.3087C154.504 32.1921 154.294 32.0521 154.038 31.9822C153.781 31.9122 153.571 31.6789 153.571 31.4457C153.571 31.0724 153.595 31.0724 154.038 31.469C154.057 31.4866 154.077 31.5036 154.096 31.5199C154.424 31.7924 154.921 31.7207 155.158 32.0755C155.438 32.472 155.531 32.4954 155.857 32.2388C156.067 32.0755 156.161 32.0521 156.067 32.1921C155.974 32.3554 156.137 32.5187 156.464 32.612C156.767 32.682 157.07 32.9619 157.14 33.2185C157.28 33.7317 157.863 33.8483 158.143 33.4284C158.213 33.2885 158.143 33.0785 157.98 32.9619C157.747 32.8219 157.724 32.7519 157.957 32.7519C158.097 32.7519 158.237 32.542 158.237 32.2854C158.237 32.0055 158.353 31.8889 158.563 31.9588C158.727 32.0288 159.053 31.9122 159.263 31.6789C159.613 31.3523 159.613 31.2357 159.31 30.7925C159.076 30.4193 159.053 30.1627 159.216 29.8594C159.473 29.3929 159.146 29.0197 158.447 29.0197C158.05 29.0197 157.77 28.5998 157.63 27.7834C157.584 27.5501 157.677 27.3868 157.887 27.3868C158.307 27.3868 158.33 27.1302 157.957 26.7103C157.7 26.4537 157.724 26.4304 157.957 26.547C158.12 26.6403 158.4 26.617 158.54 26.4771C158.913 26.1738 159.94 27.0369 160.056 27.7367C160.098 27.976 160.204 28.2245 160.326 28.412C160.531 28.7274 160.956 28.6181 161.293 28.7864C161.829 29.0896 162.436 28.8331 162.669 28.2266C162.832 27.8067 162.995 27.7367 163.695 27.8067C164.232 27.8767 164.535 27.8067 164.535 27.6201C164.535 27.2702 165.118 26.687 165.515 26.687C165.678 26.687 166.075 26.4771 166.378 26.2438C166.646 26.0341 166.637 25.6257 166.36 25.4279L166.261 25.3574C165.888 25.1008 165.445 24.9375 165.258 25.0075C165.071 25.0775 165.001 25.0075 165.095 24.8442C165.188 24.7042 165.118 24.5876 164.932 24.5876C164.768 24.5876 164.372 24.331 164.068 23.9811C163.672 23.5845 163.345 23.4446 162.972 23.5379C162.692 23.6079 162.389 23.5379 162.296 23.3979C162.226 23.258 161.782 23.0014 161.363 22.8148C160.919 22.6281 160.569 22.3949 160.569 22.3016C160.569 21.9983 161.666 21.4851 161.946 21.6484C162.389 21.9283 162.715 21.5551 162.389 21.1352C162.156 20.8553 162.156 20.762 162.389 20.692C163.159 20.4121 162.179 19.1291 160.919 18.7792C160.429 18.6393 159.94 18.5926 159.823 18.6626C159.73 18.7326 159.636 18.6159 159.636 18.406C159.636 18.1961 159.543 18.0794 159.426 18.1494C159.31 18.2194 159.17 18.0794 159.1 17.8228C159.007 17.4496 158.773 17.3563 157.957 17.3563C157.397 17.3563 156.79 17.4963 156.604 17.6362C156.324 17.8928 156.324 17.8695 156.581 17.5196C156.814 17.2397 156.814 17.123 156.627 17.123C156.464 17.123 156.417 17.0064 156.487 16.8898C156.557 16.7498 156.464 16.6565 156.277 16.6565C156.067 16.6565 155.904 16.4932 155.904 16.3066C155.904 16.12 155.671 15.91 155.391 15.8634C155.088 15.7934 154.644 15.7001 154.388 15.6534C154.061 15.5835 154.108 15.5368 154.574 15.5135C154.924 15.4902 155.204 15.3502 155.204 15.1869C155.204 14.5104 154.388 14.1605 152.522 14.0206C151.005 13.9039 150.586 13.9506 150.259 14.2538C149.862 14.6037 149.839 14.6037 149.839 14.2538C149.839 13.764 148.696 13.7407 146.573 14.2072C144.801 14.6037 144.357 14.6271 145.174 14.2772C145.384 14.1929 145.384 13.8949 145.174 13.8106C143.588 13.2041 139.109 14.5804 137.196 16.2833C136.963 16.4932 136.706 16.6565 136.613 16.6565C136.57 16.6612 136.5 16.7116 136.417 16.7898C135.752 17.4123 136.111 18.0696 137.009 18.2194C137.139 18.2452 137.116 18.4361 136.985 18.4317L136.916 18.4293C135.353 18.406 135.073 18.6859 135.866 19.4324C136.216 19.759 136.566 19.8756 136.846 19.7823C137.149 19.689 137.243 19.7356 137.149 19.8989C136.963 20.1788 137.173 20.2255 138.526 20.2022C139.039 20.1788 139.295 20.2255 139.109 20.3188C138.549 20.5521 138.176 21.0186 138.106 21.6018C138.059 21.905 137.872 22.1383 137.686 22.1383C137.499 22.1383 136.986 22.4415 136.52 22.7914C135.656 23.4446 135.237 23.6079 135.493 23.188C135.563 23.048 135.493 22.9547 135.307 22.9547C135.124 22.9547 134.848 23.1421 134.626 23.3649C134.244 23.7477 134.201 24.3216 133.697 24.5176C133.114 24.7742 132.927 24.7742 132.764 24.5409C132.461 24.0511 132.671 23.5612 133.324 23.2346C134.677 22.5815 134.21 21.0419 132.787 21.5318C132.134 21.7417 132.087 21.7417 132.227 21.3218C132.344 20.9486 132.274 20.8553 131.901 20.8553C131.364 20.8553 131.201 20.0389 131.668 19.8756C131.878 19.8056 131.878 19.6423 131.668 19.2458C131.458 18.8492 131.434 18.6393 131.644 18.406C131.994 17.9861 131.714 17.123 131.224 17.0997C131.014 17.0997 131.271 16.9131 131.831 16.6798C132.391 16.4699 132.927 16.1666 132.997 16.0267C133.09 15.8867 133.744 15.7234 134.443 15.6768C136.03 15.5135 139.599 13.9273 139.085 13.6007C138.899 13.4841 138.339 13.3908 137.826 13.3908C137.313 13.3908 136.473 13.3208 135.96 13.2508C134.84 13.1108 133.137 13.4374 133.207 13.8106C133.254 13.9506 133.114 14.0206 132.904 13.9273C132.717 13.8573 132.321 13.9972 132.041 14.2538C131.761 14.5104 131.341 14.7904 131.084 14.8837C130.804 14.977 130.688 15.1869 130.781 15.4668C130.921 16.0033 130.245 16.6798 129.148 17.123C128.728 17.2863 128.378 17.4729 128.378 17.5662C128.378 17.6595 128.169 17.7295 127.889 17.7529C127.235 17.7995 126.279 18.4527 126.279 18.8492C126.279 19.0125 126.022 19.2924 125.696 19.4557C125.299 19.6656 125.113 19.9456 125.159 20.2488C125.206 20.622 125.089 20.762 124.576 20.8087C124.18 20.8553 123.736 20.7387 123.48 20.5054C123.13 20.2022 122.967 20.1788 122.407 20.4354C122.034 20.5987 121.637 20.8553 121.544 21.0186C121.45 21.1819 120.937 21.3918 120.447 21.4618C119.444 21.6251 119.164 22.1616 120.028 22.3016C120.307 22.3482 120.564 22.5115 120.634 22.6981C120.704 22.9081 120.587 22.9547 120.237 22.8381C119.888 22.7448 119.608 22.8614 119.234 23.2346C118.745 23.7478 118.488 23.7945 116.062 23.8411C113.916 23.8645 113.403 23.8178 113.286 23.5146C113.216 23.3046 113.006 23.2346 112.726 23.3046C112.096 23.5146 111.607 23.1647 112.19 22.9314C112.61 22.7681 112.61 22.7681 112.19 22.6048C111.933 22.4882 111.7 22.5348 111.583 22.7214C111.443 22.9314 111.373 22.8847 111.303 22.4882C111.28 22.1849 111.093 21.905 110.907 21.835C110.72 21.7651 110.65 21.835 110.72 21.9517C110.79 22.0683 110.557 22.1383 110.16 22.0683C109.945 22.0395 109.897 21.7486 110.09 21.6484C110.534 21.4152 110.813 21.3918 110.977 21.5551C111.28 21.8584 111.653 21.8584 111.467 21.5551C111.397 21.4385 111.467 21.3918 111.677 21.4618C111.863 21.5318 112.073 21.7184 112.12 21.8817C112.19 22.0683 112.773 22.1849 113.683 22.2083C114.499 22.2316 115.292 22.3715 115.479 22.5115C115.735 22.7214 115.922 22.6748 116.319 22.2782C116.622 22.0216 116.995 21.7884 117.135 21.7884C117.462 21.7884 117.485 21.6484 117.205 21.1119C117.042 20.8087 117.088 20.7153 117.438 20.692C117.672 20.6687 117.881 20.8087 117.881 20.9953C117.881 21.1819 117.951 21.3218 118.021 21.3218C118.325 21.3218 119.514 20.3421 119.514 20.0855C119.514 19.8289 118.535 19.2924 117.368 18.9192C116.808 18.7326 116.832 18.3593 117.415 18.0561C117.672 17.9161 117.881 17.6595 117.881 17.4729C117.881 16.6798 118.908 15.5835 120.121 15.047C120.821 14.7437 121.38 14.4171 121.38 14.3238C121.38 14.0672 120.751 13.8573 119.981 13.8573C119.486 13.8573 118.82 14.0371 118.466 14.21C118.198 14.3406 118.581 14.5155 118.581 14.8137C118.581 14.9536 118.395 14.9536 118.138 14.7904C117.228 14.3238 116.249 14.5104 115.502 15.3035C115.106 15.7234 114.709 16.3066 114.592 16.5865C114.476 16.8898 114.289 17.123 114.173 17.123C113.916 17.123 113.963 16.6798 114.242 16.2366C114.359 16.05 114.266 15.7468 114.009 15.4668C113.519 14.9303 112.516 14.8603 112.516 15.3735C112.516 15.8167 112.19 15.8167 110.953 15.3502C110.16 15.047 109.834 15.0236 109.204 15.2336C108.597 15.4435 108.504 15.4435 108.737 15.2102C109.134 14.8137 109.087 14.5571 108.597 14.5571C108.364 14.5571 108.014 14.3005 107.851 13.9739C107.478 13.2741 106.965 13.0875 105.285 13.0175C104.585 12.9942 102.999 12.9009 101.786 12.8542C100.573 12.7843 99.2667 12.7843 98.9168 12.8542ZM143.891 14.907C143.518 15.1169 143.004 15.6068 142.748 16.0267L142.636 16.2257C142.512 16.4458 142.188 16.4244 142.095 16.19C141.978 15.8867 141.978 15.5835 142.071 15.5601C142.188 15.5135 142.631 15.2802 143.074 15.0236C143.518 14.7904 144.054 14.5804 144.241 14.5804C144.427 14.5804 144.287 14.7204 143.891 14.907ZM150.376 15.2802C150.492 15.7234 150.632 15.7701 151.519 15.6534C152.382 15.5368 152.428 15.5368 151.939 15.7701C151.612 15.8867 150.889 16.0267 150.306 16.05C149.279 16.0967 149.256 16.0733 149.676 15.6534C149.909 15.4202 150.026 15.1403 149.956 15.0236C149.862 14.8837 149.909 14.7904 150.026 14.7904C150.142 14.7904 150.306 15.0236 150.376 15.2802ZM141.745 16.5865C141.535 16.6565 141.511 16.8198 141.675 17.1464C141.931 17.6129 141.605 17.7995 141.325 17.3563C141.115 17.0064 141.371 16.4232 141.721 16.4466C141.955 16.4466 141.955 16.4932 141.745 16.5865ZM155.554 16.8898C155.484 17.0064 155.228 17.123 155.018 17.0997C154.691 17.0997 154.714 17.0531 155.088 16.8898C155.717 16.6332 155.717 16.6332 155.554 16.8898ZM158.937 18.7559C158.75 18.8725 158.377 18.9658 158.12 18.9658C157.747 18.9425 157.747 18.9192 158.237 18.7559C159.053 18.4993 159.356 18.4993 158.937 18.7559ZM161.363 19.7356C161.246 19.9222 161.316 19.9456 161.619 19.8289C161.876 19.7356 162.132 19.7823 162.226 19.9689C162.366 20.2022 162.016 20.3421 160.873 20.5521C159.053 20.8786 158.657 20.8087 160.219 20.4121C160.412 20.3659 160.589 20.3121 160.74 20.2574C161.112 20.1226 161.013 19.4557 161.409 19.4557C161.479 19.4557 161.456 19.5723 161.363 19.7356ZM137.802 19.7123C138.077 19.6873 137.972 19.9516 137.708 19.8692C137.693 19.8643 137.677 19.8587 137.663 19.8523C137.476 19.7823 137.546 19.7123 137.802 19.7123ZM160.336 20.0155C160.336 20.0622 160.126 20.1555 159.87 20.2255C159.579 20.3048 159.641 19.9878 159.937 19.9328C160.138 19.8955 160.336 19.9144 160.336 20.0155ZM81.3518 20.1788C81.6265 20.1539 81.5209 20.4181 81.2577 20.3357C81.2421 20.3308 81.2268 20.3252 81.2118 20.3188C81.0252 20.2488 81.0952 20.1788 81.3518 20.1788ZM142.375 20.3888C142.375 20.5054 142.141 20.622 141.861 20.622C141.418 20.622 141.395 20.5754 141.675 20.3888C142.118 20.0855 142.375 20.0855 142.375 20.3888ZM145.827 20.3188C145.594 20.3654 145.22 20.3654 145.01 20.3188C144.777 20.2488 144.964 20.2022 145.407 20.2022C145.85 20.2022 146.037 20.2488 145.827 20.3188ZM79.8589 20.622C79.6723 20.7387 79.3457 20.832 79.1591 20.832C78.9258 20.832 78.9725 20.762 79.2757 20.622C79.9055 20.3421 80.2788 20.3421 79.8589 20.622ZM130.711 20.622C130.711 20.7387 130.618 20.8553 130.478 20.8553C130.361 20.8553 130.245 20.7387 130.245 20.622C130.245 20.4821 130.361 20.3888 130.478 20.3888C130.618 20.3888 130.711 20.4821 130.711 20.622ZM126.746 21.1352C126.746 21.2285 126.489 21.4851 126.162 21.6951C125.439 22.1616 125.393 22.5815 126.069 22.4182C126.349 22.3482 126.489 22.3715 126.372 22.4649C126.116 22.6748 124.343 23.3046 123.41 23.5146C123.013 23.6079 122.687 23.8878 122.547 24.2377C122.384 24.6342 122.22 24.7742 121.94 24.6576C121.567 24.5176 121.567 24.4943 121.964 24.2144C122.197 24.0277 122.29 23.7945 122.174 23.6079C122.08 23.4446 122.08 23.3746 122.197 23.4679C122.225 23.4935 122.263 23.5094 122.307 23.5171C123.032 23.6433 123.371 22.6831 124.086 22.5115C124.67 22.3715 125.066 22.1383 124.996 22.0216C124.903 21.8817 124.74 21.8584 124.6 21.9283C124.46 22.0216 124.413 21.905 124.46 21.6484C124.506 21.3452 124.716 21.2285 125.043 21.2752C125.323 21.3218 125.603 21.2519 125.673 21.1119C125.836 20.8553 126.746 20.8553 126.746 21.1352ZM74.727 21.4385C73.654 21.8584 73.0475 21.8584 73.7939 21.4618C74.1205 21.2752 74.6337 21.1352 74.9603 21.1352C75.4734 21.1352 75.4268 21.1586 74.727 21.4385ZM162.202 21.2752C162.202 21.3685 161.829 21.4152 161.386 21.3685C160.943 21.3218 160.569 21.2285 160.569 21.1819C160.569 21.1352 160.943 21.0886 161.386 21.0886C161.829 21.0886 162.202 21.1819 162.202 21.2752ZM131.504 22.0683C131.364 22.5582 130.921 23.0247 130.781 22.8614C130.665 22.7448 131.294 21.5551 131.481 21.5551C131.598 21.5551 131.598 21.7884 131.504 22.0683ZM149.722 22.0216C149.652 22.1383 149.466 22.2549 149.349 22.2549C149.209 22.2549 149.186 22.1383 149.256 22.0216C149.326 21.8817 149.513 21.7884 149.629 21.7884C149.769 21.7884 149.792 21.8817 149.722 22.0216ZM142.608 22.2549C142.608 22.3715 142.491 22.4882 142.351 22.4882C142.235 22.4882 142.188 22.3715 142.258 22.2549C142.328 22.115 142.445 22.0216 142.514 22.0216C142.561 22.0216 142.608 22.115 142.608 22.2549ZM150.865 22.045C151.14 22.02 151.035 22.2842 150.771 22.2018C150.756 22.197 150.74 22.1913 150.726 22.1849C150.539 22.115 150.609 22.045 150.865 22.045ZM135.027 23.6312C134.957 23.7712 134.77 23.8878 134.63 23.8878C134.49 23.8878 134.49 23.7478 134.677 23.5379C135.003 23.1413 135.283 23.2346 135.027 23.6312ZM120.447 23.7478C120.447 23.7945 120.284 23.8878 120.098 23.9811C119.911 24.0511 119.748 24.0044 119.748 23.8878C119.748 23.7478 119.911 23.6545 120.098 23.6545C120.284 23.6545 120.447 23.7012 120.447 23.7478ZM151.659 23.8178C151.425 23.8645 151.052 23.8645 150.842 23.8178C150.609 23.7478 150.796 23.7012 151.239 23.7012C151.682 23.7012 151.869 23.7478 151.659 23.8178ZM105.518 24.9608C105.518 25.0075 105.308 25.0541 105.028 25.0541C104.772 25.0541 104.609 24.9608 104.702 24.8209C104.818 24.6342 105.518 24.7509 105.518 24.9608ZM121.474 25.0774C121.748 25.0525 121.643 25.3167 121.38 25.2343C121.364 25.2294 121.349 25.2238 121.334 25.2174C121.147 25.1474 121.217 25.0774 121.474 25.0774ZM89.3062 25.5207C89.0496 25.614 88.4898 25.8472 88.0466 26.0105C87.2768 26.2904 87.2768 26.3138 87.8133 26.5237C88.1165 26.6403 88.9097 26.6637 89.5395 26.5704L89.8318 26.5353C90.0403 26.5103 90.1374 26.7857 89.9594 26.8969C89.5395 27.1535 88.8397 27.3635 88.3032 27.3401C87.1835 27.3401 86.3437 27.5734 86.507 27.8767C86.6236 28.0633 86.1338 28.2032 85.6206 28.1099C85.5273 28.0866 85.0607 28.2499 84.5942 28.4365C84.1277 28.6231 83.7078 28.7397 83.6378 28.6698C83.4512 28.4832 85.2474 27.8067 85.5973 27.9466C85.7605 28.0166 85.9705 27.9466 86.0638 27.8067C86.3437 27.3635 84.291 27.8067 83.5445 28.3665C82.9847 28.7397 82.7514 28.7864 82.3548 28.5998C81.935 28.3898 81.9116 28.3432 82.2615 28.3432C82.4715 28.3199 82.6581 28.2032 82.6581 28.0866C82.6581 27.9466 82.868 27.8533 83.1013 27.8533C83.3579 27.8533 83.7078 27.69 83.9177 27.5034C84.1043 27.3168 84.4542 27.1769 84.6875 27.2002C84.9208 27.2002 85.294 27.2002 85.5506 27.2002C86.4137 27.1069 85.4573 26.4537 84.5476 26.5004L84.4829 26.5022C84.4116 26.5042 84.4067 26.6084 84.4776 26.617C84.8741 26.6637 85.224 26.8036 85.224 26.9436C85.224 27.0895 85.1054 27.1303 84.9637 27.0954C84.7187 27.0349 84.2924 26.9362 83.8244 26.827C83.5551 26.7682 83.5655 26.3806 83.8376 26.3364L84.4076 26.2438C85.434 26.0805 86.9502 25.8006 87.79 25.6373C89.5628 25.2641 89.9594 25.2407 89.3062 25.5207ZM154.271 25.4973C154.271 25.614 154.551 25.7773 154.878 25.8472C155.694 26.0572 155.344 26.3604 154.108 26.5237C153.268 26.617 153.152 26.5937 153.291 26.2438C153.571 25.5207 154.271 24.9841 154.271 25.4973ZM166.261 25.7306C166.354 25.8472 166.191 25.8939 165.935 25.8239C165.398 25.6839 165.328 25.5207 165.795 25.5207C165.981 25.5207 166.191 25.614 166.261 25.7306ZM33.9985 26.0105C34.2732 25.9855 34.1676 26.2498 33.9044 26.1674C33.8888 26.1625 33.8735 26.1569 33.8585 26.1505C33.6719 26.0805 33.7419 26.0105 33.9985 26.0105ZM134.14 26.9669C134.14 27.3662 133.934 27.1967 133.537 27.1572C133.512 27.1548 133.488 27.1535 133.464 27.1535C133.158 27.1535 133.22 26.8225 133.051 26.5676C133.042 26.5529 133.031 26.5382 133.021 26.5237C132.811 26.2904 132.904 26.2205 133.44 26.2205C134.07 26.2205 134.14 26.2904 134.14 26.9669ZM128.495 27.76C128.962 28.1099 128.962 28.1333 128.402 28.0399C128.075 27.9933 127.795 27.83 127.725 27.6667C127.609 27.2935 127.935 27.3168 128.495 27.76ZM29.3565 29.2529C29.2865 29.3696 29.0766 29.4862 28.9366 29.4862C28.7733 29.4862 28.7966 29.3929 29.0066 29.2529C29.4498 28.973 29.5431 28.973 29.3565 29.2529ZM146.34 29.5095C146.34 29.6262 146.177 29.7195 145.99 29.7195C145.78 29.7195 145.71 29.5795 145.78 29.3696C145.92 29.0197 146.34 29.113 146.34 29.5095ZM149.256 31.3523C149.186 31.469 149.023 31.5856 148.883 31.5856C148.766 31.5856 148.673 31.469 148.673 31.3523C148.673 31.2124 148.836 31.1191 149.046 31.1191C149.233 31.1191 149.326 31.2124 149.256 31.3523ZM88.793 34.1282C89.1896 34.4781 90.3792 34.3382 91.8721 33.825C92.1287 33.7317 91.8255 33.9416 91.1723 34.2915C90.5425 34.6414 89.4928 35.0146 88.8397 35.1079C88.2099 35.2246 87.5567 35.4579 87.4401 35.6445C87.1368 36.041 83.6378 36.3676 83.1246 36.0177C82.8214 35.8078 82.8214 35.7844 83.1246 35.7611C83.3112 35.7378 83.6611 35.7611 83.8944 35.7844C84.1277 35.8078 84.5942 35.6445 84.9441 35.4112C85.294 35.1779 85.9472 34.9447 86.3904 34.8514C87.1135 34.7347 87.2068 34.6414 87.2768 33.9883C87.3468 33.2885 87.3934 33.2418 87.8599 33.5217C88.1399 33.685 88.5598 33.9416 88.793 34.1282ZM156.231 34.8747C156.505 34.8497 156.4 35.114 156.136 35.0315C156.121 35.0267 156.106 35.021 156.091 35.0146C155.904 34.9447 155.974 34.8747 156.231 34.8747ZM36.2379 35.9244C36.2379 35.971 36.0746 36.0177 35.888 36.0177C35.7014 36.0177 35.5381 35.9244 35.5381 35.7844C35.5381 35.6678 35.7014 35.6211 35.888 35.6911C36.0746 35.7844 36.2379 35.8777 36.2379 35.9244ZM33.9985 36.041C34.2732 36.016 34.1676 36.2803 33.9044 36.1979C33.8888 36.193 33.8735 36.1874 33.8585 36.181C33.6719 36.111 33.7419 36.041 33.9985 36.041ZM33.322 36.4842C33.1121 36.8108 32.2723 37.2307 32.2723 37.0207C32.2723 36.8791 32.6934 36.5971 33.1705 36.3726C33.3152 36.3045 33.3936 36.3411 33.322 36.4842ZM25.3909 38.6303C25.7615 38.6303 25.6253 38.8105 25.257 38.7702C25.2049 38.7645 25.1553 38.7568 25.111 38.7469C24.8777 38.6769 25.0177 38.6303 25.3909 38.6303ZM27.1404 38.6769C27.1404 38.7236 26.9305 38.8169 26.6739 38.8869C26.3831 38.9662 26.445 38.6492 26.7414 38.5942C26.9419 38.5569 27.1404 38.5758 27.1404 38.6769ZM50.8871 40.5664C50.8171 40.6364 50.6771 40.5198 50.6071 40.2865C50.4672 39.9599 50.4905 39.9366 50.7238 40.1465C50.8871 40.3098 50.957 40.4964 50.8871 40.5664ZM51.2836 41.1496C51.3536 41.2662 51.3303 41.3828 51.1903 41.3828C51.0737 41.3828 50.8871 41.2662 50.8171 41.1496C50.7471 41.0096 50.7704 40.9163 50.9104 40.9163C51.027 40.9163 51.2136 41.0096 51.2836 41.1496ZM50.1173 41.6161C50.024 41.7561 50.0706 42.0127 50.2106 42.176C50.3972 42.3859 50.3739 42.4792 50.1173 42.4792C49.8797 42.4792 49.9689 42.1348 49.7903 41.9782C49.7759 41.9655 49.7605 41.9536 49.7441 41.9427C49.5574 41.8494 49.5808 41.7327 49.7907 41.5928C50.2106 41.3362 50.2806 41.3362 50.1173 41.6161ZM20.1424 41.8494C20.1424 41.966 20.0958 42.0826 20.0491 42.0826C19.9791 42.0826 19.8625 41.966 19.7925 41.8494C19.7225 41.7094 19.7692 41.6161 19.8858 41.6161C20.0258 41.6161 20.1424 41.7094 20.1424 41.8494ZM52.1001 42.8991C51.89 43.1542 51.5062 43.2741 51.2094 43.2403C50.921 43.2075 50.8636 42.8694 51.1437 42.9457C51.3769 43.0157 51.6568 42.9457 51.7268 42.8058C51.8201 42.6658 52.0067 42.5492 52.1467 42.5492C52.31 42.5492 52.2867 42.6891 52.1001 42.8991ZM51.7735 43.7622C51.8668 43.9021 51.8435 43.9488 51.6802 43.8555C51.5402 43.7855 51.3303 43.8788 51.2136 44.0887C51.0737 44.3453 50.957 44.392 50.8404 44.2054C50.7471 44.0654 50.7938 43.9488 50.9337 43.9488C51.0737 43.9488 51.1203 43.8322 51.0504 43.7155C50.9804 43.5756 51.0737 43.4823 51.2603 43.4823C51.4469 43.4823 51.6802 43.5989 51.7735 43.7622ZM50.2339 44.5086C50.2339 44.5553 50.0473 44.7419 49.8374 44.9052C49.4641 45.2084 49.4408 45.1851 49.7441 44.8119C50.0473 44.4386 50.2339 44.322 50.2339 44.5086ZM51.167 45.465C51.167 45.6516 51.1203 45.8149 51.0737 45.8149C51.027 45.8149 50.9337 45.6516 50.8404 45.465C50.7704 45.2784 50.8171 45.1151 50.9337 45.1151C51.0737 45.1151 51.167 45.2784 51.167 45.465ZM52.4266 45.1385C52.7014 45.1135 52.5957 45.3777 52.3325 45.2953C52.3169 45.2904 52.3016 45.2848 52.2867 45.2784C52.1001 45.2084 52.17 45.1385 52.4266 45.1385ZM49.954 47.0512C49.952 47.0552 49.95 47.0591 49.9481 47.0631C49.7629 47.4325 49.6041 47.4178 49.6041 47.0046C49.6041 46.5381 49.6974 46.3514 49.9073 46.4447C50.1639 46.5381 50.1639 46.678 49.954 47.0512ZM98.8935 49.8038C99.0801 50.1304 98.5203 53.5128 98.2637 53.5128C98.147 53.5128 98.0537 53.4428 98.0537 53.3261C98.0537 53.2328 97.9604 53.0229 97.8438 52.8363C97.6805 52.5797 97.5405 52.5564 97.284 52.7663C97.0507 52.9762 96.9574 52.9762 96.9574 52.743C96.9574 52.603 96.8874 52.2298 96.7941 51.9265C96.6541 51.53 96.7241 51.3667 97.1673 51.2034L97.2332 51.1748C97.3394 51.1286 97.3064 50.9701 97.1906 50.9701C96.4908 50.9468 96.5142 50.8069 97.3073 50.1304C97.9837 49.5239 98.6369 49.3839 98.8935 49.8038ZM157.304 49.6405C157.304 49.6872 156.954 49.8505 156.557 49.9904C155.857 50.2237 155.834 50.2004 156.231 49.8971C156.65 49.5705 157.304 49.4306 157.304 49.6405ZM50.4672 50.4803C50.4672 50.5969 50.4205 50.7135 50.3739 50.7135C50.3039 50.7135 50.1873 50.5969 50.1173 50.4803C50.0473 50.3403 50.094 50.247 50.2106 50.247C50.3506 50.247 50.4672 50.3403 50.4672 50.4803ZM98.287 54.5625C98.287 54.6091 97.9837 54.9823 97.5872 55.3789C97.144 55.8221 96.8874 55.9621 96.8874 55.7521C96.8874 55.5655 97.1673 55.1923 97.4939 54.9357C98.1237 54.4458 98.287 54.3525 98.287 54.5625ZM52.9398 57.385C52.9398 57.6416 53.1964 57.8749 53.6863 58.0381C54.2928 58.2481 54.3861 58.3647 54.1528 58.598C53.9429 58.8079 53.9429 58.8779 54.1528 58.8779C54.3161 58.8779 54.4327 59.1112 54.4327 59.3911C54.4327 59.7876 54.3628 59.8343 54.2228 59.601C54.1062 59.4144 53.8962 59.3678 53.7329 59.4611C53.5463 59.5777 53.4763 59.5311 53.5463 59.3211C53.6163 59.1345 53.383 58.7146 53.0564 58.388C52.4033 57.7582 52.1934 57.0118 52.6832 57.0118C52.8465 57.0118 52.9631 57.1751 52.9398 57.385ZM110.72 59.4844C110.93 59.5544 111 59.811 110.953 60.1609C110.813 60.7907 111.537 61.3039 112.213 61.0473C112.563 60.9073 112.586 60.954 112.353 61.2572C112.113 61.5312 112.305 61.9276 112.181 62.2696C112.176 62.2821 112.171 62.2945 112.166 62.3069C111.933 62.8435 111.886 62.9834 111.84 63.4266C111.84 63.6132 111.7 63.7765 111.56 63.7765C111.263 63.7765 111.506 63.18 111.209 63.1742C111.052 63.1711 110.813 63.1952 110.464 63.24C109.857 63.31 109.181 63.4733 108.971 63.5899C108.457 63.8465 107.454 63.5899 107.291 63.17C107.221 62.9834 106.988 62.8435 106.755 62.8435C106.475 62.8435 106.591 62.6802 107.128 62.3536C107.991 61.8404 107.944 61.5372 107.035 61.7704C106.661 61.8637 106.498 62.027 106.591 62.1903C106.685 62.3536 106.661 62.3769 106.498 62.2836C106.358 62.1903 105.892 62.3069 105.472 62.5169C105.052 62.7501 104.399 63.0067 104.002 63.1C103.629 63.1934 103.046 63.3566 102.719 63.4499C102.136 63.5899 102.136 63.5899 102.602 63.2167C103.046 62.8201 103.022 62.8201 102.089 62.9134C101.529 62.9834 101.086 63.0067 101.086 62.9601C101.086 62.7501 103.675 61.4672 104.655 61.2106C105.262 61.0473 105.822 60.814 105.892 60.7207C105.961 60.6041 106.125 60.5108 106.288 60.5108C106.428 60.5108 106.335 60.6741 106.078 60.8607C105.845 61.0473 105.752 61.2106 105.915 61.2106C106.405 61.2106 107.384 60.7674 107.384 60.5341C107.384 60.3941 107.244 60.3475 107.058 60.4175C106.871 60.4875 106.661 60.4175 106.545 60.2542C106.451 60.0909 106.475 60.0209 106.591 60.1142C106.708 60.1842 107.384 60.0442 108.084 59.811C109.414 59.3444 110.137 59.2511 110.72 59.4844ZM53.1264 60.0209C53.535 59.6495 53.3742 60.4712 52.8223 60.4862C52.7916 60.487 52.7607 60.4875 52.7299 60.4875C52.24 60.4641 52.1467 60.4175 52.45 60.3475C52.7066 60.2775 53.0098 60.1376 53.1264 60.0209ZM143.191 60.9773C143.378 61.2806 143.074 61.2806 142.608 60.9773C142.305 60.7907 142.305 60.744 142.654 60.744C142.864 60.744 143.121 60.8374 143.191 60.9773ZM145.64 64.1497C145.64 64.1964 145.43 64.2431 145.15 64.2431C144.894 64.2431 144.731 64.1497 144.824 64.0098C144.94 63.8232 145.64 63.9398 145.64 64.1497ZM114.966 64.593C115.106 64.6396 114.709 64.6863 114.103 64.6863C113.473 64.7096 112.983 64.6163 112.983 64.523C112.983 64.3597 114.429 64.4063 114.966 64.593ZM110.65 64.8496C110.65 65.0595 109.181 66.4358 108.877 66.5291C108.108 66.739 107.851 66.9256 107.081 67.8354C106.125 68.9784 105.752 69.7948 105.752 70.7046C105.752 71.1011 105.402 71.6376 104.725 72.3141C103.605 73.4571 102.976 73.6204 102.929 72.804C102.859 71.451 103.885 69.5149 105.868 67.3222C106.848 66.2258 107.058 65.6427 106.218 66.3891C106.031 66.5524 106.125 66.3891 106.451 66.0392C106.523 65.958 106.596 65.8846 106.668 65.8196C107.095 65.4367 107.778 65.6216 108.248 65.2928C108.854 64.8729 110.65 64.5463 110.65 64.8496ZM114.616 65.5494C115.385 65.8759 115.455 65.8759 115.829 65.4327C116.412 64.7096 117.088 64.8029 117.392 65.666C117.951 67.2755 116.925 68.1853 116.085 66.809C115.572 65.9226 115.106 65.8759 115.269 66.7157C115.362 67.2056 115.222 67.4855 114.662 67.9753C114.266 68.3019 113.869 68.8618 113.799 69.2117C113.659 69.7248 112.82 70.5413 112.423 70.5413C112.353 70.5413 112.33 70.0981 112.353 69.5382C112.4 68.6751 112.353 68.5585 111.933 68.6285C111.677 68.6751 111.303 68.8618 111.093 69.0484C110.907 69.235 110.65 69.3749 110.534 69.3749C110.44 69.3749 110.837 68.9084 111.443 68.3252C112.05 67.7421 112.47 67.1589 112.4 67.0423C112.33 66.9256 112.376 66.7857 112.516 66.6924C112.913 66.4358 112.773 66.1092 112.166 65.8293C111.84 65.6893 111.583 65.4094 111.583 65.2228C111.583 64.9662 111.793 64.9195 112.703 65.0595C113.309 65.1528 114.173 65.3627 114.616 65.5494ZM115.176 65.2228C115.106 65.3161 114.966 65.4094 114.826 65.4094C114.709 65.4094 114.616 65.3161 114.616 65.2228C114.616 65.1061 114.779 65.0128 114.966 65.0128C115.152 65.0128 115.246 65.1061 115.176 65.2228ZM122.78 68.8384C122.757 69.6082 121.357 70.1447 119.864 69.9581C119.211 69.8648 118.278 69.9115 117.811 70.0281C116.692 70.3313 116.692 69.9814 117.835 69.4449C118.395 69.165 119.234 69.025 120.284 69.025C121.147 69.025 121.847 68.9551 121.847 68.8384C121.847 68.7451 122.057 68.6751 122.314 68.6751C122.57 68.6751 122.78 68.7451 122.78 68.8384ZM117.415 71.1245C117.415 71.2877 115.572 72.1975 113.916 72.804C111.653 73.6671 110.697 73.8537 110.184 73.5738C109.577 73.2472 109.577 72.874 110.184 72.874C110.44 72.874 110.65 73.0139 110.65 73.1539C110.65 73.3871 110.697 73.3871 110.837 73.1772C110.93 73.0139 111.63 72.5707 112.4 72.1975C113.519 71.6143 113.963 71.521 114.849 71.6143C115.595 71.7076 115.805 71.6843 115.572 71.521C115.316 71.3344 115.409 71.2644 116.039 71.1711C116.311 71.1424 116.601 71.105 116.822 71.0803C117.03 71.0571 117.415 70.9152 117.415 71.1245ZM111.467 72.4074C111.397 72.524 111.117 72.6407 110.86 72.6407C110.627 72.6407 110.417 72.524 110.417 72.4074C110.417 72.2675 110.697 72.1741 111.023 72.1741C111.35 72.1741 111.56 72.2675 111.467 72.4074ZM117.998 80.8284C118.255 81.2249 117.975 81.3182 117.648 80.9217C117.462 80.7117 117.462 80.5718 117.602 80.5718C117.742 80.5718 117.928 80.6884 117.998 80.8284ZM117.998 84.7706C118.068 84.9106 117.975 84.9339 117.788 84.8639C117.368 84.7006 117.298 84.5373 117.625 84.5373C117.765 84.5373 117.928 84.6306 117.998 84.7706ZM90.4725 96.434C90.5425 96.5506 90.4492 96.6672 90.2393 96.6672C90.0293 96.6672 89.936 96.5506 90.006 96.434C90.076 96.294 90.1926 96.2007 90.2393 96.2007C90.2859 96.2007 90.4026 96.294 90.4725 96.434ZM93.1085 134.317C93.3417 134.853 93.3184 134.876 92.5486 134.853C91.8721 134.806 91.8022 134.76 92.1287 134.527C92.502 134.247 92.502 134.223 92.1521 134.083C91.9188 134.013 91.7555 133.733 91.7555 133.5C91.7555 133.08 91.8022 133.08 92.292 133.407C92.5953 133.593 92.9685 134.013 93.1085 134.317Z" />
        <path d="M129.428 13.4371C128.798 13.5537 127.842 13.6703 127.329 13.6937C126.816 13.7403 125.976 13.9036 125.486 14.0669C124.763 14.3235 124.646 14.4168 124.949 14.6034C125.509 14.9067 125.159 15.4898 124.39 15.4898C124.016 15.4898 123.713 15.3732 123.713 15.2566C123.713 14.9533 122.873 14.9533 122.29 15.2566C121.707 15.5831 121.964 16.073 123.037 16.6095C123.62 16.8894 123.83 17.146 123.806 17.5426C123.783 18.1724 123.83 18.1724 126.699 17.5659C127.889 17.3326 128.332 17.0994 129.195 16.2596C129.778 15.7231 130.245 15.0466 130.245 14.79C130.245 14.5334 130.408 14.3235 130.595 14.3235C130.968 14.3235 130.991 14.1835 130.758 13.6237C130.618 13.3204 130.385 13.2971 129.428 13.4371ZM129.778 15.0466C130.081 15.2566 130.081 15.2799 129.731 15.1866C129.521 15.1166 129.312 15.1633 129.242 15.3265C129.195 15.4665 129.102 15.4199 129.008 15.1866C128.845 14.7434 129.242 14.6501 129.778 15.0466Z" />
        <path d="M26.0441 40.7762C25.3909 41.0561 24.7611 41.3593 24.6678 41.4526C24.5745 41.5459 24.2713 41.6159 23.9913 41.6159C23.6881 41.6159 23.3382 41.7792 23.1749 41.9658C23.0116 42.1524 22.8017 42.2691 22.7084 42.1991C22.4051 42.0125 21.0755 42.5723 20.8422 42.9922C20.5157 43.622 20.9122 44.0419 21.682 43.832C22.0552 43.7153 22.5218 43.622 22.7084 43.5754C22.895 43.5287 23.2449 43.4121 23.5248 43.2721C23.7814 43.1322 24.3879 42.8756 24.8778 42.689C25.6242 42.3857 25.8575 42.0825 25.7875 41.3827C25.7875 41.3127 26.1374 41.2894 26.5806 41.336C27.3037 41.4293 27.3737 41.3827 27.3737 40.8228C27.3737 40.4432 27.1277 40.3142 26.7791 40.4644C26.5703 40.5544 26.3167 40.6634 26.0441 40.7762Z" />
        <path d="M100.386 110.687C100.06 110.78 99.3134 110.99 98.7069 111.153C98.1004 111.293 97.4239 111.62 97.214 111.853C97.004 112.11 96.6541 112.296 96.4442 112.296C96.1176 112.296 96.1176 112.343 96.4209 112.53C96.7241 112.716 96.6775 112.786 96.2576 112.903C95.5578 113.089 95.5811 113.369 96.2809 113.509C96.6075 113.579 97.074 113.439 97.3773 113.206C97.6805 112.973 98.1004 112.763 98.3103 112.763C98.5203 112.763 98.9168 112.553 99.2201 112.296C99.5933 111.97 100.06 111.83 100.713 111.853C101.249 111.876 101.483 111.923 101.273 111.993C101.04 112.04 100.853 112.18 100.853 112.296C100.853 112.506 102.299 112.996 102.952 112.996C103.139 112.996 103.839 113.253 104.469 113.579C105.168 113.906 105.845 114.092 106.171 113.999C106.591 113.906 106.685 113.976 106.685 114.466C106.685 114.769 106.848 115.165 107.035 115.329C107.221 115.492 107.384 115.749 107.384 115.912C107.384 116.075 107.548 116.332 107.734 116.495C107.944 116.658 108.061 116.938 107.991 117.102C107.828 117.498 112.75 117.382 113.799 116.985C114.106 116.858 114.168 116.45 113.912 116.238L113.799 116.145C113.403 115.842 112.82 115.585 112.493 115.562C112.073 115.562 111.886 115.445 111.91 115.189C111.933 114.932 111.607 114.699 110.93 114.512C109.95 114.209 108.551 113.043 108.551 112.483C108.551 112.303 108.314 112.426 108.152 112.349C108.078 112.313 107.997 112.256 107.921 112.18C107.524 111.76 106.825 111.713 107.058 112.11C107.174 112.296 107.105 112.32 106.825 112.226C106.591 112.133 106.265 112.063 106.078 112.063C105.892 112.063 105.775 111.946 105.822 111.783C105.961 111.083 101.809 110.244 100.386 110.687ZM108.901 116.029C108.994 116.169 108.737 116.262 108.341 116.262C107.944 116.262 107.618 116.145 107.618 116.029C107.618 115.889 107.874 115.795 108.178 115.795C108.504 115.795 108.831 115.889 108.901 116.029Z" />
        <path d="M115.829 117.195C115.059 117.568 115.036 117.894 115.829 117.894C116.132 117.894 116.249 118.081 116.249 118.548C116.249 118.898 116.365 119.317 116.529 119.481C116.972 119.924 116.109 120.087 114.686 119.831C113.566 119.621 113.403 119.644 113.216 120.017C113.029 120.344 113.123 120.507 113.729 120.787C114.522 121.184 114.942 121.277 114.756 120.974C114.709 120.88 115.269 120.787 116.039 120.81C117.322 120.81 117.508 120.88 117.928 121.463C118.157 121.784 118.633 121.784 118.861 121.463L118.978 121.3C119.538 120.46 119.864 120.274 120.028 120.694C120.074 120.834 120.354 120.787 120.751 120.577C121.427 120.227 122.85 120.297 122.967 120.694C123.013 120.81 123.223 120.927 123.433 120.927C123.66 120.927 123.578 120.634 123.732 120.466C123.777 120.417 123.834 120.367 123.9 120.32C124.273 120.041 124.226 119.947 123.433 119.177C122.967 118.734 122.36 118.361 122.104 118.361C121.824 118.361 121.614 118.221 121.614 118.034C121.614 117.568 119.701 117.101 118.441 117.311C117.672 117.428 117.275 117.381 116.995 117.125C116.645 116.845 116.482 116.868 115.829 117.195Z" />
      </Continent>
      <Continent
        className={clsx('southAmerica', selectedTimezone === timezones.southAmerica && 'selected')}
        fill={getMapBaseColor(46)}
        onClick={(event) => onClickContinent(event, timezones.southAmerica)}
      >
        <path d="M116.622 132.116C116.226 132.256 115.969 132.513 115.969 132.769C115.969 133.003 115.829 133.189 115.643 133.189C115.479 133.189 115.059 133.399 114.733 133.656C114.383 133.936 113.8 134.122 113.31 134.122C112.773 134.122 112.4 134.239 112.33 134.472C112.237 134.705 112.004 134.775 111.677 134.682C110.8 134.468 109.648 136.06 109.625 137.333C109.621 137.557 109.638 137.855 109.414 137.855C109.111 137.855 108.505 138.438 108.505 138.718C108.505 138.834 108.201 139.114 107.805 139.324C107.315 139.627 107.175 139.861 107.268 140.141C107.362 140.35 106.965 140.047 106.382 139.441C105.495 138.508 104.026 137.621 103.349 137.621C103.233 137.621 103.139 137.925 103.139 138.274C103.139 138.741 103.303 139.044 103.723 139.254C104.026 139.417 104.352 139.767 104.446 140.001C104.516 140.257 104.679 140.35 104.842 140.257C105.029 140.164 105.006 140.234 104.819 140.467C104.469 140.91 104.656 141.89 105.169 142.333C105.355 142.497 105.635 142.893 105.775 143.22C106.102 143.989 106.219 148.048 105.939 148.235C105.775 148.328 105.799 148.538 105.962 148.841C106.219 149.355 105.845 150.918 105.449 150.918C105.332 150.918 105.239 151.057 105.239 151.221C105.239 151.384 104.866 151.664 104.422 151.851C103.839 152.084 103.606 152.317 103.606 152.69C103.606 152.97 103.443 153.25 103.256 153.343C102.999 153.437 102.929 153.647 103.023 154.02C103.163 154.51 103.046 154.626 102.113 154.976C101.087 155.373 101.04 155.443 101.04 156.283C101.04 156.843 100.923 157.216 100.69 157.309C100.503 157.379 100.34 157.706 100.34 158.009C100.34 158.359 100.154 158.709 99.8737 158.849C99.6171 158.989 99.4538 159.199 99.5238 159.338C99.827 159.805 99.8503 161.415 99.5704 161.415C99.4071 161.415 99.6404 161.718 100.06 162.091C100.457 162.464 100.807 162.884 100.807 163.024C100.807 163.164 100.923 163.281 101.04 163.281C101.18 163.281 101.273 163.164 101.273 163.001C101.273 162.861 101.32 162.791 101.39 162.861C101.577 163.047 101.25 163.747 100.97 163.747C100.83 163.747 100.317 164.19 99.8037 164.727C98.9872 165.637 98.9173 165.823 99.0572 166.663C99.1272 167.176 99.3371 167.759 99.5238 167.969C99.9203 168.389 99.9903 168.879 99.6404 168.879C99.0339 168.879 99.5238 169.602 100.434 170.069C101.203 170.442 101.623 170.885 102.113 171.772C102.463 172.425 102.976 173.265 103.256 173.661C103.536 174.034 104.096 175.154 104.539 176.11C104.959 177.067 105.495 178.14 105.752 178.49C105.985 178.816 106.172 179.259 106.172 179.423C106.172 179.609 106.382 179.959 106.638 180.193C106.895 180.426 107.105 180.799 107.105 181.032C107.105 181.266 107.198 181.499 107.315 181.569C107.805 181.872 109.228 184.321 109.111 184.648C108.901 185.208 109.391 186.118 110.534 187.261C111.63 188.38 114.803 190.386 116.156 190.83C116.552 190.97 116.902 191.156 116.902 191.249C116.902 191.343 117.275 191.623 117.719 191.879C118.162 192.113 118.535 192.392 118.535 192.509C118.535 192.742 119.142 193.372 119.351 193.372C119.771 193.372 120.541 194.445 120.798 195.425C121.474 197.944 121.987 203.403 121.521 203.403C121.428 203.403 121.334 203.659 121.334 203.986C121.334 204.312 121.404 204.569 121.521 204.569C121.847 204.569 122.127 214.11 121.801 214.926C121.661 215.299 121.707 215.673 121.964 216.139C122.197 216.629 122.267 217.165 122.151 218.005C122.034 218.915 122.127 219.428 122.501 220.291C122.757 220.898 122.967 221.598 122.967 221.854C122.967 222.087 123.084 222.297 123.2 222.297C123.34 222.297 123.457 222.741 123.434 223.277C123.434 223.837 123.434 224.49 123.457 224.747C123.48 225.236 123.294 227.942 123.084 229.879C123.037 230.508 122.967 231.301 122.967 231.605C122.967 232.001 122.874 232.118 122.594 232.001C122.291 231.885 122.244 231.978 122.384 232.491C122.454 232.864 122.687 233.308 122.874 233.517C123.06 233.704 123.2 234.031 123.2 234.194C123.2 234.357 123.48 234.964 123.807 235.547C124.227 236.27 124.367 236.76 124.273 237.226C124.063 238.299 124.39 240.189 124.88 240.725C125.113 240.982 125.207 241.192 125.043 241.192C124.88 241.192 124.833 241.425 124.927 241.892C124.997 242.265 125.137 242.941 125.207 243.338C125.463 244.714 126.21 245.134 126.699 244.201C126.909 243.851 126.886 243.641 126.676 243.385C126.536 243.198 126.489 243.058 126.583 243.058C126.676 243.058 126.629 242.895 126.466 242.708C126.303 242.522 126.256 242.358 126.373 242.358C126.466 242.358 126.443 242.242 126.28 242.078C125.883 241.682 125.93 241.355 126.373 241.519C126.653 241.635 126.723 241.565 126.606 241.309C126.536 241.099 126.559 241.005 126.699 241.075C126.839 241.145 126.933 241.379 126.933 241.565C126.933 241.752 127.049 241.892 127.189 241.892C127.306 241.892 127.376 241.985 127.306 242.078C127.236 242.195 127.283 242.638 127.399 243.081C127.516 243.525 127.656 244.388 127.749 245.018C127.819 245.671 128.006 246.254 128.146 246.347C128.356 246.487 128.309 246.58 127.982 246.697C127.969 246.702 127.955 246.707 127.941 246.712C127.564 246.852 127.253 246.769 127.306 246.371C127.376 245.741 127.096 245.391 126.513 245.391C126.326 245.391 126.28 245.507 126.396 245.671C126.699 246.184 126.676 246.557 126.326 246.557C126.163 246.557 125.906 246.767 125.766 247C125.486 247.56 125.463 247.98 125.743 247.514C126.046 247.024 126.933 247.607 126.933 248.307C126.933 248.633 126.816 248.89 126.676 248.89C126.513 248.89 126.466 249.146 126.536 249.59C126.629 250.033 126.559 250.383 126.35 250.616C126.163 250.803 126.07 251.129 126.14 251.316C126.303 251.759 126.863 251.806 126.979 251.432C127.049 251.222 127.213 251.222 127.586 251.432C127.866 251.596 128.216 251.642 128.332 251.572C128.472 251.502 128.496 251.596 128.426 251.782C128.356 251.992 128.379 252.156 128.519 252.156C128.939 252.156 128.449 253.415 127.912 253.695C127.632 253.835 127.469 254.022 127.586 254.115C127.679 254.208 127.749 254.372 127.749 254.442C127.796 254.908 128.169 255.655 128.356 255.655C128.472 255.655 128.542 255.725 128.519 255.818C128.472 256.144 128.962 257.171 129.335 257.497C129.545 257.661 129.779 258.104 129.849 258.477C129.895 258.662 129.958 258.835 130.024 258.963C130.117 259.141 130.308 259.193 130.507 259.218C130.65 259.235 130.815 259.263 130.893 259.384C131.001 259.554 131.108 259.863 131.155 260.203C131.225 260.716 131.318 261.02 131.341 260.903C131.365 260.763 131.598 261.02 131.878 261.44C132.251 262.046 132.484 262.186 132.881 262.093C133.208 261.999 133.581 262.139 133.954 262.489C134.099 262.623 134.249 262.741 134.377 262.822C134.561 262.94 134.864 262.924 134.864 263.142C134.864 263.399 134.561 263.352 133.581 263.002C133.278 262.886 133.371 263.072 133.907 263.562C134.467 264.099 134.747 264.239 134.864 264.029C135.12 263.656 135.587 263.982 135.4 264.425C135.33 264.612 135.354 264.752 135.47 264.729C135.867 264.635 136.497 265.032 136.497 265.358C136.497 265.568 136.753 265.685 137.196 265.685C137.733 265.685 137.85 265.592 137.733 265.312C137.663 265.102 137.43 264.985 137.173 265.055C136.917 265.125 136.73 265.079 136.73 264.962C136.73 264.519 137.663 264.775 137.92 265.265C138.083 265.568 138.176 265.848 138.153 265.918C138.13 265.988 138.269 266.128 138.503 266.268C138.736 266.385 139.063 266.641 139.249 266.805C139.646 267.178 139.996 267.178 139.996 266.781C139.996 266.595 140.066 266.618 140.229 266.851C140.424 267.163 140.809 267.258 141.153 267.385C141.256 267.422 141.362 267.463 141.465 267.505C141.838 267.668 142.212 267.738 142.328 267.668C142.445 267.598 142.585 267.645 142.678 267.784C142.772 267.924 142.935 267.971 143.051 267.901C143.168 267.808 143.261 267.854 143.261 267.994C143.261 268.134 143.728 268.228 144.311 268.204C144.894 268.204 145.524 268.321 145.711 268.484C145.921 268.648 146.154 268.718 146.247 268.624C146.341 268.531 146.201 268.298 145.967 268.134C145.594 267.831 145.687 267.808 146.877 267.901C147.973 267.971 148.183 267.948 148.067 267.668C147.997 267.458 148.09 267.318 148.3 267.318C148.487 267.318 148.697 267.155 148.767 266.968C148.86 266.781 149.023 266.618 149.14 266.618C149.28 266.618 149.256 266.758 149.093 266.968C148.86 267.248 148.86 267.318 149.186 267.318C149.396 267.318 149.56 267.131 149.56 266.851C149.56 266.478 149.396 266.385 148.743 266.385C147.53 266.385 145.011 265.055 143.635 263.702C142.958 263.026 142.468 262.373 142.515 262.233C142.562 262.093 142.305 261.813 141.978 261.58C141.209 261.066 139.996 259.34 139.996 258.71C139.996 258.384 140.182 258.174 140.579 258.081C141.279 257.917 141.489 257.334 141.162 256.471C140.882 255.748 140.999 255.538 142.258 254.418C142.702 254.022 142.958 253.625 142.865 253.462C142.772 253.322 142.655 252.995 142.632 252.762C142.515 252.039 142.095 251.689 141.372 251.689C140.625 251.689 139.156 250.896 138.853 250.313C138.316 249.356 139.226 247.957 140.392 247.957C140.975 247.957 141.302 247.607 140.905 247.35C140.719 247.234 140.742 247.14 140.952 247.024C141.185 246.86 141.162 246.627 140.859 245.857L140.819 245.751C140.617 245.208 140.762 244.597 141.185 244.201C141.582 243.828 142.048 243.525 142.235 243.525C142.608 243.525 142.655 242.988 142.328 242.358C142.118 241.985 141.908 241.915 141.069 242.032C140.46 242.115 139.869 241.788 139.615 241.228L139.482 240.935C139.449 240.865 139.418 240.795 139.388 240.726C139.075 240.001 139.521 239.587 140.252 239.886C141.349 240.329 142.095 240.352 142.912 239.979C143.518 239.722 143.705 239.023 143.238 238.719C143.075 238.626 143.051 238.253 143.145 237.763C143.261 237.156 143.191 236.83 142.888 236.503C142.492 236.06 142.492 236.037 142.912 236.2C143.215 236.317 143.285 236.293 143.168 236.107C143.051 235.897 143.308 235.827 144.078 235.827C146.574 235.827 149.793 234.66 149.793 233.774C149.793 233.541 149.886 233.261 150.003 233.144C150.819 232.235 150.703 230.228 149.816 230.228C149.35 230.228 149.093 229.809 149.093 228.969C149.093 228.362 148.533 227.802 147.414 227.243C146.994 227.009 146.597 226.636 146.55 226.38C146.48 225.936 146.48 225.936 146.924 226.31C147.18 226.543 147.717 226.729 148.113 226.729C148.533 226.729 148.86 226.823 148.86 226.939C148.86 227.243 149.956 227.499 151.542 227.546C153.479 227.639 154.342 227.126 154.598 225.82C154.692 225.423 154.808 225.097 154.901 225.097C155.181 225.097 155.788 223.837 155.951 222.974C156.044 222.437 156.394 221.901 156.908 221.434C157.491 220.921 157.677 220.594 157.561 220.291C157.421 219.941 157.397 219.941 157.234 220.408C157.195 220.506 157.136 220.602 157.065 220.691C156.641 221.224 156.256 220.736 156.674 220.198C157.001 219.755 157.257 219.172 157.257 218.915C157.257 218.612 157.397 218.402 157.631 218.402C157.887 218.402 157.957 218.565 157.887 218.938C157.841 219.242 157.934 219.521 158.097 219.591C158.307 219.661 158.494 219.381 158.68 218.728C159.287 216.536 159.427 216.209 160.08 215.509C160.99 214.553 161.293 213.013 160.873 211.777C160.57 210.937 160.477 208.301 160.733 208.371C160.92 208.418 162.856 206.435 162.856 206.202C162.856 206.085 163.182 205.945 163.556 205.852C163.952 205.759 164.255 205.595 164.255 205.455C164.255 205.315 164.559 205.269 165.072 205.362C165.585 205.455 165.888 205.432 165.888 205.269C165.888 205.152 165.772 205.036 165.632 205.036C165.515 205.036 165.445 204.942 165.515 204.826C165.772 204.429 168.688 203.613 169.924 203.613C171.114 203.589 171.207 203.543 171.323 202.959C171.417 202.61 171.767 202.19 172.21 201.933C172.793 201.607 172.91 201.42 172.793 201C172.676 200.533 173.096 199.437 174.426 196.848C174.636 196.451 174.683 195.845 174.613 195.075C174.496 194.165 174.543 193.839 174.892 193.512C175.289 193.116 175.569 191.203 175.499 189.29C175.499 188.893 175.452 187.54 175.406 186.258C175.312 184.041 175.476 183.342 176.059 183.342C176.315 183.342 176.829 182.619 177.342 181.476C178.158 179.633 178.345 179.329 178.905 178.816C179.628 178.14 181.471 175.574 181.751 174.851C182.287 173.451 182.264 171.772 181.681 169.719C181.354 168.599 181.027 167.619 180.957 167.549C180.864 167.456 180.188 167.316 179.441 167.223C177.948 167.06 177.925 167.036 176.035 165.217C175.336 164.564 174.193 163.677 173.47 163.281C172.303 162.604 172.093 162.558 171.16 162.767C170.414 162.931 169.994 162.884 169.527 162.651C169.512 162.643 169.498 162.635 169.483 162.628C168.975 162.373 168.338 162.577 167.871 162.254C167.171 161.788 165.795 161.531 166.005 161.904C166.122 162.091 166.005 162.114 165.632 161.998C164.955 161.764 164.535 161.368 164.652 161.041C164.699 160.878 164.582 160.598 164.372 160.411C164.185 160.202 164.022 159.945 164.022 159.805C164.022 159.665 163.929 159.548 163.789 159.548C163.672 159.548 163.556 159.642 163.556 159.782C163.556 160.038 161.899 159.338 161.643 158.965C161.55 158.825 161.456 158.825 161.456 158.942C161.456 159.059 161.316 159.012 161.13 158.872C160.803 158.615 160.22 158.429 159.194 158.265C158.564 158.172 157.747 158.732 157.467 159.478C157.094 160.458 157.024 159.968 157.374 158.965C157.841 157.612 157.817 157.519 156.861 157.379C156.324 157.286 155.858 157.029 155.671 156.749C155.508 156.493 155.251 156.283 155.065 156.283C154.738 156.283 154.738 155.956 155.065 154.603C155.158 154.137 155.065 153.787 154.668 153.274C154.365 152.9 153.992 152.574 153.875 152.574C153.595 152.55 153.059 150.754 153.059 149.798C153.059 149.261 152.149 147.675 152.126 148.165C152.126 148.258 151.542 147.768 150.796 147.069C150.073 146.369 149.373 145.786 149.21 145.786C149.046 145.786 148.603 145.622 148.183 145.412C147.344 144.993 144.638 144.759 144.428 145.086C144.171 145.506 142.445 144.829 141.442 143.896C140.882 143.406 140.322 142.986 140.206 142.986C140.089 142.986 139.996 142.753 139.996 142.45C139.972 142.17 139.786 141.703 139.552 141.424C138.899 140.63 136.987 139.534 136.52 139.651C136.007 139.791 135.657 139.347 135.96 138.951C136.31 138.484 136.17 137.994 135.634 137.738C135.33 137.621 135.097 137.435 135.097 137.341C135.097 137.225 134.91 137.155 134.677 137.155C134.444 137.155 134.117 137.015 133.977 136.875C133.744 136.642 133.627 136.642 133.464 136.921C133.324 137.131 133.231 137.155 133.231 136.991C133.231 136.851 133.161 136.572 133.091 136.362C132.998 136.128 133.068 135.988 133.278 135.988C133.464 135.988 133.814 135.825 134.071 135.638C134.467 135.289 134.327 135.289 132.298 135.382C131.085 135.452 130.012 135.522 129.919 135.522C129.802 135.522 129.732 135.662 129.732 135.825C129.732 136.012 129.475 136.268 129.172 136.432C128.472 136.805 126.933 136.385 126.933 135.825C126.933 135.522 126.653 135.499 124.95 135.638L123.572 135.752C123.246 135.779 122.967 135.522 122.967 135.195C122.967 134.519 121.847 133.656 120.938 133.656C120.541 133.656 120.401 133.516 120.401 133.119C120.401 132.443 120.238 132.256 119.795 132.419C119.608 132.489 119.468 132.863 119.468 133.329C119.468 133.982 119.375 134.146 118.838 134.262C118.465 134.356 117.882 134.635 117.509 134.892C116.926 135.312 116.832 135.312 116.972 134.985C117.042 134.775 117.019 134.589 116.902 134.589C116.762 134.589 116.669 134.379 116.669 134.122C116.669 133.842 116.832 133.656 117.042 133.656C117.462 133.656 118.045 133.026 118.069 132.559C118.069 132.093 117.299 131.836 116.622 132.116ZM117.602 137.621C117.602 138.181 117.229 138.788 116.902 138.788C116.809 138.788 116.599 138.391 116.436 137.878C116.179 137.085 116.202 136.921 116.576 136.478L116.594 136.456C116.789 136.229 117.151 136.264 117.299 136.525C117.462 136.805 117.579 137.295 117.602 137.621ZM121.964 188.777C122.291 189.15 122.407 189.43 122.267 189.64C122.104 189.92 122.034 189.896 121.894 189.57C121.824 189.36 121.614 189.173 121.451 189.173C121.288 189.173 121.101 188.963 121.031 188.707C120.868 188.077 121.334 188.124 121.964 188.777ZM167.055 203.869C167.055 203.986 167.008 204.102 166.961 204.102C166.891 204.102 166.775 203.986 166.705 203.869C166.635 203.729 166.681 203.636 166.798 203.636C166.938 203.636 167.055 203.729 167.055 203.869ZM158.354 219.078C158.284 219.265 158.214 219.195 158.214 218.938C158.189 218.664 158.453 218.769 158.371 219.032C158.366 219.048 158.36 219.063 158.354 219.078ZM141.512 243.291C141.582 243.408 141.442 243.525 141.185 243.525C140.905 243.525 140.695 243.408 140.695 243.291C140.695 243.151 140.835 243.058 141.022 243.058C141.209 243.058 141.442 243.151 141.512 243.291ZM128.892 247.14C128.799 247.327 128.659 247.42 128.542 247.35C128.286 247.21 128.496 246.79 128.799 246.79C128.915 246.79 128.962 246.954 128.892 247.14ZM128.449 247.98C128.682 248.68 128.449 248.75 128.146 248.073C128.006 247.747 127.982 247.49 128.099 247.49C128.216 247.49 128.379 247.7 128.449 247.98ZM129.499 248.75C129.499 248.797 129.335 248.89 129.149 248.983C128.962 249.053 128.799 249.006 128.799 248.89C128.799 248.75 128.962 248.657 129.149 248.657C129.335 248.657 129.499 248.703 129.499 248.75ZM139.996 263.586C139.996 264.146 140.392 264.379 141.279 264.332C141.955 264.285 141.955 264.309 141.442 264.589C141.377 264.627 141.318 264.668 141.267 264.712C141.022 264.923 141.053 265.108 140.835 264.869C140.821 264.854 140.808 264.839 140.795 264.825C140.479 264.497 140.069 264.805 139.856 264.402C139.482 263.656 139.412 262.793 139.762 263.002C139.902 263.072 139.996 263.352 139.996 263.586ZM138.479 263.562C138.409 263.702 138.153 263.819 137.943 263.819C137.64 263.819 137.64 263.772 138.036 263.492C138.526 263.142 138.736 263.166 138.479 263.562Z" />
      </Continent>
      <Continent
        className={clsx('europe', selectedTimezone === timezones.europe && 'selected')}
        fill={getMapBaseColor(38)}
        onClick={(event) => onClickContinent(event, timezones.europe)}
      >
        <path d="M312.997 2.20994C312.903 2.32657 312.413 2.48986 311.854 2.58317C310.874 2.74646 308.891 3.58622 308.634 3.95945C308.541 4.07608 308.355 4.07608 308.215 4.0061C308.075 3.91279 307.958 3.95945 307.958 4.07608C307.958 4.21604 308.098 4.356 308.261 4.356C308.401 4.37933 308.798 4.42598 309.078 4.44931C309.404 4.47263 309.847 4.28602 310.104 4.0061C310.384 3.74951 310.524 3.63287 310.454 3.77283C310.361 3.93612 310.524 4.09941 310.827 4.19271C311.55 4.37933 311.737 4.356 311.574 4.09941C311.504 3.95945 311.597 3.93612 311.784 4.0061C311.994 4.07608 312.087 4.26269 311.994 4.40265C311.9 4.56594 312.46 4.58927 313.813 4.49596C314.979 4.42598 315.586 4.44931 315.306 4.56594C314.933 4.72923 315.049 4.77588 315.796 4.77588C316.472 4.79921 316.822 4.68258 316.915 4.44931C316.985 4.26269 317.242 4.09941 317.499 4.09941C318.105 4.09941 318.688 3.86614 318.688 3.60955C318.688 3.49291 318.478 3.3996 318.245 3.3996C317.989 3.3996 317.755 3.3063 317.709 3.16634C317.639 2.97972 316.122 2.74646 314.629 2.6998C314.28 2.6998 314.256 2.62982 314.489 2.3499C314.746 2.04665 314.653 2 313.953 2C313.486 2 313.067 2.09331 312.997 2.20994ZM315.772 3.3996C315.539 3.6562 315.073 3.86614 314.746 3.84281L314.14 3.81949L314.723 3.63287C315.049 3.53957 315.143 3.42293 314.956 3.3996C314.769 3.37628 314.863 3.25964 315.189 3.16634C316.122 2.86309 316.216 2.90974 315.772 3.3996ZM309.824 3.72618C309.824 3.79616 309.731 3.91279 309.591 3.98277C309.474 4.05275 309.358 4.0061 309.358 3.88947C309.358 3.74951 309.474 3.63287 309.591 3.63287C309.731 3.63287 309.824 3.67953 309.824 3.72618Z" />
        <path d="M304.972 3.25975C304.552 3.39972 304.272 3.60966 304.366 3.70296C304.459 3.79627 304.412 3.86625 304.272 3.86625C304.109 3.86625 303.992 3.74962 303.992 3.63298C303.992 3.35306 302.313 3.35306 300.61 3.63298C299.444 3.79627 299.77 3.93623 301.357 3.93623C301.776 3.93623 302.056 4.02954 301.986 4.14617C301.916 4.23948 301.893 4.33279 301.94 4.35611C303.176 4.56605 304.716 4.56605 304.599 4.35611C304.482 4.19282 304.646 4.09952 304.972 4.09952C305.345 4.09952 305.415 4.1695 305.205 4.30946C304.832 4.5194 305.905 4.79932 307.118 4.79932C307.678 4.79932 307.771 4.72934 307.491 4.56605C307.305 4.42609 306.838 4.40277 306.442 4.47275C305.812 4.61271 305.789 4.58938 306.069 4.23948C306.232 4.02954 306.582 3.86625 306.838 3.86625C307.072 3.86625 307.212 3.74962 307.142 3.63298C307.072 3.49302 306.792 3.39972 306.535 3.39972C306.302 3.39972 306.092 3.28308 306.092 3.16645C306.092 2.88653 305.952 2.88653 304.972 3.25975Z" />
        <path d="M270.169 4.28605C269.959 4.51932 269.585 4.56597 268.956 4.47267C268.256 4.35603 268.023 4.40269 267.906 4.70594C267.836 4.91588 267.906 5.17247 268.092 5.2891C268.279 5.40574 268.302 5.49904 268.162 5.49904C268.023 5.49904 267.766 5.33576 267.579 5.12582C267.416 4.91588 267.066 4.79924 266.74 4.86922C266.436 4.91588 266.156 4.89255 266.086 4.79924C266.04 4.70594 265.387 4.68261 264.64 4.70594C263.894 4.75259 263.007 4.77592 262.657 4.79924C262.144 4.79924 262.051 4.89255 262.167 5.1958C262.261 5.42906 262.587 5.87227 262.914 6.17552L263.52 6.73536L262.891 6.36213C262.517 6.12887 262.214 6.05889 262.121 6.19885C262.051 6.31548 262.424 6.61873 262.984 6.89865C264.15 7.45849 264.244 7.48182 263.87 7.01528C263.311 6.33881 265.06 7.43516 265.69 8.11164C265.993 8.48487 266.413 8.76479 266.623 8.76479C266.856 8.78812 267.439 8.97473 267.953 9.23132C269.329 9.9078 269.865 9.83782 270.052 8.99806C270.145 8.6015 270.355 8.29825 270.518 8.29825C270.868 8.29825 272.221 7.13192 271.895 7.13192C271.778 7.13192 271.871 7.01528 272.105 6.87532C272.571 6.5954 273.318 6.78202 274.087 7.38851C274.577 7.76174 274.601 7.80839 274.204 8.04166C273.738 8.32158 273.901 8.36823 274.974 8.29825C275.394 8.27493 275.79 8.36823 275.884 8.50819C276.047 8.78812 277.12 8.62483 277.516 8.27493C277.633 8.15829 277.913 8.06499 278.17 8.06499C278.403 8.06499 278.52 7.94835 278.45 7.83172C278.38 7.69176 278.006 7.59845 277.633 7.59845C277.26 7.59845 276.933 7.43516 276.84 7.22522C276.747 6.96863 276.513 6.87532 276.117 6.9453C275.79 7.01528 275.534 6.96863 275.534 6.87532C275.534 6.54875 273.411 5.68566 273.014 5.84895C272.711 5.96558 272.688 5.91893 272.898 5.70899C273.084 5.52237 273.621 5.49904 274.647 5.59235C277.143 5.87227 279.289 5.73231 279.453 5.26578C279.499 5.12582 279.733 5.03251 279.966 5.03251C280.572 5.03251 280.759 4.63596 280.269 4.37936C279.593 4.02946 276.747 3.8895 276.583 4.19275C276.513 4.37936 276.42 4.35603 276.327 4.14609C276.117 3.72621 275.067 3.79619 275.067 4.2394C275.067 4.54265 274.997 4.56598 274.717 4.33271C274.227 3.91283 270.565 3.8895 270.169 4.28605ZM272.221 5.45239C272.291 5.66233 272.175 5.75564 271.895 5.68566C271.685 5.66233 271.452 5.47572 271.382 5.31243C271.312 5.10249 271.428 5.00918 271.708 5.07916C271.918 5.10249 272.151 5.2891 272.221 5.45239ZM267.603 7.01528C267.603 7.08526 267.416 7.13192 267.183 7.13192C266.949 7.13192 266.74 6.96863 266.74 6.80534C266.74 6.54875 266.856 6.50209 267.183 6.68871C267.416 6.80534 267.603 6.96863 267.603 7.01528ZM265.97 7.59845C265.97 7.71509 265.876 7.83172 265.736 7.83172C265.62 7.83172 265.503 7.71509 265.503 7.59845C265.503 7.45849 265.62 7.36518 265.736 7.36518C265.876 7.36518 265.97 7.45849 265.97 7.59845Z" />
        <path d="M327.809 8.92841C327.482 8.99839 326.993 9.20833 326.689 9.39495C326.316 9.65154 325.173 9.81483 323.214 9.95479C320.414 10.1414 319.411 10.3047 319.155 10.6079C318.898 10.9345 317.149 11.6343 316.752 11.5877C316.449 11.541 316.332 11.6576 316.379 11.9609C316.496 12.7073 316.122 13.3372 315.353 13.7104L314.606 14.0836L315.189 14.3169C315.539 14.4802 315.632 14.5735 315.399 14.5735C315.189 14.5968 314.723 14.8534 314.349 15.18C313.813 15.6232 313.766 15.7631 314.093 15.7631C314.676 15.7631 314.583 16.2064 313.906 16.4629C313.58 16.5796 313.323 16.8362 313.323 17.0228C313.323 17.5127 313.953 17.9325 314.489 17.7926C314.746 17.7226 314.909 17.7692 314.816 17.8859C314.746 18.0025 314.979 18.2125 315.329 18.3291C315.819 18.5157 316.006 18.5157 316.169 18.2824C316.309 18.0492 316.356 18.0725 316.356 18.3524C316.356 18.8656 316.845 19.0289 318.688 19.1922C321.487 19.4021 321.791 19.4021 321.907 19.0522C321.977 18.8189 321.907 18.7723 321.627 18.8656C321.418 18.9589 321.278 18.9356 321.348 18.8189C321.417 18.7256 321.044 18.4924 320.554 18.3291C320.041 18.1658 319.621 17.9559 319.621 17.8859C319.621 17.7926 319.341 17.5593 318.968 17.3494C318.338 16.9761 317.942 15.7631 318.455 15.7631C318.595 15.7631 318.688 15.6465 318.688 15.5299C318.688 15.2033 321.138 12.7307 321.464 12.7307C321.604 12.7307 321.907 12.5441 322.141 12.3108C322.677 11.7743 323.68 11.471 326.736 10.8645C330.795 10.0481 331.611 9.53491 329.698 8.99839C328.812 8.76512 328.742 8.76512 327.809 8.92841ZM315.819 14.037C315.749 14.1069 315.539 14.1303 315.376 14.0603C315.189 13.9903 315.259 13.9203 315.516 13.9203C315.772 13.897 315.912 13.967 315.819 14.037Z" />
        <path d="M281.016 18.3512C280.922 18.4912 280.689 18.5379 280.479 18.4446C280.292 18.3746 279.873 18.5379 279.546 18.8411C279.219 19.1444 279.033 19.2377 279.126 19.0744C279.243 18.8878 279.079 18.7945 278.52 18.7945C277.517 18.7945 276.677 19.0277 276.817 19.261C276.957 19.4943 275.814 19.8208 275.091 19.7509C274.694 19.7042 274.601 19.7742 274.717 20.0074C274.857 20.2407 274.834 20.264 274.577 20.0074C274.018 19.4943 272.501 19.7275 272.501 20.334C272.501 20.6839 272.478 20.6839 271.638 20.7772C270.682 20.8705 270.355 21.1038 270.472 21.5937C270.542 21.8503 270.472 22.1302 270.355 22.2235C270.192 22.3168 270.169 22.2468 270.285 22.0602C270.425 21.8269 270.332 21.8036 269.819 21.9202C269.236 22.0602 269.189 22.0369 269.446 21.7336C269.889 21.2204 269.446 21.2438 268.209 21.8503C267.439 22.2002 267.206 22.4334 267.299 22.69C267.393 22.9466 267.346 23.0166 267.183 22.9C266.856 22.7134 265.27 23.3432 265.27 23.6697C265.27 23.9963 265.363 23.9963 266.32 23.5998C266.763 23.4132 267.136 23.2965 267.136 23.3432C267.136 23.3898 267.463 23.3432 267.836 23.2265C268.256 23.1099 268.536 23.1099 268.536 23.2499C268.536 23.3665 268.372 23.4598 268.186 23.4598C267.999 23.4598 267.836 23.6697 267.836 23.903C267.836 24.1596 267.369 24.6961 266.67 25.186C266.04 25.6525 265.503 26.1191 265.503 26.2357C265.503 26.3523 265.293 26.4923 265.037 26.5623C264.71 26.6322 264.57 26.8888 264.57 27.2854C264.57 28.0785 263.8 28.825 263.007 28.825C262.471 28.825 262.424 28.8716 262.704 29.0816C262.984 29.2915 263.007 29.3381 262.727 29.4548C262.564 29.5248 262.004 29.9913 261.514 30.4812C260.698 31.2976 260.558 31.3676 260.115 31.111C259.718 30.8544 259.602 30.8777 259.368 31.181C259.205 31.3909 258.948 31.5075 258.762 31.4376C258.529 31.3676 258.482 31.4376 258.575 31.6941C258.669 31.9507 258.529 32.0907 258.015 32.2073C257.642 32.3006 257.339 32.4639 257.339 32.5806C257.339 32.6972 257.082 32.8372 256.756 32.9305C256.429 33.0004 256.173 33.1637 256.173 33.3037C256.173 33.4203 256.079 33.467 255.986 33.397C255.869 33.327 255.473 33.467 255.076 33.7236C254.376 34.1435 254.353 34.2134 254.33 35.9163C254.306 37.3859 254.376 37.6425 254.656 37.5492C255.03 37.4092 255.146 37.8057 254.82 38.0157C254.54 38.179 254.516 39.5553 254.796 39.5553C254.913 39.5553 255.006 39.7652 255.006 40.0218C255.006 40.3017 255.356 40.6983 255.939 41.0715C257.152 41.8413 258.412 41.8413 259.462 41.0715C259.905 40.7682 260.698 40.2784 261.258 39.9985C261.794 39.7185 262.238 39.322 262.261 39.112C262.261 38.8088 262.308 38.7855 262.401 39.0887C262.471 39.2753 262.681 39.5319 262.844 39.6252C263.031 39.7185 263.171 40.0918 263.171 40.4417C263.171 40.7916 263.381 41.3281 263.637 41.6547C263.894 41.9812 264.104 42.3311 264.104 42.4244C264.104 42.5411 264.477 43.0076 264.92 43.4741C265.363 43.9407 265.62 44.3606 265.503 44.4072C265.363 44.4539 265.27 44.6405 265.27 44.8271C265.27 45.0137 265.2 45.1303 265.107 45.1303C264.734 45.0604 263.404 45.4569 263.404 45.6202C263.404 45.8535 262.028 46.18 261.841 46.0168C261.794 45.9468 261.841 45.6902 261.981 45.4569C262.214 45.0604 262.214 45.0604 262.238 45.4569C262.238 45.6902 262.354 45.8068 262.471 45.7368C262.611 45.6435 262.657 45.5036 262.587 45.3869C262.517 45.2703 262.564 45.1303 262.704 45.037C263.124 44.7804 262.961 44.2206 262.471 44.2206C262.004 44.2206 261.818 43.6608 262.214 43.4042C262.494 43.2409 262.471 41.8879 262.191 41.8879C262.098 41.8879 262.004 41.9812 262.004 42.1212C262.004 42.2378 261.864 42.3545 261.678 42.3545C261.514 42.3545 261.235 42.5177 261.071 42.7044C260.908 42.891 260.488 43.0543 260.115 43.0543C258.855 43.0543 257.992 45.5502 258.925 46.5066C259.252 46.8799 259.275 47.1831 258.972 47.3697C258.855 47.4397 258.809 47.6496 258.902 47.8596C258.995 48.1162 259.112 48.1395 259.345 47.9296C259.602 47.7196 259.672 47.7429 259.672 48.0462C259.672 48.2561 259.555 48.4194 259.415 48.4194C259.228 48.4194 259.228 48.536 259.438 48.7693C259.952 49.3758 259.322 49.8424 258.249 49.6791C257.642 49.5624 257.176 49.6324 256.826 49.8657C256.499 50.0756 255.986 50.1689 255.566 50.1223C254.47 49.959 253.467 50.5655 253.28 51.5219C253.047 52.7582 251.507 54.1578 249.758 54.741C248.731 55.0675 248.568 55.2075 248.405 55.954C248.218 56.7004 248.078 56.8404 247.285 57.0503C246.772 57.1903 246.235 57.5168 246.072 57.7501C245.746 58.2866 244.929 58.3566 244.463 57.9134C244.299 57.7268 243.856 57.5168 243.483 57.4235C242.876 57.2836 242.83 57.3069 242.993 57.7968C243.483 59.383 243.436 59.4996 242.48 59.4996C241.99 59.4996 241.477 59.4063 241.313 59.313C241.01 59.1264 239.237 59.3363 238.678 59.6396C238.234 59.8495 238.188 61.0159 238.608 61.0159C238.771 61.0159 238.911 61.1092 238.911 61.2491C238.911 61.3658 239.051 61.4124 239.214 61.3424C239.751 61.1325 240.777 61.669 240.637 62.0656C240.567 62.2522 240.614 62.4155 240.777 62.4155C240.917 62.4155 240.964 62.2988 240.894 62.1822C240.637 61.7623 241.477 61.9489 241.99 62.4155C242.293 62.6721 242.363 62.882 242.2 62.882C241.99 62.882 242.013 62.9986 242.27 63.2552C242.48 63.4418 242.643 63.7218 242.643 63.8384C242.643 63.955 242.806 64.0483 243.016 64.0483C243.203 64.0483 243.296 64.1416 243.226 64.2816C243.156 64.3982 243.18 64.5149 243.296 64.5149C243.413 64.5149 243.506 64.6548 243.506 64.7948C243.506 64.9581 243.553 65.1913 243.599 65.3313C243.693 65.6579 243.366 67.9906 243.086 68.9003C242.806 69.7401 242.153 70.2066 241.477 69.9967C241.08 69.88 237.954 69.6468 235.015 69.5534C234.339 69.5301 233.732 69.4368 233.639 69.3435C233.312 69.0169 232.449 69.1802 232.029 69.6468C231.796 69.9033 231.423 70.1133 231.213 70.1133C230.56 70.1133 230.233 70.6731 230.49 71.3496C230.606 71.6529 230.746 72.6326 230.77 73.519C230.84 75.2218 230.303 77.1813 229.603 77.7411C229.417 77.9044 229.324 78.0444 229.393 78.0444C229.487 78.0444 229.463 78.3476 229.347 78.7442C229.23 79.1641 229.23 79.444 229.37 79.444C229.487 79.444 229.58 79.6073 229.58 79.7939C229.58 79.9805 229.72 80.1205 229.883 80.0738C230.187 80.0272 230.233 81.2168 229.953 82.2432C229.79 82.803 229.837 82.8264 230.793 82.803C231.353 82.803 232.029 82.7097 232.333 82.6164C233.172 82.3365 233.779 82.6864 234.175 83.6661C234.572 84.6925 235.458 85.1124 235.948 84.5059C236.461 83.8761 237.838 83.4329 239.261 83.4095C240.03 83.4095 240.847 83.4095 241.08 83.3862C241.29 83.3862 241.64 83.0596 241.827 82.6864C242.083 82.2199 242.433 81.9399 242.9 81.8466C243.436 81.7533 243.693 81.4967 243.973 80.8203C244.159 80.3071 244.463 79.9105 244.626 79.9105C244.789 79.9105 245.046 79.7472 245.209 79.5606C245.466 79.2574 245.442 79.1407 245.162 78.9075C244.603 78.4643 244.533 77.9277 244.929 77.2979C245.769 76.0616 247.192 74.5454 247.518 74.5454C247.705 74.5454 248.405 74.2654 249.081 73.9156C250.341 73.2857 251.041 72.3527 250.574 71.8861C250.458 71.7695 250.341 71.4196 250.341 71.1397C250.341 69.9967 252.814 69.5768 254.236 70.4865C255.006 70.9764 255.543 70.8831 256.383 70.1366C256.826 69.7401 257.456 69.3668 257.829 69.2969C258.202 69.2035 258.622 68.947 258.739 68.7137C258.855 68.4804 259.112 68.2471 259.298 68.1772C259.788 67.9906 261.771 68.9003 261.771 69.3202C261.771 69.5068 261.911 69.8567 262.074 70.1366C262.308 70.5332 262.284 70.6498 261.934 70.9297C261.631 71.1397 261.584 71.2796 261.794 71.3963C261.934 71.4896 262.144 71.4429 262.238 71.2563C262.401 71.023 262.494 71.023 262.797 71.2563C263.007 71.4429 263.171 71.7228 263.171 71.8861C263.171 72.0727 263.381 72.2127 263.637 72.2127C263.894 72.2127 264.104 72.2827 264.104 72.3993C264.104 72.7492 266.436 74.4054 266.973 74.4287C267.276 74.4287 267.789 74.7087 268.116 75.0586C268.442 75.4318 268.886 75.7117 269.096 75.7117C269.306 75.7117 269.469 75.875 269.469 76.0849C269.469 76.2949 269.772 76.5748 270.169 76.7148C270.682 76.9014 270.938 77.2279 271.218 78.0444C271.452 78.6975 271.498 79.1174 271.335 79.1641C271.218 79.2107 271.102 79.444 271.102 79.6539C271.102 80.1904 268.956 80.6803 267.276 80.517C265.9 80.4004 265.503 80.517 265.503 81.0769C265.503 81.4734 267.603 82.7097 268.256 82.7097C268.396 82.7097 268.606 82.8497 268.699 83.013C268.792 83.1996 269.119 83.3862 269.422 83.4795C270.005 83.6195 270.495 82.9897 270.262 82.3832C270.192 82.1732 270.192 81.73 270.262 81.4034C270.379 80.9369 270.519 80.8436 270.892 80.9602C271.475 81.1468 272.501 80.2837 272.501 79.6073C272.501 79.374 272.711 79.1174 272.968 79.0474C273.621 78.8841 273.574 77.9977 272.851 77.5778C272.338 77.2513 272.315 77.158 272.571 76.6448C272.805 76.1782 273.015 76.0849 273.551 76.1316C273.948 76.1782 274.297 76.3182 274.321 76.4348C274.367 76.5515 274.577 76.8314 274.787 77.018C275.137 77.3679 275.184 77.3679 275.394 76.9014C275.697 76.2016 274.997 75.5018 272.968 74.522C271.592 73.8689 271.405 73.6823 271.592 73.3324C271.708 73.0991 271.732 72.9125 271.638 72.8892C271.545 72.8892 271.008 72.8658 270.472 72.8192C269.679 72.7725 269.306 72.5859 268.652 71.9094C268.209 71.4429 267.836 70.8831 267.836 70.6498C267.813 70.0666 267.416 69.6001 266.39 68.9936C265.433 68.4104 265.083 67.7106 265.503 67.1974C265.667 66.9875 265.69 66.8475 265.503 66.7309C265.037 66.451 265.27 66.0078 266.04 65.7279C266.973 65.4013 267.649 65.3546 267.183 65.6346C266.973 65.7745 266.95 66.0078 267.113 66.4277C267.393 67.1974 267.836 67.4774 268.093 67.0575C268.233 66.8009 268.303 66.8709 268.303 67.2441C268.303 67.524 268.419 67.8273 268.536 67.8972C268.676 67.9672 268.769 67.8273 268.769 67.5707C268.769 67.3141 268.886 67.0342 269.049 66.9408C269.236 66.8242 269.259 66.8942 269.166 67.1741C269.072 67.4074 269.142 67.734 269.376 67.9439C269.632 68.2238 269.655 68.3638 269.469 68.4804C269.166 68.667 270.262 69.6468 270.775 69.6468C270.962 69.6468 271.288 69.8567 271.498 70.1133C271.732 70.3699 272.175 70.5798 272.501 70.6031L273.084 70.6265L272.501 70.8131C271.941 70.9997 271.941 70.9997 272.665 71.023C273.061 71.0464 273.481 71.1397 273.551 71.2796C273.621 71.3963 273.878 71.5129 274.111 71.5129C274.321 71.5129 275.044 71.9094 275.697 72.376L276.91 73.2624L276.77 74.7087C276.653 76.0616 276.677 76.1549 277.283 76.5515C277.657 76.7847 277.796 77.018 277.633 77.1113C277.307 77.3213 277.587 78.0444 277.983 78.0444C278.17 78.0444 278.473 78.2077 278.66 78.4176C279.033 78.7675 279.033 79.1874 278.683 80.0038C278.59 80.2371 278.59 80.4937 278.73 80.587C278.846 80.657 279.009 80.9136 279.103 81.1235C279.266 81.59 279.733 81.6834 279.733 81.2635C279.733 80.6803 280.619 81.8466 280.782 82.6397C280.922 83.3629 280.992 83.4329 281.505 83.3162C281.902 83.1996 282.065 83.2696 282.065 83.5262C282.065 83.7128 282.182 83.8761 282.299 83.8761C282.439 83.8761 282.532 83.7594 282.532 83.6428C282.532 83.5028 282.648 83.4095 282.765 83.4095C282.905 83.4095 282.998 83.6661 282.998 83.9927C282.998 84.3193 283.115 84.5759 283.232 84.5759C283.675 84.5759 283.582 83.3862 283.068 82.3598C282.812 81.87 282.812 81.7533 283.022 81.9399C283.185 82.1032 283.512 82.2432 283.768 82.2432C284.095 82.2432 284.165 82.1266 284.048 81.7533C283.908 81.3568 283.978 81.2868 284.445 81.3568C284.864 81.4034 284.958 81.3335 284.911 80.9836C284.841 80.587 284.864 80.5637 285.191 80.8203C285.401 81.0069 285.564 81.0535 285.564 80.9602C285.564 80.8669 285.728 80.9136 285.914 81.0769C286.101 81.2401 286.264 81.5667 286.264 81.8233C286.264 82.2432 286.287 82.2432 286.637 81.9166C286.964 81.59 286.941 81.4967 286.521 81.0302C286.241 80.7503 286.031 80.6103 286.031 80.7036C286.031 81.0535 285.284 80.2837 285.074 79.7472C284.958 79.4207 284.678 79.2107 284.398 79.2107C284.071 79.2107 283.838 78.9775 283.675 78.5109C283.535 78.1144 283.325 77.8111 283.208 77.8111C282.952 77.8111 282.252 76.4582 282.345 76.1316C282.392 75.9916 282.765 76.1549 283.162 76.5048C283.885 77.1113 284.305 77.2979 284.048 76.878C283.955 76.7381 284.071 76.7148 284.305 76.8081C284.631 76.9247 284.678 76.878 284.538 76.5048C284.398 76.1549 284.421 76.1083 284.655 76.3415C285.214 76.9014 285.424 76.6448 284.888 76.0849C284.351 75.5251 284.351 75.5251 284.818 75.2685C285.168 75.0819 285.354 75.0819 285.471 75.2918C285.564 75.4551 285.541 75.4784 285.378 75.3851C285.214 75.2918 285.098 75.3385 285.098 75.4551C285.098 75.8983 285.728 75.735 285.868 75.2685C285.961 74.8486 286.054 74.8486 286.754 75.1052C287.197 75.2452 287.78 75.5018 288.084 75.6651C288.807 76.0616 289.553 75.945 290.043 75.3385C290.346 74.9652 290.696 74.8486 291.303 74.8719C291.746 74.8953 292.236 74.7786 292.376 74.592C292.656 74.2654 292.656 74.2654 291.839 73.9156C291.116 73.6123 290.113 71.7695 290.463 71.4196C290.603 71.2796 290.696 70.9531 290.696 70.6731C290.696 70.3699 290.883 70.1133 291.139 70.0433C291.513 69.95 291.606 69.6934 291.676 68.807C291.746 67.8273 291.839 67.6173 292.399 67.3607C292.842 67.1508 293.029 66.8709 293.029 66.4743C293.029 66.1477 293.332 65.5646 293.729 65.1447C294.102 64.7248 294.428 64.2583 294.428 64.095C294.428 63.955 294.685 63.8151 295.012 63.8151C295.362 63.8151 295.595 63.955 295.595 64.165C295.595 64.5149 296.808 64.8414 298.091 64.8181C298.511 64.7948 298.791 64.8881 298.721 65.0047C298.651 65.1214 298.487 65.2147 298.371 65.2147C298.067 65.2147 297.088 66.1477 297.368 66.1711C297.671 66.2177 297.974 66.2877 298.581 66.4976C299 66.6376 299.094 66.8242 299.047 67.2908C298.977 67.8039 299.07 67.8973 299.654 67.9672C300.097 68.0139 300.4 67.9206 300.54 67.6873C300.657 67.4774 301.007 67.3141 301.31 67.3141C301.613 67.3141 301.94 67.1974 302.01 67.0808C302.08 66.9408 302.523 66.8475 302.966 66.8475C303.526 66.8475 303.759 66.7309 303.759 66.4976C303.759 66.1011 303.876 66.1711 304.949 67.1275C305.182 67.3607 305.532 67.5473 305.695 67.5473C305.859 67.5473 306.208 67.7106 306.465 67.8972C306.698 68.0839 307.048 68.2471 307.235 68.2471C307.421 68.2471 307.748 68.4104 307.981 68.6204C308.331 68.9236 308.541 68.947 309.124 68.7603C309.707 68.5504 309.964 68.5737 310.594 68.9236C310.99 69.1802 311.714 69.4368 312.157 69.5301C312.6 69.6234 313.463 69.9967 314.046 70.3699C315.003 70.9764 315.259 71.0464 316.379 70.9064L317.639 70.7664L319.155 72.0961C321.721 74.3121 327.902 77.1346 327.482 75.875C327.436 75.735 327.226 75.5951 327.016 75.5951C326.783 75.5951 325.616 74.4987 324.403 73.1691C322.491 71.1163 322.187 70.6498 322.187 69.9733C322.187 69.2035 321.511 68.2472 320.718 67.9206C320.531 67.8506 320.508 67.7573 320.671 67.6173C320.811 67.524 321.021 67.1275 321.138 66.7542C321.371 66.0311 322.561 65.2147 323.074 65.4013C323.214 65.4479 323.354 65.3546 323.354 65.168C323.354 64.7015 323.937 64.2116 325.453 63.4418C326.153 63.0686 326.783 62.742 326.853 62.6954C326.923 62.6487 326.666 62.3688 326.269 62.0656C325.71 61.599 325.593 61.3424 325.64 60.6893C325.686 60.0361 325.523 59.6862 324.916 58.9864C323.913 57.8667 323.68 57.4235 323.494 56.1872C323.354 55.2075 323.354 55.2075 324.217 54.9742C324.707 54.8343 325.64 54.7176 326.293 54.7176C326.923 54.7176 327.576 54.601 327.716 54.461C328.182 53.9945 331.051 54.1345 332.568 54.671C333.617 55.0675 334.224 55.1608 335.064 55.0442C335.833 54.9276 336.183 54.9742 336.183 55.1608C336.183 55.3008 336.393 55.4174 336.627 55.4174C336.883 55.4174 337.233 55.5807 337.443 55.7673C337.863 56.2105 338.423 56.2105 338.586 55.7907C338.679 55.5574 338.983 55.4874 339.612 55.5807C341.269 55.7907 341.642 54.4377 340.079 53.9012C338.889 53.5047 338.399 53.0848 338.796 52.8048C339.099 52.5949 338.936 52.4316 337.583 51.4519C337.116 51.1253 337.093 51.032 337.35 50.0056C337.583 49.0492 337.583 48.8393 337.21 48.2095C336.627 47.2298 335.903 47.2531 334.994 48.3028C334.014 49.3991 333.244 49.3991 332.451 48.2795C332.124 47.8363 331.611 47.1364 331.308 46.7632C330.772 46.0867 330.772 46.0401 331.168 45.6202C331.401 45.3636 331.891 45.1537 332.264 45.1537C333.827 45.1537 334.387 43.2875 333.174 42.1679C332.311 41.3747 331.984 40.535 331.984 39.1354C331.961 36.4995 331.355 33.8869 330.025 30.9477C329.115 28.9183 328.789 27.9619 328.952 27.7753C329.092 27.612 329.698 27.3087 330.305 27.0755C332.684 26.189 335.204 24.3929 335.25 23.5764C335.25 23.4365 334.807 22.9 334.271 22.3634C333.128 21.2671 331.705 20.7772 328.952 20.544C326.479 20.3107 326.363 20.2874 325.01 19.7042C324.1 19.331 323.727 19.261 323.54 19.4476C323.33 19.6575 323.4 19.8442 323.773 20.1941C324.053 20.474 324.38 20.6373 324.497 20.5673C324.73 20.4273 326.549 21.2671 327.016 21.7103C327.459 22.1535 327.389 22.2935 326.736 22.2935C326.316 22.2935 326.199 22.3868 326.293 22.6434C326.409 22.9 326.339 22.9466 326.06 22.8533C325.873 22.7833 325.71 22.6434 325.71 22.5734C325.78 22.1069 325.36 21.3837 324.963 21.2904C324.473 21.1505 324.333 21.5004 324.823 21.6636C324.963 21.7103 324.683 21.8503 324.17 21.9436C323.657 22.0369 323.144 22.2002 323.004 22.2935C322.887 22.3868 322.281 22.4568 321.651 22.4568C321.044 22.4801 320.391 22.5967 320.204 22.76C320.041 22.9 319.645 22.9933 319.341 22.9466C318.688 22.8533 318.502 22.1535 319.085 21.9902C319.318 21.9202 319.085 21.8736 318.572 21.8503C318.058 21.8503 317.405 21.9902 317.125 22.1768C316.869 22.3634 316.286 22.5501 315.866 22.5734C315.423 22.5967 314.653 22.8766 314.14 23.1566C313.626 23.4598 312.997 23.6931 312.74 23.6931C312.017 23.7164 311.317 24.2296 311.34 24.7428C311.34 24.9994 311.177 25.2793 310.99 25.3726C310.361 25.6525 309.194 25.5592 308.891 25.2093C308.728 25.0227 308.448 24.8594 308.261 24.8594C307.608 24.8594 308.121 24.2996 309.008 24.0663L309.894 23.8097L309.101 23.2032C308.471 22.7134 307.958 22.5734 306.605 22.4801C304.879 22.3634 304.132 22.5967 305.392 22.8766C306.045 23.0166 306.092 23.1099 306.092 24.0196C306.092 24.8594 306.185 25.0694 306.675 25.2793C307.281 25.5592 307.398 25.9091 307.048 26.469C306.908 26.7022 306.745 26.7255 306.372 26.5156C306.115 26.3756 305.672 26.259 305.415 26.259C305.135 26.259 304.925 26.0957 304.925 25.8858C304.925 25.6525 304.692 25.7458 304.179 26.2357C303.759 26.6322 303.246 26.9588 303.059 26.9588C302.873 26.9588 302.406 27.2154 302.033 27.4953L301.356 28.0318L301.893 28.545C302.756 29.3381 302.243 29.6181 300.657 29.1982C299.957 29.0116 299.14 28.7317 298.837 28.5917C298.464 28.3818 298.207 28.3818 297.811 28.5917C297.251 28.8949 297.274 29.5248 297.834 29.5248C298.021 29.5248 298.161 29.6181 298.161 29.758C298.161 29.8747 298.464 29.9913 298.837 29.9913C299.21 29.9913 299.56 30.1313 299.654 30.3412C299.864 30.9477 298.464 30.8544 297.251 30.2012C296.784 29.968 296.225 29.758 296.015 29.758C295.781 29.758 295.595 29.6181 295.595 29.4548C295.595 29.2682 295.385 28.9416 295.128 28.7083C294.778 28.3818 294.732 28.1951 294.918 27.8452C295.198 27.3087 294.732 26.8422 293.682 26.6089C293.355 26.5389 293.029 26.329 292.959 26.1657C292.866 25.9558 293.285 25.9791 294.755 26.259C299.607 27.2621 300.96 27.2621 302.406 26.2357C303.153 25.6992 303.223 25.186 302.639 24.4629C302.243 23.973 300.517 23.1332 300.307 23.3198C300.26 23.3898 299.864 23.2732 299.467 23.0633C299.07 22.8533 298.697 22.69 298.627 22.69C298.441 22.7134 297.904 22.5034 296.761 22.0135C296.178 21.757 294.732 21.4537 293.565 21.3137C292.142 21.1505 291.396 20.9638 291.396 20.7772C291.396 20.4507 289.786 20.1241 289.437 20.3807C289.297 20.474 288.713 20.4973 288.13 20.4273L287.081 20.3107L287.85 20.0308C288.294 19.8675 288.573 19.6575 288.503 19.5409C288.317 19.2377 285.611 18.5145 285.331 18.6778C285.191 18.7478 285.098 18.7012 285.098 18.5845C285.098 18.4446 284.748 18.3279 284.305 18.3279C283.768 18.3279 283.418 18.4912 283.162 18.8411C282.905 19.2143 282.788 19.261 282.765 19.0277C282.765 18.8411 282.485 18.5379 282.135 18.3746C281.365 18.0247 281.202 18.0247 281.016 18.3512ZM282.065 18.7945C282.065 18.9111 281.949 19.0277 281.809 19.0277C281.692 19.0277 281.645 18.9111 281.715 18.7945C281.785 18.6545 281.902 18.5612 281.972 18.5612C282.019 18.5612 282.065 18.6545 282.065 18.7945ZM284.631 18.6545C284.631 18.8644 283.931 18.9811 283.815 18.7945C283.721 18.6545 283.885 18.5612 284.141 18.5612C284.421 18.5612 284.631 18.6078 284.631 18.6545ZM273.901 20.8939C273.901 21.0105 273.738 21.0572 273.551 20.9872C273.364 20.8939 273.201 20.8006 273.201 20.7539C273.201 20.7073 273.364 20.6606 273.551 20.6606C273.738 20.6606 273.901 20.7539 273.901 20.8939ZM271.568 21.3604C271.568 21.477 271.382 21.5937 271.172 21.5937C270.822 21.5937 270.798 21.547 271.102 21.3604C271.545 21.0805 271.568 21.0805 271.568 21.3604ZM269.096 22.9C269.166 23.0866 269.142 23.1799 269.002 23.1099C268.862 23.0399 268.769 22.8766 268.769 22.7367C268.769 22.4101 268.932 22.4801 269.096 22.9ZM270.938 22.69C270.775 22.7367 270.495 22.7367 270.355 22.69C270.192 22.62 270.309 22.5734 270.635 22.5734C270.962 22.5734 271.078 22.62 270.938 22.69ZM281.342 27.7986C281.832 27.8919 282.065 28.0785 282.065 28.3584C282.065 28.685 281.949 28.755 281.575 28.6617C281.202 28.5684 281.109 28.615 281.179 28.8483C281.319 29.2215 279.873 30.6911 279.359 30.6911C279.173 30.6911 279.033 30.7844 279.033 30.8777C279.033 31.2043 277.68 31.8574 277.33 31.7175C277.143 31.6475 276.91 31.6941 276.817 31.8341C276.747 31.9741 276.84 32.1607 277.027 32.2307C277.353 32.3473 277.353 32.4173 277.05 32.6972C276.793 32.9538 276.77 33.1637 276.933 33.4903C277.073 33.7469 277.26 34.1435 277.353 34.3534C277.47 34.5867 277.517 35.3098 277.493 35.9629C277.447 36.966 277.493 37.1526 277.82 37.1293C278.03 37.1293 278.193 37.2226 278.17 37.3625C278.146 37.5258 278.473 37.7124 278.916 37.7824C279.336 37.8757 279.733 38.039 279.779 38.179C279.826 38.3423 280.479 38.249 281.575 37.969C282.532 37.7124 283.465 37.5492 283.628 37.6191C283.815 37.6891 284.048 37.6425 284.141 37.5025C284.211 37.3625 285.028 37.1993 285.914 37.1293C287.267 37.036 287.64 37.0826 288.13 37.4558C288.783 37.9457 288.713 38.1323 287.827 38.2023C287.477 38.2256 287.197 38.3889 287.197 38.5755C287.197 38.8788 286.031 38.8555 283.698 38.5522C282.602 38.4122 279.873 39.6486 280.619 39.9518C281.016 40.1151 280.969 40.5117 280.549 40.3484C280.362 40.2551 280.199 40.0684 280.199 39.8818C280.199 39.6952 280.059 39.5553 279.919 39.5786C279.476 39.6019 278.636 39.9751 278.986 39.9985C279.406 40.0218 279.336 40.4883 278.893 40.4883C278.59 40.4883 278.543 40.6283 278.683 41.1181C278.87 41.8879 279.056 42.0979 279.196 41.7013C279.266 41.5614 279.499 41.4214 279.756 41.4214C279.989 41.4214 280.199 41.3281 280.199 41.2115C280.199 40.9315 281.109 40.7216 281.272 40.9782C281.342 41.0948 281.575 41.1881 281.785 41.1881C282.578 41.1881 282.345 43.4975 281.575 43.5208C281.505 43.5208 281.062 43.1709 280.572 42.7277C279.779 42.0279 279.616 41.9812 279.033 42.1912C278.123 42.5411 277.54 43.9407 277.703 45.3869C277.843 46.5766 277.447 47.2531 276.63 47.2531C276.42 47.2531 276.234 47.4164 276.234 47.603C276.234 47.8129 275.977 48.0695 275.674 48.2328C275.044 48.5127 274.251 48.3261 274.484 47.9529C274.904 47.2764 271.592 47.5797 270.985 48.3028C270.868 48.4427 270.099 48.6993 269.306 48.886C267.789 49.1892 267.136 49.0259 267.136 48.3028C267.136 47.6496 266.483 47.7429 264.244 48.746C263.754 48.9559 263.287 49.0726 263.194 49.0026C263.007 48.7926 264.104 47.9296 264.337 48.0695C264.43 48.1395 264.57 48.0229 264.664 47.8363C264.734 47.6496 264.967 47.4863 265.153 47.4863C265.363 47.4863 265.457 47.3697 265.387 47.2531C265.293 47.1131 265.107 47.0898 264.967 47.1831C264.78 47.2997 264.78 47.2298 264.967 46.9732C265.13 46.7865 265.2 46.5533 265.107 46.4833C265.037 46.39 265.083 46.32 265.223 46.32C265.387 46.32 265.503 46.1334 265.527 45.9001C265.55 45.6435 265.62 45.7135 265.713 46.0867C265.853 46.5999 266.016 46.6699 266.903 46.6699C267.766 46.6699 267.953 46.5766 268.046 46.2034C268.163 45.5269 268.769 45.177 269.772 45.1537C270.425 45.1537 270.635 45.0604 270.635 44.7338C270.635 44.4072 270.682 44.3839 270.868 44.6871C270.985 44.8738 271.125 44.9671 271.172 44.9204C271.452 44.6172 272.011 42.961 271.895 42.8443C271.802 42.7743 271.685 42.8443 271.638 42.9843C271.452 43.5208 271.078 43.3109 271.218 42.751C271.335 42.3311 271.475 41.4914 271.545 40.6983C271.568 40.605 271.802 40.465 272.105 40.3717C272.385 40.2784 272.665 40.1851 272.735 40.1618C272.805 40.1384 273.084 40.0451 273.388 39.9285C273.668 39.8352 273.901 39.6019 273.901 39.392C273.901 39.182 274.064 38.9021 274.251 38.7621C274.531 38.5522 274.531 38.4122 274.297 37.9457C274.018 37.3625 273.831 37.2459 272.618 36.7094C271.895 36.3828 271.638 35.8463 271.941 35.1698C272.035 34.9599 272.105 34.4934 272.105 34.1668C272.105 33.7702 272.315 33.4203 272.758 33.1171C273.131 32.8838 273.434 32.5572 273.434 32.4406C273.434 32.2073 275.067 31.3909 275.557 31.3909C275.674 31.3909 276.164 31.0177 276.653 30.5511C277.283 29.9446 277.423 29.688 277.19 29.5481C276.933 29.3848 276.98 29.1982 277.4 28.6383C277.703 28.2418 278.03 27.9619 278.146 28.0318C278.24 28.1018 278.333 28.0318 278.333 27.9152C278.333 27.6353 280.176 27.5653 281.342 27.7986ZM263.171 30.0379C263.171 30.3645 262.051 31.3909 261.701 31.3909C261.584 31.3909 261.608 31.2276 261.771 31.041C261.934 30.8544 261.981 30.6911 261.888 30.6911C261.771 30.6911 261.934 30.4812 262.238 30.2246C262.844 29.688 263.171 29.6414 263.171 30.0379ZM258.902 32.6972C258.832 32.7672 258.622 32.7905 258.459 32.7205C258.272 32.6505 258.342 32.5806 258.599 32.5806C258.855 32.5572 258.995 32.6272 258.902 32.6972ZM263.87 47.2531C263.87 47.3464 263.707 47.5097 263.497 47.5797C263.311 47.6496 263.217 47.6263 263.287 47.4863C263.357 47.3464 263.311 47.2531 263.194 47.2531C263.054 47.2531 262.937 47.1598 262.937 47.0431C262.937 46.9265 263.147 46.8799 263.404 46.9498C263.66 47.0198 263.87 47.1598 263.87 47.2531ZM277.913 47.2997C277.75 47.463 277.633 47.4863 277.633 47.3464C277.633 47.0198 277.913 46.7399 278.076 46.9032C278.123 46.9732 278.053 47.1598 277.913 47.2997ZM262.704 47.6963C262.704 47.5797 262.844 47.603 263.031 47.7663C263.521 48.2095 262.821 48.5594 262.098 48.2095C261.328 47.8596 261.351 47.4397 262.121 47.7196C262.494 47.8596 262.704 47.8596 262.704 47.6963ZM254.656 51.8951C254.586 52.0351 254.4 52.1517 254.283 52.1517C254.166 52.1517 254.19 52.0117 254.353 51.8251C254.68 51.4519 254.913 51.5219 254.656 51.8951ZM306.558 63.0919C306.115 63.3252 305.672 63.4885 305.579 63.4185C305.462 63.3719 305.392 63.4652 305.392 63.6518C305.392 63.8384 305.625 64.0717 305.929 64.1416C306.372 64.2816 306.395 64.3283 306.068 64.5149C305.859 64.6315 305.602 65.0047 305.532 65.3546C305.369 65.9378 305.322 65.9611 303.992 65.8445C303.246 65.7978 302.523 65.8445 302.406 65.9611C302.266 66.1011 301.986 65.9378 301.706 65.5646L301.217 64.9348L301.846 64.2583C302.196 63.8851 302.803 63.5585 303.176 63.4885C303.573 63.4418 304.062 63.3252 304.319 63.2319C304.925 62.9753 306.115 62.6954 306.792 62.6721C307.305 62.6487 307.281 62.6954 306.558 63.0919ZM280.199 80.2138C280.316 80.1671 280.362 80.3071 280.292 80.4937C280.199 80.6803 280.059 80.8436 279.943 80.8436C279.826 80.8436 279.663 80.8669 279.569 80.8902C279.476 80.9369 279.383 80.587 279.336 80.1438C279.266 79.3507 279.266 79.3507 279.639 79.7939C279.826 80.0505 280.083 80.2371 280.199 80.2138ZM284.631 79.6773C284.934 79.8639 284.934 79.9105 284.585 79.9105C284.375 79.9105 284.118 79.7939 284.048 79.6773C283.861 79.374 284.165 79.374 284.631 79.6773ZM283.162 80.517C283.092 80.587 282.882 80.6103 282.718 80.5403C282.532 80.4704 282.602 80.4004 282.858 80.4004C283.115 80.3771 283.255 80.447 283.162 80.517ZM283.628 81.3568C283.558 81.5434 283.488 81.4734 283.488 81.2168C283.465 80.9602 283.535 80.8203 283.605 80.9136C283.675 80.9835 283.698 81.1935 283.628 81.3568Z" />
        <path d="M217.357 26.4001C217.147 26.5867 216.774 26.7733 216.517 26.7967C216.261 26.82 216.051 26.9599 216.051 27.1232C216.051 27.2865 215.957 27.4265 215.864 27.4265C215.747 27.4265 215.467 27.5898 215.211 27.7764C214.814 28.1263 214.884 28.1263 216.237 28.0563C218.08 27.9397 218.267 28.4762 216.447 28.7328C215.351 28.8961 215.071 29.036 215.234 29.2926C215.304 29.4093 215.491 29.5259 215.607 29.5259C215.747 29.5259 215.794 29.4326 215.724 29.3393C215.561 29.0594 217.31 29.3159 217.497 29.6192C217.847 30.1324 217.683 30.459 217.1 30.459C216.75 30.459 216.517 30.5989 216.517 30.7855C216.517 31.0188 216.797 31.0888 217.637 31.0421C218.267 30.9955 219.06 31.1121 219.48 31.322C220.389 31.7419 222.442 31.8119 223.399 31.462C224.705 30.9721 225.731 30.6456 226.804 30.3423C228.274 29.9224 229.813 28.9194 229.813 28.3829C229.813 28.1263 229.533 27.8464 229.114 27.6597C228.74 27.4965 228.484 27.2865 228.554 27.1699C228.624 27.0532 228.53 26.9599 228.367 26.9599C228.181 26.9366 228.25 26.7967 228.53 26.5867C228.974 26.2368 228.95 26.2135 228.134 26.3534C227.644 26.4468 227.224 26.4001 227.131 26.2601C226.898 25.8869 226.081 25.9802 226.081 26.3768C226.081 26.5867 225.871 26.7267 225.568 26.7267C225.265 26.7267 224.378 26.7267 223.585 26.7267C222.465 26.7267 222.092 26.82 221.906 27.1466C221.696 27.5198 221.672 27.5198 221.532 27.1232C221.299 26.5634 220.716 26.6334 220.716 27.2165C220.716 27.5431 220.599 27.6597 220.366 27.5898C220.179 27.4965 219.969 27.5431 219.9 27.6597C219.643 28.0563 219.363 27.9163 219.48 27.4498C219.55 27.1466 219.34 26.89 218.733 26.5167C217.73 25.9336 217.8 25.9336 217.357 26.4001Z" />
        <path d="M237.441 33.6534C237.185 33.8867 237.255 34.4699 237.488 34.3066C237.605 34.2366 237.791 34.3066 237.884 34.4466C237.954 34.5865 237.908 34.7965 237.768 34.8898C237.558 35.0064 237.605 35.1464 237.884 35.3796C238.258 35.6595 238.281 35.6595 238.141 35.3096C238.071 35.0764 238.094 34.8431 238.234 34.7731C238.351 34.6798 238.398 34.5399 238.328 34.4232C238.258 34.3066 238.304 34.1666 238.444 34.0733C238.561 34.0033 238.678 33.7934 238.678 33.6534C238.678 33.4668 238.608 33.4668 238.468 33.7001C238.328 33.91 238.211 33.9334 238.118 33.7468C237.931 33.4668 237.675 33.4202 237.441 33.6534Z" />
        <path d="M242.433 39.3929C242.293 39.5329 242.177 39.7895 242.177 39.9528C242.177 40.3027 241.197 40.5826 240.427 40.4426C240.007 40.3493 239.751 40.466 239.494 40.8858C238.887 41.7956 238.211 42.3788 237.954 42.2388C237.838 42.1455 237.581 42.2155 237.371 42.3554C237.115 42.5887 237.091 42.7287 237.325 43.0086C237.488 43.1952 237.605 43.4051 237.581 43.4518C237.535 43.4751 237.511 43.5917 237.511 43.6851C237.511 43.8017 237.605 43.7317 237.721 43.5451C237.838 43.3585 237.978 43.3352 238.071 43.4751C238.141 43.6151 237.861 43.8483 237.395 43.9883C236.951 44.1283 236.578 44.3615 236.578 44.4782C236.578 44.6181 236.765 44.5948 237.045 44.4548C237.418 44.2682 237.511 44.2682 237.511 44.5481C237.511 44.7348 237.698 44.9214 237.931 44.9914C238.234 45.0613 238.188 45.1313 237.698 45.3413C237.325 45.4812 237.045 45.7145 237.045 45.8311C237.045 46.2043 237.511 46.3676 237.838 46.111C238.164 45.8544 238.444 45.9944 238.444 46.4143C238.444 46.5542 238.328 46.5076 238.211 46.321C238.024 46.0177 237.978 46.0411 237.978 46.391C237.978 46.7409 238.094 46.7875 238.678 46.6709C239.261 46.5542 239.354 46.5776 239.237 46.8808C239.167 47.0908 239.004 47.254 238.887 47.254C238.748 47.254 238.771 47.4873 238.934 47.7439C239.074 48.0238 239.261 48.1638 239.354 48.0472C239.587 47.7206 241.243 47.7206 241.104 48.0472C241.057 48.1871 241.197 48.537 241.43 48.7703C241.85 49.2601 241.85 50.0533 241.383 50.3332C241.243 50.4265 240.73 50.4498 240.264 50.3798C239.727 50.3332 239.377 50.3798 239.377 50.5198C239.377 50.6598 239.494 50.7064 239.611 50.6364C240.03 50.3798 239.867 50.7297 239.331 51.2196L238.794 51.7328L239.377 51.6395C240.404 51.4995 240.24 52.2693 239.144 52.7125C237.838 53.2257 237.931 53.8555 239.354 53.9255C240.474 53.9721 241.313 54.4853 240.287 54.4853C239.727 54.4853 238.678 55.2318 238.678 55.6283C238.678 55.7217 238.304 56.0016 237.861 56.2348C237.418 56.4914 237.115 56.7247 237.185 56.7947C237.441 57.0513 238.071 57.0046 238.608 56.6547C238.934 56.4448 239.401 56.3515 239.797 56.4215C240.287 56.5148 240.567 56.4215 240.847 56.1182C241.15 55.745 241.477 55.6983 242.853 55.7683C243.763 55.815 244.509 55.7683 244.509 55.6517C244.509 55.5584 244.952 55.4884 245.512 55.5117C246.539 55.5817 248.475 54.9052 248.475 54.5087C248.475 54.3687 248.311 54.2521 248.125 54.2521C247.682 54.2521 247.658 53.6456 248.102 53.4823C248.755 53.2257 249.221 52.2226 248.965 51.7095C248.638 51.1263 247.425 50.8464 246.889 51.2662C246.539 51.5228 246.515 51.5228 246.795 51.1963C247.029 50.893 247.029 50.6831 246.795 50.1466C246.655 49.7733 246.515 49.3535 246.515 49.2368C246.492 48.9336 245.839 48.3037 245.232 47.9538C244.976 47.8139 244.673 47.4173 244.603 47.0674C244.369 46.181 243.763 45.4112 243.296 45.3879C242.736 45.3879 242.76 44.6414 243.343 44.2216C243.599 44.035 243.809 43.755 243.809 43.6151C243.809 43.4518 243.973 43.1485 244.183 42.9619C244.742 42.3788 244.346 42.1222 242.876 42.1222C242.13 42.1222 241.43 42.0055 241.36 41.8889C241.29 41.7489 241.337 41.6556 241.477 41.6556C241.64 41.6556 242.013 41.4457 242.34 41.1891C242.666 40.9325 242.853 40.7226 242.76 40.7226C242.643 40.7226 242.783 40.4893 243.04 40.186C243.296 39.9061 243.576 39.5562 243.646 39.4163C243.856 39.0664 242.76 39.0664 242.433 39.3929ZM237.978 42.5887C237.978 42.7053 237.861 42.822 237.721 42.822C237.605 42.822 237.558 42.7053 237.628 42.5887C237.698 42.4487 237.814 42.3554 237.884 42.3554C237.931 42.3554 237.978 42.4487 237.978 42.5887Z" />
        <path d="M234.782 46.7405C234.222 46.9272 233.126 48.1868 233.546 48.1868C233.686 48.1868 233.732 48.3034 233.662 48.4201C233.592 48.56 233.056 48.6067 232.473 48.56C231.656 48.4667 231.4 48.5367 231.213 48.8866C231.05 49.1899 231.073 49.3531 231.283 49.4231C231.493 49.5164 231.493 49.5864 231.283 49.843C230.84 50.2862 230.91 50.4495 231.563 50.6128C232.379 50.8227 232.286 51.2193 231.213 52.0124C230.35 52.6655 229.953 53.552 230.536 53.552C230.653 53.552 230.7 53.6686 230.606 53.8319C230.49 54.0185 230.56 54.0418 230.816 53.9485C231.05 53.8785 231.213 53.9019 231.213 54.0418C231.213 54.4384 233.009 54.0418 234.199 53.412C234.782 53.1087 235.575 52.8521 235.995 52.8521C236.648 52.8521 236.765 52.7588 236.998 51.9424C237.161 51.4525 237.278 50.6361 237.278 50.1229C237.278 49.3531 237.371 49.1432 237.861 48.9333C238.538 48.63 238.608 48.0468 238.024 47.347C237.558 46.8105 235.715 46.4606 234.782 46.7405Z" />
        <path d="M260.511 70.8603C260.418 71.0702 260.091 71.3035 259.788 71.3735C259.485 71.4434 259.205 71.7 259.158 71.9333C259.018 72.5398 259.368 73.6362 259.812 73.9627C260.185 74.2193 260.161 74.2893 259.578 74.6625C259.088 74.9891 258.902 75.0358 258.762 74.8025C258.622 74.5926 258.529 74.6392 258.459 74.9425C258.389 75.199 258.482 75.5956 258.645 75.8289C258.832 76.0621 258.925 76.482 258.855 76.7386C258.505 78.5814 258.505 78.7214 259.018 79.0013C259.438 79.2112 259.578 79.1879 259.812 78.8613C259.975 78.6281 260.278 78.5115 260.465 78.5814C260.908 78.768 261.048 78.3248 261.118 76.3421C261.188 75.0124 261.118 74.6859 260.768 74.4293C260.395 74.1493 260.371 74.0094 260.605 73.2163C260.908 72.2366 260.838 70.1838 260.511 70.8603Z" />
        <path d="M284.328 85.4398C284.235 85.5098 284.165 85.7664 284.165 85.9764C284.165 86.2796 284.445 86.4196 285.144 86.5362C285.681 86.6295 286.194 86.7928 286.264 86.9328C286.357 87.0727 286.894 87.0727 287.664 86.9794C288.853 86.7928 289.297 86.4895 288.62 86.2563C288.41 86.1863 288.363 86.2563 288.48 86.4662C288.62 86.6995 288.527 86.6762 288.224 86.4196C287.99 86.2096 287.267 86.023 286.614 85.9997C285.984 85.953 285.401 85.8364 285.378 85.7198C285.331 85.6031 285.191 85.5798 285.051 85.6498C284.911 85.7431 284.748 85.6731 284.655 85.5331C284.561 85.3932 284.421 85.3465 284.328 85.4398Z" />
      </Continent>
      <Continent
        className={clsx('asia', selectedTimezone === timezones.asia && 'selected')}
        fill={getMapBaseColor(58)}
        onClick={(event) => onClickContinent(event, timezones.asia)}
      >
        <path d="M353.098 3.00077C351.512 3.39732 351.395 3.44398 351.932 3.65392C352.375 3.84053 352.375 3.86386 351.885 3.86386C351.115 3.88718 350.999 4.33039 351.605 4.96021C352.188 5.56671 352.655 5.68334 352.818 5.21681C352.911 4.98354 352.958 4.98354 352.958 5.21681C352.981 5.38009 353.145 5.52006 353.331 5.52006C353.518 5.52006 353.681 5.38009 353.681 5.21681C353.681 5.03019 353.751 5.03019 353.891 5.26346C354.008 5.45008 354.264 5.52006 354.474 5.45008C354.684 5.35677 355.314 5.45008 355.851 5.63669C356.924 5.98659 360.213 6.45312 361.776 6.45312C362.825 6.45312 363.362 6.03324 362.569 5.8233C362.289 5.75332 362.079 5.52006 362.079 5.28679C362.079 4.75027 361.519 4.56366 359.699 4.44703C357.857 4.33039 357.32 4.1671 357.623 3.79388C357.81 3.58394 357.507 3.44398 356.41 3.21071C354.124 2.74417 354.078 2.74417 353.098 3.00077Z" />
        <path d="M368.237 8.06197C367.607 8.34189 367.351 8.97171 367.701 9.39159C367.911 9.67151 367.771 9.71817 366.908 9.71817C366.324 9.71817 365.811 9.81146 365.765 9.9281C365.741 10.0447 365.298 10.208 364.808 10.278C364.155 10.3713 363.968 10.348 364.085 10.1614C364.318 9.76482 359.793 10.0914 359.396 10.5113C359.233 10.6746 358.37 10.8845 357.507 10.9545C356.644 11.0478 355.688 11.2811 355.384 11.491C355.034 11.7009 354.591 11.7943 354.241 11.7243C353.915 11.631 353.681 11.6776 353.681 11.8176C353.681 11.9342 353.518 12.0508 353.332 12.0508C353.145 12.0508 352.982 12.2375 352.958 12.4474C352.958 12.704 352.888 12.7506 352.818 12.564C352.655 12.1441 351.885 12.2141 351.722 12.6573C351.629 12.8673 351.652 13.1239 351.722 13.1938C351.815 13.2638 351.932 13.1938 351.979 13.0539C352.049 12.8439 352.212 12.8439 352.632 13.1239C352.935 13.3105 353.355 13.4504 353.588 13.4038C353.798 13.3571 354.031 13.4738 354.078 13.6137C354.218 13.987 352.375 14.0336 352.165 13.6837C352.095 13.5438 351.839 13.4504 351.629 13.4504C351.302 13.4504 351.302 13.4971 351.559 13.6604C352.049 13.9636 351.325 14.1269 349.039 14.1969L347.08 14.2436L347.756 14.7801C348.13 15.0833 348.666 15.5499 348.923 15.8064C349.226 16.1097 349.856 16.343 350.509 16.4129C351.092 16.4829 351.582 16.6229 351.582 16.7395C351.582 16.8328 351.769 16.9728 352.002 17.0194C352.212 17.0894 352.049 17.1361 351.629 17.1594C351.185 17.1594 350.556 17.0428 350.229 16.8562C349.669 16.5529 348.223 16.2263 345.844 15.8531C345.377 15.7831 345.027 15.8764 344.841 16.1097C344.561 16.4363 344.607 16.4829 345.074 16.5062C345.68 16.5296 345.54 16.5996 344.491 16.8328C343.838 16.9728 343.838 16.9961 344.281 17.2994C344.514 17.4626 345.097 17.7192 345.54 17.8592C346.31 18.1158 346.31 18.1158 345.214 18.0225C344.071 17.9292 343.441 17.4626 343.418 16.6929C343.418 16.273 342.461 15.3166 342.065 15.3166C341.901 15.3166 341.575 15.1766 341.318 15.0367C340.992 14.8267 340.805 14.8034 340.595 15.0133C340.408 15.2 340.548 15.3399 341.085 15.5265C341.505 15.6665 341.948 15.7131 342.088 15.6198C342.205 15.5499 342.251 15.5965 342.158 15.7365C342.088 15.8764 342.111 16.0397 342.251 16.133C342.671 16.3896 342.485 16.6695 341.785 16.8328C341.412 16.9261 341.085 17.1361 341.085 17.2994C341.085 17.4626 340.968 17.6493 340.805 17.6959C340.665 17.7426 341.132 18.0925 341.878 18.4657C343.138 19.0955 343.184 19.1655 343.044 19.8886C342.928 20.4951 342.998 20.7051 343.394 20.9617C343.674 21.1249 343.884 21.4049 343.884 21.5915C343.884 21.8481 344.234 21.8714 345.587 21.7781C347.033 21.6615 347.15 21.6848 346.38 21.8947C345.19 22.2213 345.051 22.6878 345.89 23.3643C346.427 23.8075 346.52 23.9708 346.287 24.2041C346.124 24.3674 345.984 24.7406 345.984 25.0205C345.984 25.4171 345.797 25.5804 345.284 25.697C344.864 25.7903 344.584 26.0002 344.584 26.2102C344.584 26.4901 344.421 26.5367 343.838 26.4201C343.418 26.3501 342.974 26.1635 342.834 26.0236C342.671 25.8603 342.718 25.8369 342.974 25.9303C343.231 26.0236 343.464 25.8136 343.768 25.1138C344.001 24.6006 344.257 24.1807 344.351 24.1807C344.467 24.1807 344.514 23.8308 344.491 23.411C344.467 22.7578 344.327 22.5945 343.488 22.2213C342.928 21.988 342.485 21.6615 342.485 21.5215C342.485 21.3815 342.251 20.8917 341.971 20.4485C341.691 20.0053 341.435 19.4454 341.388 19.2121C341.295 18.629 340.782 18.1624 340.035 17.9758C339.592 17.8592 339.499 17.7426 339.662 17.5793C340.012 17.2294 339.965 16.4596 339.592 15.9464C339.359 15.6432 338.776 15.4565 337.656 15.3399C336.7 15.2233 336.396 15.13 336.886 15.1066C337.656 15.06 337.679 15.06 337.259 14.7334C336.676 14.2902 334.927 14.2436 335.067 14.6634C335.137 14.8267 335.043 15.1066 334.857 15.2699C334.693 15.4332 334.553 15.9931 334.553 16.4829C334.553 17.276 334.46 17.4626 333.854 17.7659C332.991 18.2091 332.944 18.7456 333.737 19.0489C334.18 19.2121 334.32 19.4454 334.32 19.9586C334.32 20.7051 334.833 21.2649 335.323 21.0783C335.51 21.0083 335.953 21.1716 336.326 21.4049C336.7 21.6615 337.236 21.9647 337.539 22.0813C338.052 22.2913 338.216 22.6645 337.889 22.8511C337.773 22.9211 336.979 22.6645 336.07 22.2913C333.714 21.3349 333.317 21.3815 334.39 22.5012C334.857 22.9911 335.253 23.4809 335.253 23.5976C335.207 24.414 332.664 26.2102 330.308 27.0966C329.702 27.3298 329.095 27.6331 328.955 27.7964C328.792 27.983 329.118 28.9394 330.028 30.9688C331.358 33.908 331.964 36.5206 331.988 39.1565C331.988 40.5561 332.314 41.3959 333.177 42.189C334.39 43.3087 333.83 45.1748 332.267 45.1748C331.894 45.1748 331.404 45.3848 331.171 45.6413C330.775 46.0612 330.775 46.1079 331.311 46.7844C331.614 47.1576 332.127 47.8574 332.454 48.3006C333.247 49.4203 334.017 49.4203 334.997 48.3239C335.906 47.2742 336.63 47.2509 337.213 48.2306C337.586 48.8371 337.586 49.0937 337.353 50.0268C337.119 51.0298 337.119 51.1465 337.563 51.473C339.102 52.616 339.102 52.616 338.799 52.826C338.402 53.1059 338.892 53.5258 340.082 53.9223C341.645 54.4589 341.272 55.8118 339.615 55.6019C338.986 55.5086 338.682 55.5785 338.589 55.8118C338.426 56.2317 337.866 56.2317 337.446 55.7885C337.236 55.6019 336.886 55.4386 336.63 55.4386C336.396 55.4386 336.186 55.3219 336.186 55.182C336.186 54.9954 335.836 54.9487 335.067 55.0653C334.227 55.182 333.62 55.0887 332.571 54.6921C331.054 54.1556 328.185 54.0156 327.719 54.4822C327.579 54.6221 326.926 54.7388 326.296 54.7388C325.643 54.7388 324.71 54.8554 324.22 54.9954C323.357 55.2286 323.357 55.2286 323.497 56.2084C323.683 57.4447 323.916 57.8879 324.92 59.0076C325.549 59.7074 325.689 60.0573 325.643 60.7104C325.596 61.3869 325.713 61.6435 326.226 62.0167C326.576 62.2733 326.856 62.6232 326.856 62.7865C326.856 63.2064 327.649 63.4397 328.489 63.253C329.025 63.1364 329.235 63.2064 329.538 63.6962C329.911 64.2328 330.215 66.4721 329.888 66.2855C329.818 66.2389 329.212 66.1455 328.559 66.0756C327.485 65.9823 327.346 66.0289 327.112 66.5654C326.972 66.8687 326.902 67.2419 326.972 67.3585C327.066 67.4752 326.786 67.5685 326.389 67.5685C325.736 67.5685 325.689 67.6151 325.923 68.035C326.063 68.2916 326.319 68.5015 326.506 68.5015C326.692 68.5015 327.019 68.8048 327.252 69.2014C327.462 69.5746 327.742 69.9012 327.835 69.9012C327.929 69.9012 328.022 70.1111 328.022 70.391C328.022 70.6943 328.115 70.8109 328.302 70.6943C328.442 70.601 328.675 70.6943 328.815 70.8809C328.978 71.0908 329.352 71.3008 329.655 71.3474C330.448 71.4407 330.565 71.674 330.425 72.7237C330.355 73.2369 330.378 73.6334 330.518 73.6334C330.891 73.6334 331.568 75.0564 331.428 75.5695C331.264 76.246 331.801 77.2724 332.244 77.1791C332.524 77.1324 332.571 77.2957 332.501 77.8789C332.407 78.4621 332.454 78.5787 332.757 78.4621C333.34 78.2521 333.784 79.0219 334.04 80.6315C334.18 81.4712 334.344 82.311 334.414 82.4976C334.53 82.8008 334.227 82.9408 332.967 83.2674C331.054 83.7106 330.168 83.5473 328.838 82.4976C328.349 82.101 327.719 81.7978 327.439 81.7978C326.739 81.7978 326.156 81.098 325.993 80.0483C325.899 79.4418 325.946 79.1152 326.179 78.9986C326.366 78.8586 326.482 78.3454 326.506 77.669C326.506 76.5959 326.529 76.5493 327.159 76.5493C327.509 76.5726 327.789 76.456 327.789 76.316C327.789 76.1527 327.462 76.1061 327.019 76.1527C325.899 76.2694 321.164 73.8434 319.158 72.1172L317.642 70.7876L316.382 70.9275C315.262 71.0675 315.006 70.9975 314.049 70.391C313.466 70.0178 312.603 69.6446 312.16 69.5513C311.717 69.4579 310.993 69.2013 310.574 68.9448C309.99 68.6182 309.687 68.5715 309.197 68.7581L308.591 68.9914L309.127 69.5513C309.407 69.8545 310.247 70.391 310.97 70.7409C311.973 71.2308 312.416 71.604 312.72 72.2338C313.513 73.8667 312.486 74.5665 309.244 74.6831C308.008 74.7298 307.028 74.6598 307.028 74.5432C307.028 74.4265 306.725 74.3332 306.351 74.3332C305.955 74.3332 305.582 74.2166 305.512 74.1C305.418 73.96 305.162 73.9134 304.929 73.9833C304.625 74.0766 304.415 73.96 304.229 73.6334C304.042 73.2835 303.832 73.1902 303.482 73.3069C303.132 73.4002 302.876 73.3069 302.619 73.0036C302.269 72.5837 302.083 72.5604 300.333 72.7004C298.467 72.887 297.067 73.4235 296.391 74.2399C296.204 74.4732 295.738 74.4965 294.455 74.3566C292.729 74.1933 292.029 74.4032 292.612 74.9864C293.172 75.5462 292.285 75.8728 290.116 75.9195C288.973 75.9428 287.993 76.0594 287.947 76.176C287.9 76.316 287.713 76.3393 287.527 76.2694C287.34 76.1994 287.2 76.2694 287.2 76.4093C287.2 76.5493 287.363 76.6659 287.55 76.6659C287.76 76.6659 287.9 76.8758 287.9 77.2491C287.9 77.7856 288.087 77.9256 288.763 77.8556C288.903 77.8323 288.996 77.9955 288.95 78.2288C288.926 78.4621 288.95 78.742 289.043 78.882C289.113 78.9986 289.183 79.3252 289.23 79.5817C289.253 79.8383 289.206 79.9083 289.16 79.745C288.95 79.2785 288.437 79.4418 288.483 79.9783C288.506 80.3282 288.763 80.5848 289.323 80.7714C289.766 80.9347 289.976 81.0747 289.813 81.0747C289.58 81.098 289.58 81.1913 289.813 81.6112C289.976 81.9144 290.209 82.2177 290.326 82.2877C290.419 82.381 290.326 82.5676 290.093 82.7075C289.626 82.9641 289.719 83.0108 290.816 83.0341C291.189 83.0341 291.212 83.0574 290.886 83.1274C290.653 83.1974 290.466 83.3374 290.466 83.4773C290.466 83.5939 290.629 83.6639 290.816 83.5706C291.002 83.5006 291.166 83.5473 291.166 83.6639C291.166 83.8039 291.329 83.8972 291.516 83.8972C291.702 83.8972 291.866 83.7806 291.866 83.6406C291.866 83.2674 292.962 83.5939 293.498 84.1305C293.848 84.4804 294.222 84.597 294.898 84.5503C295.808 84.4804 295.948 84.3171 295.854 83.4773C295.761 82.8008 297.651 83.4073 298.724 84.4104C299.423 85.0402 301.36 84.8069 301.966 83.9905C302.456 83.3607 302.852 83.2907 303.529 83.7106C303.879 83.9439 304.065 83.9205 304.462 83.5473C305.045 83.0341 305.372 83.244 304.835 83.8039C304.555 84.0605 304.509 84.2704 304.672 84.4337C304.812 84.5737 304.812 84.8536 304.695 85.1802C304.579 85.4601 304.579 85.7867 304.672 85.9033C304.952 86.2065 305.162 87.7694 304.952 88.0027C304.649 88.3526 303.809 90.9185 303.529 92.3182C303.249 93.7644 302.736 94.4176 301.873 94.4176C301.569 94.4409 300.963 94.4875 300.566 94.5342C299.89 94.6275 299.797 94.7441 299.797 95.3506C299.797 96.4703 300.263 98.0332 300.753 98.5697C300.986 98.8263 301.196 99.1995 301.196 99.3628C301.196 99.7127 302.876 101.626 303.179 101.626C303.249 101.626 303.482 101.462 303.669 101.276C303.879 101.089 304.229 100.926 304.485 100.926C304.719 100.926 304.929 101.089 304.929 101.276C304.929 101.462 305.045 101.626 305.162 101.626C305.302 101.626 305.395 101.742 305.395 101.859C305.395 101.999 305.582 102.302 305.815 102.559C306.025 102.792 306.421 103.398 306.678 103.888C306.935 104.378 307.331 104.985 307.564 105.241C308.124 105.848 308.731 106.991 308.567 107.154C308.358 107.364 309.197 108.624 309.524 108.624C309.71 108.624 309.897 108.787 309.967 108.973C310.06 109.16 310.224 109.253 310.364 109.16C310.48 109.09 310.527 109.137 310.457 109.253C310.364 109.393 310.504 109.767 310.76 110.093C311.017 110.42 311.227 110.84 311.227 111.026C311.227 111.236 311.343 111.469 311.483 111.563C311.647 111.656 311.693 112.053 311.6 112.589C311.507 113.242 311.553 113.545 311.833 113.779C312.02 113.942 312.136 114.269 312.09 114.502C311.997 114.782 312.16 115.062 312.51 115.295C312.836 115.505 313.14 115.832 313.21 116.041C313.256 116.275 313.629 116.648 314.003 116.905C314.376 117.138 314.656 117.418 314.609 117.511C314.539 117.581 314.772 117.768 315.076 117.908C315.519 118.118 315.659 118.351 315.659 118.911C315.659 119.354 315.822 119.774 316.079 119.96C316.289 120.124 316.475 120.357 316.475 120.474C316.475 120.59 316.872 121.08 317.362 121.547C317.828 122.036 318.225 122.573 318.225 122.76C318.225 122.946 318.318 123.086 318.435 123.086C318.551 123.086 318.761 123.413 318.901 123.833C319.158 124.556 319.158 124.556 318.691 124.136C318.435 123.903 318.295 123.623 318.365 123.529C318.435 123.413 318.318 123.389 318.108 123.459C317.875 123.553 317.758 123.506 317.805 123.296C317.852 123.133 317.688 122.946 317.478 122.9C317.175 122.853 317.105 122.923 317.198 123.249C317.315 123.646 317.315 123.646 316.988 123.249C316.802 123.016 316.569 122.923 316.522 123.039C316.359 123.273 317.618 125.232 317.805 125.046C318.015 124.859 318.995 125.909 319.041 126.329C319.041 126.538 319.135 126.818 319.204 126.935C319.391 127.262 320.184 129.898 320.208 130.317C320.278 131.437 320.604 131.904 321.421 132.067C321.887 132.137 322.354 132.114 322.494 131.974C322.61 131.857 322.843 131.81 323.007 131.857C323.17 131.927 323.66 131.647 324.08 131.274C324.593 130.784 325.106 130.551 325.596 130.551C326.389 130.551 327.905 129.991 328.139 129.618C328.209 129.501 328.628 129.384 329.072 129.384C329.632 129.384 329.981 129.221 330.331 128.778C330.635 128.381 331.474 127.938 332.734 127.495C334.997 126.678 336.093 126.072 336 125.652C335.86 125.022 336.233 124.532 337.189 124.136C339.009 123.389 339.242 123.319 340.012 123.319C340.875 123.319 341.225 123.016 341.365 122.153C341.412 121.71 341.598 121.547 342.135 121.5C342.998 121.383 343.418 121.057 343.418 120.497C343.418 119.984 344.257 119.354 344.957 119.354C345.4 119.354 345.447 119.237 345.424 117.931C345.4 116.415 345.657 116.018 346.264 116.648C346.567 116.951 346.66 116.928 346.963 116.578C347.313 116.158 347.267 115.528 346.893 115.762C346.497 115.995 346.683 115.412 347.243 114.758C347.92 113.942 348.55 112.729 348.55 112.239C348.55 112.029 348.34 111.889 348.06 111.889C347.78 111.889 347.523 111.703 347.453 111.423C347.383 111.166 347.243 110.956 347.127 110.956C347.01 110.956 346.917 110.84 346.917 110.7C346.917 110.21 346.17 109.72 345.097 109.533C342.998 109.137 341.971 107.83 341.971 105.568C341.971 104.121 341.691 103.842 341.202 104.821C340.712 105.754 337.959 108.554 337.633 108.437C337.469 108.39 337.353 108.437 337.353 108.554C337.353 108.67 336.979 108.717 336.536 108.67C336.093 108.624 335.72 108.484 335.72 108.367C335.72 108.25 335.627 108.157 335.487 108.157C335.37 108.157 335.253 108.297 335.253 108.484C335.253 109.02 334.787 109.16 334.344 108.764C334.11 108.554 333.807 108.39 333.644 108.39C333.457 108.39 333.41 108.227 333.55 107.854C333.807 107.107 333.644 105.055 333.317 104.845C332.757 104.495 332.477 104.681 332.197 105.591L331.918 106.524L331.474 105.894C331.008 105.148 330.938 104.891 331.288 104.891C331.428 104.891 331.521 105.055 331.521 105.264C331.521 105.451 331.638 105.544 331.754 105.474C332.151 105.218 332.011 104.611 331.521 104.495C331.264 104.425 331.031 104.285 331.031 104.168C331.008 104.052 330.984 103.865 330.961 103.725C330.961 103.608 330.728 103.352 330.471 103.165C330.215 102.978 330.005 102.699 330.005 102.512C329.981 102.349 329.865 102.232 329.702 102.255C329.538 102.302 329.422 102.232 329.422 102.115C329.422 101.999 329.212 101.812 328.955 101.742C328.698 101.672 328.489 101.439 328.489 101.252C328.489 101.066 328.395 100.926 328.302 100.926C327.975 100.926 327.066 98.6864 327.229 98.2198C327.322 97.9399 327.299 97.8699 327.159 98.0565C327.019 98.2431 326.832 98.2665 326.622 98.1265C326.319 97.9399 326.319 97.8932 326.622 97.8932C326.832 97.8932 327.159 97.7066 327.369 97.4734C328.139 96.6803 328.395 96.5403 328.815 96.7736C329.025 96.8902 329.375 96.9368 329.608 96.8669C329.888 96.7736 330.168 96.9835 330.541 97.4967C330.821 97.9166 331.054 98.3831 331.054 98.5231C331.054 98.663 331.148 98.8263 331.264 98.873C331.381 98.8963 331.778 99.4561 332.127 100.086C332.617 100.902 333.037 101.276 333.62 101.486C334.064 101.649 334.74 102.045 335.113 102.395C335.487 102.745 335.953 103.025 336.163 103.025C336.466 103.025 336.466 103.072 336.233 103.235C336 103.375 336.163 103.468 336.746 103.585C337.213 103.678 337.539 103.842 337.469 103.982C337.376 104.098 337.446 104.191 337.563 104.191C337.703 104.191 337.819 104.075 337.819 103.958C337.819 103.795 337.983 103.795 338.286 103.958C338.542 104.098 338.869 104.145 339.009 104.075C339.126 103.982 339.615 103.865 340.082 103.818C341.015 103.702 341.621 103.328 341.412 103.002C341.225 102.675 342.205 102.769 342.415 103.072C342.508 103.235 342.718 103.795 342.834 104.308C343.138 105.451 344.024 105.941 346.194 106.174C347.943 106.361 348.666 106.477 349.483 106.664C350.486 106.921 352.515 106.967 352.515 106.757C352.515 106.664 352.935 106.617 353.448 106.687C354.008 106.781 354.381 106.734 354.381 106.571C354.381 106.431 354.684 106.408 355.221 106.524C355.688 106.641 356.714 106.617 357.577 106.477C359.303 106.221 359.746 106.314 359.746 106.991C359.746 107.294 359.91 107.457 360.19 107.457C360.469 107.457 360.679 107.69 360.796 108.087C360.983 108.88 361.566 109.557 362.032 109.557C362.219 109.557 362.686 109.86 363.035 110.256C363.992 111.236 364.948 111.679 365.858 111.516C366.581 111.399 366.604 111.399 366.185 111.749C365.718 112.123 365.345 112.216 364.598 112.146C364.365 112.123 364.178 112.216 364.178 112.379C364.178 112.822 367.398 115.855 367.841 115.855C369.054 115.808 370.803 114.222 370.267 113.685C370.033 113.452 370.197 112.589 370.477 112.589C370.57 112.589 370.593 112.846 370.523 113.172C370.477 113.522 370.523 113.755 370.687 113.755C370.85 113.755 370.897 113.872 370.827 113.989C370.733 114.129 370.85 114.549 371.036 114.968C371.293 115.458 371.363 115.971 371.27 116.508C371.176 116.951 371.223 117.674 371.363 118.141C371.97 120.264 372.413 121.896 372.693 123.133C372.856 123.856 373.392 125.209 373.859 126.165C374.349 127.098 374.932 128.241 375.142 128.685C375.375 129.128 375.725 130.107 375.912 130.877C376.122 131.624 376.635 132.767 377.078 133.443C377.941 134.726 379.108 137.479 379.108 138.179C379.108 138.785 380.367 140.721 380.974 141.048C381.65 141.398 382.49 140.861 382.677 139.928C382.84 139.182 383.656 138.808 384.519 139.088C384.869 139.182 384.963 139.135 384.893 138.925C384.823 138.738 384.636 138.598 384.449 138.598C384.076 138.598 384.076 138.412 384.333 137.479C384.473 137.012 384.683 136.849 385.103 136.849C385.616 136.849 385.662 136.779 385.522 136.172C385.289 135.029 385.312 133.163 385.592 132.837C385.966 132.393 385.989 130.481 385.639 128.685C385.196 126.189 385.312 125.512 386.246 125.745C386.689 125.862 386.829 125.792 386.945 125.302C387.015 124.976 387.225 124.719 387.412 124.719C388.205 124.719 388.905 124.159 388.905 123.506C388.905 123.156 388.998 122.853 389.115 122.853C389.488 122.853 391.821 120.497 392.567 119.354C392.964 118.747 393.687 118.071 394.2 117.838C394.69 117.581 395.32 117.138 395.623 116.858C396.159 116.368 396.206 116.088 396.089 114.945C396.043 114.642 396.183 114.315 396.369 114.222C396.556 114.129 396.953 113.942 397.256 113.779C397.676 113.569 397.816 113.569 397.909 113.849C398.049 114.199 398.935 114.152 400.195 113.755C400.568 113.615 401.151 113.522 401.478 113.522C402.294 113.499 403.251 112.752 403.064 112.286C402.994 112.076 403.017 111.843 403.111 111.773C403.227 111.726 403.367 111.889 403.437 112.146C403.507 112.426 403.671 112.566 403.787 112.496C404.044 112.332 404.3 112.869 404.3 113.545C404.3 114.105 404.814 114.968 405.863 116.158C406.283 116.625 406.633 117.138 406.633 117.278C406.633 117.418 406.82 117.488 407.053 117.441C407.52 117.301 408.126 117.838 407.963 118.257C407.916 118.421 407.963 118.724 408.079 118.934C408.219 119.237 408.196 119.354 407.986 119.354C407.776 119.354 407.776 119.424 407.986 119.634C408.219 119.867 408.336 119.867 408.453 119.657C408.733 119.237 409.129 119.61 409.642 120.777C410.039 121.64 410.085 122.13 409.992 123.599L409.852 125.349L410.505 125.535C411.812 125.885 412.232 125.862 412.558 125.372C412.745 125.116 413.258 124.649 413.724 124.346C414.261 123.973 414.564 123.576 414.564 123.249C414.588 122.526 416.127 125.769 416.337 126.935C416.407 127.448 416.64 128.148 416.804 128.498C416.99 128.848 417.13 129.384 417.13 129.734C417.13 130.084 417.27 130.317 417.457 130.317C417.643 130.317 417.9 130.644 418.04 131.041C418.39 132.02 418.366 132.184 417.947 132.044C417.69 131.927 417.643 131.997 417.737 132.277C417.807 132.487 417.97 132.65 418.087 132.65C418.18 132.65 418.157 132.977 418.017 133.373C417.83 133.91 417.713 134.003 417.573 133.77C417.457 133.606 417.363 133.56 417.363 133.7C417.363 134.143 417.737 134.493 418.18 134.446C418.693 134.4 418.95 135.123 418.553 135.519C418.39 135.683 418.203 135.659 417.947 135.449C417.737 135.263 417.597 135.263 417.597 135.403C417.597 135.543 417.783 135.753 417.993 135.893C418.436 136.126 418.553 137.712 418.226 139.065C417.993 140.045 418.296 142.074 418.6 141.771C419.066 141.304 419.696 141.468 419.696 142.027C419.696 142.471 419.743 142.494 419.929 142.214C420.093 141.934 420.163 141.934 420.163 142.191C420.163 142.354 420.489 142.891 420.862 143.38C421.492 144.173 421.539 144.313 421.212 144.5C420.862 144.687 420.862 144.733 421.189 144.99C421.516 145.247 421.562 145.247 421.562 144.873C421.562 144.477 421.586 144.477 421.912 144.92C422.099 145.2 422.215 145.853 422.192 146.39C422.169 146.926 422.262 147.439 422.402 147.533C422.542 147.626 422.729 148.139 422.822 148.676C422.892 149.235 423.218 149.935 423.545 150.285C423.872 150.635 424.128 151.148 424.128 151.428C424.128 151.801 424.571 152.245 425.831 153.084C427.674 154.321 428.094 154.717 428 155.114C427.954 155.277 427.977 155.324 428.024 155.207C428.234 154.857 428.84 155.044 428.7 155.417C428.607 155.697 428.654 155.767 428.91 155.65C429.143 155.557 429.26 155.674 429.26 155.977C429.26 156.233 429.377 156.443 429.517 156.443C429.68 156.443 429.727 156.233 429.657 155.953C429.563 155.58 429.587 155.534 429.843 155.744C430.053 155.93 430.193 155.93 430.31 155.744C430.543 155.37 430.216 155.044 429.54 155.02C429.143 154.997 429.05 154.95 429.33 154.88C429.54 154.834 429.727 154.624 429.727 154.437C429.727 154.251 429.657 154.134 429.563 154.181C429.447 154.251 429.353 154.041 429.353 153.737C429.33 153.434 429.027 152.804 428.677 152.338C428.117 151.615 428.047 151.311 428.047 149.842C428.07 148.909 427.954 147.836 427.79 147.463C427.534 146.786 425.691 144.78 425.341 144.78C425.248 144.78 424.968 144.453 424.735 144.057C424.431 143.567 424.175 143.38 423.895 143.45C423.172 143.637 422.565 142.984 422.239 141.701C422.099 141.024 421.819 140.348 421.679 140.231C421.329 139.951 421.026 138.482 421.306 138.482C421.539 138.482 421.516 137.875 421.259 137.642C421.166 137.549 421.096 137.712 421.096 137.992C421.096 138.598 420.443 138.925 420.139 138.458C419.836 138.015 420.023 136.056 420.489 134.516C420.722 133.816 420.909 132.603 420.932 131.834C420.979 130.481 421.002 130.434 421.632 130.364C422.215 130.294 422.262 130.341 422.262 131.204C422.262 132.067 422.285 132.114 423.078 132.137C424.082 132.184 424.828 132.557 424.828 133.023C424.828 133.233 424.991 133.35 425.248 133.303C425.504 133.257 425.761 133.49 426.041 134.05C426.274 134.493 426.438 135.006 426.461 135.146C426.461 135.519 427.301 136.382 427.65 136.382C427.767 136.382 427.814 136.242 427.72 136.079C427.627 135.916 427.65 135.869 427.767 135.986C427.977 136.196 428 136.289 428.07 136.966C428.117 137.525 428.56 137.385 428.56 136.802C428.56 136.359 428.607 136.336 428.98 136.662C429.19 136.849 429.587 137.082 429.843 137.175C430.17 137.292 430.216 137.385 430.007 137.525C429.587 137.782 429.68 140.348 430.123 140.348C430.286 140.348 430.543 140.138 430.706 139.881C430.87 139.601 431.453 139.135 431.989 138.808C432.899 138.295 433.272 137.712 433.342 136.732C433.366 136.592 433.482 136.522 433.645 136.569C434.042 136.639 435.255 135.986 436.001 135.309C436.328 135.006 436.725 134.749 436.888 134.749C437.214 134.749 437.471 133.537 437.331 132.603C437.284 132.254 437.354 132.02 437.448 132.09C437.564 132.16 437.658 131.997 437.658 131.74C437.658 131.484 437.564 131.227 437.448 131.157C437.354 131.087 437.191 130.527 437.121 129.921C437.051 129.314 436.818 128.475 436.631 128.101C436.421 127.705 436.258 127.238 436.258 127.028C436.258 126.842 435.955 126.375 435.558 126.002C435.185 125.629 434.858 125.209 434.858 125.046C434.858 124.882 434.672 124.696 434.415 124.602C433.879 124.439 431.126 121.803 431.126 121.477C431.126 121.337 430.776 120.87 430.333 120.404C429.657 119.727 429.54 119.424 429.54 118.514C429.54 117.838 429.657 117.418 429.843 117.348C430.007 117.278 430.333 116.858 430.543 116.415C430.753 115.971 431.01 115.668 431.126 115.738C431.219 115.808 431.383 115.692 431.453 115.505C431.523 115.318 431.709 115.178 431.873 115.202C432.036 115.248 432.153 115.108 432.153 114.898C432.106 114.409 433.692 113.849 434.322 114.105C434.555 114.199 434.998 114.269 435.302 114.245C435.745 114.199 435.815 114.269 435.698 114.572C435.488 115.155 436.281 116.765 436.795 116.788C437.331 116.788 437.518 116.251 437.191 115.622C437.005 115.272 437.005 115.155 437.168 115.248C437.331 115.342 437.424 115.225 437.424 114.945C437.424 114.665 437.564 114.432 437.728 114.432C437.868 114.409 438.124 114.385 438.241 114.339C438.381 114.315 438.614 114.269 438.777 114.269C438.941 114.245 439.011 114.105 438.917 113.942C438.824 113.802 438.847 113.755 439.011 113.849C439.151 113.919 439.314 113.895 439.407 113.779C439.5 113.615 439.757 113.615 440.13 113.755C440.97 114.082 441.227 114.035 441.063 113.615C440.993 113.429 441.017 113.336 441.157 113.406C441.297 113.476 441.39 113.429 441.39 113.289C441.39 113.149 441.507 113.102 441.647 113.196C442.02 113.406 442.323 112.822 442.206 112.123C442.113 111.563 442.113 111.563 442.37 112.169C442.626 112.776 443.489 113.102 443.489 112.612C443.489 112.496 443.909 112.169 444.446 111.913C444.959 111.633 445.332 111.516 445.239 111.656C445.169 111.773 445.216 111.889 445.332 111.889C445.472 111.889 445.589 111.773 445.589 111.656C445.589 111.516 445.659 111.446 445.775 111.446C446.592 111.586 447.058 111.353 447.292 110.746C447.432 110.35 447.758 110 448.015 109.93C448.295 109.86 448.715 109.487 448.971 109.09C449.228 108.717 449.554 108.39 449.718 108.39C450.091 108.39 450.557 107.527 450.347 107.177C450.254 107.037 450.277 106.967 450.394 107.037C450.651 107.201 451.211 106.641 451.024 106.431C450.931 106.361 451.094 106.291 451.374 106.291C451.677 106.291 451.887 106.151 451.887 105.918C451.887 105.731 451.794 105.614 451.677 105.684C451.374 105.871 451.164 104.845 451.42 104.541C451.56 104.378 451.584 104.191 451.49 104.121C451.42 104.028 451.467 103.958 451.607 103.958C451.77 103.958 451.887 103.748 451.887 103.515C451.887 103.258 452.004 102.978 452.12 102.908C452.26 102.839 452.354 102.512 452.354 102.209C452.354 101.416 452.587 100.926 452.96 100.926C453.147 100.926 453.31 100.786 453.31 100.622C453.427 99.7827 453.403 99.3628 453.217 98.9896C453.1 98.733 453.123 98.6164 453.31 98.6397C453.473 98.6864 453.59 98.2198 453.636 97.2401C453.66 96.447 453.59 95.7938 453.473 95.7938C453.38 95.7938 453.263 95.8871 453.263 96.0271C453.24 96.1437 453.217 96.3537 453.193 96.4703C453.147 96.9368 452.82 97.0068 452.447 96.6103C452.19 96.377 451.864 96.2837 451.514 96.3537C450.721 96.5636 451.024 95.9105 451.91 95.5139C452.61 95.1873 452.75 94.9774 452.354 94.7441C452.214 94.6508 452.167 94.5109 452.237 94.4176C452.284 94.3009 452.05 94.161 451.677 94.091C450.931 93.9277 450.557 93.3445 451.094 93.1346C451.257 93.0646 451.397 93.1346 451.35 93.2745C451.327 93.4145 451.444 93.5778 451.607 93.6245C451.747 93.6944 451.887 93.5778 451.887 93.4145C451.887 93.1346 451.094 92.4348 449.904 91.6417C449.788 91.5484 449.508 91.1985 449.321 90.8719C448.388 89.239 447.362 88.166 446.289 87.6528C445.239 87.1629 445.146 87.0463 445.286 86.5098C445.355 86.1832 445.729 85.6234 446.102 85.2502C446.778 84.5737 446.965 84.2238 446.475 84.5037C446.312 84.597 446.522 84.3404 446.942 83.9205C447.455 83.454 447.945 83.1974 448.411 83.1974C448.971 83.1974 449.088 83.1041 449.088 82.6376C449.088 81.9144 448.948 81.8444 447.642 81.7045C447.058 81.6578 446.359 81.4945 446.102 81.3546C445.612 81.098 444.329 81.238 444.516 81.5412C444.562 81.6112 444.469 81.8911 444.282 82.1244C444.003 82.4976 443.863 82.5443 443.373 82.311C443.046 82.171 442.79 81.8444 442.79 81.6345C442.766 80.9347 442.253 80.4682 441.437 80.4448C440.364 80.4215 439.524 79.8383 439.524 79.0919C439.524 78.6254 439.617 78.5087 439.897 78.6254C440.597 78.882 441.413 78.5087 441.53 77.8556C441.6 77.5057 441.973 77.0158 442.346 76.7825C442.72 76.526 443.023 76.176 443.023 76.0128C443.023 75.4762 443.606 75.0797 444.259 75.1963C445.192 75.3829 445.495 75.8495 445.122 76.6193C444.679 77.4823 444.702 77.669 445.192 78.0422C445.589 78.3221 445.589 78.3921 445.262 78.742C444.912 79.1385 444.982 79.4651 445.379 79.4651C445.495 79.4651 445.985 79.0219 446.499 78.5087C447.245 77.7389 447.618 77.529 448.481 77.4357C449.064 77.3657 449.718 77.389 449.951 77.4823C450.184 77.599 450.464 77.6456 450.557 77.6223C450.651 77.6223 450.954 77.6923 451.187 77.8322C451.584 78.0422 451.654 78.3221 451.654 79.6517C451.654 80.7481 451.724 81.1213 451.887 80.8647C452.004 80.6781 452.12 80.6081 452.12 80.7481C452.12 81.0747 453.263 81.5645 453.427 81.3079C453.497 81.1913 453.73 81.1913 454.01 81.3313C454.243 81.4712 454.453 81.5179 454.453 81.4479C454.453 81.3779 454.71 81.4945 455.036 81.7045C455.619 82.0777 455.829 82.7309 455.386 82.7309C454.989 82.7309 455.129 83.1274 455.806 83.8505C456.786 84.9236 456.996 85.2268 456.856 85.3901C456.599 85.6467 456.622 86.813 456.879 86.9763C457.019 87.0696 457.135 87.3729 457.135 87.6761C457.135 88.166 457.229 88.2126 458.185 88.1427C458.768 88.1193 459.282 87.9794 459.305 87.8627C459.352 87.7228 459.585 87.6295 459.841 87.6295C460.075 87.6295 460.285 87.5362 460.285 87.3962C460.285 87.2796 460.425 87.2329 460.611 87.3029C461.078 87.4895 461.917 86.5798 461.917 85.9266C461.917 84.9936 460.541 82.5676 459.422 81.5179C458.838 80.958 458.139 80.2816 457.859 79.9783C457.602 79.6751 456.972 79.2085 456.482 78.9286C456.016 78.672 455.456 78.2055 455.269 77.9022C454.919 77.389 454.943 77.3424 455.549 76.9692C455.899 76.7825 456.482 76.2227 456.809 75.7328C457.369 74.9397 457.392 74.8231 457.089 74.2399C456.692 73.4935 456.762 72.3271 457.205 72.3971C457.345 72.4438 457.485 72.2572 457.485 71.9772C457.485 71.7206 457.579 71.5574 457.695 71.6273C457.812 71.7206 458.022 71.4874 458.139 71.1375C458.279 70.7642 458.395 70.6243 458.395 70.8109C458.418 71.0675 458.512 71.0442 458.768 70.6943C459.048 70.3444 459.095 70.321 459.118 70.6243C459.118 71.1375 460.751 71.604 461.498 71.3241C462.244 71.0442 463.55 69.5279 463.55 68.9681C463.55 68.3383 464.203 65.9356 464.46 65.609C464.577 65.4691 464.67 64.396 464.67 63.253C464.67 62.0867 464.763 60.897 464.903 60.5705C465.067 60.1039 464.997 59.7074 464.553 58.7043C464.25 58.0045 463.877 57.3514 463.714 57.2581C463.574 57.1648 463.527 57.0248 463.62 56.9315C463.714 56.8382 463.55 56.6049 463.294 56.4183C462.431 55.8351 461.521 53.3392 462.151 53.3392C462.617 53.3392 464.437 54.8554 464.577 55.3219C464.647 55.5785 465.067 56.1617 465.486 56.6049C465.906 57.0481 466.49 57.9112 466.769 58.4944C467.049 59.0775 467.586 59.8473 467.959 60.1739C468.332 60.5005 468.752 61.1303 468.915 61.5968C469.055 62.0634 469.265 62.4366 469.382 62.4366C469.499 62.4366 469.685 62.7865 469.825 63.2064C469.989 63.6496 470.408 64.3027 470.805 64.676L471.528 65.3524L471.365 64.5127C471.202 63.6729 471.225 63.6729 471.855 63.7896C472.228 63.8595 472.694 64.1395 472.928 64.396C473.254 64.7926 473.324 64.8159 473.348 64.5127C473.348 64.0695 472.578 63.1364 472.228 63.1364C472.088 63.1364 471.715 62.9031 471.411 62.5999C471.085 62.32 470.572 61.9001 470.245 61.6901C469.919 61.4569 469.499 60.827 469.312 60.2905C469.125 59.7307 468.892 59.1009 468.799 58.8676C468.542 58.2611 469.662 58.3544 470.782 59.0542C471.761 59.684 471.948 59.7307 471.948 59.4508C471.948 59.3575 471.598 59.0775 471.202 58.821C470.782 58.5877 470.012 57.9579 469.475 57.4447C468.962 56.9082 467.609 55.6485 466.49 54.6221C465.37 53.5958 464.297 52.4761 464.11 52.1262C463.69 51.3331 462.757 50.3067 462.454 50.3067C462.314 50.3067 461.964 50.0268 461.661 49.6535C460.821 48.6971 460.378 48.4172 459.958 48.5805C459.631 48.6971 459.701 48.8138 460.308 49.187C461.031 49.6302 461.031 49.6302 460.611 50.0968C460.215 50.54 460.215 50.6099 460.541 50.8432C461.031 51.2164 461.778 52.3828 461.638 52.5461C461.451 52.7327 460.285 51.403 460.285 51.0065C460.285 50.8899 459.631 50.4467 458.838 50.0501C458.022 49.6769 457.066 49.187 456.716 49.0004C455.946 48.5805 453.893 48.3939 454.196 48.7671C454.43 49.0704 454.103 50.0734 453.753 50.0734C453.636 50.0734 453.52 49.8635 453.52 49.6069C453.52 49.3503 453.357 49.1404 453.17 49.1404C452.983 49.1404 452.82 48.9304 452.82 48.6738C452.82 48.4172 452.703 48.2073 452.563 48.2073C452.423 48.2073 452.354 48.0673 452.423 47.8807C452.61 47.3909 451.77 46.7144 451.257 46.9243C451.024 47.0409 450.721 47.0876 450.557 47.0643C450.417 47.041 450.231 47.1343 450.137 47.2742C449.998 47.4842 450.651 47.6008 451.77 47.5541C452.027 47.5541 452.097 47.6008 451.934 47.6708C451.7 47.7641 451.724 47.8807 452.027 48.2306C452.237 48.4639 452.307 48.6738 452.167 48.6738C452.05 48.6738 451.98 48.9071 452.05 49.187C452.12 49.5369 452.004 49.4436 451.7 48.9071C451.304 48.184 451.117 48.0673 450.044 47.9507C448.038 47.6941 448.015 47.6474 448.901 46.5278C449.904 45.2448 450.277 44.5217 449.974 44.4284C449.718 44.335 450.697 42.1423 451.001 42.1423C451.094 42.1423 451.071 41.8857 450.954 41.5592C450.767 41.0926 450.814 40.9293 451.071 40.836C451.257 40.7427 451.42 40.4395 451.42 40.1362C451.42 39.763 451.607 39.5064 451.91 39.3898C452.68 39.0865 456.016 39.0399 456.202 39.3198C456.272 39.4598 456.552 39.5764 456.809 39.5764C457.066 39.5764 457.229 39.5064 457.159 39.3898C457.112 39.2965 457.905 39.2498 458.932 39.2731C460.704 39.3431 460.798 39.3198 460.681 38.8533C460.588 38.5034 460.658 38.4101 460.938 38.5034C461.871 38.76 462.991 38.9699 463.177 38.8999C463.317 38.8766 463.667 38.8999 463.97 38.9932C464.413 39.1099 464.46 39.2032 464.25 39.4598C463.947 39.833 464.204 40.0429 464.997 40.0429C466.839 40.0429 467.283 39.9963 467.283 39.7863C467.283 39.6464 467.539 39.5997 467.936 39.693C468.309 39.763 468.915 39.7164 469.265 39.5764C470.082 39.2731 469.825 39.0165 468.309 38.62C467.283 38.3634 467.143 38.0368 467.563 36.9638C467.679 36.6372 467.796 36.0074 467.819 35.5875C467.866 34.8644 467.936 34.8177 468.799 34.7244C469.312 34.6778 469.942 34.6544 470.198 34.6544C470.455 34.6778 470.945 34.6778 471.318 34.7011C471.715 34.7011 471.948 34.8177 471.948 35.0277C471.948 35.4942 472.834 36.3573 473.138 36.194C473.254 36.1007 473.558 36.2873 473.814 36.5906C474.117 36.9405 474.257 37.0105 474.281 36.8005C474.281 36.6139 474.351 36.5439 474.444 36.6372C474.537 36.7072 474.724 36.4506 474.84 36.0307C475.05 35.4476 475.237 35.2376 475.75 35.191C476.31 35.121 476.683 34.6778 476.147 34.6778C476.03 34.6778 475.657 34.3979 475.307 34.048C474.701 33.4415 474.701 33.4181 475.19 33.2548C475.75 33.0449 476.45 33.2315 476.24 33.5348C476.1 33.7914 477.336 35.3776 477.686 35.3776C478.13 35.3776 477.943 36.1474 477.383 36.6606C477.01 36.9871 476.893 37.3137 476.963 37.6869C477.01 38.0135 476.963 38.3867 476.823 38.5733C476.66 38.7366 476.613 39.2032 476.707 39.5764C476.87 40.4395 476.52 41.4425 476.03 41.4425C475.844 41.4425 475.68 41.5358 475.68 41.6758C475.68 41.7924 475.447 41.9091 475.167 41.9091C474.631 41.9091 474.631 41.9091 475.237 42.4689C475.68 42.8888 475.82 43.2154 475.75 43.6119C475.564 44.5916 476.777 46.1779 479.483 48.5105C480.112 49.0471 481.069 49.8868 481.582 50.4C482.118 50.8899 483.028 51.683 483.611 52.1495C484.171 52.616 484.894 53.2692 485.198 53.6191C485.501 53.969 486.061 54.4822 486.411 54.7388L487.087 55.2286L487.46 54.5055C487.88 53.6891 487.764 52.616 487.204 52.1495C486.784 51.7996 486.784 51.0998 487.204 50.9365C487.39 50.8665 487.787 50.8665 488.09 50.9365C488.533 51.0298 488.44 50.8665 487.6 50.1434C486.784 49.397 486.62 49.117 486.76 48.6972C486.9 48.2539 487.017 48.2073 487.74 48.3472C488.44 48.4872 488.557 48.4406 488.463 48.0673C488.417 47.8341 487.973 47.3675 487.484 47.0409C486.061 46.0612 485.501 44.6383 486.76 45.2215C487.087 45.3614 487.414 45.4081 487.507 45.3381C487.717 45.1048 486.644 44.0085 486.201 44.0085C485.687 43.9852 484.544 42.9588 484.544 42.5156C484.544 42.0257 483.681 41.6058 483.005 41.7691C482.165 41.979 481.232 40.9993 481.419 40.0896C481.535 39.5531 481.419 39.2731 480.952 38.7833C480.369 38.2234 480.369 38.0602 480.882 38.1535C480.975 38.1768 481.279 38.0602 481.559 37.9202C481.815 37.7802 481.978 37.7569 481.885 37.8969C481.815 38.0135 481.908 38.1768 482.095 38.2701C482.468 38.4101 482.772 38.0602 482.585 37.7336C482.282 37.2437 483.145 37.337 483.611 37.8269C484.195 38.4334 484.544 38.4101 484.544 37.7336C484.544 36.8472 486.714 36.7772 488.487 37.5936C489.84 38.2234 490.096 38.2234 489.723 37.5703C489.56 37.2671 489.63 37.0338 490.05 36.6139C490.353 36.3107 490.679 35.7975 490.749 35.4942C490.819 35.1676 491.146 34.8411 491.566 34.7011C491.939 34.5845 492.242 34.3512 492.242 34.2346C492.242 34.0946 492.382 33.978 492.569 33.978C492.732 33.978 493.035 33.8147 493.199 33.6048C493.502 33.2315 494.412 33.2549 496.278 33.6514C496.814 33.768 496.908 33.7214 496.908 33.3248C496.908 33.0216 496.698 32.7883 496.394 32.6717C496.091 32.6017 495.695 32.3918 495.508 32.2285C495.321 32.0652 494.901 31.8552 494.575 31.7386C493.665 31.412 491.076 29.9191 491.076 29.7092C491.076 29.6159 491.286 29.3126 491.519 29.0327C492.126 28.4029 491.776 27.8197 490.563 27.4698C490.073 27.3065 489.723 27.0966 489.793 26.9799C489.863 26.84 490.539 27.0499 491.263 27.4232C492.382 27.983 492.872 28.0996 494.225 28.0763C495.578 28.0763 496.068 28.1696 497.024 28.6828C497.654 29.0327 498.424 29.3126 498.704 29.3126C498.96 29.3126 499.847 29.5226 500.617 29.7558C501.946 30.1757 502.063 30.1757 502.296 29.8025C502.506 29.4759 502.529 29.4759 502.413 29.8258C502.273 30.2924 502.856 30.3857 503.439 30.0124C503.742 29.8258 503.696 29.7558 503.276 29.6392C502.973 29.5459 502.809 29.4059 502.879 29.2893C502.949 29.1727 502.879 29.0794 502.763 29.0794C502.623 29.0794 502.296 28.8694 502.063 28.6362C501.596 28.1696 501.596 28.1696 502.203 28.0996C502.553 28.053 502.879 27.8897 502.926 27.7031C502.996 27.4698 502.926 27.4232 502.669 27.5165C502.436 27.6098 502.343 27.5631 502.413 27.3765C502.459 27.2132 502.226 26.91 501.853 26.7C500.757 26.0469 496.768 25.1138 495.088 25.1138C493.922 25.1138 493.245 24.9738 492.359 24.554C491.099 23.9241 480.626 21.3116 477.196 20.7517C476.053 20.5651 474.211 20.3785 473.114 20.3318C470.128 20.1919 467.446 19.9819 466.933 19.8186C466.559 19.7253 466.583 19.7953 467.119 20.3085C467.469 20.6351 467.866 20.915 467.982 20.915C468.099 20.915 468.589 21.1483 469.079 21.4282L469.965 21.9647L469.312 22.0347C468.939 22.0813 468.519 22.0114 468.356 21.8714C468.192 21.7314 467.563 21.5448 466.956 21.4748C466.35 21.3815 465.813 21.2416 465.766 21.1483C465.696 21.0316 465.86 21.0083 466.116 21.0783C466.373 21.1483 466.583 21.0783 466.583 20.9383C466.583 20.7984 466.513 20.7051 466.42 20.7051C466.303 20.7284 465.86 20.7051 465.416 20.6817L464.6 20.6118L465.416 20.4485L466.233 20.2852L465.486 20.2152C464.04 20.0752 463.9 20.0986 463.9 20.4485C463.9 20.7051 463.644 20.7984 462.851 20.775C460.961 20.7051 456.669 20.6817 456.319 20.7284C455.503 20.845 453.52 20.4018 453.52 20.0752C453.52 19.632 452.423 19.1188 450.557 18.6756C449.274 18.3724 448.248 18.3257 445.332 18.4424C442.416 18.559 441.553 18.5357 440.877 18.2558C440.434 18.0691 439.57 17.8592 438.941 17.7659C437.331 17.5793 436.701 17.3927 436.865 17.1594C437.238 16.5529 432.596 15.9464 431.896 16.5062C431.686 16.6929 431.569 16.7162 431.663 16.5529C431.919 16.1563 431.406 16.063 427.044 15.6665C422.309 15.2466 422.472 15.2466 423.195 15.6665L423.778 15.9931L422.915 16.0164C422.239 16.0164 422.075 16.0864 422.169 16.3663C422.262 16.5529 422.589 16.7162 422.915 16.7162C423.242 16.7162 423.428 16.7862 423.335 16.8795C423.265 16.9728 423.475 17.1827 423.825 17.346L424.478 17.6493L423.778 17.5093C423.405 17.416 422.869 17.3693 422.612 17.3693C422.355 17.3927 421.492 17.416 420.676 17.416C419.883 17.416 418.973 17.5326 418.693 17.6259C418.366 17.7426 417.713 17.7426 417.107 17.6026C416.547 17.486 415.847 17.3693 415.567 17.346C415.311 17.3227 415.124 17.2294 415.194 17.1127C415.241 17.0194 415.171 16.9495 415.007 16.9495C414.867 16.9495 414.797 17.1361 414.867 17.346C415.357 18.9089 415.381 18.8156 414.541 18.6756C413.328 18.4424 411.508 17.556 410.645 16.7395C410.225 16.343 409.712 16.0164 409.526 16.0164C409.339 16.0164 409.199 15.8764 409.199 15.7131C409.199 15.06 404.977 13.8703 404.627 14.4302C404.557 14.5235 404.37 14.5468 404.23 14.4535C404.067 14.3602 403.694 14.2436 403.367 14.2202C403.041 14.1736 402.318 14.0336 401.734 13.8936C400.988 13.7304 400.661 13.7304 400.615 13.8936C400.568 14.0336 400.428 14.1036 400.312 14.0103C400.195 13.9403 400.102 14.0103 400.102 14.1269C400.102 14.2669 400.312 14.5002 400.568 14.6634C400.825 14.8267 401.035 15.0133 401.035 15.0833C401.035 15.2 399.052 15.1766 396.836 14.99C396.136 14.9434 395.343 14.8967 395.086 14.8734C394.736 14.8501 394.69 14.8034 394.946 14.6401C395.133 14.5235 395.156 14.3835 395.016 14.2902C394.69 14.1036 389.045 14.0336 388.741 14.2202C388.392 14.4302 386.805 13.4971 386.805 13.0772C386.805 12.844 386.502 12.7506 385.732 12.7506C384.869 12.7506 384.636 12.8439 384.543 13.1938C384.473 13.4971 384.286 13.5904 383.983 13.5204C383.75 13.4505 383.376 13.4738 383.166 13.5437C382.863 13.6604 382.84 13.7537 383.073 14.0103C383.283 14.2902 383.19 14.4068 382.443 14.5935C381.324 14.92 380.204 14.92 380.391 14.6168C380.46 14.4768 380.787 14.3835 381.09 14.3835C381.394 14.3835 381.743 14.2435 381.837 14.0803C382.07 13.7304 382.677 13.3571 384.053 12.6807C384.659 12.4007 385.219 12.0275 385.289 11.8409C385.499 11.3044 384.473 10.3946 383.773 10.4879C383.446 10.5346 383.33 10.5346 383.493 10.4879C383.726 10.3946 383.703 10.3013 383.376 10.0447C382.886 9.67151 382.513 9.62486 382.723 9.95143C382.956 10.3247 382.56 10.2314 382.093 9.78815C381.743 9.48489 381.137 9.39159 378.804 9.39159C377.241 9.36826 375.702 9.46156 375.399 9.5782C374.979 9.71815 374.885 9.69484 374.955 9.46158C375.119 8.99504 374.722 8.31856 374.302 8.31856C374.116 8.31856 374.022 8.41188 374.092 8.55183C374.326 8.90173 373.323 8.83176 372.716 8.4352C372.086 8.01532 368.937 7.75871 368.237 8.06197ZM386.339 13.6837C386.595 13.847 386.502 13.917 385.942 13.917C385.522 13.917 385.173 13.8004 385.173 13.6837C385.173 13.3805 385.872 13.3805 386.339 13.6837ZM349.623 22.5712C350.182 23.201 349.902 23.341 349.249 22.7811C348.946 22.5246 348.573 22.3146 348.41 22.3146C348.246 22.3146 348.06 22.2213 347.99 22.128C347.756 21.7314 349.226 22.1513 349.623 22.5712ZM495.158 25.8136C495.228 25.9536 495.135 25.9769 494.948 25.9069C494.528 25.7436 494.458 25.5804 494.785 25.5804C494.925 25.5804 495.088 25.6737 495.158 25.8136ZM490.726 29.7792C490.913 30.0824 490.376 30.0824 489.91 29.7792C489.63 29.5925 489.653 29.5459 490.073 29.5459C490.353 29.5459 490.656 29.6392 490.726 29.7792ZM409.596 46.7377C410.529 48.9071 410.645 49.3503 410.365 49.7702C410.132 50.1434 410.109 50.1434 410.225 49.7702C410.295 49.5602 410.249 49.3736 410.132 49.3736C409.992 49.3736 409.899 49.5836 409.899 49.8402C409.899 50.1201 410.062 50.3067 410.319 50.3067C410.552 50.3067 410.622 50.3767 410.459 50.4933C410.319 50.5866 410.132 50.8665 410.039 51.1231C409.969 51.4264 409.502 51.753 408.896 51.9629C408.056 52.2895 407.916 52.4061 407.986 52.9426C408.056 53.4558 407.939 53.5724 407.123 53.829C406.423 54.039 406.027 54.039 405.653 53.8524C405.14 53.6191 405.14 53.5958 405.793 53.5958C406.446 53.5724 407.333 53.0126 407.333 52.5927C407.333 52.4994 407.543 52.1962 407.823 51.9162C408.079 51.6596 408.243 51.2864 408.173 51.0998C408.103 50.9365 408.173 50.7032 408.336 50.6099C409.106 50.1201 409.316 48.2773 408.826 46.5044C408.616 45.8513 409.292 46.0379 409.596 46.7377ZM364.738 63.5796C364.808 63.6962 364.715 63.8362 364.528 63.9295C364.342 63.9995 364.178 64.2094 364.178 64.4194C364.178 64.606 364.015 64.7693 363.829 64.7693C363.595 64.7693 363.479 65.0025 363.479 65.4924C363.479 66.1222 363.432 66.1922 363.129 65.9356C362.709 65.5857 362.662 64.536 363.059 64.536C363.199 64.536 363.269 64.466 363.199 64.396C362.989 64.1861 363.782 63.4163 364.225 63.393C364.435 63.3697 364.668 63.463 364.738 63.5796ZM366.115 63.533C365.951 63.5796 365.671 63.5796 365.531 63.533C365.368 63.463 365.485 63.4163 365.811 63.4163C366.138 63.4163 366.255 63.463 366.115 63.533ZM369.31 63.8362C369.567 64.0228 369.474 64.0461 368.96 63.9295C368.097 63.7429 367.911 63.6029 368.517 63.6029C368.75 63.6029 369.124 63.7196 369.31 63.8362ZM370.71 63.8129C370.71 63.9528 370.5 64.0228 370.243 63.9995C369.987 63.9762 369.777 63.8829 369.777 63.7896C369.777 63.6962 369.987 63.6029 370.243 63.6029C370.5 63.6029 370.71 63.6962 370.71 63.8129ZM342.648 64.2094C342.578 64.2794 342.368 64.3027 342.205 64.2328C342.018 64.1628 342.088 64.0928 342.345 64.0928C342.601 64.0695 342.741 64.1395 342.648 64.2094ZM340.852 65.2358C340.852 65.3524 340.735 65.4691 340.595 65.4691C340.478 65.4691 340.432 65.5857 340.502 65.7023C340.595 65.8423 340.852 65.889 341.108 65.819C341.435 65.7023 341.575 65.819 341.691 66.2622C341.761 66.5888 341.971 66.962 342.158 67.1253C342.415 67.3352 342.415 67.4985 342.111 68.0817C341.878 68.5482 341.645 68.7348 341.412 68.6415C341.202 68.5715 340.898 68.5015 340.712 68.5015C340.478 68.5015 340.385 68.175 340.385 67.2419C340.385 66.5421 340.292 65.889 340.152 65.819C340.012 65.7257 340.035 65.5624 340.222 65.3524C340.548 64.9559 340.852 64.9092 340.852 65.2358ZM340.035 66.4021C340.269 67.1719 339.895 67.4285 339.545 66.752C339.149 65.9589 339.149 65.9356 339.545 65.9356C339.732 65.9356 339.942 66.1456 340.035 66.4021ZM339.335 68.2216C339.195 68.3616 338.752 67.1719 338.776 66.6821C338.776 66.4721 338.939 66.7054 339.126 67.1953C339.312 67.7084 339.405 68.1516 339.335 68.2216ZM332.944 73.4468C333.084 73.7501 333.527 74.2866 333.947 74.6365C334.344 75.0097 334.53 75.243 334.367 75.1497C334.18 75.0564 333.9 75.173 333.714 75.4063C333.457 75.7328 333.317 75.7562 333.247 75.5462C333.177 75.4063 332.897 75.2663 332.641 75.2663C332.034 75.2663 331.544 74.6365 331.521 73.82C331.521 72.8636 332.501 72.6071 332.944 73.4468ZM303.995 100.109C303.995 100.296 303.902 100.459 303.762 100.459C303.646 100.459 303.599 100.296 303.669 100.109C303.762 99.9227 303.855 99.7594 303.902 99.7594C303.949 99.7594 303.995 99.9227 303.995 100.109ZM393.803 117.721C393.803 117.838 393.757 117.954 393.71 117.954C393.64 117.954 393.523 117.838 393.453 117.721C393.384 117.581 393.43 117.488 393.547 117.488C393.687 117.488 393.803 117.581 393.803 117.721ZM318.085 124.206C318.155 124.322 318.061 124.416 317.875 124.416C317.688 124.416 317.595 124.322 317.665 124.206C317.735 124.112 317.828 124.019 317.875 124.019C317.922 124.019 318.015 124.112 318.085 124.206ZM427.511 135.426C427.581 135.543 427.534 135.706 427.394 135.799C427.277 135.869 427.161 135.776 427.161 135.589C427.161 135.169 427.301 135.099 427.511 135.426ZM432.409 137.549C432.642 137.805 432.712 138.015 432.572 138.015C432.432 138.015 432.199 137.805 432.059 137.549C431.919 137.292 431.849 137.082 431.896 137.082C431.943 137.082 432.176 137.292 432.409 137.549Z" />
        <path d="M419.463 13.1944C419.463 13.591 420.139 13.9642 420.583 13.8242C420.886 13.7309 421.096 13.8009 421.189 14.0342C421.282 14.2675 421.166 14.3841 420.839 14.3841C420.559 14.3841 420.443 14.4774 420.536 14.6407C420.606 14.7806 420.909 14.8273 421.212 14.7573C421.492 14.6873 422.612 14.734 423.662 14.874C425.574 15.1072 426.181 15.0139 425.528 14.6174C424.945 14.2441 422.822 13.6843 421.982 13.6843C421.562 13.6843 421.002 13.521 420.769 13.3344C420.256 12.9612 419.463 12.8679 419.463 13.1944Z" />
        <path d="M471.248 66.0764C471.248 66.263 471.155 66.3563 471.015 66.2863C470.898 66.2163 470.782 66.263 470.782 66.4029C470.782 66.5662 470.922 66.6362 471.108 66.5662C471.691 66.333 473.231 68.5957 472.974 69.3188C472.904 69.4821 472.951 69.692 473.091 69.7853C473.604 70.0886 473.324 70.3218 472.671 70.1352C471.808 69.902 471.598 70.0886 472.088 70.6251C472.415 70.9983 472.438 71.0683 472.135 71.1849C471.925 71.2782 471.785 71.6281 471.785 72.0713C471.785 72.4912 471.831 72.6778 471.878 72.5379C472.041 72.0947 473.044 72.8411 473.208 73.5409C473.371 74.1707 473.978 74.2874 474.187 73.7509C474.281 73.5643 474.467 73.401 474.654 73.401C475.284 73.401 474.981 72.9578 474.047 72.5612C473.534 72.3279 473.114 72.0713 473.114 71.9547C473.114 71.6981 473.441 71.7214 473.744 72.0247C473.931 72.2113 474.117 72.188 474.467 71.978C474.841 71.7448 475.144 71.7681 476.217 72.2113C476.94 72.5145 477.64 72.6778 477.803 72.5612C477.99 72.4679 478.013 72.4912 477.92 72.6545C477.826 72.8178 477.873 72.9344 478.013 72.9344C478.2 72.9344 478.246 72.7245 478.153 72.3746C477.943 71.4882 478.223 71.0683 479.039 71.0683C479.809 71.045 480.579 70.6018 480.579 70.1819C480.579 70.0419 480.416 69.9719 480.206 70.0652C479.926 70.1819 479.856 70.0186 479.902 69.1322C479.949 68.3857 480.042 68.1058 480.276 68.1991C480.439 68.2691 480.579 68.2224 480.579 68.0825C480.579 67.9658 480.369 67.8025 480.112 67.7326C479.716 67.6393 479.623 67.7559 479.506 68.5257C479.343 69.4821 478.946 69.5521 478.946 68.5957C478.946 67.9192 478.573 67.8492 478.363 68.5023C478.223 68.9456 478.176 68.9456 476.543 68.3391C475.027 67.8025 473.908 67.2427 473.581 66.8695C473.348 66.5896 471.761 65.7031 471.505 65.7031C471.365 65.7031 471.248 65.8664 471.248 66.0764Z" />
        <path d="M474.747 73.9842C474.747 74.1941 474.63 74.2874 474.49 74.1941C474.304 74.0775 474.257 74.2174 474.374 74.6373C474.444 74.9872 474.397 75.3138 474.257 75.3838C474.094 75.5004 474.141 75.687 474.397 75.9669C474.584 76.1769 474.7 76.5268 474.607 76.76C474.514 77.0166 474.584 77.1333 474.84 77.1333C475.027 77.1333 475.307 77.3432 475.447 77.5998C475.82 78.3229 475.89 80.469 475.54 80.8656C475.237 81.2155 475.214 81.3088 475.214 81.9153C475.214 82.1252 474.934 82.4051 474.584 82.5684C474.211 82.7084 473.861 82.9416 473.767 83.0816C473.511 83.4082 473.091 82.7084 473.254 82.2185C473.417 81.7053 473.091 81.542 472.601 81.8919C472.274 82.1252 472.274 82.2418 472.554 82.7317C472.718 83.0583 472.858 83.1749 472.881 82.9883C472.881 82.8017 472.928 82.7084 472.998 82.7783C473.137 82.9183 472.578 84.1313 472.368 84.1313C472.088 84.1313 472.134 85.0177 472.414 85.111C472.554 85.1577 472.554 85.321 472.414 85.5076C472.251 85.7875 472.158 85.7875 471.855 85.5542C471.575 85.321 471.248 85.2976 470.338 85.5309C469.708 85.6709 468.939 85.7875 468.635 85.7642C467.656 85.6709 467.282 85.7642 467.282 86.0441C467.282 86.4406 466.116 87.7469 465.603 87.9336C465.37 88.0035 465.183 88.2835 465.183 88.5401C465.183 88.7966 465.043 89.1232 464.903 89.2632C464.74 89.4031 464.367 89.7064 464.063 89.9397L463.527 90.4062L464.157 91.106C464.507 91.5026 464.88 91.7591 465.02 91.6658C465.16 91.5959 465.183 91.6192 465.113 91.7591C465.02 91.8758 465.253 92.4123 465.603 92.9022C465.953 93.392 466.233 93.9752 466.233 94.1618C466.233 94.3484 466.466 94.535 466.769 94.5817C467.049 94.6283 467.282 94.7916 467.282 94.9316C467.282 95.0949 467.516 94.9549 467.819 94.6283C468.262 94.1385 468.332 93.8119 468.286 92.5289C468.262 91.6658 468.239 90.8961 468.192 90.7794C468.099 90.1963 468.262 90.2196 468.939 90.8727C469.708 91.6425 470.455 91.8525 470.222 91.2226C470.035 90.7561 470.478 89.963 470.898 89.963C471.085 89.963 471.248 90.0563 471.248 90.1963C471.248 90.6395 471.598 90.4295 471.878 89.823C472.018 89.4965 472.111 89.0999 472.041 88.9133C471.971 88.75 472.018 88.5401 472.158 88.4467C472.298 88.3768 472.414 88.4467 472.414 88.6334C472.414 89.0532 473.347 89.893 473.907 90.0096C474.257 90.0563 474.397 89.893 474.56 89.3098C474.677 88.89 474.91 88.5634 475.097 88.5634C475.61 88.5634 475.517 87.9102 474.98 87.7469C474.724 87.677 474.514 87.5137 474.514 87.397C474.514 87.3037 474.724 87.2571 474.98 87.3271C475.237 87.397 475.447 87.5137 475.447 87.607C475.447 87.677 475.913 87.7469 476.497 87.7469C477.453 87.7469 477.546 87.7003 477.546 87.1638L477.57 86.5806L477.919 87.2338C478.316 87.9802 478.806 87.8403 478.643 87.0238C478.526 86.3707 478.969 86.2774 479.506 86.8372C479.856 87.1638 479.902 87.1638 480.252 86.8372C480.439 86.6273 480.532 86.3473 480.462 86.2307C480.369 86.0907 480.415 85.8342 480.555 85.6709C480.742 85.4376 480.695 85.2277 480.345 84.8311C480.089 84.5279 479.856 84.0846 479.856 83.8047C479.832 83.5248 479.786 83.1516 479.762 82.965C479.739 82.7783 479.692 82.4518 479.669 82.2418C479.669 82.0319 479.436 81.6353 479.179 81.3321C478.923 81.0288 478.713 80.6556 478.713 80.469C478.713 80.2357 478.853 80.1891 479.179 80.2824C479.436 80.3757 479.646 80.329 479.646 80.1891C479.646 80.0491 479.552 79.9325 479.412 79.9325C479.296 79.9325 479.179 79.4193 479.179 78.7895C479.179 77.6931 479.109 77.5532 477.663 76.0602C476.823 75.1972 476.147 74.3341 476.147 74.1708C476.147 74.0075 476.007 73.8675 475.82 73.8675C475.633 73.8675 475.33 73.7976 475.12 73.7276C474.864 73.6343 474.747 73.7042 474.747 73.9842ZM475.913 74.6606C475.913 74.7073 475.75 74.8006 475.563 74.8939C475.377 74.9639 475.214 74.9172 475.214 74.8006C475.214 74.6606 475.377 74.5673 475.563 74.5673C475.75 74.5673 475.913 74.614 475.913 74.6606ZM471.645 87.7469C471.831 88.2601 471.645 88.3768 470.921 88.1902C470.432 88.0735 470.362 87.9802 470.572 87.7236C470.921 87.2804 471.458 87.3037 471.645 87.7469ZM469.942 88.4467C469.778 88.8433 469.242 88.9133 469.032 88.5867C468.939 88.4001 469.242 88.2368 469.895 88.1202C469.988 88.1202 470.012 88.2601 469.942 88.4467ZM467.912 88.8433C467.842 89.0299 467.772 88.9599 467.772 88.7033C467.749 88.4467 467.819 88.3068 467.889 88.4001C467.959 88.4701 467.982 88.68 467.912 88.8433ZM468.589 88.9599C468.425 89.5898 467.866 90.1963 467.469 90.1963C467.376 90.1963 467.306 90.0563 467.352 89.893C467.376 89.7297 467.096 89.5898 466.653 89.5431C466.233 89.4965 465.883 89.3565 465.883 89.2165C465.883 89.0532 466.279 89.0299 467.026 89.1232C468.052 89.2632 468.192 89.2165 468.332 88.7966C468.565 88.0968 468.752 88.2368 468.589 88.9599Z" />
        <path d="M301.99 85.6473C301.756 85.834 301.15 85.9972 300.683 85.9972C300.193 85.9972 299.797 86.0905 299.797 86.2305C299.797 86.3471 299.587 86.4638 299.307 86.4638C298.607 86.4638 298.84 87.2102 299.657 87.5835C300.24 87.8867 300.38 87.8634 300.986 87.4202C301.336 87.1403 301.756 86.9303 301.873 86.9303C302.013 86.9303 302.059 86.7903 301.99 86.6037C301.92 86.3938 302.059 86.1372 302.339 85.9506C302.829 85.6473 302.993 85.2974 302.643 85.2974C302.526 85.2974 302.246 85.4607 301.99 85.6473Z" />
        <path d="M455.689 120.1C455.479 120.427 455.526 123.786 455.736 124.579C455.876 125.022 455.853 125.046 455.549 124.649C454.803 123.739 454.71 124.486 455.316 126.375C455.713 127.588 456.319 128.451 456.786 128.451C456.926 128.451 457.019 128.265 457.019 128.055C457.019 127.845 457.089 127.752 457.182 127.822C457.252 127.915 457.206 128.241 457.019 128.545C456.692 129.151 456.856 129.968 457.252 129.734C457.369 129.664 457.485 129.711 457.485 129.828C457.485 130.224 456.786 130.131 456.646 129.734C456.552 129.548 456.342 129.384 456.156 129.384C455.923 129.384 456.016 129.641 456.506 130.248C456.879 130.737 457.369 131.461 457.579 131.88C458.069 132.86 458.419 133.047 458.792 132.534C459.188 132.02 458.978 131.041 458.419 130.527C457.999 130.154 457.999 130.131 458.535 129.851C459.118 129.524 459.888 129.758 459.328 130.084C459.095 130.248 459.095 130.364 459.352 130.667C459.655 131.041 459.701 131.041 459.958 130.667C460.215 130.317 460.261 130.317 460.471 130.714C460.611 130.947 460.798 131.041 460.891 130.947C461.008 130.854 460.938 130.551 460.775 130.294C460.238 129.408 460.658 129.618 461.731 130.761C462.711 131.81 462.874 132.184 462.337 132.184C462.244 132.184 462.174 132.114 462.197 131.997C462.267 131.764 461.614 130.784 461.404 130.784C461.078 130.784 461.218 131.507 461.568 131.624C461.801 131.717 461.918 132.09 461.918 132.743C461.918 133.373 461.988 133.583 462.127 133.373C462.291 133.117 462.431 133.117 462.874 133.443C463.69 134.003 463.83 133.91 463.41 133.093C463.201 132.65 463.131 132.3 463.27 132.207C463.387 132.137 463.527 131.787 463.597 131.437C463.69 130.924 463.597 130.761 463.061 130.527C462.337 130.248 462.151 129.851 462.734 129.851C462.921 129.851 463.084 129.944 463.084 130.084C463.084 130.364 463.317 130.387 463.76 130.108C463.994 129.944 464.017 129.804 463.807 129.408C463.48 128.848 463.084 128.755 463.084 129.268C463.084 129.594 463.014 129.594 462.384 129.268C461.988 129.058 461.591 128.965 461.474 129.035C461.358 129.081 461.194 128.988 461.124 128.778C461.008 128.475 460.775 128.428 459.958 128.545C459.165 128.661 458.908 128.638 458.815 128.335C458.675 128.008 458.932 127.892 459.771 127.915C459.911 127.938 460.051 127.868 460.051 127.798C460.051 127.565 459.235 127.075 458.838 127.052C458.372 127.052 458.255 126.725 458.372 125.839C458.442 125.419 458.605 125.186 458.885 125.186C459.468 125.186 459.725 123.226 459.212 122.456C459.025 122.153 458.838 121.477 458.838 120.963C458.838 120.054 458.815 120.054 458.069 120.194C457.625 120.264 457.252 120.217 457.159 120.077C456.926 119.727 455.923 119.75 455.689 120.1Z" />
        <path d="M463.784 132.44C463.784 132.673 464.087 133.093 464.46 133.373C464.857 133.653 465.09 134.003 465.02 134.213C464.927 134.469 464.833 134.446 464.6 134.12C464.157 133.536 463.97 133.863 464.134 134.843C464.227 135.333 464.437 135.682 464.623 135.682C464.833 135.682 464.95 135.939 464.95 136.359C464.95 136.872 465.113 137.082 465.603 137.292C465.976 137.455 466.28 137.689 466.28 137.852C466.28 138.015 466.35 138.365 466.443 138.622C466.56 139.042 466.513 139.112 466 139.112C465.673 139.112 465.417 139.042 465.417 138.925C465.417 138.808 465.323 138.715 465.183 138.715C465.067 138.715 465.02 139.018 465.113 139.461C465.23 139.998 465.183 140.138 464.973 140.021C464.787 139.905 464.647 140.021 464.553 140.324C464.39 140.931 464.297 140.931 463.784 140.301C463.434 139.835 463.294 139.811 462.757 140.068C462.431 140.208 462.151 140.488 462.151 140.698C462.151 140.908 461.941 141.048 461.568 141.048C461.241 141.048 460.938 141.188 460.891 141.328C460.845 141.491 460.728 141.957 460.588 142.331C460.471 142.727 460.401 143.17 460.471 143.334C460.611 143.754 461.218 143.66 461.218 143.194C461.218 142.984 461.381 142.587 461.568 142.354C461.918 141.911 461.918 141.911 461.918 142.401C461.918 142.937 462.244 143.1 462.501 142.68C462.571 142.564 462.781 142.517 462.967 142.611C463.201 142.68 463.317 142.564 463.317 142.237C463.317 141.817 463.387 141.771 463.76 141.981C464.134 142.167 464.227 142.447 464.204 143.427C464.18 144.477 464.25 144.687 464.833 145.153C465.207 145.456 465.72 145.713 466 145.76C466.28 145.806 466.49 146.04 466.536 146.366C466.63 146.926 467.143 146.949 467.189 146.389C467.189 146.203 467.283 145.876 467.376 145.643C467.493 145.386 467.446 144.967 467.259 144.593C466.839 143.754 466.863 143.544 467.353 143.474C467.586 143.45 467.749 143.567 467.749 143.824C467.749 144.033 467.913 144.43 468.099 144.687C468.402 145.106 468.426 145.083 468.449 144.267C468.449 143.8 468.566 143.334 468.682 143.264C468.822 143.194 468.916 142.914 468.916 142.634C468.916 141.771 467.913 138.482 467.703 138.622C467.563 138.715 467.563 138.388 467.656 137.875C467.796 137.105 467.773 137.035 467.399 137.152C467.119 137.269 466.979 137.175 466.886 136.849C466.746 136.312 466.35 136.219 466.35 136.732C466.35 136.919 466.186 137.082 465.976 137.082C465.696 137.082 465.65 136.989 465.836 136.779C465.953 136.616 465.976 136.289 465.86 136.032C465.557 135.263 465.626 135.169 466.35 135.263L467.026 135.356L466.583 134.633C466.326 134.213 466.116 133.63 466.116 133.326C466.116 132.44 465.743 132.09 464.74 132.044C463.947 131.997 463.784 132.067 463.784 132.44Z" />
        <path d="M459.585 133.561C459.585 133.701 459.702 133.864 459.818 133.934C459.982 134.027 460.028 134.634 459.982 135.357C459.888 136.663 459.982 136.826 460.495 136.313C460.728 136.103 460.845 136.103 460.915 136.313C460.961 136.476 461.171 136.523 461.358 136.453C461.848 136.267 461.754 136.92 461.218 137.386C460.821 137.713 460.798 137.853 461.031 138.249C461.195 138.506 461.451 138.716 461.591 138.716C461.754 138.716 461.918 138.879 462.011 139.066C462.221 139.649 462.571 139.462 462.851 138.599C462.991 138.156 463.224 137.783 463.341 137.783C463.481 137.783 463.504 137.876 463.434 138.016C463.271 138.273 463.994 138.319 464.577 138.109C465.137 137.899 465.02 137.2 464.367 136.85C463.9 136.593 463.784 136.36 463.784 135.637C463.784 134.914 463.714 134.75 463.317 134.75C463.061 134.75 462.851 134.844 462.851 134.984C462.851 135.1 462.664 135.217 462.407 135.217C462.081 135.217 462.011 135.077 462.104 134.634C462.198 134.144 462.104 134.05 461.661 134.05C461.358 134.027 460.845 133.887 460.518 133.701C459.818 133.304 459.585 133.281 459.585 133.561ZM463.247 135.73C463.177 135.917 463.107 135.847 463.107 135.59C463.084 135.333 463.154 135.194 463.224 135.287C463.294 135.357 463.317 135.567 463.247 135.73ZM463.084 137.433C463.084 137.55 462.967 137.503 462.851 137.316C462.734 137.13 462.617 136.85 462.617 136.733C462.617 136.593 462.734 136.663 462.851 136.85C462.967 137.036 463.084 137.293 463.084 137.433ZM463.481 137.363C463.411 137.55 463.341 137.48 463.341 137.223C463.317 136.966 463.387 136.826 463.457 136.92C463.527 136.99 463.551 137.2 463.481 137.363ZM462.547 137.829C462.477 138.016 462.407 137.946 462.407 137.689C462.384 137.433 462.454 137.293 462.524 137.386C462.594 137.456 462.617 137.666 462.547 137.829Z" />
        <path d="M455.246 135.217C455.153 135.66 454.896 136.313 454.64 136.663C454.383 137.037 454.266 137.433 454.336 137.55C454.43 137.69 454.406 137.76 454.29 137.736C454.01 137.643 452.19 139.509 452.004 140.092C451.91 140.396 451.957 140.745 452.097 140.885C452.284 141.072 452.354 141.072 452.354 140.885C452.354 140.769 452.89 140.116 453.52 139.462C454.173 138.809 454.686 138.18 454.686 138.04C454.686 137.666 455.689 136.687 456.179 136.593C456.809 136.477 456.902 136.01 456.319 135.847C455.992 135.777 455.853 135.544 455.853 135.1C455.853 134.75 455.946 134.541 456.086 134.634C456.226 134.704 456.319 134.681 456.319 134.541C456.319 134.424 456.109 134.331 455.853 134.354C455.526 134.377 455.339 134.611 455.246 135.217Z" />
        <path d="M385.499 137.83C385.406 137.97 385.452 138.156 385.592 138.203C385.756 138.249 385.872 138.483 385.872 138.739C385.872 139.019 385.756 139.113 385.522 139.043C385.336 138.949 385.172 139.019 385.172 139.159C385.172 139.299 385.289 139.416 385.406 139.416C385.709 139.416 385.709 140.419 385.429 140.699C385.242 140.909 385.662 144.104 385.989 144.968C386.199 145.504 387.108 145.691 387.855 145.317C389.068 144.711 389.371 144.314 389.371 143.311C389.371 142.355 388.438 140.139 387.832 139.626C387.645 139.486 387.505 139.229 387.505 139.089C387.505 138.739 386.292 137.55 385.942 137.55C385.779 137.55 385.592 137.666 385.499 137.83Z" />
        <path d="M451.724 142.634C451.607 142.728 451.351 143.194 451.164 143.661C451.001 144.151 450.768 144.547 450.651 144.547C450.558 144.547 450.348 144.85 450.231 145.224C450.091 145.597 449.718 146.087 449.415 146.297C448.948 146.6 448.878 146.74 449.111 147.02C449.321 147.3 449.275 147.346 448.761 147.346C448.412 147.346 447.968 147.51 447.782 147.696C447.572 147.883 447.222 148.046 446.965 148.046C446.662 148.046 446.522 148.209 446.522 148.536C446.522 148.816 446.102 149.539 445.589 150.146C444.819 151.079 444.493 151.289 443.606 151.405C443.046 151.499 442.51 151.662 442.44 151.802C442.347 151.918 442.183 151.965 442.067 151.895C441.88 151.778 441.74 152.642 441.833 153.248C441.857 153.341 441.763 153.645 441.623 153.901C441.414 154.274 441.297 154.321 440.854 154.064C440.574 153.901 440.154 153.808 439.921 153.855C439.711 153.878 439.454 153.738 439.384 153.551C439.174 153.038 438.521 153.458 438.078 154.368C437.681 155.138 437.705 156.001 438.078 158.007C438.404 159.663 438.708 160.153 439.291 160.013C439.687 159.896 439.757 159.989 439.734 160.666C439.734 161.109 439.804 161.949 439.874 162.555C440.061 163.722 440.037 163.698 441.81 163.675C442.627 163.675 442.79 163.745 442.79 164.118C442.79 164.655 443.163 164.958 443.373 164.608C443.466 164.468 443.7 164.422 443.933 164.492C444.166 164.562 444.563 164.398 444.843 164.142C445.122 163.862 445.356 163.768 445.356 163.908C445.356 164.025 445.519 164.142 445.706 164.142C445.892 164.142 446.056 164.258 446.056 164.398C446.056 164.562 446.265 164.608 446.545 164.538C447.315 164.328 447.688 164.608 447.688 165.331C447.688 166.148 447.828 166.171 449.111 165.425L450.138 164.865L450.418 165.378C450.674 165.891 450.698 165.891 450.954 165.215C451.094 164.841 451.141 164.352 451.071 164.118C450.977 163.885 451.117 163.302 451.327 162.812C451.607 162.252 451.654 161.926 451.491 161.879C451.047 161.716 451.444 160.783 452.284 159.989C452.727 159.593 453.17 159.243 453.31 159.243C453.427 159.243 453.473 158.73 453.38 158.007C453.217 156.444 453.59 155.861 454.756 155.931C456.016 156.024 456.086 155.557 454.943 154.554C454.15 153.878 453.963 153.598 454.103 153.201C454.197 152.875 454.08 152.478 453.777 152.035C453.357 151.452 453.31 151.265 453.567 150.729C453.753 150.286 453.753 150.052 453.567 149.866C453.24 149.539 453.683 149.073 454.546 148.839C455.386 148.629 455.526 148.326 454.803 148.139C454.103 147.976 454.033 147.58 454.71 147.58C455.293 147.58 456.086 147.183 456.086 146.857C456.086 146.74 455.853 146.53 455.573 146.413C454.896 146.11 454.15 145.644 453.963 145.41C453.893 145.317 453.683 145.247 453.497 145.294C453.287 145.34 453.194 145.2 453.24 144.897C453.264 144.617 453.054 144.197 452.704 143.894C452.4 143.614 452.19 143.241 452.26 143.054C452.4 142.658 452.097 142.401 451.724 142.634Z" />
        <path d="M413.282 146.18C412.815 146.927 413.538 147.953 416.267 150.519C417.06 151.265 417.597 151.989 417.597 152.315C417.597 152.478 418.063 152.945 418.647 153.365C419.23 153.761 419.673 154.181 419.65 154.275C419.533 154.764 420.443 157.12 420.816 157.26C421.469 157.47 423.195 160.573 423.335 161.739C423.382 162.299 424.082 163.325 425.131 164.375C425.481 164.725 425.761 165.098 425.761 165.215C425.761 165.308 426.694 166.311 427.837 167.431C429.89 169.46 429.937 169.484 430.66 169.227C431.243 169.04 431.476 169.04 431.733 169.32C432.013 169.6 432.036 169.787 431.803 170.277C431.5 170.93 431.126 171.116 431.126 170.627C431.126 170.277 430.823 170.603 430.753 171.023C430.707 171.28 430.94 171.373 431.593 171.373C432.386 171.373 433.226 171.886 433.039 172.283C433.016 172.353 433.366 172.493 433.856 172.609C434.346 172.703 434.952 172.866 435.209 172.983C436.025 173.356 436.445 173.449 436.585 173.286C436.865 173.006 438.684 173.286 439.547 173.706C440.434 174.172 443.933 174.709 444.539 174.499C444.726 174.406 444.889 174.476 444.889 174.639C444.889 174.779 445.029 174.872 445.192 174.802C445.332 174.755 445.799 174.849 446.219 175.035C447.245 175.455 447.502 175.432 447.339 174.895C447.222 174.522 447.292 174.476 447.688 174.569C447.968 174.639 448.155 174.895 448.155 175.199C448.178 175.689 448.178 175.689 448.505 175.245C448.692 174.989 448.855 174.895 448.855 175.059C448.855 175.199 449.251 175.385 449.741 175.455C450.208 175.549 450.838 175.689 451.094 175.805C451.421 175.922 452.027 175.898 452.727 175.689C453.357 175.525 453.73 175.362 453.543 175.362C453.357 175.339 453.287 175.269 453.357 175.199C453.45 175.105 453.637 175.129 453.8 175.269C454.267 175.619 455.083 175.572 455.41 175.199C455.62 174.942 455.76 174.942 455.993 175.129C456.529 175.572 459.795 175.595 461.218 175.175C461.918 174.965 462.641 174.825 462.804 174.849C462.991 174.872 463.247 174.802 463.411 174.662C463.551 174.546 463.831 174.522 463.994 174.616C464.297 174.802 465.79 174.592 466.047 174.312C466.42 173.939 465.813 173.869 463.667 174.009C463.037 174.056 462.431 173.962 462.314 173.846C462.198 173.682 461.964 173.822 461.661 174.242C461.405 174.592 461.078 174.802 460.938 174.732C460.635 174.546 459.282 174.359 457.882 174.312C457.112 174.312 456.693 174.406 456.553 174.639C456.389 174.895 456.319 174.919 456.319 174.685C456.319 174.476 456.156 174.429 455.876 174.522C455.62 174.592 455.363 174.546 455.27 174.406C455.2 174.289 455.246 174.172 455.386 174.172C455.526 174.172 455.573 174.056 455.48 173.916C455.363 173.729 455.293 173.729 455.176 173.916C454.99 174.196 454.127 174.149 453.847 173.869C453.543 173.566 452.564 173.706 452.424 174.079C452.354 174.266 452.167 174.382 451.981 174.312C451.817 174.242 451.654 174.312 451.584 174.476C451.537 174.662 451.374 174.616 451.094 174.336C450.581 173.822 450.208 173.916 449.904 174.592C449.788 174.872 449.555 175.105 449.391 175.105C449.018 175.105 448.995 174.732 449.345 174.522C449.695 174.289 448.692 173.636 448.272 173.799C447.782 173.986 447.222 173.729 447.222 173.333C447.222 172.983 447.175 172.959 445.589 172.913C444.819 172.889 444.539 172.796 444.469 172.493C444.423 172.19 444.516 172.12 444.866 172.19C445.519 172.306 446.802 171.863 446.685 171.536C446.639 171.396 445.962 171.326 444.959 171.35C443.84 171.396 443.07 171.28 442.603 171.047C442.23 170.86 441.857 170.767 441.763 170.837C441.693 170.93 441.484 170.79 441.297 170.557C440.994 170.137 440.97 170.16 440.597 170.743C440.294 171.186 440.037 171.326 439.571 171.256C439.221 171.233 438.521 171.186 438.008 171.163C437.285 171.14 437.005 171.023 436.842 170.627C436.702 170.347 436.375 170.09 436.072 170.043C435.768 169.997 435.209 169.81 434.789 169.6C434.346 169.344 434.042 169.297 433.972 169.46C433.926 169.624 433.646 169.624 433.179 169.484C432.783 169.39 432.363 169.344 432.246 169.414C432.13 169.507 432.083 169.227 432.176 168.76C432.246 168.341 432.339 167.198 432.409 166.241C432.526 163.862 432.689 163.395 433.249 163.699C433.762 163.955 434.625 163.605 434.625 163.139C434.625 162.952 434.556 162.952 434.416 163.185C434.276 163.395 434.159 163.419 434.066 163.232C433.972 163.092 433.949 162.882 434.019 162.766C434.089 162.649 433.949 162.509 433.692 162.439C433.226 162.322 433.133 162.182 432.899 161.039C432.713 160.27 432.269 160.246 431.5 160.946C430.87 161.529 430.917 161.809 431.64 161.809C431.873 161.809 432.06 161.926 432.06 162.066C432.06 162.229 431.85 162.276 431.546 162.182C431.243 162.112 430.94 162.042 430.847 162.042C430.73 162.042 430.66 161.926 430.66 161.786C430.66 161.646 430.497 161.389 430.31 161.203C430.123 160.993 429.96 160.526 429.96 160.153C429.96 159.593 429.867 159.476 429.424 159.476C429.12 159.476 428.677 159.336 428.444 159.15C428.047 158.87 428.024 158.777 428.327 158.427C428.794 157.867 428.864 157.027 428.467 156.701C428.281 156.537 428.187 156.234 428.257 156.047C428.374 155.767 428.304 155.698 428.024 155.814C427.767 155.907 427.627 155.837 427.627 155.628C427.627 155.441 427.371 155.231 427.044 155.138C426.718 155.068 426.461 154.858 426.461 154.671C426.461 154.368 426.344 154.298 425.528 154.251C425.341 154.228 425.178 154.018 425.178 153.761C425.178 153.481 424.992 153.272 424.712 153.225C424.455 153.178 424.128 153.038 423.988 152.922C423.872 152.828 423.545 152.782 423.265 152.852C422.915 152.945 422.729 152.852 422.612 152.502C422.542 152.222 422.356 152.012 422.239 152.012C422.122 152.012 422.029 151.895 422.029 151.755C422.029 151.452 421.679 151.102 420.629 150.379C420.186 150.076 419.673 149.679 419.51 149.516C419.346 149.353 419.09 149.213 418.973 149.213C418.857 149.213 418.763 149.049 418.763 148.863C418.763 148.653 418.647 148.373 418.53 148.21C418.39 148.046 418.087 147.673 417.83 147.37C417.457 146.927 417.154 146.833 416.221 146.833C415.404 146.833 414.891 146.693 414.541 146.39C413.981 145.9 413.491 145.83 413.282 146.18ZM453.52 174.779C453.52 174.825 453.357 174.872 453.17 174.872C452.984 174.872 452.82 174.779 452.82 174.639C452.82 174.522 452.984 174.476 453.17 174.546C453.357 174.639 453.52 174.732 453.52 174.779Z" />
        <path d="M373.042 149.656C373.042 150.053 373.136 150.403 373.276 150.496C373.392 150.566 373.439 150.776 373.369 150.939C373.299 151.126 373.346 151.336 373.486 151.429C373.626 151.499 373.742 151.709 373.742 151.849C373.742 152.012 373.649 151.989 373.486 151.779C373.276 151.499 373.252 151.499 373.369 151.849C373.439 152.082 373.509 152.479 373.509 152.735C373.509 153.272 374.046 153.318 374.442 152.805C374.699 152.502 374.675 152.432 374.255 152.315C373.976 152.245 373.812 152.082 373.882 151.965C374.116 151.592 373.742 150.123 373.462 150.286C373.299 150.403 373.252 150.193 373.322 149.703C373.369 149.306 373.346 148.98 373.229 148.98C373.136 148.98 373.042 149.283 373.042 149.656ZM374.209 152.712C374.209 152.829 374.116 152.945 373.976 152.945C373.859 152.945 373.742 152.829 373.742 152.712C373.742 152.572 373.859 152.479 373.976 152.479C374.116 152.479 374.209 152.572 374.209 152.712Z" />
        <path d="M471.248 153.551C471.061 153.878 470.921 154.555 470.898 155.068C470.898 155.558 470.805 155.977 470.665 155.977C470.548 155.977 470.595 156.094 470.781 156.234C471.015 156.397 471.038 156.514 470.875 156.584C470.711 156.654 470.665 156.934 470.735 157.19C470.851 157.68 470.875 157.68 471.038 157.237C471.178 156.864 471.225 156.911 471.225 157.47C471.248 158.03 471.178 158.1 470.781 158.007C470.222 157.867 469.988 158.147 470.198 158.753C470.362 159.336 471.715 159.57 471.715 159.033C471.738 158.73 471.761 158.73 471.971 159.08C472.438 159.873 472.578 159.243 472.158 158.357C471.551 157.144 471.738 156.631 472.671 156.957C473.604 157.284 473.721 157.097 473.021 156.421C472.461 155.884 472.461 155.861 472.904 155.558C473.324 155.254 473.534 154.111 473.161 154.111C473.067 154.111 472.741 154.275 472.461 154.485C471.948 154.834 471.948 154.811 471.948 153.901C471.948 152.805 471.691 152.688 471.248 153.551Z" />
        <path d="M465.557 154.811C465.02 155.371 464.507 155.721 464.274 155.651C464.064 155.581 463.084 155.465 462.104 155.371C461.101 155.301 460.285 155.138 460.285 155.045C460.285 154.811 459.445 154.601 459.188 154.788C459.072 154.858 458.908 155.115 458.815 155.348C458.722 155.581 458.535 155.698 458.419 155.628C458.092 155.418 457.252 156.374 457.252 156.957C457.252 157.191 457.159 157.377 457.066 157.377C456.949 157.377 456.902 157.681 456.972 158.054C457.019 158.45 456.902 158.964 456.692 159.29C456.482 159.57 456.319 160.153 456.319 160.55C456.319 161.343 455.713 162.742 455.363 162.742C455.246 162.742 455.176 163.022 455.246 163.372C455.293 163.746 455.363 164.189 455.363 164.399C455.386 164.655 455.549 164.749 455.923 164.679C456.389 164.585 456.459 164.679 456.506 165.448C456.552 165.938 456.459 166.848 456.296 167.478C455.969 168.831 456.156 169.087 457.276 168.901C458.022 168.761 458.092 168.691 458.115 167.921C458.162 164.259 458.302 163.209 458.722 163.069C459.352 162.859 459.492 163.139 459.165 163.955C458.908 164.632 458.955 164.749 459.562 165.285C460.051 165.752 460.168 166.008 460.051 166.358C459.865 166.848 460.168 167.408 460.611 167.408C460.751 167.408 461.055 167.198 461.288 166.941C461.498 166.685 461.801 166.475 461.941 166.475C462.128 166.475 462.104 166.615 461.871 166.871C461.288 167.548 461.218 168.271 461.708 168.714C462.128 169.087 462.174 169.087 462.617 168.691C462.897 168.434 463.037 168.108 462.967 167.851C462.897 167.641 462.921 167.385 463.014 167.338C463.107 167.268 463.201 166.825 463.201 166.311C463.177 165.588 463.107 165.448 462.757 165.495C462.244 165.588 461.708 164.912 461.848 164.352C461.918 164.119 461.638 163.396 461.241 162.719C460.658 161.786 460.565 161.46 460.775 161.203C460.915 161.016 461.195 160.876 461.358 160.876C461.521 160.876 461.918 160.55 462.198 160.153C462.594 159.617 462.851 159.477 463.247 159.593C463.667 159.687 463.784 159.617 463.784 159.243C463.784 158.614 463.201 158.404 462.477 158.777C462.174 158.94 461.544 159.104 461.078 159.104C460.495 159.127 460.098 159.313 459.795 159.687C459.375 160.2 458.652 160.223 458.652 159.71C458.652 159.593 458.442 159.267 458.162 158.987C457.789 158.614 457.695 158.334 457.812 157.774C458.045 156.584 458.255 156.514 461.218 156.724C465.183 156.981 465.883 156.957 465.883 156.421C465.883 156.281 466.14 155.838 466.466 155.418C466.816 154.951 467.003 154.485 466.91 154.252C466.7 153.715 466.583 153.762 465.557 154.811Z" />
        <path d="M472.765 163.163C472.391 163.209 471.995 163.373 471.901 163.513C471.761 163.723 471.715 163.723 471.715 163.513C471.715 163.303 471.552 163.349 471.248 163.606C470.688 164.072 470.642 164.492 471.132 164.282C471.388 164.189 471.482 164.306 471.482 164.702C471.482 165.239 471.528 165.262 472.298 165.006C472.741 164.866 473.114 164.609 473.114 164.446C473.114 164.189 473.208 164.189 473.581 164.376C473.861 164.516 474.257 164.562 474.491 164.492C474.747 164.422 475.284 164.562 475.727 164.842C476.707 165.449 476.847 165.425 476.847 164.772C476.847 163.583 475.097 162.906 472.765 163.163Z" />
        <path d="M469.125 174.265C469.079 174.312 468.402 174.429 467.632 174.522C466.653 174.615 466.116 174.802 465.883 175.105C465.696 175.362 465.043 175.758 464.437 176.038C463.76 176.342 463.294 176.715 463.177 177.041C463.107 177.321 462.874 177.718 462.687 177.951C462.431 178.231 462.431 178.371 462.617 178.488C463.107 178.791 464.647 178.091 465.486 177.228C465.953 176.738 466.746 176.225 467.329 176.038C468.635 175.618 470.128 174.685 469.965 174.405C469.825 174.172 469.312 174.102 469.125 174.265Z" />
      </Continent>
      <Continent
        className={clsx('africa', selectedTimezone === timezones.africa && 'selected')}
        fill={getMapBaseColor(28)}
        onClick={(event) => onClickContinent(event, timezones.africa)}
      >
        <path d="M260.073 82.4435C259.28 82.8867 258.114 82.9334 257.531 82.5602C257.391 82.4668 257.181 82.5368 257.088 82.6768C256.971 82.8867 256.761 82.8867 256.271 82.6535C255.758 82.4202 255.595 82.4202 255.501 82.6535C255.455 82.7934 255.245 82.9334 255.011 82.9334C254.802 82.9334 254.475 83.0733 254.335 83.2133C254.102 83.4466 253.938 83.4466 253.589 83.2133C252.959 82.8401 252.306 82.8634 249.156 83.3766C246.521 83.8198 244.958 84.3563 244.958 84.8462C244.958 84.9628 244.794 85.0328 244.608 84.9628C244.421 84.8695 244.211 84.9162 244.141 85.0328C244.048 85.1727 243.838 85.2661 243.651 85.2661C243.465 85.2661 243.045 85.5226 242.718 85.8492C242.392 86.1758 242.018 86.4324 241.855 86.4557C241.272 86.5024 240.409 86.3857 240.222 86.2224C240.129 86.1291 239.453 86.1291 238.706 86.2224C237.236 86.4324 236.373 86.0825 236.163 85.2427C236.023 84.7762 235.697 84.6829 235.254 84.9861C235.114 85.0794 234.741 85.7326 234.437 86.4324C233.644 88.3219 233.108 88.9517 231.661 89.6281C230.262 90.3046 228.862 91.5409 228.862 92.1241C228.862 92.334 228.629 92.8472 228.349 93.2205C227.929 93.7803 227.812 94.2468 227.812 95.4132C227.812 96.2996 227.696 97.0227 227.533 97.186C227.369 97.3493 227.229 97.6059 227.229 97.7458C227.229 97.8625 226.856 98.3057 226.39 98.6789C225.923 99.0755 225.34 99.5653 225.083 99.7986C224.547 100.288 223.217 100.895 222.681 100.895C222.167 100.895 221.841 101.268 221.234 102.574C220.931 103.204 220.605 103.694 220.511 103.694C219.881 103.694 218.598 105.607 218.598 106.563C218.598 107.123 217.992 108.056 217.059 108.943C216.779 109.199 216.312 110.039 216.009 110.809C215.729 111.579 215.309 112.348 215.076 112.488C214.866 112.652 214.61 113.165 214.516 113.655C214.446 114.145 214.306 114.798 214.236 115.101C214.143 115.498 214.19 115.614 214.423 115.544C214.586 115.498 214.983 115.707 215.263 116.011C215.799 116.594 215.962 117.224 215.589 117.224C215.193 117.224 215.076 118.39 215.449 118.927C215.939 119.626 215.893 121.632 215.333 123.055C215.076 123.709 214.866 124.502 214.866 124.852C214.866 125.551 213.723 127.487 213.303 127.487C213.14 127.487 213 127.581 213 127.721C213 127.837 213.163 127.954 213.35 127.954C213.56 127.954 213.816 128.234 213.956 128.584C214.096 128.957 214.306 129.424 214.423 129.634C214.54 129.867 214.54 130.077 214.4 130.17C214.26 130.24 214.166 130.8 214.166 131.406C214.19 132.316 214.306 132.619 214.866 133.179C215.636 133.926 215.776 134.322 215.193 134.089C214.889 133.972 214.843 134.042 214.983 134.509C215.239 135.325 215.403 135.419 215.916 135.022C216.242 134.765 216.289 134.625 216.102 134.555C215.753 134.439 215.683 133.786 216.009 133.786C216.126 133.786 216.336 134.089 216.476 134.485C216.592 134.859 216.826 135.185 216.942 135.185C217.082 135.185 217.199 135.349 217.199 135.535C217.199 135.722 217.362 135.885 217.549 135.885C217.735 135.885 217.899 135.955 217.899 136.072C217.899 136.165 218.435 136.795 219.065 137.495C219.811 138.311 220.231 138.964 220.255 139.361C220.278 140.34 220.418 140.854 220.721 141.017C220.861 141.11 220.908 141.32 220.838 141.46C220.744 141.6 220.768 141.717 220.908 141.717C221.048 141.717 221.094 141.857 221.048 141.997C220.978 142.183 221.328 142.463 221.911 142.72C222.891 143.116 223.707 143.77 226.506 146.289C229.329 148.831 230.029 149.041 232.151 148.015C234.391 146.942 237.913 146.849 239.872 147.805C240.129 147.922 240.526 147.875 240.922 147.665C243.068 146.592 244.678 145.916 245.144 145.916C245.424 145.916 245.727 145.752 245.797 145.589C245.961 145.146 247.85 144.749 250.02 144.703C251.396 144.656 251.769 144.726 252.189 145.099C252.772 145.636 253.589 147.129 253.589 147.642C253.589 147.828 253.822 148.225 254.125 148.482C254.522 148.878 254.825 148.948 255.455 148.831C255.898 148.761 256.691 148.645 257.204 148.598C257.717 148.528 258.464 148.458 258.884 148.412C259.49 148.318 259.653 148.412 259.747 148.855C259.817 149.135 260.097 149.461 260.353 149.531C260.61 149.601 260.82 149.811 260.82 149.974C260.82 150.161 260.96 150.418 261.146 150.558C261.59 150.931 261.38 153.194 260.796 154.36C260.493 154.943 260.447 155.246 260.633 155.293C260.796 155.363 260.843 155.55 260.703 155.853C260.61 156.086 260.47 156.763 260.423 157.346C260.307 158.349 260.003 158.839 259.607 158.605C259.304 158.419 259.397 159.072 259.747 159.818C259.933 160.192 260.26 160.915 260.493 161.428C260.727 161.941 261.31 162.734 261.753 163.177C263.409 164.787 265.485 167.866 265.485 168.729C265.485 169.009 265.602 169.289 265.718 169.359C265.858 169.452 265.905 169.592 265.835 169.709C265.765 169.825 265.952 170.479 266.278 171.155C266.582 171.832 267.025 172.951 267.281 173.628C267.701 174.841 267.701 174.887 267.258 175.191C266.838 175.517 266.838 175.587 267.585 177.453C268.098 178.76 268.331 179.739 268.308 180.439C268.261 181.652 267.888 182.772 267.491 182.772C267.048 182.795 265.975 184.731 265.812 185.828C265.625 187.227 265.019 189.303 264.832 189.303C264.529 189.303 264.669 193.595 264.995 194.365C265.159 194.785 265.392 195.135 265.462 195.135C265.695 195.135 266.698 197.211 267.701 199.684C267.911 200.197 268.354 201.083 268.704 201.667C269.194 202.506 269.334 203.043 269.357 204.349C269.381 205.235 269.497 206.495 269.637 207.148C269.777 207.778 270.057 209.248 270.267 210.391C270.71 212.793 271.037 213.54 272.25 214.869C272.81 215.476 273.3 216.386 273.65 217.435C274.209 219.162 274.979 220.841 275.306 221.051C275.422 221.121 275.586 221.658 275.656 222.264C275.796 223.174 275.749 223.36 275.446 223.36C275.259 223.36 275.049 223.5 274.979 223.64C274.886 223.967 275.259 224.76 275.609 225.017C275.726 225.11 275.772 225.436 275.702 225.763C275.539 226.556 275.889 227.116 276.215 226.626C276.355 226.393 276.449 226.37 276.449 226.556C276.449 226.719 276.589 226.859 276.752 226.859C276.939 226.859 277.265 227.069 277.498 227.326C278.012 227.909 278.478 227.909 279.061 227.396C279.341 227.139 279.854 226.999 280.461 227.023C280.997 227.046 281.651 226.906 281.977 226.696C282.654 226.253 284.333 226.136 285.616 226.44C286.223 226.579 286.619 226.579 286.829 226.393C287.016 226.253 287.389 226.16 287.692 226.206C287.995 226.253 288.205 226.16 288.182 226.02C288.159 225.856 288.579 225.693 289.138 225.646C290.375 225.506 292.731 223.967 294.06 222.474C294.317 222.171 294.924 221.541 295.437 221.074C296.3 220.235 297.909 217.949 297.909 217.552C297.909 217.249 299.472 215.663 299.775 215.663C300.032 215.663 300.452 214.799 301.035 212.98C301.455 211.674 301.548 209.947 301.175 210.181C300.755 210.437 300.942 209.878 301.478 209.248C301.758 208.921 302.738 208.361 303.624 207.988C304.814 207.522 305.397 207.125 305.794 206.565C306.12 206.075 306.237 205.702 306.097 205.562C305.957 205.422 305.957 205.119 306.12 204.722C306.424 203.976 306.377 202.11 306.074 201.806C305.934 201.667 305.84 201.457 305.84 201.34C305.84 201.2 305.957 201.247 306.074 201.433C306.26 201.713 306.307 201.69 306.307 201.247C306.307 200.967 306.19 200.757 306.027 200.78C305.864 200.827 305.747 200.523 305.724 200.034C305.7 199.59 305.561 199.194 305.397 199.147C305.257 199.101 305.141 198.821 305.141 198.541C305.141 197.934 307.24 195.602 307.8 195.602C308.01 195.602 308.173 195.508 308.173 195.415C308.173 195.322 308.36 194.972 308.593 194.668C309.433 193.525 310.296 192.849 311.602 192.312C312.349 192.033 313.002 191.706 313.072 191.636C313.142 191.543 313.352 191.379 313.562 191.263C314.168 190.936 315.731 188.813 316.058 187.857C316.291 187.181 316.291 186.761 316.104 185.991C315.941 185.431 315.848 183.682 315.871 181.955C315.941 178.06 315.964 178.223 315.334 177.943C314.891 177.733 314.681 177.337 314.401 176.077C313.888 174.048 313.911 173.744 314.471 173.604C314.728 173.534 314.938 173.348 314.938 173.208C314.938 173.045 314.775 172.975 314.588 173.068C314.121 173.231 313.842 172.578 314.168 172.065C314.355 171.762 314.261 171.482 313.748 170.852C313.375 170.432 313.072 169.895 313.072 169.709C313.072 169.219 313.842 167.05 314.075 166.886C314.308 166.723 315.381 164.25 315.568 163.457C315.638 163.177 315.918 162.827 316.174 162.688C316.897 162.291 318.11 160.985 318.577 160.122C319.58 158.139 322.729 154.756 324.642 153.543C325.901 152.75 329.61 148.691 329.634 148.085C329.634 147.992 330.007 147.315 330.45 146.615C330.893 145.916 331.267 145.169 331.267 144.936C331.267 144.469 332.69 141.414 332.969 141.297C333.063 141.25 333.133 141.11 333.133 140.97C333.133 140.854 333.389 140.224 333.716 139.571C334.042 138.941 334.299 138.054 334.299 137.588C334.299 136.981 334.416 136.748 334.766 136.655C335.255 136.515 335.395 136.212 334.999 136.072C334.789 136.002 334.672 134.765 334.812 133.926C334.929 133.272 333.949 132.993 333.436 133.529C332.923 134.042 330.917 134.742 330.427 134.555C330.24 134.485 329.82 134.555 329.517 134.719C329.214 134.882 328.724 134.952 328.444 134.882C328.141 134.789 327.698 134.929 327.301 135.232C326.835 135.559 326.415 135.675 325.878 135.605C325.365 135.535 324.852 135.628 324.385 135.908C323.172 136.655 322.286 136.352 321.353 134.882C321.119 134.532 320.7 134.182 320.373 134.112L319.813 133.972L320.35 133.692C321.236 133.249 321.143 132.573 320.023 131.406C319.463 130.823 318.857 130.123 318.67 129.843C318.484 129.587 318.227 129.354 318.087 129.354C317.97 129.354 317.667 129.027 317.457 128.654C317.224 128.257 316.874 127.884 316.641 127.791C316.407 127.697 316.081 127.487 315.918 127.324C315.754 127.184 315.474 127.091 315.311 127.161C315.101 127.231 314.961 127.068 314.891 126.694C314.798 126.274 314.868 126.088 315.101 126.088C315.381 126.088 315.381 126.041 315.078 125.738C314.635 125.271 314.541 125.295 314.378 125.901C314.238 126.438 314.215 126.438 313.888 125.971C313.678 125.715 313.422 125.085 313.305 124.572C312.652 121.889 312.325 121.213 311.182 120.326C309.876 119.3 309.549 118.553 309.479 116.291C309.433 115.404 309.269 114.518 309.13 114.331C308.966 114.121 308.803 113.701 308.78 113.375C308.78 113.048 308.71 112.792 308.64 112.792C308.57 112.792 308.08 112.395 307.52 111.905C306.424 110.925 305.77 109.293 306.494 109.293C306.75 109.293 306.68 109.129 306.237 108.733C305.91 108.43 305.351 107.566 304.954 106.82C304.581 106.05 304.044 105.047 303.764 104.557C303.508 104.067 303.274 103.554 303.274 103.391C303.274 103.228 303.088 102.831 302.878 102.528C302.645 102.201 302.435 101.758 302.411 101.525C302.365 101.268 301.968 100.662 301.525 100.125C301.082 99.612 300.709 98.9355 300.709 98.6556C300.709 98.3524 300.615 98.0724 300.499 97.9791C300.359 97.9091 300.195 97.116 300.102 96.2296C300.032 95.3432 299.822 94.4335 299.659 94.2002C299.052 93.3838 296.276 93.6637 295.25 94.6434C294.97 94.9 294.574 95.0399 294.294 94.97C294.037 94.9233 293.361 94.7367 292.777 94.6201C292.194 94.4801 291.518 94.2702 291.261 94.1535C290.421 93.7803 288.859 93.4771 288.275 93.5937C287.832 93.687 287.646 93.5937 287.529 93.2205C287.412 92.8472 287.179 92.7306 286.643 92.7306C285.383 92.7306 284.217 92.2874 284.077 91.7975C284.007 91.5176 283.75 91.331 283.493 91.331C283.214 91.331 282.934 91.2144 282.864 91.0977C282.514 90.5379 280.298 91.0511 279.248 91.9375C278.525 92.5673 278.315 93.617 278.758 94.3401C279.131 94.9466 278.198 96.2063 277.358 96.2296C277.172 96.2296 276.589 95.9264 276.076 95.5765C275.119 94.8767 273.65 94.3635 272.693 94.3635C271.877 94.3635 271.154 93.7803 270.967 92.9406C270.85 92.474 270.594 92.1474 270.197 91.9841C269.847 91.8442 269.264 91.5876 268.891 91.3777C268.471 91.1677 267.678 91.0278 266.978 91.0511C266.208 91.0744 265.439 90.9111 264.809 90.6312C263.876 90.1647 263.246 89.4415 263.549 89.1383C263.619 89.0683 263.526 88.9983 263.316 88.9983C263.106 88.9983 262.919 89.0916 262.919 89.2316C262.919 89.6281 262.243 89.4882 261.986 89.0217C261.8 88.6484 261.893 88.4618 262.453 87.972C262.849 87.6454 263.316 87.4354 263.502 87.5054C263.689 87.5987 263.852 87.5287 263.852 87.3654C263.852 87.2255 263.736 87.1788 263.619 87.2488C263.456 87.3421 263.432 87.0855 263.526 86.479C263.642 85.6859 263.596 85.4993 263.176 85.2894C262.593 84.9628 262.546 84.4496 263.059 83.9364C263.246 83.7265 263.362 83.4232 263.292 83.2366C263.222 83.05 263.269 82.98 263.386 83.05C263.526 83.12 263.619 83.0967 263.619 82.98C263.619 82.5368 263.292 82.4902 262.873 82.8634C262.453 83.2366 262.429 83.2366 262.313 82.7701C262.196 82.3735 261.636 82.0236 261.053 82.0003C260.983 82.0003 260.54 82.2103 260.073 82.4435ZM304.581 157.766C304.767 158.022 304.791 158.255 304.627 158.442C304.511 158.605 304.464 158.815 304.534 158.932C304.674 159.188 304.184 160.378 303.928 160.378C303.811 160.378 303.741 160.495 303.741 160.635C303.741 160.798 303.554 161.055 303.344 161.218C302.948 161.521 302.948 161.568 303.344 161.871C303.928 162.338 303.578 162.478 302.341 162.244C301.712 162.128 301.175 162.151 301.012 162.268C300.569 162.641 300.452 162.128 300.592 160.448C300.825 157.952 300.849 157.882 301.805 157.579C302.295 157.416 302.785 157.206 302.901 157.112C303.158 156.856 304.208 157.252 304.581 157.766ZM296.51 166.723C296.51 167.073 296.626 167.423 296.766 167.493C296.906 167.586 296.953 167.819 296.883 168.006C296.813 168.193 296.86 168.566 297 168.846C297.21 169.219 297.21 169.359 297 169.429C296.533 169.592 296.72 170.409 297.303 170.619C297.863 170.852 298.143 171.318 298.143 172.065C298.143 172.321 298.399 173.045 298.726 173.674C299.449 175.097 299.449 175.121 298.726 174.794C298.353 174.631 298.166 174.398 298.259 174.188C298.329 174.024 298.213 173.651 298.026 173.371C297.839 173.091 297.676 172.718 297.676 172.555C297.676 172.368 297.373 171.855 296.976 171.412C296.603 170.992 296.253 170.455 296.23 170.222C296.23 170.012 296.183 169.499 296.137 169.126C296.09 168.729 296.02 167.586 295.973 166.56C295.903 164.88 295.927 164.764 296.183 165.393C296.346 165.767 296.486 166.373 296.51 166.723ZM304.907 177.407C305.047 177.663 305.187 178.363 305.234 178.923C305.257 179.506 305.421 180.113 305.561 180.253C305.91 180.602 305.934 181.536 305.607 181.722C305.281 181.932 305.467 185.174 305.817 185.291C305.957 185.338 306.074 185.478 306.074 185.594C306.074 185.734 305.98 185.758 305.84 185.688C305.724 185.618 305.607 185.664 305.607 185.781C305.607 185.921 305.491 186.038 305.351 186.038C305.211 186.038 305.164 185.828 305.257 185.571C305.351 185.291 305.304 184.965 305.187 184.801C304.954 184.498 304.604 182.702 304.651 181.955C304.814 179.18 304.767 178.386 304.511 177.873C304.348 177.547 304.208 177.22 304.208 177.103C304.208 176.754 304.674 176.963 304.907 177.407Z" />
        <path d="M330.8 182.12C330.613 182.423 330.45 183.006 330.403 183.449C330.333 184.429 330.193 184.569 329.634 184.219C329.26 183.986 329.167 183.986 329.167 184.266C329.167 184.452 329.307 184.662 329.47 184.732C329.61 184.802 329.517 184.802 329.237 184.732C328.7 184.616 328.607 184.755 328.677 185.619C328.7 185.829 328.607 186.062 328.467 186.155C328.351 186.225 328.234 186.505 328.234 186.738C328.234 187.042 328.117 187.135 327.884 187.065C327.674 186.972 327.534 187.065 327.534 187.275C327.534 187.741 325.878 188.908 324.618 189.328C324.105 189.491 323.592 189.724 323.475 189.817C323.335 189.911 323.009 190.004 322.705 190.004C322.286 190.004 322.169 190.121 322.169 190.634C322.169 191.007 321.959 191.59 321.726 191.963C321.259 192.663 321.259 193.223 321.726 195.906C322.006 197.469 322.006 197.515 321.329 198.822C320.956 199.545 320.559 200.221 320.466 200.291C319.65 200.874 319.346 202.997 319.93 204.093C320.116 204.467 320.21 205.12 320.14 205.726C320.07 206.333 320.116 206.799 320.279 206.893C320.419 206.986 320.536 207.242 320.536 207.476C320.536 207.989 321.236 208.665 321.772 208.665C321.982 208.665 322.169 208.759 322.169 208.899C322.169 209.249 323.382 209.155 323.895 208.782C324.152 208.595 324.642 208.432 324.991 208.432C325.691 208.432 326.601 207.499 326.601 206.823C326.601 206.613 326.741 206.239 326.928 205.983C327.091 205.726 327.371 204.933 327.534 204.233C327.721 203.534 328.047 202.577 328.257 202.111C328.491 201.644 328.817 200.758 328.957 200.128C329.12 199.498 329.634 197.982 330.123 196.722C330.613 195.486 331.08 194.063 331.15 193.573C331.43 191.893 331.5 191.637 331.803 191.707C332.083 191.754 332.106 191.544 331.826 189.771L331.663 188.721L332.129 189.281L332.596 189.864L332.993 189.118C333.272 188.558 333.319 188.255 333.133 187.928C333.016 187.671 332.899 186.995 332.899 186.435C332.899 185.875 332.783 185.059 332.643 184.616C332.316 183.589 331.406 181.63 331.266 181.606C331.196 181.606 330.986 181.84 330.8 182.12Z" />
      </Continent>
      <Continent
        className={clsx('australia', selectedTimezone === timezones.australia && 'selected')}
        fill={getMapBaseColor(32)}
        onClick={(event) => onClickContinent(event, timezones.australia)}
      >
        <path d="M478.338 158.423C477.895 158.68 477.405 158.983 477.242 159.1C477.055 159.263 476.915 159.263 476.822 159.123C476.658 158.866 475.772 158.796 475.772 159.03C475.772 159.426 476.798 160.522 477.172 160.522C477.825 160.522 478.338 160.779 478.338 161.106C478.338 161.782 479.131 162.109 480.507 162.015C481.487 161.945 481.837 161.969 481.72 162.155C481.627 162.319 481.324 162.365 481.02 162.295C480.694 162.202 480.367 162.295 480.204 162.505C480.041 162.715 479.714 162.809 479.504 162.739C479.038 162.599 478.338 162.809 478.338 163.112C478.338 163.228 478.595 163.438 478.921 163.555C479.248 163.672 479.504 163.928 479.504 164.138C479.504 164.325 479.621 164.488 479.761 164.488C479.877 164.488 479.924 164.605 479.854 164.721C479.644 165.071 479.971 165.911 480.227 165.748C480.367 165.678 480.624 165.748 480.811 165.911C480.997 166.051 481.23 166.121 481.3 166.028C481.394 165.934 481.347 165.794 481.207 165.701C481.044 165.584 480.997 165.398 481.137 165.211C481.3 164.931 481.464 164.955 482.187 165.281C482.653 165.514 483.237 165.864 483.47 166.051C483.703 166.261 484.216 166.471 484.589 166.494C485.499 166.611 488.952 168.267 488.928 168.57C488.905 168.687 489.045 169.013 489.208 169.27C489.395 169.527 489.535 169.9 489.535 170.133C489.535 170.366 489.605 170.553 489.721 170.553C489.815 170.553 489.885 170.926 489.908 171.369C489.908 171.999 489.838 172.186 489.511 172.186C489.068 172.186 487.902 173.679 487.902 174.239C487.902 174.565 490.001 174.565 490.258 174.215C490.328 174.122 490.748 173.982 491.191 173.935C491.937 173.842 492.077 173.912 492.87 174.962C493.734 176.081 493.757 176.105 495.017 176.105C495.74 176.081 496.369 176.151 496.463 176.221C496.533 176.315 496.789 176.268 497.023 176.151C497.373 175.965 497.419 175.988 497.349 176.385C497.303 176.641 497.186 176.968 497.116 177.084C497.046 177.224 497.069 177.318 497.186 177.318C497.466 177.318 497.699 176.758 497.699 176.128C497.699 175.895 497.816 175.685 497.932 175.685C498.072 175.685 498.166 175.522 498.166 175.312C498.166 175.125 498.259 175.008 498.376 175.078C498.492 175.148 498.586 174.962 498.586 174.659C498.586 173.725 500.288 172.512 500.825 173.072C500.918 173.166 501.385 173.352 501.851 173.469C502.364 173.609 502.808 173.912 502.948 174.239C503.064 174.519 503.251 174.752 503.344 174.752C503.461 174.752 503.531 174.938 503.531 175.195C503.531 175.428 503.694 175.755 503.881 175.918C504.067 176.081 504.161 176.291 504.091 176.408C504.021 176.525 504.137 176.618 504.324 176.618C504.534 176.618 504.744 176.875 504.837 177.178C504.93 177.598 505.234 177.808 506.05 178.041C506.657 178.204 507.473 178.321 507.846 178.297C508.243 178.297 508.779 178.437 509.059 178.624C509.666 178.997 511.369 179.067 511.579 178.717C511.672 178.577 511.649 178.484 511.532 178.507C511.439 178.531 511.182 178.531 510.995 178.507C510.715 178.484 510.715 178.437 510.949 178.297C511.159 178.158 510.949 177.994 510.296 177.784C509.456 177.504 509.013 177.084 509.596 177.084C510.132 177.084 509.782 176.641 509.199 176.571C508.663 176.525 508.569 176.408 508.616 175.941C508.686 175.428 508.639 175.405 508.01 175.522C507.38 175.662 507.31 175.592 507.006 174.682C506.82 174.145 506.283 173.329 505.817 172.862C505.327 172.396 504.93 171.789 504.93 171.533C504.93 171.113 505.07 171.02 505.747 171.02C506.377 171.02 506.563 170.926 506.563 170.6C506.563 170.11 505.793 169.317 505.257 169.247C505.047 169.223 504.72 169.037 504.511 168.85C504.277 168.664 503.834 168.454 503.507 168.36C502.994 168.244 502.924 168.127 503.018 167.637C503.064 167.311 503.204 167.054 503.321 167.054C503.437 167.054 503.531 166.891 503.531 166.681C503.531 166.354 503.461 166.354 503.181 166.587C502.878 166.844 502.738 166.797 502.364 166.401C502.108 166.121 501.898 165.794 501.898 165.654C501.898 165.514 501.805 165.444 501.688 165.538C501.571 165.608 501.315 165.491 501.128 165.304C500.918 165.118 500.568 164.955 500.312 164.955C500.055 164.955 499.729 164.791 499.565 164.605C499.425 164.418 498.702 164.115 497.979 163.905C497.233 163.695 496.23 163.275 495.716 162.995C495.203 162.692 494.643 162.505 494.503 162.552C494.34 162.622 494.2 162.552 494.2 162.435C494.2 162.319 493.757 162.155 493.244 162.062C492.707 161.969 491.564 161.526 490.701 161.059C489.232 160.289 489.068 160.243 488.415 160.546C488.019 160.709 487.645 161.036 487.599 161.269C487.529 161.526 487.249 161.689 486.876 161.689C486.549 161.689 486.269 161.782 486.269 161.875C486.269 161.992 486.082 162.295 485.872 162.575C485.639 162.832 485.289 163.275 485.103 163.532C484.916 163.812 484.543 164.022 484.263 164.022C483.866 164.022 483.703 163.812 483.47 162.902C483.307 162.295 483.12 161.945 483.073 162.109C482.91 162.529 482.537 162.202 482.537 161.619C482.537 161.199 482.583 161.152 482.793 161.456C483.003 161.735 483.027 161.712 482.887 161.339C482.793 161.082 482.7 160.476 482.653 159.986C482.537 158.913 482.35 158.586 481.954 158.75C481.79 158.796 481.277 158.656 480.811 158.4C479.784 157.863 479.388 157.863 478.338 158.423Z" />
        <path d="M495.647 179.231C495.53 179.394 495.343 179.79 495.25 180.117C495.157 180.444 494.877 181.05 494.597 181.47C494.34 181.913 494.107 182.683 494.06 183.196C494.014 183.709 493.874 184.269 493.734 184.432C493.617 184.596 493.454 185.389 493.384 186.205C493.244 187.978 492.451 190.544 491.844 191.221C491.588 191.477 491.401 191.804 491.401 191.944C491.401 192.41 490.678 192.947 490.095 192.947C489.722 192.947 489.185 192.597 488.602 192.014C488.089 191.5 487.482 191.081 487.249 191.081C487.016 191.081 486.666 190.824 486.502 190.497C486.339 190.171 486.059 189.914 485.896 189.914C485.709 189.914 485.523 189.658 485.429 189.331C485.266 188.701 484.87 188.538 484.87 189.098C484.87 189.331 484.45 189.051 483.727 188.375L482.584 187.278L483.05 186.788C483.307 186.509 483.61 186.065 483.727 185.762C483.843 185.459 484.053 185.296 484.193 185.365C484.333 185.459 484.38 185.669 484.31 185.832C484.193 186.135 485.033 186.485 485.383 186.275C485.569 186.159 485.476 185.062 485.243 184.829C485.173 184.759 484.963 184.852 484.753 185.016C484.45 185.272 484.403 185.226 484.403 184.689C484.403 184.362 484.52 184.106 484.66 184.129C484.8 184.176 485.173 183.733 485.499 183.149C485.966 182.286 485.989 182.076 485.733 181.866C485.546 181.75 485.453 181.493 485.499 181.33C485.569 181.19 485.499 181.05 485.359 181.05C485.219 181.05 485.103 181.143 485.103 181.26C485.103 181.377 484.893 181.517 484.613 181.587C484.263 181.68 484.17 181.633 484.263 181.377C484.426 180.933 484.263 180.957 483.47 181.563C482.933 181.96 482.7 182.006 482.49 181.797C482.327 181.633 481.977 181.517 481.697 181.517C481.394 181.517 481.021 181.377 480.857 181.213C480.694 181.027 480.414 180.933 480.251 181.003C480.111 181.05 479.971 180.887 479.971 180.653C479.971 180.304 479.924 180.28 479.738 180.584C479.528 180.887 479.458 180.887 479.248 180.514C479.108 180.304 478.921 180.164 478.781 180.234C478.665 180.327 478.571 180.164 478.571 179.907C478.571 179.487 478.501 179.464 477.988 179.65C477.662 179.767 477.405 179.79 477.405 179.72C477.405 179.627 477.288 179.674 477.125 179.837C476.798 180.164 477.288 180.84 477.732 180.677C478.175 180.514 478.408 181.26 478.058 181.68C477.848 181.96 477.452 182.006 476.542 181.913C475.842 181.843 475.306 181.89 475.306 182.006C475.306 182.123 475.189 182.216 475.026 182.216C474.606 182.216 473.463 183.616 473.579 183.966C473.649 184.153 473.486 184.386 473.23 184.549C472.95 184.689 472.74 184.922 472.74 185.062C472.74 185.226 472.506 185.575 472.226 185.879C471.807 186.322 471.76 186.485 472.017 186.625C472.553 186.975 472.063 187.325 471.2 187.208C470.734 187.162 470.197 187.232 470.034 187.372C469.754 187.605 469.73 187.582 469.847 187.255C469.94 187.022 469.847 186.835 469.614 186.742C469.404 186.672 469.241 186.462 469.241 186.299C469.241 186.112 468.914 185.739 468.518 185.459C467.841 184.992 467.748 184.969 467.491 185.319C467.328 185.529 467.165 185.669 467.118 185.599C466.861 185.365 465.508 185.249 465.508 185.482C465.508 185.599 465.625 185.715 465.742 185.715C465.882 185.715 465.975 185.949 465.975 186.229C465.952 186.742 465.952 186.765 465.625 186.322C465.298 185.902 465.275 185.902 465.275 186.275C465.275 186.578 465.158 186.648 464.809 186.532C464.412 186.415 464.342 186.485 464.342 187.022C464.342 187.372 464.272 187.605 464.202 187.535C463.945 187.278 462.849 188.561 462.802 189.168C462.779 189.518 462.732 189.891 462.732 190.008C462.709 190.101 462.336 190.124 461.869 190.031C461.333 189.938 461.076 189.961 461.17 190.124C461.263 190.241 461.216 190.404 461.076 190.497C460.913 190.591 460.913 190.777 461.076 191.104C461.24 191.36 461.24 191.547 461.1 191.547C460.983 191.547 460.843 191.734 460.773 191.944C460.726 192.177 460.586 191.92 460.493 191.36C460.4 190.824 460.237 190.381 460.167 190.381C460.073 190.381 459.607 190.777 459.14 191.267C458.464 191.944 458.277 192.34 458.277 192.97C458.277 193.6 458.067 194.02 457.531 194.626C457.111 195.093 456.574 195.723 456.341 196.072C455.758 196.912 454.638 197.449 453.332 197.495C452.679 197.519 452.049 197.682 451.792 197.915C451.559 198.125 451.092 198.312 450.766 198.312C450.439 198.312 450.043 198.475 449.879 198.662C449.669 198.918 449.296 198.988 448.69 198.918C448.177 198.872 447.733 198.918 447.64 199.035C447.57 199.152 447.36 199.245 447.174 199.245C446.987 199.245 446.497 199.548 446.077 199.921C445.681 200.318 444.957 200.784 444.491 200.994C444.024 201.228 443.511 201.671 443.325 202.021L443.045 202.627L442.905 201.927C442.788 201.368 442.718 201.298 442.532 201.578C441.948 202.464 441.715 203.094 441.715 203.724C441.715 204.097 441.458 204.797 441.132 205.263C440.432 206.29 440.432 206.5 440.969 208.272C441.505 210.022 441.389 210.908 440.782 209.602L440.362 208.692L440.129 209.275L439.896 209.859L439.639 209.275C439.406 208.762 439.382 208.832 439.499 209.719C439.569 210.278 439.802 211.048 439.989 211.421C440.199 211.818 440.385 212.658 440.432 213.334C440.479 213.987 440.595 214.827 440.665 215.224C440.759 215.597 440.829 216.227 440.805 216.623C440.805 216.997 440.759 218.583 440.735 220.099C440.689 222.082 440.572 222.922 440.385 222.968C440.222 223.038 440.082 223.318 440.082 223.598C440.082 224.298 439.452 225.138 438.893 225.138C438.613 225.138 438.356 225.371 438.216 225.744C437.889 226.584 437.913 226.77 438.426 226.77C438.683 226.77 438.939 226.957 439.009 227.19C439.196 227.75 439.942 228.1 441.295 228.263C442.345 228.38 443.581 228.077 443.581 227.704C443.581 227.517 444.328 227.214 445.004 227.12C445.214 227.097 445.587 226.864 445.844 226.607C446.427 226.024 448.107 225.791 450.859 225.931L452.865 226.047L453.892 224.998C454.428 224.438 455.081 223.971 455.291 223.971C455.501 223.971 455.734 223.878 455.804 223.761C455.874 223.645 456.224 223.481 456.574 223.388C456.947 223.318 457.414 223.085 457.624 222.875C457.904 222.618 458.254 222.572 458.79 222.665C459.56 222.805 462.779 222.082 463.292 221.639C463.432 221.522 464.505 221.359 465.695 221.265C467.724 221.102 467.888 221.125 468.424 221.639C468.727 221.918 469.124 222.105 469.287 222.058C469.451 221.988 469.987 222.198 470.454 222.502C470.943 222.782 471.457 223.038 471.62 223.038C471.83 223.038 471.83 223.108 471.62 223.318C471.223 223.715 471.27 224.438 471.667 224.438C472.203 224.438 472.46 226.561 471.97 226.864C471.643 227.074 471.69 227.19 472.343 227.774C473.09 228.427 473.09 228.427 473.16 227.913C473.253 227.33 474.746 225.837 475.236 225.837C475.422 225.837 476.052 225.371 476.635 224.788C477.218 224.205 477.755 223.785 477.848 223.878C477.918 223.971 477.848 224.088 477.708 224.135C477.545 224.205 477.405 224.438 477.405 224.671C477.405 224.904 477.125 225.254 476.798 225.487C476.472 225.697 476.052 226.234 475.865 226.677C475.679 227.12 475.282 227.587 475.002 227.68C474.699 227.797 474.326 228.077 474.139 228.31C473.813 228.683 473.836 228.707 474.373 228.567C474.699 228.473 475.142 228.403 475.376 228.403C475.609 228.403 476.005 228.03 476.262 227.587C476.542 227.144 476.845 226.77 476.962 226.77C477.428 226.77 477.125 227.937 476.472 228.59C475.702 229.36 475.772 229.616 476.705 229.43C477.242 229.313 477.382 229.406 477.638 230.06C477.918 230.736 477.918 230.899 477.498 231.646C477.102 232.369 477.055 232.602 477.288 233.279C477.452 233.722 477.802 234.165 478.081 234.258C478.338 234.375 478.571 234.538 478.571 234.655C478.571 234.772 478.991 234.865 479.528 234.865C480.064 234.865 480.554 234.981 480.647 235.098C480.717 235.238 481.021 235.448 481.301 235.611C481.744 235.845 482 235.798 482.933 235.308C483.54 234.981 484.1 234.795 484.17 234.888C484.24 234.981 484.683 235.261 485.173 235.495C485.639 235.751 486.036 236.101 486.036 236.288C486.036 236.568 486.129 236.568 486.526 236.194C487.879 234.981 490.351 233.768 491.564 233.768C493.057 233.768 494.41 233.022 494.434 232.206C494.434 232.042 494.69 231.646 495.017 231.296C495.343 230.946 495.6 230.549 495.6 230.386C495.6 230.246 496.066 229.686 496.65 229.15C497.233 228.613 498.212 227.564 498.796 226.817C500.219 225.044 501.362 223.948 502.435 223.388C502.901 223.131 503.298 222.805 503.298 222.665C503.298 222.525 503.578 222.105 503.951 221.709C505.001 220.566 505.397 220.006 505.397 219.726C505.397 219.562 505.63 219.073 505.934 218.629C507.263 216.6 508.01 215.224 508.103 214.547C508.173 214.151 508.383 213.428 508.57 212.914C508.943 211.958 508.826 211.398 508.453 212.331C508.336 212.634 508.22 212.774 508.22 212.658C508.196 212.401 508.616 210.745 508.826 210.208C508.896 210.022 508.943 209.765 508.919 209.672C508.896 209.555 509.106 209.135 509.363 208.692C509.619 208.272 509.829 207.736 509.806 207.503C509.806 207.176 509.759 207.153 509.666 207.409C509.316 208.366 508.43 208.529 508.43 207.619C508.43 207.386 508.336 207.176 508.22 207.176C508.103 207.176 507.963 206.849 507.893 206.476C507.8 206.08 507.59 205.776 507.403 205.776C507.217 205.776 507.03 205.543 506.983 205.24C506.937 204.96 506.773 204.75 506.61 204.797C506.353 204.843 506.33 204.563 506.47 203.537C506.61 202.277 506.61 202.207 506.144 202.324C505.84 202.417 505.607 202.324 505.537 202.114C505.42 201.811 505.327 201.811 505.024 202.067C504.721 202.324 504.674 202.231 504.721 201.414C504.767 200.831 504.627 200.225 504.394 199.851C503.904 199.128 503.904 199.012 504.347 199.012C504.581 199.012 504.674 198.825 504.604 198.475C504.557 198.195 504.394 197.985 504.231 198.032C503.834 198.125 502.388 197.122 502.225 196.632C502.155 196.399 501.968 196.282 501.758 196.376C501.595 196.446 501.432 196.376 501.432 196.212C501.432 196.072 501.338 196.002 501.222 196.072C500.872 196.306 500.428 195.536 500.545 194.929C500.615 194.603 500.568 194.276 500.452 194.206C500.359 194.136 500.335 193.973 500.405 193.833C500.685 193.39 500.498 191.477 500.149 191.081C499.799 190.707 499.752 188.211 500.102 187.675C500.195 187.512 500.009 187.232 499.635 186.998C499.076 186.625 499.052 186.555 499.402 186.299C499.682 186.089 499.729 185.972 499.519 185.762C499.355 185.599 499.076 185.482 498.889 185.505C498.632 185.505 498.656 185.575 498.982 185.715C499.449 185.925 499.449 185.925 498.982 185.949C498.726 185.995 498.446 185.995 498.352 185.972C498.236 185.972 498.166 186.042 498.166 186.182C498.166 186.299 498.003 186.415 497.793 186.415C497.513 186.415 497.466 186.182 497.513 185.249C497.559 184.386 497.676 184.106 497.886 184.199C498.119 184.292 498.189 184.083 498.166 183.429C498.142 182.963 498.142 182.52 498.166 182.45C498.329 181.96 498.329 181.517 498.142 181.517C497.909 181.517 497.839 181.983 497.909 183.126C497.956 183.593 497.886 183.849 497.746 183.756C497.606 183.686 497.536 183.429 497.583 183.196C497.606 182.986 497.536 182.683 497.419 182.543C497.186 182.216 497.256 181.727 497.583 181.447C497.723 181.33 497.653 181.283 497.419 181.377C497.023 181.493 496.416 180.793 496.673 180.514C496.743 180.444 496.743 180.07 496.65 179.674C496.486 178.927 496.066 178.741 495.647 179.231ZM440.549 210.558C440.549 210.745 440.502 210.908 440.455 210.908C440.409 210.908 440.315 210.745 440.222 210.558C440.152 210.372 440.199 210.208 440.315 210.208C440.455 210.208 440.549 210.372 440.549 210.558ZM484.87 234.352C484.706 234.562 484.52 234.655 484.473 234.608C484.31 234.445 484.636 234.002 484.916 234.002C485.056 234.002 485.056 234.142 484.87 234.352Z" />
        <path d="M481.511 239.554C481.184 240.044 480.857 240.883 480.811 241.42C480.741 241.933 480.577 242.586 480.438 242.843C480.298 243.123 480.204 243.753 480.251 244.243C480.298 245.059 480.368 245.176 480.974 245.292C481.697 245.432 483.237 244.849 483.237 244.452C483.237 244.289 483.353 244.289 483.54 244.452C483.773 244.639 484.007 244.546 484.473 244.033C484.823 243.683 485.103 243.216 485.103 243.006C485.103 242.796 485.289 242.633 485.546 242.633C485.826 242.633 486.083 242.423 486.176 242.096C486.269 241.817 486.572 241.21 486.876 240.744C487.599 239.601 487.296 239.344 485.733 239.787C483.68 240.347 483.61 240.347 483.12 239.904C482.864 239.647 482.56 239.414 482.42 239.344C482.304 239.297 482.21 239.111 482.257 238.947C482.374 238.458 482.164 238.644 481.511 239.554Z" />
      </Continent>
    </svg>
  );
}
