import os
from pathlib import Path
import pytest
import string
from font_collector import VariableFontFace, FontFile, FontLoader, FontType, NormalFontFace, InvalidVariableFontFaceException, Name
from langcodes import Language
from typing import Hashable


dir_path = os.path.dirname(os.path.realpath(__file__))


def test__init__():
    font_index = 0
    families_prefix = [Name("families_prefix", Language.get("en"))]
    families_suffix = [Name("families_suffix", Language.get("en"))]
    exact_names_suffix = [Name("exact_names_suffix", Language.get("en"))]
    weight = 400
    is_italic = False
    font_type = FontType.TRUETYPE
    named_instance_coordinates = {"ital": False}

    font = VariableFontFace(
        font_index,
        families_prefix,
        families_suffix,
        exact_names_suffix,
        weight,
        is_italic,
        font_type,
        named_instance_coordinates
    )

    assert font.font_index == font_index
    assert font.families_prefix == families_prefix
    assert font.families_suffix == families_suffix
    assert font.exact_names_suffix == exact_names_suffix
    assert font.weight == weight
    assert font.is_italic == is_italic
    assert font.named_instance_coordinates == named_instance_coordinates
    assert font.font_file == None


    families_prefix = []
    with pytest.raises(InvalidVariableFontFaceException) as exc_info:
        font = VariableFontFace(
            font_index,
            families_prefix,
            families_suffix,
            exact_names_suffix,
            weight,
            is_italic,
            font_type,
            named_instance_coordinates
        )
    assert str(exc_info.value) == "The font does not contain an valid family prefix."


def test_font_index_property():
    font = VariableFontFace(0, [Name("test", Language.get("en"))], [], [], 400, False, FontType.TRUETYPE, {})
    font_index = 1
    with pytest.raises(AttributeError) as exc_info:
        font.font_index = font_index


def test_families_prefix_property():
    font = VariableFontFace(0, [Name("test", Language.get("en"))], [], [], 400, False, FontType.TRUETYPE, {})
    families_prefix = [Name("test", Language.get("en"))]
    with pytest.raises(AttributeError) as exc_info:
        font.families_prefix = families_prefix


def test_families_suffix_property():
    font = VariableFontFace(0, [Name("test", Language.get("en"))], [], [], 400, False, FontType.TRUETYPE, {})
    families_suffix = [Name("test", Language.get("en"))]
    with pytest.raises(AttributeError) as exc_info:
        font.families_suffix = families_suffix


def test_family_names_property():
    font = VariableFontFace(0, [Name("family", Language.get("fr-CA"))], [Name("suffix name", Language.get("en"))], [], 400, False, FontType.TRUETYPE, {})
    with pytest.raises(AttributeError) as exc_info:
        font.family_names = ""
    assert str(exc_info.value) == "You cannot set the family name for an variable font. It is automatically generated by the families_prefix and the families_suffix."

    assert font.family_names == [Name("family suffix name", Language.get("en"))]

    font = VariableFontFace(0, [Name("family", Language.get("fr-CA")), Name("family 2", Language.get("fr-CA"))], [Name("suffix name", Language.get("en"))], [], 400, False, FontType.TRUETYPE, {})
    assert font.family_names == [Name("family suffix name", Language.get("en")), Name("family 2 suffix name", Language.get("en"))]

    font = VariableFontFace(0, [Name("family", Language.get("fr-CA")), Name("family 2", Language.get("fr-CA"))], [Name("suffix name", Language.get("en")), Name("suffix name 2", Language.get("es"))], [], 400, False, FontType.TRUETYPE, {})
    assert font.family_names == [
        Name("family suffix name", Language.get("en")),
        Name("family suffix name 2", Language.get("es")),
        Name("family 2 suffix name", Language.get("en")),
        Name("family 2 suffix name 2", Language.get("es")),
    ]


def test_exact_names_suffix_property():
    font = VariableFontFace(0, [Name("test", Language.get("en"))], [], [], 400, False, FontType.TRUETYPE, {})
    exact_names_suffix = [Name("test", Language.get("en"))]
    with pytest.raises(AttributeError) as exc_info:
        font.exact_names_suffix = exact_names_suffix


def test_exact_names_property():
    font = VariableFontFace(0, [Name("family", Language.get("fr-CA"))], [], [Name("exact name", Language.get("en"))], 400, False, FontType.TRUETYPE, {})
    with pytest.raises(AttributeError) as exc_info:
        font.exact_names = ""
    assert str(exc_info.value) == "You cannot set the exact name for an variable font. It is automatically generated by the families_prefix and the exact_names_suffix."

    assert font.exact_names == [Name("family exact name", Language.get("en"))]

    font = VariableFontFace(0, [Name("family", Language.get("fr-CA")), Name("family 2", Language.get("fr-CA"))], [], [Name("exact name", Language.get("en"))], 400, False, FontType.TRUETYPE, {})
    assert font.exact_names == [Name("family exact name", Language.get("en")), Name("family 2 exact name", Language.get("en"))]

    font = VariableFontFace(0, [Name("family", Language.get("fr-CA")), Name("family 2", Language.get("fr-CA"))], [], [Name("exact name", Language.get("en")), Name("exact name 2", Language.get("es"))], 400, False, FontType.TRUETYPE, {})
    assert font.exact_names == [
        Name("family exact name", Language.get("en")),
        Name("family exact name 2", Language.get("es")),
        Name("family 2 exact name", Language.get("en")),
        Name("family 2 exact name 2", Language.get("es")),
    ]


def test_weight_property():
    font = VariableFontFace(0, [Name("test", Language.get("en"))], [], [], 400, False, FontType.TRUETYPE, {})
    weight = 900
    with pytest.raises(AttributeError) as exc_info:
        font.weight = weight


def test_is_italic_property():
    font = VariableFontFace(0, [Name("test", Language.get("en"))], [], [], 400, False, FontType.TRUETYPE, {})
    is_italic = True
    with pytest.raises(AttributeError) as exc_info:
        font.is_italic = is_italic


def test_is_glyph_emboldened_property():
    font = VariableFontFace(0, [Name("test", Language.get("en"))], [], [], 400, False, FontType.TRUETYPE, {})
    assert not font.is_glyph_emboldened

    is_glyph_emboldened = True
    with pytest.raises(AttributeError) as exc_info:
        font.is_glyph_emboldened = is_glyph_emboldened
    assert str(exc_info.value) == "You cannot set is_glyph_emboldened for an variable font. It is automatically generated by the weight."

    font = VariableFontFace(0, [Name("test", Language.get("en"))], [], [], 401, False, FontType.TRUETYPE, {})
    assert font.is_glyph_emboldened


def test_font_type_property():
    font = VariableFontFace(0, [Name("test", Language.get("en"))], [], [], 400, False, FontType.TRUETYPE, {})
    font_type = FontType.OPENTYPE
    with pytest.raises(AttributeError) as exc_info:
        font.font_type = font_type


def test_named_instance_coordinates_property():
    font = VariableFontFace(0, [Name("test", Language.get("en"))], [], [], 400, False, FontType.TRUETYPE, {})
    named_instance_coordinates = {"wght": 800}
    with pytest.raises(AttributeError) as exc_info:
        font.font_type = named_instance_coordinates


def test_link_face_to_a_font_file():
    font_face_1 = VariableFontFace(0, [Name("test", Language.get("en"))], [], [], 400, False, FontType.TRUETYPE, {})
    font_face_2 = VariableFontFace(1, [Name("test", Language.get("en"))], [], [], 400, False, FontType.TRUETYPE, {})
    font_collection_path = Path(os.path.join(os.path.dirname(dir_path), "file", "fonts", "truetype_font_collection.ttc"))
    font_file = FontFile(font_collection_path, [font_face_1], True)
    font_face_2.link_face_to_a_font_file(font_file)
    assert font_face_2.font_file == font_file


def test_get_best_family_prefix_from_lang():
    # This method is difficult to test since it depend on the OS language.
    # In this test, we suppose that the OS language isn't fr-BE or fr. If it is, then, the test will fail
    font_index = 0
    families_suffix = []
    exact_names_suffix = []
    weight = 400
    is_italic = True
    font_type = FontType.TRUETYPE
    named_instance_coordinates = {}

    name_1 = Name("", Language.get("fr-CA"))
    name_2 = Name("", Language.get("fr-BE"))
    name_3 = Name("", Language.get("fr"))

    families_prefix = [name_1, name_2, name_3]

    font = VariableFontFace(
        font_index,
        families_prefix,
        families_suffix,
        exact_names_suffix,
        weight,
        is_italic,
        font_type,
        named_instance_coordinates,
    )

    assert font.get_best_family_prefix_from_lang() == name_1


def test_get_family_prefix_from_lang():
    font_index = 0
    families_suffix = []
    exact_names_suffix = []
    weight = 400
    is_italic = True
    font_type = FontType.TRUETYPE
    named_instance_coordinates = {}

    name_1 = Name("", Language.get("fr-CA"))
    name_2 = Name("", Language.get("fr-BE"))
    name_3 = Name("", Language.get("fr"))

    families_prefix = [name_1, name_2, name_3]

    font = VariableFontFace(
        font_index,
        families_prefix,
        families_suffix,
        exact_names_suffix,
        weight,
        is_italic,
        font_type,
        named_instance_coordinates,
    )

    assert font.get_family_prefix_from_lang("fr-BE") == name_2



def test_get_missing_glyphs():
    font_path = Path(os.path.join(os.path.dirname(dir_path), "file", "fonts", "Asap-VariableFont_wdth,wght.ttf"))
    font_file = FontFile.from_font_path(font_path)

    assert len(font_file.font_faces) == 9
    font_face = font_file.font_faces[0]
    assert isinstance(font_face, VariableFontFace)

    missing_glyphs = font_face.get_missing_glyphs(string.ascii_letters + string.digits + "éｦ&*╠" )
    assert missing_glyphs == {"ｦ", "╠"}


def test_variable_font_to_collection():
    font_path = Path(os.path.join(os.path.dirname(dir_path), "file", "fonts", "Asap-VariableFont_wdth,wght.ttf"))
    font_file = FontFile.from_font_path(font_path)

    assert len(font_file.font_faces) == 9
    font_face = font_file.font_faces[0]
    assert isinstance(font_face, VariableFontFace)

    save_path = Path(os.path.join(dir_path, "Asap - Test.ttf"))
    nbr_generated_font = len(FontLoader.load_generated_fonts())
    try:
        generated_fonts = font_face.variable_font_to_collection(save_path, False)
        font = FontFile.from_font_path(save_path)
    except Exception:
        pass
    finally:
        # Always delete the generated font. Even if there is an exception
        if os.path.isfile(save_path):
            os.remove(save_path)

    assert generated_fonts == font
    assert nbr_generated_font == len(FontLoader.load_generated_fonts())

    expected_fonts = [
        NormalFontFace(
            0,
            [Name("Asap Thin", Language.get("en-US"))],
            [Name("Asap Thin", Language.get("en-US"))],
            100,
            False,
            False,
            FontType.TRUETYPE
        ),
        NormalFontFace(
            1,
            [Name("Asap ExtraLight", Language.get("en-US"))],
            [Name("Asap ExtraLight", Language.get("en-US"))],
            200,
            False,
            False,
            FontType.TRUETYPE
        ),
        NormalFontFace(
            2,
            [Name("Asap Light", Language.get("en-US"))],
            [Name("Asap Light", Language.get("en-US"))],
            300,
            False,
            False,
            FontType.TRUETYPE
        ),
        NormalFontFace(
            3,
            [Name("Asap", Language.get("en-US"))],
            [Name("Asap Regular", Language.get("en-US"))],
            400,
            False,
            False,
            FontType.TRUETYPE
        ),
        NormalFontFace(
            4,
            [Name("Asap Medium", Language.get("en-US"))],
            [Name("Asap Medium", Language.get("en-US"))],
            500,
            False,
            True,
            FontType.TRUETYPE
        ),
        NormalFontFace(
            5,
            [Name("Asap SemiBold", Language.get("en-US"))],
            [Name("Asap SemiBold", Language.get("en-US"))],
            600,
            False,
            True,
            FontType.TRUETYPE
        ),
        NormalFontFace(
            6,
            [Name("Asap", Language.get("en-US"))],
            [Name("Asap Bold", Language.get("en-US"))],
            700,
            False,
            True,
            FontType.TRUETYPE
        ),
        NormalFontFace(
            7,
            [Name("Asap ExtraBold", Language.get("en-US"))],
            [Name("Asap ExtraBold", Language.get("en-US"))],
            800,
            False,
            True,
            FontType.TRUETYPE
        ),
        NormalFontFace(
            8,
            [Name("Asap Black", Language.get("en-US"))],
            [Name("Asap Black", Language.get("en-US"))],
            900,
            False,
            True,
            FontType.TRUETYPE
        )
    ]

    # The generated collection can be in any order. To test it, we need to try all the possible font_index
    for font in expected_fonts:
        font_found = False

        for i in range(len(expected_fonts)):
            font._NormalFontFace__font_index = i # little hack
            if font in generated_fonts.font_faces:
                font_found = True
                break

        assert font_found

    try:
        font_face.variable_font_to_collection(save_path)
    except Exception:
        pass
    finally:
        assert nbr_generated_font + 1 == len(FontLoader.load_generated_fonts())

        # Always delete the generated font. Even if there is an exception
        if os.path.isfile(save_path):
            os.remove(save_path)


def test__eq__():
    font_1 = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )
    font_2 = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )
    assert font_1 == font_2

    font_4 = VariableFontFace(
        1, # different
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )
    assert font_1 != font_4

    font_5 = VariableFontFace(
        0,
        [Name("families_prefix different", Language.get("fr"))], # different
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )
    assert font_1 != font_5

    font_6 = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix different", Language.get("fr"))], # different
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )
    assert font_1 != font_6

    font_7 = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix different", Language.get("fr"))], # different
        400,
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )
    assert font_1 != font_7

    font_8 = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix", Language.get("fr"))],
        401, # different
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )
    assert font_1 != font_8

    font_9 = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        True, # different
        FontType.TRUETYPE,
        {"ital": False}
    )
    assert font_1 != font_9

    font_10 = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        False,
        FontType.OPENTYPE, # different
        {"ital": False}
    )
    assert font_1 != font_10

    font_11 = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        False,
        FontType.TRUETYPE,
        {"ital": True} # different
    )
    assert font_1 != font_11

    font_12 = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        False,
        FontType.TRUETYPE,
        {"ital different": False} # different
    )
    assert font_1 != font_12

    font_13 = VariableFontFace(
        0,
        [Name("families_prefix_1", Language.get("fr")), Name("families_prefix_2", Language.get("fr"))],
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )
    font_14 = VariableFontFace(
        0,
        [Name("families_prefix_2", Language.get("fr")), Name("families_prefix_1", Language.get("fr"))], # order different
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )
    assert font_13 != font_14

    font_15 = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix_1", Language.get("fr")), Name("families_suffix_2", Language.get("fr"))],
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )
    font_16 = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix_2", Language.get("fr")), Name("families_suffix_1", Language.get("fr"))], # order different
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )
    assert font_15 != font_16

    font_17 = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix_1", Language.get("fr")), Name("exact_names_suffix_2", Language.get("fr"))],
        400,
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )
    font_18 = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix_2", Language.get("fr")), Name("exact_names_suffix_1", Language.get("fr"))], # order different
        400,
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )
    assert font_17 != font_18

    assert font_1 != "test"

def test__hash__():
    font_1 = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )
    font_2 = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )
    assert isinstance(font_1, Hashable)
    assert {font_1} == {font_2}

    font_4 = VariableFontFace(
        1, # different
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )
    assert {font_1} != {font_4}

    font_5 = VariableFontFace(
        0,
        [Name("families_prefix different", Language.get("fr"))], # different
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )
    assert {font_1} != {font_5}

    font_6 = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix different", Language.get("fr"))], # different
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )
    assert {font_1} != {font_6}

    font_7 = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix different", Language.get("fr"))], # different
        400,
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )
    assert {font_1} != {font_7}

    font_8 = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix", Language.get("fr"))],
        401, # different
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )
    assert {font_1} != {font_8}

    font_9 = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        True, # different
        FontType.TRUETYPE,
        {"ital": False}
    )
    assert {font_1} != {font_9}

    font_10 = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        False,
        FontType.OPENTYPE, # different
        {"ital": False}
    )
    assert {font_1} != {font_10}

    font_11 = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        False,
        FontType.TRUETYPE,
        {"ital": True} # different
    )
    assert {font_1} != {font_11}

    font_12 = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        False,
        FontType.TRUETYPE,
        {"ital different": False} # different
    )
    assert {font_1} != {font_12}

    font_13 = VariableFontFace(
        0,
        [Name("families_prefix_1", Language.get("fr")), Name("families_prefix_2", Language.get("fr"))],
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )
    font_14 = VariableFontFace(
        0,
        [Name("families_prefix_2", Language.get("fr")), Name("families_prefix_1", Language.get("fr"))], # order different
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )
    assert {font_13} != {font_14}

    font_15 = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix_1", Language.get("fr")), Name("families_suffix_2", Language.get("fr"))],
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )
    font_16 = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix_2", Language.get("fr")), Name("families_suffix_1", Language.get("fr"))], # order different
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )
    assert {font_15} != {font_16}

    font_17 = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix_1", Language.get("fr")), Name("exact_names_suffix_2", Language.get("fr"))],
        400,
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )
    font_18 = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix_2", Language.get("fr")), Name("exact_names_suffix_1", Language.get("fr"))], # order different
        400,
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )
    assert {font_17} != {font_18}

    assert {font_1} != {"test"}

def test__repr__():
    font = VariableFontFace(
        0,
        [Name("families_prefix", Language.get("fr"))],
        [Name("families_suffix", Language.get("fr"))],
        [Name("exact_names_suffix", Language.get("fr"))],
        400,
        False,
        FontType.TRUETYPE,
        {"ital": False}
    )

    assert repr(font) == 'VariableFontFace(Font index="0", Family names="[Name(value="families_prefix families_suffix", lang_code="fr")]", Exact names="[Name(value="families_prefix exact_names_suffix", lang_code="fr")]", Weight="400", Italic="False", Glyph emboldened="False", Font type="TRUETYPE", Named instance coordinates="{\'ital\': False}")'
