# Chapter 7: Symbols - The Linker's Address Book

## Introduction to Symbols

In the realm of C programming and linking, symbols serve as the fundamental building blocks that enable separate compilation and modular programming. They are much more than just names for functions and variables - they represent a sophisticated addressing mechanism that allows the linker to piece together your program from multiple object files. In this comprehensive chapter, we'll dive deep into the world of symbols, understanding their types, properties, and how they influence the linking process.

## Symbol Types and Properties

### Basic Symbol Structure

Every symbol in an object file contains several pieces of information:
1. Name: The identifier used in your source code
2. Value: Usually an offset or address
3. Size: How many bytes the symbol occupies
4. Type: What kind of symbol it is (function, object, etc.)
5. Binding: How visible the symbol is (local, global, weak)
6. Section: Which section contains the symbol

Let's examine this with a practical example:

```c
// symbols.c
#include <stdio.h>

// Global variable with initialization
int global_counter = 42;

// Global variable without initialization
int uninitialized_array[1000];

// Static variable with initialization
static double internal_pi = 3.14159;

// Constant data
const char *message = "Hello, World!";

// Function definition
void print_message(void) {
    printf("%s\n", message);
}

// Static function
static int get_internal_value(void) {
    return global_counter * 2;
}

// Weak symbol
__attribute__((weak))
int get_configuration(void) {
    return 100;
}

int main(void) {
    print_message();
    return get_internal_value();
}
```

Let's compile this and examine the symbols:

```bash
$ gcc -c symbols.c
$ readelf --symbols symbols.o
```

This produces a detailed symbol table. Let's analyze each category:

### Symbol Bindings

1. **Local Binding (l)**
```
Symbol table '.symtab' contains 16 entries:
   Num:    Value          Size Type    Bind   Vis      Ndx Name
    ...
     5: 0000000000000000     8 OBJECT  LOCAL  DEFAULT    3 internal_pi
     6: 0000000000000020    28 FUNC    LOCAL  DEFAULT    1 get_internal_value
```

Local symbols (marked with 'l' or LOCAL) are only visible within their object file. In our example, `internal_pi` and `get_internal_value` are local because they're declared static. These symbols can't be referenced from other object files.

2. **Global Binding (g)**
```
    12: 0000000000000000     4 OBJECT  GLOBAL DEFAULT    2 global_counter
    13: 0000000000000000    18 FUNC    GLOBAL DEFAULT    1 print_message
    14: 0000000000000000  4000 OBJECT  GLOBAL DEFAULT    4 uninitialized_array
```

Global symbols (marked with 'g' or GLOBAL) are visible to all object files. They form the interface between different compilation units.

3. **Weak Binding (w)**
```
    15: 0000000000000048    15 FUNC    WEAK   DEFAULT    1 get_configuration
```

Weak symbols provide a default implementation that can be overridden by a strong (global) symbol.

### Symbol Types

1. **Function Symbols (FUNC)**
```c
void normal_function(void) {
    // Function code
}

static void static_function(void) {
    // Internal function
}

__attribute__((weak))
void weak_function(void) {
    // Overrideable function
}
```

2. **Object Symbols (OBJECT)**
```c
// Different types of object symbols
int global_var = 42;              // Initialized data
int uninit_var;                   // BSS data
static int static_var = 100;      // Local initialized data
const int constant = 255;         // Read-only data
```

3. **Section Symbols (SECTION)**
These are automatically generated by the assembler to mark section boundaries.

### Advanced Symbol Properties

Let's create a more complex example to demonstrate advanced symbol properties:

```c
// advanced_symbols.c
#include <stdio.h>

// Symbol versioning
__asm__(".symver original_function,original_function@VERS_1.0");
__asm__(".symver new_function,new_function@@VERS_2.0");

// Symbol visibility
__attribute__((visibility("hidden")))
void internal_helper(void) {
    printf("Internal helper\n");
}

// Symbol aliasing
void real_function(void) {
    printf("Real function\n");
}
void __attribute__((alias("real_function"))) aliased_function(void);

// Thread-local storage
__thread int per_thread_counter = 0;

// Constructor/destructor functions
__attribute__((constructor))
void initialization(void) {
    printf("Library initialization\n");
}

__attribute__((destructor))
void cleanup(void) {
    printf("Library cleanup\n");
}
```

## Symbol Resolution Process

The linker follows a sophisticated process when resolving symbols:

### 1. Symbol Collection Phase

During this phase, the linker:
1. Reads all input object files
2. Creates a global symbol table
3. Notes all undefined symbols
4. Records symbol properties and attributes

Let's see this with multiple files:

```c
// module1.c
int shared_data = 42;
extern void external_function(void);

void module1_function(void) {
    external_function();
}

// module2.c
extern int shared_data;
void external_function(void) {
    shared_data++;
}

// main.c
extern void module1_function(void);
extern int shared_data;

int main(void) {
    module1_function();
    return shared_data;
}
```

Compile and examine:
```bash
$ gcc -c module1.c module2.c main.c
$ nm module1.o module2.o main.o
```

### 2. Symbol Resolution Rules

The linker follows specific rules when resolving symbols:

1. **Strong vs Weak Resolution**
```c
// file1.c
int variable = 100;  // Strong definition

// file2.c
__attribute__((weak)) int variable = 50;  // Weak definition
```

2. **Common Symbol Resolution**
```c
// file1.c
int shared;  // Common symbol

// file2.c
int shared = 42;  // Initialized definition wins
```

3. **Multiple Definition Handling**
```c
// Generates link error if both are strong
int duplicate_symbol = 1;  // file1.c
int duplicate_symbol = 2;  // file2.c
```

### Symbol Table Manipulation

We can use various tools to manipulate symbol tables:

1. **nm Tool Usage**
```bash
$ nm --demangle --numeric-sort program
$ nm --undefined-only program
$ nm --extern-only program
```

2. **objcopy for Symbol Manipulation**
```bash
$ objcopy --add-symbol new_symbol=.data:0x100 input.o output.o
$ objcopy --redefine-sym old_name=new_name input.o output.o
```

3. **strip for Symbol Removal**
```bash
$ strip --strip-unneeded program
$ strip --strip-debug program
```

## Practical Symbol Usage Patterns

### 1. Library Interface Design

```c
// lib_interface.h
#ifndef LIB_INTERFACE_H
#define LIB_INTERFACE_H

// Public API
#define API __attribute__((visibility("default")))

API void public_function(void);
API extern int public_variable;

#endif

// lib_implementation.c
#include "lib_interface.h"

// Internal symbols
static void helper_function(void);
static int internal_state;

API void public_function(void) {
    helper_function();
}

API int public_variable = 42;
```

### 2. Symbol Versioning

```c
// version_script.map
VERS_1.0 {
    global:
        original_function;
    local:
        *;
};

VERS_2.0 {
    global:
        new_function;
} VERS_1.0;
```

Compile with version script:
```bash
$ gcc -shared -Wl,--version-script=version_script.map -o libversioned.so lib.c
```

### 3. Symbol Interposition

```c
// interpose.c
#define _GNU_SOURCE
#include <dlfcn.h>

typedef int (*original_func)(const char *);

int open(const char *pathname, int flags, ...) {
    original_func original_open = dlsym(RTLD_NEXT, "open");
    printf("Opening: %s\n", pathname);
    return original_open(pathname, flags);
}
```

## Debugging Symbol Issues

### 1. Using GDB with Symbols

```bash
$ gcc -g program.c -o program
$ gdb ./program
(gdb) info functions
(gdb) info variables
(gdb) p 'static_variable'
```

### 2. Handling Symbol Conflicts

```bash
$ gcc -fvisibility=hidden -o program program.c  # Hide all symbols by default
$ gcc -Wl,--verbose -o program program.c        # See linker's symbol resolution
```

### 3. Symbol Versioning Debug

```bash
$ readelf --version-info libversioned.so
$ objdump -T libversioned.so
```

## Advanced Symbol Topics

### 1. Thread-Local Symbols

```c
// thread_symbols.c
#include <pthread.h>

__thread int per_thread_counter = 0;

void *thread_func(void *arg) {
    per_thread_counter++;  // Each thread has its own copy
    return NULL;
}
```

### 2. Symbol Initialization Priority

```c
__attribute__((constructor(101)))
void early_init(void) {
    // Runs early during initialization
}

__attribute__((constructor(200)))
void late_init(void) {
    // Runs later
}
```

### 3. Dynamic Symbol Loading

```c
#include <dlfcn.h>

void *handle = dlopen("./libdynamic.so", RTLD_LAZY);
void (*dynamic_func)(void) = dlsym(handle, "dynamic_function");
```

## Common Symbol-Related Issues and Solutions

1. **Undefined Reference Errors**
```bash
$ gcc -c module1.c module2.c
$ nm module1.o module2.o | grep undefined
```

2. **Multiple Definition Errors**
```bash
$ gcc -fcommon  # Allow common symbols (old behavior)
$ gcc -fno-common  # Modern approach, catch multiple definitions
```

3. **Symbol Visibility Issues**
```bash
$ readelf --dyn-syms library.so  # Check exported symbols
$ objdump -T library.so          # Show dynamic symbols
```

This comprehensive coverage of symbols provides the foundation for understanding how the linker works with your code. In the next chapter, we'll explore how to control the physical layout of these symbols in memory using linker scripts.
