# Before main(): The Secret Life of Global Variables in C

Many C programmers assume their program starts executing at `main()`, but there's actually quite a bit happening before that first line of `main()` runs. In this post, we'll dive deep into the initialization of global variables and explore the hidden startup sequence that sets up your program's environment.

## The Mystery of Initialized Globals

Let's start with a simple example that demonstrates global variable initialization:

```c
// globals.c
#include <stdio.h>

// Global variables with various initializations
int simple_global = 42;
int array_global[] = {1, 2, 3, 4, 5};
char *string_global = "Hello, World!";

// Global struct with initializer
struct Point {
    int x;
    int y;
} point_global = {10, 20};

int main() {
    printf("simple_global: %d\n", simple_global);
    printf("array_global[2]: %d\n", array_global[2]);
    printf("string_global: %s\n", string_global);
    printf("point_global: (%d,%d)\n", point_global.x, point_global.y);
    return 0;
}
```

Let's compile this with debugging information and examine what happens before `main()`:

```bash
$ gcc -g -o globals globals.c
```

## Using GDB to Peek Behind the Curtain

Now let's use GDB to examine these variables before `main()` executes:

```bash
$ gdb ./globals
(gdb) break main
Breakpoint 1 at 0x1149: file globals.c, line 15.
(gdb) run
Starting program: ./globals

Breakpoint 1, main () at globals.c:15
(gdb) print simple_global
$1 = 42
(gdb) print array_global
$2 = {1, 2, 3, 4, 5}
(gdb) print point_global
$3 = {x = 10, y = 20}
```

Notice that all our global variables are already initialized when we hit `main()`! But how did this happen?

## The Hidden Startup Sequence

Let's examine the startup sequence by looking at the call stack when we break even earlier:

```bash
(gdb) break _start
(gdb) run
Starting program: ./globals

Breakpoint 1, _start () at ../sysdeps/x86_64/start.S:120
(gdb) bt
#0  _start () at ../sysdeps/x86_64/start.S:120
```

The actual sequence is:
1. `_start` (assembly entry point)
2. `__libc_start_main` (C runtime initialization)
3. Various initialization functions
4. Finally, your `main()`

We can see this by setting breakpoints and stepping through:

```bash
(gdb) break __libc_start_main
(gdb) continue
Continuing.

Breakpoint 2, __libc_start_main () at ../csu/libc-start.c:332
(gdb) info functions __do_global_*
All functions matching regular expression "__do_global_*":

Non-debugging symbols:
0x0000000000001090  __do_global_dtors_aux
0x00000000000010f0  __do_global_ctors_aux
```

## The .init_array Section

Global variable initialization is handled through special functions stored in the `.init_array` section. Let's examine this section:

```bash
$ readelf -W -d globals | grep INIT
 0x000000000000001b (INIT)               0x1000
 0x0000000000000019 (INIT_ARRAY)         0x3dc8
 0x000000000000001a (INIT_ARRAYSZ)       0x8
```

These initialization functions are automatically generated by the compiler. Let's create a more complex example to see them in action:

```c
// complex_globals.c
#include <stdio.h>
#include <stdlib.h>

// A global that requires computation
int computed_global = 40 + 2;

// A global that requires function calls
char *allocated_string = NULL;

// Constructor function that runs before main
__attribute__((constructor))
void init_globals() {
    allocated_string = strdup("Dynamically allocated string");
    printf("Constructor running!\n");
}

int main() {
    printf("In main(): computed_global = %d\n", computed_global);
    printf("In main(): allocated_string = %s\n", allocated_string);
    free(allocated_string);
    return 0;
}
```

Compile and run under GDB:

```bash
$ gcc -g -o complex_globals complex_globals.c
$ gdb ./complex_globals
(gdb) break main
(gdb) run
Constructor running!
Breakpoint 1, main () at complex_globals.c:18
```

Notice that our constructor function ran before we hit `main()`!

## Understanding Global Variable Storage

Let's look at where these initialized globals are stored in the executable:

```bash
$ objdump -s -j .data complex_globals

complex_globals:     file format elf64-x86-64

Contents of section .data:
 4000 2a000000 00000000 00000000          *...........
```

The computed value (42) is already stored in the `.data` section! This is because the compiler evaluated the constant expression at compile time.

## Types of Global Variable Initialization

There are several types of global variable initialization, each handled differently:

1. **Constant Initialization**
```c
int simple = 42;                  // Direct to .data
const int constant = 100;         // Goes to .rodata
```

2. **Zero Initialization**
```c
int zero_global = 0;             // Optimized to .bss
int implicit_zero_global;        // Automatically in .bss
```

3. **Dynamic Initialization**
```c
int random_global = rand();      // Requires runtime init
```

Let's see how these are handled:

```c
// init_types.c
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

int constant_init = 42;
int zero_init = 0;
int dynamic_init = 0;  // Will be set by constructor

__attribute__((constructor))
void init_dynamic() {
    srand(time(NULL));
    dynamic_init = rand();
}

int main() {
    printf("Values: %d, %d, %d\n", 
           constant_init, zero_init, dynamic_init);
    return 0;
}
```

## Global Constructors and Destructors

C++ programmers might be familiar with global constructors and destructors, but C has them too! We can use function attributes:

```c
// lifecycle.c
#include <stdio.h>

__attribute__((constructor))
void before_main() {
    printf("Constructor: Running before main\n");
}

__attribute__((destructor))
void after_main() {
    printf("Destructor: Running after main\n");
}

int main() {
    printf("Main: Now executing\n");
    return 0;
}
```

Running this shows the execution order:
```bash
$ ./lifecycle
Constructor: Running before main
Main: Now executing
Destructor: Running after main
```

## Common Pitfalls and Best Practices

1. **Initialization Order Dependencies**
```c
// Dangerous: relies on initialization order
int a = 42;
int b = a + 1;  // May not work as expected
```

2. **Dynamic Allocation**
```c
// Dangerous: memory leak if program terminates abnormally
char *global_buf = NULL;

__attribute__((constructor))
void init_buf() {
    global_buf = malloc(1024);
}
```

3. **Circular Dependencies**
```c
// Dangerous: undefined behavior
extern int b;
int a = b + 1;
int b = a + 1;
```

## Best Practices for Global Variables

1. **Use Static Initialization When Possible**
```c
// Good: compile-time constant
static const int CONFIG_VALUE = 100;

// Avoid: runtime computation
static int config_value = compute_value();
```

2. **Consider Using Singleton Functions**
```c
// Better than raw global variables
int* get_config(void) {
    static int config = 0;
    static int initialized = 0;
    
    if (!initialized) {
        config = compute_value();
        initialized = 1;
    }
    return &config;
}
```

3. **Use Guards for Dynamic Initialization**
```c
static int initialized = 0;
static char* global_resource = NULL;

void init_resource(void) {
    if (!initialized) {
        global_resource = allocate_resource();
        initialized = 1;
    }
}
```

## Debugging Global Initialization

Here are some useful GDB commands for debugging global initialization:

```bash
# Break before any initialization
(gdb) break _start

# Watch a global variable
(gdb) watch global_variable

# Show constructor functions
(gdb) info functions __attribute__
```

## Conclusion

Understanding how global variables are initialized is crucial for:
- Debugging initialization-related bugs
- Writing reliable constructors and destructors
- Managing program startup performance
- Handling complex initialization dependencies

In our next post, "The Linker: Bringing Your C Code Together," we'll explore how the linker handles these global variables when combining multiple object files, and how it resolves initialization ordering across translation units.

For readers interested in more details about memory layout and initialization ordering, these topics will be covered in more depth in Post 8, "Customizing the Layout: Introduction to Linker Scripts," where we'll learn how to take control of the initialization process through linker scripts.
