# -*- coding: utf-8 -*-
"""Utility functions for quantization hadamard transformation."""

import scipy.linalg
import torch
import torch.nn.functional as F

from .functional import is_pow2

# Adapted from https://github.com/Cornell-RelaxML/quip-sharp/blob/main/lib/utils/matmul_had.py
# Copied from https://github.com/spcl/QuaRot/blob/main/fake_quant/hadamard_utils.py

__all__ = ["random_hadamard_matrix", "get_hadamard_matrices", "hardmard_transform"]


def _get_hadamard_K(n: int, transpose: bool = False, force: bool = False) -> tuple[torch.Tensor | None, int]:
    """Get the Hadamard matrix and its dimension for a given input size."""
    hadamard_K, K = None, None
    if n % 172 == 0:  # llama-2-7b up
        assert is_pow2(n // 172)
        K = 172
        hadamard_K = _get_hadamard_172().T if transpose else _get_hadamard_172()
    elif n % 156 == 0:  # llama-1-30b 3x hidden
        assert is_pow2(n // 156)
        K = 156
        hadamard_K = _get_hadamard_156().T if transpose else _get_hadamard_156()
    elif n % 140 == 0:  # llama-1-30b intermediate
        assert is_pow2(n // 140)
        K = 140
        hadamard_K = _get_hadamard_140().T if transpose else _get_hadamard_140()
    elif n % 108 == 0:  # llama-1-13b intermediate
        assert is_pow2(n // 108)
        K = 108
        hadamard_K = _get_hadamard_108().T if transpose else _get_hadamard_108()
    elif n % 60 == 0:  # llama-1-13b 3x hidden
        assert is_pow2(n // 60)
        K = 60
        hadamard_K = _get_hadamard_60().T if transpose else _get_hadamard_60()
    elif n % 52 == 0:  # llama-1-13b 1x hidden
        assert is_pow2(n // 52)
        K = 52
        hadamard_K = _get_hadamard_52().T if transpose else _get_hadamard_52()
    elif n % 36 == 0:
        assert is_pow2(n // 36)
        K = 36
        hadamard_K = _get_hadamard_36().T if transpose else _get_hadamard_36()
    elif n % 28 == 0:
        assert is_pow2(n // 28)
        K = 28
        hadamard_K = _get_hadamard_28().T if transpose else _get_hadamard_28()
    elif n % 40 == 0:
        assert is_pow2(n // 40)
        K = 40
        hadamard_K = _get_hadamard_40().T if transpose else _get_hadamard_40()
    elif n % 20 == 0:
        assert is_pow2(n // 20)
        K = 20
        hadamard_K = _get_hadamard_20().T if transpose else _get_hadamard_20()
    elif n % 12 == 0:
        assert is_pow2(n // 12)
        K = 12
        hadamard_K = _get_hadamard_12().T if transpose else _get_hadamard_12()
    else:
        assert is_pow2(n)
        if force:
            hadamard_K = torch.Tensor(scipy.linalg.hadamard(n))
        K = 1

    return hadamard_K, K


def _matmul_hadU(X: torch.Tensor, hadamard_K: torch.Tensor | None, K: int) -> torch.Tensor:
    """Apply Hadamard matrix to the input tensor."""
    n = X.shape[-1]
    input = X.clone().view(-1, n, 1)
    output = input.clone()
    while input.shape[1] > K:
        input = input.view(input.shape[0], input.shape[1] // 2, 2, input.shape[2])
        output = output.view(input.shape)
        output[:, :, 0, :] = input[:, :, 0, :] + input[:, :, 1, :]
        output[:, :, 1, :] = input[:, :, 0, :] - input[:, :, 1, :]
        output = output.view(input.shape[0], input.shape[1], -1)
        (input, output) = (output, input)
    del output

    if K > 1:
        # Do not explicitly repeat - OOM
        # input = torch.bmm(
        #     hadK.repeat(len(input), 1, 1).to(input.device).to(input.dtype), input)
        # Use bcast instead
        assert hadamard_K is not None
        input = hadamard_K.view(1, K, K).to(input) @ input

    return input.view(X.shape) / torch.tensor(n).sqrt()


def random_hadamard_matrix(size: int) -> torch.Tensor:
    """Generate a random Hadamard matrix of size `size`."""
    # See https://cornell-relaxml.github.io/quip-sharp/ , Section "Randomized Hadamard Transformation"
    Q = torch.randint(low=0, high=2, size=(size,)).to(torch.float64)
    Q = Q * 2 - 1
    Q = torch.diag(Q)
    hadamard_K, K = _get_hadamard_K(Q.shape[-1])
    return _matmul_hadU(Q, hadamard_K=hadamard_K, K=K)


def get_hadamard_matrices(n: int) -> tuple[torch.Tensor, torch.Tensor | None, int]:
    """Get the Hadamard matrix and its dimension for a given input size."""
    H2, K2 = _get_hadamard_K(n, force=True)
    if K2 > 1:
        H1, K1 = _get_hadamard_K(n // K2, force=True)
        assert K1 == 1
    else:
        H1, K1 = H2, K2
    assert H1 is not None
    return H1, H2, K2


def hardmard_transform(
    X: torch.Tensor, H1: torch.Tensor, H2: torch.Tensor | None, K: int, scaled: bool = False
) -> torch.Tensor:
    """Apply Hadamard matrix to the input tensor."""
    shape = X.shape
    n = shape[-1]
    XH = X.view(-1, K, n // K)
    XH = F.linear(XH, H1)
    if not scaled:
        XH = XH.mul_(1.0 / torch.tensor(n).sqrt().to(XH.device, XH.dtype))
    if K > 1:
        XH = H2 @ XH
    return XH.view(shape)


# region hadamard matrices
# hadamard matrices for had12, had36.pal2, had52,will, had60.pal, had108.pal, had140.pal, had156.will, had172.will:
# http://www.neilsloane.com/hadamard/index.html
def _get_hadamard_12() -> torch.Tensor:
    return torch.FloatTensor(
        [
            [+1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1],
            [+1, +1, -1, +1, -1, -1, -1, +1, +1, +1, -1, +1],
            [+1, +1, +1, -1, +1, -1, -1, -1, +1, +1, +1, -1],
            [+1, -1, +1, +1, -1, +1, -1, -1, -1, +1, +1, +1],
            [+1, +1, -1, +1, +1, -1, +1, -1, -1, -1, +1, +1],
            [+1, +1, +1, -1, +1, +1, -1, +1, -1, -1, -1, +1],
            [+1, +1, +1, +1, -1, +1, +1, -1, +1, -1, -1, -1],
            [+1, -1, +1, +1, +1, -1, +1, +1, -1, +1, -1, -1],
            [+1, -1, -1, +1, +1, +1, -1, +1, +1, -1, +1, -1],
            [+1, -1, -1, -1, +1, +1, +1, -1, +1, +1, -1, +1],
            [+1, +1, -1, -1, -1, +1, +1, +1, -1, +1, +1, -1],
            [+1, -1, +1, -1, -1, -1, +1, +1, +1, -1, +1, +1],
        ]
    )


def _get_hadamard_40() -> torch.Tensor:
    return torch.FloatTensor(
        [
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
            ],
        ]
    )


def _get_hadamard_20() -> torch.Tensor:
    return torch.FloatTensor(
        [
            [+1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1],
            [+1, +1, -1, +1, +1, -1, -1, -1, -1, +1, -1, +1, -1, +1, +1, +1, +1, -1, -1, +1],
            [+1, +1, +1, -1, +1, +1, -1, -1, -1, -1, +1, -1, +1, -1, +1, +1, +1, +1, -1, -1],
            [+1, -1, +1, +1, -1, +1, +1, -1, -1, -1, -1, +1, -1, +1, -1, +1, +1, +1, +1, -1],
            [+1, -1, -1, +1, +1, -1, +1, +1, -1, -1, -1, -1, +1, -1, +1, -1, +1, +1, +1, +1],
            [+1, +1, -1, -1, +1, +1, -1, +1, +1, -1, -1, -1, -1, +1, -1, +1, -1, +1, +1, +1],
            [+1, +1, +1, -1, -1, +1, +1, -1, +1, +1, -1, -1, -1, -1, +1, -1, +1, -1, +1, +1],
            [+1, +1, +1, +1, -1, -1, +1, +1, -1, +1, +1, -1, -1, -1, -1, +1, -1, +1, -1, +1],
            [+1, +1, +1, +1, +1, -1, -1, +1, +1, -1, +1, +1, -1, -1, -1, -1, +1, -1, +1, -1],
            [+1, -1, +1, +1, +1, +1, -1, -1, +1, +1, -1, +1, +1, -1, -1, -1, -1, +1, -1, +1],
            [+1, +1, -1, +1, +1, +1, +1, -1, -1, +1, +1, -1, +1, +1, -1, -1, -1, -1, +1, -1],
            [+1, -1, +1, -1, +1, +1, +1, +1, -1, -1, +1, +1, -1, +1, +1, -1, -1, -1, -1, +1],
            [+1, +1, -1, +1, -1, +1, +1, +1, +1, -1, -1, +1, +1, -1, +1, +1, -1, -1, -1, -1],
            [+1, -1, +1, -1, +1, -1, +1, +1, +1, +1, -1, -1, +1, +1, -1, +1, +1, -1, -1, -1],
            [+1, -1, -1, +1, -1, +1, -1, +1, +1, +1, +1, -1, -1, +1, +1, -1, +1, +1, -1, -1],
            [+1, -1, -1, -1, +1, -1, +1, -1, +1, +1, +1, +1, -1, -1, +1, +1, -1, +1, +1, -1],
            [+1, -1, -1, -1, -1, +1, -1, +1, -1, +1, +1, +1, +1, -1, -1, +1, +1, -1, +1, +1],
            [+1, +1, -1, -1, -1, -1, +1, -1, +1, -1, +1, +1, +1, +1, -1, -1, +1, +1, -1, +1],
            [+1, +1, +1, -1, -1, -1, -1, +1, -1, +1, -1, +1, +1, +1, +1, -1, -1, +1, +1, -1],
            [+1, -1, +1, +1, -1, -1, -1, -1, +1, -1, +1, -1, +1, +1, +1, +1, -1, -1, +1, +1],
        ]
    )


def _get_hadamard_28() -> torch.Tensor:
    return torch.FloatTensor(
        [
            [
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
            ],
        ]
    )


def _get_hadamard_36() -> torch.Tensor:
    return torch.FloatTensor(
        [
            [
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
            ],
        ]
    )


def _get_hadamard_60() -> torch.Tensor:
    return torch.FloatTensor(
        [
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
            ],
        ]
    )


def _get_hadamard_52() -> torch.Tensor:
    return torch.FloatTensor(
        [
            [
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
            ],
            [
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
            ],
            [
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
            ],
            [
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
            ],
        ]
    )


def _get_hadamard_108() -> torch.Tensor:
    return torch.FloatTensor(
        [
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
            ],
        ]
    )


def _get_hadamard_140() -> torch.Tensor:
    return torch.FloatTensor(
        [
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
            ],
        ]
    )


def _get_hadamard_156() -> torch.Tensor:
    return torch.FloatTensor(
        [
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
            ],
            [
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
            ],
            [
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
            ],
            [
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
            ],
            [
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
            ],
            [
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
            ],
            [
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
            ],
            [
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                +1,
            ],
            [
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
            ],
            [
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
            ],
            [
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
            ],
            [
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
            ],
            [
                +1,
                +1,
                -1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                -1,
                +1,
                -1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                +1,
                -1,
                -1,
                +1,
                -1,
                -1,
                -1,
                -1,
                +1,
                -1,
                +1,
                -1,
                -1,
                +1,
                +1,
                +1,
                -1,
   