// Copyright (c) Microsoft Corporation
// License: MIT OR Apache-2.0

#![allow(missing_docs)]

pub use bindings::*;
#[cfg(any(driver_model__driver_type = "WDM", driver_model__driver_type = "KMDF"))]
pub use kernel_mode::*;
#[cfg(any(driver_model__driver_type = "KMDF", driver_model__driver_type = "UMDF"))]
pub use wdf::*;

use crate::types::NTSTATUS;

#[allow(non_upper_case_globals)]
#[rustversion::attr(
    any(
        all(not(nightly), since(1.78)),
        all(nightly, since(2024-02-09)),
    ),
    // required until following issue is resolved: https://github.com/rust-lang/rust-bindgen/issues/2710
    allow(clippy::manual_c_str_literals)
)]
#[allow(clippy::unreadable_literal)]
mod bindings {
    // allow wildcards for types module since underlying c code relies on all
    // type definitions being in scope
    #[allow(clippy::wildcard_imports)]
    use crate::types::*;

    include!(concat!(env!("OUT_DIR"), "/constants.rs"));
}

#[cfg(any(driver_model__driver_type = "KMDF", driver_model__driver_type = "UMDF"))]
mod wdf {
    use crate::types::{PVOID, PWDF_OBJECT_ATTRIBUTES};

    pub const WDF_NO_OBJECT_ATTRIBUTES: PWDF_OBJECT_ATTRIBUTES = core::ptr::null_mut();
    pub const WDF_NO_EVENT_CALLBACK: PVOID = core::ptr::null_mut();
    pub const WDF_NO_HANDLE: PVOID = core::ptr::null_mut();
    pub const WDF_NO_CONTEXT: PVOID = core::ptr::null_mut();
    pub const WDF_NO_SEND_OPTIONS: PVOID = core::ptr::null_mut();
}

#[cfg(any(driver_model__driver_type = "WDM", driver_model__driver_type = "KMDF"))]
mod kernel_mode {
    use crate::types::POOL_FLAGS;

    // Macros with MSVC C Integer Constant Suffixes are not supported by bindgen, so they must be manually ported or imported from elsewhere: https://github.com/rust-lang/rust-bindgen/issues/2600
    pub const POOL_FLAG_REQUIRED_START: POOL_FLAGS = 0x0000_0000_0000_0001;
    pub const POOL_FLAG_USE_QUOTA: POOL_FLAGS = 0x0000_0000_0000_0001; // Charge quota
    pub const POOL_FLAG_UNINITIALIZED: POOL_FLAGS = 0x0000_0000_0000_0002; // Don't zero-initialize allocation
    pub const POOL_FLAG_SESSION: POOL_FLAGS = 0x0000_0000_0000_0004; // Use session specific pool
    pub const POOL_FLAG_CACHE_ALIGNED: POOL_FLAGS = 0x0000_0000_0000_0008; // Cache aligned allocation
    pub const POOL_FLAG_RESERVED1: POOL_FLAGS = 0x0000_0000_0000_0010; // Reserved for system use
    pub const POOL_FLAG_RAISE_ON_FAILURE: POOL_FLAGS = 0x0000_0000_0000_0020; // Raise exception on failure
    pub const POOL_FLAG_NON_PAGED: POOL_FLAGS = 0x0000_0000_0000_0040; // Non paged pool NX
    pub const POOL_FLAG_NON_PAGED_EXECUTE: POOL_FLAGS = 0x0000_0000_0000_0080; // Non paged pool executable
    pub const POOL_FLAG_PAGED: POOL_FLAGS = 0x0000_0000_0000_0100; // Paged pool
    pub const POOL_FLAG_RESERVED2: POOL_FLAGS = 0x0000_0000_0000_0200; // Reserved for system use
    pub const POOL_FLAG_RESERVED3: POOL_FLAGS = 0x0000_0000_0000_0400; // Reserved for system use
    pub const POOL_FLAG_REQUIRED_END: POOL_FLAGS = 0x0000_0000_8000_0000;
    pub const POOL_FLAG_OPTIONAL_START: POOL_FLAGS = 0x0000_0001_0000_0000;
    pub const POOL_FLAG_SPECIAL_POOL: POOL_FLAGS = 0x0000_0001_0000_0000; // Make special pool allocation
    pub const POOL_FLAG_OPTIONAL_END: POOL_FLAGS = 0x8000_0000_0000_0000;
}

// Due to linker issues with windows_sys, these definitions are manually
// imported definitions from windows_sys::Win32::Foundation:
pub const STATUS_ABANDONED: NTSTATUS = 128_i32;
pub const STATUS_ABANDONED_WAIT_0: NTSTATUS = 128_i32;
pub const STATUS_ABANDONED_WAIT_63: NTSTATUS = 191_i32;
pub const STATUS_ABANDON_HIBERFILE: NTSTATUS = 1_073_741_875_i32;
pub const STATUS_ABIOS_INVALID_COMMAND: NTSTATUS = -1_073_741_549_i32;
pub const STATUS_ABIOS_INVALID_LID: NTSTATUS = -1_073_741_548_i32;
pub const STATUS_ABIOS_INVALID_SELECTOR: NTSTATUS = -1_073_741_546_i32;
pub const STATUS_ABIOS_LID_ALREADY_OWNED: NTSTATUS = -1_073_741_551_i32;
pub const STATUS_ABIOS_LID_NOT_EXIST: NTSTATUS = -1_073_741_552_i32;
pub const STATUS_ABIOS_NOT_LID_OWNER: NTSTATUS = -1_073_741_550_i32;
pub const STATUS_ABIOS_NOT_PRESENT: NTSTATUS = -1_073_741_553_i32;
pub const STATUS_ABIOS_SELECTOR_NOT_AVAILABLE: NTSTATUS = -1_073_741_547_i32;
pub const STATUS_ACCESS_AUDIT_BY_POLICY: NTSTATUS = 1_073_741_874_i32;
pub const STATUS_ACCESS_DENIED: NTSTATUS = -1_073_741_790_i32;
pub const STATUS_ACCESS_DISABLED_BY_POLICY_DEFAULT: NTSTATUS = -1_073_740_959_i32;
pub const STATUS_ACCESS_DISABLED_BY_POLICY_OTHER: NTSTATUS = -1_073_740_956_i32;
pub const STATUS_ACCESS_DISABLED_BY_POLICY_PATH: NTSTATUS = -1_073_740_958_i32;
pub const STATUS_ACCESS_DISABLED_BY_POLICY_PUBLISHER: NTSTATUS = -1_073_740_957_i32;
pub const STATUS_ACCESS_DISABLED_NO_SAFER_UI_BY_POLICY: NTSTATUS = -1_073_740_942_i32;
pub const STATUS_ACCESS_VIOLATION: NTSTATUS = -1_073_741_819_i32;
pub const STATUS_ACPI_ACQUIRE_GLOBAL_LOCK: NTSTATUS = -1_072_431_086_i32;
pub const STATUS_ACPI_ADDRESS_NOT_MAPPED: NTSTATUS = -1_072_431_092_i32;
pub const STATUS_ACPI_ALREADY_INITIALIZED: NTSTATUS = -1_072_431_085_i32;
pub const STATUS_ACPI_ASSERT_FAILED: NTSTATUS = -1_072_431_101_i32;
pub const STATUS_ACPI_FATAL: NTSTATUS = -1_072_431_098_i32;
pub const STATUS_ACPI_HANDLER_COLLISION: NTSTATUS = -1_072_431_090_i32;
pub const STATUS_ACPI_INCORRECT_ARGUMENT_COUNT: NTSTATUS = -1_072_431_093_i32;
pub const STATUS_ACPI_INVALID_ACCESS_SIZE: NTSTATUS = -1_072_431_087_i32;
pub const STATUS_ACPI_INVALID_ARGTYPE: NTSTATUS = -1_072_431_096_i32;
pub const STATUS_ACPI_INVALID_ARGUMENT: NTSTATUS = -1_072_431_099_i32;
pub const STATUS_ACPI_INVALID_DATA: NTSTATUS = -1_072_431_089_i32;
pub const STATUS_ACPI_INVALID_EVENTTYPE: NTSTATUS = -1_072_431_091_i32;
pub const STATUS_ACPI_INVALID_INDEX: NTSTATUS = -1_072_431_100_i32;
pub const STATUS_ACPI_INVALID_MUTEX_LEVEL: NTSTATUS = -1_072_431_083_i32;
pub const STATUS_ACPI_INVALID_OBJTYPE: NTSTATUS = -1_072_431_095_i32;
pub const STATUS_ACPI_INVALID_OPCODE: NTSTATUS = -1_072_431_103_i32;
pub const STATUS_ACPI_INVALID_REGION: NTSTATUS = -1_072_431_088_i32;
pub const STATUS_ACPI_INVALID_SUPERNAME: NTSTATUS = -1_072_431_097_i32;
pub const STATUS_ACPI_INVALID_TABLE: NTSTATUS = -1_072_431_079_i32;
pub const STATUS_ACPI_INVALID_TARGETTYPE: NTSTATUS = -1_072_431_094_i32;
pub const STATUS_ACPI_MUTEX_NOT_OWNED: NTSTATUS = -1_072_431_082_i32;
pub const STATUS_ACPI_MUTEX_NOT_OWNER: NTSTATUS = -1_072_431_081_i32;
pub const STATUS_ACPI_NOT_INITIALIZED: NTSTATUS = -1_072_431_084_i32;
pub const STATUS_ACPI_POWER_REQUEST_FAILED: NTSTATUS = -1_072_431_071_i32;
pub const STATUS_ACPI_REG_HANDLER_FAILED: NTSTATUS = -1_072_431_072_i32;
pub const STATUS_ACPI_RS_ACCESS: NTSTATUS = -1_072_431_080_i32;
pub const STATUS_ACPI_STACK_OVERFLOW: NTSTATUS = -1_072_431_102_i32;
pub const STATUS_ADAPTER_HARDWARE_ERROR: NTSTATUS = -1_073_741_630_i32;
pub const STATUS_ADDRESS_ALREADY_ASSOCIATED: NTSTATUS = -1_073_741_256_i32;
pub const STATUS_ADDRESS_ALREADY_EXISTS: NTSTATUS = -1_073_741_302_i32;
pub const STATUS_ADDRESS_CLOSED: NTSTATUS = -1_073_741_301_i32;
pub const STATUS_ADDRESS_NOT_ASSOCIATED: NTSTATUS = -1_073_741_255_i32;
pub const STATUS_ADMINLESS_ACCESS_DENIED: NTSTATUS = -1_073_700_348_i32;
pub const STATUS_ADVANCED_INSTALLER_FAILED: NTSTATUS = -1_072_365_536_i32;
pub const STATUS_AGENTS_EXHAUSTED: NTSTATUS = -1_073_741_691_i32;
pub const STATUS_ALERTED: NTSTATUS = 257_i32;
pub const STATUS_ALIAS_EXISTS: NTSTATUS = -1_073_741_484_i32;
pub const STATUS_ALLOCATE_BUCKET: NTSTATUS = -1_073_741_265_i32;
pub const STATUS_ALLOTTED_SPACE_EXCEEDED: NTSTATUS = -1_073_741_671_i32;
pub const STATUS_ALL_SIDS_FILTERED: NTSTATUS = -1_073_740_962_i32;
pub const STATUS_ALL_USER_TRUST_QUOTA_EXCEEDED: NTSTATUS = -1_073_740_798_i32;
pub const STATUS_ALPC_CHECK_COMPLETION_LIST: NTSTATUS = 1_073_741_872_i32;
pub const STATUS_ALREADY_COMMITTED: NTSTATUS = -1_073_741_791_i32;
pub const STATUS_ALREADY_COMPLETE: NTSTATUS = 255_i32;
pub const STATUS_ALREADY_DISCONNECTED: NTSTATUS = -2_147_483_611_i32;
pub const STATUS_ALREADY_HAS_STREAM_ID: NTSTATUS = -1_073_740_530_i32;
pub const STATUS_ALREADY_INITIALIZED: NTSTATUS = -1_073_740_528_i32;
pub const STATUS_ALREADY_REGISTERED: NTSTATUS = -1_073_740_008_i32;
pub const STATUS_ALREADY_WIN32: NTSTATUS = 1_073_741_851_i32;
pub const STATUS_AMBIGUOUS_SYSTEM_DEVICE: NTSTATUS = -1_073_740_719_i32;
pub const STATUS_APC_RETURNED_WHILE_IMPERSONATING: NTSTATUS = -1_073_740_015_i32;
pub const STATUS_APISET_NOT_HOSTED: NTSTATUS = -1_073_740_671_i32;
pub const STATUS_APISET_NOT_PRESENT: NTSTATUS = -1_073_740_670_i32;
pub const STATUS_APPEXEC_APP_COMPAT_BLOCK: NTSTATUS = -1_058_275_320_i32;
pub const STATUS_APPEXEC_CALLER_WAIT_TIMEOUT: NTSTATUS = -1_058_275_319_i32;
pub const STATUS_APPEXEC_CALLER_WAIT_TIMEOUT_LICENSING: NTSTATUS = -1_058_275_317_i32;
pub const STATUS_APPEXEC_CALLER_WAIT_TIMEOUT_RESOURCES: NTSTATUS = -1_058_275_316_i32;
pub const STATUS_APPEXEC_CALLER_WAIT_TIMEOUT_TERMINATION: NTSTATUS = -1_058_275_318_i32;
pub const STATUS_APPEXEC_CONDITION_NOT_SATISFIED: NTSTATUS = -1_058_275_328_i32;
pub const STATUS_APPEXEC_HANDLE_INVALIDATED: NTSTATUS = -1_058_275_327_i32;
pub const STATUS_APPEXEC_HOST_ID_MISMATCH: NTSTATUS = -1_058_275_322_i32;
pub const STATUS_APPEXEC_INVALID_HOST_GENERATION: NTSTATUS = -1_058_275_326_i32;
pub const STATUS_APPEXEC_INVALID_HOST_STATE: NTSTATUS = -1_058_275_324_i32;
pub const STATUS_APPEXEC_NO_DONOR: NTSTATUS = -1_058_275_323_i32;
pub const STATUS_APPEXEC_UNEXPECTED_PROCESS_REGISTRATION: NTSTATUS = -1_058_275_325_i32;
pub const STATUS_APPEXEC_UNKNOWN_USER: NTSTATUS = -1_058_275_321_i32;
pub const STATUS_APPHELP_BLOCK: NTSTATUS = -1_073_740_963_i32;
pub const STATUS_APPX_FILE_NOT_ENCRYPTED: NTSTATUS = -1_073_740_634_i32;
pub const STATUS_APPX_INTEGRITY_FAILURE_CLR_NGEN: NTSTATUS = -1_073_740_673_i32;
pub const STATUS_APP_DATA_CORRUPT: NTSTATUS = -1_073_700_221_i32;
pub const STATUS_APP_DATA_EXPIRED: NTSTATUS = -1_073_700_222_i32;
pub const STATUS_APP_DATA_LIMIT_EXCEEDED: NTSTATUS = -1_073_700_220_i32;
pub const STATUS_APP_DATA_NOT_FOUND: NTSTATUS = -1_073_700_223_i32;
pub const STATUS_APP_DATA_REBOOT_REQUIRED: NTSTATUS = -1_073_700_219_i32;
pub const STATUS_APP_INIT_FAILURE: NTSTATUS = -1_073_741_499_i32;
pub const STATUS_ARBITRATION_UNHANDLED: NTSTATUS = 1_073_741_862_i32;
pub const STATUS_ARRAY_BOUNDS_EXCEEDED: NTSTATUS = -1_073_741_684_i32;
pub const STATUS_ASSERTION_FAILURE: NTSTATUS = -1_073_740_768_i32;
pub const STATUS_ATTACHED_EXECUTABLE_MEMORY_WRITE: NTSTATUS = -1_073_739_995_i32;
pub const STATUS_ATTRIBUTE_NOT_PRESENT: NTSTATUS = -1_073_740_532_i32;
pub const STATUS_AUDIO_ENGINE_NODE_NOT_FOUND: NTSTATUS = -1_069_285_375_i32;
pub const STATUS_AUDITING_DISABLED: NTSTATUS = -1_073_740_970_i32;
pub const STATUS_AUDIT_FAILED: NTSTATUS = -1_073_741_244_i32;
pub const STATUS_AUTHIP_FAILURE: NTSTATUS = -1_073_700_730_i32;
pub const STATUS_AUTH_TAG_MISMATCH: NTSTATUS = -1_073_700_862_i32;
pub const STATUS_BACKUP_CONTROLLER: NTSTATUS = -1_073_741_433_i32;
pub const STATUS_BAD_BINDINGS: NTSTATUS = -1_073_740_965_i32;
pub const STATUS_BAD_CLUSTERS: NTSTATUS = -1_073_739_771_i32;
pub const STATUS_BAD_COMPRESSION_BUFFER: NTSTATUS = -1_073_741_246_i32;
pub const STATUS_BAD_CURRENT_DIRECTORY: NTSTATUS = 1_073_741_831_i32;
pub const STATUS_BAD_DATA: NTSTATUS = -1_073_739_509_i32;
pub const STATUS_BAD_DESCRIPTOR_FORMAT: NTSTATUS = -1_073_741_593_i32;
pub const STATUS_BAD_DEVICE_TYPE: NTSTATUS = -1_073_741_621_i32;
pub const STATUS_BAD_DLL_ENTRYPOINT: NTSTATUS = -1_073_741_231_i32;
pub const STATUS_BAD_FILE_TYPE: NTSTATUS = -1_073_739_517_i32;
pub const STATUS_BAD_FUNCTION_TABLE: NTSTATUS = -1_073_741_569_i32;
pub const STATUS_BAD_IMPERSONATION_LEVEL: NTSTATUS = -1_073_741_659_i32;
pub const STATUS_BAD_INHERITANCE_ACL: NTSTATUS = -1_073_741_699_i32;
pub const STATUS_BAD_INITIAL_PC: NTSTATUS = -1_073_741_814_i32;
pub const STATUS_BAD_INITIAL_STACK: NTSTATUS = -1_073_741_815_i32;
pub const STATUS_BAD_KEY: NTSTATUS = -1_073_739_510_i32;
pub const STATUS_BAD_LOGON_SESSION_STATE: NTSTATUS = -1_073_741_564_i32;
pub const STATUS_BAD_MASTER_BOOT_RECORD: NTSTATUS = -1_073_741_655_i32;
pub const STATUS_BAD_MCFG_TABLE: NTSTATUS = -1_073_739_512_i32;
pub const STATUS_BAD_NETWORK_NAME: NTSTATUS = -1_073_741_620_i32;
pub const STATUS_BAD_NETWORK_PATH: NTSTATUS = -1_073_741_634_i32;
pub const STATUS_BAD_REMOTE_ADAPTER: NTSTATUS = -1_073_741_627_i32;
pub const STATUS_BAD_SERVICE_ENTRYPOINT: NTSTATUS = -1_073_741_230_i32;
pub const STATUS_BAD_STACK: NTSTATUS = -1_073_741_784_i32;
pub const STATUS_BAD_TOKEN_TYPE: NTSTATUS = -1_073_741_656_i32;
pub const STATUS_BAD_VALIDATION_CLASS: NTSTATUS = -1_073_741_657_i32;
pub const STATUS_BAD_WORKING_SET_LIMIT: NTSTATUS = -1_073_741_748_i32;
pub const STATUS_BCD_NOT_ALL_ENTRIES_IMPORTED: NTSTATUS = -2_143_748_095_i32;
pub const STATUS_BCD_NOT_ALL_ENTRIES_SYNCHRONIZED: NTSTATUS = -2_143_748_093_i32;
pub const STATUS_BCD_TOO_MANY_ELEMENTS: NTSTATUS = -1_070_006_270_i32;
pub const STATUS_BEGINNING_OF_MEDIA: NTSTATUS = -2_147_483_617_i32;
pub const STATUS_BEYOND_VDL: NTSTATUS = -1_073_740_750_i32;
pub const STATUS_BIOS_FAILED_TO_CONNECT_INTERRUPT: NTSTATUS = -1_073_741_458_i32;
pub const STATUS_BIZRULES_NOT_ENABLED: NTSTATUS = 1_073_741_876_i32;
pub const STATUS_BLOCKED_BY_PARENTAL_CONTROLS: NTSTATUS = -1_073_740_664_i32;
pub const STATUS_BLOCK_SHARED: NTSTATUS = -1_073_739_499_i32;
pub const STATUS_BLOCK_SOURCE_WEAK_REFERENCE_INVALID: NTSTATUS = -1_073_739_501_i32;
pub const STATUS_BLOCK_TARGET_WEAK_REFERENCE_INVALID: NTSTATUS = -1_073_739_500_i32;
pub const STATUS_BLOCK_TOO_MANY_REFERENCES: NTSTATUS = -1_073_740_660_i32;
pub const STATUS_BLOCK_WEAK_REFERENCE_INVALID: NTSTATUS = -1_073_739_502_i32;
pub const STATUS_BREAKPOINT: NTSTATUS = -2_147_483_645_i32;
pub const STATUS_BTH_ATT_ATTRIBUTE_NOT_FOUND: NTSTATUS = -1_069_416_438_i32;
pub const STATUS_BTH_ATT_ATTRIBUTE_NOT_LONG: NTSTATUS = -1_069_416_437_i32;
pub const STATUS_BTH_ATT_INSUFFICIENT_AUTHENTICATION: NTSTATUS = -1_069_416_443_i32;
pub const STATUS_BTH_ATT_INSUFFICIENT_AUTHORIZATION: NTSTATUS = -1_069_416_440_i32;
pub const STATUS_BTH_ATT_INSUFFICIENT_ENCRYPTION: NTSTATUS = -1_069_416_433_i32;
pub const STATUS_BTH_ATT_INSUFFICIENT_ENCRYPTION_KEY_SIZE: NTSTATUS = -1_069_416_436_i32;
pub const STATUS_BTH_ATT_INSUFFICIENT_RESOURCES: NTSTATUS = -1_069_416_431_i32;
pub const STATUS_BTH_ATT_INVALID_ATTRIBUTE_VALUE_LENGTH: NTSTATUS = -1_069_416_435_i32;
pub const STATUS_BTH_ATT_INVALID_HANDLE: NTSTATUS = -1_069_416_447_i32;
pub const STATUS_BTH_ATT_INVALID_OFFSET: NTSTATUS = -1_069_416_441_i32;
pub const STATUS_BTH_ATT_INVALID_PDU: NTSTATUS = -1_069_416_444_i32;
pub const STATUS_BTH_ATT_PREPARE_QUEUE_FULL: NTSTATUS = -1_069_416_439_i32;
pub const STATUS_BTH_ATT_READ_NOT_PERMITTED: NTSTATUS = -1_069_416_446_i32;
pub const STATUS_BTH_ATT_REQUEST_NOT_SUPPORTED: NTSTATUS = -1_069_416_442_i32;
pub const STATUS_BTH_ATT_UNKNOWN_ERROR: NTSTATUS = -1_069_412_352_i32;
pub const STATUS_BTH_ATT_UNLIKELY: NTSTATUS = -1_069_416_434_i32;
pub const STATUS_BTH_ATT_UNSUPPORTED_GROUP_TYPE: NTSTATUS = -1_069_416_432_i32;
pub const STATUS_BTH_ATT_WRITE_NOT_PERMITTED: NTSTATUS = -1_069_416_445_i32;
pub const STATUS_BUFFER_ALL_ZEROS: NTSTATUS = 279_i32;
pub const STATUS_BUFFER_OVERFLOW: NTSTATUS = -2_147_483_643_i32;
pub const STATUS_BUFFER_TOO_SMALL: NTSTATUS = -1_073_741_789_i32;
pub const STATUS_BUS_RESET: NTSTATUS = -2_147_483_619_i32;
pub const STATUS_BYPASSIO_FLT_NOT_SUPPORTED: NTSTATUS = -1_073_740_590_i32;
pub const STATUS_CACHE_PAGE_LOCKED: NTSTATUS = 277_i32;
pub const STATUS_CALLBACK_BYPASS: NTSTATUS = -1_073_740_541_i32;
pub const STATUS_CALLBACK_INVOKE_INLINE: NTSTATUS = -1_073_740_661_i32;
pub const STATUS_CALLBACK_POP_STACK: NTSTATUS = -1_073_740_765_i32;
pub const STATUS_CALLBACK_RETURNED_LANG: NTSTATUS = -1_073_740_001_i32;
pub const STATUS_CALLBACK_RETURNED_LDR_LOCK: NTSTATUS = -1_073_740_002_i32;
pub const STATUS_CALLBACK_RETURNED_PRI_BACK: NTSTATUS = -1_073_740_000_i32;
pub const STATUS_CALLBACK_RETURNED_THREAD_AFFINITY: NTSTATUS = -1_073_739_999_i32;
pub const STATUS_CALLBACK_RETURNED_THREAD_PRIORITY: NTSTATUS = -1_073_740_005_i32;
pub const STATUS_CALLBACK_RETURNED_TRANSACTION: NTSTATUS = -1_073_740_003_i32;
pub const STATUS_CALLBACK_RETURNED_WHILE_IMPERSONATING: NTSTATUS = -1_073_740_016_i32;
pub const STATUS_CANCELLED: NTSTATUS = -1_073_741_536_i32;
pub const STATUS_CANNOT_ABORT_TRANSACTIONS: NTSTATUS = -1_072_103_347_i32;
pub const STATUS_CANNOT_ACCEPT_TRANSACTED_WORK: NTSTATUS = -1_072_103_348_i32;
pub const STATUS_CANNOT_BREAK_OPLOCK: NTSTATUS = -1_073_739_511_i32;
pub const STATUS_CANNOT_DELETE: NTSTATUS = -1_073_741_535_i32;
pub const STATUS_CANNOT_EXECUTE_FILE_IN_TRANSACTION: NTSTATUS = -1_072_103_356_i32;
pub const STATUS_CANNOT_GRANT_REQUESTED_OPLOCK: NTSTATUS = -2_147_483_602_i32;
pub const STATUS_CANNOT_IMPERSONATE: NTSTATUS = -1_073_741_555_i32;
pub const STATUS_CANNOT_LOAD_REGISTRY_FILE: NTSTATUS = -1_073_741_288_i32;
pub const STATUS_CANNOT_MAKE: NTSTATUS = -1_073_741_078_i32;
pub const STATUS_CANNOT_SWITCH_RUNLEVEL: NTSTATUS = -1_073_700_543_i32;
pub const STATUS_CANT_ACCESS_DOMAIN_INFO: NTSTATUS = -1_073_741_606_i32;
pub const STATUS_CANT_BREAK_TRANSACTIONAL_DEPENDENCY: NTSTATUS = -1_072_103_369_i32;
pub const STATUS_CANT_CLEAR_ENCRYPTION_FLAG: NTSTATUS = -1_073_740_616_i32;
pub const STATUS_CANT_CREATE_MORE_STREAM_MINIVERSIONS: NTSTATUS = -1_072_103_386_i32;
pub const STATUS_CANT_CROSS_RM_BOUNDARY: NTSTATUS = -1_072_103_368_i32;
pub const STATUS_CANT_DISABLE_MANDATORY: NTSTATUS = -1_073_741_731_i32;
pub const STATUS_CANT_ENABLE_DENY_ONLY: NTSTATUS = -1_073_741_133_i32;
pub const STATUS_CANT_OPEN_ANONYMOUS: NTSTATUS = -1_073_741_658_i32;
pub const STATUS_CANT_OPEN_MINIVERSION_WITH_MODIFY_INTENT: NTSTATUS = -1_072_103_387_i32;
pub const STATUS_CANT_RECOVER_WITH_HANDLE_OPEN: NTSTATUS = -2_145_845_199_i32;
pub const STATUS_CANT_TERMINATE_SELF: NTSTATUS = -1_073_741_605_i32;
pub const STATUS_CANT_WAIT: NTSTATUS = -1_073_741_608_i32;
pub const STATUS_CARDBUS_NOT_SUPPORTED: NTSTATUS = 1_073_741_863_i32;
pub const STATUS_CASE_DIFFERING_NAMES_IN_DIR: NTSTATUS = -1_073_740_621_i32;
pub const STATUS_CASE_SENSITIVE_PATH: NTSTATUS = -1_073_740_614_i32;
pub const STATUS_CC_NEEDS_CALLBACK_SECTION_DRAIN: NTSTATUS = -1_073_700_856_i32;
pub const STATUS_CERTIFICATE_MAPPING_NOT_UNIQUE: NTSTATUS = -1_073_740_012_i32;
pub const STATUS_CERTIFICATE_VALIDATION_PREFERENCE_CONFLICT: NTSTATUS = -1_073_741_387_i32;
pub const STATUS_CHECKING_FILE_SYSTEM: NTSTATUS = 1_073_741_844_i32;
pub const STATUS_CHECKOUT_REQUIRED: NTSTATUS = -1_073_739_518_i32;
pub const STATUS_CHILD_MUST_BE_VOLATILE: NTSTATUS = -1_073_741_439_i32;
pub const STATUS_CHILD_PROCESS_BLOCKED: NTSTATUS = -1_073_740_643_i32;
pub const STATUS_CIMFS_IMAGE_CORRUPT: NTSTATUS = -1_073_692_671_i32;
pub const STATUS_CIMFS_IMAGE_VERSION_NOT_SUPPORTED: NTSTATUS = -1_073_692_670_i32;
pub const STATUS_CLEANER_CARTRIDGE_INSTALLED: NTSTATUS = -2_147_483_609_i32;
pub const STATUS_CLIENT_SERVER_PARAMETERS_INVALID: NTSTATUS = -1_073_741_277_i32;
pub const STATUS_CLIP_DEVICE_LICENSE_MISSING: NTSTATUS = -1_058_406_397_i32;
pub const STATUS_CLIP_KEYHOLDER_LICENSE_MISSING_OR_INVALID: NTSTATUS = -1_058_406_395_i32;
pub const STATUS_CLIP_LICENSE_DEVICE_ID_MISMATCH: NTSTATUS = -1_058_406_390_i32;
pub const STATUS_CLIP_LICENSE_EXPIRED: NTSTATUS = -1_058_406_394_i32;
pub const STATUS_CLIP_LICENSE_HARDWARE_ID_OUT_OF_TOLERANCE: NTSTATUS = -1_058_406_391_i32;
pub const STATUS_CLIP_LICENSE_INVALID_SIGNATURE: NTSTATUS = -1_058_406_396_i32;
pub const STATUS_CLIP_LICENSE_NOT_FOUND: NTSTATUS = -1_058_406_398_i32;
pub const STATUS_CLIP_LICENSE_NOT_SIGNED: NTSTATUS = -1_058_406_392_i32;
pub const STATUS_CLIP_LICENSE_SIGNED_BY_UNKNOWN_SOURCE: NTSTATUS = -1_058_406_393_i32;
pub const STATUS_CLOUD_FILE_ACCESS_DENIED: NTSTATUS = -1_073_688_808_i32;
pub const STATUS_CLOUD_FILE_ALREADY_CONNECTED: NTSTATUS = -1_073_688_823_i32;
pub const STATUS_CLOUD_FILE_AUTHENTICATION_FAILED: NTSTATUS = -1_073_688_817_i32;
pub const STATUS_CLOUD_FILE_CONNECTED_PROVIDER_ONLY: NTSTATUS = -1_073_688_819_i32;
pub const STATUS_CLOUD_FILE_DEHYDRATION_DISALLOWED: NTSTATUS = -1_073_688_800_i32;
pub const STATUS_CLOUD_FILE_INCOMPATIBLE_HARDLINKS: NTSTATUS = -1_073_688_807_i32;
pub const STATUS_CLOUD_FILE_INSUFFICIENT_RESOURCES: NTSTATUS = -1_073_688_816_i32;
pub const STATUS_CLOUD_FILE_INVALID_REQUEST: NTSTATUS = -1_073_688_821_i32;
pub const STATUS_CLOUD_FILE_IN_USE: NTSTATUS = -1_073_688_812_i32;
pub const STATUS_CLOUD_FILE_METADATA_CORRUPT: NTSTATUS = -1_073_688_830_i32;
pub const STATUS_CLOUD_FILE_METADATA_TOO_LARGE: NTSTATUS = -1_073_688_829_i32;
pub const STATUS_CLOUD_FILE_NETWORK_UNAVAILABLE: NTSTATUS = -1_073_688_815_i32;
pub const STATUS_CLOUD_FILE_NOT_IN_SYNC: NTSTATUS = -1_073_688_824_i32;
pub const STATUS_CLOUD_FILE_NOT_SUPPORTED: NTSTATUS = -1_073_688_822_i32;
pub const STATUS_CLOUD_FILE_NOT_UNDER_SYNC_ROOT: NTSTATUS = -1_073_688_813_i32;
pub const STATUS_CLOUD_FILE_PINNED: NTSTATUS = -1_073_688_811_i32;
pub const STATUS_CLOUD_FILE_PROPERTY_BLOB_CHECKSUM_MISMATCH: NTSTATUS = -2_147_430_656_i32;
pub const STATUS_CLOUD_FILE_PROPERTY_BLOB_TOO_LARGE: NTSTATUS = -2_147_430_652_i32;
pub const STATUS_CLOUD_FILE_PROPERTY_CORRUPT: NTSTATUS = -1_073_688_809_i32;
pub const STATUS_CLOUD_FILE_PROPERTY_LOCK_CONFLICT: NTSTATUS = -1_073_688_806_i32;
pub const STATUS_CLOUD_FILE_PROPERTY_VERSION_NOT_SUPPORTED: NTSTATUS = -1_073_688_826_i32;
pub const STATUS_CLOUD_FILE_PROVIDER_NOT_RUNNING: NTSTATUS = -1_073_688_831_i32;
pub const STATUS_CLOUD_FILE_PROVIDER_TERMINATED: NTSTATUS = -1_073_688_803_i32;
pub const STATUS_CLOUD_FILE_READ_ONLY_VOLUME: NTSTATUS = -1_073_688_820_i32;
pub const STATUS_CLOUD_FILE_REQUEST_ABORTED: NTSTATUS = -1_073_688_810_i32;
pub const STATUS_CLOUD_FILE_REQUEST_CANCELED: NTSTATUS = -1_073_688_805_i32;
pub const STATUS_CLOUD_FILE_REQUEST_TIMEOUT: NTSTATUS = -1_073_688_801_i32;
pub const STATUS_CLOUD_FILE_SYNC_ROOT_METADATA_CORRUPT: NTSTATUS = -1_073_688_832_i32;
pub const STATUS_CLOUD_FILE_TOO_MANY_PROPERTY_BLOBS: NTSTATUS = -2_147_430_651_i32;
pub const STATUS_CLOUD_FILE_UNSUCCESSFUL: NTSTATUS = -1_073_688_814_i32;
pub const STATUS_CLOUD_FILE_US_MESSAGE_TIMEOUT: NTSTATUS = -1_073_688_799_i32;
pub const STATUS_CLOUD_FILE_VALIDATION_FAILED: NTSTATUS = -1_073_688_818_i32;
pub const STATUS_CLUSTER_CAM_TICKET_REPLAY_DETECTED: NTSTATUS = -1_072_496_591_i32;
pub const STATUS_CLUSTER_CSV_AUTO_PAUSE_ERROR: NTSTATUS = -1_072_496_607_i32;
pub const STATUS_CLUSTER_CSV_INVALID_HANDLE: NTSTATUS = -1_072_496_599_i32;
pub const STATUS_CLUSTER_CSV_NOT_REDIRECTED: NTSTATUS = -1_072_496_605_i32;
pub const STATUS_CLUSTER_CSV_NO_SNAPSHOTS: NTSTATUS = -1_072_496_601_i32;
pub const STATUS_CLUSTER_CSV_READ_OPLOCK_BREAK_IN_PROGRESS: NTSTATUS = -1_072_496_608_i32;
pub const STATUS_CLUSTER_CSV_REDIRECTED: NTSTATUS = -1_072_496_606_i32;
pub const STATUS_CLUSTER_CSV_SNAPSHOT_CREATION_IN_PROGRESS: NTSTATUS = -1_072_496_603_i32;
pub const STATUS_CLUSTER_CSV_SUPPORTED_ONLY_ON_COORDINATOR: NTSTATUS = -1_072_496_592_i32;
pub const STATUS_CLUSTER_CSV_VOLUME_DRAINING: NTSTATUS = -1_072_496_604_i32;
pub const STATUS_CLUSTER_CSV_VOLUME_DRAINING_SUCCEEDED_DOWNLEVEL: NTSTATUS = -1_072_496_602_i32;
pub const STATUS_CLUSTER_CSV_VOLUME_NOT_LOCAL: NTSTATUS = -1_072_496_615_i32;
pub const STATUS_CLUSTER_INVALID_NETWORK: NTSTATUS = -1_072_496_624_i32;
pub const STATUS_CLUSTER_INVALID_NETWORK_PROVIDER: NTSTATUS = -1_072_496_629_i32;
pub const STATUS_CLUSTER_INVALID_NODE: NTSTATUS = -1_072_496_639_i32;
pub const STATUS_CLUSTER_INVALID_REQUEST: NTSTATUS = -1_072_496_630_i32;
pub const STATUS_CLUSTER_JOIN_IN_PROGRESS: NTSTATUS = -1_072_496_637_i32;
pub const STATUS_CLUSTER_JOIN_NOT_IN_PROGRESS: NTSTATUS = -1_072_496_625_i32;
pub const STATUS_CLUSTER_LOCAL_NODE_NOT_FOUND: NTSTATUS = -1_072_496_635_i32;
pub const STATUS_CLUSTER_NETINTERFACE_EXISTS: NTSTATUS = -1_072_496_632_i32;
pub const STATUS_CLUSTER_NETINTERFACE_NOT_FOUND: NTSTATUS = -1_072_496_631_i32;
pub const STATUS_CLUSTER_NETWORK_ALREADY_OFFLINE: NTSTATUS = -2_146_238_460_i32;
pub const STATUS_CLUSTER_NETWORK_ALREADY_ONLINE: NTSTATUS = -2_146_238_461_i32;
pub const STATUS_CLUSTER_NETWORK_EXISTS: NTSTATUS = -1_072_496_634_i32;
pub const STATUS_CLUSTER_NETWORK_NOT_FOUND: NTSTATUS = -1_072_496_633_i32;
pub const STATUS_CLUSTER_NETWORK_NOT_INTERNAL: NTSTATUS = -1_072_496_618_i32;
pub const STATUS_CLUSTER_NODE_ALREADY_DOWN: NTSTATUS = -2_146_238_462_i32;
pub const STATUS_CLUSTER_NODE_ALREADY_MEMBER: NTSTATUS = -2_146_238_459_i32;
pub const STATUS_CLUSTER_NODE_ALREADY_UP: NTSTATUS = -2_146_238_463_i32;
pub const STATUS_CLUSTER_NODE_DOWN: NTSTATUS = -1_072_496_628_i32;
pub const STATUS_CLUSTER_NODE_EXISTS: NTSTATUS = -1_072_496_638_i32;
pub const STATUS_CLUSTER_NODE_NOT_FOUND: NTSTATUS = -1_072_496_636_i32;
pub const STATUS_CLUSTER_NODE_NOT_MEMBER: NTSTATUS = -1_072_496_626_i32;
pub const STATUS_CLUSTER_NODE_NOT_PAUSED: NTSTATUS = -1_072_496_620_i32;
pub const STATUS_CLUSTER_NODE_PAUSED: NTSTATUS = -1_072_496_621_i32;
pub const STATUS_CLUSTER_NODE_UNREACHABLE: NTSTATUS = -1_072_496_627_i32;
pub const STATUS_CLUSTER_NODE_UP: NTSTATUS = -1_072_496_622_i32;
pub const STATUS_CLUSTER_NON_CSV_PATH: NTSTATUS = -1_072_496_616_i32;
pub const STATUS_CLUSTER_NO_NET_ADAPTERS: NTSTATUS = -1_072_496_623_i32;
pub const STATUS_CLUSTER_NO_SECURITY_CONTEXT: NTSTATUS = -1_072_496_619_i32;
pub const STATUS_CLUSTER_POISONED: NTSTATUS = -1_072_496_617_i32;
pub const STATUS_COMMITMENT_LIMIT: NTSTATUS = -1_073_741_523_i32;
pub const STATUS_COMMITMENT_MINIMUM: NTSTATUS = -1_073_741_112_i32;
pub const STATUS_COMPRESSED_FILE_NOT_SUPPORTED: NTSTATUS = -1_073_740_677_i32;
pub const STATUS_COMPRESSION_DISABLED: NTSTATUS = -1_073_740_762_i32;
pub const STATUS_COMPRESSION_NOT_ALLOWED_IN_TRANSACTION: NTSTATUS = -1_072_103_338_i32;
pub const STATUS_COMPRESSION_NOT_BENEFICIAL: NTSTATUS = -1_073_740_689_i32;
pub const STATUS_CONFLICTING_ADDRESSES: NTSTATUS = -1_073_741_800_i32;
pub const STATUS_CONNECTION_ABORTED: NTSTATUS = -1_073_741_247_i32;
pub const STATUS_CONNECTION_ACTIVE: NTSTATUS = -1_073_741_253_i32;
pub const STATUS_CONNECTION_COUNT_LIMIT: NTSTATUS = -1_073_741_242_i32;
pub const STATUS_CONNECTION_DISCONNECTED: NTSTATUS = -1_073_741_300_i32;
pub const STATUS_CONNECTION_INVALID: NTSTATUS = -1_073_741_254_i32;
pub const STATUS_CONNECTION_IN_USE: NTSTATUS = -1_073_741_560_i32;
pub const STATUS_CONNECTION_REFUSED: NTSTATUS = -1_073_741_258_i32;
pub const STATUS_CONNECTION_RESET: NTSTATUS = -1_073_741_299_i32;
pub const STATUS_CONTAINER_ASSIGNED: NTSTATUS = -1_073_740_536_i32;
pub const STATUS_CONTENT_BLOCKED: NTSTATUS = -1_073_739_772_i32;
pub const STATUS_CONTEXT_MISMATCH: NTSTATUS = -1_073_740_007_i32;
pub const STATUS_CONTEXT_STOWED_EXCEPTION: NTSTATUS = -1_073_741_188_i32;
pub const STATUS_CONTROL_C_EXIT: NTSTATUS = -1_073_741_510_i32;
pub const STATUS_CONTROL_STACK_VIOLATION: NTSTATUS = -1_073_741_390_i32;
pub const STATUS_CONVERT_TO_LARGE: NTSTATUS = -1_073_741_268_i32;
pub const STATUS_COPY_PROTECTION_FAILURE: NTSTATUS = -1_073_741_051_i32;
pub const STATUS_CORRUPT_LOG_CLEARED: NTSTATUS = -1_073_739_763_i32;
pub const STATUS_CORRUPT_LOG_CORRUPTED: NTSTATUS = -1_073_739_766_i32;
pub const STATUS_CORRUPT_LOG_DELETED_FULL: NTSTATUS = -1_073_739_764_i32;
pub const STATUS_CORRUPT_LOG_OVERFULL: NTSTATUS = -1_073_739_767_i32;
pub const STATUS_CORRUPT_LOG_UNAVAILABLE: NTSTATUS = -1_073_739_765_i32;
pub const STATUS_CORRUPT_LOG_UPLEVEL_RECORDS: NTSTATUS = -1_073_739_759_i32;
pub const STATUS_CORRUPT_SYSTEM_FILE: NTSTATUS = -1_073_741_116_i32;
pub const STATUS_COULD_NOT_INTERPRET: NTSTATUS = -1_073_741_639_i32;
pub const STATUS_COULD_NOT_RESIZE_LOG: NTSTATUS = -2_145_845_239_i32;
pub const STATUS_CPU_SET_INVALID: NTSTATUS = -1_073_741_393_i32;
pub const STATUS_CRASH_DUMP: NTSTATUS = 278_i32;
pub const STATUS_CRC_ERROR: NTSTATUS = -1_073_741_761_i32;
pub const STATUS_CRED_REQUIRES_CONFIRMATION: NTSTATUS = -1_073_740_736_i32;
pub const STATUS_CRM_PROTOCOL_ALREADY_EXISTS: NTSTATUS = -1_072_103_409_i32;
pub const STATUS_CRM_PROTOCOL_NOT_FOUND: NTSTATUS = -1_072_103_407_i32;
pub const STATUS_CROSSREALM_DELEGATION_FAILURE: NTSTATUS = -1_073_740_789_i32;
pub const STATUS_CROSS_PARTITION_VIOLATION: NTSTATUS = -1_073_740_277_i32;
pub const STATUS_CRYPTO_SYSTEM_INVALID: NTSTATUS = -1_073_741_069_i32;
pub const STATUS_CSS_AUTHENTICATION_FAILURE: NTSTATUS = -1_073_741_050_i32;
pub const STATUS_CSS_KEY_NOT_ESTABLISHED: NTSTATUS = -1_073_741_048_i32;
pub const STATUS_CSS_KEY_NOT_PRESENT: NTSTATUS = -1_073_741_049_i32;
pub const STATUS_CSS_REGION_MISMATCH: NTSTATUS = -1_073_741_046_i32;
pub const STATUS_CSS_RESETS_EXHAUSTED: NTSTATUS = -1_073_741_045_i32;
pub const STATUS_CSS_SCRAMBLED_SECTOR: NTSTATUS = -1_073_741_047_i32;
pub const STATUS_CSV_IO_PAUSE_TIMEOUT: NTSTATUS = -1_072_496_600_i32;
pub const STATUS_CS_ENCRYPTION_EXISTING_ENCRYPTED_FILE: NTSTATUS = -1_073_740_733_i32;
pub const STATUS_CS_ENCRYPTION_FILE_NOT_CSE: NTSTATUS = -1_073_740_731_i32;
pub const STATUS_CS_ENCRYPTION_INVALID_SERVER_RESPONSE: NTSTATUS = -1_073_740_735_i32;
pub const STATUS_CS_ENCRYPTION_NEW_ENCRYPTED_FILE: NTSTATUS = -1_073_740_732_i32;
pub const STATUS_CS_ENCRYPTION_UNSUPPORTED_SERVER: NTSTATUS = -1_073_740_734_i32;
pub const STATUS_CTLOG_INCONSISTENT_TRACKING_FILE: NTSTATUS = -1_069_940_700_i32;
pub const STATUS_CTLOG_INVALID_TRACKING_STATE: NTSTATUS = -1_069_940_701_i32;
pub const STATUS_CTLOG_LOGFILE_SIZE_EXCEEDED_MAXSIZE: NTSTATUS = -1_069_940_703_i32;
pub const STATUS_CTLOG_TRACKING_NOT_INITIALIZED: NTSTATUS = -1_069_940_704_i32;
pub const STATUS_CTLOG_VHD_CHANGED_OFFLINE: NTSTATUS = -1_069_940_702_i32;
pub const STATUS_CTL_FILE_NOT_SUPPORTED: NTSTATUS = -1_073_741_737_i32;
pub const STATUS_CTX_BAD_VIDEO_MODE: NTSTATUS = -1_073_086_440_i32;
pub const STATUS_CTX_CDM_CONNECT: NTSTATUS = 1_074_397_188_i32;
pub const STATUS_CTX_CDM_DISCONNECT: NTSTATUS = 1_074_397_189_i32;
pub const STATUS_CTX_CLIENT_LICENSE_IN_USE: NTSTATUS = -1_073_086_412_i32;
pub const STATUS_CTX_CLIENT_LICENSE_NOT_SET: NTSTATUS = -1_073_086_413_i32;
pub const STATUS_CTX_CLIENT_QUERY_TIMEOUT: NTSTATUS = -1_073_086_426_i32;
pub const STATUS_CTX_CLOSE_PENDING: NTSTATUS = -1_073_086_458_i32;
pub const STATUS_CTX_CONSOLE_CONNECT: NTSTATUS = -1_073_086_424_i32;
pub const STATUS_CTX_CONSOLE_DISCONNECT: NTSTATUS = -1_073_086_425_i32;
pub const STATUS_CTX_GRAPHICS_INVALID: NTSTATUS = -1_073_086_430_i32;
pub const STATUS_CTX_INVALID_MODEMNAME: NTSTATUS = -1_073_086_455_i32;
pub const STATUS_CTX_INVALID_PD: NTSTATUS = -1_073_086_462_i32;
pub const STATUS_CTX_INVALID_WD: NTSTATUS = -1_073_086_418_i32;
pub const STATUS_CTX_LICENSE_CLIENT_INVALID: NTSTATUS = -1_073_086_446_i32;
pub const STATUS_CTX_LICENSE_EXPIRED: NTSTATUS = -1_073_086_444_i32;
pub const STATUS_CTX_LICENSE_NOT_AVAILABLE: NTSTATUS = -1_073_086_445_i32;
pub const STATUS_CTX_LOGON_DISABLED: NTSTATUS = -1_073_086_409_i32;
pub const STATUS_CTX_MODEM_INF_NOT_FOUND: NTSTATUS = -1_073_086_456_i32;
pub const STATUS_CTX_MODEM_RESPONSE_BUSY: NTSTATUS = -1_073_086_450_i32;
pub const STATUS_CTX_MODEM_RESPONSE_NO_CARRIER: NTSTATUS = -1_073_086_452_i32;
pub const STATUS_CTX_MODEM_RESPONSE_NO_DIALTONE: NTSTATUS = -1_073_086_451_i32;
pub const STATUS_CTX_MODEM_RESPONSE_TIMEOUT: NTSTATUS = -1_073_086_453_i32;
pub const STATUS_CTX_MODEM_RESPONSE_VOICE: NTSTATUS = -1_073_086_449_i32;
pub const STATUS_CTX_NOT_CONSOLE: NTSTATUS = -1_073_086_428_i32;
pub const STATUS_CTX_NO_OUTBUF: NTSTATUS = -1_073_086_457_i32;
pub const STATUS_CTX_PD_NOT_FOUND: NTSTATUS = -1_073_086_461_i32;
pub const STATUS_CTX_RESPONSE_ERROR: NTSTATUS = -1_073_086_454_i32;
pub const STATUS_CTX_SECURITY_LAYER_ERROR: NTSTATUS = -1_073_086_408_i32;
pub const STATUS_CTX_SHADOW_DENIED: NTSTATUS = -1_073_086_422_i32;
pub const STATUS_CTX_SHADOW_DISABLED: NTSTATUS = -1_073_086_415_i32;
pub const STATUS_CTX_SHADOW_ENDED_BY_MODE_CHANGE: NTSTATUS = -1_073_086_411_i32;
pub const STATUS_CTX_SHADOW_INVALID: NTSTATUS = -1_073_086_416_i32;
pub const STATUS_CTX_SHADOW_NOT_RUNNING: NTSTATUS = -1_073_086_410_i32;
pub const STATUS_CTX_TD_ERROR: NTSTATUS = -1_073_086_448_i32;
pub const STATUS_CTX_WD_NOT_FOUND: NTSTATUS = -1_073_086_417_i32;
pub const STATUS_CTX_WINSTATION_ACCESS_DENIED: NTSTATUS = -1_073_086_421_i32;
pub const STATUS_CTX_WINSTATION_BUSY: NTSTATUS = -1_073_086_441_i32;
pub const STATUS_CTX_WINSTATION_NAME_COLLISION: NTSTATUS = -1_073_086_442_i32;
pub const STATUS_CTX_WINSTATION_NAME_INVALID: NTSTATUS = -1_073_086_463_i32;
pub const STATUS_CTX_WINSTATION_NOT_FOUND: NTSTATUS = -1_073_086_443_i32;
pub const STATUS_CURRENT_DOMAIN_NOT_ALLOWED: NTSTATUS = -1_073_741_079_i32;
pub const STATUS_CURRENT_TRANSACTION_NOT_VALID: NTSTATUS = -1_072_103_400_i32;
pub const STATUS_DATATYPE_MISALIGNMENT: NTSTATUS = -2_147_483_646_i32;
pub const STATUS_DATATYPE_MISALIGNMENT_ERROR: NTSTATUS = -1_073_741_115_i32;
pub const STATUS_DATA_CHECKSUM_ERROR: NTSTATUS = -1_073_740_688_i32;
pub const STATUS_DATA_ERROR: NTSTATUS = -1_073_741_762_i32;
pub const STATUS_DATA_LATE_ERROR: NTSTATUS = -1_073_741_763_i32;
pub const STATUS_DATA_LOST_REPAIR: NTSTATUS = -2_147_481_597_i32;
pub const STATUS_DATA_NOT_ACCEPTED: NTSTATUS = -1_073_741_285_i32;
pub const STATUS_DATA_OVERRUN: NTSTATUS = -1_073_741_764_i32;
pub const STATUS_DATA_OVERWRITTEN: NTSTATUS = 304_i32;
pub const STATUS_DAX_MAPPING_EXISTS: NTSTATUS = -1_073_740_644_i32;
pub const STATUS_DEBUGGER_INACTIVE: NTSTATUS = -1_073_740_972_i32;
pub const STATUS_DEBUG_ATTACH_FAILED: NTSTATUS = -1_073_741_287_i32;
pub const STATUS_DECRYPTION_FAILED: NTSTATUS = -1_073_741_173_i32;
pub const STATUS_DELAY_LOAD_FAILED: NTSTATUS = -1_073_740_782_i32;
pub const STATUS_DELETE_PENDING: NTSTATUS = -1_073_741_738_i32;
pub const STATUS_DESTINATION_ELEMENT_FULL: NTSTATUS = -1_073_741_180_i32;
pub const STATUS_DEVICE_ALREADY_ATTACHED: NTSTATUS = -1_073_741_768_i32;
pub const STATUS_DEVICE_BUSY: NTSTATUS = -2_147_483_631_i32;
pub const STATUS_DEVICE_CONFIGURATION_ERROR: NTSTATUS = -1_073_741_438_i32;
pub const STATUS_DEVICE_DATA_ERROR: NTSTATUS = -1_073_741_668_i32;
pub const STATUS_DEVICE_DOES_NOT_EXIST: NTSTATUS = -1_073_741_632_i32;
pub const STATUS_DEVICE_DOOR_OPEN: NTSTATUS = -2_147_482_999_i32;
pub const STATUS_DEVICE_ENUMERATION_ERROR: NTSTATUS = -1_073_740_954_i32;
pub const STATUS_DEVICE_FEATURE_NOT_SUPPORTED: NTSTATUS = -1_073_740_701_i32;
pub const STATUS_DEVICE_HARDWARE_ERROR: NTSTATUS = -1_073_740_669_i32;
pub const STATUS_DEVICE_HINT_NAME_BUFFER_TOO_SMALL: NTSTATUS = -1_073_740_650_i32;
pub const STATUS_DEVICE_HUNG: NTSTATUS = -1_073_740_537_i32;
pub const STATUS_DEVICE_INSUFFICIENT_RESOURCES: NTSTATUS = -1_073_740_696_i32;
pub const STATUS_DEVICE_IN_MAINTENANCE: NTSTATUS = -1_073_740_647_i32;
pub const STATUS_DEVICE_NOT_CONNECTED: NTSTATUS = -1_073_741_667_i32;
pub const STATUS_DEVICE_NOT_PARTITIONED: NTSTATUS = -1_073_741_452_i32;
pub const STATUS_DEVICE_NOT_READY: NTSTATUS = -1_073_741_661_i32;
pub const STATUS_DEVICE_OFF_LINE: NTSTATUS = -2_147_483_632_i32;
pub const STATUS_DEVICE_PAPER_EMPTY: NTSTATUS = -2_147_483_634_i32;
pub const STATUS_DEVICE_POWERED_OFF: NTSTATUS = -2_147_483_633_i32;
pub const STATUS_DEVICE_POWER_CYCLE_REQUIRED: NTSTATUS = -2_147_483_599_i32;
pub const STATUS_DEVICE_POWER_FAILURE: NTSTATUS = -1_073_741_666_i32;
pub const STATUS_DEVICE_PROTOCOL_ERROR: NTSTATUS = -1_073_741_434_i32;
pub const STATUS_DEVICE_REMOVED: NTSTATUS = -1_073_741_130_i32;
pub const STATUS_DEVICE_REQUIRES_CLEANING: NTSTATUS = -2_147_483_000_i32;
pub const STATUS_DEVICE_RESET_REQUIRED: NTSTATUS = -2_147_483_210_i32;
pub const STATUS_DEVICE_SUPPORT_IN_PROGRESS: NTSTATUS = -2_147_483_600_i32;
pub const STATUS_DEVICE_UNREACHABLE: NTSTATUS = -1_073_740_700_i32;
pub const STATUS_DEVICE_UNRESPONSIVE: NTSTATUS = -1_073_740_534_i32;
pub const STATUS_DFS_EXIT_PATH_FOUND: NTSTATUS = -1_073_741_669_i32;
pub const STATUS_DFS_UNAVAILABLE: NTSTATUS = -1_073_741_203_i32;
pub const STATUS_DIF_BINDING_API_NOT_FOUND: NTSTATUS = -1_073_738_625_i32;
pub const STATUS_DIF_IOCALLBACK_NOT_REPLACED: NTSTATUS = -1_073_738_634_i32;
pub const STATUS_DIF_LIVEDUMP_LIMIT_EXCEEDED: NTSTATUS = -1_073_738_633_i32;
pub const STATUS_DIF_VOLATILE_DRIVER_HOTPATCHED: NTSTATUS = -1_073_738_631_i32;
pub const STATUS_DIF_VOLATILE_DRIVER_IS_NOT_RUNNING: NTSTATUS = -1_073_738_629_i32;
pub const STATUS_DIF_VOLATILE_INVALID_INFO: NTSTATUS = -1_073_738_630_i32;
pub const STATUS_DIF_VOLATILE_NOT_ALLOWED: NTSTATUS = -1_073_738_626_i32;
pub const STATUS_DIF_VOLATILE_PLUGIN_CHANGE_NOT_ALLOWED: NTSTATUS = -1_073_738_627_i32;
pub const STATUS_DIF_VOLATILE_PLUGIN_IS_NOT_RUNNING: NTSTATUS = -1_073_738_628_i32;
pub const STATUS_DIF_VOLATILE_SECTION_NOT_LOCKED: NTSTATUS = -1_073_738_632_i32;
pub const STATUS_DIRECTORY_IS_A_REPARSE_POINT: NTSTATUS = -1_073_741_183_i32;
pub const STATUS_DIRECTORY_NOT_EMPTY: NTSTATUS = -1_073_741_567_i32;
pub const STATUS_DIRECTORY_NOT_RM: NTSTATUS = -1_072_103_416_i32;
pub const STATUS_DIRECTORY_NOT_SUPPORTED: NTSTATUS = -1_073_740_676_i32;
pub const STATUS_DIRECTORY_SERVICE_REQUIRED: NTSTATUS = -1_073_741_135_i32;
pub const STATUS_DISK_CORRUPT_ERROR: NTSTATUS = -1_073_741_774_i32;
pub const STATUS_DISK_FULL: NTSTATUS = -1_073_741_697_i32;
pub const STATUS_DISK_OPERATION_FAILED: NTSTATUS = -1_073_741_462_i32;
pub const STATUS_DISK_QUOTA_EXCEEDED: NTSTATUS = -1_073_739_774_i32;
pub const STATUS_DISK_RECALIBRATE_FAILED: NTSTATUS = -1_073_741_463_i32;
pub const STATUS_DISK_REPAIR_DISABLED: NTSTATUS = -1_073_739_776_i32;
pub const STATUS_DISK_REPAIR_REDIRECTED: NTSTATUS = 1_073_743_879_i32;
pub const STATUS_DISK_REPAIR_UNSUCCESSFUL: NTSTATUS = -1_073_739_768_i32;
pub const STATUS_DISK_RESET_FAILED: NTSTATUS = -1_073_741_461_i32;
pub const STATUS_DISK_RESOURCES_EXHAUSTED: NTSTATUS = -1_073_740_703_i32;
pub const STATUS_DLL_INIT_FAILED: NTSTATUS = -1_073_741_502_i32;
pub const STATUS_DLL_INIT_FAILED_LOGOFF: NTSTATUS = -1_073_741_205_i32;
pub const STATUS_DLL_MIGHT_BE_INCOMPATIBLE: NTSTATUS = -2_147_483_604_i32;
pub const STATUS_DLL_MIGHT_BE_INSECURE: NTSTATUS = -2_147_483_605_i32;
pub const STATUS_DLL_NOT_FOUND: NTSTATUS = -1_073_741_515_i32;
pub const STATUS_DM_OPERATION_LIMIT_EXCEEDED: NTSTATUS = -1_070_135_808_i32;
pub const STATUS_DOMAIN_CONTROLLER_NOT_FOUND: NTSTATUS = -1_073_741_261_i32;
pub const STATUS_DOMAIN_CTRLR_CONFIG_ERROR: NTSTATUS = -1_073_741_474_i32;
pub const STATUS_DOMAIN_EXISTS: NTSTATUS = -1_073_741_600_i32;
pub const STATUS_DOMAIN_LIMIT_EXCEEDED: NTSTATUS = -1_073_741_599_i32;
pub const STATUS_DOMAIN_TRUST_INCONSISTENT: NTSTATUS = -1_073_741_413_i32;
pub const STATUS_DRIVERS_LEAKING_LOCKED_PAGES: NTSTATUS = 1_073_741_869_i32;
pub const STATUS_DRIVER_BLOCKED: NTSTATUS = -1_073_740_948_i32;
pub const STATUS_DRIVER_BLOCKED_CRITICAL: NTSTATUS = -1_073_740_949_i32;
pub const STATUS_DRIVER_CANCEL_TIMEOUT: NTSTATUS = -1_073_741_282_i32;
pub const STATUS_DRIVER_DATABASE_ERROR: NTSTATUS = -1_073_740_947_i32;
pub const STATUS_DRIVER_ENTRYPOINT_NOT_FOUND: NTSTATUS = -1_073_741_213_i32;
pub const STATUS_DRIVER_FAILED_PRIOR_UNLOAD: NTSTATUS = -1_073_740_914_i32;
pub const STATUS_DRIVER_FAILED_SLEEP: NTSTATUS = -1_073_741_118_i32;
pub const STATUS_DRIVER_INTERNAL_ERROR: NTSTATUS = -1_073_741_437_i32;
pub const STATUS_DRIVER_ORDINAL_NOT_FOUND: NTSTATUS = -1_073_741_214_i32;
pub const STATUS_DRIVER_PROCESS_TERMINATED: NTSTATUS = -1_073_740_720_i32;
pub const STATUS_DRIVER_UNABLE_TO_LOAD: NTSTATUS = -1_073_741_204_i32;
pub const STATUS_DS_ADMIN_LIMIT_EXCEEDED: NTSTATUS = -1_073_741_119_i32;
pub const STATUS_DS_AG_CANT_HAVE_UNIVERSAL_MEMBER: NTSTATUS = -1_073_740_968_i32;
pub const STATUS_DS_ATTRIBUTE_OR_VALUE_EXISTS: NTSTATUS = -1_073_741_148_i32;
pub const STATUS_DS_ATTRIBUTE_TYPE_UNDEFINED: NTSTATUS = -1_073_741_149_i32;
pub const STATUS_DS_BUSY: NTSTATUS = -1_073_741_147_i32;
pub const STATUS_DS_CANT_MOD_OBJ_CLASS: NTSTATUS = -1_073_741_138_i32;
pub const STATUS_DS_CANT_MOD_PRIMARYGROUPID: NTSTATUS = -1_073_741_104_i32;
pub const STATUS_DS_CANT_ON_NON_LEAF: NTSTATUS = -1_073_741_140_i32;
pub const STATUS_DS_CANT_ON_RDN: NTSTATUS = -1_073_741_139_i32;
pub const STATUS_DS_CANT_START: NTSTATUS = -1_073_741_087_i32;
pub const STATUS_DS_CROSS_DOM_MOVE_FAILED: NTSTATUS = -1_073_741_137_i32;
pub const STATUS_DS_DOMAIN_NAME_EXISTS_IN_FOREST: NTSTATUS = -1_073_740_774_i32;
pub const STATUS_DS_DOMAIN_RENAME_IN_PROGRESS: NTSTATUS = -1_073_739_775_i32;
pub const STATUS_DS_DUPLICATE_ID_FOUND: NTSTATUS = -1_073_740_795_i32;
pub const STATUS_DS_FLAT_NAME_EXISTS_IN_FOREST: NTSTATUS = -1_073_740_773_i32;
pub const STATUS_DS_GC_NOT_AVAILABLE: NTSTATUS = -1_073_741_136_i32;
pub const STATUS_DS_GC_REQUIRED: NTSTATUS = -1_073_741_084_i32;
pub const STATUS_DS_GLOBAL_CANT_HAVE_CROSSDOMAIN_MEMBER: NTSTATUS = -1_073_741_094_i32;
pub const STATUS_DS_GLOBAL_CANT_HAVE_LOCAL_MEMBER: NTSTATUS = -1_073_741_097_i32;
pub const STATUS_DS_GLOBAL_CANT_HAVE_UNIVERSAL_MEMBER: NTSTATUS = -1_073_741_096_i32;
pub const STATUS_DS_GROUP_CONVERSION_ERROR: NTSTATUS = -1_073_740_794_i32;
pub const STATUS_DS_HAVE_PRIMARY_MEMBERS: NTSTATUS = -1_073_741_092_i32;
pub const STATUS_DS_INCORRECT_ROLE_OWNER: NTSTATUS = -1_073_741_143_i32;
pub const STATUS_DS_INIT_FAILURE: NTSTATUS = -1_073_741_086_i32;
pub const STATUS_DS_INIT_FAILURE_CONSOLE: NTSTATUS = -1_073_741_076_i32;
pub const STATUS_DS_INVALID_ATTRIBUTE_SYNTAX: NTSTATUS = -1_073_741_150_i32;
pub const STATUS_DS_INVALID_GROUP_TYPE: NTSTATUS = -1_073_741_100_i32;
pub const STATUS_DS_LOCAL_CANT_HAVE_CROSSDOMAIN_LOCAL_MEMBER: NTSTATUS = -1_073_741_093_i32;
pub const STATUS_DS_LOCAL_MEMBER_OF_LOCAL_ONLY: NTSTATUS = -1_073_741_083_i32;
pub const STATUS_DS_MACHINE_ACCOUNT_QUOTA_EXCEEDED: NTSTATUS = -1_073_741_081_i32;
pub const STATUS_DS_MEMBERSHIP_EVALUATED_LOCALLY: NTSTATUS = 289_i32;
pub const STATUS_DS_NAME_NOT_UNIQUE: NTSTATUS = -1_073_740_796_i32;
pub const STATUS_DS_NO_ATTRIBUTE_OR_VALUE: NTSTATUS = -1_073_741_151_i32;
pub const STATUS_DS_NO_FPO_IN_UNIVERSAL_GROUPS: NTSTATUS = -1_073_741_082_i32;
pub const STATUS_DS_NO_MORE_RIDS: NTSTATUS = -1_073_741_144_i32;
pub const STATUS_DS_NO_NEST_GLOBALGROUP_IN_MIXEDDOMAIN: NTSTATUS = -1_073_741_099_i32;
pub const STATUS_DS_NO_NEST_LOCALGROUP_IN_MIXEDDOMAIN: NTSTATUS = -1_073_741_098_i32;
pub const STATUS_DS_NO_RIDS_ALLOCATED: NTSTATUS = -1_073_741_145_i32;
pub const STATUS_DS_OBJ_CLASS_VIOLATION: NTSTATUS = -1_073_741_141_i32;
pub const STATUS_DS_OID_MAPPED_GROUP_CANT_HAVE_MEMBERS: NTSTATUS = -1_073_700_729_i32;
pub const STATUS_DS_OID_NOT_FOUND: NTSTATUS = -1_073_700_728_i32;
pub const STATUS_DS_RIDMGR_DISABLED: NTSTATUS = -1_073_741_126_i32;
pub const STATUS_DS_RIDMGR_INIT_ERROR: NTSTATUS = -1_073_741_142_i32;
pub const STATUS_DS_SAM_INIT_FAILURE: NTSTATUS = -1_073_741_109_i32;
pub const STATUS_DS_SAM_INIT_FAILURE_CONSOLE: NTSTATUS = -1_073_741_075_i32;
pub const STATUS_DS_SENSITIVE_GROUP_VIOLATION: NTSTATUS = -1_073_741_107_i32;
pub const STATUS_DS_SHUTTING_DOWN: NTSTATUS = 1_073_742_704_i32;
pub const STATUS_DS_SRC_SID_EXISTS_IN_FOREST: NTSTATUS = -1_073_740_775_i32;
pub const STATUS_DS_UNAVAILABLE: NTSTATUS = -1_073_741_146_i32;
pub const STATUS_DS_UNIVERSAL_CANT_HAVE_LOCAL_MEMBER: NTSTATUS = -1_073_741_095_i32;
pub const STATUS_DS_VERSION_CHECK_FAILURE: NTSTATUS = -1_073_740_971_i32;
pub const STATUS_DUPLICATE_NAME: NTSTATUS = -1_073_741_635_i32;
pub const STATUS_DUPLICATE_OBJECTID: NTSTATUS = -1_073_741_270_i32;
pub const STATUS_DUPLICATE_PRIVILEGES: NTSTATUS = -1_073_741_402_i32;
pub const STATUS_DYNAMIC_CODE_BLOCKED: NTSTATUS = -1_073_740_284_i32;
pub const STATUS_EAS_NOT_SUPPORTED: NTSTATUS = -1_073_741_745_i32;
pub const STATUS_EA_CORRUPT_ERROR: NTSTATUS = -1_073_741_741_i32;
pub const STATUS_EA_LIST_INCONSISTENT: NTSTATUS = -2_147_483_628_i32;
pub const STATUS_EA_TOO_LARGE: NTSTATUS = -1_073_741_744_i32;
pub const STATUS_EFS_ALG_BLOB_TOO_BIG: NTSTATUS = -1_073_740_974_i32;
pub const STATUS_EFS_NOT_ALLOWED_IN_TRANSACTION: NTSTATUS = -1_072_103_362_i32;
pub const STATUS_ELEVATION_REQUIRED: NTSTATUS = -1_073_740_756_i32;
pub const STATUS_EMULATION_BREAKPOINT: NTSTATUS = 1_073_741_880_i32;
pub const STATUS_EMULATION_SYSCALL: NTSTATUS = 1_073_741_881_i32;
pub const STATUS_ENCLAVE_FAILURE: NTSTATUS = -1_073_740_657_i32;
pub const STATUS_ENCLAVE_IS_TERMINATING: NTSTATUS = -1_073_740_526_i32;
pub const STATUS_ENCLAVE_NOT_TERMINATED: NTSTATUS = -1_073_740_527_i32;
pub const STATUS_ENCLAVE_VIOLATION: NTSTATUS = -1_073_740_638_i32;
pub const STATUS_ENCOUNTERED_WRITE_IN_PROGRESS: NTSTATUS = -1_073_740_749_i32;
pub const STATUS_ENCRYPTED_FILE_NOT_SUPPORTED: NTSTATUS = -1_073_740_605_i32;
pub const STATUS_ENCRYPTED_IO_NOT_POSSIBLE: NTSTATUS = -1_073_739_760_i32;
pub const STATUS_ENCRYPTING_METADATA_DISALLOWED: NTSTATUS = -1_073_740_617_i32;
pub const STATUS_ENCRYPTION_DISABLED: NTSTATUS = -1_073_740_618_i32;
pub const STATUS_ENCRYPTION_FAILED: NTSTATUS = -1_073_741_174_i32;
pub const STATUS_END_OF_FILE: NTSTATUS = -1_073_741_807_i32;
pub const STATUS_END_OF_MEDIA: NTSTATUS = -2_147_483_618_i32;
pub const STATUS_ENLISTMENT_NOT_FOUND: NTSTATUS = -1_072_103_344_i32;
pub const STATUS_ENLISTMENT_NOT_SUPERIOR: NTSTATUS = -1_072_103_373_i32;
pub const STATUS_ENTRYPOINT_NOT_FOUND: NTSTATUS = -1_073_741_511_i32;
pub const STATUS_EOF_ON_GHOSTED_RANGE: NTSTATUS = -1_073_700_857_i32;
pub const STATUS_EOM_OVERFLOW: NTSTATUS = -1_073_741_449_i32;
pub const STATUS_ERROR_PROCESS_NOT_IN_JOB: NTSTATUS = -1_073_741_394_i32;
pub const STATUS_EVALUATION_EXPIRATION: NTSTATUS = -1_073_741_208_i32;
pub const STATUS_EVENTLOG_CANT_START: NTSTATUS = -1_073_741_425_i32;
pub const STATUS_EVENTLOG_FILE_CHANGED: NTSTATUS = -1_073_741_417_i32;
pub const STATUS_EVENTLOG_FILE_CORRUPT: NTSTATUS = -1_073_741_426_i32;
pub const STATUS_EVENT_DONE: NTSTATUS = 1_073_741_842_i32;
pub const STATUS_EVENT_PENDING: NTSTATUS = 1_073_741_843_i32;
pub const STATUS_EXECUTABLE_MEMORY_WRITE: NTSTATUS = -1_073_739_997_i32;
pub const STATUS_EXPIRED_HANDLE: NTSTATUS = -1_072_103_328_i32;
pub const STATUS_EXTERNAL_BACKING_PROVIDER_UNKNOWN: NTSTATUS = -1_073_740_690_i32;
pub const STATUS_EXTERNAL_SYSKEY_NOT_SUPPORTED: NTSTATUS = -1_073_740_639_i32;
pub const STATUS_EXTRANEOUS_INFORMATION: NTSTATUS = -2_147_483_625_i32;
pub const STATUS_FAILED_DRIVER_ENTRY: NTSTATUS = -1_073_740_955_i32;
pub const STATUS_FAILED_STACK_SWITCH: NTSTATUS = -1_073_740_941_i32;
pub const STATUS_FAIL_CHECK: NTSTATUS = -1_073_741_271_i32;
pub const STATUS_FAIL_FAST_EXCEPTION: NTSTATUS = -1_073_740_286_i32;
pub const STATUS_FASTPATH_REJECTED: NTSTATUS = -1_073_700_844_i32;
pub const STATUS_FATAL_APP_EXIT: NTSTATUS = 1_073_741_845_i32;
pub const STATUS_FATAL_MEMORY_EXHAUSTION: NTSTATUS = -1_073_741_395_i32;
pub const STATUS_FATAL_USER_CALLBACK_EXCEPTION: NTSTATUS = -1_073_740_771_i32;
pub const STATUS_FILEMARK_DETECTED: NTSTATUS = -2_147_483_621_i32;
pub const STATUS_FILES_OPEN: NTSTATUS = -1_073_741_561_i32;
pub const STATUS_FILE_CHECKED_OUT: NTSTATUS = -1_073_739_519_i32;
pub const STATUS_FILE_CLOSED: NTSTATUS = -1_073_741_528_i32;
pub const STATUS_FILE_CORRUPT_ERROR: NTSTATUS = -1_073_741_566_i32;
pub const STATUS_FILE_DELETED: NTSTATUS = -1_073_741_533_i32;
pub const STATUS_FILE_ENCRYPTED: NTSTATUS = -1_073_741_165_i32;
pub const STATUS_FILE_FORCED_CLOSED: NTSTATUS = -1_073_741_642_i32;
pub const STATUS_FILE_HANDLE_REVOKED: NTSTATUS = -1_073_739_504_i32;
pub const STATUS_FILE_IDENTITY_NOT_PERSISTENT: NTSTATUS = -1_072_103_370_i32;
pub const STATUS_FILE_INVALID: NTSTATUS = -1_073_741_672_i32;
pub const STATUS_FILE_IS_A_DIRECTORY: NTSTATUS = -1_073_741_638_i32;
pub const STATUS_FILE_IS_OFFLINE: NTSTATUS = -1_073_741_209_i32;
pub const STATUS_FILE_LOCKED_WITH_ONLY_READERS: NTSTATUS = 298_i32;
pub const STATUS_FILE_LOCKED_WITH_WRITERS: NTSTATUS = 299_i32;
pub const STATUS_FILE_LOCK_CONFLICT: NTSTATUS = -1_073_741_740_i32;
pub const STATUS_FILE_METADATA_OPTIMIZATION_IN_PROGRESS: NTSTATUS = -1_073_741_397_i32;
pub const STATUS_FILE_NOT_AVAILABLE: NTSTATUS = -1_073_740_697_i32;
pub const STATUS_FILE_NOT_ENCRYPTED: NTSTATUS = -1_073_741_167_i32;
pub const STATUS_FILE_NOT_SUPPORTED: NTSTATUS = -1_073_740_620_i32;
pub const STATUS_FILE_PROTECTED_UNDER_DPL: NTSTATUS = -1_073_740_637_i32;
pub const STATUS_FILE_RENAMED: NTSTATUS = -1_073_741_611_i32;
pub const STATUS_FILE_SNAP_INVALID_PARAMETER: NTSTATUS = -1_073_679_099_i32;
pub const STATUS_FILE_SNAP_IN_PROGRESS: NTSTATUS = -1_073_679_104_i32;
pub const STATUS_FILE_SNAP_IO_NOT_COORDINATED: NTSTATUS = -1_073_679_101_i32;
pub const STATUS_FILE_SNAP_MODIFY_NOT_SUPPORTED: NTSTATUS = -1_073_679_102_i32;
pub const STATUS_FILE_SNAP_UNEXPECTED_ERROR: NTSTATUS = -1_073_679_100_i32;
pub const STATUS_FILE_SNAP_USER_SECTION_NOT_SUPPORTED: NTSTATUS = -1_073_679_103_i32;
pub const STATUS_FILE_SYSTEM_LIMITATION: NTSTATUS = -1_073_740_761_i32;
pub const STATUS_FILE_SYSTEM_VIRTUALIZATION_BUSY: NTSTATUS = -1_073_689_085_i32;
pub const STATUS_FILE_SYSTEM_VIRTUALIZATION_INVALID_OPERATION: NTSTATUS = -1_073_689_083_i32;
pub const STATUS_FILE_SYSTEM_VIRTUALIZATION_METADATA_CORRUPT: NTSTATUS = -1_073_689_086_i32;
pub const STATUS_FILE_SYSTEM_VIRTUALIZATION_PROVIDER_UNKNOWN: NTSTATUS = -1_073_689_084_i32;
pub const STATUS_FILE_SYSTEM_VIRTUALIZATION_UNAVAILABLE: NTSTATUS = -1_073_689_087_i32;
pub const STATUS_FILE_TOO_LARGE: NTSTATUS = -1_073_739_516_i32;
pub const STATUS_FIRMWARE_IMAGE_INVALID: NTSTATUS = -1_073_740_667_i32;
pub const STATUS_FIRMWARE_SLOT_INVALID: NTSTATUS = -1_073_740_668_i32;
pub const STATUS_FIRMWARE_UPDATED: NTSTATUS = 1_073_741_868_i32;
pub const STATUS_FLOATED_SECTION: NTSTATUS = -1_072_103_349_i32;
pub const STATUS_FLOAT_DENORMAL_OPERAND: NTSTATUS = -1_073_741_683_i32;
pub const STATUS_FLOAT_DIVIDE_BY_ZERO: NTSTATUS = -1_073_741_682_i32;
pub const STATUS_FLOAT_INEXACT_RESULT: NTSTATUS = -1_073_741_681_i32;
pub const STATUS_FLOAT_INVALID_OPERATION: NTSTATUS = -1_073_741_680_i32;
pub const STATUS_FLOAT_MULTIPLE_FAULTS: NTSTATUS = -1_073_741_132_i32;
pub const STATUS_FLOAT_MULTIPLE_TRAPS: NTSTATUS = -1_073_741_131_i32;
pub const STATUS_FLOAT_OVERFLOW: NTSTATUS = -1_073_741_679_i32;
pub const STATUS_FLOAT_STACK_CHECK: NTSTATUS = -1_073_741_678_i32;
pub const STATUS_FLOAT_UNDERFLOW: NTSTATUS = -1_073_741_677_i32;
pub const STATUS_FLOPPY_BAD_REGISTERS: NTSTATUS = -1_073_741_464_i32;
pub const STATUS_FLOPPY_ID_MARK_NOT_FOUND: NTSTATUS = -1_073_741_467_i32;
pub const STATUS_FLOPPY_UNKNOWN_ERROR: NTSTATUS = -1_073_741_465_i32;
pub const STATUS_FLOPPY_VOLUME: NTSTATUS = -1_073_741_468_i32;
pub const STATUS_FLOPPY_WRONG_CYLINDER: NTSTATUS = -1_073_741_466_i32;
pub const STATUS_FLT_ALREADY_ENLISTED: NTSTATUS = -1_071_906_789_i32;
pub const STATUS_FLT_BUFFER_TOO_SMALL: NTSTATUS = -2_145_648_639_i32;
pub const STATUS_FLT_CBDQ_DISABLED: NTSTATUS = -1_071_906_802_i32;
pub const STATUS_FLT_CONTEXT_ALLOCATION_NOT_FOUND: NTSTATUS = -1_071_906_794_i32;
pub const STATUS_FLT_CONTEXT_ALREADY_DEFINED: NTSTATUS = -1_071_906_814_i32;
pub const STATUS_FLT_CONTEXT_ALREADY_LINKED: NTSTATUS = -1_071_906_788_i32;
pub const STATUS_FLT_DELETING_OBJECT: NTSTATUS = -1_071_906_805_i32;
pub const STATUS_FLT_DISALLOW_FAST_IO: NTSTATUS = -1_071_906_812_i32;
pub const STATUS_FLT_DO_NOT_ATTACH: NTSTATUS = -1_071_906_801_i32;
pub const STATUS_FLT_DO_NOT_DETACH: NTSTATUS = -1_071_906_800_i32;
pub const STATUS_FLT_DUPLICATE_ENTRY: NTSTATUS = -1_071_906_803_i32;
pub const STATUS_FLT_FILTER_NOT_FOUND: NTSTATUS = -1_071_906_797_i32;
pub const STATUS_FLT_FILTER_NOT_READY: NTSTATUS = -1_071_906_808_i32;
pub const STATUS_FLT_INSTANCE_ALTITUDE_COLLISION: NTSTATUS = -1_071_906_799_i32;
pub const STATUS_FLT_INSTANCE_NAME_COLLISION: NTSTATUS = -1_071_906_798_i32;
pub const STATUS_FLT_INSTANCE_NOT_FOUND: NTSTATUS = -1_071_906_795_i32;
pub const STATUS_FLT_INTERNAL_ERROR: NTSTATUS = -1_071_906_806_i32;
pub const STATUS_FLT_INVALID_ASYNCHRONOUS_REQUEST: NTSTATUS = -1_071_906_813_i32;
pub const STATUS_FLT_INVALID_CONTEXT_REGISTRATION: NTSTATUS = -1_071_906_793_i32;
pub const STATUS_FLT_INVALID_NAME_REQUEST: NTSTATUS = -1_071_906_811_i32;
pub const STATUS_FLT_IO_COMPLETE: NTSTATUS = 1_835_009_i32;
pub const STATUS_FLT_MUST_BE_NONPAGED_POOL: NTSTATUS = -1_071_906_804_i32;
pub const STATUS_FLT_NAME_CACHE_MISS: NTSTATUS = -1_071_906_792_i32;
pub const STATUS_FLT_NOT_INITIALIZED: NTSTATUS = -1_071_906_809_i32;
pub const STATUS_FLT_NOT_SAFE_TO_POST_OPERATION: NTSTATUS = -1_071_906_810_i32;
pub const STATUS_FLT_NO_DEVICE_OBJECT: NTSTATUS = -1_071_906_791_i32;
pub const STATUS_FLT_NO_HANDLER_DEFINED: NTSTATUS = -1_071_906_815_i32;
pub const STATUS_FLT_NO_WAITER_FOR_REPLY: NTSTATUS = -1_071_906_784_i32;
pub const STATUS_FLT_POST_OPERATION_CLEANUP: NTSTATUS = -1_071_906_807_i32;
pub const STATUS_FLT_REGISTRATION_BUSY: NTSTATUS = -1_071_906_781_i32;
pub const STATUS_FLT_VOLUME_ALREADY_MOUNTED: NTSTATUS = -1_071_906_790_i32;
pub const STATUS_FLT_VOLUME_NOT_FOUND: NTSTATUS = -1_071_906_796_i32;
pub const STATUS_FLT_WCOS_NOT_SUPPORTED: NTSTATUS = -1_071_906_780_i32;
pub const STATUS_FORMS_AUTH_REQUIRED: NTSTATUS = -1_073_739_515_i32;
pub const STATUS_FOUND_OUT_OF_SCOPE: NTSTATUS = -1_073_741_266_i32;
pub const STATUS_FREE_SPACE_TOO_FRAGMENTED: NTSTATUS = -1_073_740_645_i32;
pub const STATUS_FREE_VM_NOT_AT_BASE: NTSTATUS = -1_073_741_665_i32;
pub const STATUS_FSFILTER_OP_COMPLETED_SUCCESSFULLY: NTSTATUS = 294_i32;
pub const STATUS_FS_DRIVER_REQUIRED: NTSTATUS = -1_073_741_412_i32;
pub const STATUS_FS_METADATA_INCONSISTENT: NTSTATUS = -1_073_740_520_i32;
pub const STATUS_FT_DI_SCAN_REQUIRED: NTSTATUS = -1_073_740_692_i32;
pub const STATUS_FT_MISSING_MEMBER: NTSTATUS = -1_073_741_473_i32;
pub const STATUS_FT_ORPHANING: NTSTATUS = -1_073_741_459_i32;
pub const STATUS_FT_READ_FAILURE: NTSTATUS = -1_073_740_629_i32;
pub const STATUS_FT_READ_FROM_COPY: NTSTATUS = 1_073_741_877_i32;
pub const STATUS_FT_READ_FROM_COPY_FAILURE: NTSTATUS = -1_073_740_609_i32;
pub const STATUS_FT_READ_RECOVERY_FROM_BACKUP: NTSTATUS = 1_073_741_834_i32;
pub const STATUS_FT_WRITE_FAILURE: NTSTATUS = -1_073_740_693_i32;
pub const STATUS_FT_WRITE_RECOVERY: NTSTATUS = 1_073_741_835_i32;
pub const STATUS_FULLSCREEN_MODE: NTSTATUS = -1_073_741_479_i32;
pub const STATUS_FVE_ACTION_NOT_ALLOWED: NTSTATUS = -1_071_579_127_i32;
pub const STATUS_FVE_AUTH_INVALID_APPLICATION: NTSTATUS = -1_071_579_109_i32;
pub const STATUS_FVE_AUTH_INVALID_CONFIG: NTSTATUS = -1_071_579_108_i32;
pub const STATUS_FVE_BAD_DATA: NTSTATUS = -1_071_579_126_i32;
pub const STATUS_FVE_BAD_INFORMATION: NTSTATUS = -1_071_579_134_i32;
pub const STATUS_FVE_BAD_METADATA_POINTER: NTSTATUS = -1_071_579_105_i32;
pub const STATUS_FVE_BAD_PARTITION_SIZE: NTSTATUS = -1_071_579_131_i32;
pub const STATUS_FVE_CONV_READ_ERROR: NTSTATUS = -1_071_579_123_i32;
pub const STATUS_FVE_CONV_RECOVERY_FAILED: NTSTATUS = -1_071_579_096_i32;
pub const STATUS_FVE_CONV_WRITE_ERROR: NTSTATUS = -1_071_579_122_i32;
pub const STATUS_FVE_DATASET_FULL: NTSTATUS = -1_071_579_069_i32;
pub const STATUS_FVE_DEBUGGER_ENABLED: NTSTATUS = -1_071_579_107_i32;
pub const STATUS_FVE_DEVICE_LOCKEDOUT: NTSTATUS = -1_071_579_077_i32;
pub const STATUS_FVE_DRY_RUN_FAILED: NTSTATUS = -1_071_579_106_i32;
pub const STATUS_FVE_EDRIVE_BAND_ENUMERATION_FAILED: NTSTATUS = -1_071_579_071_i32;
pub const STATUS_FVE_EDRIVE_DRY_RUN_FAILED: NTSTATUS = -1_071_579_080_i32;
pub const STATUS_FVE_ENH_PIN_INVALID: NTSTATUS = -1_071_579_087_i32;
pub const STATUS_FVE_FAILED_AUTHENTICATION: NTSTATUS = -1_071_579_119_i32;
pub const STATUS_FVE_FAILED_SECTOR_SIZE: NTSTATUS = -1_071_579_120_i32;
pub const STATUS_FVE_FAILED_WRONG_FS: NTSTATUS = -1_071_579_132_i32;
pub const STATUS_FVE_FS_MOUNTED: NTSTATUS = -1_071_579_129_i32;
pub const STATUS_FVE_FS_NOT_EXTENDED: NTSTATUS = -1_071_579_130_i32;
pub const STATUS_FVE_FULL_ENCRYPTION_NOT_ALLOWED_ON_TP_STORAGE: NTSTATUS = -1_071_579_086_i32;
pub const STATUS_FVE_INVALID_DATUM_TYPE: NTSTATUS = -1_071_579_094_i32;
pub const STATUS_FVE_KEYFILE_INVALID: NTSTATUS = -1_071_579_116_i32;
pub const STATUS_FVE_KEYFILE_NOT_FOUND: NTSTATUS = -1_071_579_117_i32;
pub const STATUS_FVE_KEYFILE_NO_VMK: NTSTATUS = -1_071_579_115_i32;
pub const STATUS_FVE_LOCKED_VOLUME: NTSTATUS = -1_071_579_136_i32;
pub const STATUS_FVE_METADATA_FULL: NTSTATUS = -1_071_579_068_i32;
pub const STATUS_FVE_MOR_FAILED: NTSTATUS = -1_071_579_099_i32;
pub const STATUS_FVE_NOT_ALLOWED_ON_CLUSTER: NTSTATUS = -1_071_579_083_i32;
pub const STATUS_FVE_NOT_ALLOWED_ON_CSV_STACK: NTSTATUS = -1_071_579_084_i32;
pub const STATUS_FVE_NOT_ALLOWED_TO_UPGRADE_WHILE_CONVERTING: NTSTATUS = -1_071_579_082_i32;
pub const STATUS_FVE_NOT_DATA_VOLUME: NTSTATUS = -1_071_579_124_i32;
pub const STATUS_FVE_NOT_DE_VOLUME: NTSTATUS = -1_071_579_075_i32;
pub const STATUS_FVE_NOT_ENCRYPTED: NTSTATUS = -1_071_579_135_i32;
pub const STATUS_FVE_NOT_OS_VOLUME: NTSTATUS = -1_071_579_118_i32;
pub const STATUS_FVE_NO_AUTOUNLOCK_MASTER_KEY: NTSTATUS = -1_071_579_100_i32;
pub const STATUS_FVE_NO_FEATURE_LICENSE: NTSTATUS = -1_071_579_098_i32;
pub const STATUS_FVE_NO_LICENSE: NTSTATUS = -1_071_579_128_i32;
pub const STATUS_FVE_OLD_METADATA_COPY: NTSTATUS = -1_071_579_104_i32;
pub const STATUS_FVE_OSV_KSR_NOT_ALLOWED: NTSTATUS = -1_071_579_072_i32;
pub const STATUS_FVE_OVERLAPPED_UPDATE: NTSTATUS = -1_071_579_121_i32;
pub const STATUS_FVE_PARTIAL_METADATA: NTSTATUS = -2_145_320_959_i32;
pub const STATUS_FVE_PIN_INVALID: NTSTATUS = -1_071_579_110_i32;
pub const STATUS_FVE_POLICY_ON_RDV_EXCLUSION_LIST: NTSTATUS = -1_071_579_070_i32;
pub const STATUS_FVE_POLICY_USER_DISABLE_RDV_NOT_ALLOWED: NTSTATUS = -1_071_579_097_i32;
pub const STATUS_FVE_PROTECTION_CANNOT_BE_DISABLED: NTSTATUS = -1_071_579_073_i32;
pub const STATUS_FVE_PROTECTION_DISABLED: NTSTATUS = -1_071_579_074_i32;
pub const STATUS_FVE_RAW_ACCESS: NTSTATUS = -1_071_579_102_i32;
pub const STATUS_FVE_RAW_BLOCKED: NTSTATUS = -1_071_579_101_i32;
pub const STATUS_FVE_REBOOT_REQUIRED: NTSTATUS = -1_071_579_103_i32;
pub const STATUS_FVE_SECUREBOOT_CONFIG_CHANGE: NTSTATUS = -1_071_579_078_i32;
pub const STATUS_FVE_SECUREBOOT_DISABLED: NTSTATUS = -1_071_579_079_i32;
pub const STATUS_FVE_TOO_SMALL: NTSTATUS = -1_071_579_133_i32;
pub const STATUS_FVE_TPM_DISABLED: NTSTATUS = -1_071_579_114_i32;
pub const STATUS_FVE_TPM_INVALID_PCR: NTSTATUS = -1_071_579_112_i32;
pub const STATUS_FVE_TPM_NO_VMK: NTSTATUS = -1_071_579_111_i32;
pub const STATUS_FVE_TPM_SRK_AUTH_NOT_ZERO: NTSTATUS = -1_071_579_113_i32;
pub const STATUS_FVE_TRANSIENT_STATE: NTSTATUS = -2_145_320_958_i32;
pub const STATUS_FVE_VIRTUALIZED_SPACE_TOO_BIG: NTSTATUS = -1_071_579_095_i32;
pub const STATUS_FVE_VOLUME_EXTEND_PREVENTS_EOW_DECRYPT: NTSTATUS = -1_071_579_076_i32;
pub const STATUS_FVE_VOLUME_NOT_BOUND: NTSTATUS = -1_071_579_125_i32;
pub const STATUS_FVE_VOLUME_TOO_SMALL: NTSTATUS = -1_071_579_088_i32;
pub const STATUS_FVE_WIPE_CANCEL_NOT_APPLICABLE: NTSTATUS = -1_071_579_081_i32;
pub const STATUS_FVE_WIPE_NOT_ALLOWED_ON_TP_STORAGE: NTSTATUS = -1_071_579_085_i32;
pub const STATUS_FWP_ACTION_INCOMPATIBLE_WITH_LAYER: NTSTATUS = -1_071_513_556_i32;
pub const STATUS_FWP_ACTION_INCOMPATIBLE_WITH_SUBLAYER: NTSTATUS = -1_071_513_555_i32;
pub const STATUS_FWP_ALREADY_EXISTS: NTSTATUS = -1_071_513_591_i32;
pub const STATUS_FWP_BUILTIN_OBJECT: NTSTATUS = -1_071_513_577_i32;
pub const STATUS_FWP_CALLOUT_NOTIFICATION_FAILED: NTSTATUS = -1_071_513_545_i32;
pub const STATUS_FWP_CALLOUT_NOT_FOUND: NTSTATUS = -1_071_513_599_i32;
pub const STATUS_FWP_CANNOT_PEND: NTSTATUS = -1_071_513_341_i32;
pub const STATUS_FWP_CONDITION_NOT_FOUND: NTSTATUS = -1_071_513_598_i32;
pub const STATUS_FWP_CONNECTIONS_DISABLED: NTSTATUS = -1_071_513_535_i32;
pub const STATUS_FWP_CONTEXT_INCOMPATIBLE_WITH_CALLOUT: NTSTATUS = -1_071_513_553_i32;
pub const STATUS_FWP_CONTEXT_INCOMPATIBLE_WITH_LAYER: NTSTATUS = -1_071_513_554_i32;
pub const STATUS_FWP_DROP_NOICMP: NTSTATUS = -1_071_513_340_i32;
pub const STATUS_FWP_DUPLICATE_AUTH_METHOD: NTSTATUS = -1_071_513_540_i32;
pub const STATUS_FWP_DUPLICATE_CONDITION: NTSTATUS = -1_071_513_558_i32;
pub const STATUS_FWP_DUPLICATE_KEYMOD: NTSTATUS = -1_071_513_557_i32;
pub const STATUS_FWP_DYNAMIC_SESSION_IN_PROGRESS: NTSTATUS = -1_071_513_589_i32;
pub const STATUS_FWP_EM_NOT_SUPPORTED: NTSTATUS = -1_071_513_550_i32;
pub const STATUS_FWP_FILTER_NOT_FOUND: NTSTATUS = -1_071_513_597_i32;
pub const STATUS_FWP_IKEEXT_NOT_RUNNING: NTSTATUS = -1_071_513_532_i32;
pub const STATUS_FWP_INCOMPATIBLE_AUTH_METHOD: NTSTATUS = -1_071_513_552_i32;
pub const STATUS_FWP_INCOMPATIBLE_CIPHER_TRANSFORM: NTSTATUS = -1_071_513_542_i32;
pub const STATUS_FWP_INCOMPATIBLE_DH_GROUP: NTSTATUS = -1_071_513_551_i32;
pub const STATUS_FWP_INCOMPATIBLE_LAYER: NTSTATUS = -1_071_513_580_i32;
pub const STATUS_FWP_INCOMPATIBLE_SA_STATE: NTSTATUS = -1_071_513_573_i32;
pub const STATUS_FWP_INCOMPATIBLE_TXN: NTSTATUS = -1_071_513_583_i32;
pub const STATUS_FWP_INJECT_HANDLE_CLOSING: NTSTATUS = -1_071_513_343_i32;
pub const STATUS_FWP_INJECT_HANDLE_STALE: NTSTATUS = -1_071_513_342_i32;
pub const STATUS_FWP_INVALID_ACTION_TYPE: NTSTATUS = -1_071_513_564_i32;
pub const STATUS_FWP_INVALID_AUTH_TRANSFORM: NTSTATUS = -1_071_513_544_i32;
pub const STATUS_FWP_INVALID_CIPHER_TRANSFORM: NTSTATUS = -1_071_513_543_i32;
pub const STATUS_FWP_INVALID_DNS_NAME: NTSTATUS = -1_071_513_534_i32;
pub const STATUS_FWP_INVALID_ENUMERATOR: NTSTATUS = -1_071_513_571_i32;
pub const STATUS_FWP_INVALID_FLAGS: NTSTATUS = -1_071_513_570_i32;
pub const STATUS_FWP_INVALID_INTERVAL: NTSTATUS = -1_071_513_567_i32;
pub const STATUS_FWP_INVALID_NET_MASK: NTSTATUS = -1_071_513_569_i32;
pub const STATUS_FWP_INVALID_PARAMETER: NTSTATUS = -1_071_513_547_i32;
pub const STATUS_FWP_INVALID_RANGE: NTSTATUS = -1_071_513_568_i32;
pub const STATUS_FWP_INVALID_TRANSFORM_COMBINATION: NTSTATUS = -1_071_513_541_i32;
pub const STATUS_FWP_INVALID_TUNNEL_ENDPOINT: NTSTATUS = -1_071_513_539_i32;
pub const STATUS_FWP_INVALID_WEIGHT: NTSTATUS = -1_071_513_563_i32;
pub const STATUS_FWP_IN_USE: NTSTATUS = -1_071_513_590_i32;
pub const STATUS_FWP_KEY_DICTATION_INVALID_KEYING_MATERIAL: NTSTATUS = -1_071_513_536_i32;
pub const STATUS_FWP_KEY_DICTATOR_ALREADY_REGISTERED: NTSTATUS = -1_071_513_537_i32;
pub const STATUS_FWP_KM_CLIENTS_ONLY: NTSTATUS = -1_071_513_579_i32;
pub const STATUS_FWP_L2_DRIVER_NOT_READY: NTSTATUS = -1_071_513_538_i32;
pub const STATUS_FWP_LAYER_NOT_FOUND: NTSTATUS = -1_071_513_596_i32;
pub const STATUS_FWP_LIFETIME_MISMATCH: NTSTATUS = -1_071_513_578_i32;
pub const STATUS_FWP_MATCH_TYPE_MISMATCH: NTSTATUS = -1_071_513_562_i32;
pub const STATUS_FWP_NET_EVENTS_DISABLED: NTSTATUS = -1_071_513_581_i32;
pub const STATUS_FWP_NEVER_MATCH: NTSTATUS = -1_071_513_549_i32;
pub const STATUS_FWP_NOTIFICATION_DROPPED: NTSTATUS = -1_071_513_575_i32;
pub const STATUS_FWP_NOT_FOUND: NTSTATUS = -1_071_513_592_i32;
pub const STATUS_FWP_NO_TXN_IN_PROGRESS: NTSTATUS = -1_071_513_587_i32;
pub const STATUS_FWP_NULL_DISPLAY_NAME: NTSTATUS = -1_071_513_565_i32;
pub const STATUS_FWP_NULL_POINTER: NTSTATUS = -1_071_513_572_i32;
pub const STATUS_FWP_OUT_OF_BOUNDS: NTSTATUS = -1_071_513_560_i32;
pub const STATUS_FWP_PROVIDER_CONTEXT_MISMATCH: NTSTATUS = -1_071_513_548_i32;
pub const STATUS_FWP_PROVIDER_CONTEXT_NOT_FOUND: NTSTATUS = -1_071_513_594_i32;
pub const STATUS_FWP_PROVIDER_NOT_FOUND: NTSTATUS = -1_071_513_595_i32;
pub const STATUS_FWP_RESERVED: NTSTATUS = -1_071_513_559_i32;
pub const STATUS_FWP_SESSION_ABORTED: NTSTATUS = -1_071_513_584_i32;
pub const STATUS_FWP_STILL_ON: NTSTATUS = -1_071_513_533_i32;
pub const STATUS_FWP_SUBLAYER_NOT_FOUND: NTSTATUS = -1_071_513_593_i32;
pub const STATUS_FWP_TCPIP_NOT_READY: NTSTATUS = -1_071_513_344_i32;
pub const STATUS_FWP_TIMEOUT: NTSTATUS = -1_071_513_582_i32;
pub const STATUS_FWP_TOO_MANY_CALLOUTS: NTSTATUS = -1_071_513_576_i32;
pub const STATUS_FWP_TOO_MANY_SUBLAYERS: NTSTATUS = -1_071_513_546_i32;
pub const STATUS_FWP_TRAFFIC_MISMATCH: NTSTATUS = -1_071_513_574_i32;
pub const STATUS_FWP_TXN_ABORTED: NTSTATUS = -1_071_513_585_i32;
pub const STATUS_FWP_TXN_IN_PROGRESS: NTSTATUS = -1_071_513_586_i32;
pub const STATUS_FWP_TYPE_MISMATCH: NTSTATUS = -1_071_513_561_i32;
pub const STATUS_FWP_WRONG_SESSION: NTSTATUS = -1_071_513_588_i32;
pub const STATUS_FWP_ZERO_LENGTH_ARRAY: NTSTATUS = -1_071_513_566_i32;
pub const STATUS_GDI_HANDLE_LEAK: NTSTATUS = -2_143_354_879_i32;
pub const STATUS_GENERIC_COMMAND_FAILED: NTSTATUS = -1_072_365_530_i32;
pub const STATUS_GENERIC_NOT_MAPPED: NTSTATUS = -1_073_741_594_i32;
pub const STATUS_GHOSTED: NTSTATUS = 303_i32;
pub const STATUS_GPIO_CLIENT_INFORMATION_INVALID: NTSTATUS = -1_073_700_574_i32;
pub const STATUS_GPIO_INCOMPATIBLE_CONNECT_MODE: NTSTATUS = -1_073_700_570_i32;
pub const STATUS_GPIO_INTERRUPT_ALREADY_UNMASKED: NTSTATUS = -2_147_442_393_i32;
pub const STATUS_GPIO_INVALID_REGISTRATION_PACKET: NTSTATUS = -1_073_700_572_i32;
pub const STATUS_GPIO_OPERATION_DENIED: NTSTATUS = -1_073_700_571_i32;
pub const STATUS_GPIO_VERSION_NOT_SUPPORTED: NTSTATUS = -1_073_700_573_i32;
pub const STATUS_GRACEFUL_DISCONNECT: NTSTATUS = -1_073_741_257_i32;
pub const STATUS_GRAPHICS_ADAPTER_ACCESS_NOT_EXCLUDED: NTSTATUS = -1_071_774_661_i32;
pub const STATUS_GRAPHICS_ADAPTER_CHAIN_NOT_READY: NTSTATUS = -1_071_774_669_i32;
pub const STATUS_GRAPHICS_ADAPTER_MUST_HAVE_AT_LEAST_ONE_SOURCE: NTSTATUS = -1_071_774_936_i32;
pub const STATUS_GRAPHICS_ADAPTER_MUST_HAVE_AT_LEAST_ONE_TARGET: NTSTATUS = -1_071_774_935_i32;
pub const STATUS_GRAPHICS_ADAPTER_WAS_RESET: NTSTATUS = -1_071_775_741_i32;
pub const STATUS_GRAPHICS_ALLOCATION_BUSY: NTSTATUS = -1_071_775_486_i32;
pub const STATUS_GRAPHICS_ALLOCATION_CLOSED: NTSTATUS = -1_071_775_470_i32;
pub const STATUS_GRAPHICS_ALLOCATION_CONTENT_LOST: NTSTATUS = -1_071_775_466_i32;
pub const STATUS_GRAPHICS_ALLOCATION_INVALID: NTSTATUS = -1_071_775_482_i32;
pub const STATUS_GRAPHICS_CANCEL_VIDPN_TOPOLOGY_AUGMENTATION: NTSTATUS = -1_071_774_886_i32;
pub const STATUS_GRAPHICS_CANNOTCOLORCONVERT: NTSTATUS = -1_071_775_736_i32;
pub const STATUS_GRAPHICS_CANT_ACCESS_ACTIVE_VIDPN: NTSTATUS = -1_071_774_909_i32;
pub const STATUS_GRAPHICS_CANT_EVICT_PINNED_ALLOCATION: NTSTATUS = -1_071_775_479_i32;
pub const STATUS_GRAPHICS_CANT_LOCK_MEMORY: NTSTATUS = -1_071_775_487_i32;
pub const STATUS_GRAPHICS_CANT_RENDER_LOCKED_ALLOCATION: NTSTATUS = -1_071_775_471_i32;
pub const STATUS_GRAPHICS_CHAINLINKS_NOT_ENUMERATED: NTSTATUS = -1_071_774_670_i32;
pub const STATUS_GRAPHICS_CHAINLINKS_NOT_POWERED_ON: NTSTATUS = -1_071_774_667_i32;
pub const STATUS_GRAPHICS_CHAINLINKS_NOT_STARTED: NTSTATUS = -1_071_774_668_i32;
pub const STATUS_GRAPHICS_CHILD_DESCRIPTOR_NOT_SUPPORTED: NTSTATUS = -1_071_774_719_i32;
pub const STATUS_GRAPHICS_CLIENTVIDPN_NOT_SET: NTSTATUS = -1_071_774_884_i32;
pub const STATUS_GRAPHICS_COPP_NOT_SUPPORTED: NTSTATUS = -1_071_774_463_i32;
pub const STATUS_GRAPHICS_DATASET_IS_EMPTY: NTSTATUS = 1_075_708_747_i32;
pub const STATUS_GRAPHICS_DDCCI_INVALID_CAPABILITIES_STRING: NTSTATUS = -1_071_774_329_i32;
pub const STATUS_GRAPHICS_DDCCI_INVALID_DATA: NTSTATUS = -1_071_774_331_i32;
pub const STATUS_GRAPHICS_DDCCI_INVALID_MESSAGE_CHECKSUM: NTSTATUS = -1_071_774_325_i32;
pub const STATUS_GRAPHICS_DDCCI_INVALID_MESSAGE_COMMAND: NTSTATUS = -1_071_774_327_i32;
pub const STATUS_GRAPHICS_DDCCI_INVALID_MESSAGE_LENGTH: NTSTATUS = -1_071_774_326_i32;
pub const STATUS_GRAPHICS_DDCCI_MONITOR_RETURNED_INVALID_TIMING_STATUS_BYTE: NTSTATUS =
    -1_071_774_330_i32;
pub const STATUS_GRAPHICS_DDCCI_VCP_NOT_SUPPORTED: NTSTATUS = -1_071_774_332_i32;
pub const STATUS_GRAPHICS_DEPENDABLE_CHILD_STATUS: NTSTATUS = 1_075_708_988_i32;
pub const STATUS_GRAPHICS_DISPLAY_DEVICE_NOT_ATTACHED_TO_DESKTOP: NTSTATUS = -1_071_774_238_i32;
pub const STATUS_GRAPHICS_DRIVER_MISMATCH: NTSTATUS = -1_071_775_735_i32;
pub const STATUS_GRAPHICS_EMPTY_ADAPTER_MONITOR_MODE_SUPPORT_INTERSECTION: NTSTATUS =
    -1_071_774_939_i32;
pub const STATUS_GRAPHICS_FREQUENCYRANGE_ALREADY_IN_SET: NTSTATUS = -1_071_774_945_i32;
pub const STATUS_GRAPHICS_FREQUENCYRANGE_NOT_IN_SET: NTSTATUS = -1_071_774_947_i32;
pub const STATUS_GRAPHICS_GAMMA_RAMP_NOT_SUPPORTED: NTSTATUS = -1_071_774_904_i32;
pub const STATUS_GRAPHICS_GPU_EXCEPTION_ON_DEVICE: NTSTATUS = -1_071_775_232_i32;
pub const STATUS_GRAPHICS_I2C_DEVICE_DOES_NOT_EXIST: NTSTATUS = -1_071_774_335_i32;
pub const STATUS_GRAPHICS_I2C_ERROR_RECEIVING_DATA: NTSTATUS = -1_071_774_333_i32;
pub const STATUS_GRAPHICS_I2C_ERROR_TRANSMITTING_DATA: NTSTATUS = -1_071_774_334_i32;
pub const STATUS_GRAPHICS_I2C_NOT_SUPPORTED: NTSTATUS = -1_071_774_336_i32;
pub const STATUS_GRAPHICS_INCOMPATIBLE_PRIVATE_FORMAT: NTSTATUS = -1_071_774_891_i32;
pub const STATUS_GRAPHICS_INCONSISTENT_DEVICE_LINK_STATE: NTSTATUS = -1_071_774_666_i32;
pub const STATUS_GRAPHICS_INDIRECT_DISPLAY_ABANDON_SWAPCHAIN: NTSTATUS = -1_071_775_726_i32;
pub const STATUS_GRAPHICS_INDIRECT_DISPLAY_DEVICE_STOPPED: NTSTATUS = -1_071_775_725_i32;
pub const STATUS_GRAPHICS_INSUFFICIENT_DMA_BUFFER: NTSTATUS = -1_071_775_743_i32;
pub const STATUS_GRAPHICS_INTERNAL_ERROR: NTSTATUS = -1_071_774_233_i32;
pub const STATUS_GRAPHICS_INVALID_ACTIVE_REGION: NTSTATUS = -1_071_774_965_i32;
pub const STATUS_GRAPHICS_INVALID_ALLOCATION_HANDLE: NTSTATUS = -1_071_775_468_i32;
pub const STATUS_GRAPHICS_INVALID_ALLOCATION_INSTANCE: NTSTATUS = -1_071_775_469_i32;
pub const STATUS_GRAPHICS_INVALID_ALLOCATION_USAGE: NTSTATUS = -1_071_775_472_i32;
pub const STATUS_GRAPHICS_INVALID_CLIENT_TYPE: NTSTATUS = -1_071_774_885_i32;
pub const STATUS_GRAPHICS_INVALID_COLORBASIS: NTSTATUS = -1_071_774_914_i32;
pub const STATUS_GRAPHICS_INVALID_COPYPROTECTION_TYPE: NTSTATUS = -1_071_774_897_i32;
pub const STATUS_GRAPHICS_INVALID_DISPLAY_ADAPTER: NTSTATUS = -1_071_775_742_i32;
pub const STATUS_GRAPHICS_INVALID_DRIVER_MODEL: NTSTATUS = -1_071_775_740_i32;
pub const STATUS_GRAPHICS_INVALID_FREQUENCY: NTSTATUS = -1_071_774_966_i32;
pub const STATUS_GRAPHICS_INVALID_GAMMA_RAMP: NTSTATUS = -1_071_774_905_i32;
pub const STATUS_GRAPHICS_INVALID_MODE_PRUNING_ALGORITHM: NTSTATUS = -1_071_774_890_i32;
pub const STATUS_GRAPHICS_INVALID_MONITORDESCRIPTOR: NTSTATUS = -1_071_774_933_i32;
pub const STATUS_GRAPHICS_INVALID_MONITORDESCRIPTORSET: NTSTATUS = -1_071_774_934_i32;
pub const STATUS_GRAPHICS_INVALID_MONITOR_CAPABILITY_ORIGIN: NTSTATUS = -1_071_774_889_i32;
pub const STATUS_GRAPHICS_INVALID_MONITOR_FREQUENCYRANGE: NTSTATUS = -1_071_774_948_i32;
pub const STATUS_GRAPHICS_INVALID_MONITOR_FREQUENCYRANGESET: NTSTATUS = -1_071_774_949_i32;
pub const STATUS_GRAPHICS_INVALID_MONITOR_FREQUENCYRANGE_CONSTRAINT: NTSTATUS = -1_071_774_888_i32;
pub const STATUS_GRAPHICS_INVALID_MONITOR_SOURCEMODESET: NTSTATUS = -1_071_774_943_i32;
pub const STATUS_GRAPHICS_INVALID_MONITOR_SOURCE_MODE: NTSTATUS = -1_071_774_942_i32;
pub const STATUS_GRAPHICS_INVALID_PATH_CONTENT_GEOMETRY_TRANSFORMATION: NTSTATUS =
    -1_071_774_907_i32;
pub const STATUS_GRAPHICS_INVALID_PATH_CONTENT_TYPE: NTSTATUS = -1_071_774_898_i32;
pub const STATUS_GRAPHICS_INVALID_PATH_IMPORTANCE_ORDINAL: NTSTATUS = -1_071_774_908_i32;
pub const STATUS_GRAPHICS_INVALID_PHYSICAL_MONITOR_HANDLE: NTSTATUS = -1_071_774_324_i32;
pub const STATUS_GRAPHICS_INVALID_PIXELFORMAT: NTSTATUS = -1_071_774_915_i32;
pub const STATUS_GRAPHICS_INVALID_PIXELVALUEACCESSMODE: NTSTATUS = -1_071_774_913_i32;
pub const STATUS_GRAPHICS_INVALID_POINTER: NTSTATUS = -1_071_774_236_i32;
pub const STATUS_GRAPHICS_INVALID_PRIMARYSURFACE_SIZE: NTSTATUS = -1_071_774_918_i32;
pub const STATUS_GRAPHICS_INVALID_SCANLINE_ORDERING: NTSTATUS = -1_071_774_894_i32;
pub const STATUS_GRAPHICS_INVALID_STRIDE: NTSTATUS = -1_071_774_916_i32;
pub const STATUS_GRAPHICS_INVALID_TOTAL_REGION: NTSTATUS = -1_071_774_964_i32;
pub const STATUS_GRAPHICS_INVALID_VIDEOPRESENTSOURCESET: NTSTATUS = -1_071_774_955_i32;
pub const STATUS_GRAPHICS_INVALID_VIDEOPRESENTTARGETSET: NTSTATUS = -1_071_774_954_i32;
pub const STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_SOURCE: NTSTATUS = -1_071_774_972_i32;
pub const STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_SOURCE_MODE: NTSTATUS = -1_071_774_960_i32;
pub const STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_TARGET: NTSTATUS = -1_071_774_971_i32;
pub const STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_TARGET_MODE: NTSTATUS = -1_071_774_959_i32;
pub const STATUS_GRAPHICS_INVALID_VIDPN: NTSTATUS = -1_071_774_973_i32;
pub const STATUS_GRAPHICS_INVALID_VIDPN_PRESENT_PATH: NTSTATUS = -1_071_774_951_i32;
pub const STATUS_GRAPHICS_INVALID_VIDPN_SOURCEMODESET: NTSTATUS = -1_071_774_968_i32;
pub const STATUS_GRAPHICS_INVALID_VIDPN_TARGETMODESET: NTSTATUS = -1_071_774_967_i32;
pub const STATUS_GRAPHICS_INVALID_VIDPN_TARGET_SUBSET_TYPE: NTSTATUS = -1_071_774_929_i32;
pub const STATUS_GRAPHICS_INVALID_VIDPN_TOPOLOGY: NTSTATUS = -1_071_774_976_i32;
pub const STATUS_GRAPHICS_INVALID_VIDPN_TOPOLOGY_RECOMMENDATION_REASON: NTSTATUS =
    -1_071_774_899_i32;
pub const STATUS_GRAPHICS_INVALID_VISIBLEREGION_SIZE: NTSTATUS = -1_071_774_917_i32;
pub const STATUS_GRAPHICS_LEADLINK_NOT_ENUMERATED: NTSTATUS = -1_071_774_671_i32;
pub const STATUS_GRAPHICS_LEADLINK_START_DEFERRED: NTSTATUS = 1_075_708_983_i32;
pub const STATUS_GRAPHICS_LINK_CONFIGURATION_IN_PROGRESS: NTSTATUS = -2_145_517_568_i32;
pub const STATUS_GRAPHICS_MAX_NUM_PATHS_REACHED: NTSTATUS = -1_071_774_887_i32;
pub const STATUS_GRAPHICS_MCA_INTERNAL_ERROR: NTSTATUS = -1_071_774_328_i32;
pub const STATUS_GRAPHICS_MIRRORING_DEVICES_NOT_SUPPORTED: NTSTATUS = -1_071_774_237_i32;
pub const STATUS_GRAPHICS_MODE_ALREADY_IN_MODESET: NTSTATUS = -1_071_774_956_i32;
pub const STATUS_GRAPHICS_MODE_ID_MUST_BE_UNIQUE: NTSTATUS = -1_071_774_940_i32;
pub const STATUS_GRAPHICS_MODE_NOT_IN_MODESET: NTSTATUS = -1_071_774_902_i32;
pub const STATUS_GRAPHICS_MODE_NOT_PINNED: NTSTATUS = 1_075_708_679_i32;
pub const STATUS_GRAPHICS_MONITORDESCRIPTOR_ALREADY_IN_SET: NTSTATUS = -1_071_774_931_i32;
pub const STATUS_GRAPHICS_MONITORDESCRIPTOR_ID_MUST_BE_UNIQUE: NTSTATUS = -1_071_774_930_i32;
pub const STATUS_GRAPHICS_MONITORDESCRIPTOR_NOT_IN_SET: NTSTATUS = -1_071_774_932_i32;
pub const STATUS_GRAPHICS_MONITOR_COULD_NOT_BE_ASSOCIATED_WITH_ADAPTER: NTSTATUS =
    -1_071_774_924_i32;
pub const STATUS_GRAPHICS_MONITOR_NOT_CONNECTED: NTSTATUS = -1_071_774_920_i32;
pub const STATUS_GRAPHICS_MONITOR_NO_LONGER_EXISTS: NTSTATUS = -1_071_774_323_i32;
pub const STATUS_GRAPHICS_MPO_ALLOCATION_UNPINNED: NTSTATUS = -1_071_775_720_i32;
pub const STATUS_GRAPHICS_MULTISAMPLING_NOT_SUPPORTED: NTSTATUS = -1_071_774_903_i32;
pub const STATUS_GRAPHICS_NOT_A_LINKED_ADAPTER: NTSTATUS = -1_071_774_672_i32;
pub const STATUS_GRAPHICS_NOT_EXCLUSIVE_MODE_OWNER: NTSTATUS = -1_071_775_744_i32;
pub const STATUS_GRAPHICS_NOT_POST_DEVICE_DRIVER: NTSTATUS = -1_071_774_664_i32;
pub const STATUS_GRAPHICS_NO_ACTIVE_VIDPN: NTSTATUS = -1_071_774_922_i32;
pub const STATUS_GRAPHICS_NO_AVAILABLE_IMPORTANCE_ORDINALS: NTSTATUS = -1_071_774_892_i32;
pub const STATUS_GRAPHICS_NO_AVAILABLE_VIDPN_TARGET: NTSTATUS = -1_071_774_925_i32;
pub const STATUS_GRAPHICS_NO_DISPLAY_DEVICE_CORRESPONDS_TO_NAME: NTSTATUS = -1_071_774_239_i32;
pub const STATUS_GRAPHICS_NO_DISPLAY_MODE_MANAGEMENT_SUPPORT: NTSTATUS = -1_071_774_911_i32;
pub const STATUS_GRAPHICS_NO_MONITORS_CORRESPOND_TO_DISPLAY_DEVICE: NTSTATUS = -1_071_774_235_i32;
pub const STATUS_GRAPHICS_NO_MORE_ELEMENTS_IN_DATASET: NTSTATUS = 1_075_708_748_i32;
pub const STATUS_GRAPHICS_NO_PREFERRED_MODE: NTSTATUS = 1_075_708_702_i32;
pub const STATUS_GRAPHICS_NO_RECOMMENDED_FUNCTIONAL_VIDPN: NTSTATUS = -1_071_774_941_i32;
pub const STATUS_GRAPHICS_NO_RECOMMENDED_VIDPN_TOPOLOGY: NTSTATUS = -1_071_774_950_i32;
pub const STATUS_GRAPHICS_NO_VIDEO_MEMORY: NTSTATUS = -1_071_775_488_i32;
pub const STATUS_GRAPHICS_NO_VIDPNMGR: NTSTATUS = -1_071_774_923_i32;
pub const STATUS_GRAPHICS_ONLY_CONSOLE_SESSION_SUPPORTED: NTSTATUS = -1_071_774_240_i32;
pub const STATUS_GRAPHICS_OPM_ALL_HDCP_HARDWARE_ALREADY_IN_USE: NTSTATUS = -1_071_774_440_i32;
pub const STATUS_GRAPHICS_OPM_DRIVER_INTERNAL_ERROR: NTSTATUS = -1_071_774_434_i32;
pub const STATUS_GRAPHICS_OPM_HDCP_SRM_NEVER_SET: NTSTATUS = -1_071_774_442_i32;
pub const STATUS_GRAPHICS_OPM_INTERNAL_ERROR: NTSTATUS = -1_071_774_453_i32;
pub const STATUS_GRAPHICS_OPM_INVALID_CONFIGURATION_REQUEST: NTSTATUS = -1_071_774_431_i32;
pub const STATUS_GRAPHICS_OPM_INVALID_ENCRYPTED_PARAMETERS: NTSTATUS = -1_071_774_461_i32;
pub const STATUS_GRAPHICS_OPM_INVALID_HANDLE: NTSTATUS = -1_071_774_452_i32;
pub const STATUS_GRAPHICS_OPM_INVALID_INFORMATION_REQUEST: NTSTATUS = -1_071_774_435_i32;
pub const STATUS_GRAPHICS_OPM_INVALID_SRM: NTSTATUS = -1_071_774_446_i32;
pub const STATUS_GRAPHICS_OPM_NOT_SUPPORTED: NTSTATUS = -1_071_774_464_i32;
pub const STATUS_GRAPHICS_OPM_NO_PROTECTED_OUTPUTS_EXIST: NTSTATUS = -1_071_774_459_i32;
pub const STATUS_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_ACP: NTSTATUS = -1_071_774_444_i32;
pub const STATUS_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_CGMSA: NTSTATUS = -1_071_774_443_i32;
pub const STATUS_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_HDCP: NTSTATUS = -1_071_774_445_i32;
pub const STATUS_GRAPHICS_OPM_PROTECTED_OUTPUT_DOES_NOT_HAVE_COPP_SEMANTICS: NTSTATUS =
    -1_071_774_436_i32;
pub const STATUS_GRAPHICS_OPM_PROTECTED_OUTPUT_DOES_NOT_HAVE_OPM_SEMANTICS: NTSTATUS =
    -1_071_774_433_i32;
pub const STATUS_GRAPHICS_OPM_PROTECTED_OUTPUT_NO_LONGER_EXISTS: NTSTATUS = -1_071_774_438_i32;
pub const STATUS_GRAPHICS_OPM_RESOLUTION_TOO_HIGH: NTSTATUS = -1_071_774_441_i32;
pub const STATUS_GRAPHICS_OPM_SIGNALING_NOT_SUPPORTED: NTSTATUS = -1_071_774_432_i32;
pub const STATUS_GRAPHICS_OPM_SPANNING_MODE_ENABLED: NTSTATUS = -1_071_774_449_i32;
pub const STATUS_GRAPHICS_OPM_THEATER_MODE_ENABLED: NTSTATUS = -1_071_774_448_i32;
pub const STATUS_GRAPHICS_PARAMETER_ARRAY_TOO_SMALL: NTSTATUS = -1_071_774_234_i32;
pub const STATUS_GRAPHICS_PARTIAL_DATA_POPULATED: NTSTATUS = 1_075_707_914_i32;
pub const STATUS_GRAPHICS_PATH_ALREADY_IN_TOPOLOGY: NTSTATUS = -1_071_774_957_i32;
pub const STATUS_GRAPHICS_PATH_CONTENT_GEOMETRY_TRANSFORMATION_NOT_PINNED: NTSTATUS =
    1_075_708_753_i32;
pub const STATUS_GRAPHICS_PATH_CONTENT_GEOMETRY_TRANSFORMATION_NOT_SUPPORTED: NTSTATUS =
    -1_071_774_906_i32;
pub const STATUS_GRAPHICS_PATH_NOT_IN_TOPOLOGY: NTSTATUS = -1_071_774_937_i32;
pub const STATUS_GRAPHICS_PINNED_MODE_MUST_REMAIN_IN_SET: NTSTATUS = -1_071_774_958_i32;
pub const STATUS_GRAPHICS_POLLING_TOO_FREQUENTLY: NTSTATUS = 1_075_708_985_i32;
pub const STATUS_GRAPHICS_PRESENT_BUFFER_NOT_BOUND: NTSTATUS = -1_071_775_728_i32;
pub const STATUS_GRAPHICS_PRESENT_DENIED: NTSTATUS = -1_071_775_737_i32;
pub const STATUS_GRAPHICS_PRESENT_INVALID_WINDOW: NTSTATUS = -1_071_775_729_i32;
pub const STATUS_GRAPHICS_PRESENT_MODE_CHANGED: NTSTATUS = -1_071_775_739_i32;
pub const STATUS_GRAPHICS_PRESENT_OCCLUDED: NTSTATUS = -1_071_775_738_i32;
pub const STATUS_GRAPHICS_PRESENT_REDIRECTION_DISABLED: NTSTATUS = -1_071_775_733_i32;
pub const STATUS_GRAPHICS_PRESENT_UNOCCLUDED: NTSTATUS = -1_071_775_732_i32;
pub const STATUS_GRAPHICS_PVP_HFS_FAILED: NTSTATUS = -1_071_774_447_i32;
pub const STATUS_GRAPHICS_PVP_INVALID_CERTIFICATE_LENGTH: NTSTATUS = -1_071_774_450_i32;
pub const STATUS_GRAPHICS_RESOURCES_NOT_RELATED: NTSTATUS = -1_071_774_928_i32;
pub const STATUS_GRAPHICS_SESSION_TYPE_CHANGE_IN_PROGRESS: NTSTATUS = -1_071_774_232_i32;
pub const STATUS_GRAPHICS_SKIP_ALLOCATION_PREPARATION: NTSTATUS = 1_075_708_417_i32;
pub const STATUS_GRAPHICS_SOURCE_ALREADY_IN_SET: NTSTATUS = -1_071_774_953_i32;
pub const STATUS_GRAPHICS_SOURCE_ID_MUST_BE_UNIQUE: NTSTATUS = -1_071_774_927_i32;
pub const STATUS_GRAPHICS_SOURCE_NOT_IN_TOPOLOGY: NTSTATUS = -1_071_774_919_i32;
pub const STATUS_GRAPHICS_SPECIFIED_CHILD_ALREADY_CONNECTED: NTSTATUS = -1_071_774_720_i32;
pub const STATUS_GRAPHICS_STALE_MODESET: NTSTATUS = -1_071_774_944_i32;
pub const STATUS_GRAPHICS_STALE_VIDPN_TOPOLOGY: NTSTATUS = -1_071_774_921_i32;
pub const STATUS_GRAPHICS_START_DEFERRED: NTSTATUS = 1_075_708_986_i32;
pub const STATUS_GRAPHICS_TARGET_ALREADY_IN_SET: NTSTATUS = -1_071_774_952_i32;
pub const STATUS_GRAPHICS_TARGET_ID_MUST_BE_UNIQUE: NTSTATUS = -1_071_774_926_i32;
pub const STATUS_GRAPHICS_TARGET_NOT_IN_TOPOLOGY: NTSTATUS = -1_071_774_912_i32;
pub const STATUS_GRAPHICS_TOO_MANY_REFERENCES: NTSTATUS = -1_071_775_485_i32;
pub const STATUS_GRAPHICS_TOPOLOGY_CHANGES_NOT_ALLOWED: NTSTATUS = -1_071_774_893_i32;
pub const STATUS_GRAPHICS_TRY_AGAIN_LATER: NTSTATUS = -1_071_775_484_i32;
pub const STATUS_GRAPHICS_TRY_AGAIN_NOW: NTSTATUS = -1_071_775_483_i32;
pub const STATUS_GRAPHICS_UAB_NOT_SUPPORTED: NTSTATUS = -1_071_774_462_i32;
pub const STATUS_GRAPHICS_UNASSIGNED_MODESET_ALREADY_EXISTS: NTSTATUS = -1_071_774_896_i32;
pub const STATUS_GRAPHICS_UNKNOWN_CHILD_STATUS: NTSTATUS = 1_075_708_975_i32;
pub const STATUS_GRAPHICS_UNSWIZZLING_APERTURE_UNAVAILABLE: NTSTATUS = -1_071_775_481_i32;
pub const STATUS_GRAPHICS_UNSWIZZLING_APERTURE_UNSUPPORTED: NTSTATUS = -1_071_775_480_i32;
pub const STATUS_GRAPHICS_VAIL_STATE_CHANGED: NTSTATUS = -1_071_775_727_i32;
pub const STATUS_GRAPHICS_VIDEO_PRESENT_TARGETS_LESS_THAN_SOURCES: NTSTATUS = -1_071_774_938_i32;
pub const STATUS_GRAPHICS_VIDPN_MODALITY_NOT_SUPPORTED: NTSTATUS = -1_071_774_970_i32;
pub const STATUS_GRAPHICS_VIDPN_SOURCE_IN_USE: NTSTATUS = -1_071_774_910_i32;
pub const STATUS_GRAPHICS_VIDPN_TOPOLOGY_CURRENTLY_NOT_SUPPORTED: NTSTATUS = -1_071_774_974_i32;
pub const STATUS_GRAPHICS_VIDPN_TOPOLOGY_NOT_SUPPORTED: NTSTATUS = -1_071_774_975_i32;
pub const STATUS_GRAPHICS_WINDOWDC_NOT_AVAILABLE: NTSTATUS = -1_071_775_731_i32;
pub const STATUS_GRAPHICS_WINDOWLESS_PRESENT_DISABLED: NTSTATUS = -1_071_775_730_i32;
pub const STATUS_GRAPHICS_WRONG_ALLOCATION_DEVICE: NTSTATUS = -1_071_775_467_i32;
pub const STATUS_GROUP_EXISTS: NTSTATUS = -1_073_741_723_i32;
pub const STATUS_GUARD_PAGE_VIOLATION: NTSTATUS = -2_147_483_647_i32;
pub const STATUS_GUIDS_EXHAUSTED: NTSTATUS = -1_073_741_693_i32;
pub const STATUS_GUID_SUBSTITUTION_MADE: NTSTATUS = -2_147_483_636_i32;
pub const STATUS_HANDLES_CLOSED: NTSTATUS = -2_147_483_638_i32;
pub const STATUS_HANDLE_NOT_CLOSABLE: NTSTATUS = -1_073_741_259_i32;
pub const STATUS_HANDLE_NO_LONGER_VALID: NTSTATUS = -1_072_103_384_i32;
pub const STATUS_HANDLE_REVOKED: NTSTATUS = -1_073_700_858_i32;
pub const STATUS_HARDWARE_MEMORY_ERROR: NTSTATUS = -1_073_740_023_i32;
pub const STATUS_HASH_NOT_PRESENT: NTSTATUS = -1_073_700_607_i32;
pub const STATUS_HASH_NOT_SUPPORTED: NTSTATUS = -1_073_700_608_i32;
pub const STATUS_HAS_SYSTEM_CRITICAL_FILES: NTSTATUS = -1_073_740_611_i32;
pub const STATUS_HDAUDIO_CONNECTION_LIST_NOT_SUPPORTED: NTSTATUS = -1_069_285_373_i32;
pub const STATUS_HDAUDIO_EMPTY_CONNECTION_LIST: NTSTATUS = -1_069_285_374_i32;
pub const STATUS_HDAUDIO_NO_LOGICAL_DEVICES_CREATED: NTSTATUS = -1_069_285_372_i32;
pub const STATUS_HDAUDIO_NULL_LINKED_LIST_ENTRY: NTSTATUS = -1_069_285_371_i32;
pub const STATUS_HEAP_CORRUPTION: NTSTATUS = -1_073_740_940_i32;
pub const STATUS_HEURISTIC_DAMAGE_POSSIBLE: NTSTATUS = 1_075_380_225_i32;
pub const STATUS_HIBERNATED: NTSTATUS = 1_073_741_866_i32;
pub const STATUS_HIBERNATION_FAILURE: NTSTATUS = -1_073_740_783_i32;
pub const STATUS_HIVE_UNLOADED: NTSTATUS = -1_073_740_763_i32;
pub const STATUS_HMAC_NOT_SUPPORTED: NTSTATUS = -1_073_700_863_i32;
pub const STATUS_HOPLIMIT_EXCEEDED: NTSTATUS = -1_073_700_846_i32;
pub const STATUS_HOST_DOWN: NTSTATUS = -1_073_740_976_i32;
pub const STATUS_HOST_UNREACHABLE: NTSTATUS = -1_073_741_251_i32;
pub const STATUS_HUNG_DISPLAY_DRIVER_THREAD: NTSTATUS = -1_073_740_779_i32;
pub const STATUS_HV_ACCESS_DENIED: NTSTATUS = -1_070_268_410_i32;
pub const STATUS_HV_ACKNOWLEDGED: NTSTATUS = -1_070_268_394_i32;
pub const STATUS_HV_CALL_PENDING: NTSTATUS = -1_070_268_295_i32;
pub const STATUS_HV_CPUID_FEATURE_VALIDATION_ERROR: NTSTATUS = -1_070_268_356_i32;
pub const STATUS_HV_CPUID_XSAVE_FEATURE_VALIDATION_ERROR: NTSTATUS = -1_070_268_355_i32;
pub const STATUS_HV_DEVICE_NOT_IN_DOMAIN: NTSTATUS = -1_070_268_298_i32;
pub const STATUS_HV_EVENT_BUFFER_ALREADY_FREED: NTSTATUS = -1_070_268_300_i32;
pub const STATUS_HV_FEATURE_UNAVAILABLE: NTSTATUS = -1_070_268_386_i32;
pub const STATUS_HV_INACTIVE: NTSTATUS = -1_070_268_388_i32;
pub const STATUS_HV_INSUFFICIENT_BUFFER: NTSTATUS = -1_070_268_365_i32;
pub const STATUS_HV_INSUFFICIENT_BUFFERS: NTSTATUS = -1_070_268_397_i32;
pub const STATUS_HV_INSUFFICIENT_CONTIGUOUS_MEMORY: NTSTATUS = -1_070_268_299_i32;
pub const STATUS_HV_INSUFFICIENT_CONTIGUOUS_MEMORY_MIRRORING: NTSTATUS = -1_070_268_286_i32;
pub const STATUS_HV_INSUFFICIENT_CONTIGUOUS_ROOT_MEMORY: NTSTATUS = -1_070_268_285_i32;
pub const STATUS_HV_INSUFFICIENT_CONTIGUOUS_ROOT_MEMORY_MIRRORING: NTSTATUS = -1_070_268_283_i32;
pub const STATUS_HV_INSUFFICIENT_DEVICE_DOMAINS: NTSTATUS = -1_070_268_360_i32;
pub const STATUS_HV_INSUFFICIENT_MEMORY: NTSTATUS = -1_070_268_405_i32;
pub const STATUS_HV_INSUFFICIENT_MEMORY_MIRRORING: NTSTATUS = -1_070_268_287_i32;
pub const STATUS_HV_INSUFFICIENT_ROOT_MEMORY: NTSTATUS = -1_070_268_301_i32;
pub const STATUS_HV_INSUFFICIENT_ROOT_MEMORY_MIRRORING: NTSTATUS = -1_070_268_284_i32;
pub const STATUS_HV_INVALID_ALIGNMENT: NTSTATUS = -1_070_268_412_i32;
pub const STATUS_HV_INVALID_CONNECTION_ID: NTSTATUS = -1_070_268_398_i32;
pub const STATUS_HV_INVALID_CPU_GROUP_ID: NTSTATUS = -1_070_268_305_i32;
pub const STATUS_HV_INVALID_CPU_GROUP_STATE: NTSTATUS = -1_070_268_304_i32;
pub const STATUS_HV_INVALID_DEVICE_ID: NTSTATUS = -1_070_268_329_i32;
pub const STATUS_HV_INVALID_DEVICE_STATE: NTSTATUS = -1_070_268_328_i32;
pub const STATUS_HV_INVALID_HYPERCALL_CODE: NTSTATUS = -1_070_268_414_i32;
pub const STATUS_HV_INVALID_HYPERCALL_INPUT: NTSTATUS = -1_070_268_413_i32;
pub const STATUS_HV_INVALID_LP_INDEX: NTSTATUS = -1_070_268_351_i32;
pub const STATUS_HV_INVALID_PARAMETER: NTSTATUS = -1_070_268_411_i32;
pub const STATUS_HV_INVALID_PARTITION_ID: NTSTATUS = -1_070_268_403_i32;
pub const STATUS_HV_INVALID_PARTITION_STATE: NTSTATUS = -1_070_268_409_i32;
pub const STATUS_HV_INVALID_PORT_ID: NTSTATUS = -1_070_268_399_i32;
pub const STATUS_HV_INVALID_PROXIMITY_DOMAIN_INFO: NTSTATUS = -1_070_268_390_i32;
pub const STATUS_HV_INVALID_REGISTER_VALUE: NTSTATUS = -1_070_268_336_i32;
pub const STATUS_HV_INVALID_SAVE_RESTORE_STATE: NTSTATUS = -1_070_268_393_i32;
pub const STATUS_HV_INVALID_SYNIC_STATE: NTSTATUS = -1_070_268_392_i32;
pub const STATUS_HV_INVALID_VP_INDEX: NTSTATUS = -1_070_268_402_i32;
pub const STATUS_HV_INVALID_VP_STATE: NTSTATUS = -1_070_268_395_i32;
pub const STATUS_HV_INVALID_VTL_STATE: NTSTATUS = -1_070_268_335_i32;
pub const STATUS_HV_MSR_ACCESS_FAILED: NTSTATUS = -1_070_268_288_i32;
pub const STATUS_HV_NESTED_VM_EXIT: NTSTATUS = -1_070_268_297_i32;
pub const STATUS_HV_NOT_ACKNOWLEDGED: NTSTATUS = -1_070_268_396_i32;
pub const STATUS_HV_NOT_ALLOWED_WITH_NESTED_VIRT_ACTIVE: NTSTATUS = -1_070_268_302_i32;
pub const STATUS_HV_NOT_PRESENT: NTSTATUS = -1_070_264_320_i32;
pub const STATUS_HV_NO_DATA: NTSTATUS = -1_070_268_389_i32;
pub const STATUS_HV_NO_RESOURCES: NTSTATUS = -1_070_268_387_i32;
pub const STATUS_HV_NX_NOT_DETECTED: NTSTATUS = -1_070_268_331_i32;
pub const STATUS_HV_OBJECT_IN_USE: NTSTATUS = -1_070_268_391_i32;
pub const STATUS_HV_OPERATION_DENIED: NTSTATUS = -1_070_268_408_i32;
pub const STATUS_HV_OPERATION_FAILED: NTSTATUS = -1_070_268_303_i32;
pub const STATUS_HV_PAGE_REQUEST_INVALID: NTSTATUS = -1_070_268_320_i32;
pub const STATUS_HV_PARTITION_TOO_DEEP: NTSTATUS = -1_070_268_404_i32;
pub const STATUS_HV_PENDING_PAGE_REQUESTS: NTSTATUS = 3_473_497_i32;
pub const STATUS_HV_PROCESSOR_STARTUP_TIMEOUT: NTSTATUS = -1_070_268_354_i32;
pub const STATUS_HV_PROPERTY_VALUE_OUT_OF_RANGE: NTSTATUS = -1_070_268_406_i32;
pub const STATUS_HV_SMX_ENABLED: NTSTATUS = -1_070_268_353_i32;
pub const STATUS_HV_UNKNOWN_PROPERTY: NTSTATUS = -1_070_268_407_i32;
pub const STATUS_ILLEGAL_CHARACTER: NTSTATUS = -1_073_741_471_i32;
pub const STATUS_ILLEGAL_DLL_RELOCATION: NTSTATUS = -1_073_741_207_i32;
pub const STATUS_ILLEGAL_ELEMENT_ADDRESS: NTSTATUS = -1_073_741_179_i32;
pub const STATUS_ILLEGAL_FLOAT_CONTEXT: NTSTATUS = -1_073_741_494_i32;
pub const STATUS_ILLEGAL_FUNCTION: NTSTATUS = -1_073_741_649_i32;
pub const STATUS_ILLEGAL_INSTRUCTION: NTSTATUS = -1_073_741_795_i32;
pub const STATUS_ILL_FORMED_PASSWORD: NTSTATUS = -1_073_741_717_i32;
pub const STATUS_ILL_FORMED_SERVICE_ENTRY: NTSTATUS = -1_073_741_472_i32;
pub const STATUS_IMAGE_ALREADY_LOADED: NTSTATUS = -1_073_741_554_i32;
pub const STATUS_IMAGE_ALREADY_LOADED_AS_DLL: NTSTATUS = -1_073_741_411_i32;
pub const STATUS_IMAGE_AT_DIFFERENT_BASE: NTSTATUS = 1_073_741_878_i32;
pub const STATUS_IMAGE_CERT_EXPIRED: NTSTATUS = -1_073_740_283_i32;
pub const STATUS_IMAGE_CERT_REVOKED: NTSTATUS = -1_073_740_285_i32;
pub const STATUS_IMAGE_CHECKSUM_MISMATCH: NTSTATUS = -1_073_741_279_i32;
pub const STATUS_IMAGE_LOADED_AS_PATCH_IMAGE: NTSTATUS = -1_073_740_608_i32;
pub const STATUS_IMAGE_MACHINE_TYPE_MISMATCH: NTSTATUS = 1_073_741_838_i32;
pub const STATUS_IMAGE_MACHINE_TYPE_MISMATCH_EXE: NTSTATUS = 1_073_741_859_i32;
pub const STATUS_IMAGE_MP_UP_MISMATCH: NTSTATUS = -1_073_741_239_i32;
pub const STATUS_IMAGE_NOT_AT_BASE: NTSTATUS = 1_073_741_827_i32;
pub const STATUS_IMAGE_SUBSYSTEM_NOT_PRESENT: NTSTATUS = -1_073_741_405_i32;
pub const STATUS_IMPLEMENTATION_LIMIT: NTSTATUS = -1_073_740_757_i32;
pub const STATUS_INCOMPATIBLE_DRIVER_BLOCKED: NTSTATUS = -1_073_740_764_i32;
pub const STATUS_INCOMPATIBLE_FILE_MAP: NTSTATUS = -1_073_741_747_i32;
pub const STATUS_INCOMPATIBLE_WITH_GLOBAL_SHORT_NAME_REGISTRY_SETTING: NTSTATUS =
    -1_073_741_410_i32;
pub const STATUS_INCORRECT_ACCOUNT_TYPE: NTSTATUS = -1_073_700_727_i32;
pub const STATUS_INDEX_OUT_OF_BOUNDS: NTSTATUS = -1_073_740_591_i32;
pub const STATUS_INDOUBT_TRANSACTIONS_EXIST: NTSTATUS = -1_072_103_366_i32;
pub const STATUS_INFO_LENGTH_MISMATCH: NTSTATUS = -1_073_741_820_i32;
pub const STATUS_INSTANCE_NOT_AVAILABLE: NTSTATUS = -1_073_741_653_i32;
pub const STATUS_INSTRUCTION_MISALIGNMENT: NTSTATUS = -1_073_741_654_i32;
pub const STATUS_INSUFFICIENT_LOGON_INFO: NTSTATUS = -1_073_741_232_i32;
pub const STATUS_INSUFFICIENT_NVRAM_RESOURCES: NTSTATUS = -1_073_740_716_i32;
pub const STATUS_INSUFFICIENT_POWER: NTSTATUS = -1_073_741_090_i32;
pub const STATUS_INSUFFICIENT_RESOURCES: NTSTATUS = -1_073_741_670_i32;
pub const STATUS_INSUFFICIENT_RESOURCE_FOR_SPECIFIED_SHARED_SECTION_SIZE: NTSTATUS =
    -1_073_740_778_i32;
pub const STATUS_INSUFFICIENT_VIRTUAL_ADDR_RESOURCES: NTSTATUS = -1_073_740_606_i32;
pub const STATUS_INSUFF_SERVER_RESOURCES: NTSTATUS = -1_073_741_307_i32;
pub const STATUS_INTEGER_DIVIDE_BY_ZERO: NTSTATUS = -1_073_741_676_i32;
pub const STATUS_INTEGER_OVERFLOW: NTSTATUS = -1_073_741_675_i32;
pub const STATUS_INTERMIXED_KERNEL_EA_OPERATION: NTSTATUS = -1_073_740_687_i32;
pub const STATUS_INTERNAL_DB_CORRUPTION: NTSTATUS = -1_073_741_596_i32;
pub const STATUS_INTERNAL_DB_ERROR: NTSTATUS = -1_073_741_480_i32;
pub const STATUS_INTERNAL_ERROR: NTSTATUS = -1_073_741_595_i32;
pub const STATUS_INTERRUPTED: NTSTATUS = -1_073_740_523_i32;
pub const STATUS_INTERRUPT_STILL_CONNECTED: NTSTATUS = 296_i32;
pub const STATUS_INTERRUPT_VECTOR_ALREADY_CONNECTED: NTSTATUS = 295_i32;
pub const STATUS_INVALID_ACCOUNT_NAME: NTSTATUS = -1_073_741_726_i32;
pub const STATUS_INVALID_ACE_CONDITION: NTSTATUS = -1_073_741_406_i32;
pub const STATUS_INVALID_ACL: NTSTATUS = -1_073_741_705_i32;
pub const STATUS_INVALID_ADDRESS: NTSTATUS = -1_073_741_503_i32;
pub const STATUS_INVALID_ADDRESS_COMPONENT: NTSTATUS = -1_073_741_305_i32;
pub const STATUS_INVALID_ADDRESS_WILDCARD: NTSTATUS = -1_073_741_304_i32;
pub const STATUS_INVALID_BLOCK_LENGTH: NTSTATUS = -1_073_741_453_i32;
pub const STATUS_INVALID_BUFFER_SIZE: NTSTATUS = -1_073_741_306_i32;
pub const STATUS_INVALID_CAP: NTSTATUS = -1_073_740_539_i32;
pub const STATUS_INVALID_CID: NTSTATUS = -1_073_741_813_i32;
pub const STATUS_INVALID_COMPUTER_NAME: NTSTATUS = -1_073_741_534_i32;
pub const STATUS_INVALID_CONNECTION: NTSTATUS = -1_073_741_504_i32;
pub const STATUS_INVALID_CRUNTIME_PARAMETER: NTSTATUS = -1_073_740_777_i32;
pub const STATUS_INVALID_DEVICE_OBJECT_PARAMETER: NTSTATUS = -1_073_740_951_i32;
pub const STATUS_INVALID_DEVICE_REQUEST: NTSTATUS = -1_073_741_808_i32;
pub const STATUS_INVALID_DEVICE_STATE: NTSTATUS = -1_073_741_436_i32;
pub const STATUS_INVALID_DISPOSITION: NTSTATUS = -1_073_741_786_i32;
pub const STATUS_INVALID_DOMAIN_ROLE: NTSTATUS = -1_073_741_602_i32;
pub const STATUS_INVALID_DOMAIN_STATE: NTSTATUS = -1_073_741_603_i32;
pub const STATUS_INVALID_EA_FLAG: NTSTATUS = -2_147_483_627_i32;
pub const STATUS_INVALID_EA_NAME: NTSTATUS = -2_147_483_629_i32;
pub const STATUS_INVALID_EXCEPTION_HANDLER: NTSTATUS = -1_073_741_403_i32;
pub const STATUS_INVALID_FIELD_IN_PARAMETER_LIST: NTSTATUS = -1_073_740_683_i32;
pub const STATUS_INVALID_FILE_FOR_SECTION: NTSTATUS = -1_073_741_792_i32;
pub const STATUS_INVALID_GROUP_ATTRIBUTES: NTSTATUS = -1_073_741_660_i32;
pub const STATUS_INVALID_HANDLE: NTSTATUS = -1_073_741_816_i32;
pub const STATUS_INVALID_HW_PROFILE: NTSTATUS = -1_073_741_216_i32;
pub const STATUS_INVALID_IDN_NORMALIZATION: NTSTATUS = -1_073_740_010_i32;
pub const STATUS_INVALID_ID_AUTHORITY: NTSTATUS = -1_073_741_692_i32;
pub const STATUS_INVALID_IMAGE_FORMAT: NTSTATUS = -1_073_741_701_i32;
pub const STATUS_INVALID_IMAGE_HASH: NTSTATUS = -1_073_740_760_i32;
pub const STATUS_INVALID_IMAGE_LE_FORMAT: NTSTATUS = -1_073_741_522_i32;
pub const STATUS_INVALID_IMAGE_NE_FORMAT: NTSTATUS = -1_073_741_541_i32;
pub const STATUS_INVALID_IMAGE_NOT_MZ: NTSTATUS = -1_073_741_521_i32;
pub const STATUS_INVALID_IMAGE_PROTECT: NTSTATUS = -1_073_741_520_i32;
pub const STATUS_INVALID_IMAGE_WIN_16: NTSTATUS = -1_073_741_519_i32;
pub const STATUS_INVALID_IMAGE_WIN_32: NTSTATUS = -1_073_740_967_i32;
pub const STATUS_INVALID_IMAGE_WIN_64: NTSTATUS = -1_073_740_966_i32;
pub const STATUS_INVALID_IMPORT_OF_NON_DLL: NTSTATUS = -1_073_740_945_i32;
pub const STATUS_INVALID_INFO_CLASS: NTSTATUS = -1_073_741_821_i32;
pub const STATUS_INVALID_INITIATOR_TARGET_PATH: NTSTATUS = -1_073_740_681_i32;
pub const STATUS_INVALID_KERNEL_INFO_VERSION: NTSTATUS = -1_073_700_860_i32;
pub const STATUS_INVALID_LABEL: NTSTATUS = -1_073_740_730_i32;
pub const STATUS_INVALID_LDT_DESCRIPTOR: NTSTATUS = -1_073_741_542_i32;
pub const STATUS_INVALID_LDT_OFFSET: NTSTATUS = -1_073_741_543_i32;
pub const STATUS_INVALID_LDT_SIZE: NTSTATUS = -1_073_741_544_i32;
pub const STATUS_INVALID_LEVEL: NTSTATUS = -1_073_741_496_i32;
pub const STATUS_INVALID_LOCK_RANGE: NTSTATUS = -1_073_741_407_i32;
pub const STATUS_INVALID_LOCK_SEQUENCE: NTSTATUS = -1_073_741_794_i32;
pub const STATUS_INVALID_LOGON_HOURS: NTSTATUS = -1_073_741_713_i32;
pub const STATUS_INVALID_LOGON_TYPE: NTSTATUS = -1_073_741_557_i32;
pub const STATUS_INVALID_MEMBER: NTSTATUS = -1_073_741_445_i32;
pub const STATUS_INVALID_MESSAGE: NTSTATUS = -1_073_740_030_i32;
pub const STATUS_INVALID_NETWORK_RESPONSE: NTSTATUS = -1_073_741_629_i32;
pub const STATUS_INVALID_OFFSET_ALIGNMENT: NTSTATUS = -1_073_740_684_i32;
pub const STATUS_INVALID_OPLOCK_PROTOCOL: NTSTATUS = -1_073_741_597_i32;
pub const STATUS_INVALID_OWNER: NTSTATUS = -1_073_741_734_i32;
pub const STATUS_INVALID_PACKAGE_SID_LENGTH: NTSTATUS = -1_073_700_350_i32;
pub const STATUS_INVALID_PAGE_PROTECTION: NTSTATUS = -1_073_741_755_i32;
pub const STATUS_INVALID_PARAMETER: NTSTATUS = -1_073_741_811_i32;
pub const STATUS_INVALID_PARAMETER_1: NTSTATUS = -1_073_741_585_i32;
pub const STATUS_INVALID_PARAMETER_10: NTSTATUS = -1_073_741_576_i32;
pub const STATUS_INVALID_PARAMETER_11: NTSTATUS = -1_073_741_575_i32;
pub const STATUS_INVALID_PARAMETER_12: NTSTATUS = -1_073_741_574_i32;
pub const STATUS_INVALID_PARAMETER_2: NTSTATUS = -1_073_741_584_i32;
pub const STATUS_INVALID_PARAMETER_3: NTSTATUS = -1_073_741_583_i32;
pub const STATUS_INVALID_PARAMETER_4: NTSTATUS = -1_073_741_582_i32;
pub const STATUS_INVALID_PARAMETER_5: NTSTATUS = -1_073_741_581_i32;
pub const STATUS_INVALID_PARAMETER_6: NTSTATUS = -1_073_741_580_i32;
pub const STATUS_INVALID_PARAMETER_7: NTSTATUS = -1_073_741_579_i32;
pub const STATUS_INVALID_PARAMETER_8: NTSTATUS = -1_073_741_578_i32;
pub const STATUS_INVALID_PARAMETER_9: NTSTATUS = -1_073_741_577_i32;
pub const STATUS_INVALID_PARAMETER_MIX: NTSTATUS = -1_073_741_776_i32;
pub const STATUS_INVALID_PEP_INFO_VERSION: NTSTATUS = -1_073_700_859_i32;
pub const STATUS_INVALID_PIPE_STATE: NTSTATUS = -1_073_741_651_i32;
pub const STATUS_INVALID_PLUGPLAY_DEVICE_PATH: NTSTATUS = -1_073_741_215_i32;
pub const STATUS_INVALID_PORT_ATTRIBUTES: NTSTATUS = -1_073_741_778_i32;
pub const STATUS_INVALID_PORT_HANDLE: NTSTATUS = -1_073_741_758_i32;
pub const STATUS_INVALID_PRIMARY_GROUP: NTSTATUS = -1_073_741_733_i32;
pub const STATUS_INVALID_QUOTA_LOWER: NTSTATUS = -1_073_741_775_i32;
pub const STATUS_INVALID_READ_MODE: NTSTATUS = -1_073_741_644_i32;
pub const STATUS_INVALID_RUNLEVEL_SETTING: NTSTATUS = -1_073_700_542_i32;
pub const STATUS_INVALID_SECURITY_DESCR: NTSTATUS = -1_073_741_703_i32;
pub const STATUS_INVALID_SERVER_STATE: NTSTATUS = -1_073_741_604_i32;
pub const STATUS_INVALID_SESSION: NTSTATUS = -1_073_740_715_i32;
pub const STATUS_INVALID_SID: NTSTATUS = -1_073_741_704_i32;
pub const STATUS_INVALID_SIGNATURE: NTSTATUS = -1_073_700_864_i32;
pub const STATUS_INVALID_STATE_TRANSITION: NTSTATUS = -1_073_700_861_i32;
pub const STATUS_INVALID_SUB_AUTHORITY: NTSTATUS = -1_073_741_706_i32;
pub const STATUS_INVALID_SYSTEM_SERVICE: NTSTATUS = -1_073_741_796_i32;
pub const STATUS_INVALID_TASK_INDEX: NTSTATUS = -1_073_740_543_i32;
pub const STATUS_INVALID_TASK_NAME: NTSTATUS = -1_073_740_544_i32;
pub const STATUS_INVALID_THREAD: NTSTATUS = -1_073_740_004_i32;
pub const STATUS_INVALID_TOKEN: NTSTATUS = -1_073_740_699_i32;
pub const STATUS_INVALID_TRANSACTION: NTSTATUS = -1_072_103_422_i32;
pub const STATUS_INVALID_UNWIND_TARGET: NTSTATUS = -1_073_741_783_i32;
pub const STATUS_INVALID_USER_BUFFER: NTSTATUS = -1_073_741_592_i32;
pub const STATUS_INVALID_USER_PRINCIPAL_NAME: NTSTATUS = -1_073_740_772_i32;
pub const STATUS_INVALID_VARIANT: NTSTATUS = -1_073_741_262_i32;
pub const STATUS_INVALID_VIEW_SIZE: NTSTATUS = -1_073_741_793_i32;
pub const STATUS_INVALID_VOLUME_LABEL: NTSTATUS = -1_073_741_690_i32;
pub const STATUS_INVALID_WEIGHT: NTSTATUS = -1_073_740_712_i32;
pub const STATUS_INVALID_WORKSTATION: NTSTATUS = -1_073_741_712_i32;
pub const STATUS_IN_PAGE_ERROR: NTSTATUS = -1_073_741_818_i32;
pub const STATUS_IORING_COMPLETION_QUEUE_TOO_BIG: NTSTATUS = -1_069_154_299_i32;
pub const STATUS_IORING_COMPLETION_QUEUE_TOO_FULL: NTSTATUS = -1_069_154_296_i32;
pub const STATUS_IORING_CORRUPT: NTSTATUS = -1_069_154_297_i32;
pub const STATUS_IORING_REQUIRED_FLAG_NOT_SUPPORTED: NTSTATUS = -1_069_154_303_i32;
pub const STATUS_IORING_SUBMISSION_QUEUE_FULL: NTSTATUS = -1_069_154_302_i32;
pub const STATUS_IORING_SUBMISSION_QUEUE_TOO_BIG: NTSTATUS = -1_069_154_300_i32;
pub const STATUS_IORING_SUBMIT_IN_PROGRESS: NTSTATUS = -1_069_154_298_i32;
pub const STATUS_IORING_VERSION_NOT_SUPPORTED: NTSTATUS = -1_069_154_301_i32;
pub const STATUS_IO_DEVICE_ERROR: NTSTATUS = -1_073_741_435_i32;
pub const STATUS_IO_DEVICE_INVALID_DATA: NTSTATUS = -1_073_741_392_i32;
pub const STATUS_IO_OPERATION_TIMEOUT: NTSTATUS = -1_073_740_675_i32;
pub const STATUS_IO_PREEMPTED: NTSTATUS = -1_068_433_407_i32;
pub const STATUS_IO_PRIVILEGE_FAILED: NTSTATUS = -1_073_741_513_i32;
pub const STATUS_IO_REISSUE_AS_CACHED: NTSTATUS = -1_073_479_623_i32;
pub const STATUS_IO_REPARSE_DATA_INVALID: NTSTATUS = -1_073_741_192_i32;
pub const STATUS_IO_REPARSE_TAG_INVALID: NTSTATUS = -1_073_741_194_i32;
pub const STATUS_IO_REPARSE_TAG_MISMATCH: NTSTATUS = -1_073_741_193_i32;
pub const STATUS_IO_REPARSE_TAG_NOT_HANDLED: NTSTATUS = -1_073_741_191_i32;
pub const STATUS_IO_TIMEOUT: NTSTATUS = -1_073_741_643_i32;
pub const STATUS_IO_UNALIGNED_WRITE: NTSTATUS = -1_073_741_391_i32;
pub const STATUS_IPSEC_AUTH_FIREWALL_DROP: NTSTATUS = -1_070_202_872_i32;
pub const STATUS_IPSEC_BAD_SPI: NTSTATUS = -1_070_202_879_i32;
pub const STATUS_IPSEC_CLEAR_TEXT_DROP: NTSTATUS = -1_070_202_873_i32;
pub const STATUS_IPSEC_DOSP_BLOCK: NTSTATUS = -1_070_170_112_i32;
pub const STATUS_IPSEC_DOSP_INVALID_PACKET: NTSTATUS = -1_070_170_110_i32;
pub const STATUS_IPSEC_DOSP_KEYMOD_NOT_ALLOWED: NTSTATUS = -1_070_170_107_i32;
pub const STATUS_IPSEC_DOSP_MAX_ENTRIES: NTSTATUS = -1_070_170_108_i32;
pub const STATUS_IPSEC_DOSP_MAX_PER_IP_RATELIMIT_QUEUES: NTSTATUS = -1_070_170_106_i32;
pub const STATUS_IPSEC_DOSP_RECEIVED_MULTICAST: NTSTATUS = -1_070_170_111_i32;
pub const STATUS_IPSEC_DOSP_STATE_LOOKUP_FAILED: NTSTATUS = -1_070_170_109_i32;
pub const STATUS_IPSEC_INTEGRITY_CHECK_FAILED: NTSTATUS = -1_070_202_874_i32;
pub const STATUS_IPSEC_INVALID_PACKET: NTSTATUS = -1_070_202_875_i32;
pub const STATUS_IPSEC_QUEUE_OVERFLOW: NTSTATUS = -1_073_700_848_i32;
pub const STATUS_IPSEC_REPLAY_CHECK_FAILED: NTSTATUS = -1_070_202_876_i32;
pub const STATUS_IPSEC_SA_LIFETIME_EXPIRED: NTSTATUS = -1_070_202_878_i32;
pub const STATUS_IPSEC_THROTTLE_DROP: NTSTATUS = -1_070_202_871_i32;
pub const STATUS_IPSEC_WRONG_SA: NTSTATUS = -1_070_202_877_i32;
pub const STATUS_IP_ADDRESS_CONFLICT1: NTSTATUS = -1_073_741_228_i32;
pub const STATUS_IP_ADDRESS_CONFLICT2: NTSTATUS = -1_073_741_227_i32;
pub const STATUS_ISSUING_CA_UNTRUSTED: NTSTATUS = -1_073_740_918_i32;
pub const STATUS_ISSUING_CA_UNTRUSTED_KDC: NTSTATUS = -1_073_740_787_i32;
pub const STATUS_JOB_NOT_EMPTY: NTSTATUS = -1_073_740_529_i32;
pub const STATUS_JOB_NO_CONTAINER: NTSTATUS = -1_073_740_535_i32;
pub const STATUS_JOURNAL_DELETE_IN_PROGRESS: NTSTATUS = -1_073_741_129_i32;
pub const STATUS_JOURNAL_ENTRY_DELETED: NTSTATUS = -1_073_741_105_i32;
pub const STATUS_JOURNAL_NOT_ACTIVE: NTSTATUS = -1_073_741_128_i32;
pub const STATUS_KDC_CERT_EXPIRED: NTSTATUS = -1_073_740_786_i32;
pub const STATUS_KDC_CERT_REVOKED: NTSTATUS = -1_073_740_785_i32;
pub const STATUS_KDC_INVALID_REQUEST: NTSTATUS = -1_073_741_061_i32;
pub const STATUS_KDC_UNABLE_TO_REFER: NTSTATUS = -1_073_741_060_i32;
pub const STATUS_KDC_UNKNOWN_ETYPE: NTSTATUS = -1_073_741_059_i32;
pub const STATUS_KERNEL_APC: NTSTATUS = 256_i32;
pub const STATUS_KERNEL_EXECUTABLE_MEMORY_WRITE: NTSTATUS = -1_073_739_996_i32;
pub const STATUS_KEY_DELETED: NTSTATUS = -1_073_741_444_i32;
pub const STATUS_KEY_HAS_CHILDREN: NTSTATUS = -1_073_741_440_i32;
pub const STATUS_LAST_ADMIN: NTSTATUS = -1_073_741_719_i32;
pub const STATUS_LICENSE_QUOTA_EXCEEDED: NTSTATUS = -1_073_741_223_i32;
pub const STATUS_LICENSE_VIOLATION: NTSTATUS = -1_073_741_206_i32;
pub const STATUS_LINK_FAILED: NTSTATUS = -1_073_741_506_i32;
pub const STATUS_LINK_TIMEOUT: NTSTATUS = -1_073_741_505_i32;
pub const STATUS_LM_CROSS_ENCRYPTION_REQUIRED: NTSTATUS = -1_073_741_441_i32;
pub const STATUS_LOCAL_DISCONNECT: NTSTATUS = -1_073_741_509_i32;
pub const STATUS_LOCAL_POLICY_MODIFICATION_NOT_SUPPORTED: NTSTATUS = -1_073_700_726_i32;
pub const STATUS_LOCAL_USER_SESSION_KEY: NTSTATUS = 1_073_741_830_i32;
pub const STATUS_LOCK_NOT_GRANTED: NTSTATUS = -1_073_741_739_i32;
pub const STATUS_LOGIN_TIME_RESTRICTION: NTSTATUS = -1_073_741_241_i32;
pub const STATUS_LOGIN_WKSTA_RESTRICTION: NTSTATUS = -1_073_741_240_i32;
pub const STATUS_LOGON_NOT_GRANTED: NTSTATUS = -1_073_741_483_i32;
pub const STATUS_LOGON_SERVER_CONFLICT: NTSTATUS = -1_073_741_518_i32;
pub const STATUS_LOGON_SESSION_COLLISION: NTSTATUS = -1_073_741_563_i32;
pub const STATUS_LOGON_SESSION_EXISTS: NTSTATUS = -1_073_741_586_i32;
pub const STATUS_LOG_APPENDED_FLUSH_FAILED: NTSTATUS = -1_072_037_841_i32;
pub const STATUS_LOG_ARCHIVE_IN_PROGRESS: NTSTATUS = -1_072_037_855_i32;
pub const STATUS_LOG_ARCHIVE_NOT_IN_PROGRESS: NTSTATUS = -1_072_037_856_i32;
pub const STATUS_LOG_BLOCKS_EXHAUSTED: NTSTATUS = -1_072_037_882_i32;
pub const STATUS_LOG_BLOCK_INCOMPLETE: NTSTATUS = -1_072_037_884_i32;
pub const STATUS_LOG_BLOCK_INVALID: NTSTATUS = -1_072_037_878_i32;
pub const STATUS_LOG_BLOCK_VERSION: NTSTATUS = -1_072_037_879_i32;
pub const STATUS_LOG_CANT_DELETE: NTSTATUS = -1_072_037_871_i32;
pub const STATUS_LOG_CLIENT_ALREADY_REGISTERED: NTSTATUS = -1_072_037_852_i32;
pub const STATUS_LOG_CLIENT_NOT_REGISTERED: NTSTATUS = -1_072_037_851_i32;
pub const STATUS_LOG_CONTAINER_LIMIT_EXCEEDED: NTSTATUS = -1_072_037_870_i32;
pub const STATUS_LOG_CONTAINER_OPEN_FAILED: NTSTATUS = -1_072_037_847_i32;
pub const STATUS_LOG_CONTAINER_READ_FAILED: NTSTATUS = -1_072_037_849_i32;
pub const STATUS_LOG_CONTAINER_STATE_INVALID: NTSTATUS = -1_072_037_846_i32;
pub const STATUS_LOG_CONTAINER_WRITE_FAILED: NTSTATUS = -1_072_037_848_i32;
pub const STATUS_LOG_CORRUPTION_DETECTED: NTSTATUS = -1_072_103_376_i32;
pub const STATUS_LOG_DEDICATED: NTSTATUS = -1_072_037_857_i32;
pub const STATUS_LOG_EPHEMERAL: NTSTATUS = -1_072_037_854_i32;
pub const STATUS_LOG_FILE_FULL: NTSTATUS = -1_073_741_432_i32;
pub const STATUS_LOG_FULL: NTSTATUS = -1_072_037_859_i32;
pub const STATUS_LOG_FULL_HANDLER_IN_PROGRESS: NTSTATUS = -1_072_037_850_i32;
pub const STATUS_LOG_GROWTH_FAILED: NTSTATUS = -1_072_103_399_i32;
pub const STATUS_LOG_HARD_ERROR: NTSTATUS = 1_073_741_850_i32;
pub const STATUS_LOG_INCONSISTENT_SECURITY: NTSTATUS = -1_072_037_842_i32;
pub const STATUS_LOG_INVALID_RANGE: NTSTATUS = -1_072_037_883_i32;
pub const STATUS_LOG_METADATA_CORRUPT: NTSTATUS = -1_072_037_875_i32;
pub const STATUS_LOG_METADATA_FLUSH_FAILED: NTSTATUS = -1_072_037_843_i32;
pub const STATUS_LOG_METADATA_INCONSISTENT: NTSTATUS = -1_072_037_873_i32;
pub const STATUS_LOG_METADATA_INVALID: NTSTATUS = -1_072_037_874_i32;
pub const STATUS_LOG_MULTIPLEXED: NTSTATUS = -1_072_037_858_i32;
pub const STATUS_LOG_NOT_ENOUGH_CONTAINERS: NTSTATUS = -1_072_037_853_i32;
pub const STATUS_LOG_NO_RESTART: NTSTATUS = 1_075_445_772_i32;
pub const STATUS_LOG_PINNED: NTSTATUS = -1_072_037_844_i32;
pub const STATUS_LOG_PINNED_ARCHIVE_TAIL: NTSTATUS = -1_072_037_864_i32;
pub const STATUS_LOG_PINNED_RESERVATION: NTSTATUS = -1_072_037_840_i32;
pub const STATUS_LOG_POLICY_ALREADY_INSTALLED: NTSTATUS = -1_072_037_868_i32;
pub const STATUS_LOG_POLICY_CONFLICT: NTSTATUS = -1_072_037_865_i32;
pub const STATUS_LOG_POLICY_INVALID: NTSTATUS = -1_072_037_866_i32;
pub const STATUS_LOG_POLICY_NOT_INSTALLED: NTSTATUS = -1_072_037_867_i32;
pub const STATUS_LOG_READ_CONTEXT_INVALID: NTSTATUS = -1_072_037_881_i32;
pub const STATUS_LOG_READ_MODE_INVALID: NTSTATUS = -1_072_037_877_i32;
pub const STATUS_LOG_RECORDS_RESERVED_INVALID: NTSTATUS = -1_072_037_862_i32;
pub const STATUS_LOG_RECORD_NONEXISTENT: NTSTATUS = -1_072_037_863_i32;
pub const STATUS_LOG_RESERVATION_INVALID: NTSTATUS = -1_072_037_872_i32;
pub const STATUS_LOG_RESIZE_INVALID_SIZE: NTSTATUS = -1_072_103_413_i32;
pub const STATUS_LOG_RESTART_INVALID: NTSTATUS = -1_072_037_880_i32;
pub const STATUS_LOG_SECTOR_INVALID: NTSTATUS = -1_072_037_887_i32;
pub const STATUS_LOG_SECTOR_PARITY_INVALID: NTSTATUS = -1_072_037_886_i32;
pub const STATUS_LOG_SECTOR_REMAPPED: NTSTATUS = -1_072_037_885_i32;
pub const STATUS_LOG_SPACE_RESERVED_INVALID: NTSTATUS = -1_072_037_861_i32;
pub const STATUS_LOG_START_OF_LOG: NTSTATUS = -1_072_037_869_i32;
pub const STATUS_LOG_STATE_INVALID: NTSTATUS = -1_072_037_845_i32;
pub const STATUS_LOG_TAIL_INVALID: NTSTATUS = -1_072_037_860_i32;
pub const STATUS_LONGJUMP: NTSTATUS = -2_147_483_610_i32;
pub const STATUS_LOST_MODE_LOGON_RESTRICTION: NTSTATUS = -1_073_741_043_i32;
pub const STATUS_LOST_WRITEBEHIND_DATA: NTSTATUS = -1_073_741_278_i32;
pub const STATUS_LOST_WRITEBEHIND_DATA_LOCAL_DISK_ERROR: NTSTATUS = -1_073_700_734_i32;
pub const STATUS_LOST_WRITEBEHIND_DATA_NETWORK_DISCONNECTED: NTSTATUS = -1_073_700_736_i32;
pub const STATUS_LOST_WRITEBEHIND_DATA_NETWORK_SERVER_ERROR: NTSTATUS = -1_073_700_735_i32;
pub const STATUS_LPAC_ACCESS_DENIED: NTSTATUS = -1_073_700_349_i32;
pub const STATUS_LPC_HANDLE_COUNT_EXCEEDED: NTSTATUS = -1_073_739_998_i32;
pub const STATUS_LPC_INVALID_CONNECTION_USAGE: NTSTATUS = -1_073_740_026_i32;
pub const STATUS_LPC_RECEIVE_BUFFER_EXPECTED: NTSTATUS = -1_073_740_027_i32;
pub const STATUS_LPC_REPLY_LOST: NTSTATUS = -1_073_741_229_i32;
pub const STATUS_LPC_REQUESTS_NOT_ALLOWED: NTSTATUS = -1_073_740_025_i32;
pub const STATUS_LUIDS_EXHAUSTED: NTSTATUS = -1_073_741_707_i32;
pub const STATUS_MAGAZINE_NOT_PRESENT: NTSTATUS = -1_073_741_178_i32;
pub const STATUS_MAPPED_ALIGNMENT: NTSTATUS = -1_073_741_280_i32;
pub const STATUS_MAPPED_FILE_SIZE_ZERO: NTSTATUS = -1_073_741_538_i32;
pub const STATUS_MARKED_TO_DISALLOW_WRITES: NTSTATUS = -1_073_740_659_i32;
pub const STATUS_MARSHALL_OVERFLOW: NTSTATUS = -1_073_741_263_i32;
pub const STATUS_MAX_REFERRALS_EXCEEDED: NTSTATUS = -1_073_741_068_i32;
pub const STATUS_MCA_EXCEPTION: NTSTATUS = -1_073_740_013_i32;
pub const STATUS_MCA_OCCURED: NTSTATUS = -1_073_740_950_i32;
pub const STATUS_MEDIA_CHANGED: NTSTATUS = -2_147_483_620_i32;
pub const STATUS_MEDIA_CHECK: NTSTATUS = -2_147_483_616_i32;
pub const STATUS_MEDIA_WRITE_PROTECTED: NTSTATUS = -1_073_741_662_i32;
pub const STATUS_MEMBERS_PRIMARY_GROUP: NTSTATUS = -1_073_741_529_i32;
pub const STATUS_MEMBER_IN_ALIAS: NTSTATUS = -1_073_741_485_i32;
pub const STATUS_MEMBER_IN_GROUP: NTSTATUS = -1_073_741_721_i32;
pub const STATUS_MEMBER_NOT_IN_ALIAS: NTSTATUS = -1_073_741_486_i32;
pub const STATUS_MEMBER_NOT_IN_GROUP: NTSTATUS = -1_073_741_720_i32;
pub const STATUS_MEMORY_NOT_ALLOCATED: NTSTATUS = -1_073_741_664_i32;
pub const STATUS_MESSAGE_LOST: NTSTATUS = -1_073_740_031_i32;
pub const STATUS_MESSAGE_NOT_FOUND: NTSTATUS = -1_073_741_559_i32;
pub const STATUS_MESSAGE_RETRIEVED: NTSTATUS = 1_073_741_870_i32;
pub const STATUS_MFT_TOO_FRAGMENTED: NTSTATUS = -1_073_741_052_i32;
pub const STATUS_MINIVERSION_INACCESSIBLE_FROM_SPECIFIED_TRANSACTION: NTSTATUS = -1_072_103_388_i32;
pub const STATUS_MISSING_SYSTEMFILE: NTSTATUS = -1_073_741_501_i32;
pub const STATUS_MONITOR_INVALID_DESCRIPTOR_CHECKSUM: NTSTATUS = -1_071_841_277_i32;
pub const STATUS_MONITOR_INVALID_DETAILED_TIMING_BLOCK: NTSTATUS = -1_071_841_271_i32;
pub const STATUS_MONITOR_INVALID_MANUFACTURE_DATE: NTSTATUS = -1_071_841_270_i32;
pub const STATUS_MONITOR_INVALID_SERIAL_NUMBER_MONDSC_BLOCK: NTSTATUS = -1_071_841_274_i32;
pub const STATUS_MONITOR_INVALID_STANDARD_TIMING_BLOCK: NTSTATUS = -1_071_841_276_i32;
pub const STATUS_MONITOR_INVALID_USER_FRIENDLY_MONDSC_BLOCK: NTSTATUS = -1_071_841_273_i32;
pub const STATUS_MONITOR_NO_DESCRIPTOR: NTSTATUS = -1_071_841_279_i32;
pub const STATUS_MONITOR_NO_MORE_DESCRIPTOR_DATA: NTSTATUS = -1_071_841_272_i32;
pub const STATUS_MONITOR_UNKNOWN_DESCRIPTOR_FORMAT: NTSTATUS = -1_071_841_278_i32;
pub const STATUS_MONITOR_WMI_DATABLOCK_REGISTRATION_FAILED: NTSTATUS = -1_071_841_275_i32;
pub const STATUS_MORE_ENTRIES: NTSTATUS = 261_i32;
pub const STATUS_MORE_PROCESSING_REQUIRED: NTSTATUS = -1_073_741_802_i32;
pub const STATUS_MOUNT_POINT_NOT_RESOLVED: NTSTATUS = -1_073_740_952_i32;
pub const STATUS_MP_PROCESSOR_MISMATCH: NTSTATUS = 1_073_741_865_i32;
pub const STATUS_MUI_FILE_NOT_FOUND: NTSTATUS = -1_073_020_927_i32;
pub const STATUS_MUI_FILE_NOT_LOADED: NTSTATUS = -1_073_020_922_i32;
pub const STATUS_MUI_INVALID_FILE: NTSTATUS = -1_073_020_926_i32;
pub const STATUS_MUI_INVALID_LOCALE_NAME: NTSTATUS = -1_073_020_924_i32;
pub const STATUS_MUI_INVALID_RC_CONFIG: NTSTATUS = -1_073_020_925_i32;
pub const STATUS_MUI_INVALID_ULTIMATEFALLBACK_NAME: NTSTATUS = -1_073_020_923_i32;
pub const STATUS_MULTIPLE_FAULT_VIOLATION: NTSTATUS = -1_073_741_080_i32;
pub const STATUS_MUST_BE_KDC: NTSTATUS = -1_073_741_067_i32;
pub const STATUS_MUTANT_LIMIT_EXCEEDED: NTSTATUS = -1_073_741_423_i32;
pub const STATUS_MUTANT_NOT_OWNED: NTSTATUS = -1_073_741_754_i32;
pub const STATUS_MUTUAL_AUTHENTICATION_FAILED: NTSTATUS = -1_073_741_117_i32;
pub const STATUS_NAME_TOO_LONG: NTSTATUS = -1_073_741_562_i32;
pub const STATUS_NDIS_ADAPTER_NOT_FOUND: NTSTATUS = -1_071_448_058_i32;
pub const STATUS_NDIS_ADAPTER_NOT_READY: NTSTATUS = -1_071_448_047_i32;
pub const STATUS_NDIS_ADAPTER_REMOVED: NTSTATUS = -1_071_448_040_i32;
pub const STATUS_NDIS_ALREADY_MAPPED: NTSTATUS = -1_071_448_035_i32;
pub const STATUS_NDIS_BAD_CHARACTERISTICS: NTSTATUS = -1_071_448_059_i32;
pub const STATUS_NDIS_BAD_VERSION: NTSTATUS = -1_071_448_060_i32;
pub const STATUS_NDIS_BUFFER_TOO_SHORT: NTSTATUS = -1_071_448_042_i32;
pub const STATUS_NDIS_CLOSING: NTSTATUS = -1_071_448_062_i32;
pub const STATUS_NDIS_DEVICE_FAILED: NTSTATUS = -1_071_448_056_i32;
pub const STATUS_NDIS_DOT11_AP_BAND_CURRENTLY_NOT_AVAILABLE: NTSTATUS = -1_071_439_866_i32;
pub const STATUS_NDIS_DOT11_AP_BAND_NOT_ALLOWED: NTSTATUS = -1_071_439_864_i32;
pub const STATUS_NDIS_DOT11_AP_CHANNEL_CURRENTLY_NOT_AVAILABLE: NTSTATUS = -1_071_439_867_i32;
pub const STATUS_NDIS_DOT11_AP_CHANNEL_NOT_ALLOWED: NTSTATUS = -1_071_439_865_i32;
pub const STATUS_NDIS_DOT11_AUTO_CONFIG_ENABLED: NTSTATUS = -1_071_439_872_i32;
pub const STATUS_NDIS_DOT11_MEDIA_IN_USE: NTSTATUS = -1_071_439_871_i32;
pub const STATUS_NDIS_DOT11_POWER_STATE_INVALID: NTSTATUS = -1_071_439_870_i32;
pub const STATUS_NDIS_ERROR_READING_FILE: NTSTATUS = -1_071_448_036_i32;
pub const STATUS_NDIS_FILE_NOT_FOUND: NTSTATUS = -1_071_448_037_i32;
pub const STATUS_NDIS_GROUP_ADDRESS_IN_USE: NTSTATUS = -1_071_448_038_i32;
pub const STATUS_NDIS_INDICATION_REQUIRED: NTSTATUS = 1_076_035_585_i32;
pub const STATUS_NDIS_INTERFACE_NOT_FOUND: NTSTATUS = -1_071_448_021_i32;
pub const STATUS_NDIS_INVALID_ADDRESS: NTSTATUS = -1_071_448_030_i32;
pub const STATUS_NDIS_INVALID_DATA: NTSTATUS = -1_071_448_043_i32;
pub const STATUS_NDIS_INVALID_DEVICE_REQUEST: NTSTATUS = -1_071_448_048_i32;
pub const STATUS_NDIS_INVALID_LENGTH: NTSTATUS = -1_071_448_044_i32;
pub const STATUS_NDIS_INVALID_OID: NTSTATUS = -1_071_448_041_i32;
pub const STATUS_NDIS_INVALID_PACKET: NTSTATUS = -1_071_448_049_i32;
pub const STATUS_NDIS_INVALID_PORT: NTSTATUS = -1_071_448_019_i32;
pub const STATUS_NDIS_INVALID_PORT_STATE: NTSTATUS = -1_071_448_018_i32;
pub const STATUS_NDIS_LOW_POWER_STATE: NTSTATUS = -1_071_448_017_i32;
pub const STATUS_NDIS_MEDIA_DISCONNECTED: NTSTATUS = -1_071_448_033_i32;
pub const STATUS_NDIS_MULTICAST_EXISTS: NTSTATUS = -1_071_448_054_i32;
pub const STATUS_NDIS_MULTICAST_FULL: NTSTATUS = -1_071_448_055_i32;
pub const STATUS_NDIS_MULTICAST_NOT_FOUND: NTSTATUS = -1_071_448_053_i32;
pub const STATUS_NDIS_NOT_SUPPORTED: NTSTATUS = -1_071_447_877_i32;
pub const STATUS_NDIS_NO_QUEUES: NTSTATUS = -1_071_448_015_i32;
pub const STATUS_NDIS_OFFLOAD_CONNECTION_REJECTED: NTSTATUS = -1_071_443_950_i32;
pub const STATUS_NDIS_OFFLOAD_PATH_REJECTED: NTSTATUS = -1_071_443_949_i32;
pub const STATUS_NDIS_OFFLOAD_POLICY: NTSTATUS = -1_071_443_953_i32;
pub const STATUS_NDIS_OPEN_FAILED: NTSTATUS = -1_071_448_057_i32;
pub const STATUS_NDIS_PAUSED: NTSTATUS = -1_071_448_022_i32;
pub const STATUS_NDIS_PM_PROTOCOL_OFFLOAD_LIST_FULL: NTSTATUS = -1_071_439_868_i32;
pub const STATUS_NDIS_PM_WOL_PATTERN_LIST_FULL: NTSTATUS = -1_071_439_869_i32;
pub const STATUS_NDIS_REINIT_REQUIRED: NTSTATUS = -1_071_448_016_i32;
pub const STATUS_NDIS_REQUEST_ABORTED: NTSTATUS = -1_071_448_052_i32;
pub const STATUS_NDIS_RESET_IN_PROGRESS: NTSTATUS = -1_071_448_051_i32;
pub const STATUS_NDIS_RESOURCE_CONFLICT: NTSTATUS = -1_071_448_034_i32;
pub const STATUS_NDIS_UNSUPPORTED_MEDIA: NTSTATUS = -1_071_448_039_i32;
pub const STATUS_NDIS_UNSUPPORTED_REVISION: NTSTATUS = -1_071_448_020_i32;
pub const STATUS_ND_QUEUE_OVERFLOW: NTSTATUS = -1_073_700_847_i32;
pub const STATUS_NEEDS_REGISTRATION: NTSTATUS = -1_073_740_663_i32;
pub const STATUS_NEEDS_REMEDIATION: NTSTATUS = -1_073_740_702_i32;
pub const STATUS_NETLOGON_NOT_STARTED: NTSTATUS = -1_073_741_422_i32;
pub const STATUS_NETWORK_ACCESS_DENIED: NTSTATUS = -1_073_741_622_i32;
pub const STATUS_NETWORK_ACCESS_DENIED_EDP: NTSTATUS = -1_073_740_658_i32;
pub const STATUS_NETWORK_AUTHENTICATION_PROMPT_CANCELED: NTSTATUS = -1_067_646_972_i32;
pub const STATUS_NETWORK_BUSY: NTSTATUS = -1_073_741_633_i32;
pub const STATUS_NETWORK_CREDENTIAL_CONFLICT: NTSTATUS = -1_073_741_419_i32;
pub const STATUS_NETWORK_NAME_DELETED: NTSTATUS = -1_073_741_623_i32;
pub const STATUS_NETWORK_OPEN_RESTRICTION: NTSTATUS = -1_073_741_311_i32;
pub const STATUS_NETWORK_SESSION_EXPIRED: NTSTATUS = -1_073_740_964_i32;
pub const STATUS_NETWORK_UNREACHABLE: NTSTATUS = -1_073_741_252_i32;
pub const STATUS_NET_WRITE_FAULT: NTSTATUS = -1_073_741_614_i32;
pub const STATUS_NOINTERFACE: NTSTATUS = -1_073_741_127_i32;
pub const STATUS_NOLOGON_INTERDOMAIN_TRUST_ACCOUNT: NTSTATUS = -1_073_741_416_i32;
pub const STATUS_NOLOGON_SERVER_TRUST_ACCOUNT: NTSTATUS = -1_073_741_414_i32;
pub const STATUS_NOLOGON_WORKSTATION_TRUST_ACCOUNT: NTSTATUS = -1_073_741_415_i32;
pub const STATUS_NONCONTINUABLE_EXCEPTION: NTSTATUS = -1_073_741_787_i32;
pub const STATUS_NONEXISTENT_EA_ENTRY: NTSTATUS = -1_073_741_743_i32;
pub const STATUS_NONEXISTENT_SECTOR: NTSTATUS = -1_073_741_803_i32;
pub const STATUS_NONE_MAPPED: NTSTATUS = -1_073_741_709_i32;
pub const STATUS_NOTHING_TO_TERMINATE: NTSTATUS = 290_i32;
pub const STATUS_NOTIFICATION_GUID_ALREADY_DEFINED: NTSTATUS = -1_073_741_404_i32;
pub const STATUS_NOTIFY_CLEANUP: NTSTATUS = 267_i32;
pub const STATUS_NOTIFY_ENUM_DIR: NTSTATUS = 268_i32;
pub const STATUS_NOT_ALLOWED_ON_SYSTEM_FILE: NTSTATUS = -1_073_741_401_i32;
pub const STATUS_NOT_ALL_ASSIGNED: NTSTATUS = 262_i32;
pub const STATUS_NOT_APPCONTAINER: NTSTATUS = -1_073_700_352_i32;
pub const STATUS_NOT_A_CLOUD_FILE: NTSTATUS = -1_073_688_825_i32;
pub const STATUS_NOT_A_CLOUD_SYNC_ROOT: NTSTATUS = -1_073_688_802_i32;
pub const STATUS_NOT_A_DAX_VOLUME: NTSTATUS = -1_073_740_623_i32;
pub const STATUS_NOT_A_DIRECTORY: NTSTATUS = -1_073_741_565_i32;
pub const STATUS_NOT_A_REPARSE_POINT: NTSTATUS = -1_073_741_195_i32;
pub const STATUS_NOT_A_TIERED_VOLUME: NTSTATUS = -1_073_740_531_i32;
pub const STATUS_NOT_CAPABLE: NTSTATUS = -1_073_740_759_i32;
pub const STATUS_NOT_CLIENT_SESSION: NTSTATUS = -1_073_741_289_i32;
pub const STATUS_NOT_COMMITTED: NTSTATUS = -1_073_741_779_i32;
pub const STATUS_NOT_DAX_MAPPABLE: NTSTATUS = -1_073_740_622_i32;
pub const STATUS_NOT_EXPORT_FORMAT: NTSTATUS = -1_073_741_166_i32;
pub const STATUS_NOT_FOUND: NTSTATUS = -1_073_741_275_i32;
pub const STATUS_NOT_GUI_PROCESS: NTSTATUS = -1_073_740_538_i32;
pub const STATUS_NOT_IMPLEMENTED: NTSTATUS = -1_073_741_822_i32;
pub const STATUS_NOT_LOCKED: NTSTATUS = -1_073_741_782_i32;
pub const STATUS_NOT_LOGON_PROCESS: NTSTATUS = -1_073_741_587_i32;
pub const STATUS_NOT_MAPPED_DATA: NTSTATUS = -1_073_741_688_i32;
pub const STATUS_NOT_MAPPED_VIEW: NTSTATUS = -1_073_741_799_i32;
pub const STATUS_NOT_READ_FROM_COPY: NTSTATUS = -1_073_740_694_i32;
pub const STATUS_NOT_REDUNDANT_STORAGE: NTSTATUS = -1_073_740_679_i32;
pub const STATUS_NOT_REGISTRY_FILE: NTSTATUS = -1_073_741_476_i32;
pub const STATUS_NOT_SAFE_MODE_DRIVER: NTSTATUS = -1_073_740_961_i32;
pub const STATUS_NOT_SAME_DEVICE: NTSTATUS = -1_073_741_612_i32;
pub const STATUS_NOT_SAME_OBJECT: NTSTATUS = -1_073_741_396_i32;
pub const STATUS_NOT_SERVER_SESSION: NTSTATUS = -1_073_741_290_i32;
pub const STATUS_NOT_SNAPSHOT_VOLUME: NTSTATUS = -1_072_103_353_i32;
pub const STATUS_NOT_SUPPORTED: NTSTATUS = -1_073_741_637_i32;
pub const STATUS_NOT_SUPPORTED_IN_APPCONTAINER: NTSTATUS = -1_073_700_351_i32;
pub const STATUS_NOT_SUPPORTED_ON_DAX: NTSTATUS = -1_073_740_646_i32;
pub const STATUS_NOT_SUPPORTED_ON_SBS: NTSTATUS = -1_073_741_056_i32;
pub const STATUS_NOT_SUPPORTED_WITH_AUDITING: NTSTATUS = -1_073_740_595_i32;
pub const STATUS_NOT_SUPPORTED_WITH_BTT: NTSTATUS = -1_073_740_619_i32;
pub const STATUS_NOT_SUPPORTED_WITH_BYPASSIO: NTSTATUS = -1_073_740_601_i32;
pub const STATUS_NOT_SUPPORTED_WITH_CACHED_HANDLE: NTSTATUS = -1_073_740_587_i32;
pub const STATUS_NOT_SUPPORTED_WITH_COMPRESSION: NTSTATUS = -1_073_740_598_i32;
pub const STATUS_NOT_SUPPORTED_WITH_DEDUPLICATION: NTSTATUS = -1_073_740_596_i32;
pub const STATUS_NOT_SUPPORTED_WITH_ENCRYPTION: NTSTATUS = -1_073_740_599_i32;
pub const STATUS_NOT_SUPPORTED_WITH_MONITORING: NTSTATUS = -1_073_740_594_i32;
pub const STATUS_NOT_SUPPORTED_WITH_REPLICATION: NTSTATUS = -1_073_740_597_i32;
pub const STATUS_NOT_SUPPORTED_WITH_SNAPSHOT: NTSTATUS = -1_073_740_593_i32;
pub const STATUS_NOT_SUPPORTED_WITH_VIRTUALIZATION: NTSTATUS = -1_073_740_592_i32;
pub const STATUS_NOT_TINY_STREAM: NTSTATUS = -1_073_741_274_i32;
pub const STATUS_NO_ACE_CONDITION: NTSTATUS = -2_147_483_601_i32;
pub const STATUS_NO_APPLICABLE_APP_LICENSES_FOUND: NTSTATUS = -1_058_406_399_i32;
pub const STATUS_NO_APPLICATION_PACKAGE: NTSTATUS = -1_073_741_398_i32;
pub const STATUS_NO_BROWSER_SERVERS_FOUND: NTSTATUS = -1_073_741_284_i32;
pub const STATUS_NO_BYPASSIO_DRIVER_SUPPORT: NTSTATUS = -1_073_740_600_i32;
pub const STATUS_NO_CALLBACK_ACTIVE: NTSTATUS = -1_073_741_224_i32;
pub const STATUS_NO_DATA_DETECTED: NTSTATUS = -2_147_483_614_i32;
pub const STATUS_NO_EAS_ON_FILE: NTSTATUS = -1_073_741_742_i32;
pub const STATUS_NO_EFS: NTSTATUS = -1_073_741_170_i32;
pub const STATUS_NO_EVENT_PAIR: NTSTATUS = -1_073_741_490_i32;
pub const STATUS_NO_GUID_TRANSLATION: NTSTATUS = -1_073_741_556_i32;
pub const STATUS_NO_IMPERSONATION_TOKEN: NTSTATUS = -1_073_741_732_i32;
pub const STATUS_NO_INHERITANCE: NTSTATUS = -2_147_483_637_i32;
pub const STATUS_NO_IP_ADDRESSES: NTSTATUS = -1_073_741_071_i32;
pub const STATUS_NO_KERB_KEY: NTSTATUS = -1_073_741_022_i32;
pub const STATUS_NO_KEY: NTSTATUS = -1_073_739_508_i32;
pub const STATUS_NO_LDT: NTSTATUS = -1_073_741_545_i32;
pub const STATUS_NO_LINK_TRACKING_IN_TRANSACTION: NTSTATUS = -1_072_103_335_i32;
pub const STATUS_NO_LOGON_SERVERS: NTSTATUS = -1_073_741_730_i32;
pub const STATUS_NO_LOG_SPACE: NTSTATUS = -1_073_741_443_i32;
pub const STATUS_NO_MATCH: NTSTATUS = -1_073_741_198_i32;
pub const STATUS_NO_MEDIA: NTSTATUS = -1_073_741_448_i32;
pub const STATUS_NO_MEDIA_IN_DEVICE: NTSTATUS = -1_073_741_805_i32;
pub const STATUS_NO_MEMORY: NTSTATUS = -1_073_741_801_i32;
pub const STATUS_NO_MORE_EAS: NTSTATUS = -2_147_483_630_i32;
pub const STATUS_NO_MORE_ENTRIES: NTSTATUS = -2_147_483_622_i32;
pub const STATUS_NO_MORE_FILES: NTSTATUS = -2_147_483_642_i32;
pub const STATUS_NO_MORE_MATCHES: NTSTATUS = -1_073_741_197_i32;
pub const STATUS_NO_PAGEFILE: NTSTATUS = -1_073_741_497_i32;
pub const STATUS_NO_PA_DATA: NTSTATUS = -1_073_741_064_i32;
pub const STATUS_NO_PHYSICALLY_ALIGNED_FREE_SPACE_FOUND: NTSTATUS = -1_073_740_635_i32;
pub const STATUS_NO_QUOTAS_FOR_ACCOUNT: NTSTATUS = 269_i32;
pub const STATUS_NO_RANGES_PROCESSED: NTSTATUS = -1_073_740_704_i32;
pub const STATUS_NO_RECOVERY_POLICY: NTSTATUS = -1_073_741_171_i32;
pub const STATUS_NO_S4U_PROT_SUPPORT: NTSTATUS = -1_073_740_790_i32;
pub const STATUS_NO_SAVEPOINT_WITH_OPEN_FILES: NTSTATUS = -1_072_103_352_i32;
pub const STATUS_NO_SECRETS: NTSTATUS = -1_073_740_943_i32;
pub const STATUS_NO_SECURITY_CONTEXT: NTSTATUS = -1_073_740_755_i32;
pub const STATUS_NO_SECURITY_ON_OBJECT: NTSTATUS = -1_073_741_609_i32;
pub const STATUS_NO_SPOOL_SPACE: NTSTATUS = -1_073_741_625_i32;
pub const STATUS_NO_SUCH_ALIAS: NTSTATUS = -1_073_741_487_i32;
pub const STATUS_NO_SUCH_DEVICE: NTSTATUS = -1_073_741_810_i32;
pub const STATUS_NO_SUCH_DOMAIN: NTSTATUS = -1_073_741_601_i32;
pub const STATUS_NO_SUCH_FILE: NTSTATUS = -1_073_741_809_i32;
pub const STATUS_NO_SUCH_GROUP: NTSTATUS = -1_073_741_722_i32;
pub const STATUS_NO_SUCH_MEMBER: NTSTATUS = -1_073_741_446_i32;
pub const STATUS_NO_SUCH_PACKAGE: NTSTATUS = -1_073_741_570_i32;
pub const STATUS_NO_SUCH_PRIVILEGE: NTSTATUS = -1_073_741_728_i32;
pub const STATUS_NO_TGT_REPLY: NTSTATUS = -1_073_741_073_i32;
pub const STATUS_NO_TOKEN: NTSTATUS = -1_073_741_700_i32;
pub const STATUS_NO_TRACKING_SERVICE: NTSTATUS = -1_073_741_153_i32;
pub const STATUS_NO_TRUST_LSA_SECRET: NTSTATUS = -1_073_741_430_i32;
pub const STATUS_NO_TRUST_SAM_ACCOUNT: NTSTATUS = -1_073_741_429_i32;
pub const STATUS_NO_TXF_METADATA: NTSTATUS = -2_145_845_207_i32;
pub const STATUS_NO_UNICODE_TRANSLATION: NTSTATUS = -1_073_740_009_i32;
pub const STATUS_NO_USER_KEYS: NTSTATUS = -1_073_741_168_i32;
pub const STATUS_NO_USER_SESSION_KEY: NTSTATUS = -1_073_741_310_i32;
pub const STATUS_NO_WORK_DONE: NTSTATUS = -2_147_483_598_i32;
pub const STATUS_NO_YIELD_PERFORMED: NTSTATUS = 1_073_741_860_i32;
pub const STATUS_NTLM_BLOCKED: NTSTATUS = -1_073_740_776_i32;
pub const STATUS_NT_CROSS_ENCRYPTION_REQUIRED: NTSTATUS = -1_073_741_475_i32;
pub const STATUS_NULL_LM_PASSWORD: NTSTATUS = 1_073_741_837_i32;
pub const STATUS_OBJECTID_EXISTS: NTSTATUS = -1_073_741_269_i32;
pub const STATUS_OBJECTID_NOT_FOUND: NTSTATUS = -1_073_741_072_i32;
pub const STATUS_OBJECT_IS_IMMUTABLE: NTSTATUS = -1_073_740_610_i32;
pub const STATUS_OBJECT_NAME_COLLISION: NTSTATUS = -1_073_741_771_i32;
pub const STATUS_OBJECT_NAME_EXISTS: NTSTATUS = 1_073_741_824_i32;
pub const STATUS_OBJECT_NAME_INVALID: NTSTATUS = -1_073_741_773_i32;
pub const STATUS_OBJECT_NAME_NOT_FOUND: NTSTATUS = -1_073_741_772_i32;
pub const STATUS_OBJECT_NOT_EXTERNALLY_BACKED: NTSTATUS = -1_073_740_691_i32;
pub const STATUS_OBJECT_NO_LONGER_EXISTS: NTSTATUS = -1_072_103_391_i32;
pub const STATUS_OBJECT_PATH_INVALID: NTSTATUS = -1_073_741_767_i32;
pub const STATUS_OBJECT_PATH_NOT_FOUND: NTSTATUS = -1_073_741_766_i32;
pub const STATUS_OBJECT_PATH_SYNTAX_BAD: NTSTATUS = -1_073_741_765_i32;
pub const STATUS_OBJECT_TYPE_MISMATCH: NTSTATUS = -1_073_741_788_i32;
pub const STATUS_OFFLOAD_READ_FILE_NOT_SUPPORTED: NTSTATUS = -1_073_700_189_i32;
pub const STATUS_OFFLOAD_READ_FLT_NOT_SUPPORTED: NTSTATUS = -1_073_700_191_i32;
pub const STATUS_OFFLOAD_WRITE_FILE_NOT_SUPPORTED: NTSTATUS = -1_073_700_188_i32;
pub const STATUS_OFFLOAD_WRITE_FLT_NOT_SUPPORTED: NTSTATUS = -1_073_700_190_i32;
pub const STATUS_ONLY_IF_CONNECTED: NTSTATUS = -1_073_741_108_i32;
pub const STATUS_OPEN_FAILED: NTSTATUS = -1_073_741_514_i32;
pub const STATUS_OPERATION_IN_PROGRESS: NTSTATUS = -1_073_740_682_i32;
pub const STATUS_OPERATION_NOT_SUPPORTED_IN_TRANSACTION: NTSTATUS = -1_072_103_334_i32;
pub const STATUS_OPLOCK_BREAK_IN_PROGRESS: NTSTATUS = 264_i32;
pub const STATUS_OPLOCK_HANDLE_CLOSED: NTSTATUS = 534_i32;
pub const STATUS_OPLOCK_NOT_GRANTED: NTSTATUS = -1_073_741_598_i32;
pub const STATUS_OPLOCK_SWITCHED_TO_NEW_HANDLE: NTSTATUS = 533_i32;
pub const STATUS_ORDINAL_NOT_FOUND: NTSTATUS = -1_073_741_512_i32;
pub const STATUS_ORPHAN_NAME_EXHAUSTED: NTSTATUS = -1_073_739_762_i32;
pub const STATUS_PACKAGE_NOT_AVAILABLE: NTSTATUS = -1_073_740_649_i32;
pub const STATUS_PACKAGE_UPDATING: NTSTATUS = -1_073_740_695_i32;
pub const STATUS_PAGEFILE_CREATE_FAILED: NTSTATUS = -1_073_741_498_i32;
pub const STATUS_PAGEFILE_NOT_SUPPORTED: NTSTATUS = -1_073_740_603_i32;
pub const STATUS_PAGEFILE_QUOTA: NTSTATUS = -1_073_741_817_i32;
pub const STATUS_PAGEFILE_QUOTA_EXCEEDED: NTSTATUS = -1_073_741_524_i32;
pub const STATUS_PAGE_FAULT_COPY_ON_WRITE: NTSTATUS = 274_i32;
pub const STATUS_PAGE_FAULT_DEMAND_ZERO: NTSTATUS = 273_i32;
pub const STATUS_PAGE_FAULT_GUARD_PAGE: NTSTATUS = 275_i32;
pub const STATUS_PAGE_FAULT_PAGING_FILE: NTSTATUS = 276_i32;
pub const STATUS_PAGE_FAULT_RETRY: NTSTATUS = 873_i32;
pub const STATUS_PAGE_FAULT_TRANSITION: NTSTATUS = 272_i32;
pub const STATUS_PARAMETER_QUOTA_EXCEEDED: NTSTATUS = -1_073_740_784_i32;
pub const STATUS_PARITY_ERROR: NTSTATUS = -1_073_741_781_i32;
pub const STATUS_PARTIAL_COPY: NTSTATUS = -2_147_483_635_i32;
pub const STATUS_PARTITION_FAILURE: NTSTATUS = -1_073_741_454_i32;
pub const STATUS_PARTITION_TERMINATING: NTSTATUS = -1_073_740_640_i32;
pub const STATUS_PASSWORD_CHANGE_REQUIRED: NTSTATUS = -1_073_741_044_i32;
pub const STATUS_PASSWORD_RESTRICTION: NTSTATUS = -1_073_741_716_i32;
pub const STATUS_PATCH_CONFLICT: NTSTATUS = -1_073_740_628_i32;
pub const STATUS_PATCH_DEFERRED: NTSTATUS = 1_073_741_879_i32;
pub const STATUS_PATCH_NOT_REGISTERED: NTSTATUS = -1_073_740_588_i32;
pub const STATUS_PATH_NOT_COVERED: NTSTATUS = -1_073_741_225_i32;
pub const STATUS_PCP_ATTESTATION_CHALLENGE_NOT_SET: NTSTATUS = -1_071_046_638_i32;
pub const STATUS_PCP_AUTHENTICATION_FAILED: NTSTATUS = -1_071_046_648_i32;
pub const STATUS_PCP_AUTHENTICATION_IGNORED: NTSTATUS = -1_071_046_647_i32;
pub const STATUS_PCP_BUFFER_LENGTH_MISMATCH: NTSTATUS = -1_071_046_626_i32;
pub const STATUS_PCP_BUFFER_TOO_SMALL: NTSTATUS = -1_071_046_650_i32;
pub const STATUS_PCP_CLAIM_TYPE_NOT_SUPPORTED: NTSTATUS = -1_071_046_628_i32;
pub const STATUS_PCP_DEVICE_NOT_FOUND: NTSTATUS = -1_071_046_643_i32;
pub const STATUS_PCP_DEVICE_NOT_READY: NTSTATUS = -1_071_046_655_i32;
pub const STATUS_PCP_ERROR_MASK: NTSTATUS = -1_071_046_656_i32;
pub const STATUS_PCP_FLAG_NOT_SUPPORTED: NTSTATUS = -1_071_046_652_i32;
pub const STATUS_PCP_IFX_RSA_KEY_CREATION_BLOCKED: NTSTATUS = -1_071_046_625_i32;
pub const STATUS_PCP_INTERNAL_ERROR: NTSTATUS = -1_071_046_649_i32;
pub const STATUS_PCP_INVALID_HANDLE: NTSTATUS = -1_071_046_654_i32;
pub const STATUS_PCP_INVALID_PARAMETER: NTSTATUS = -1_071_046_653_i32;
pub const STATUS_PCP_KEY_ALREADY_FINALIZED: NTSTATUS = -1_071_046_636_i32;
pub const STATUS_PCP_KEY_HANDLE_INVALIDATED: NTSTATUS = -1_071_046_622_i32;
pub const STATUS_PCP_KEY_NOT_AIK: NTSTATUS = -1_071_046_631_i32;
pub const STATUS_PCP_KEY_NOT_AUTHENTICATED: NTSTATUS = -1_071_046_632_i32;
pub const STATUS_PCP_KEY_NOT_FINALIZED: NTSTATUS = -1_071_046_639_i32;
pub const STATUS_PCP_KEY_NOT_LOADED: NTSTATUS = -1_071_046_641_i32;
pub const STATUS_PCP_KEY_NOT_SIGNING_KEY: NTSTATUS = -1_071_046_630_i32;
pub const STATUS_PCP_KEY_USAGE_POLICY_INVALID: NTSTATUS = -1_071_046_634_i32;
pub const STATUS_PCP_KEY_USAGE_POLICY_NOT_SUPPORTED: NTSTATUS = -1_071_046_635_i32;
pub const STATUS_PCP_LOCKED_OUT: NTSTATUS = -1_071_046_629_i32;
pub const STATUS_PCP_NOT_PCR_BOUND: NTSTATUS = -1_071_046_637_i32;
pub const STATUS_PCP_NOT_SUPPORTED: NTSTATUS = -1_071_046_651_i32;
pub const STATUS_PCP_NO_KEY_CERTIFICATION: NTSTATUS = -1_071_046_640_i32;
pub const STATUS_PCP_POLICY_NOT_FOUND: NTSTATUS = -1_071_046_646_i32;
pub const STATUS_PCP_PROFILE_NOT_FOUND: NTSTATUS = -1_071_046_645_i32;
pub const STATUS_PCP_RAW_POLICY_NOT_SUPPORTED: NTSTATUS = -1_071_046_623_i32;
pub const STATUS_PCP_SOFT_KEY_ERROR: NTSTATUS = -1_071_046_633_i32;
pub const STATUS_PCP_TICKET_MISSING: NTSTATUS = -1_071_046_624_i32;
pub const STATUS_PCP_TPM_VERSION_NOT_SUPPORTED: NTSTATUS = -1_071_046_627_i32;
pub const STATUS_PCP_UNSUPPORTED_PSS_SALT: NTSTATUS = 1_076_437_027_i32;
pub const STATUS_PCP_VALIDATION_FAILED: NTSTATUS = -1_071_046_644_i32;
pub const STATUS_PCP_WRONG_PARENT: NTSTATUS = -1_071_046_642_i32;
pub const STATUS_PENDING: NTSTATUS = 259_i32;
pub const STATUS_PER_USER_TRUST_QUOTA_EXCEEDED: NTSTATUS = -1_073_740_799_i32;
pub const STATUS_PIPE_BROKEN: NTSTATUS = -1_073_741_493_i32;
pub const STATUS_PIPE_BUSY: NTSTATUS = -1_073_741_650_i32;
pub const STATUS_PIPE_CLOSING: NTSTATUS = -1_073_741_647_i32;
pub const STATUS_PIPE_CONNECTED: NTSTATUS = -1_073_741_646_i32;
pub const STATUS_PIPE_DISCONNECTED: NTSTATUS = -1_073_741_648_i32;
pub const STATUS_PIPE_EMPTY: NTSTATUS = -1_073_741_607_i32;
pub const STATUS_PIPE_LISTENING: NTSTATUS = -1_073_741_645_i32;
pub const STATUS_PIPE_NOT_AVAILABLE: NTSTATUS = -1_073_741_652_i32;
pub const STATUS_PKINIT_CLIENT_FAILURE: NTSTATUS = -1_073_740_916_i32;
pub const STATUS_PKINIT_FAILURE: NTSTATUS = -1_073_741_024_i32;
pub const STATUS_PKINIT_NAME_MISMATCH: NTSTATUS = -1_073_741_063_i32;
pub const STATUS_PKU2U_CERT_FAILURE: NTSTATUS = -1_073_740_753_i32;
pub const STATUS_PLATFORM_MANIFEST_BINARY_ID_NOT_FOUND: NTSTATUS = -1_058_340_859_i32;
pub const STATUS_PLATFORM_MANIFEST_CATALOG_NOT_AUTHORIZED: NTSTATUS = -1_058_340_860_i32;
pub const STATUS_PLATFORM_MANIFEST_FILE_NOT_AUTHORIZED: NTSTATUS = -1_058_340_861_i32;
pub const STATUS_PLATFORM_MANIFEST_INVALID: NTSTATUS = -1_058_340_862_i32;
pub const STATUS_PLATFORM_MANIFEST_NOT_ACTIVE: NTSTATUS = -1_058_340_858_i32;
pub const STATUS_PLATFORM_MANIFEST_NOT_AUTHORIZED: NTSTATUS = -1_058_340_863_i32;
pub const STATUS_PLATFORM_MANIFEST_NOT_SIGNED: NTSTATUS = -1_058_340_857_i32;
pub const STATUS_PLUGPLAY_NO_DEVICE: NTSTATUS = -1_073_741_218_i32;
pub const STATUS_PLUGPLAY_QUERY_VETOED: NTSTATUS = -2_147_483_608_i32;
pub const STATUS_PNP_BAD_MPS_TABLE: NTSTATUS = -1_073_479_627_i32;
pub const STATUS_PNP_DEVICE_CONFIGURATION_PENDING: NTSTATUS = -1_073_740_651_i32;
pub const STATUS_PNP_DRIVER_CONFIGURATION_INCOMPLETE: NTSTATUS = -1_073_740_653_i32;
pub const STATUS_PNP_DRIVER_CONFIGURATION_NOT_FOUND: NTSTATUS = -1_073_740_654_i32;
pub const STATUS_PNP_DRIVER_PACKAGE_NOT_FOUND: NTSTATUS = -1_073_740_655_i32;
pub const STATUS_PNP_FUNCTION_DRIVER_REQUIRED: NTSTATUS = -1_073_740_652_i32;
pub const STATUS_PNP_INVALID_ID: NTSTATUS = -1_073_479_624_i32;
pub const STATUS_PNP_IRQ_TRANSLATION_FAILED: NTSTATUS = -1_073_479_625_i32;
pub const STATUS_PNP_NO_COMPAT_DRIVERS: NTSTATUS = -1_073_740_656_i32;
pub const STATUS_PNP_REBOOT_REQUIRED: NTSTATUS = -1_073_741_102_i32;
pub const STATUS_PNP_RESTART_ENUMERATION: NTSTATUS = -1_073_741_106_i32;
pub const STATUS_PNP_TRANSLATION_FAILED: NTSTATUS = -1_073_479_626_i32;
pub const STATUS_POLICY_OBJECT_NOT_FOUND: NTSTATUS = -1_073_741_158_i32;
pub const STATUS_POLICY_ONLY_IN_DS: NTSTATUS = -1_073_741_157_i32;
pub const STATUS_PORT_ALREADY_HAS_COMPLETION_LIST: NTSTATUS = -1_073_740_006_i32;
pub const STATUS_PORT_ALREADY_SET: NTSTATUS = -1_073_741_752_i32;
pub const STATUS_PORT_CLOSED: NTSTATUS = -1_073_740_032_i32;
pub const STATUS_PORT_CONNECTION_REFUSED: NTSTATUS = -1_073_741_759_i32;
pub const STATUS_PORT_DISCONNECTED: NTSTATUS = -1_073_741_769_i32;
pub const STATUS_PORT_DO_NOT_DISTURB: NTSTATUS = -1_073_741_770_i32;
pub const STATUS_PORT_MESSAGE_TOO_LONG: NTSTATUS = -1_073_741_777_i32;
pub const STATUS_PORT_NOT_SET: NTSTATUS = -1_073_740_973_i32;
pub const STATUS_PORT_UNREACHABLE: NTSTATUS = -1_073_741_249_i32;
pub const STATUS_POSSIBLE_DEADLOCK: NTSTATUS = -1_073_741_420_i32;
pub const STATUS_POWER_STATE_INVALID: NTSTATUS = -1_073_741_101_i32;
pub const STATUS_PREDEFINED_HANDLE: NTSTATUS = 1_073_741_846_i32;
pub const STATUS_PRENT4_MACHINE_ACCOUNT: NTSTATUS = -1_073_740_969_i32;
pub const STATUS_PRIMARY_TRANSPORT_CONNECT_FAILED: NTSTATUS = 270_i32;
pub const STATUS_PRINT_CANCELLED: NTSTATUS = -1_073_741_624_i32;
pub const STATUS_PRINT_QUEUE_FULL: NTSTATUS = -1_073_741_626_i32;
pub const STATUS_PRIVILEGED_INSTRUCTION: NTSTATUS = -1_073_741_674_i32;
pub const STATUS_PRIVILEGE_NOT_HELD: NTSTATUS = -1_073_741_727_i32;
pub const STATUS_PROACTIVE_SCAN_IN_PROGRESS: NTSTATUS = -1_073_739_761_i32;
pub const STATUS_PROCEDURE_NOT_FOUND: NTSTATUS = -1_073_741_702_i32;
pub const STATUS_PROCESS_CLONED: NTSTATUS = 297_i32;
pub const STATUS_PROCESS_IN_JOB: NTSTATUS = 292_i32;
pub const STATUS_PROCESS_IS_PROTECTED: NTSTATUS = -1_073_740_014_i32;
pub const STATUS_PROCESS_IS_TERMINATING: NTSTATUS = -1_073_741_558_i32;
pub const STATUS_PROCESS_NOT_IN_JOB: NTSTATUS = 291_i32;
pub const STATUS_PROFILING_AT_LIMIT: NTSTATUS = -1_073_741_613_i32;
pub const STATUS_PROFILING_NOT_STARTED: NTSTATUS = -1_073_741_641_i32;
pub const STATUS_PROFILING_NOT_STOPPED: NTSTATUS = -1_073_741_640_i32;
pub const STATUS_PROPSET_NOT_FOUND: NTSTATUS = -1_073_741_264_i32;
pub const STATUS_PROTOCOL_NOT_SUPPORTED: NTSTATUS = -1_073_700_845_i32;
pub const STATUS_PROTOCOL_UNREACHABLE: NTSTATUS = -1_073_741_250_i32;
pub const STATUS_PTE_CHANGED: NTSTATUS = -1_073_740_748_i32;
pub const STATUS_PURGE_FAILED: NTSTATUS = -1_073_740_747_i32;
pub const STATUS_PWD_HISTORY_CONFLICT: NTSTATUS = -1_073_741_220_i32;
pub const STATUS_PWD_TOO_LONG: NTSTATUS = -1_073_741_190_i32;
pub const STATUS_PWD_TOO_RECENT: NTSTATUS = -1_073_741_221_i32;
pub const STATUS_PWD_TOO_SHORT: NTSTATUS = -1_073_741_222_i32;
pub const STATUS_QUERY_STORAGE_ERROR: NTSTATUS = -2_143_682_559_i32;
pub const STATUS_QUIC_ALPN_NEG_FAILURE: NTSTATUS = -1_071_382_521_i32;
pub const STATUS_QUIC_CONNECTION_IDLE: NTSTATUS = -1_071_382_523_i32;
pub const STATUS_QUIC_CONNECTION_TIMEOUT: NTSTATUS = -1_071_382_522_i32;
pub const STATUS_QUIC_HANDSHAKE_FAILURE: NTSTATUS = -1_071_382_528_i32;
pub const STATUS_QUIC_INTERNAL_ERROR: NTSTATUS = -1_071_382_525_i32;
pub const STATUS_QUIC_PROTOCOL_VIOLATION: NTSTATUS = -1_071_382_524_i32;
pub const STATUS_QUIC_USER_CANCELED: NTSTATUS = -1_071_382_526_i32;
pub const STATUS_QUIC_VER_NEG_FAILURE: NTSTATUS = -1_071_382_527_i32;
pub const STATUS_QUOTA_ACTIVITY: NTSTATUS = -1_073_740_662_i32;
pub const STATUS_QUOTA_EXCEEDED: NTSTATUS = -1_073_741_756_i32;
pub const STATUS_QUOTA_LIST_INCONSISTENT: NTSTATUS = -1_073_741_210_i32;
pub const STATUS_QUOTA_NOT_ENABLED: NTSTATUS = -1_073_741_399_i32;
pub const STATUS_RANGE_LIST_CONFLICT: NTSTATUS = -1_073_741_182_i32;
pub const STATUS_RANGE_NOT_FOUND: NTSTATUS = -1_073_741_172_i32;
pub const STATUS_RANGE_NOT_LOCKED: NTSTATUS = -1_073_741_698_i32;
pub const STATUS_RDBSS_CONTINUE_OPERATION: NTSTATUS = -1_069_481_982_i32;
pub const STATUS_RDBSS_POST_OPERATION: NTSTATUS = -1_069_481_981_i32;
pub const STATUS_RDBSS_RESTART_OPERATION: NTSTATUS = -1_069_481_983_i32;
pub const STATUS_RDBSS_RETRY_LOOKUP: NTSTATUS = -1_069_481_980_i32;
pub const STATUS_RDP_PROTOCOL_ERROR: NTSTATUS = -1_073_086_414_i32;
pub const STATUS_RECEIVE_EXPEDITED: NTSTATUS = 1_073_741_840_i32;
pub const STATUS_RECEIVE_PARTIAL: NTSTATUS = 1_073_741_839_i32;
pub const STATUS_RECEIVE_PARTIAL_EXPEDITED: NTSTATUS = 1_073_741_841_i32;
pub const STATUS_RECOVERABLE_BUGCHECK: NTSTATUS = -2_147_483_596_i32;
pub const STATUS_RECOVERY_FAILURE: NTSTATUS = -1_073_741_273_i32;
pub const STATUS_RECOVERY_NOT_NEEDED: NTSTATUS = 1_075_380_276_i32;
pub const STATUS_RECURSIVE_DISPATCH: NTSTATUS = -1_073_740_028_i32;
pub const STATUS_REDIRECTOR_HAS_OPEN_HANDLES: NTSTATUS = -2_147_483_613_i32;
pub const STATUS_REDIRECTOR_NOT_STARTED: NTSTATUS = -1_073_741_573_i32;
pub const STATUS_REDIRECTOR_PAUSED: NTSTATUS = -1_073_741_615_i32;
pub const STATUS_REDIRECTOR_STARTED: NTSTATUS = -1_073_741_572_i32;
pub const STATUS_REGISTRY_CORRUPT: NTSTATUS = -1_073_741_492_i32;
pub const STATUS_REGISTRY_HIVE_RECOVERED: NTSTATUS = -2_147_483_606_i32;
pub const STATUS_REGISTRY_IO_FAILED: NTSTATUS = -1_073_741_491_i32;
pub const STATUS_REGISTRY_QUOTA_LIMIT: NTSTATUS = -1_073_741_226_i32;
pub const STATUS_REGISTRY_RECOVERED: NTSTATUS = 1_073_741_833_i32;
pub const STATUS_REG_NAT_CONSUMPTION: NTSTATUS = -1_073_741_111_i32;
pub const STATUS_REINITIALIZATION_NEEDED: NTSTATUS = -1_073_741_177_i32;
pub const STATUS_REMOTE_DISCONNECT: NTSTATUS = -1_073_741_508_i32;
pub const STATUS_REMOTE_FILE_VERSION_MISMATCH: NTSTATUS = -1_072_103_412_i32;
pub const STATUS_REMOTE_NOT_LISTENING: NTSTATUS = -1_073_741_636_i32;
pub const STATUS_REMOTE_RESOURCES: NTSTATUS = -1_073_741_507_i32;
pub const STATUS_REMOTE_SESSION_LIMIT: NTSTATUS = -1_073_741_418_i32;
pub const STATUS_REMOTE_STORAGE_MEDIA_ERROR: NTSTATUS = -1_073_741_154_i32;
pub const STATUS_REMOTE_STORAGE_NOT_ACTIVE: NTSTATUS = -1_073_741_155_i32;
pub const STATUS_REPAIR_NEEDED: NTSTATUS = -1_073_741_400_i32;
pub const STATUS_REPARSE: NTSTATUS = 260_i32;
pub const STATUS_REPARSE_ATTRIBUTE_CONFLICT: NTSTATUS = -1_073_741_134_i32;
pub const STATUS_REPARSE_GLOBAL: NTSTATUS = 872_i32;
pub const STATUS_REPARSE_OBJECT: NTSTATUS = 280_i32;
pub const STATUS_REPARSE_POINT_ENCOUNTERED: NTSTATUS = -1_073_740_533_i32;
pub const STATUS_REPARSE_POINT_NOT_RESOLVED: NTSTATUS = -1_073_741_184_i32;
pub const STATUS_REPLY_MESSAGE_MISMATCH: NTSTATUS = -1_073_741_281_i32;
pub const STATUS_REQUEST_ABORTED: NTSTATUS = -1_073_741_248_i32;
pub const STATUS_REQUEST_CANCELED: NTSTATUS = -1_073_740_029_i32;
pub const STATUS_REQUEST_NOT_ACCEPTED: NTSTATUS = -1_073_741_616_i32;
pub const STATUS_REQUEST_OUT_OF_SEQUENCE: NTSTATUS = -1_073_740_758_i32;
pub const STATUS_REQUEST_PAUSED: NTSTATUS = -1_073_740_711_i32;
pub const STATUS_RESIDENT_FILE_NOT_SUPPORTED: NTSTATUS = -1_073_740_678_i32;
pub const STATUS_RESOURCEMANAGER_NOT_FOUND: NTSTATUS = -1_072_103_345_i32;
pub const STATUS_RESOURCEMANAGER_READ_ONLY: NTSTATUS = 514_i32;
pub const STATUS_RESOURCE_DATA_NOT_FOUND: NTSTATUS = -1_073_741_687_i32;
pub const STATUS_RESOURCE_ENUM_USER_STOP: NTSTATUS = -1_073_020_921_i32;
pub const STATUS_RESOURCE_IN_USE: NTSTATUS = -1_073_740_024_i32;
pub const STATUS_RESOURCE_LANG_NOT_FOUND: NTSTATUS = -1_073_741_308_i32;
pub const STATUS_RESOURCE_NAME_NOT_FOUND: NTSTATUS = -1_073_741_685_i32;
pub const STATUS_RESOURCE_NOT_OWNED: NTSTATUS = -1_073_741_212_i32;
pub const STATUS_RESOURCE_REQUIREMENTS_CHANGED: NTSTATUS = 281_i32;
pub const STATUS_RESOURCE_TYPE_NOT_FOUND: NTSTATUS = -1_073_741_686_i32;
pub const STATUS_RESTART_BOOT_APPLICATION: NTSTATUS = -1_073_740_717_i32;
pub const STATUS_RESUME_HIBERNATION: NTSTATUS = 1_073_741_867_i32;
pub const STATUS_RETRY: NTSTATUS = -1_073_741_267_i32;
pub const STATUS_RETURN_ADDRESS_HIJACK_ATTEMPT: NTSTATUS = -2_147_483_597_i32;
pub const STATUS_REVISION_MISMATCH: NTSTATUS = -1_073_741_735_i32;
pub const STATUS_REVOCATION_OFFLINE_C: NTSTATUS = -1_073_740_917_i32;
pub const STATUS_REVOCATION_OFFLINE_KDC: NTSTATUS = -1_073_740_788_i32;
pub const STATUS_RING_NEWLY_EMPTY: NTSTATUS = 531_i32;
pub const STATUS_RING_PREVIOUSLY_ABOVE_QUOTA: NTSTATUS = 530_i32;
pub const STATUS_RING_PREVIOUSLY_EMPTY: NTSTATUS = 528_i32;
pub const STATUS_RING_PREVIOUSLY_FULL: NTSTATUS = 529_i32;
pub const STATUS_RING_SIGNAL_OPPOSITE_ENDPOINT: NTSTATUS = 532_i32;
pub const STATUS_RKF_ACTIVE_KEY: NTSTATUS = -1_069_547_514_i32;
pub const STATUS_RKF_BLOB_FULL: NTSTATUS = -1_069_547_517_i32;
pub const STATUS_RKF_DUPLICATE_KEY: NTSTATUS = -1_069_547_518_i32;
pub const STATUS_RKF_FILE_BLOCKED: NTSTATUS = -1_069_547_515_i32;
pub const STATUS_RKF_KEY_NOT_FOUND: NTSTATUS = -1_069_547_519_i32;
pub const STATUS_RKF_STORE_FULL: NTSTATUS = -1_069_547_516_i32;
pub const STATUS_RM_ALREADY_STARTED: NTSTATUS = 1_075_380_277_i32;
pub const STATUS_RM_CANNOT_BE_FROZEN_FOR_SNAPSHOT: NTSTATUS = -1_072_103_331_i32;
pub const STATUS_RM_DISCONNECTED: NTSTATUS = -1_072_103_374_i32;
pub const STATUS_RM_METADATA_CORRUPT: NTSTATUS = -1_072_103_418_i32;
pub const STATUS_RM_NOT_ACTIVE: NTSTATUS = -1_072_103_419_i32;
pub const STATUS_ROLLBACK_TIMER_EXPIRED: NTSTATUS = -1_072_103_364_i32;
pub const STATUS_RTPM_CONTEXT_COMPLETE: NTSTATUS = 2_699_265_i32;
pub const STATUS_RTPM_CONTEXT_CONTINUE: NTSTATUS = 2_699_264_i32;
pub const STATUS_RTPM_INVALID_CONTEXT: NTSTATUS = -1_071_042_556_i32;
pub const STATUS_RTPM_NO_RESULT: NTSTATUS = -1_071_042_558_i32;
pub const STATUS_RTPM_PCR_READ_INCOMPLETE: NTSTATUS = -1_071_042_557_i32;
pub const STATUS_RTPM_UNSUPPORTED_CMD: NTSTATUS = -1_071_042_555_i32;
pub const STATUS_RUNLEVEL_SWITCH_AGENT_TIMEOUT: NTSTATUS = -1_073_700_539_i32;
pub const STATUS_RUNLEVEL_SWITCH_IN_PROGRESS: NTSTATUS = -1_073_700_538_i32;
pub const STATUS_RUNLEVEL_SWITCH_TIMEOUT: NTSTATUS = -1_073_700_541_i32;
pub const STATUS_RWRAW_ENCRYPTED_FILE_NOT_ENCRYPTED: NTSTATUS = -1_073_740_633_i32;
pub const STATUS_RWRAW_ENCRYPTED_INVALID_EDATAINFO_FILEOFFSET: NTSTATUS = -1_073_740_632_i32;
pub const STATUS_RWRAW_ENCRYPTED_INVALID_EDATAINFO_FILERANGE: NTSTATUS = -1_073_740_631_i32;
pub const STATUS_RWRAW_ENCRYPTED_INVALID_EDATAINFO_PARAMETER: NTSTATUS = -1_073_740_630_i32;
pub const STATUS_RXACT_COMMITTED: NTSTATUS = 266_i32;
pub const STATUS_RXACT_COMMIT_FAILURE: NTSTATUS = -1_073_741_539_i32;
pub const STATUS_RXACT_COMMIT_NECESSARY: NTSTATUS = -2_147_483_624_i32;
pub const STATUS_RXACT_INVALID_STATE: NTSTATUS = -1_073_741_540_i32;
pub const STATUS_RXACT_STATE_CREATED: NTSTATUS = 1_073_741_828_i32;
pub const STATUS_SAM_INIT_FAILURE: NTSTATUS = -1_073_741_085_i32;
pub const STATUS_SAM_NEED_BOOTKEY_FLOPPY: NTSTATUS = -1_073_741_088_i32;
pub const STATUS_SAM_NEED_BOOTKEY_PASSWORD: NTSTATUS = -1_073_741_089_i32;
pub const STATUS_SCRUB_DATA_DISABLED: NTSTATUS = -1_073_740_680_i32;
pub const STATUS_SECCORE_INVALID_COMMAND: NTSTATUS = -1_058_537_472_i32;
pub const STATUS_SECONDARY_IC_PROVIDER_NOT_REGISTERED: NTSTATUS = -1_073_700_575_i32;
pub const STATUS_SECRET_TOO_LONG: NTSTATUS = -1_073_741_481_i32;
pub const STATUS_SECTION_DIRECT_MAP_ONLY: NTSTATUS = -1_073_739_503_i32;
pub const STATUS_SECTION_NOT_EXTENDED: NTSTATUS = -1_073_741_689_i32;
pub const STATUS_SECTION_NOT_IMAGE: NTSTATUS = -1_073_741_751_i32;
pub const STATUS_SECTION_PROTECTION: NTSTATUS = -1_073_741_746_i32;
pub const STATUS_SECTION_TOO_BIG: NTSTATUS = -1_073_741_760_i32;
pub const STATUS_SECUREBOOT_FILE_REPLACED: NTSTATUS = -1_069_350_905_i32;
pub const STATUS_SECUREBOOT_INVALID_POLICY: NTSTATUS = -1_069_350_909_i32;
pub const STATUS_SECUREBOOT_NOT_BASE_POLICY: NTSTATUS = -1_069_350_897_i32;
pub const STATUS_SECUREBOOT_NOT_ENABLED: NTSTATUS = -2_143_092_730_i32;
pub const STATUS_SECUREBOOT_NOT_SUPPLEMENTAL_POLICY: NTSTATUS = -1_069_350_896_i32;
pub const STATUS_SECUREBOOT_PLATFORM_ID_MISMATCH: NTSTATUS = -1_069_350_901_i32;
pub const STATUS_SECUREBOOT_POLICY_MISSING_ANTIROLLBACKVERSION: NTSTATUS = -1_069_350_902_i32;
pub const STATUS_SECUREBOOT_POLICY_NOT_AUTHORIZED: NTSTATUS = -1_069_350_904_i32;
pub const STATUS_SECUREBOOT_POLICY_NOT_SIGNED: NTSTATUS = -1_069_350_907_i32;
pub const STATUS_SECUREBOOT_POLICY_PUBLISHER_NOT_FOUND: NTSTATUS = -1_069_350_908_i32;
pub const STATUS_SECUREBOOT_POLICY_ROLLBACK_DETECTED: NTSTATUS = -1_069_350_900_i32;
pub const STATUS_SECUREBOOT_POLICY_UNKNOWN: NTSTATUS = -1_069_350_903_i32;
pub const STATUS_SECUREBOOT_POLICY_UPGRADE_MISMATCH: NTSTATUS = -1_069_350_899_i32;
pub const STATUS_SECUREBOOT_POLICY_VIOLATION: NTSTATUS = -1_069_350_910_i32;
pub const STATUS_SECUREBOOT_REQUIRED_POLICY_FILE_MISSING: NTSTATUS = -1_069_350_898_i32;
pub const STATUS_SECUREBOOT_ROLLBACK_DETECTED: NTSTATUS = -1_069_350_911_i32;
pub const STATUS_SECURITY_STREAM_IS_INCONSISTENT: NTSTATUS = -1_073_741_408_i32;
pub const STATUS_SEGMENT_NOTIFICATION: NTSTATUS = 1_073_741_829_i32;
pub const STATUS_SEMAPHORE_LIMIT_EXCEEDED: NTSTATUS = -1_073_741_753_i32;
pub const STATUS_SERIAL_COUNTER_TIMEOUT: NTSTATUS = 1_073_741_836_i32;
pub const STATUS_SERIAL_MORE_WRITES: NTSTATUS = 1_073_741_832_i32;
pub const STATUS_SERIAL_NO_DEVICE_INITED: NTSTATUS = -1_073_741_488_i32;
pub const STATUS_SERVER_DISABLED: NTSTATUS = -1_073_741_696_i32;
pub const STATUS_SERVER_HAS_OPEN_HANDLES: NTSTATUS = -2_147_483_612_i32;
pub const STATUS_SERVER_NOT_DISABLED: NTSTATUS = -1_073_741_695_i32;
pub const STATUS_SERVER_SHUTDOWN_IN_PROGRESS: NTSTATUS = -1_073_741_057_i32;
pub const STATUS_SERVER_SID_MISMATCH: NTSTATUS = -1_073_741_152_i32;
pub const STATUS_SERVER_TRANSPORT_CONFLICT: NTSTATUS = -1_073_741_388_i32;
pub const STATUS_SERVER_UNAVAILABLE: NTSTATUS = -1_073_740_698_i32;
pub const STATUS_SERVICES_FAILED_AUTOSTART: NTSTATUS = 1_073_783_108_i32;
pub const STATUS_SERVICE_NOTIFICATION: NTSTATUS = 1_073_741_848_i32;
pub const STATUS_SESSION_KEY_TOO_SHORT: NTSTATUS = -1_073_740_521_i32;
pub const STATUS_SETMARK_DETECTED: NTSTATUS = -2_147_483_615_i32;
pub const STATUS_SET_CONTEXT_DENIED: NTSTATUS = -1_073_740_278_i32;
pub const STATUS_SHARED_IRQ_BUSY: NTSTATUS = -1_073_741_460_i32;
pub const STATUS_SHARED_POLICY: NTSTATUS = -1_073_741_159_i32;
pub const STATUS_SHARE_UNAVAILABLE: NTSTATUS = -1_073_740_672_i32;
pub const STATUS_SHARING_PAUSED: NTSTATUS = -1_073_741_617_i32;
pub const STATUS_SHARING_VIOLATION: NTSTATUS = -1_073_741_757_i32;
pub const STATUS_SHORT_NAMES_NOT_ENABLED_ON_VOLUME: NTSTATUS = -1_073_741_409_i32;
pub const STATUS_SHUTDOWN_IN_PROGRESS: NTSTATUS = -1_073_741_058_i32;
pub const STATUS_SINGLE_STEP: NTSTATUS = -2_147_483_644_i32;
pub const STATUS_SMARTCARD_CARD_BLOCKED: NTSTATUS = -1_073_740_927_i32;
pub const STATUS_SMARTCARD_CARD_NOT_AUTHENTICATED: NTSTATUS = -1_073_740_926_i32;
pub const STATUS_SMARTCARD_CERT_EXPIRED: NTSTATUS = -1_073_740_915_i32;
pub const STATUS_SMARTCARD_CERT_REVOKED: NTSTATUS = -1_073_740_919_i32;
pub const STATUS_SMARTCARD_IO_ERROR: NTSTATUS = -1_073_740_921_i32;
pub const STATUS_SMARTCARD_LOGON_REQUIRED: NTSTATUS = -1_073_741_062_i32;
pub const STATUS_SMARTCARD_NO_CARD: NTSTATUS = -1_073_740_925_i32;
pub const STATUS_SMARTCARD_NO_CERTIFICATE: NTSTATUS = -1_073_740_923_i32;
pub const STATUS_SMARTCARD_NO_KEYSET: NTSTATUS = -1_073_740_922_i32;
pub const STATUS_SMARTCARD_NO_KEY_CONTAINER: NTSTATUS = -1_073_740_924_i32;
pub const STATUS_SMARTCARD_SILENT_CONTEXT: NTSTATUS = -1_073_740_913_i32;
pub const STATUS_SMARTCARD_SUBSYSTEM_FAILURE: NTSTATUS = -1_073_741_023_i32;
pub const STATUS_SMARTCARD_WRONG_PIN: NTSTATUS = -1_073_740_928_i32;
pub const STATUS_SMB1_NOT_AVAILABLE: NTSTATUS = -1_073_740_525_i32;
pub const STATUS_SMB_BAD_CLUSTER_DIALECT: NTSTATUS = -1_067_646_975_i32;
pub const STATUS_SMB_GUEST_LOGON_BLOCKED: NTSTATUS = -1_067_646_974_i32;
pub const STATUS_SMB_NO_PREAUTH_INTEGRITY_HASH_OVERLAP: NTSTATUS = -1_067_646_976_i32;
pub const STATUS_SMB_NO_SIGNING_ALGORITHM_OVERLAP: NTSTATUS = -1_067_646_973_i32;
pub const STATUS_SMI_PRIMITIVE_INSTALLER_FAILED: NTSTATUS = -1_072_365_531_i32;
pub const STATUS_SMR_GARBAGE_COLLECTION_REQUIRED: NTSTATUS = -1_073_740_524_i32;
pub const STATUS_SOME_NOT_MAPPED: NTSTATUS = 263_i32;
pub const STATUS_SOURCE_ELEMENT_EMPTY: NTSTATUS = -1_073_741_181_i32;
pub const STATUS_SPACES_ALLOCATION_SIZE_INVALID: NTSTATUS = -1_058_602_994_i32;
pub const STATUS_SPACES_CACHE_FULL: NTSTATUS = -1_058_602_970_i32;
pub const STATUS_SPACES_COMPLETE: NTSTATUS = 15_138_818_i32;
pub const STATUS_SPACES_CORRUPT_METADATA: NTSTATUS = -1_058_602_986_i32;
pub const STATUS_SPACES_DRIVE_LOST_DATA: NTSTATUS = -1_058_602_979_i32;
pub const STATUS_SPACES_DRIVE_NOT_READY: NTSTATUS = -1_058_602_981_i32;
pub const STATUS_SPACES_DRIVE_OPERATIONAL_STATE_INVALID: NTSTATUS = -1_058_602_990_i32;
pub const STATUS_SPACES_DRIVE_REDUNDANCY_INVALID: NTSTATUS = -1_058_603_002_i32;
pub const STATUS_SPACES_DRIVE_SECTOR_SIZE_INVALID: NTSTATUS = -1_058_603_004_i32;
pub const STATUS_SPACES_DRIVE_SPLIT: NTSTATUS = -1_058_602_980_i32;
pub const STATUS_SPACES_DRT_FULL: NTSTATUS = -1_058_602_985_i32;
pub const STATUS_SPACES_ENCLOSURE_AWARE_INVALID: NTSTATUS = -1_058_602_993_i32;
pub const STATUS_SPACES_ENTRY_INCOMPLETE: NTSTATUS = -1_058_602_978_i32;
pub const STATUS_SPACES_ENTRY_INVALID: NTSTATUS = -1_058_602_977_i32;
pub const STATUS_SPACES_EXTENDED_ERROR: NTSTATUS = -1_058_602_996_i32;
pub const STATUS_SPACES_FAULT_DOMAIN_TYPE_INVALID: NTSTATUS = -1_058_603_007_i32;
pub const STATUS_SPACES_FLUSH_METADATA: NTSTATUS = -1_058_602_971_i32;
pub const STATUS_SPACES_INCONSISTENCY: NTSTATUS = -1_058_602_984_i32;
pub const STATUS_SPACES_INTERLEAVE_LENGTH_INVALID: NTSTATUS = -1_058_602_999_i32;
pub const STATUS_SPACES_LOG_NOT_READY: NTSTATUS = -1_058_602_983_i32;
pub const STATUS_SPACES_MAP_REQUIRED: NTSTATUS = -1_058_602_988_i32;
pub const STATUS_SPACES_MARK_DIRTY: NTSTATUS = -1_058_602_976_i32;
pub const STATUS_SPACES_NOT_ENOUGH_DRIVES: NTSTATUS = -1_058_602_997_i32;
pub const STATUS_SPACES_NO_REDUNDANCY: NTSTATUS = -1_058_602_982_i32;
pub const STATUS_SPACES_NUMBER_OF_COLUMNS_INVALID: NTSTATUS = -1_058_602_998_i32;
pub const STATUS_SPACES_NUMBER_OF_DATA_COPIES_INVALID: NTSTATUS = -1_058_603_001_i32;
pub const STATUS_SPACES_NUMBER_OF_GROUPS_INVALID: NTSTATUS = -1_058_602_991_i32;
pub const STATUS_SPACES_PAUSE: NTSTATUS = 15_138_817_i32;
pub const STATUS_SPACES_PD_INVALID_DATA: NTSTATUS = -1_058_602_972_i32;
pub const STATUS_SPACES_PD_LENGTH_MISMATCH: NTSTATUS = -1_058_602_974_i32;
pub const STATUS_SPACES_PD_NOT_FOUND: NTSTATUS = -1_058_602_975_i32;
pub const STATUS_SPACES_PD_UNSUPPORTED_VERSION: NTSTATUS = -1_058_602_973_i32;
pub const STATUS_SPACES_PROVISIONING_TYPE_INVALID: NTSTATUS = -1_058_602_995_i32;
pub const STATUS_SPACES_REDIRECT: NTSTATUS = 15_138_819_i32;
pub const STATUS_SPACES_REPAIRED: NTSTATUS = 15_138_816_i32;
pub const STATUS_SPACES_REPAIR_IN_PROGRESS: NTSTATUS = -1_058_602_969_i32;
pub const STATUS_SPACES_RESILIENCY_TYPE_INVALID: NTSTATUS = -1_058_603_005_i32;
pub const STATUS_SPACES_UNSUPPORTED_VERSION: NTSTATUS = -1_058_602_987_i32;
pub const STATUS_SPACES_UPDATE_COLUMN_STATE: NTSTATUS = -1_058_602_989_i32;
pub const STATUS_SPACES_WRITE_CACHE_SIZE_INVALID: NTSTATUS = -1_058_602_992_i32;
pub const STATUS_SPARSE_FILE_NOT_SUPPORTED: NTSTATUS = -1_073_740_604_i32;
pub const STATUS_SPARSE_NOT_ALLOWED_IN_TRANSACTION: NTSTATUS = -1_072_103_351_i32;
pub const STATUS_SPECIAL_ACCOUNT: NTSTATUS = -1_073_741_532_i32;
pub const STATUS_SPECIAL_GROUP: NTSTATUS = -1_073_741_531_i32;
pub const STATUS_SPECIAL_USER: NTSTATUS = -1_073_741_530_i32;
pub const STATUS_STACK_BUFFER_OVERRUN: NTSTATUS = -1_073_740_791_i32;
pub const STATUS_STACK_OVERFLOW: NTSTATUS = -1_073_741_571_i32;
pub const STATUS_STACK_OVERFLOW_READ: NTSTATUS = -1_073_741_272_i32;
pub const STATUS_STOPPED_ON_SYMLINK: NTSTATUS = -2_147_483_603_i32;
pub const STATUS_STORAGE_LOST_DATA_PERSISTENCE: NTSTATUS = -1_073_740_642_i32;
pub const STATUS_STORAGE_RESERVE_ALREADY_EXISTS: NTSTATUS = -1_073_740_625_i32;
pub const STATUS_STORAGE_RESERVE_DOES_NOT_EXIST: NTSTATUS = -1_073_740_626_i32;
pub const STATUS_STORAGE_RESERVE_ID_INVALID: NTSTATUS = -1_073_740_627_i32;
pub const STATUS_STORAGE_RESERVE_NOT_EMPTY: NTSTATUS = -1_073_740_624_i32;
pub const STATUS_STORAGE_STACK_ACCESS_DENIED: NTSTATUS = -1_073_740_607_i32;
pub const STATUS_STORAGE_TOPOLOGY_ID_MISMATCH: NTSTATUS = -1_073_740_666_i32;
pub const STATUS_STOWED_EXCEPTION: NTSTATUS = -1_073_741_189_i32;
pub const STATUS_STREAM_MINIVERSION_NOT_FOUND: NTSTATUS = -1_072_103_390_i32;
pub const STATUS_STREAM_MINIVERSION_NOT_VALID: NTSTATUS = -1_072_103_389_i32;
pub const STATUS_STRICT_CFG_VIOLATION: NTSTATUS = -1_073_740_282_i32;
pub const STATUS_STRONG_CRYPTO_NOT_SUPPORTED: NTSTATUS = -1_073_741_066_i32;
pub const STATUS_SUCCESS: NTSTATUS = 0_i32;
pub const STATUS_SUSPEND_COUNT_EXCEEDED: NTSTATUS = -1_073_741_750_i32;
pub const STATUS_SVHDX_ERROR_NOT_AVAILABLE: NTSTATUS = -1_067_647_232_i32;
pub const STATUS_SVHDX_ERROR_STORED: NTSTATUS = -1_067_712_512_i32;
pub const STATUS_SVHDX_NO_INITIATOR: NTSTATUS = -1_067_647_221_i32;
pub const STATUS_SVHDX_RESERVATION_CONFLICT: NTSTATUS = -1_067_647_225_i32;
pub const STATUS_SVHDX_UNIT_ATTENTION_AVAILABLE: NTSTATUS = -1_067_647_231_i32;
pub const STATUS_SVHDX_UNIT_ATTENTION_CAPACITY_DATA_CHANGED: NTSTATUS = -1_067_647_230_i32;
pub const STATUS_SVHDX_UNIT_ATTENTION_OPERATING_DEFINITION_CHANGED: NTSTATUS = -1_067_647_226_i32;
pub const STATUS_SVHDX_UNIT_ATTENTION_REGISTRATIONS_PREEMPTED: NTSTATUS = -1_067_647_227_i32;
pub const STATUS_SVHDX_UNIT_ATTENTION_RESERVATIONS_PREEMPTED: NTSTATUS = -1_067_647_229_i32;
pub const STATUS_SVHDX_UNIT_ATTENTION_RESERVATIONS_RELEASED: NTSTATUS = -1_067_647_228_i32;
pub const STATUS_SVHDX_VERSION_MISMATCH: NTSTATUS = -1_067_647_223_i32;
pub const STATUS_SVHDX_WRONG_FILE_TYPE: NTSTATUS = -1_067_647_224_i32;
pub const STATUS_SXS_ACTIVATION_CONTEXT_DISABLED: NTSTATUS = -1_072_365_561_i32;
pub const STATUS_SXS_ASSEMBLY_IS_NOT_A_DEPLOYMENT: NTSTATUS = -1_072_365_538_i32;
pub const STATUS_SXS_ASSEMBLY_MISSING: NTSTATUS = -1_072_365_556_i32;
pub const STATUS_SXS_ASSEMBLY_NOT_FOUND: NTSTATUS = -1_072_365_564_i32;
pub const STATUS_SXS_CANT_GEN_ACTCTX: NTSTATUS = -1_072_365_566_i32;
pub const STATUS_SXS_COMPONENT_STORE_CORRUPT: NTSTATUS = -1_072_365_542_i32;
pub const STATUS_SXS_CORRUPTION: NTSTATUS = -1_072_365_547_i32;
pub const STATUS_SXS_CORRUPT_ACTIVATION_STACK: NTSTATUS = -1_072_365_548_i32;
pub const STATUS_SXS_EARLY_DEACTIVATION: NTSTATUS = -1_072_365_553_i32;
pub const STATUS_SXS_FILE_HASH_MISMATCH: NTSTATUS = -1_072_365_541_i32;
pub const STATUS_SXS_FILE_HASH_MISSING: NTSTATUS = -1_072_365_529_i32;
pub const STATUS_SXS_FILE_NOT_PART_OF_ASSEMBLY: NTSTATUS = -1_072_365_537_i32;
pub const STATUS_SXS_IDENTITIES_DIFFERENT: NTSTATUS = -1_072_365_539_i32;
pub const STATUS_SXS_IDENTITY_DUPLICATE_ATTRIBUTE: NTSTATUS = -1_072_365_544_i32;
pub const STATUS_SXS_IDENTITY_PARSE_ERROR: NTSTATUS = -1_072_365_543_i32;
pub const STATUS_SXS_INVALID_ACTCTXDATA_FORMAT: NTSTATUS = -1_072_365_565_i32;
pub const STATUS_SXS_INVALID_DEACTIVATION: NTSTATUS = -1_072_365_552_i32;
pub const STATUS_SXS_INVALID_IDENTITY_ATTRIBUTE_NAME: NTSTATUS = -1_072_365_545_i32;
pub const STATUS_SXS_INVALID_IDENTITY_ATTRIBUTE_VALUE: NTSTATUS = -1_072_365_546_i32;
pub const STATUS_SXS_KEY_NOT_FOUND: NTSTATUS = -1_072_365_560_i32;
pub const STATUS_SXS_MANIFEST_FORMAT_ERROR: NTSTATUS = -1_072_365_563_i32;
pub const STATUS_SXS_MANIFEST_IDENTITY_SAME_BUT_CONTENTS_DIFFERENT: NTSTATUS = -1_072_365_540_i32;
pub const STATUS_SXS_MANIFEST_PARSE_ERROR: NTSTATUS = -1_072_365_562_i32;
pub const STATUS_SXS_MANIFEST_TOO_BIG: NTSTATUS = -1_072_365_534_i32;
pub const STATUS_SXS_MULTIPLE_DEACTIVATION: NTSTATUS = -1_072_365_551_i32;
pub const STATUS_SXS_PROCESS_DEFAULT_ALREADY_SET: NTSTATUS = -1_072_365_554_i32;
pub const STATUS_SXS_PROCESS_TERMINATION_REQUESTED: NTSTATUS = -1_072_365_549_i32;
pub const STATUS_SXS_RELEASE_ACTIVATION_CONTEXT: NTSTATUS = 1_075_118_093_i32;
pub const STATUS_SXS_SECTION_NOT_FOUND: NTSTATUS = -1_072_365_567_i32;
pub const STATUS_SXS_SETTING_NOT_REGISTERED: NTSTATUS = -1_072_365_533_i32;
pub const STATUS_SXS_SYSTEM_DEFAULT_ACTIVATION_CONTEXT_EMPTY: NTSTATUS = -1_072_365_550_i32;
pub const STATUS_SXS_THREAD_QUERIES_DISABLED: NTSTATUS = -1_072_365_557_i32;
pub const STATUS_SXS_TRANSACTION_CLOSURE_INCOMPLETE: NTSTATUS = -1_072_365_532_i32;
pub const STATUS_SXS_VERSION_CONFLICT: NTSTATUS = -1_072_365_559_i32;
pub const STATUS_SXS_WRONG_SECTION_TYPE: NTSTATUS = -1_072_365_558_i32;
pub const STATUS_SYMLINK_CLASS_DISABLED: NTSTATUS = -1_073_740_011_i32;
pub const STATUS_SYNCHRONIZATION_REQUIRED: NTSTATUS = -1_073_741_516_i32;
pub const STATUS_SYSTEM_DEVICE_NOT_FOUND: NTSTATUS = -1_073_740_718_i32;
pub const STATUS_SYSTEM_HIVE_TOO_LARGE: NTSTATUS = -1_073_740_946_i32;
pub const STATUS_SYSTEM_IMAGE_BAD_SIGNATURE: NTSTATUS = -1_073_741_103_i32;
pub const STATUS_SYSTEM_INTEGRITY_INVALID_POLICY: NTSTATUS = -1_058_471_933_i32;
pub const STATUS_SYSTEM_INTEGRITY_POLICY_NOT_SIGNED: NTSTATUS = -1_058_471_932_i32;
pub const STATUS_SYSTEM_INTEGRITY_POLICY_VIOLATION: NTSTATUS = -1_058_471_934_i32;
pub const STATUS_SYSTEM_INTEGRITY_REPUTATION_DANGEROUS_EXT: NTSTATUS = -1_058_471_927_i32;
pub const STATUS_SYSTEM_INTEGRITY_REPUTATION_MALICIOUS: NTSTATUS = -1_058_471_929_i32;
pub const STATUS_SYSTEM_INTEGRITY_REPUTATION_OFFLINE: NTSTATUS = -1_058_471_926_i32;
pub const STATUS_SYSTEM_INTEGRITY_REPUTATION_PUA: NTSTATUS = -1_058_471_928_i32;
pub const STATUS_SYSTEM_INTEGRITY_REPUTATION_UNATTAINABLE: NTSTATUS = -1_058_471_924_i32;
pub const STATUS_SYSTEM_INTEGRITY_REPUTATION_UNFRIENDLY_FILE: NTSTATUS = -1_058_471_925_i32;
pub const STATUS_SYSTEM_INTEGRITY_ROLLBACK_DETECTED: NTSTATUS = -1_058_471_935_i32;
pub const STATUS_SYSTEM_INTEGRITY_SUPPLEMENTAL_POLICY_NOT_AUTHORIZED: NTSTATUS = -1_058_471_930_i32;
pub const STATUS_SYSTEM_INTEGRITY_TOO_MANY_POLICIES: NTSTATUS = -1_058_471_931_i32;
pub const STATUS_SYSTEM_NEEDS_REMEDIATION: NTSTATUS = -1_073_740_674_i32;
pub const STATUS_SYSTEM_POWERSTATE_COMPLEX_TRANSITION: NTSTATUS = 1_073_741_873_i32;
pub const STATUS_SYSTEM_POWERSTATE_TRANSITION: NTSTATUS = 1_073_741_871_i32;
pub const STATUS_SYSTEM_PROCESS_TERMINATED: NTSTATUS = -1_073_741_286_i32;
pub const STATUS_SYSTEM_SHUTDOWN: NTSTATUS = -1_073_741_077_i32;
pub const STATUS_THREADPOOL_FREE_LIBRARY_ON_COMPLETION_FAILED: NTSTATUS = -1_073_740_018_i32;
pub const STATUS_THREADPOOL_HANDLE_EXCEPTION: NTSTATUS = -1_073_740_022_i32;
pub const STATUS_THREADPOOL_RELEASED_DURING_OPERATION: NTSTATUS = -1_073_740_017_i32;
pub const STATUS_THREADPOOL_RELEASE_MUTEX_ON_COMPLETION_FAILED: NTSTATUS = -1_073_740_019_i32;
pub const STATUS_THREADPOOL_RELEASE_SEMAPHORE_ON_COMPLETION_FAILED: NTSTATUS = -1_073_740_020_i32;
pub const STATUS_THREADPOOL_SET_EVENT_ON_COMPLETION_FAILED: NTSTATUS = -1_073_740_021_i32;
pub const STATUS_THREAD_ALREADY_IN_SESSION: NTSTATUS = -1_073_740_714_i32;
pub const STATUS_THREAD_ALREADY_IN_TASK: NTSTATUS = -1_073_740_542_i32;
pub const STATUS_THREAD_IS_TERMINATING: NTSTATUS = -1_073_741_749_i32;
pub const STATUS_THREAD_NOT_IN_PROCESS: NTSTATUS = -1_073_741_526_i32;
pub const STATUS_THREAD_NOT_IN_SESSION: NTSTATUS = -1_073_740_713_i32;
pub const STATUS_THREAD_NOT_RUNNING: NTSTATUS = -1_073_740_522_i32;
pub const STATUS_THREAD_WAS_SUSPENDED: NTSTATUS = 1_073_741_825_i32;
pub const STATUS_TIMEOUT: NTSTATUS = 258_i32;
pub const STATUS_TIMER_NOT_CANCELED: NTSTATUS = -1_073_741_812_i32;
pub const STATUS_TIMER_RESOLUTION_NOT_SET: NTSTATUS = -1_073_741_243_i32;
pub const STATUS_TIMER_RESUME_IGNORED: NTSTATUS = 1_073_741_861_i32;
pub const STATUS_TIME_DIFFERENCE_AT_DC: NTSTATUS = -1_073_741_517_i32;
pub const STATUS_TM_IDENTITY_MISMATCH: NTSTATUS = -1_072_103_350_i32;
pub const STATUS_TM_INITIALIZATION_FAILED: NTSTATUS = -1_072_103_420_i32;
pub const STATUS_TM_VOLATILE: NTSTATUS = -1_072_103_365_i32;
pub const STATUS_TOKEN_ALREADY_IN_USE: NTSTATUS = -1_073_741_525_i32;
pub const STATUS_TOO_LATE: NTSTATUS = -1_073_741_431_i32;
pub const STATUS_TOO_MANY_ADDRESSES: NTSTATUS = -1_073_741_303_i32;
pub const STATUS_TOO_MANY_COMMANDS: NTSTATUS = -1_073_741_631_i32;
pub const STATUS_TOO_MANY_CONTEXT_IDS: NTSTATUS = -1_073_741_478_i32;
pub const STATUS_TOO_MANY_GUIDS_REQUESTED: NTSTATUS = -1_073_741_694_i32;
pub const STATUS_TOO_MANY_LINKS: NTSTATUS = -1_073_741_211_i32;
pub const STATUS_TOO_MANY_LUIDS_REQUESTED: NTSTATUS = -1_073_741_708_i32;
pub const STATUS_TOO_MANY_NAMES: NTSTATUS = -1_073_741_619_i32;
pub const STATUS_TOO_MANY_NODES: NTSTATUS = -1_073_741_298_i32;
pub const STATUS_TOO_MANY_OPENED_FILES: NTSTATUS = -1_073_741_537_i32;
pub const STATUS_TOO_MANY_PAGING_FILES: NTSTATUS = -1_073_741_673_i32;
pub const STATUS_TOO_MANY_PRINCIPALS: NTSTATUS = -1_073_741_065_i32;
pub const STATUS_TOO_MANY_SECRETS: NTSTATUS = -1_073_741_482_i32;
pub const STATUS_TOO_MANY_SEGMENT_DESCRIPTORS: NTSTATUS = -1_073_740_685_i32;
pub const STATUS_TOO_MANY_SESSIONS: NTSTATUS = -1_073_741_618_i32;
pub const STATUS_TOO_MANY_SIDS: NTSTATUS = -1_073_741_442_i32;
pub const STATUS_TOO_MANY_THREADS: NTSTATUS = -1_073_741_527_i32;
pub const STATUS_TPM_20_E_ASYMMETRIC: NTSTATUS = -1_071_054_719_i32;
pub const STATUS_TPM_20_E_ATTRIBUTES: NTSTATUS = -1_071_054_718_i32;
pub const STATUS_TPM_20_E_AUTHSIZE: NTSTATUS = -1_071_054_524_i32;
pub const STATUS_TPM_20_E_AUTH_CONTEXT: NTSTATUS = -1_071_054_523_i32;
pub const STATUS_TPM_20_E_AUTH_FAIL: NTSTATUS = -1_071_054_706_i32;
pub const STATUS_TPM_20_E_AUTH_MISSING: NTSTATUS = -1_071_054_555_i32;
pub const STATUS_TPM_20_E_AUTH_TYPE: NTSTATUS = -1_071_054_556_i32;
pub const STATUS_TPM_20_E_AUTH_UNAVAILABLE: NTSTATUS = -1_071_054_545_i32;
pub const STATUS_TPM_20_E_BAD_AUTH: NTSTATUS = -1_071_054_686_i32;
pub const STATUS_TPM_20_E_BAD_CONTEXT: NTSTATUS = -1_071_054_512_i32;
pub const STATUS_TPM_20_E_BINDING: NTSTATUS = -1_071_054_683_i32;
pub const STATUS_TPM_20_E_COMMAND_CODE: NTSTATUS = -1_071_054_525_i32;
pub const STATUS_TPM_20_E_COMMAND_SIZE: NTSTATUS = -1_071_054_526_i32;
pub const STATUS_TPM_20_E_CPHASH: NTSTATUS = -1_071_054_511_i32;
pub const STATUS_TPM_20_E_CURVE: NTSTATUS = -1_071_054_682_i32;
pub const STATUS_TPM_20_E_DISABLED: NTSTATUS = -1_071_054_560_i32;
pub const STATUS_TPM_20_E_ECC_CURVE: NTSTATUS = -1_071_054_557_i32;
pub const STATUS_TPM_20_E_ECC_POINT: NTSTATUS = -1_071_054_681_i32;
pub const STATUS_TPM_20_E_EXCLUSIVE: NTSTATUS = -1_071_054_559_i32;
pub const STATUS_TPM_20_E_EXPIRED: NTSTATUS = -1_071_054_685_i32;
pub const STATUS_TPM_20_E_FAILURE: NTSTATUS = -1_071_054_591_i32;
pub const STATUS_TPM_20_E_HANDLE: NTSTATUS = -1_071_054_709_i32;
pub const STATUS_TPM_20_E_HASH: NTSTATUS = -1_071_054_717_i32;
pub const STATUS_TPM_20_E_HIERARCHY: NTSTATUS = -1_071_054_715_i32;
pub const STATUS_TPM_20_E_HMAC: NTSTATUS = -1_071_054_567_i32;
pub const STATUS_TPM_20_E_INITIALIZE: NTSTATUS = -1_071_054_592_i32;
pub const STATUS_TPM_20_E_INSUFFICIENT: NTSTATUS = -1_071_054_694_i32;
pub const STATUS_TPM_20_E_INTEGRITY: NTSTATUS = -1_071_054_689_i32;
pub const STATUS_TPM_20_E_KDF: NTSTATUS = -1_071_054_708_i32;
pub const STATUS_TPM_20_E_KEY: NTSTATUS = -1_071_054_692_i32;
pub const STATUS_TPM_20_E_KEY_SIZE: NTSTATUS = -1_071_054_713_i32;
pub const STATUS_TPM_20_E_MGF: NTSTATUS = -1_071_054_712_i32;
pub const STATUS_TPM_20_E_MODE: NTSTATUS = -1_071_054_711_i32;
pub const STATUS_TPM_20_E_NEEDS_TEST: NTSTATUS = -1_071_054_509_i32;
pub const STATUS_TPM_20_E_NONCE: NTSTATUS = -1_071_054_705_i32;
pub const STATUS_TPM_20_E_NO_RESULT: NTSTATUS = -1_071_054_508_i32;
pub const STATUS_TPM_20_E_NV_AUTHORIZATION: NTSTATUS = -1_071_054_519_i32;
pub const STATUS_TPM_20_E_NV_DEFINED: NTSTATUS = -1_071_054_516_i32;
pub const STATUS_TPM_20_E_NV_LOCKED: NTSTATUS = -1_071_054_520_i32;
pub const STATUS_TPM_20_E_NV_RANGE: NTSTATUS = -1_071_054_522_i32;
pub const STATUS_TPM_20_E_NV_SIZE: NTSTATUS = -1_071_054_521_i32;
pub const STATUS_TPM_20_E_NV_SPACE: NTSTATUS = -1_071_054_517_i32;
pub const STATUS_TPM_20_E_NV_UNINITIALIZED: NTSTATUS = -1_071_054_518_i32;
pub const STATUS_TPM_20_E_PARENT: NTSTATUS = -1_071_054_510_i32;
pub const STATUS_TPM_20_E_PCR: NTSTATUS = -1_071_054_553_i32;
pub const STATUS_TPM_20_E_PCR_CHANGED: NTSTATUS = -1_071_054_552_i32;
pub const STATUS_TPM_20_E_POLICY: NTSTATUS = -1_071_054_554_i32;
pub const STATUS_TPM_20_E_POLICY_CC: NTSTATUS = -1_071_054_684_i32;
pub const STATUS_TPM_20_E_POLICY_FAIL: NTSTATUS = -1_071_054_691_i32;
pub const STATUS_TPM_20_E_PP: NTSTATUS = -1_071_054_704_i32;
pub const STATUS_TPM_20_E_PRIVATE: NTSTATUS = -1_071_054_581_i32;
pub const STATUS_TPM_20_E_RANGE: NTSTATUS = -1_071_054_707_i32;
pub const STATUS_TPM_20_E_REBOOT: NTSTATUS = -1_071_054_544_i32;
pub const STATUS_TPM_20_E_RESERVED_BITS: NTSTATUS = -1_071_054_687_i32;
pub const STATUS_TPM_20_E_SCHEME: NTSTATUS = -1_071_054_702_i32;
pub const STATUS_TPM_20_E_SELECTOR: NTSTATUS = -1_071_054_696_i32;
pub const STATUS_TPM_20_E_SENSITIVE: NTSTATUS = -1_071_054_507_i32;
pub const STATUS_TPM_20_E_SEQUENCE: NTSTATUS = -1_071_054_589_i32;
pub const STATUS_TPM_20_E_SIGNATURE: NTSTATUS = -1_071_054_693_i32;
pub const STATUS_TPM_20_E_SIZE: NTSTATUS = -1_071_054_699_i32;
pub const STATUS_TPM_20_E_SYMMETRIC: NTSTATUS = -1_071_054_698_i32;
pub const STATUS_TPM_20_E_TAG: NTSTATUS = -1_071_054_697_i32;
pub const STATUS_TPM_20_E_TICKET: NTSTATUS = -1_071_054_688_i32;
pub const STATUS_TPM_20_E_TOO_MANY_CONTEXTS: NTSTATUS = -1_071_054_546_i32;
pub const STATUS_TPM_20_E_TYPE: NTSTATUS = -1_071_054_710_i32;
pub const STATUS_TPM_20_E_UNBALANCED: NTSTATUS = -1_071_054_543_i32;
pub const STATUS_TPM_20_E_UPGRADE: NTSTATUS = -1_071_054_547_i32;
pub const STATUS_TPM_20_E_VALUE: NTSTATUS = -1_071_054_716_i32;
pub const STATUS_TPM_ACCESS_DENIED: NTSTATUS = -1_071_050_748_i32;
pub const STATUS_TPM_AREA_LOCKED: NTSTATUS = -1_071_054_788_i32;
pub const STATUS_TPM_AUDITFAILURE: NTSTATUS = -1_071_054_844_i32;
pub const STATUS_TPM_AUDITFAIL_SUCCESSFUL: NTSTATUS = -1_071_054_799_i32;
pub const STATUS_TPM_AUDITFAIL_UNSUCCESSFUL: NTSTATUS = -1_071_054_800_i32;
pub const STATUS_TPM_AUTH2FAIL: NTSTATUS = -1_071_054_819_i32;
pub const STATUS_TPM_AUTHFAIL: NTSTATUS = -1_071_054_847_i32;
pub const STATUS_TPM_AUTH_CONFLICT: NTSTATUS = -1_071_054_789_i32;
pub const STATUS_TPM_BADCONTEXT: NTSTATUS = -1_071_054_758_i32;
pub const STATUS_TPM_BADINDEX: NTSTATUS = -1_071_054_846_i32;
pub const STATUS_TPM_BADTAG: NTSTATUS = -1_071_054_818_i32;
pub const STATUS_TPM_BAD_ATTRIBUTES: NTSTATUS = -1_071_054_782_i32;
pub const STATUS_TPM_BAD_COUNTER: NTSTATUS = -1_071_054_779_i32;
pub const STATUS_TPM_BAD_DATASIZE: NTSTATUS = -1_071_054_805_i32;
pub const STATUS_TPM_BAD_DELEGATE: NTSTATUS = -1_071_054_759_i32;
pub const STATUS_TPM_BAD_HANDLE: NTSTATUS = -1_071_054_760_i32;
pub const STATUS_TPM_BAD_KEY_PROPERTY: NTSTATUS = -1_071_054_808_i32;
pub const STATUS_TPM_BAD_LOCALITY: NTSTATUS = -1_071_054_787_i32;
pub const STATUS_TPM_BAD_MIGRATION: NTSTATUS = -1_071_054_807_i32;
pub const STATUS_TPM_BAD_MODE: NTSTATUS = -1_071_054_804_i32;
pub const STATUS_TPM_BAD_ORDINAL: NTSTATUS = -1_071_054_838_i32;
pub const STATUS_TPM_BAD_PARAMETER: NTSTATUS = -1_071_054_845_i32;
pub const STATUS_TPM_BAD_PARAM_SIZE: NTSTATUS = -1_071_054_823_i32;
pub const STATUS_TPM_BAD_PRESENCE: NTSTATUS = -1_071_054_803_i32;
pub const STATUS_TPM_BAD_SCHEME: NTSTATUS = -1_071_054_806_i32;
pub const STATUS_TPM_BAD_SIGNATURE: NTSTATUS = -1_071_054_750_i32;
pub const STATUS_TPM_BAD_TYPE: NTSTATUS = -1_071_054_796_i32;
pub const STATUS_TPM_BAD_VERSION: NTSTATUS = -1_071_054_802_i32;
pub const STATUS_TPM_CLEAR_DISABLED: NTSTATUS = -1_071_054_843_i32;
pub const STATUS_TPM_COMMAND_BLOCKED: NTSTATUS = -1_071_053_824_i32;
pub const STATUS_TPM_COMMAND_CANCELED: NTSTATUS = -1_071_050_751_i32;
pub const STATUS_TPM_CONTEXT_GAP: NTSTATUS = -1_071_054_777_i32;
pub const STATUS_TPM_DAA_INPUT_DATA0: NTSTATUS = -1_071_054_767_i32;
pub const STATUS_TPM_DAA_INPUT_DATA1: NTSTATUS = -1_071_054_766_i32;
pub const STATUS_TPM_DAA_ISSUER_SETTINGS: NTSTATUS = -1_071_054_765_i32;
pub const STATUS_TPM_DAA_ISSUER_VALIDITY: NTSTATUS = -1_071_054_762_i32;
pub const STATUS_TPM_DAA_RESOURCES: NTSTATUS = -1_071_054_768_i32;
pub const STATUS_TPM_DAA_STAGE: NTSTATUS = -1_071_054_763_i32;
pub const STATUS_TPM_DAA_TPM_SETTINGS: NTSTATUS = -1_071_054_764_i32;
pub const STATUS_TPM_DAA_WRONG_W: NTSTATUS = -1_071_054_761_i32;
pub const STATUS_TPM_DEACTIVATED: NTSTATUS = -1_071_054_842_i32;
pub const STATUS_TPM_DECRYPT_ERROR: NTSTATUS = -1_071_054_815_i32;
pub const STATUS_TPM_DEFEND_LOCK_RUNNING: NTSTATUS = -1_071_052_797_i32;
pub const STATUS_TPM_DELEGATE_ADMIN: NTSTATUS = -1_071_054_771_i32;
pub const STATUS_TPM_DELEGATE_FAMILY: NTSTATUS = -1_071_054_772_i32;
pub const STATUS_TPM_DELEGATE_LOCK: NTSTATUS = -1_071_054_773_i32;
pub const STATUS_TPM_DISABLED: NTSTATUS = -1_071_054_841_i32;
pub const STATUS_TPM_DISABLED_CMD: NTSTATUS = -1_071_054_840_i32;
pub const STATUS_TPM_DOING_SELFTEST: NTSTATUS = -1_071_052_798_i32;
pub const STATUS_TPM_DUPLICATE_VHANDLE: NTSTATUS = -1_071_053_822_i32;
pub const STATUS_TPM_EMBEDDED_COMMAND_BLOCKED: NTSTATUS = -1_071_053_821_i32;
pub const STATUS_TPM_EMBEDDED_COMMAND_UNSUPPORTED: NTSTATUS = -1_071_053_820_i32;
pub const STATUS_TPM_ENCRYPT_ERROR: NTSTATUS = -1_071_054_816_i32;
pub const STATUS_TPM_ERROR_MASK: NTSTATUS = -1_071_054_848_i32;
pub const STATUS_TPM_FAIL: NTSTATUS = -1_071_054_839_i32;
pub const STATUS_TPM_FAILEDSELFTEST: NTSTATUS = -1_071_054_820_i32;
pub const STATUS_TPM_FAMILYCOUNT: NTSTATUS = -1_071_054_784_i32;
pub const STATUS_TPM_INAPPROPRIATE_ENC: NTSTATUS = -1_071_054_834_i32;
pub const STATUS_TPM_INAPPROPRIATE_SIG: NTSTATUS = -1_071_054_809_i32;
pub const STATUS_TPM_INSTALL_DISABLED: NTSTATUS = -1_071_054_837_i32;
pub const STATUS_TPM_INSUFFICIENT_BUFFER: NTSTATUS = -1_071_050_747_i32;
pub const STATUS_TPM_INVALID_AUTHHANDLE: NTSTATUS = -1_071_054_814_i32;
pub const STATUS_TPM_INVALID_FAMILY: NTSTATUS = -1_071_054_793_i32;
pub const STATUS_TPM_INVALID_HANDLE: NTSTATUS = -1_071_053_823_i32;
pub const STATUS_TPM_INVALID_KEYHANDLE: NTSTATUS = -1_071_054_836_i32;
pub const STATUS_TPM_INVALID_KEYUSAGE: NTSTATUS = -1_071_054_812_i32;
pub const STATUS_TPM_INVALID_PCR_INFO: NTSTATUS = -1_071_054_832_i32;
pub const STATUS_TPM_INVALID_POSTINIT: NTSTATUS = -1_071_054_810_i32;
pub const STATUS_TPM_INVALID_RESOURCE: NTSTATUS = -1_071_054_795_i32;
pub const STATUS_TPM_INVALID_STRUCTURE: NTSTATUS = -1_071_054_781_i32;
pub const STATUS_TPM_IOERROR: NTSTATUS = -1_071_054_817_i32;
pub const STATUS_TPM_KEYNOTFOUND: NTSTATUS = -1_071_054_835_i32;
pub const STATUS_TPM_KEY_NOTSUPPORTED: NTSTATUS = -1_071_054_790_i32;
pub const STATUS_TPM_KEY_OWNER_CONTROL: NTSTATUS = -1_071_054_780_i32;
pub const STATUS_TPM_MAXNVWRITES: NTSTATUS = -1_071_054_776_i32;
pub const STATUS_TPM_MA_AUTHORITY: NTSTATUS = -1_071_054_753_i32;
pub const STATUS_TPM_MA_DESTINATION: NTSTATUS = -1_071_054_755_i32;
pub const STATUS_TPM_MA_SOURCE: NTSTATUS = -1_071_054_754_i32;
pub const STATUS_TPM_MA_TICKET_SIGNATURE: NTSTATUS = -1_071_054_756_i32;
pub const STATUS_TPM_MIGRATEFAIL: NTSTATUS = -1_071_054_833_i32;
pub const STATUS_TPM_NEEDS_SELFTEST: NTSTATUS = -1_071_052_799_i32;
pub const STATUS_TPM_NOCONTEXTSPACE: NTSTATUS = -1_071_054_749_i32;
pub const STATUS_TPM_NOOPERATOR: NTSTATUS = -1_071_054_775_i32;
pub const STATUS_TPM_NOSPACE: NTSTATUS = -1_071_054_831_i32;
pub const STATUS_TPM_NOSRK: NTSTATUS = -1_071_054_830_i32;
pub const STATUS_TPM_NOTFIPS: NTSTATUS = -1_071_054_794_i32;
pub const STATUS_TPM_NOTLOCAL: NTSTATUS = -1_071_054_797_i32;
pub const STATUS_TPM_NOTRESETABLE: NTSTATUS = -1_071_054_798_i32;
pub const STATUS_TPM_NOTSEALED_BLOB: NTSTATUS = -1_071_054_829_i32;
pub const STATUS_TPM_NOT_FOUND: NTSTATUS = -1_071_050_749_i32;
pub const STATUS_TPM_NOT_FULLWRITE: NTSTATUS = -1_071_054_778_i32;
pub const STATUS_TPM_NO_ENDORSEMENT: NTSTATUS = -1_071_054_813_i32;
pub const STATUS_TPM_NO_NV_PERMISSION: NTSTATUS = -1_071_054_792_i32;
pub const STATUS_TPM_NO_WRAP_TRANSPORT: NTSTATUS = -1_071_054_801_i32;
pub const STATUS_TPM_OWNER_CONTROL: NTSTATUS = -1_071_054_769_i32;
pub const STATUS_TPM_OWNER_SET: NTSTATUS = -1_071_054_828_i32;
pub const STATUS_TPM_PERMANENTEK: NTSTATUS = -1_071_054_751_i32;
pub const STATUS_TPM_PER_NOWRITE: NTSTATUS = -1_071_054_785_i32;
pub const STATUS_TPM_PPI_FUNCTION_UNSUPPORTED: NTSTATUS = -1_071_050_746_i32;
pub const STATUS_TPM_READ_ONLY: NTSTATUS = -1_071_054_786_i32;
pub const STATUS_TPM_REQUIRES_SIGN: NTSTATUS = -1_071_054_791_i32;
pub const STATUS_TPM_RESOURCEMISSING: NTSTATUS = -1_071_054_774_i32;
pub const STATUS_TPM_RESOURCES: NTSTATUS = -1_071_054_827_i32;
pub const STATUS_TPM_RETRY: NTSTATUS = -1_071_052_800_i32;
pub const STATUS_TPM_SHA_ERROR: NTSTATUS = -1_071_054_821_i32;
pub const STATUS_TPM_SHA_THREAD: NTSTATUS = -1_071_054_822_i32;
pub const STATUS_TPM_SHORTRANDOM: NTSTATUS = -1_071_054_826_i32;
pub const STATUS_TPM_SIZE: NTSTATUS = -1_071_054_825_i32;
pub const STATUS_TPM_TOOMANYCONTEXTS: NTSTATUS = -1_071_054_757_i32;
pub const STATUS_TPM_TOO_MANY_CONTEXTS: NTSTATUS = -1_071_050_750_i32;
pub const STATUS_TPM_TRANSPORT_NOTEXCLUSIVE: NTSTATUS = -1_071_054_770_i32;
pub const STATUS_TPM_WRITE_LOCKED: NTSTATUS = -1_071_054_783_i32;
pub const STATUS_TPM_WRONGPCRVAL: NTSTATUS = -1_071_054_824_i32;
pub const STATUS_TPM_WRONG_ENTITYTYPE: NTSTATUS = -1_071_054_811_i32;
pub const STATUS_TPM_ZERO_EXHAUST_ENABLED: NTSTATUS = -1_071_038_464_i32;
pub const STATUS_TRANSACTED_MAPPING_UNSUPPORTED_REMOTE: NTSTATUS = -1_072_103_360_i32;
pub const STATUS_TRANSACTIONAL_CONFLICT: NTSTATUS = -1_072_103_423_i32;
pub const STATUS_TRANSACTIONAL_OPEN_NOT_ALLOWED: NTSTATUS = -1_072_103_361_i32;
pub const STATUS_TRANSACTIONMANAGER_IDENTITY_MISMATCH: NTSTATUS = -1_072_103_332_i32;
pub const STATUS_TRANSACTIONMANAGER_NOT_FOUND: NTSTATUS = -1_072_103_343_i32;
pub const STATUS_TRANSACTIONMANAGER_NOT_ONLINE: NTSTATUS = -1_072_103_342_i32;
pub const STATUS_TRANSACTIONMANAGER_RECOVERY_NAME_COLLISION: NTSTATUS = -1_072_103_341_i32;
pub const STATUS_TRANSACTIONS_NOT_FROZEN: NTSTATUS = -1_072_103_355_i32;
pub const STATUS_TRANSACTIONS_UNSUPPORTED_REMOTE: NTSTATUS = -1_072_103_414_i32;
pub const STATUS_TRANSACTION_ABORTED: NTSTATUS = -1_073_741_297_i32;
pub const STATUS_TRANSACTION_ALREADY_ABORTED: NTSTATUS = -1_072_103_403_i32;
pub const STATUS_TRANSACTION_ALREADY_COMMITTED: NTSTATUS = -1_072_103_402_i32;
pub const STATUS_TRANSACTION_FREEZE_IN_PROGRESS: NTSTATUS = -1_072_103_354_i32;
pub const STATUS_TRANSACTION_INTEGRITY_VIOLATED: NTSTATUS = -1_072_103_333_i32;
pub const STATUS_TRANSACTION_INVALID_ID: NTSTATUS = -1_073_741_292_i32;
pub const STATUS_TRANSACTION_INVALID_MARSHALL_BUFFER: NTSTATUS = -1_072_103_401_i32;
pub const STATUS_TRANSACTION_INVALID_TYPE: NTSTATUS = -1_073_741_291_i32;
pub const STATUS_TRANSACTION_MUST_WRITETHROUGH: NTSTATUS = -1_072_103_330_i32;
pub const STATUS_TRANSACTION_NOT_ACTIVE: NTSTATUS = -1_072_103_421_i32;
pub const STATUS_TRANSACTION_NOT_ENLISTED: NTSTATUS = -1_072_103_327_i32;
pub const STATUS_TRANSACTION_NOT_FOUND: NTSTATUS = -1_072_103_346_i32;
pub const STATUS_TRANSACTION_NOT_JOINED: NTSTATUS = -1_072_103_417_i32;
pub const STATUS_TRANSACTION_NOT_REQUESTED: NTSTATUS = -1_072_103_404_i32;
pub const STATUS_TRANSACTION_NOT_ROOT: NTSTATUS = -1_072_103_340_i32;
pub const STATUS_TRANSACTION_NO_MATCH: NTSTATUS = -1_073_741_294_i32;
pub const STATUS_TRANSACTION_NO_RELEASE: NTSTATUS = -1_073_741_295_i32;
pub const STATUS_TRANSACTION_NO_SUPERIOR: NTSTATUS = -1_072_103_329_i32;
pub const STATUS_TRANSACTION_OBJECT_EXPIRED: NTSTATUS = -1_072_103_339_i32;
pub const STATUS_TRANSACTION_PROPAGATION_FAILED: NTSTATUS = -1_072_103_408_i32;
pub const STATUS_TRANSACTION_RECORD_TOO_LONG: NTSTATUS = -1_072_103_336_i32;
pub const STATUS_TRANSACTION_REQUEST_NOT_VALID: NTSTATUS = -1_072_103_405_i32;
pub const STATUS_TRANSACTION_REQUIRED_PROMOTION: NTSTATUS = -1_072_103_357_i32;
pub const STATUS_TRANSACTION_RESPONDED: NTSTATUS = -1_073_741_293_i32;
pub const STATUS_TRANSACTION_RESPONSE_NOT_ENLISTED: NTSTATUS = -1_072_103_337_i32;
pub const STATUS_TRANSACTION_SCOPE_CALLBACKS_NOT_SET: NTSTATUS = -2_145_845_182_i32;
pub const STATUS_TRANSACTION_SUPERIOR_EXISTS: NTSTATUS = -1_072_103_406_i32;
pub const STATUS_TRANSACTION_TIMED_OUT: NTSTATUS = -1_073_741_296_i32;
pub const STATUS_TRANSLATION_COMPLETE: NTSTATUS = 288_i32;
pub const STATUS_TRANSPORT_FULL: NTSTATUS = -1_073_741_110_i32;
pub const STATUS_TRIGGERED_EXECUTABLE_MEMORY_WRITE: NTSTATUS = -1_073_739_994_i32;
pub const STATUS_TRIM_READ_ZERO_NOT_SUPPORTED: NTSTATUS = -1_073_740_686_i32;
pub const STATUS_TRUSTED_DOMAIN_FAILURE: NTSTATUS = -1_073_741_428_i32;
pub const STATUS_TRUSTED_RELATIONSHIP_FAILURE: NTSTATUS = -1_073_741_427_i32;
pub const STATUS_TRUST_FAILURE: NTSTATUS = -1_073_741_424_i32;
pub const STATUS_TS_INCOMPATIBLE_SESSIONS: NTSTATUS = -1_073_086_407_i32;
pub const STATUS_TS_VIDEO_SUBSYSTEM_ERROR: NTSTATUS = -1_073_086_406_i32;
pub const STATUS_TXF_ATTRIBUTE_CORRUPT: NTSTATUS = -1_072_103_363_i32;
pub const STATUS_TXF_DIR_NOT_EMPTY: NTSTATUS = -1_072_103_367_i32;
pub const STATUS_TXF_METADATA_ALREADY_PRESENT: NTSTATUS = -2_145_845_183_i32;
pub const STATUS_UNABLE_TO_DECOMMIT_VM: NTSTATUS = -1_073_741_780_i32;
pub const STATUS_UNABLE_TO_DELETE_SECTION: NTSTATUS = -1_073_741_797_i32;
pub const STATUS_UNABLE_TO_FREE_VM: NTSTATUS = -1_073_741_798_i32;
pub const STATUS_UNABLE_TO_LOCK_MEDIA: NTSTATUS = -1_073_741_451_i32;
pub const STATUS_UNABLE_TO_UNLOAD_MEDIA: NTSTATUS = -1_073_741_450_i32;
pub const STATUS_UNDEFINED_CHARACTER: NTSTATUS = -1_073_741_469_i32;
pub const STATUS_UNDEFINED_SCOPE: NTSTATUS = -1_073_740_540_i32;
pub const STATUS_UNEXPECTED_IO_ERROR: NTSTATUS = -1_073_741_591_i32;
pub const STATUS_UNEXPECTED_MM_CREATE_ERR: NTSTATUS = -1_073_741_590_i32;
pub const STATUS_UNEXPECTED_MM_EXTEND_ERR: NTSTATUS = -1_073_741_588_i32;
pub const STATUS_UNEXPECTED_MM_MAP_ERROR: NTSTATUS = -1_073_741_589_i32;
pub const STATUS_UNEXPECTED_NETWORK_ERROR: NTSTATUS = -1_073_741_628_i32;
pub const STATUS_UNFINISHED_CONTEXT_DELETED: NTSTATUS = -1_073_741_074_i32;
pub const STATUS_UNHANDLED_EXCEPTION: NTSTATUS = -1_073_741_500_i32;
pub const STATUS_UNKNOWN_REVISION: NTSTATUS = -1_073_741_736_i32;
pub const STATUS_UNMAPPABLE_CHARACTER: NTSTATUS = -1_073_741_470_i32;
pub const STATUS_UNRECOGNIZED_MEDIA: NTSTATUS = -1_073_741_804_i32;
pub const STATUS_UNRECOGNIZED_VOLUME: NTSTATUS = -1_073_741_489_i32;
pub const STATUS_UNSATISFIED_DEPENDENCIES: NTSTATUS = -1_073_740_615_i32;
pub const STATUS_UNSUCCESSFUL: NTSTATUS = -1_073_741_823_i32;
pub const STATUS_UNSUPPORTED_COMPRESSION: NTSTATUS = -1_073_741_217_i32;
pub const STATUS_UNSUPPORTED_PAGING_MODE: NTSTATUS = -1_073_740_613_i32;
pub const STATUS_UNSUPPORTED_PREAUTH: NTSTATUS = -1_073_740_975_i32;
pub const STATUS_UNTRUSTED_MOUNT_POINT: NTSTATUS = -1_073_740_612_i32;
pub const STATUS_UNWIND: NTSTATUS = -1_073_741_785_i32;
pub const STATUS_UNWIND_CONSOLIDATE: NTSTATUS = -2_147_483_607_i32;
pub const STATUS_USER2USER_REQUIRED: NTSTATUS = -1_073_740_792_i32;
pub const STATUS_USER_APC: NTSTATUS = 192_i32;
pub const STATUS_USER_DELETE_TRUST_QUOTA_EXCEEDED: NTSTATUS = -1_073_740_797_i32;
pub const STATUS_USER_EXISTS: NTSTATUS = -1_073_741_725_i32;
pub const STATUS_USER_MAPPED_FILE: NTSTATUS = -1_073_741_245_i32;
pub const STATUS_USER_SESSION_DELETED: NTSTATUS = -1_073_741_309_i32;
pub const STATUS_VALIDATE_CONTINUE: NTSTATUS = -1_073_741_199_i32;
pub const STATUS_VALID_CATALOG_HASH: NTSTATUS = 301_i32;
pub const STATUS_VALID_IMAGE_HASH: NTSTATUS = 300_i32;
pub const STATUS_VALID_STRONG_CODE_HASH: NTSTATUS = 302_i32;
pub const STATUS_VARIABLE_NOT_FOUND: NTSTATUS = -1_073_741_568_i32;
pub const STATUS_VDM_DISALLOWED: NTSTATUS = -1_073_740_780_i32;
pub const STATUS_VDM_HARD_ERROR: NTSTATUS = -1_073_741_283_i32;
pub const STATUS_VERIFIER_STOP: NTSTATUS = -1_073_740_767_i32;
pub const STATUS_VERIFY_REQUIRED: NTSTATUS = -2_147_483_626_i32;
pub const STATUS_VHDSET_BACKING_STORAGE_NOT_FOUND: NTSTATUS = -1_067_647_220_i32;
pub const STATUS_VHD_ALREADY_AT_OR_BELOW_MINIMUM_VIRTUAL_SIZE: NTSTATUS = -1_069_940_685_i32;
pub const STATUS_VHD_BITMAP_MISMATCH: NTSTATUS = -1_069_940_724_i32;
pub const STATUS_VHD_BLOCK_ALLOCATION_FAILURE: NTSTATUS = -1_069_940_727_i32;
pub const STATUS_VHD_BLOCK_ALLOCATION_TABLE_CORRUPT: NTSTATUS = -1_069_940_726_i32;
pub const STATUS_VHD_CHANGE_TRACKING_DISABLED: NTSTATUS = -1_069_940_694_i32;
pub const STATUS_VHD_CHILD_PARENT_ID_MISMATCH: NTSTATUS = -1_069_940_722_i32;
pub const STATUS_VHD_CHILD_PARENT_SIZE_MISMATCH: NTSTATUS = -1_069_940_713_i32;
pub const STATUS_VHD_CHILD_PARENT_TIMESTAMP_MISMATCH: NTSTATUS = -1_069_940_721_i32;
pub const STATUS_VHD_COULD_NOT_COMPUTE_MINIMUM_VIRTUAL_SIZE: NTSTATUS = -1_069_940_686_i32;
pub const STATUS_VHD_DIFFERENCING_CHAIN_CYCLE_DETECTED: NTSTATUS = -1_069_940_712_i32;
pub const STATUS_VHD_DIFFERENCING_CHAIN_ERROR_IN_PARENT: NTSTATUS = -1_069_940_711_i32;
pub const STATUS_VHD_DRIVE_FOOTER_CHECKSUM_MISMATCH: NTSTATUS = -1_069_940_734_i32;
pub const STATUS_VHD_DRIVE_FOOTER_CORRUPT: NTSTATUS = -1_069_940_733_i32;
pub const STATUS_VHD_DRIVE_FOOTER_MISSING: NTSTATUS = -1_069_940_735_i32;
pub const STATUS_VHD_FORMAT_UNKNOWN: NTSTATUS = -1_069_940_732_i32;
pub const STATUS_VHD_FORMAT_UNSUPPORTED_VERSION: NTSTATUS = -1_069_940_731_i32;
pub const STATUS_VHD_INVALID_BLOCK_SIZE: NTSTATUS = -1_069_940_725_i32;
pub const STATUS_VHD_INVALID_CHANGE_TRACKING_ID: NTSTATUS = -1_069_940_695_i32;
pub const STATUS_VHD_INVALID_FILE_SIZE: NTSTATUS = -1_069_940_717_i32;
pub const STATUS_VHD_INVALID_SIZE: NTSTATUS = -1_069_940_718_i32;
pub const STATUS_VHD_INVALID_STATE: NTSTATUS = -1_069_940_708_i32;
pub const STATUS_VHD_INVALID_TYPE: NTSTATUS = -1_069_940_709_i32;
pub const STATUS_VHD_METADATA_FULL: NTSTATUS = -1_069_940_696_i32;
pub const STATUS_VHD_METADATA_READ_FAILURE: NTSTATUS = -1_069_940_720_i32;
pub const STATUS_VHD_METADATA_WRITE_FAILURE: NTSTATUS = -1_069_940_719_i32;
pub const STATUS_VHD_MISSING_CHANGE_TRACKING_INFORMATION: NTSTATUS = -1_069_940_688_i32;
pub const STATUS_VHD_PARENT_VHD_ACCESS_DENIED: NTSTATUS = -1_069_940_714_i32;
pub const STATUS_VHD_PARENT_VHD_NOT_FOUND: NTSTATUS = -1_069_940_723_i32;
pub const STATUS_VHD_RESIZE_WOULD_TRUNCATE_DATA: NTSTATUS = -1_069_940_687_i32;
pub const STATUS_VHD_SHARED: NTSTATUS = -1_067_647_222_i32;
pub const STATUS_VHD_SPARSE_HEADER_CHECKSUM_MISMATCH: NTSTATUS = -1_069_940_730_i32;
pub const STATUS_VHD_SPARSE_HEADER_CORRUPT: NTSTATUS = -1_069_940_728_i32;
pub const STATUS_VHD_SPARSE_HEADER_UNSUPPORTED_VERSION: NTSTATUS = -1_069_940_729_i32;
pub const STATUS_VIDEO_DRIVER_DEBUG_REPORT_REQUEST: NTSTATUS = 1_075_511_532_i32;
pub const STATUS_VIDEO_HUNG_DISPLAY_DRIVER_THREAD: NTSTATUS = -1_071_972_118_i32;
pub const STATUS_VIDEO_HUNG_DISPLAY_DRIVER_THREAD_RECOVERED: NTSTATUS = -2_145_713_941_i32;
pub const STATUS_VID_CHILD_GPA_PAGE_SET_CORRUPTED: NTSTATUS = -1_070_137_330_i32;
pub const STATUS_VID_DUPLICATE_HANDLER: NTSTATUS = -1_070_137_343_i32;
pub const STATUS_VID_EXCEEDED_KM_CONTEXT_COUNT_LIMIT: NTSTATUS = -1_070_137_314_i32;
pub const STATUS_VID_EXCEEDED_MBP_ENTRY_MAP_LIMIT: NTSTATUS = -1_070_137_332_i32;
pub const STATUS_VID_HANDLER_NOT_PRESENT: NTSTATUS = -1_070_137_340_i32;
pub const STATUS_VID_INSUFFICIENT_RESOURCES_HV_DEPOSIT: NTSTATUS = -1_070_137_299_i32;
pub const STATUS_VID_INSUFFICIENT_RESOURCES_PHYSICAL_BUFFER: NTSTATUS = -1_070_137_300_i32;
pub const STATUS_VID_INSUFFICIENT_RESOURCES_RESERVE: NTSTATUS = -1_070_137_301_i32;
pub const STATUS_VID_INSUFFICIENT_RESOURCES_WITHDRAW: NTSTATUS = -1_070_137_297_i32;
pub const STATUS_VID_INVALID_CHILD_GPA_PAGE_SET: NTSTATUS = -1_070_137_310_i32;
pub const STATUS_VID_INVALID_GPA_RANGE_HANDLE: NTSTATUS = -1_070_137_323_i32;
pub const STATUS_VID_INVALID_MEMORY_BLOCK_HANDLE: NTSTATUS = -1_070_137_326_i32;
pub const STATUS_VID_INVALID_MESSAGE_QUEUE_HANDLE: NTSTATUS = -1_070_137_324_i32;
pub const STATUS_VID_INVALID_NUMA_NODE_INDEX: NTSTATUS = -1_070_137_328_i32;
pub const STATUS_VID_INVALID_NUMA_SETTINGS: NTSTATUS = -1_070_137_329_i32;
pub const STATUS_VID_INVALID_OBJECT_NAME: NTSTATUS = -1_070_137_339_i32;
pub const STATUS_VID_INVALID_PPM_HANDLE: NTSTATUS = -1_070_137_320_i32;
pub const STATUS_VID_INVALID_PROCESSOR_STATE: NTSTATUS = -1_070_137_315_i32;
pub const STATUS_VID_KM_INTERFACE_ALREADY_INITIALIZED: NTSTATUS = -1_070_137_313_i32;
pub const STATUS_VID_MBPS_ARE_LOCKED: NTSTATUS = -1_070_137_319_i32;
pub const STATUS_VID_MBP_ALREADY_LOCKED_USING_RESERVED_PAGE: NTSTATUS = -1_070_137_307_i32;
pub const STATUS_VID_MBP_COUNT_EXCEEDED_LIMIT: NTSTATUS = -1_070_137_306_i32;
pub const STATUS_VID_MB_PROPERTY_ALREADY_SET_RESET: NTSTATUS = -1_070_137_312_i32;
pub const STATUS_VID_MB_STILL_REFERENCED: NTSTATUS = -1_070_137_331_i32;
pub const STATUS_VID_MEMORY_BLOCK_LOCK_COUNT_EXCEEDED: NTSTATUS = -1_070_137_321_i32;
pub const STATUS_VID_MEMORY_TYPE_NOT_SUPPORTED: NTSTATUS = -1_070_137_298_i32;
pub const STATUS_VID_MESSAGE_QUEUE_ALREADY_EXISTS: NTSTATUS = -1_070_137_333_i32;
pub const STATUS_VID_MESSAGE_QUEUE_CLOSED: NTSTATUS = -1_070_137_318_i32;
pub const STATUS_VID_MESSAGE_QUEUE_NAME_TOO_LONG: NTSTATUS = -1_070_137_337_i32;
pub const STATUS_VID_MMIO_RANGE_DESTROYED: NTSTATUS = -1_070_137_311_i32;
pub const STATUS_VID_NOTIFICATION_QUEUE_ALREADY_ASSOCIATED: NTSTATUS = -1_070_137_327_i32;
pub const STATUS_VID_NO_MEMORY_BLOCK_NOTIFICATION_QUEUE: NTSTATUS = -1_070_137_322_i32;
pub const STATUS_VID_PAGE_RANGE_OVERFLOW: NTSTATUS = -1_070_137_325_i32;
pub const STATUS_VID_PARTITION_ALREADY_EXISTS: NTSTATUS = -1_070_137_336_i32;
pub const STATUS_VID_PARTITION_DOES_NOT_EXIST: NTSTATUS = -1_070_137_335_i32;
pub const STATUS_VID_PARTITION_NAME_NOT_FOUND: NTSTATUS = -1_070_137_334_i32;
pub const STATUS_VID_PARTITION_NAME_TOO_LONG: NTSTATUS = -1_070_137_338_i32;
pub const STATUS_VID_PROCESS_ALREADY_SET: NTSTATUS = -1_070_137_296_i32;
pub const STATUS_VID_QUEUE_FULL: NTSTATUS = -1_070_137_341_i32;
pub const STATUS_VID_REMOTE_NODE_PARENT_GPA_PAGES_USED: NTSTATUS = -2_143_879_167_i32;
pub const STATUS_VID_RESERVE_PAGE_SET_IS_BEING_USED: NTSTATUS = -1_070_137_309_i32;
pub const STATUS_VID_RESERVE_PAGE_SET_TOO_SMALL: NTSTATUS = -1_070_137_308_i32;
pub const STATUS_VID_SAVED_STATE_CORRUPT: NTSTATUS = -1_070_137_305_i32;
pub const STATUS_VID_SAVED_STATE_INCOMPATIBLE: NTSTATUS = -1_070_137_303_i32;
pub const STATUS_VID_SAVED_STATE_UNRECOGNIZED_ITEM: NTSTATUS = -1_070_137_304_i32;
pub const STATUS_VID_STOP_PENDING: NTSTATUS = -1_070_137_316_i32;
pub const STATUS_VID_TOO_MANY_HANDLERS: NTSTATUS = -1_070_137_342_i32;
pub const STATUS_VID_VIRTUAL_PROCESSOR_LIMIT_EXCEEDED: NTSTATUS = -1_070_137_317_i32;
pub const STATUS_VID_VTL_ACCESS_DENIED: NTSTATUS = -1_070_137_302_i32;
pub const STATUS_VIRTDISK_DISK_ALREADY_OWNED: NTSTATUS = -1_069_940_706_i32;
pub const STATUS_VIRTDISK_DISK_ONLINE_AND_WRITABLE: NTSTATUS = -1_069_940_705_i32;
pub const STATUS_VIRTDISK_NOT_VIRTUAL_DISK: NTSTATUS = -1_069_940_715_i32;
pub const STATUS_VIRTDISK_PROVIDER_NOT_FOUND: NTSTATUS = -1_069_940_716_i32;
pub const STATUS_VIRTDISK_UNSUPPORTED_DISK_SECTOR_SIZE: NTSTATUS = -1_069_940_707_i32;
pub const STATUS_VIRTUAL_CIRCUIT_CLOSED: NTSTATUS = -1_073_741_610_i32;
pub const STATUS_VIRTUAL_DISK_LIMITATION: NTSTATUS = -1_069_940_710_i32;
pub const STATUS_VIRUS_DELETED: NTSTATUS = -1_073_739_513_i32;
pub const STATUS_VIRUS_INFECTED: NTSTATUS = -1_073_739_514_i32;
pub const STATUS_VOLMGR_ALL_DISKS_FAILED: NTSTATUS = -1_070_071_767_i32;
pub const STATUS_VOLMGR_BAD_BOOT_DISK: NTSTATUS = -1_070_071_729_i32;
pub const STATUS_VOLMGR_DATABASE_FULL: NTSTATUS = -1_070_071_807_i32;
pub const STATUS_VOLMGR_DIFFERENT_SECTOR_SIZE: NTSTATUS = -1_070_071_730_i32;
pub const STATUS_VOLMGR_DISK_CONFIGURATION_CORRUPTED: NTSTATUS = -1_070_071_806_i32;
pub const STATUS_VOLMGR_DISK_CONFIGURATION_NOT_IN_SYNC: NTSTATUS = -1_070_071_805_i32;
pub const STATUS_VOLMGR_DISK_CONTAINS_NON_SIMPLE_VOLUME: NTSTATUS = -1_070_071_803_i32;
pub const STATUS_VOLMGR_DISK_DUPLICATE: NTSTATUS = -1_070_071_802_i32;
pub const STATUS_VOLMGR_DISK_DYNAMIC: NTSTATUS = -1_070_071_801_i32;
pub const STATUS_VOLMGR_DISK_ID_INVALID: NTSTATUS = -1_070_071_800_i32;
pub const STATUS_VOLMGR_DISK_INVALID: NTSTATUS = -1_070_071_799_i32;
pub const STATUS_VOLMGR_DISK_LAST_VOTER: NTSTATUS = -1_070_071_798_i32;
pub const STATUS_VOLMGR_DISK_LAYOUT_INVALID: NTSTATUS = -1_070_071_797_i32;
pub const STATUS_VOLMGR_DISK_LAYOUT_NON_BASIC_BETWEEN_BASIC_PARTITIONS: NTSTATUS =
    -1_070_071_796_i32;
pub const STATUS_VOLMGR_DISK_LAYOUT_NOT_CYLINDER_ALIGNED: NTSTATUS = -1_070_071_795_i32;
pub const STATUS_VOLMGR_DISK_LAYOUT_PARTITIONS_TOO_SMALL: NTSTATUS = -1_070_071_794_i32;
pub const STATUS_VOLMGR_DISK_LAYOUT_PRIMARY_BETWEEN_LOGICAL_PARTITIONS: NTSTATUS =
    -1_070_071_793_i32;
pub const STATUS_VOLMGR_DISK_LAYOUT_TOO_MANY_PARTITIONS: NTSTATUS = -1_070_071_792_i32;
pub const STATUS_VOLMGR_DISK_MISSING: NTSTATUS = -1_070_071_791_i32;
pub const STATUS_VOLMGR_DISK_NOT_EMPTY: NTSTATUS = -1_070_071_790_i32;
pub const STATUS_VOLMGR_DISK_NOT_ENOUGH_SPACE: NTSTATUS = -1_070_071_789_i32;
pub const STATUS_VOLMGR_DISK_REVECTORING_FAILED: NTSTATUS = -1_070_071_788_i32;
pub const STATUS_VOLMGR_DISK_SECTOR_SIZE_INVALID: NTSTATUS = -1_070_071_787_i32;
pub const STATUS_VOLMGR_DISK_SET_NOT_CONTAINED: NTSTATUS = -1_070_071_786_i32;
pub const STATUS_VOLMGR_DISK_USED_BY_MULTIPLE_MEMBERS: NTSTATUS = -1_070_071_785_i32;
pub const STATUS_VOLMGR_DISK_USED_BY_MULTIPLE_PLEXES: NTSTATUS = -1_070_071_784_i32;
pub const STATUS_VOLMGR_DYNAMIC_DISK_NOT_SUPPORTED: NTSTATUS = -1_070_071_783_i32;
pub const STATUS_VOLMGR_EXTENT_ALREADY_USED: NTSTATUS = -1_070_071_782_i32;
pub const STATUS_VOLMGR_EXTENT_NOT_CONTIGUOUS: NTSTATUS = -1_070_071_781_i32;
pub const STATUS_VOLMGR_EXTENT_NOT_IN_PUBLIC_REGION: NTSTATUS = -1_070_071_780_i32;
pub const STATUS_VOLMGR_EXTENT_NOT_SECTOR_ALIGNED: NTSTATUS = -1_070_071_779_i32;
pub const STATUS_VOLMGR_EXTENT_OVERLAPS_EBR_PARTITION: NTSTATUS = -1_070_071_778_i32;
pub const STATUS_VOLMGR_EXTENT_VOLUME_LENGTHS_DO_NOT_MATCH: NTSTATUS = -1_070_071_777_i32;
pub const STATUS_VOLMGR_FAULT_TOLERANT_NOT_SUPPORTED: NTSTATUS = -1_070_071_776_i32;
pub const STATUS_VOLMGR_INCOMPLETE_DISK_MIGRATION: NTSTATUS = -2_143_813_630_i32;
pub const STATUS_VOLMGR_INCOMPLETE_REGENERATION: NTSTATUS = -2_143_813_631_i32;
pub const STATUS_VOLMGR_INTERLEAVE_LENGTH_INVALID: NTSTATUS = -1_070_071_775_i32;
pub const STATUS_VOLMGR_MAXIMUM_REGISTERED_USERS: NTSTATUS = -1_070_071_774_i32;
pub const STATUS_VOLMGR_MEMBER_INDEX_DUPLICATE: NTSTATUS = -1_070_071_772_i32;
pub const STATUS_VOLMGR_MEMBER_INDEX_INVALID: NTSTATUS = -1_070_071_771_i32;
pub const STATUS_VOLMGR_MEMBER_IN_SYNC: NTSTATUS = -1_070_071_773_i32;
pub const STATUS_VOLMGR_MEMBER_MISSING: NTSTATUS = -1_070_071_770_i32;
pub const STATUS_VOLMGR_MEMBER_NOT_DETACHED: NTSTATUS = -1_070_071_769_i32;
pub const STATUS_VOLMGR_MEMBER_REGENERATING: NTSTATUS = -1_070_071_768_i32;
pub const STATUS_VOLMGR_MIRROR_NOT_SUPPORTED: NTSTATUS = -1_070_071_717_i32;
pub const STATUS_VOLMGR_NOTIFICATION_RESET: NTSTATUS = -1_070_071_764_i32;
pub const STATUS_VOLMGR_NOT_PRIMARY_PACK: NTSTATUS = -1_070_071_726_i32;
pub const STATUS_VOLMGR_NO_REGISTERED_USERS: NTSTATUS = -1_070_071_766_i32;
pub const STATUS_VOLMGR_NO_SUCH_USER: NTSTATUS = -1_070_071_765_i32;
pub const STATUS_VOLMGR_NO_VALID_LOG_COPIES: NTSTATUS = -1_070_071_720_i32;
pub const STATUS_VOLMGR_NUMBER_OF_DISKS_INVALID: NTSTATUS = -1_070_071_718_i32;
pub const STATUS_VOLMGR_NUMBER_OF_DISKS_IN_MEMBER_INVALID: NTSTATUS = -1_070_071_723_i32;
pub const STATUS_VOLMGR_NUMBER_OF_DISKS_IN_PLEX_INVALID: NTSTATUS = -1_070_071_724_i32;
pub const STATUS_VOLMGR_NUMBER_OF_EXTENTS_INVALID: NTSTATUS = -1_070_071_731_i32;
pub const STATUS_VOLMGR_NUMBER_OF_MEMBERS_INVALID: NTSTATUS = -1_070_071_763_i32;
pub const STATUS_VOLMGR_NUMBER_OF_PLEXES_INVALID: NTSTATUS = -1_070_071_762_i32;
pub const STATUS_VOLMGR_PACK_CONFIG_OFFLINE: NTSTATUS = -1_070_071_728_i32;
pub const STATUS_VOLMGR_PACK_CONFIG_ONLINE: NTSTATUS = -1_070_071_727_i32;
pub const STATUS_VOLMGR_PACK_CONFIG_UPDATE_FAILED: NTSTATUS = -1_070_071_804_i32;
pub const STATUS_VOLMGR_PACK_DUPLICATE: NTSTATUS = -1_070_071_761_i32;
pub const STATUS_VOLMGR_PACK_HAS_QUORUM: NTSTATUS = -1_070_071_756_i32;
pub const STATUS_VOLMGR_PACK_ID_INVALID: NTSTATUS = -1_070_071_760_i32;
pub const STATUS_VOLMGR_PACK_INVALID: NTSTATUS = -1_070_071_759_i32;
pub const STATUS_VOLMGR_PACK_LOG_UPDATE_FAILED: NTSTATUS = -1_070_071_725_i32;
pub const STATUS_VOLMGR_PACK_NAME_INVALID: NTSTATUS = -1_070_071_758_i32;
pub const STATUS_VOLMGR_PACK_OFFLINE: NTSTATUS = -1_070_071_757_i32;
pub const STATUS_VOLMGR_PACK_WITHOUT_QUORUM: NTSTATUS = -1_070_071_755_i32;
pub const STATUS_VOLMGR_PARTITION_STYLE_INVALID: NTSTATUS = -1_070_071_754_i32;
pub const STATUS_VOLMGR_PARTITION_UPDATE_FAILED: NTSTATUS = -1_070_071_753_i32;
pub const STATUS_VOLMGR_PLEX_INDEX_DUPLICATE: NTSTATUS = -1_070_071_751_i32;
pub const STATUS_VOLMGR_PLEX_INDEX_INVALID: NTSTATUS = -1_070_071_750_i32;
pub const STATUS_VOLMGR_PLEX_IN_SYNC: NTSTATUS = -1_070_071_752_i32;
pub const STATUS_VOLMGR_PLEX_LAST_ACTIVE: NTSTATUS = -1_070_071_749_i32;
pub const STATUS_VOLMGR_PLEX_MISSING: NTSTATUS = -1_070_071_748_i32;
pub const STATUS_VOLMGR_PLEX_NOT_RAID5: NTSTATUS = -1_070_071_745_i32;
pub const STATUS_VOLMGR_PLEX_NOT_SIMPLE: NTSTATUS = -1_070_071_744_i32;
pub const STATUS_VOLMGR_PLEX_NOT_SIMPLE_SPANNED: NTSTATUS = -1_070_071_721_i32;
pub const STATUS_VOLMGR_PLEX_REGENERATING: NTSTATUS = -1_070_071_747_i32;
pub const STATUS_VOLMGR_PLEX_TYPE_INVALID: NTSTATUS = -1_070_071_746_i32;
pub const STATUS_VOLMGR_PRIMARY_PACK_PRESENT: NTSTATUS = -1_070_071_719_i32;
pub const STATUS_VOLMGR_RAID5_NOT_SUPPORTED: NTSTATUS = -1_070_071_716_i32;
pub const STATUS_VOLMGR_STRUCTURE_SIZE_INVALID: NTSTATUS = -1_070_071_743_i32;
pub const STATUS_VOLMGR_TOO_MANY_NOTIFICATION_REQUESTS: NTSTATUS = -1_070_071_742_i32;
pub const STATUS_VOLMGR_TRANSACTION_IN_PROGRESS: NTSTATUS = -1_070_071_741_i32;
pub const STATUS_VOLMGR_UNEXPECTED_DISK_LAYOUT_CHANGE: NTSTATUS = -1_070_071_740_i32;
pub const STATUS_VOLMGR_VOLUME_CONTAINS_MISSING_DISK: NTSTATUS = -1_070_071_739_i32;
pub const STATUS_VOLMGR_VOLUME_ID_INVALID: NTSTATUS = -1_070_071_738_i32;
pub const STATUS_VOLMGR_VOLUME_LENGTH_INVALID: NTSTATUS = -1_070_071_737_i32;
pub const STATUS_VOLMGR_VOLUME_LENGTH_NOT_SECTOR_SIZE_MULTIPLE: NTSTATUS = -1_070_071_736_i32;
pub const STATUS_VOLMGR_VOLUME_MIRRORED: NTSTATUS = -1_070_071_722_i32;
pub const STATUS_VOLMGR_VOLUME_NOT_MIRRORED: NTSTATUS = -1_070_071_735_i32;
pub const STATUS_VOLMGR_VOLUME_NOT_RETAINED: NTSTATUS = -1_070_071_734_i32;
pub const STATUS_VOLMGR_VOLUME_OFFLINE: NTSTATUS = -1_070_071_733_i32;
pub const STATUS_VOLMGR_VOLUME_RETAINED: NTSTATUS = -1_070_071_732_i32;
pub const STATUS_VOLSNAP_ACTIVATION_TIMEOUT: NTSTATUS = -1_068_498_940_i32;
pub const STATUS_VOLSNAP_BOOTFILE_NOT_VALID: NTSTATUS = -1_068_498_941_i32;
pub const STATUS_VOLSNAP_HIBERNATE_READY: NTSTATUS = 293_i32;
pub const STATUS_VOLSNAP_NO_BYPASSIO_WITH_SNAPSHOT: NTSTATUS = -1_068_498_939_i32;
pub const STATUS_VOLSNAP_PREPARE_HIBERNATE: NTSTATUS = -1_073_740_793_i32;
pub const STATUS_VOLUME_DIRTY: NTSTATUS = -1_073_739_770_i32;
pub const STATUS_VOLUME_DISMOUNTED: NTSTATUS = -1_073_741_202_i32;
pub const STATUS_VOLUME_MOUNTED: NTSTATUS = 265_i32;
pub const STATUS_VOLUME_NOT_CLUSTER_ALIGNED: NTSTATUS = -1_073_740_636_i32;
pub const STATUS_VOLUME_NOT_SUPPORTED: NTSTATUS = -1_073_740_602_i32;
pub const STATUS_VOLUME_NOT_UPGRADED: NTSTATUS = -1_073_741_156_i32;
pub const STATUS_VOLUME_WRITE_ACCESS_DENIED: NTSTATUS = -1_073_740_589_i32;
pub const STATUS_VRF_VOLATILE_CFG_AND_IO_ENABLED: NTSTATUS = -1_073_738_744_i32;
pub const STATUS_VRF_VOLATILE_NMI_REGISTERED: NTSTATUS = -1_073_738_738_i32;
pub const STATUS_VRF_VOLATILE_NOT_RUNNABLE_SYSTEM: NTSTATUS = -1_073_738_741_i32;
pub const STATUS_VRF_VOLATILE_NOT_STOPPABLE: NTSTATUS = -1_073_738_743_i32;
pub const STATUS_VRF_VOLATILE_NOT_SUPPORTED_RULECLASS: NTSTATUS = -1_073_738_740_i32;
pub const STATUS_VRF_VOLATILE_PROTECTED_DRIVER: NTSTATUS = -1_073_738_739_i32;
pub const STATUS_VRF_VOLATILE_SAFE_MODE: NTSTATUS = -1_073_738_742_i32;
pub const STATUS_VRF_VOLATILE_SETTINGS_CONFLICT: NTSTATUS = -1_073_738_737_i32;
pub const STATUS_VSM_DMA_PROTECTION_NOT_IN_USE: NTSTATUS = -1_069_219_839_i32;
pub const STATUS_VSM_NOT_INITIALIZED: NTSTATUS = -1_069_219_840_i32;
pub const STATUS_WAIT_0: NTSTATUS = 0_i32;
pub const STATUS_WAIT_1: NTSTATUS = 1_i32;
pub const STATUS_WAIT_2: NTSTATUS = 2_i32;
pub const STATUS_WAIT_3: NTSTATUS = 3_i32;
pub const STATUS_WAIT_63: NTSTATUS = 63_i32;
pub const STATUS_WAIT_FOR_OPLOCK: NTSTATUS = 871_i32;
pub const STATUS_WAKE_SYSTEM: NTSTATUS = 1_073_742_484_i32;
pub const STATUS_WAKE_SYSTEM_DEBUGGER: NTSTATUS = -2_147_483_641_i32;
pub const STATUS_WAS_LOCKED: NTSTATUS = 1_073_741_849_i32;
pub const STATUS_WAS_UNLOCKED: NTSTATUS = 1_073_741_847_i32;
pub const STATUS_WEAK_WHFBKEY_BLOCKED: NTSTATUS = -1_073_741_389_i32;
pub const STATUS_WIM_NOT_BOOTABLE: NTSTATUS = -1_073_740_665_i32;
pub const STATUS_WMI_ALREADY_DISABLED: NTSTATUS = -1_073_741_054_i32;
pub const STATUS_WMI_ALREADY_ENABLED: NTSTATUS = -1_073_741_053_i32;
pub const STATUS_WMI_GUID_DISCONNECTED: NTSTATUS = -1_073_741_055_i32;
pub const STATUS_WMI_GUID_NOT_FOUND: NTSTATUS = -1_073_741_163_i32;
pub const STATUS_WMI_INSTANCE_NOT_FOUND: NTSTATUS = -1_073_741_162_i32;
pub const STATUS_WMI_ITEMID_NOT_FOUND: NTSTATUS = -1_073_741_161_i32;
pub const STATUS_WMI_NOT_SUPPORTED: NTSTATUS = -1_073_741_091_i32;
pub const STATUS_WMI_READ_ONLY: NTSTATUS = -1_073_741_114_i32;
pub const STATUS_WMI_SET_FAILURE: NTSTATUS = -1_073_741_113_i32;
pub const STATUS_WMI_TRY_AGAIN: NTSTATUS = -1_073_741_160_i32;
pub const STATUS_WOF_FILE_RESOURCE_TABLE_CORRUPT: NTSTATUS = -1_073_700_185_i32;
pub const STATUS_WOF_WIM_HEADER_CORRUPT: NTSTATUS = -1_073_700_187_i32;
pub const STATUS_WOF_WIM_RESOURCE_TABLE_CORRUPT: NTSTATUS = -1_073_700_186_i32;
pub const STATUS_WORKING_SET_LIMIT_RANGE: NTSTATUS = 1_073_741_826_i32;
pub const STATUS_WORKING_SET_QUOTA: NTSTATUS = -1_073_741_663_i32;
pub const STATUS_WOW_ASSERTION: NTSTATUS = -1_073_702_760_i32;
pub const STATUS_WRONG_COMPARTMENT: NTSTATUS = -1_073_700_731_i32;
pub const STATUS_WRONG_CREDENTIAL_HANDLE: NTSTATUS = -1_073_741_070_i32;
pub const STATUS_WRONG_EFS: NTSTATUS = -1_073_741_169_i32;
pub const STATUS_WRONG_PASSWORD_CORE: NTSTATUS = -1_073_741_495_i32;
pub const STATUS_WRONG_VOLUME: NTSTATUS = -1_073_741_806_i32;
pub const STATUS_WX86_BREAKPOINT: NTSTATUS = 1_073_741_855_i32;
pub const STATUS_WX86_CONTINUE: NTSTATUS = 1_073_741_853_i32;
pub const STATUS_WX86_CREATEWX86TIB: NTSTATUS = 1_073_741_864_i32;
pub const STATUS_WX86_EXCEPTION_CHAIN: NTSTATUS = 1_073_741_858_i32;
pub const STATUS_WX86_EXCEPTION_CONTINUE: NTSTATUS = 1_073_741_856_i32;
pub const STATUS_WX86_EXCEPTION_LASTCHANCE: NTSTATUS = 1_073_741_857_i32;
pub const STATUS_WX86_FLOAT_STACK_CHECK: NTSTATUS = -1_073_741_200_i32;
pub const STATUS_WX86_INTERNAL_ERROR: NTSTATUS = -1_073_741_201_i32;
pub const STATUS_WX86_SINGLE_STEP: NTSTATUS = 1_073_741_854_i32;
pub const STATUS_WX86_UNSIMULATE: NTSTATUS = 1_073_741_852_i32;
pub const STATUS_XMLDSIG_ERROR: NTSTATUS = -1_073_700_732_i32;
pub const STATUS_XML_ENCODING_MISMATCH: NTSTATUS = -1_072_365_535_i32;
pub const STATUS_XML_PARSE_ERROR: NTSTATUS = -1_073_700_733_i32;
