//
//  Copyright (c) Microsoft Corporation. All rights reserved.
//  Licensed under the MIT License. See License.txt in the project root for license information.
//

using System;
using AppKit;

namespace Microsoft.VisualStudio.Text.Editor
{
    /// <summary>
    /// A native <see cref="NSEvent"/> key input event that allows for indicating that
    /// the event has been handled. This event will wrap <see cref="NSEventType.KeyUp"/>,
    /// <see cref="NSEventType.KeyDown"/>, and  <see cref="NSEventType.FlagsChanged"/>
    /// events. 
    /// </summary>
    public sealed class KeyEventArgs : InputEventArgs
    {
        /// <summary>
        /// The characters associated with a key-up or key-down event.
        /// </summary>
        public string Characters => Event.Characters;

        /// <summary>
        /// The characters generated by a key event as if no modifier key (except for Shift) applies.
        /// </summary>
        public string CharactersIgnoringModifiers => Event.CharactersIgnoringModifiers;

        internal KeyEventArgs(NSEvent @event) : base(@event)
        {
            switch (@event.Type)
            {
                case NSEventType.KeyUp:
                case NSEventType.KeyDown:
                case NSEventType.FlagsChanged:
                    break;
                default:
                    throw new ArgumentException(
                        "event type must be KeyUp, KeyDown, or FlagsChanged",
                        nameof(@event));
            }
        }
    }
}