//
//  Copyright (c) Microsoft Corporation. All rights reserved.
//  Licensed under the MIT License. See License.txt in the project root for license information.
//
namespace Microsoft.VisualStudio.Text.Classification
{
    using System;
    using System.Collections.Generic;

    /// <summary>
    /// The service that maintains the collection of all known classification types.
    /// </summary>
    /// <remarks>This is a MEF component part, and should be imported as follows:
    /// [Import]
    /// IClassificationTypeRegistryService registry = null;
    /// </remarks>
    public interface IClassificationTypeRegistryService
    {
        /// <summary>
        /// Gets the <see cref="IClassificationType"></see> object identified by the specified <paramref name="type"/>.
        /// </summary>
        /// <param name="type">
        /// The name of the classification type.
        /// </param>
        /// <returns>
        /// The classification type, <c>null</c> if there is no classification type of that name.
        /// </returns>
        IClassificationType GetClassificationType(string type);

        /// <summary>
        /// Initializes a new instance of a <see cref="IClassificationType"/> and adds it to the registry.
        /// </summary>
        /// <param name="type">The name of the classification type to create.</param>
        /// <param name="baseTypes">The base types of the classification.</param>
        /// <returns>A new <see cref="IClassificationType"/>.</returns>
        /// <exception cref="InvalidOperationException"><paramref name="type"/> is already in the registry.</exception>
        IClassificationType CreateClassificationType(string type, IEnumerable<IClassificationType> baseTypes);

        /// <summary>
        /// Creates an <see cref="IClassificationType"/> that persists only for the duration of
        /// this session. This <see cref="IClassificationType"/> must inherit from at least one
        /// <see cref="IClassificationType"/>.
        /// </summary>
        /// <param name="baseTypes">
        /// The base types for this <see cref="IClassificationType"/>.
        /// </param>
        /// <returns>
        /// A new <see cref="IClassificationType"/> that inherits from all of <paramref name="baseTypes"/>.
        /// </returns>
        /// <exception cref="ArgumentNullException"><paramref name="baseTypes"/> is null.</exception>
        /// <exception cref="InvalidOperationException"><paramref name="baseTypes"/> has zero items.</exception>
        /// <remarks>
        /// <para>
        /// This function is intended primarily to aid in the runtime display of overlapping classifications.
        /// </para>
        /// <para>
        /// The classification names generated by this function are determined at run time and are subject to
        /// change in future revisions. The only guarantee made is that if two transient <see cref="IClassificationType"/> objects
        /// are created with the same base types, they will have the same classification name.
        /// </para>
        /// </remarks>
        IClassificationType CreateTransientClassificationType(IEnumerable<IClassificationType> baseTypes);

        /// <summary>
        /// Creates an <see cref="IClassificationType"/> that persists only for the duration of
        /// this session. This <see cref="IClassificationType"/> must inherit from at least one
        /// <see cref="IClassificationType"/>.
        /// </summary>
        /// <param name="baseTypes">
        /// The base types for this <see cref="IClassificationType"/>.
        /// </param>
        /// <returns>
        /// A new <see cref="IClassificationType"/> which inherits from all <paramref name="baseTypes"/>.
        /// </returns>
        /// <exception cref="ArgumentNullException"><paramref name="baseTypes"/> is null.</exception>
        /// <exception cref="InvalidOperationException"><paramref name="baseTypes"/> has zero items.</exception>
        /// <remarks>
        /// <para>
        /// This function is intended primarily to aid in the runtime display of overlapping classifications.
        /// </para>
        /// 
        /// <para>
        /// The classification names generated by this function are determined at run time and are subject to
        /// change in future revisions. The only guarantee made is that if two transient <see cref="IClassificationType"/> objects
        /// are created with the same base types, they will have the same classification name.
        /// </para>
        /// </remarks>
        IClassificationType CreateTransientClassificationType(params IClassificationType[] baseTypes);

        /// <summary>
        /// Gets the <see cref="IClassificationType"></see> object identified by the specified <paramref name="type"/>.
        /// </summary>
        /// <param name="layer">
        /// Specifies the layer to which this classification belongs. Layers are specializations
        /// of <see cref="IClassificationType"/>s that can be used to make all classifications from
        /// one source supersede another.
        /// </param>
        /// <param name="type">
        /// The name of the classification type.
        /// </param>
        /// <returns>
        /// The classification type, <c>null</c> if there is no classification type of that name.
        /// </returns>
        ILayeredClassificationType GetClassificationType(ClassificationLayer layer, string type);

        /// <summary>
        /// Initializes a new instance of a <see cref="IClassificationType"/> and adds it to the registry.
        /// </summary>
        /// <param name="layer">
        /// Specifies the layer to which this classification belongs. Layers are specializations
        /// of <see cref="IClassificationType"/>s that can be used to make all classifications from
        /// one source supersede another.
        /// </param>
        /// <param name="type">The name of the classification type to create.</param>
        /// <param name="baseTypes">The base types of the classification.</param>
        /// <returns>A new <see cref="IClassificationType"/>.</returns>
        /// <exception cref="InvalidOperationException"><paramref name="type"/> is already in the registry.</exception>
        ILayeredClassificationType CreateClassificationType(ClassificationLayer layer, string type, IEnumerable<IClassificationType> baseTypes);
    }
}
