// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

const chai = require('chai');

const { TimexProperty, TimexSet } = require('../index.js');

const assert = chai.assert;
const expect = chai.expect;
chai.should();

describe('No Network', () => {
    describe('Datatypes', () => {
        describe('Timex parsing', () => {
            describe('date', () => {
                it('complete date', () => {
                    const timex = new TimexProperty('2017-05-29');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('definite', 'date');
                    timex.should.have.property('year', 2017);
                    timex.should.have.property('month', 5);
                    timex.should.have.property('dayOfMonth', 29);
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('month and dayOfMonth', () => {
                    const timex = new TimexProperty('XXXX-12-05');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('date');
                    timex.should.not.have.property('year');
                    timex.should.have.property('month', 12);
                    timex.should.have.property('dayOfMonth', 5);
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('dayOfWeek', () => {
                    const timex = new TimexProperty('XXXX-WXX-3');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('date');
                    timex.should.not.have.property('year');
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.have.property('dayOfWeek', 3);
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
            });
            describe('time', () => {
                it('hours, minutes and seconds', () => {
                    const timex = new TimexProperty('T17:30:05');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('time');
                    timex.should.not.have.property('year');
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.have.property('hour', 17);
                    timex.should.have.property('minute', 30);
                    timex.should.have.property('second', 5);
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('hours and minutes', () => {
                    const timex = new TimexProperty('T17:30');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('time');
                    timex.should.not.have.property('year');
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.have.property('hour', 17);
                    timex.should.have.property('minute', 30);
                    timex.should.have.property('second', 0);
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('hours', () => {
                    const timex = new TimexProperty('T17');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('time');
                    timex.should.not.have.property('year');
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.have.property('hour', 17);
                    timex.should.have.property('minute', 0);
                    timex.should.have.property('second', 0);
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
            });
            describe('datetime', () => {
                it('now', () => {
                    const timex = new TimexProperty('PRESENT_REF');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('present', 'datetime', 'date', 'time');
                    timex.should.not.have.property('year');
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.have.property('now', true);
                });
                it('full datetime', () => {
                    const timex = new TimexProperty('1984-01-03T18:30:45');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('definite', 'datetime', 'date', 'time');
                    timex.should.have.property('year', 1984);
                    timex.should.have.property('month', 1);
                    timex.should.have.property('dayOfMonth', 3);
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.have.property('hour', 18);
                    timex.should.have.property('minute', 30);
                    timex.should.have.property('second', 45);
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('paricular time on particular day of week (e.g. "Wednesday 4pm")', () => {
                    const timex = new TimexProperty('XXXX-WXX-3T16');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('datetime', 'date', 'time');
                    timex.should.not.have.property('year');
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.have.property('dayOfWeek', 3);
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.have.property('hour', 16);
                    timex.should.have.property('minute', 0);
                    timex.should.have.property('second', 0);
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
            });
            describe('daterange', () => {
                it('year', () => {
                    const timex = new TimexProperty('2016');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('daterange');
                    timex.should.have.property('year', 2016);
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('year season (e.g. "summer of 1999")', () => {
                    const timex = new TimexProperty('1999-SU');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('daterange');
                    timex.should.have.property('year', 1999);
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.have.property('season', 'SU');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('year and week', () => {
                    const timex = new TimexProperty('2017-W37');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('daterange');
                    timex.should.have.property('year', 2017);
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.have.property('weekOfYear', 37);
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('season (e.g. "summer")', () => {
                    const timex = new TimexProperty('SU');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('daterange');
                    timex.should.not.have.property('year');
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.have.property('season', 'SU');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('season (e.g. "winter")', () => {
                    const timex = new TimexProperty('WI');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('daterange');
                    timex.should.not.have.property('year');
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.have.property('season', 'WI');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('year and weekend (e.g. "this weekend" at least today:-))', () => {
                    const timex = new TimexProperty('2017-W37-WE');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('daterange');
                    timex.should.have.property('year', 2017);
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.have.property('weekOfYear', 37);
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.have.property('weekend', true);
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('month (e.g. "may")', () => {
                    const timex = new TimexProperty('XXXX-05');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('daterange');
                    timex.should.not.have.property('year');
                    timex.should.have.property('month', 5);
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('month (e.g. "July 2020")', () => {
                    const timex = new TimexProperty('2020-07');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('daterange');
                    timex.should.have.property('year', 2020);
                    timex.should.have.property('month', 7);
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('week of month ("first week of january")', () => {
                    const timex = new TimexProperty('XXXX-01-W01');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('daterange');
                    timex.should.not.have.property('year');
                    timex.should.have.property('month', 1);
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.have.property('weekOfMonth', 1);
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('wednesday to saturday', () => {
                    const timex = new TimexProperty('(XXXX-WXX-3,XXXX-WXX-6,P3D)');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('date', 'duration', 'daterange');
                    timex.should.not.have.property('year');
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.have.property('dayOfWeek', 3);
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.have.property('days', 3);
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('1 january to 5 august', () => {
                    const timex = new TimexProperty('(XXXX-01-01,XXXX-08-05,P216D)');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('date', 'duration', 'daterange');
                    timex.should.not.have.property('year');
                    timex.should.have.property('month', 1);
                    timex.should.have.property('dayOfMonth', 1);
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.have.property('days', 216);
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('1 january 2015 to 5 august 2015', () => {
                    const timex = new TimexProperty('(2015-01-01,2015-08-05,P216D)');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('definite', 'date', 'duration', 'daterange');
                    timex.should.have.property('year', 2015);
                    timex.should.have.property('month', 1);
                    timex.should.have.property('dayOfMonth', 1);
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.have.property('days', 216);
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
            });
            describe('TimeRange', () => {
                it('daytime', () => {
                    const timex = new TimexProperty('TDT');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('timerange');
                    timex.should.not.have.property('year');
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.have.property('partOfDay', 'DT');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('night', () => {
                    const timex = new TimexProperty('TNI');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('timerange');
                    timex.should.not.have.property('year');
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.have.property('partOfDay', 'NI');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('morning', () => {
                    const timex = new TimexProperty('TMO');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('timerange');
                    timex.should.not.have.property('year');
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.have.property('partOfDay', 'MO');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('afternoon', () => {
                    const timex = new TimexProperty('TAF');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('timerange');
                    timex.should.not.have.property('year');
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.have.property('partOfDay', 'AF');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('evening', () => {
                    const timex = new TimexProperty('TEV');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('timerange');
                    timex.should.not.have.property('year');
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.have.property('partOfDay', 'EV');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('4:30pm to 4:45pm', () => {
                    // BUG Luis actually incorrectly returns "(T16:30,T16:45,PT0H)" 
                    const timex = new TimexProperty('(T16:30,T16:45,PT15M)');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('time', 'duration', 'timerange');
                    timex.should.not.have.property('year');
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.have.property('hour', 16);
                    timex.should.have.property('minute', 30);
                    timex.should.have.property('second', 0);
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.have.property('minutes', 15);
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
            });
            describe('DateTimeRange', () => {
                it('friday evening', () => {
                    const timex = new TimexProperty('XXXX-WXX-5TEV');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('date', 'timerange', 'datetimerange');
                    timex.should.not.have.property('year');
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.have.property('dayOfWeek', 5);
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.have.property('partOfDay', 'EV');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('last night', () => {
                    const timex = new TimexProperty('2017-09-07TNI');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('definite', 'date', 'timerange', 'datetimerange');
                    timex.should.have.property('year', 2017);
                    timex.should.have.property('month', 9);
                    timex.should.have.property('dayOfMonth', 7);
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.have.property('partOfDay', 'NI');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('last 5 minutes', () => {
                    const timex = new TimexProperty('(2017-09-08T21:19:29,2017-09-08T21:24:29,PT5M)');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('date', 'time', 'datetime', 'duration', 'datetimerange', 'daterange', 'definite', 'timerange');
                    timex.should.have.property('year', 2017);
                    timex.should.have.property('month', 9);
                    timex.should.have.property('dayOfMonth', 8);
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.have.property('hour', 21);
                    timex.should.have.property('minute', 19);
                    timex.should.have.property('second', 29);
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.have.property('minutes', 5);
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('4pm wednesday to 3pm saturday', () => {
                    const timex = new TimexProperty('(XXXX-WXX-3T16,XXXX-WXX-6T15,PT71H)');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('date', 'time', 'datetime', 'daterange', 'timerange', 'datetimerange', 'duration');
                    timex.should.not.have.property('year');
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.have.property('dayOfWeek', 3);
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.have.property('hour', 16);
                    timex.should.have.property('minute', 0);
                    timex.should.have.property('second', 0);
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.have.property('hours', 71);
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
            });
            describe('Duration', () => {
                it('years', () => {
                    const timex = new TimexProperty('P2Y');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('duration');
                    timex.should.not.have.property('year');
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.have.property('years', 2);
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('months', () => {
                    const timex = new TimexProperty('P4M');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('duration');
                    timex.should.not.have.property('year');
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.have.property('months', 4);
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('weeks', () => {
                    const timex = new TimexProperty('P6W');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('duration');
                    timex.should.not.have.property('year');
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.have.property('weeks', 6);
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('weeks (floating point)', () => {
                    const timex = new TimexProperty('P2.5W');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('duration');
                    timex.should.not.have.property('year');
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.have.property('weeks', 2.5);
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('days', () => {
                    const timex = new TimexProperty('P1D');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('duration');
                    timex.should.not.have.property('year');
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.have.property('days', 1);
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('hours', () => {
                    const timex = new TimexProperty('PT5H');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('duration');
                    timex.should.not.have.property('year');
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.have.property('hours', 5);
                    timex.should.not.have.property('minutes');
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('minutes', () => {
                    const timex = new TimexProperty('PT30M');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('duration');
                    timex.should.not.have.property('year');
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.have.property('minutes', 30);
                    timex.should.not.have.property('seconds');
                    timex.should.not.have.property('now');
                });
                it('seconds', () => {
                    const timex = new TimexProperty('PT45S');
                    timex.should.have.property('types').that.is.a('Set').that.has.keys('duration');
                    timex.should.not.have.property('year');
                    timex.should.not.have.property('month');
                    timex.should.not.have.property('dayOfMonth');
                    timex.should.not.have.property('dayOfWeek');
                    timex.should.not.have.property('weekOfYear');
                    timex.should.not.have.property('weekOfMonth');
                    timex.should.not.have.property('season');
                    timex.should.not.have.property('hour');
                    timex.should.not.have.property('minute');
                    timex.should.not.have.property('second');
                    timex.should.not.have.property('weekend');
                    timex.should.not.have.property('partOfDay');
                    timex.should.not.have.property('years');
                    timex.should.not.have.property('months');
                    timex.should.not.have.property('weeks');
                    timex.should.not.have.property('days');
                    timex.should.not.have.property('hours');
                    timex.should.not.have.property('minutes');
                    timex.should.have.property('seconds', 45);
                    timex.should.not.have.property('now');
                });
            });
        });
    });
});
