/*
 * File:   main.c
 * Author: M91406
 *
 * Created on July 8, 2019, 1:52 PM
 */

#include <xc.h> // include processor files - each processor file is guarded.  
#include <stdint.h> // include standard integer data types
#include <stdbool.h> // include standard boolean data types

#include "main.h" // include main header
#include "config/apps.h"

/*******************************************************************************
 * @fn     int main(void)
 * @ingroup firmware-flow
 * @brief  Application main function executed after device comes out of RESET
 * @return Signed Integer (0=failure, 1=success)
 * @details
 * This function is the starting point of the firmware. It is called after the
 * device is coming out of RESET, starting to execute code. The startup sequence 
 * is as follows:
 * 
 * 1) Boot():
 * 
 * Configures fundamental system components 
 * 
 * Fundamental components are peripherals which are essential for the CPU to run,
 * such as
 * 
 * - system oscillator
 * - auxiliary oscillator
 * - general purpose input/output configuration
 * - digital signal processor configuration
 *      
 * 2) sysUserPeriperhals_Initialize()
 * 
 * User-defined, static peripheral configurations (optional)
 * 
 * Specific peripheral modules required to support circuit functions
 * of the hardware, which are not covered by any task library, such as
 * 
 * - Digital-To-Analog Converter (DAC):
 *   This peripheral instance is used to provide an analog debugging signal which 
 *   can be used to output internal high-speed signals/values on an oscilloscope.
 * 
 * - Operational Amplifier (OPA)
 *   The analog debugging signal generated by the DAC is buffered by one of the 
 *   internal operational amplifiers to stabilize the signal when probed.
 *          
 * 3) rtos_Execute()
 * 
 * This application executes five tasks simultaneously:
 * 
 * - High Priority Tasks
 *     - Power Supply State Machine & Control 
 *     - System Fault Monitor & Diagnostics
 * 
 * - Low Priority Tasks
 *     - LCD Display Control & Update
 *     - On-Board Push-Button HMI Control
 *     - On-Board Debugging LED Control
 * 
 * The operating system supports four individual task queues:
 * 
 * - Queue Initialize
 * - Queue Startup Sequence
 * - Queue Execute Sequence
 * - Queue Stop Sequence
 * 
 * Each 'Task' is an independent process which is called at integer multiples of
 * the execution frequency. The default frequency (OS base clock) set in this code 
 * version is 100 us (= 10 kHz).
 * 
 * At device startup the OS will execute all tasks listed in Queue Initialize 
 * where all initialization functions of all listed user tasks will be executed
 * in one sequence without timing control. Each of the listed tasks must make 
 * sure all required device resources are configured within its individual 
 * initialization routine.
 * 
 * After all initialization functions have been executed, the OS executes all
 * tasks of Queue Startup Sequence. This intermediary step allows the alignment
 * and deterministic startup behavior of all listed user tasks.
 * 
 * After startup the OS is continuously executing Queue Execute. In each 
 * execution cycle the task execution timing settings Period and Offset are 
 * applied. 
 * 
 * The individual, effective task execution rate is set within each user-task 
 * configuration. The operating system is managing the internal counters of each 
 * task and raises execution triggers at integer multiples of the OS base clock.
 * Timing settings Priority and Offset can be modified during runtime allowing 
 * readjustments to the application code execution profile. When called, the 
 * OS collects statistical data about the individual task execution and attaches
 * the results to the individual task object data structure to be observed by 
 * other software processes and allows the detection of task execution quantum
 * violations.
 * 
 * High- and low-priority task execution is achieved by setting the high-priority 
 * parameter when Queue Execute gets called from the high priority OS interrupt.
 * when Queue Execute gets called from the low-priority OS routine, only low-
 * priority tasks will get called. This allows tasks to be dynamically lifted 
 * or dropped in priority during runtime while their individual task execution 
 * range is kept unchanged during priority transitions.
 * 
 *********************************************************************************/

int main(void) {

    volatile uint16_t retval = 1;
    
    // Initialize basic system configuration
    retval &= Boot();
    
    // Initialize special, application-specific peripherals
    retval &= sysUserPeriperhals_Initialize();

    // Main program execution
    retval &= rtos_Execute();
    
    // if the firmware ever ends up here, reset the CPU
    CPU_RESET();

    // Main program execution
    retval &= rtos_Execute();

    // When the OS is exited, a CPU RESET will be triggered
    // This section of the code should never be reached.
    
    return (0);
}

// ______________________________________
// end of file
