"""add recipe yield quantity

Revision ID: b1020f328e98
Revises: 3897397b4631
Create Date: 2024-10-23 15:50:59.888793

"""

import sqlalchemy as sa
from alembic import op
from sqlalchemy import orm

from mealie.db.models._model_utils.guid import GUID
from mealie.services.scraper.cleaner import clean_yield

# revision identifiers, used by Alembic.
revision = "b1020f328e98"
down_revision: str | None = "3897397b4631"
branch_labels: str | tuple[str, ...] | None = None
depends_on: str | tuple[str, ...] | None = None


# Intermediate table definitions
class SqlAlchemyBase(orm.DeclarativeBase):
    pass


class RecipeModel(SqlAlchemyBase):
    __tablename__ = "recipes"

    id: orm.Mapped[GUID] = orm.mapped_column(GUID, primary_key=True, default=GUID.generate)
    recipe_yield: orm.Mapped[str | None] = orm.mapped_column(sa.String)
    recipe_yield_quantity: orm.Mapped[float] = orm.mapped_column(sa.Float, index=True, default=0)
    recipe_servings: orm.Mapped[float] = orm.mapped_column(sa.Float, index=True, default=0)


def parse_recipe_yields():
    bind = op.get_bind()
    session = orm.Session(bind=bind)

    for recipe in session.query(RecipeModel).all():
        try:
            recipe.recipe_servings, recipe.recipe_yield_quantity, recipe.recipe_yield = clean_yield(recipe.recipe_yield)
        except Exception:
            recipe.recipe_servings = 0
            recipe.recipe_yield_quantity = 0

    session.commit()


def upgrade():
    # ### commands auto generated by Alembic - please adjust! ###
    with op.batch_alter_table("recipes", schema=None) as batch_op:
        batch_op.add_column(sa.Column("recipe_yield_quantity", sa.Float(), nullable=False, server_default="0"))
        batch_op.create_index(batch_op.f("ix_recipes_recipe_yield_quantity"), ["recipe_yield_quantity"], unique=False)
        batch_op.add_column(sa.Column("recipe_servings", sa.Float(), nullable=False, server_default="0"))
        batch_op.create_index(batch_op.f("ix_recipes_recipe_servings"), ["recipe_servings"], unique=False)

    # ### end Alembic commands ###

    parse_recipe_yields()


def downgrade():
    # ### commands auto generated by Alembic - please adjust! ###
    with op.batch_alter_table("recipes", schema=None) as batch_op:
        batch_op.drop_index(batch_op.f("ix_recipes_recipe_servings"))
        batch_op.drop_column("recipe_servings")
        batch_op.drop_index(batch_op.f("ix_recipes_recipe_yield_quantity"))
        batch_op.drop_column("recipe_yield_quantity")

    # ### end Alembic commands ###
