package com.dimple.system.web.controller;

import com.dimple.common.core.constant.UserConstants;
import com.dimple.common.core.domain.ResponseEntity;
import com.dimple.common.core.utils.StringUtils;
import com.dimple.common.core.utils.bean.BeanMapper;
import com.dimple.common.core.utils.poi.ExcelUtil;
import com.dimple.common.core.utils.response.ResponseEntityUtils;
import com.dimple.common.core.web.controller.BaseController;
import com.dimple.common.core.web.page.TableDataInfo;
import com.dimple.common.core.web.vo.params.AjaxResult;
import com.dimple.common.log.annotation.OperationLog;
import com.dimple.common.log.enums.BusinessType;
import com.dimple.common.redis.constants.CacheConstants;
import com.dimple.common.redis.service.RedisService;
import com.dimple.common.security.annotation.InnerAuth;
import com.dimple.common.security.annotation.RequiresPermissions;
import com.dimple.common.security.utils.SecurityUtils;
import com.dimple.system.api.model.LoginUser;
import com.dimple.system.api.model.SysRoleBO;
import com.dimple.system.api.model.SysUserBO;
import com.dimple.system.service.entity.SysUser;
import com.dimple.system.service.service.SysConfigService;
import com.dimple.system.service.service.SysPermissionService;
import com.dimple.system.service.service.SysRoleService;
import com.dimple.system.service.service.SysUserService;
import com.dimple.system.web.controller.vo.SysRoleVO;
import com.dimple.system.web.controller.vo.SysUserVO;
import com.dimple.system.web.controller.vo.params.SysUserVOParams;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 用户信息
 *
 * @author Dimple
 */
@RestController
@RequestMapping("/user")
public class SysUserController extends BaseController {
    @Autowired
    private SysUserService userService;

    @Autowired
    private SysRoleService roleService;
    @Autowired
    private SysPermissionService permissionService;

    @Autowired
    private SysConfigService configService;
    @Autowired
    private RedisService redisService;

    /**
     * 获取用户列表
     */
    @RequiresPermissions("system:user:list")
    @GetMapping("/list")
    public TableDataInfo list(SysUserVOParams user) {
        startPage();
        SysUserBO sysUserBO = BeanMapper.convert(user, SysUserBO.class);
        List<SysUserBO> list = userService.selectUserList(sysUserBO);
        return getDataTable(BeanMapper.convertList(list, SysUserVO.class));
    }

    @RequiresPermissions("system:user:unlock")
    @OperationLog(title = "账户解锁", businessType = BusinessType.OTHER)
    @GetMapping("/unlock/{userName}")
    public AjaxResult unlock(@PathVariable("userName") String userName) {
        redisService.deleteObject(CacheConstants.PWD_ERR_CNT_KEY_DEFINE.formatKey(userName));
        return success();
    }

    @OperationLog(title = "用户管理", businessType = BusinessType.EXPORT)
    @RequiresPermissions("system:user:export")
    @PostMapping("/export")
    public void export(HttpServletResponse response, SysUserVOParams user) {
        SysUserBO sysUserBO = BeanMapper.convert(user, SysUserBO.class);
        List<SysUserBO> list = userService.selectUserList(sysUserBO);
        ExcelUtil<SysUserVO> util = new ExcelUtil<>(SysUserVO.class);
        util.exportExcel(response, BeanMapper.convertList(list, SysUserVO.class), "用户数据");
    }

    @OperationLog(title = "用户管理", businessType = BusinessType.IMPORT)
    @RequiresPermissions("system:user:import")
    @PostMapping("/importData")
    public AjaxResult importData(MultipartFile file, boolean updateSupport) throws Exception {
        ExcelUtil<SysUserVOParams> util = new ExcelUtil<>(SysUserVOParams.class);
        List<SysUserVOParams> userList = util.importExcel(file.getInputStream());
        String operName = SecurityUtils.getUsername();
        String message = userService.importUser(BeanMapper.convertList(userList, SysUserBO.class), updateSupport, operName);
        return success(message);
    }

    @PostMapping("/importTemplate")
    public void importTemplate(HttpServletResponse response) throws IOException {
        ExcelUtil<SysUser> util = new ExcelUtil<>(SysUser.class);
        util.importTemplateExcel(response, "用户数据");
    }

    /**
     * 获取当前用户信息
     */
    @InnerAuth
    @GetMapping("/info/{username}")
    public ResponseEntity<LoginUser> info(@PathVariable("username") String username) {
        SysUserBO sysUser = userService.selectUserByUserName(username);
        if (StringUtils.isNull(sysUser)) {
            return ResponseEntityUtils.fail("用户名或密码错误");
        }
        // 角色集合
        Set<String> roles = permissionService.getRolePermission(sysUser);
        // 权限集合
        Set<String> permissions = permissionService.getMenuPermission(sysUser);
        LoginUser sysUserVo = new LoginUser();
        sysUserVo.setSysUser(sysUser);
        sysUserVo.setRoles(roles);
        sysUserVo.setPermissions(permissions);
        return ResponseEntityUtils.ok(sysUserVo);
    }

    /**
     * 注册用户信息
     */
    @InnerAuth
    @PostMapping("/register")
    public ResponseEntity<Boolean> register(@RequestBody SysUserVOParams sysUser) {
        SysUserBO sysUserBO = BeanMapper.convert(sysUser, SysUserBO.class);
        String username = sysUser.getUserName();
        if (!("true".equals(configService.selectConfigByKey("sys.account.registerUser")))) {
            return ResponseEntityUtils.fail("当前系统没有开启注册功能！");
        }
        if (UserConstants.NOT_UNIQUE.equals(userService.checkUserNameUnique(sysUserBO))) {
            return ResponseEntityUtils.fail("保存用户'" + username + "'失败，注册账号已存在");
        }
        return ResponseEntityUtils.ok(userService.registerUser(sysUserBO));
    }

    /**
     * 获取用户信息
     *
     * @return 用户信息
     */
    @GetMapping("getInfo")
    public AjaxResult getInfo() {
        SysUserBO user = userService.selectUserById(SecurityUtils.getUserId());
        // 角色集合
        Set<String> roles = permissionService.getRolePermission(user);
        // 权限集合
        Set<String> permissions = permissionService.getMenuPermission(user);
        AjaxResult ajax = AjaxResult.success();
        ajax.put("user", user);
        ajax.put("roles", roles);
        ajax.put("permissions", permissions);
        return ajax;
    }

    /**
     * 根据用户编号获取详细信息
     */
    @RequiresPermissions("system:user:query")
    @GetMapping({"/", "/{userId}"})
    public AjaxResult getInfo(@PathVariable(value = "userId", required = false) Long userId) {
        userService.checkUserDataScope(userId);
        AjaxResult ajax = AjaxResult.success();
        List<SysRoleBO> roles = roleService.selectRoleAll();
        ajax.put("roles", SysUser.isAdmin(userId) ? roles : roles.stream().filter(r -> !r.isAdmin()).collect(Collectors.toList()));
        if (StringUtils.isNotNull(userId)) {
            SysUserBO sysUser = userService.selectUserById(userId);
            ajax.put(AjaxResult.DATA_TAG, sysUser);
            ajax.put("roleIds", sysUser.getRoles().stream().map(e->e.getRoleId()).collect(Collectors.toList()));
        }
        return ajax;
    }

    /**
     * 新增用户
     */
    @RequiresPermissions("system:user:add")
    @OperationLog(title = "用户管理", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@Validated @RequestBody SysUserVOParams user) {
        SysUserBO sysUserBO = BeanMapper.convert(user, SysUserBO.class);
        if (UserConstants.NOT_UNIQUE.equals(userService.checkUserNameUnique(sysUserBO))) {
            return error("新增用户'" + user.getUserName() + "'失败，登录账号已存在");
        } else if (StringUtils.isNotEmpty(user.getPhonenumber())
                && UserConstants.NOT_UNIQUE.equals(userService.checkPhoneUnique(sysUserBO))) {
            return error("新增用户'" + user.getUserName() + "'失败，手机号码已存在");
        } else if (StringUtils.isNotEmpty(user.getEmail())
                && UserConstants.NOT_UNIQUE.equals(userService.checkEmailUnique(sysUserBO))) {
            return error("新增用户'" + user.getUserName() + "'失败，邮箱账号已存在");
        }
        sysUserBO.setPassword(SecurityUtils.encryptPassword(user.getPassword()));
        return toAjax(userService.insertUser(sysUserBO));
    }

    /**
     * 修改用户
     */
    @RequiresPermissions("system:user:edit")
    @OperationLog(title = "用户管理", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@Validated @RequestBody SysUser user) {
        SysUserBO sysUserBO = BeanMapper.convert(user, SysUserBO.class);
        userService.checkUserAllowed(sysUserBO);
        userService.checkUserDataScope(user.getUserId());
        if (UserConstants.NOT_UNIQUE.equals(userService.checkUserNameUnique(sysUserBO))) {
            return error("修改用户'" + user.getUserName() + "'失败，登录账号已存在");
        } else if (StringUtils.isNotEmpty(user.getPhonenumber())
                && UserConstants.NOT_UNIQUE.equals(userService.checkPhoneUnique(sysUserBO))) {
            return error("修改用户'" + user.getUserName() + "'失败，手机号码已存在");
        } else if (StringUtils.isNotEmpty(user.getEmail())
                && UserConstants.NOT_UNIQUE.equals(userService.checkEmailUnique(sysUserBO))) {
            return error("修改用户'" + user.getUserName() + "'失败，邮箱账号已存在");
        }
        return toAjax(userService.updateUser(sysUserBO));
    }

    /**
     * 删除用户
     */
    @RequiresPermissions("system:user:remove")
    @OperationLog(title = "用户管理", businessType = BusinessType.DELETE)
    @DeleteMapping("/{userIds}")
    public AjaxResult remove(@PathVariable Long[] userIds) {
        if (ArrayUtils.contains(userIds, SecurityUtils.getUserId())) {
            return error("当前用户不能删除");
        }
        return toAjax(userService.deleteUserByIds(userIds));
    }

    /**
     * 重置密码
     */
    @RequiresPermissions("system:user:edit")
    @OperationLog(title = "用户管理", businessType = BusinessType.UPDATE)
    @PutMapping("/resetPwd")
    public AjaxResult resetPwd(@RequestBody SysUserVOParams user) {
        SysUserBO sysUserBO = BeanMapper.convert(user, SysUserBO.class);
        userService.checkUserAllowed(sysUserBO);
        userService.checkUserDataScope(sysUserBO.getUserId());
        sysUserBO.setPassword(SecurityUtils.encryptPassword(user.getPassword()));
        return toAjax(userService.resetPwd(sysUserBO));
    }

    /**
     * 状态修改
     */
    @RequiresPermissions("system:user:edit")
    @OperationLog(title = "用户管理", businessType = BusinessType.UPDATE)
    @PutMapping("/changeStatus")
    public AjaxResult changeStatus(@RequestBody SysUserVOParams user) {
        SysUserBO sysUserBO = BeanMapper.convert(user, SysUserBO.class);
        userService.checkUserAllowed(sysUserBO);
        userService.checkUserDataScope(user.getUserId());
        return toAjax(userService.updateUserStatus(sysUserBO));
    }

    /**
     * 根据用户编号获取授权角色
     */
    @RequiresPermissions("system:user:query")
    @GetMapping("/authRole/{userId}")
    public AjaxResult authRole(@PathVariable("userId") Long userId) {
        AjaxResult ajax = AjaxResult.success();
        SysUserVO user = BeanMapper.convert(userService.selectUserById(userId), SysUserVO.class);
        List<SysRoleVO> roles = BeanMapper.convertList(roleService.selectRolesByUserId(userId), SysRoleVO.class);
        ajax.put("user", BeanMapper.convert(user, SysUserVO.class));
        ajax.put("roles", SysUser.isAdmin(userId) ? roles : roles.stream().filter(r -> !r.isAdmin()).collect(Collectors.toList()));
        return ajax;
    }

    /**
     * 用户授权角色
     */
    @RequiresPermissions("system:user:edit")
    @OperationLog(title = "用户管理", businessType = BusinessType.GRANT)
    @PutMapping("/authRole")
    public AjaxResult insertAuthRole(Long userId, Long[] roleIds) {
        userService.checkUserDataScope(userId);
        userService.insertUserAuth(userId, roleIds);
        return success();
    }
}
