package com.dimple.system.service.service.impl;

import com.dimple.common.core.constant.UserConstants;
import com.dimple.common.core.exception.ServiceException;
import com.dimple.common.core.utils.StringUtils;
import com.dimple.common.core.utils.bean.BeanMapper;
import com.dimple.common.security.utils.DictUtils;
import com.dimple.system.api.model.SysDictDataBO;
import com.dimple.system.service.entity.SysDictData;
import com.dimple.system.service.entity.SysDictType;
import com.dimple.system.service.mapper.SysDictDataMapper;
import com.dimple.system.service.mapper.SysDictTypeMapper;
import com.dimple.system.service.service.SysDictTypeService;
import com.dimple.system.service.service.bo.SysDictTypeBO;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.PostConstruct;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 字典 业务层处理
 *
 * @author Dimple
 */
@Service
public class SysDictTypeServiceImpl implements SysDictTypeService {
    @Autowired
    private SysDictTypeMapper dictTypeMapper;

    @Autowired
    private SysDictDataMapper dictDataMapper;

    /**
     * 项目启动时，初始化字典到缓存
     */
    @PostConstruct
    public void init() {
        loadingDictCache();
    }

    /**
     * 根据条件分页查询字典类型
     *
     * @param dictType 字典类型信息
     * @return 字典类型集合信息
     */
    @Override
    public List<SysDictTypeBO> selectDictTypeList(SysDictTypeBO dictType) {
        SysDictType sysDictType = BeanMapper.convert(dictType, SysDictType.class);
        return BeanMapper.convertList(dictTypeMapper.selectDictTypeList(sysDictType), SysDictTypeBO.class);
    }

    /**
     * 根据所有字典类型
     *
     * @return 字典类型集合信息
     */
    @Override
    public List<SysDictTypeBO> selectDictTypeAll() {
        return BeanMapper.convertList(dictTypeMapper.selectDictTypeAll(), SysDictTypeBO.class);
    }

    /**
     * 根据字典类型查询字典数据
     *
     * @param dictType 字典类型
     * @return 字典数据集合信息
     */
    @Override
    public List<SysDictDataBO> selectDictDataByType(String dictType) {
        List<SysDictDataBO> dictDatas = DictUtils.getDictCache(dictType);
        if (StringUtils.isNotEmpty(dictDatas)) {
            return dictDatas;
        }
        dictDatas = BeanMapper.convertList(dictDataMapper.selectDictDataByType(dictType), SysDictDataBO.class);
        if (StringUtils.isNotEmpty(dictDatas)) {
            DictUtils.setDictCache(dictType, dictDatas);
            return dictDatas;
        }
        return null;
    }

    /**
     * 根据字典类型ID查询信息
     *
     * @param id 字典类型ID
     * @return 字典类型
     */
    @Override
    public SysDictTypeBO selectDictTypeById(Long id) {
        return BeanMapper.convert(dictTypeMapper.selectDictTypeById(id), SysDictTypeBO.class);
    }

    /**
     * 根据字典类型查询信息
     *
     * @param dictType 字典类型
     * @return 字典类型
     */
    @Override
    public SysDictTypeBO selectDictTypeByType(String dictType) {
        return BeanMapper.convert(dictTypeMapper.selectDictTypeByType(dictType), SysDictTypeBO.class);
    }

    /**
     * 批量删除字典类型信息
     *
     * @param ids 需要删除的字典ID
     */
    @Override
    public void deleteDictTypeByIds(Long[] ids) {
        for (Long id : ids) {
            SysDictTypeBO dictType = selectDictTypeById(id);
            if (dictDataMapper.countDictDataByType(dictType.getDictType()) > 0) {
                throw new ServiceException(String.format("%1$s已分配,不能删除", dictType.getDictName()));
            }
            dictTypeMapper.deleteDictTypeById(id);
            DictUtils.removeDictCache(dictType.getDictType());
        }
    }

    /**
     * 加载字典缓存数据
     */
    @Override
    public void loadingDictCache() {
        SysDictData dictData = new SysDictData();
        dictData.setStatus("0");
        Map<String, List<SysDictData>> dictDataMap = dictDataMapper.selectDictDataList(dictData).stream().collect(Collectors.groupingBy(SysDictData::getDictType));
        for (Map.Entry<String, List<SysDictData>> entry : dictDataMap.entrySet()) {
            DictUtils.setDictCache(entry.getKey(), entry.getValue().stream().sorted(Comparator.comparing(SysDictData::getDictSort)).map(e -> BeanMapper.convert(e, SysDictDataBO.class)).collect(Collectors.toList()));
        }
    }

    /**
     * 清空字典缓存数据
     */
    @Override
    public void clearDictCache() {
        DictUtils.clearDictCache();
    }

    /**
     * 重置字典缓存数据
     */
    @Override
    public void resetDictCache() {
        clearDictCache();
        loadingDictCache();
    }

    /**
     * 新增保存字典类型信息
     *
     * @param dict 字典类型信息
     * @return affected lines
     */
    @Override
    public int insertDictType(SysDictTypeBO dict) {
        SysDictType sysDictType = BeanMapper.convert(dict, SysDictType.class);
        int row = dictTypeMapper.insertDictType(sysDictType);
        if (row > 0) {
            DictUtils.setDictCache(dict.getDictType(), null);
        }
        return row;
    }

    /**
     * 修改保存字典类型信息
     *
     * @param dict 字典类型信息
     * @return affected lines
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public int updateDictType(SysDictTypeBO dict) {
        SysDictType oldDict = dictTypeMapper.selectDictTypeById(dict.getId());
        dictDataMapper.updateDictDataType(oldDict.getDictType(), dict.getDictType());
        SysDictType sysDictType = BeanMapper.convert(dict, SysDictType.class);
        int row = dictTypeMapper.updateDictType(sysDictType);
        if (row > 0) {
            List<SysDictDataBO> dictDatas = BeanMapper.convertList(dictDataMapper.selectDictDataByType(dict.getDictType()), SysDictDataBO.class);
            DictUtils.setDictCache(dict.getDictType(), dictDatas);
        }
        return row;
    }

    /**
     * 校验字典类型称是否唯一
     *
     * @param dict 字典类型
     * @return affected lines
     */
    @Override
    public String checkDictTypeUnique(SysDictTypeBO dict) {
        Long id = StringUtils.isNull(dict.getId()) ? -1L : dict.getId();
        SysDictType dictType = dictTypeMapper.checkDictTypeUnique(dict.getDictType());
        if (StringUtils.isNotNull(dictType) && dictType.getId().longValue() != id.longValue()) {
            return UserConstants.NOT_UNIQUE;
        }
        return UserConstants.UNIQUE;
    }
}
