#!/usr/bin/env node
// Copyright IBM Corp. 2020. All Rights Reserved.
// Node module: loopback-next
// This file is licensed under the MIT License.
// License text available at https://opensource.org/licenses/MIT

/**
 * This is an internal script to update docs/site MONOREPO.md file with filling
 * up all packages with lexicographical order
 */
'use strict';

const path = require('node:path');
const fse = require('fs-extra');
const createMarkdownTable = require('markdown-table');
const {getPackages, runMain} = require('./script-util');

const MONOREPO_FILE_DIST = 'docs/site';
const MONOREPO_FILE_NAME = 'MONOREPO.md';

function getPackageRelativeUri(pkg) {
  return path.relative(pkg.rootPath, pkg.location).replace(/\\/g, '/');
}

async function getSortedPackages() {
  const packages = await getPackages();
  packages.sort((p1, p2) =>
    getPackageRelativeUri(p1).localeCompare(getPackageRelativeUri(p2)),
  );
  return packages;
}

function getPackageFields(pkg) {
  const packageJson = fse.readJsonSync(path.join(pkg.location, 'package.json'));

  const relativeUri = getPackageRelativeUri(pkg);
  const pkgUrl = `https://github.com/loopbackio/loopback-next/tree/master/${relativeUri}`;

  return [
    `[${relativeUri}](${pkgUrl})`,
    pkg.private ? '_(private)_' : pkg.name,
    packageJson.description,
  ];
}

function updateTable(packages) {
  const header = ['Package', 'npm', 'Description'];
  const rows = packages.map(getPackageFields);

  return createMarkdownTable([header, ...rows]);
}

async function updateMonorepoFile() {
  const packages = await getSortedPackages();
  const markdownTable = updateTable(packages);
  const monorepoFilePath = path.join(MONOREPO_FILE_DIST, MONOREPO_FILE_NAME);

  const content = [
    '<!-- Do not edit this file. It is automatically generated by update-monorepo-file script -->',
    '',
    '# Monorepo overview',
    '',
    'The [loopback-next](https://github.com/loopbackio/loopback-next) repository uses',
    '[lerna](https://lernajs.io/) to manage multiple packages for LoopBack 4.',
    '',
    'Please run the following command to update packages information after adding new',
    'one in the monorepo: `npm run update-monorepo-file`',
    '',
    markdownTable,
    '',
    'We use npm scripts declared in',
    '[package.json](https://github.com/loopbackio/loopback-next/blob/master/package.json)',
    'to work with the monorepo managed by lerna. See',
    '[Developing LoopBack](./DEVELOPING.md) for more details.',
  ].join('\n');

  fse.writeFileSync(monorepoFilePath, content + '\n', {encoding: 'utf-8'});
}

module.exports = updateMonorepoFile;

runMain(module, updateMonorepoFile);
