/**
 * Copyright (c) 2016 - 2022 Alex Grant (@localnerve), LocalNerve LLC
 * Copyrights licensed under the BSD License. See the accompanying LICENSE file for terms.
 *
 * A limited mock for amqplib
 */
/* global Promise */
'use strict';

function Channel (chanError, msg, ack, nack) {
  this.chanError = chanError;
  this.msg = msg;
  this.cbAck = ack;
  this.cbNack = nack;
  this.error = new Error('channel');
}
Channel.prototype = {
  // Return promise, reject if chanError is true.
  assertQueue: function () {
    var self = this;

    return new Promise(function (resolve, reject) {
      if (self.chanError) {
        return reject(self.error);
      }
      resolve();
    });
  },
  sendToQueue: function () {
  },
  consume: function (q, cb) {
    cb(this.msg);
  },
  nack: function (msg) {
    this.cbNack(msg);
  },
  ack: function (msg) {
    this.cbAck(msg);
  }
};

function Connection (connError, chanError, msg, ack, nack) {
  this.connError = connError;
  this.chanError = chanError;
  this.msg = msg;
  this.cbAck = ack;
  this.cbNack = nack;
  this.error = new Error('connection');
}
Connection.prototype = {
  // Return promise that receives a Channel on resolve
  // or reject if connError is true.
  createChannel: function () {
    var self = this;

    return new Promise(function (resolve, reject) {
      if (self.connError) {
        return reject(self.error);
      }
      resolve(
        new Channel(
          self.chanError, self.msg, self.cbAck, self.cbNack
        )
      );
    });
  }
};

var errorProfile = {
  connect: false,
  connection: false,
  channel: false
};

var consumerMessage = {
  content: {
    toString: function() {
      return JSON.stringify({});
    }
  }
};

var consumerAck = function () {
};

var consumerNack = function () {
};

// Return promise that receives a connection on resolve
// or reject if emulateError is true.
function connect () {
  return new Promise(function (resolve, reject) {
    if (errorProfile.connect) {
      return reject(new Error('connect'));
    }
    resolve(
      new Connection(
        errorProfile.connection, errorProfile.channel,
        consumerMessage, consumerAck, consumerNack
      )
    );
  });
}

function setErrors (profile) {
  errorProfile.connect = profile.connect;
  errorProfile.connection = profile.connection;
  errorProfile.channel = profile.channel;
}

function setConsumerMessage (msg) {
  consumerMessage.content = new Buffer(JSON.stringify(msg));
}

function getConsumerMessage(msg) {
  return JSON.parse(msg.content.toString());
}

function setConsumerAck (cb) {
  consumerAck = cb;
}

function setConsumerNack (cb) {
  consumerNack = cb;
}

module.exports = {
  connect: connect,
  setErrors: setErrors,
  setConsumerMessage: setConsumerMessage,
  getConsumerMessage: getConsumerMessage,
  setConsumerAck: setConsumerAck,
  setConsumerNack: setConsumerNack
};
