/***
 * Copyright (c) 2016 - 2022 Alex Grant (@localnerve), LocalNerve LLC
 * Copyrights licensed under the BSD License. See the accompanying LICENSE file for terms.
 *
 * Environment specific configuration for mail.
 *
 * Environment variables can override the following:
 *   MAIL_SERVICE - A string that denotes a mail service known to nodemailer, default Mandrill
 *   MAIL_USERNAME - The authenticated service user of the mail service, default MANDRILL_USERNAME from env
 *   MAIL_PASSWORD - The authenticated service pass for the mail service, default MANDRILL_APIKEY from env
 *   MAIL_TO - The mailto of all mail messages from this app, defaults to NODE_ENV chosen headers
 *   MAIL_FROM - The mailfrom of all mail messages from this app, defaults to NODE_ENV chosen headers
 *   QUEUE_NAME - The name of the outgoing mail queue, defaults to 'outgoing-mail'
 *   QUEUE_URL - The url of the queue service, defaults to NODE_ENV chosen url
 */
'use strict';

/***
 * Environment specific mail header default values.
 */
var mailHeaders = {
  development: {
    mailTo: 'react-pwa-dev@localnerve.com',
    mailFrom: 'react-pwa-from-dev@localnerve.com'
  },
  production: {
    mailTo: 'react-pwa-prod@localnerve.com',
    mailFrom: 'react-pwa-from-prod@localnerve.com'
  }
};

/**
 * Get the MAIL_SERVICE configuration value.
 * Defaults to 'Sparkpost'.
 * @returns {String} The MAIL_SERVICE configuration value.
 */
function MAIL_SERVICE () {
  return process.env.MAIL_SERVICE || 'Sparkpost';
}

/**
 * Get the MAIL_USERNAME configuration value.
 * Defaults to MANDRILL_USERNAME from process.env
 * @returns {String} The MAIL_USERNAME configuration value.
 */
function MAIL_USERNAME () {
  return process.env.MAIL_USERNAME || process.env.SPARKPOST_USERNAME;
}

/**
 * Get the MAIL_PASSWORD configuration value.
 * Defaults to MANDRILL_APIKEY from process.env
 * @returns {String} The MAIL_PASSWORD configuration value.
 */
function MAIL_PASSWORD () {
  return process.env.MAIL_PASSWORD || process.env.SPARKPOST_APIKEY;
}

/**
 * Get the MAIL_TO configuration value.
 * Defaults to environment specific mail header defaults.
 * @returns {String} The MAIL_TO configuration value.
 */
function mailTo (env) {
  return process.env.MAIL_TO || mailHeaders[env].mailTo ||
    mailHeaders.development.mailTo;
}

/**
 * Get the MAIL_FROM configuration value.
 * Defaults to environment specific mail header defaults.
 * @returns {String} The MAIL_FROM configuration value.
 */
function mailFrom (env) {
  return process.env.MAIL_FROM || mailHeaders[env].mailFrom ||
    mailHeaders.development.mailFrom;
}

/**
 * Get the QUEUE_NAME configuration value.
 * Defaults to 'outgoing-mail'.
 * @returns {String} The QUEUE_NAME configuration value.
 */
function QUEUE_NAME () {
  return process.env.QUEUE_NAME || 'outgoing-mail';
}

/**
 * Get the QUEUE_URL configuration value.
 * Defaults to environment specific mail queue url defaults.
 * @returns {String} The QUEUE_URL configuration value.
 */
function QUEUE_URL (env) {
  return process.env.QUEUE_URL ||
    (env === 'production' ? process.env.CLOUDAMQP_URL : 'amqp://localhost');
}

/**
 * Make the contact configuration object.
 *
 * @param {Object} nconf - The nconfig object.
 * @returns {Object} The contact configuration object.
 */
function makeConfig(nconf) {
  var env = nconf.get('NODE_ENV');

  return {
    mail: {
      service: MAIL_SERVICE,
      username: MAIL_USERNAME,
      password: MAIL_PASSWORD,
      subject: 'React Pwa Contact Form Submission',

      /**
       * @see mailTo
       */
      to: function () {
        return mailTo(env);
      },

      /**
       * @see mailFrom
       */
      from: function () {
        return mailFrom(env);
      }
    },
    queue: {
      name: QUEUE_NAME,

      /**
       * @see QUEUE_URL
       */
      url: function () {
        return QUEUE_URL(env);
      }
    }
  };
}

module.exports = makeConfig;
