// SPDX-License-Identifier: GPL-2.0-or-later
/*
 * Copyright (c) 2017 Richard Palethorpe <rpalethorpe@suse.com>
 * Copyright (c) 2016 Jan Horn <jann@thejh.net>
 */
/*
 * Test for CVE-2016-10044, which was fixed in commit
 * 22f6b4d34fcf039c aio: mark AIO pseudo-fs noexec.
 *
 * The test checks that we can not implicitly mark AIO mappings as
 * executable using the READ_IMPLIES_EXEC personality.
 */

#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include "lapi/personality.h"
#include "tst_test.h"
#include "tst_safe_stdio.h"
#include "lapi/syscalls.h"

static FILE * f;

static void cleanup(void)
{
	if (f)
		SAFE_FCLOSE(f);
}

static void run(void)
{
	void *ctx = 0;
	char perms[8], line[BUFSIZ];

	SAFE_PERSONALITY(READ_IMPLIES_EXEC);
	if (tst_syscall(__NR_io_setup, 1, &ctx))
		tst_brk(TBROK | TERRNO, "Failed to create AIO context");

	f = SAFE_FOPEN("/proc/self/maps", "r");
	while (fgets(line, BUFSIZ, f) != NULL) {
		if (strstr(line, "[aio]") != NULL)
			goto found_mapping;
	}
	tst_brk(TCONF, "Could not find mapping in /proc/self/maps");

found_mapping:
	if (sscanf(line, "%*x-%*x %s", perms) != 1)
		tst_brk(TBROK, "failed to find permission string in %s", line);
	if (strchr(perms, (int)'x'))
		tst_res(TFAIL, "AIO mapping is executable: %s!", perms);
	else
		tst_res(TPASS, "AIO mapping is not executable: %s", perms);

	if (tst_syscall(__NR_io_destroy, ctx))
		tst_brk(TBROK | TERRNO, "Failed to destroy AIO context");

	SAFE_FCLOSE(f);
	f = NULL;
}

static struct tst_test test = {
	.test_all = run,
	.cleanup = cleanup,
	.tags = (const struct tst_tag[]) {
		{"linux-git", "22f6b4d34fcf"},
		{"CVE", "2016-10044"},
		{}
	}
};
