/**
 * ascii-folder.js
 * https://github.com/mplatt/fold-to-ascii-js
 *
 * This is a JavaScript port of the Apache Lucene ASCII Folding Filter.
 *
 * The Apache Lucene ASCII Folding Filter is licensed to the Apache Software
 * Foundation (ASF) under one or more contributor license agreements. See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The ASF licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
class ASCIIFolder {
  static foldReplacing(str = "", replacement = ""): string {
    return this._fold(str, () => replacement);
  }

  static foldMaintaining(str = ""): string {
    return this._fold(str, (char: string) => char);
  }

  static _fold(str: string, fallback: (character: string) => string): string {
    if (str === null) return "";

    if (typeof str === "number") return "" + str;

    if (typeof str !== "string") throw new Error("Invalid input data type");

    return str
      .split("")
      .map(character => {
        if (character.charCodeAt(0) < 128) {
          return character;
        } else {
          const replacement = this.mapping.get(character.charCodeAt(0));
          return replacement === undefined ? fallback(character) : replacement;
        }
      })
      .join("");
  }

  private static mapping = new Map([
    [0xc0, "A"],
    [0xc1, "A"],
    [0xc2, "A"],
    [0xc3, "A"],
    [0xc4, "A"],
    [0xc5, "A"],
    [0x100, "A"],
    [0x102, "A"],
    [0x104, "A"],
    [0x18f, "A"],
    [0x1cd, "A"],
    [0x1de, "A"],
    [0x1e0, "A"],
    [0x1fa, "A"],
    [0x200, "A"],
    [0x202, "A"],
    [0x226, "A"],
    [0x23a, "A"],
    [0x1d00, "A"],
    [0x1e00, "A"],
    [0x1ea0, "A"],
    [0x1ea2, "A"],
    [0x1ea4, "A"],
    [0x1ea6, "A"],
    [0x1ea8, "A"],
    [0x1eaa, "A"],
    [0x1eac, "A"],
    [0x1eae, "A"],
    [0x1eb0, "A"],
    [0x1eb2, "A"],
    [0x1eb4, "A"],
    [0x1eb6, "A"],
    [0x24b6, "A"],
    [0xff21, "A"],
    [0xe0, "a"],
    [0xe1, "a"],
    [0xe2, "a"],
    [0xe3, "a"],
    [0xe4, "a"],
    [0xe5, "a"],
    [0x101, "a"],
    [0x103, "a"],
    [0x105, "a"],
    [0x1ce, "a"],
    [0x1df, "a"],
    [0x1e1, "a"],
    [0x1fb, "a"],
    [0x201, "a"],
    [0x203, "a"],
    [0x227, "a"],
    [0x250, "a"],
    [0x259, "a"],
    [0x25a, "a"],
    [0x1d8f, "a"],
    [0x1d95, "a"],
    [0x1e01, "a"],
    [0x1e9a, "a"],
    [0x1ea1, "a"],
    [0x1ea3, "a"],
    [0x1ea5, "a"],
    [0x1ea7, "a"],
    [0x1ea9, "a"],
    [0x1eab, "a"],
    [0x1ead, "a"],
    [0x1eaf, "a"],
    [0x1eb1, "a"],
    [0x1eb3, "a"],
    [0x1eb5, "a"],
    [0x1eb7, "a"],
    [0x2090, "a"],
    [0x2094, "a"],
    [0x24d0, "a"],
    [0x2c65, "a"],
    [0x2c6f, "a"],
    [0xff41, "a"],
    [0xa732, "AA"],
    [0xc6, "AE"],
    [0x1e2, "AE"],
    [0x1fc, "AE"],
    [0x1d01, "AE"],
    [0xa734, "AO"],
    [0xa736, "AU"],
    [0xa738, "AV"],
    [0xa73a, "AV"],
    [0xa73c, "AY"],
    [0x249c, "(a)"],
    [0xa733, "aa"],
    [0xe6, "ae"],
    [0x1e3, "ae"],
    [0x1fd, "ae"],
    [0x1d02, "ae"],
    [0xa735, "ao"],
    [0xa737, "au"],
    [0xa739, "av"],
    [0xa73b, "av"],
    [0xa73d, "ay"],
    [0x181, "B"],
    [0x182, "B"],
    [0x243, "B"],
    [0x299, "B"],
    [0x1d03, "B"],
    [0x1e02, "B"],
    [0x1e04, "B"],
    [0x1e06, "B"],
    [0x24b7, "B"],
    [0xff22, "B"],
    [0x180, "b"],
    [0x183, "b"],
    [0x253, "b"],
    [0x1d6c, "b"],
    [0x1d80, "b"],
    [0x1e03, "b"],
    [0x1e05, "b"],
    [0x1e07, "b"],
    [0x24d1, "b"],
    [0xff42, "b"],
    [0x249d, "(b)"],
    [0xc7, "C"],
    [0x106, "C"],
    [0x108, "C"],
    [0x10a, "C"],
    [0x10c, "C"],
    [0x187, "C"],
    [0x23b, "C"],
    [0x297, "C"],
    [0x1d04, "C"],
    [0x1e08, "C"],
    [0x24b8, "C"],
    [0xff23, "C"],
    [0xe7, "c"],
    [0x107, "c"],
    [0x109, "c"],
    [0x10b, "c"],
    [0x10d, "c"],
    [0x188, "c"],
    [0x23c, "c"],
    [0x255, "c"],
    [0x1e09, "c"],
    [0x2184, "c"],
    [0x24d2, "c"],
    [0xa73e, "c"],
    [0xa73f, "c"],
    [0xff43, "c"],
    [0x249e, "(c)"],
    [0xd0, "D"],
    [0x10e, "D"],
    [0x110, "D"],
    [0x189, "D"],
    [0x18a, "D"],
    [0x18b, "D"],
    [0x1d05, "D"],
    [0x1d06, "D"],
    [0x1e0a, "D"],
    [0x1e0c, "D"],
    [0x1e0e, "D"],
    [0x1e10, "D"],
    [0x1e12, "D"],
    [0x24b9, "D"],
    [0xa779, "D"],
    [0xff24, "D"],
    [0xf0, "d"],
    [0x10f, "d"],
    [0x111, "d"],
    [0x18c, "d"],
    [0x221, "d"],
    [0x256, "d"],
    [0x257, "d"],
    [0x1d6d, "d"],
    [0x1d81, "d"],
    [0x1d91, "d"],
    [0x1e0b, "d"],
    [0x1e0d, "d"],
    [0x1e0f, "d"],
    [0x1e11, "d"],
    [0x1e13, "d"],
    [0x24d3, "d"],
    [0xa77a, "d"],
    [0xff44, "d"],
    [0x1c4, "DZ"],
    [0x1f1, "DZ"],
    [0x1c5, "Dz"],
    [0x1f2, "Dz"],
    [0x249f, "(d)"],
    [0x238, "db"],
    [0x1c6, "dz"],
    [0x1f3, "dz"],
    [0x2a3, "dz"],
    [0x2a5, "dz"],
    [0xc8, "E"],
    [0xc9, "E"],
    [0xca, "E"],
    [0xcb, "E"],
    [0x112, "E"],
    [0x114, "E"],
    [0x116, "E"],
    [0x118, "E"],
    [0x11a, "E"],
    [0x18e, "E"],
    [0x190, "E"],
    [0x204, "E"],
    [0x206, "E"],
    [0x228, "E"],
    [0x246, "E"],
    [0x1d07, "E"],
    [0x1e14, "E"],
    [0x1e16, "E"],
    [0x1e18, "E"],
    [0x1e1a, "E"],
    [0x1e1c, "E"],
    [0x1eb8, "E"],
    [0x1eba, "E"],
    [0x1ebc, "E"],
    [0x1ebe, "E"],
    [0x1ec0, "E"],
    [0x1ec2, "E"],
    [0x1ec4, "E"],
    [0x1ec6, "E"],
    [0x24ba, "E"],
    [0x2c7b, "E"],
    [0xff25, "E"],
    [0xe8, "e"],
    [0xe9, "e"],
    [0xea, "e"],
    [0xeb, "e"],
    [0x113, "e"],
    [0x115, "e"],
    [0x117, "e"],
    [0x119, "e"],
    [0x11b, "e"],
    [0x1dd, "e"],
    [0x205, "e"],
    [0x207, "e"],
    [0x229, "e"],
    [0x247, "e"],
    [0x258, "e"],
    [0x25b, "e"],
    [0x25c, "e"],
    [0x25d, "e"],
    [0x25e, "e"],
    [0x29a, "e"],
    [0x1d08, "e"],
    [0x1d92, "e"],
    [0x1d93, "e"],
    [0x1d94, "e"],
    [0x1e15, "e"],
    [0x1e17, "e"],
    [0x1e19, "e"],
    [0x1e1b, "e"],
    [0x1e1d, "e"],
    [0x1eb9, "e"],
    [0x1ebb, "e"],
    [0x1ebd, "e"],
    [0x1ebf, "e"],
    [0x1ec1, "e"],
    [0x1ec3, "e"],
    [0x1ec5, "e"],
    [0x1ec7, "e"],
    [0x2091, "e"],
    [0x24d4, "e"],
    [0x2c78, "e"],
    [0xff45, "e"],
    [0x24a0, "(e)"],
    [0x191, "F"],
    [0x1e1e, "F"],
    [0x24bb, "F"],
    [0xa730, "F"],
    [0xa77b, "F"],
    [0xa7fb, "F"],
    [0xff26, "F"],
    [0x192, "f"],
    [0x1d6e, "f"],
    [0x1d82, "f"],
    [0x1e1f, "f"],
    [0x1e9b, "f"],
    [0x24d5, "f"],
    [0xa77c, "f"],
    [0xff46, "f"],
    [0x24a1, "(f)"],
    [0xfb00, "ff"],
    [0xfb03, "ffi"],
    [0xfb04, "ffl"],
    [0xfb01, "fi"],
    [0xfb02, "fl"],
    [0x11c, "G"],
    [0x11e, "G"],
    [0x120, "G"],
    [0x122, "G"],
    [0x193, "G"],
    [0x1e4, "G"],
    [0x1e5, "G"],
    [0x1e6, "G"],
    [0x1e7, "G"],
    [0x1f4, "G"],
    [0x262, "G"],
    [0x29b, "G"],
    [0x1e20, "G"],
    [0x24bc, "G"],
    [0xa77d, "G"],
    [0xa77e, "G"],
    [0xff27, "G"],
    [0x11d, "g"],
    [0x11f, "g"],
    [0x121, "g"],
    [0x123, "g"],
    [0x1f5, "g"],
    [0x260, "g"],
    [0x261, "g"],
    [0x1d77, "g"],
    [0x1d79, "g"],
    [0x1d83, "g"],
    [0x1e21, "g"],
    [0x24d6, "g"],
    [0xa77f, "g"],
    [0xff47, "g"],
    [0x24a2, "(g)"],
    [0x124, "H"],
    [0x126, "H"],
    [0x21e, "H"],
    [0x29c, "H"],
    [0x1e22, "H"],
    [0x1e24, "H"],
    [0x1e26, "H"],
    [0x1e28, "H"],
    [0x1e2a, "H"],
    [0x24bd, "H"],
    [0x2c67, "H"],
    [0x2c75, "H"],
    [0xff28, "H"],
    [0x125, "h"],
    [0x127, "h"],
    [0x21f, "h"],
    [0x265, "h"],
    [0x266, "h"],
    [0x2ae, "h"],
    [0x2af, "h"],
    [0x1e23, "h"],
    [0x1e25, "h"],
    [0x1e27, "h"],
    [0x1e29, "h"],
    [0x1e2b, "h"],
    [0x1e96, "h"],
    [0x24d7, "h"],
    [0x2c68, "h"],
    [0x2c76, "h"],
    [0xff48, "h"],
    [0x1f6, "HV"],
    [0x24a3, "(h)"],
    [0x195, "hv"],
    [0xcc, "I"],
    [0xcd, "I"],
    [0xce, "I"],
    [0xcf, "I"],
    [0x128, "I"],
    [0x12a, "I"],
    [0x12c, "I"],
    [0x12e, "I"],
    [0x130, "I"],
    [0x196, "I"],
    [0x197, "I"],
    [0x1cf, "I"],
    [0x208, "I"],
    [0x20a, "I"],
    [0x26a, "I"],
    [0x1d7b, "I"],
    [0x1e2c, "I"],
    [0x1e2e, "I"],
    [0x1ec8, "I"],
    [0x1eca, "I"],
    [0x24be, "I"],
    [0xa7fe, "I"],
    [0xff29, "I"],
    [0xec, "i"],
    [0xed, "i"],
    [0xee, "i"],
    [0xef, "i"],
    [0x129, "i"],
    [0x12b, "i"],
    [0x12d, "i"],
    [0x12f, "i"],
    [0x131, "i"],
    [0x1d0, "i"],
    [0x209, "i"],
    [0x20b, "i"],
    [0x268, "i"],
    [0x1d09, "i"],
    [0x1d62, "i"],
    [0x1d7c, "i"],
    [0x1d96, "i"],
    [0x1e2d, "i"],
    [0x1e2f, "i"],
    [0x1ec9, "i"],
    [0x1ecb, "i"],
    [0x2071, "i"],
    [0x24d8, "i"],
    [0xff49, "i"],
    [0x132, "IJ"],
    [0x24a4, "(i)"],
    [0x133, "ij"],
    [0x134, "J"],
    [0x248, "J"],
    [0x1d0a, "J"],
    [0x24bf, "J"],
    [0xff2a, "J"],
    [0x135, "j"],
    [0x1f0, "j"],
    [0x237, "j"],
    [0x249, "j"],
    [0x25f, "j"],
    [0x284, "j"],
    [0x29d, "j"],
    [0x24d9, "j"],
    [0x2c7c, "j"],
    [0xff4a, "j"],
    [0x24a5, "(j)"],
    [0x136, "K"],
    [0x198, "K"],
    [0x1e8, "K"],
    [0x1d0b, "K"],
    [0x1e30, "K"],
    [0x1e32, "K"],
    [0x1e34, "K"],
    [0x24c0, "K"],
    [0x2c69, "K"],
    [0xa740, "K"],
    [0xa742, "K"],
    [0xa744, "K"],
    [0xff2b, "K"],
    [0x137, "k"],
    [0x199, "k"],
    [0x1e9, "k"],
    [0x29e, "k"],
    [0x1d84, "k"],
    [0x1e31, "k"],
    [0x1e33, "k"],
    [0x1e35, "k"],
    [0x24da, "k"],
    [0x2c6a, "k"],
    [0xa741, "k"],
    [0xa743, "k"],
    [0xa745, "k"],
    [0xff4b, "k"],
    [0x24a6, "(k)"],
    [0x139, "L"],
    [0x13b, "L"],
    [0x13d, "L"],
    [0x13f, "L"],
    [0x141, "L"],
    [0x23d, "L"],
    [0x29f, "L"],
    [0x1d0c, "L"],
    [0x1e36, "L"],
    [0x1e38, "L"],
    [0x1e3a, "L"],
    [0x1e3c, "L"],
    [0x24c1, "L"],
    [0x2c60, "L"],
    [0x2c62, "L"],
    [0xa746, "L"],
    [0xa748, "L"],
    [0xa780, "L"],
    [0xff2c, "L"],
    [0x13a, "l"],
    [0x13c, "l"],
    [0x13e, "l"],
    [0x140, "l"],
    [0x142, "l"],
    [0x19a, "l"],
    [0x234, "l"],
    [0x26b, "l"],
    [0x26c, "l"],
    [0x26d, "l"],
    [0x1d85, "l"],
    [0x1e37, "l"],
    [0x1e39, "l"],
    [0x1e3b, "l"],
    [0x1e3d, "l"],
    [0x24db, "l"],
    [0x2c61, "l"],
    [0xa747, "l"],
    [0xa749, "l"],
    [0xa781, "l"],
    [0xff4c, "l"],
    [0x1c7, "LJ"],
    [0x1efa, "LL"],
    [0x1c8, "Lj"],
    [0x24a7, "(l)"],
    [0x1c9, "lj"],
    [0x1efb, "ll"],
    [0x2aa, "ls"],
    [0x2ab, "lz"],
    [0x19c, "M"],
    [0x1d0d, "M"],
    [0x1e3e, "M"],
    [0x1e40, "M"],
    [0x1e42, "M"],
    [0x24c2, "M"],
    [0x2c6e, "M"],
    [0xa7fd, "M"],
    [0xa7ff, "M"],
    [0xff2d, "M"],
    [0x26f, "m"],
    [0x270, "m"],
    [0x271, "m"],
    [0x1d6f, "m"],
    [0x1d86, "m"],
    [0x1e3f, "m"],
    [0x1e41, "m"],
    [0x1e43, "m"],
    [0x24dc, "m"],
    [0xff4d, "m"],
    [0x24a8, "(m)"],
    [0xd1, "N"],
    [0x143, "N"],
    [0x145, "N"],
    [0x147, "N"],
    [0x14a, "N"],
    [0x19d, "N"],
    [0x1f8, "N"],
    [0x220, "N"],
    [0x274, "N"],
    [0x1d0e, "N"],
    [0x1e44, "N"],
    [0x1e46, "N"],
    [0x1e48, "N"],
    [0x1e4a, "N"],
    [0x24c3, "N"],
    [0xff2e, "N"],
    [0xf1, "n"],
    [0x144, "n"],
    [0x146, "n"],
    [0x148, "n"],
    [0x149, "n"],
    [0x14b, "n"],
    [0x19e, "n"],
    [0x1f9, "n"],
    [0x235, "n"],
    [0x272, "n"],
    [0x273, "n"],
    [0x1d70, "n"],
    [0x1d87, "n"],
    [0x1e45, "n"],
    [0x1e47, "n"],
    [0x1e49, "n"],
    [0x1e4b, "n"],
    [0x207f, "n"],
    [0x24dd, "n"],
    [0xff4e, "n"],
    [0x1ca, "NJ"],
    [0x1cb, "Nj"],
    [0x24a9, "(n)"],
    [0x1cc, "nj"],
    [0xd2, "O"],
    [0xd3, "O"],
    [0xd4, "O"],
    [0xd5, "O"],
    [0xd6, "O"],
    [0xd8, "O"],
    [0x14c, "O"],
    [0x14e, "O"],
    [0x150, "O"],
    [0x186, "O"],
    [0x19f, "O"],
    [0x1a0, "O"],
    [0x1d1, "O"],
    [0x1ea, "O"],
    [0x1ec, "O"],
    [0x1fe, "O"],
    [0x20c, "O"],
    [0x20e, "O"],
    [0x22a, "O"],
    [0x22c, "O"],
    [0x22e, "O"],
    [0x230, "O"],
    [0x1d0f, "O"],
    [0x1d10, "O"],
    [0x1e4c, "O"],
    [0x1e4e, "O"],
    [0x1e50, "O"],
    [0x1e52, "O"],
    [0x1ecc, "O"],
    [0x1ece, "O"],
    [0x1ed0, "O"],
    [0x1ed2, "O"],
    [0x1ed4, "O"],
    [0x1ed6, "O"],
    [0x1ed8, "O"],
    [0x1eda, "O"],
    [0x1edc, "O"],
    [0x1ede, "O"],
    [0x1ee0, "O"],
    [0x1ee2, "O"],
    [0x24c4, "O"],
    [0xa74a, "O"],
    [0xa74c, "O"],
    [0xff2f, "O"],
    [0xf2, "o"],
    [0xf3, "o"],
    [0xf4, "o"],
    [0xf5, "o"],
    [0xf6, "o"],
    [0xf8, "o"],
    [0x14d, "o"],
    [0x14f, "o"],
    [0x151, "o"],
    [0x1a1, "o"],
    [0x1d2, "o"],
    [0x1eb, "o"],
    [0x1ed, "o"],
    [0x1ff, "o"],
    [0x20d, "o"],
    [0x20f, "o"],
    [0x22b, "o"],
    [0x22d, "o"],
    [0x22f, "o"],
    [0x231, "o"],
    [0x254, "o"],
    [0x275, "o"],
    [0x1d16, "o"],
    [0x1d17, "o"],
    [0x1d97, "o"],
    [0x1e4d, "o"],
    [0x1e4f, "o"],
    [0x1e51, "o"],
    [0x1e53, "o"],
    [0x1ecd, "o"],
    [0x1ecf, "o"],
    [0x1ed1, "o"],
    [0x1ed3, "o"],
    [0x1ed5, "o"],
    [0x1ed7, "o"],
    [0x1ed9, "o"],
    [0x1edb, "o"],
    [0x1edd, "o"],
    [0x1edf, "o"],
    [0x1ee1, "o"],
    [0x1ee3, "o"],
    [0x2092, "o"],
    [0x24de, "o"],
    [0x2c7a, "o"],
    [0xa74b, "o"],
    [0xa74d, "o"],
    [0xff4f, "o"],
    [0x152, "OE"],
    [0x276, "OE"],
    [0xa74e, "OO"],
    [0x222, "OU"],
    [0x1d15, "OU"],
    [0x24aa, "(o)"],
    [0x153, "oe"],
    [0x1d14, "oe"],
    [0xa74f, "oo"],
    [0x223, "ou"],
    [0x1a4, "P"],
    [0x1d18, "P"],
    [0x1e54, "P"],
    [0x1e56, "P"],
    [0x24c5, "P"],
    [0x2c63, "P"],
    [0xa750, "P"],
    [0xa752, "P"],
    [0xa754, "P"],
    [0xff30, "P"],
    [0x1a5, "p"],
    [0x1d71, "p"],
    [0x1d7d, "p"],
    [0x1d88, "p"],
    [0x1e55, "p"],
    [0x1e57, "p"],
    [0x24df, "p"],
    [0xa751, "p"],
    [0xa753, "p"],
    [0xa755, "p"],
    [0xa7fc, "p"],
    [0xff50, "p"],
    [0x24ab, "(p)"],
    [0x24a, "Q"],
    [0x24c6, "Q"],
    [0xa756, "Q"],
    [0xa758, "Q"],
    [0xff31, "Q"],
    [0x138, "q"],
    [0x24b, "q"],
    [0x2a0, "q"],
    [0x24e0, "q"],
    [0xa757, "q"],
    [0xa759, "q"],
    [0xff51, "q"],
    [0x24ac, "(q)"],
    [0x239, "qp"],
    [0x154, "R"],
    [0x156, "R"],
    [0x158, "R"],
    [0x210, "R"],
    [0x212, "R"],
    [0x24c, "R"],
    [0x280, "R"],
    [0x281, "R"],
    [0x1d19, "R"],
    [0x1d1a, "R"],
    [0x1e58, "R"],
    [0x1e5a, "R"],
    [0x1e5c, "R"],
    [0x1e5e, "R"],
    [0x24c7, "R"],
    [0x2c64, "R"],
    [0xa75a, "R"],
    [0xa782, "R"],
    [0xff32, "R"],
    [0x155, "r"],
    [0x157, "r"],
    [0x159, "r"],
    [0x211, "r"],
    [0x213, "r"],
    [0x24d, "r"],
    [0x27c, "r"],
    [0x27d, "r"],
    [0x27e, "r"],
    [0x27f, "r"],
    [0x1d63, "r"],
    [0x1d72, "r"],
    [0x1d73, "r"],
    [0x1d89, "r"],
    [0x1e59, "r"],
    [0x1e5b, "r"],
    [0x1e5d, "r"],
    [0x1e5f, "r"],
    [0x24e1, "r"],
    [0xa75b, "r"],
    [0xa783, "r"],
    [0xff52, "r"],
    [0x24ad, "(r)"],
    [0x15a, "S"],
    [0x15c, "S"],
    [0x15e, "S"],
    [0x160, "S"],
    [0x218, "S"],
    [0x1e60, "S"],
    [0x1e62, "S"],
    [0x1e64, "S"],
    [0x1e66, "S"],
    [0x1e68, "S"],
    [0x24c8, "S"],
    [0xa731, "S"],
    [0xa785, "S"],
    [0xff33, "S"],
    [0x15b, "s"],
    [0x15d, "s"],
    [0x15f, "s"],
    [0x161, "s"],
    [0x17f, "s"],
    [0x219, "s"],
    [0x23f, "s"],
    [0x282, "s"],
    [0x1d74, "s"],
    [0x1d8a, "s"],
    [0x1e61, "s"],
    [0x1e63, "s"],
    [0x1e65, "s"],
    [0x1e67, "s"],
    [0x1e69, "s"],
    [0x1e9c, "s"],
    [0x1e9d, "s"],
    [0x24e2, "s"],
    [0xa784, "s"],
    [0xff53, "s"],
    [0x1e9e, "SS"],
    [0x24ae, "(s)"],
    [0xdf, "ss"],
    [0xfb06, "st"],
    [0x162, "T"],
    [0x164, "T"],
    [0x166, "T"],
    [0x1ac, "T"],
    [0x1ae, "T"],
    [0x21a, "T"],
    [0x23e, "T"],
    [0x1d1b, "T"],
    [0x1e6a, "T"],
    [0x1e6c, "T"],
    [0x1e6e, "T"],
    [0x1e70, "T"],
    [0x24c9, "T"],
    [0xa786, "T"],
    [0xff34, "T"],
    [0x163, "t"],
    [0x165, "t"],
    [0x167, "t"],
    [0x1ab, "t"],
    [0x1ad, "t"],
    [0x21b, "t"],
    [0x236, "t"],
    [0x287, "t"],
    [0x288, "t"],
    [0x1d75, "t"],
    [0x1e6b, "t"],
    [0x1e6d, "t"],
    [0x1e6f, "t"],
    [0x1e71, "t"],
    [0x1e97, "t"],
    [0x24e3, "t"],
    [0x2c66, "t"],
    [0xff54, "t"],
    [0xde, "TH"],
    [0xa766, "TH"],
    [0xa728, "TZ"],
    [0x24af, "(t)"],
    [0x2a8, "tc"],
    [0xfe, "th"],
    [0x1d7a, "th"],
    [0xa767, "th"],
    [0x2a6, "ts"],
    [0xa729, "tz"],
    [0xd9, "U"],
    [0xda, "U"],
    [0xdb, "U"],
    [0xdc, "U"],
    [0x168, "U"],
    [0x16a, "U"],
    [0x16c, "U"],
    [0x16e, "U"],
    [0x170, "U"],
    [0x172, "U"],
    [0x1af, "U"],
    [0x1d3, "U"],
    [0x1d5, "U"],
    [0x1d7, "U"],
    [0x1d9, "U"],
    [0x1db, "U"],
    [0x214, "U"],
    [0x216, "U"],
    [0x244, "U"],
    [0x1d1c, "U"],
    [0x1d7e, "U"],
    [0x1e72, "U"],
    [0x1e74, "U"],
    [0x1e76, "U"],
    [0x1e78, "U"],
    [0x1e7a, "U"],
    [0x1ee4, "U"],
    [0x1ee6, "U"],
    [0x1ee8, "U"],
    [0x1eea, "U"],
    [0x1eec, "U"],
    [0x1eee, "U"],
    [0x1ef0, "U"],
    [0x24ca, "U"],
    [0xff35, "U"],
    [0xf9, "u"],
    [0xfa, "u"],
    [0xfb, "u"],
    [0xfc, "u"],
    [0x169, "u"],
    [0x16b, "u"],
    [0x16d, "u"],
    [0x16f, "u"],
    [0x171, "u"],
    [0x173, "u"],
    [0x1b0, "u"],
    [0x1d4, "u"],
    [0x1d6, "u"],
    [0x1d8, "u"],
    [0x1da, "u"],
    [0x1dc, "u"],
    [0x215, "u"],
    [0x217, "u"],
    [0x289, "u"],
    [0x1d64, "u"],
    [0x1d99, "u"],
    [0x1e73, "u"],
    [0x1e75, "u"],
    [0x1e77, "u"],
    [0x1e79, "u"],
    [0x1e7b, "u"],
    [0x1ee5, "u"],
    [0x1ee7, "u"],
    [0x1ee9, "u"],
    [0x1eeb, "u"],
    [0x1eed, "u"],
    [0x1eef, "u"],
    [0x1ef1, "u"],
    [0x24e4, "u"],
    [0xff55, "u"],
    [0x24b0, "(u)"],
    [0x1d6b, "ue"],
    [0x1b2, "V"],
    [0x245, "V"],
    [0x1d20, "V"],
    [0x1e7c, "V"],
    [0x1e7e, "V"],
    [0x1efc, "V"],
    [0x24cb, "V"],
    [0xa75e, "V"],
    [0xa768, "V"],
    [0xff36, "V"],
    [0x28b, "v"],
    [0x28c, "v"],
    [0x1d65, "v"],
    [0x1d8c, "v"],
    [0x1e7d, "v"],
    [0x1e7f, "v"],
    [0x24e5, "v"],
    [0x2c71, "v"],
    [0x2c74, "v"],
    [0xa75f, "v"],
    [0xff56, "v"],
    [0xa760, "VY"],
    [0x24b1, "(v)"],
    [0xa761, "vy"],
    [0x174, "W"],
    [0x1f7, "W"],
    [0x1d21, "W"],
    [0x1e80, "W"],
    [0x1e82, "W"],
    [0x1e84, "W"],
    [0x1e86, "W"],
    [0x1e88, "W"],
    [0x24cc, "W"],
    [0x2c72, "W"],
    [0xff37, "W"],
    [0x175, "w"],
    [0x1bf, "w"],
    [0x28d, "w"],
    [0x1e81, "w"],
    [0x1e83, "w"],
    [0x1e85, "w"],
    [0x1e87, "w"],
    [0x1e89, "w"],
    [0x1e98, "w"],
    [0x24e6, "w"],
    [0x2c73, "w"],
    [0xff57, "w"],
    [0x24b2, "(w)"],
    [0x1e8a, "X"],
    [0x1e8c, "X"],
    [0x24cd, "X"],
    [0xff38, "X"],
    [0x1d8d, "x"],
    [0x1e8b, "x"],
    [0x1e8d, "x"],
    [0x2093, "x"],
    [0x24e7, "x"],
    [0xff58, "x"],
    [0x24b3, "(x)"],
    [0xdd, "Y"],
    [0x176, "Y"],
    [0x178, "Y"],
    [0x1b3, "Y"],
    [0x232, "Y"],
    [0x24e, "Y"],
    [0x28f, "Y"],
    [0x1e8e, "Y"],
    [0x1ef2, "Y"],
    [0x1ef4, "Y"],
    [0x1ef6, "Y"],
    [0x1ef8, "Y"],
    [0x1efe, "Y"],
    [0x24ce, "Y"],
    [0xff39, "Y"],
    [0xfd, "y"],
    [0xff, "y"],
    [0x177, "y"],
    [0x1b4, "y"],
    [0x233, "y"],
    [0x24f, "y"],
    [0x28e, "y"],
    [0x1e8f, "y"],
    [0x1e99, "y"],
    [0x1ef3, "y"],
    [0x1ef5, "y"],
    [0x1ef7, "y"],
    [0x1ef9, "y"],
    [0x1eff, "y"],
    [0x24e8, "y"],
    [0xff59, "y"],
    [0x24b4, "(y)"],
    [0x179, "Z"],
    [0x17b, "Z"],
    [0x17d, "Z"],
    [0x1b5, "Z"],
    [0x21c, "Z"],
    [0x224, "Z"],
    [0x1d22, "Z"],
    [0x1e90, "Z"],
    [0x1e92, "Z"],
    [0x1e94, "Z"],
    [0x24cf, "Z"],
    [0x2c6b, "Z"],
    [0xa762, "Z"],
    [0xff3a, "Z"],
    [0x17a, "z"],
    [0x17c, "z"],
    [0x17e, "z"],
    [0x1b6, "z"],
    [0x21d, "z"],
    [0x225, "z"],
    [0x240, "z"],
    [0x290, "z"],
    [0x291, "z"],
    [0x1d76, "z"],
    [0x1d8e, "z"],
    [0x1e91, "z"],
    [0x1e93, "z"],
    [0x1e95, "z"],
    [0x24e9, "z"],
    [0x2c6c, "z"],
    [0xa763, "z"],
    [0xff5a, "z"],
    [0x24b5, "(z)"],
    [0x2070, "0"],
    [0x2080, "0"],
    [0x24ea, "0"],
    [0x24ff, "0"],
    [0xff10, "0"],
    [0xb9, "1"],
    [0x2081, "1"],
    [0x2460, "1"],
    [0x24f5, "1"],
    [0x2776, "1"],
    [0x2780, "1"],
    [0x278a, "1"],
    [0xff11, "1"],
    [0x2488, "1."],
    [0x2474, "(1)"],
    [0xb2, "2"],
    [0x2082, "2"],
    [0x2461, "2"],
    [0x24f6, "2"],
    [0x2777, "2"],
    [0x2781, "2"],
    [0x278b, "2"],
    [0xff12, "2"],
    [0x2489, "2."],
    [0x2475, "(2)"],
    [0xb3, "3"],
    [0x2083, "3"],
    [0x2462, "3"],
    [0x24f7, "3"],
    [0x2778, "3"],
    [0x2782, "3"],
    [0x278c, "3"],
    [0xff13, "3"],
    [0x248a, "3."],
    [0x2476, "(3)"],
    [0x2074, "4"],
    [0x2084, "4"],
    [0x2463, "4"],
    [0x24f8, "4"],
    [0x2779, "4"],
    [0x2783, "4"],
    [0x278d, "4"],
    [0xff14, "4"],
    [0x248b, "4."],
    [0x2477, "(4)"],
    [0x2075, "5"],
    [0x2085, "5"],
    [0x2464, "5"],
    [0x24f9, "5"],
    [0x277a, "5"],
    [0x2784, "5"],
    [0x278e, "5"],
    [0xff15, "5"],
    [0x248c, "5."],
    [0x2478, "(5)"],
    [0x2076, "6"],
    [0x2086, "6"],
    [0x2465, "6"],
    [0x24fa, "6"],
    [0x277b, "6"],
    [0x2785, "6"],
    [0x278f, "6"],
    [0xff16, "6"],
    [0x248d, "6."],
    [0x2479, "(6)"],
    [0x2077, "7"],
    [0x2087, "7"],
    [0x2466, "7"],
    [0x24fb, "7"],
    [0x277c, "7"],
    [0x2786, "7"],
    [0x2790, "7"],
    [0xff17, "7"],
    [0x248e, "7."],
    [0x247a, "(7)"],
    [0x2078, "8"],
    [0x2088, "8"],
    [0x2467, "8"],
    [0x24fc, "8"],
    [0x277d, "8"],
    [0x2787, "8"],
    [0x2791, "8"],
    [0xff18, "8"],
    [0x248f, "8."],
    [0x247b, "(8)"],
    [0x2079, "9"],
    [0x2089, "9"],
    [0x2468, "9"],
    [0x24fd, "9"],
    [0x277e, "9"],
    [0x2788, "9"],
    [0x2792, "9"],
    [0xff19, "9"],
    [0x2490, "9."],
    [0x247c, "(9)"],
    [0x2469, "10"],
    [0x24fe, "10"],
    [0x277f, "10"],
    [0x2789, "10"],
    [0x2793, "10"],
    [0x2491, "10."],
    [0x247d, "(10)"],
    [0x246a, "11"],
    [0x24eb, "11"],
    [0x2492, "11."],
    [0x247e, "(11)"],
    [0x246b, "12"],
    [0x24ec, "12"],
    [0x2493, "12."],
    [0x247f, "(12)"],
    [0x246c, "13"],
    [0x24ed, "13"],
    [0x2494, "13."],
    [0x2480, "(13)"],
    [0x246d, "14"],
    [0x24ee, "14"],
    [0x2495, "14."],
    [0x2481, "(14)"],
    [0x246e, "15"],
    [0x24ef, "15"],
    [0x2496, "15."],
    [0x2482, "(15)"],
    [0x246f, "16"],
    [0x24f0, "16"],
    [0x2497, "16."],
    [0x2483, "(16)"],
    [0x2470, "17"],
    [0x24f1, "17"],
    [0x2498, "17."],
    [0x2484, "(17)"],
    [0x2471, "18"],
    [0x24f2, "18"],
    [0x2499, "18."],
    [0x2485, "(18)"],
    [0x2472, "19"],
    [0x24f3, "19"],
    [0x249a, "19."],
    [0x2486, "(19)"],
    [0x2473, "20"],
    [0x24f4, "20"],
    [0x249b, "20."],
    [0x2487, "(20)"],
    [0xab, '"'],
    [0xbb, '"'],
    [0x201c, '"'],
    [0x201d, '"'],
    [0x201e, '"'],
    [0x2033, '"'],
    [0x2036, '"'],
    [0x275d, '"'],
    [0x275e, '"'],
    [0x276e, '"'],
    [0x276f, '"'],
    [0xff02, '"'],
    [0x2018, "'"],
    [0x2019, "'"],
    [0x201a, "'"],
    [0x201b, "'"],
    [0x2032, "'"],
    [0x2035, "'"],
    [0x2039, "'"],
    [0x203a, "'"],
    [0x275b, "'"],
    [0x275c, "'"],
    [0xff07, "'"],
    [0x2010, "-"],
    [0x2011, "-"],
    [0x2012, "-"],
    [0x2013, "-"],
    [0x2014, "-"],
    [0x207b, "-"],
    [0x208b, "-"],
    [0xff0d, "-"],
    [0x2045, "["],
    [0x2772, "["],
    [0xff3b, "["],
    [0x2046, "]"],
    [0x2773, "]"],
    [0xff3d, "]"],
    [0x207d, "("],
    [0x208d, "("],
    [0x2768, "("],
    [0x276a, "("],
    [0xff08, "("],
    [0x2e28, "(("],
    [0x207e, ")"],
    [0x208e, ")"],
    [0x2769, ")"],
    [0x276b, ")"],
    [0xff09, ")"],
    [0x2e29, "))"],
    [0x276c, "<"],
    [0x2770, "<"],
    [0xff1c, "<"],
    [0x276d, ">"],
    [0x2771, ">"],
    [0xff1e, ">"],
    [0x2774, "{"],
    [0xff5b, "{"],
    [0x2775, "}"],
    [0xff5d, "}"],
    [0x207a, "+"],
    [0x208a, "+"],
    [0xff0b, "+"],
    [0x207c, "="],
    [0x208c, "="],
    [0xff1d, "="],
    [0xff01, "!"],
    [0x203c, "!!"],
    [0x2049, "!?"],
    [0xff03, "#"],
    [0xff04, "$"],
    [0x2052, "%"],
    [0xff05, "%"],
    [0xff06, "&"],
    [0x204e, "*"],
    [0xff0a, "*"],
    [0xff0c, ","],
    [0xff0e, "."],
    [0x2044, "/"],
    [0xff0f, "/"],
    [0xff1a, ":"],
    [0x204f, ";"],
    [0xff1b, ";"],
    [0xff1f, "?"],
    [0x2047, "??"],
    [0x2048, "?!"],
    [0xff20, "@"],
    [0xff3c, "\\"],
    [0x2038, "^"],
    [0xff3e, "^"],
    [0xff3f, "_"],
    [0x2053, "~"],
    [0xff5e, "~"],
  ]);
}

export default ASCIIFolder;
