#!/usr/bin/env python3

import os
import tarfile
import requests
import scipy.io
import pickle
import json

from collections import defaultdict
from PIL import Image
from torch.utils.data import Dataset

from learn2learn.data.utils import download_file

DATA_DIR = 'fgvc_fungi'
DATA_URL = 'https://labs.gbif.org/fgvcx/2018/fungi_train_val.tgz'
ANNOTATIONS_URL = 'https://labs.gbif.org/fgvcx/2018/train_val_annotations.tgz'

# Splits from "Meta-Datasets", Triantafillou et al, 2020
SPLITS = {
    'train': [
        '0226.Cortinarius malicorius',
        '0283.Cystoderma jasonis',
        '0256.Cortinarius triumphans',
        '0334.Entoloma sepium',
        '0919.Resinicium bicolor',
        '0267.Crepidotus applanatus',
        '1378.Cerioporus varius',
        '1258.Coprinopsis romagnesiana',
        '1025.Spathularia flavida',
        '0144.Clavulinopsis luteoalba',
        '0856.Pluteus ephebeus',
        '0811.Phellinus lundellii',
        '0114.Camarops polysperma',
        '0255.Cortinarius traganus',
        '0585.Lactarius hepaticus',
        '0599.Lactarius romagnesii',
        '1053.Suillus flavidus',
        '1263.Coprinopsis stercorea',
        '0422.Helvella lacunosa',
        '0049.Antrodia serialis',
        '1156.Armillaria lutea',
        '1054.Suillus granulatus',
        '0721.Mycena capillaris',
        '0678.Marasmius bulliardii',
        '0538.Inocybe lanuginosa',
        '1058.Syzygospora tumefaciens',
        '0860.Pluteus nanus',
        '0206.Cortinarius comptulus',
        '0452.Cuphophyllus flavipes',
        '0611.Lactifluus volemus',
        '1306.Lactarius azonites',
        '0751.Mycena sanguinolenta',
        '0753.Mycena speirea',
        '0178.Cheilymenia granulata',
        '1140.Cladonia scabriuscula',
        '0385.Gloeophyllum sepiarium',
        '0668.Macrocystidia cucumis',
        '1217.Parasola plicatilis',
        '0734.Mycena haematopus',
        '0894.Psathyrella pseudogracilis',
        '0451.Hygrocybe conica var. conica',
        '0042.Amanita strobiliformis',
        '0458.Hygrocybe intermedia',
        '1010.Scleroderma areolatum',
        '0497.Hypholoma capnoides',
        '0652.Leucogyrophana mollusca',
        '0284.Cystolepiota adulterina',
        '0030.Amanita crocea',
        '0934.Russula alnetorum',
        '0373.Ganoderma lucidum',
        '0140.Clavulina rugosa',
        '0347.Exidia saccharina',
        '0804.Phaeomarasmius erinaceus',
        '0037.Amanita muscaria',
        '0040.Amanita porphyria',
        '0160.Clitocybe trulliformis',
        '0643.Lepiota subincarnata',
        '0829.Pholiota flammans',
        '0858.Pluteus leoninus',
        '0111.Calyptella capula',
        '0925.Rhytisma acerinum',
        '0767.Nodulisporium cecidiogenes',
        '0499.Hypholoma fasciculare',
        '0269.Crepidotus cesatii',
        '0041.Amanita rubescens var. rubescens',
        '0298.Diatrype disciformis',
        '0513.Hypoxylon howeanum',
        '0775.Mucidula mucida',
        '0215.Cortinarius flexipes',
        '1383.Amyloporia sinuosa',
        '0336.Entoloma sericellum',
        '0411.Hapalopilus rutilans',
        '0630.Lepiota boudieri',
        '1368.Neoboletus luridiformis var. pseudosulphureus',
        '0791.Peniophora lycii',
        '0625.Lentinellus cochleatus',
        '1071.Trametes ochracea',
        '1392.Aureoboletus projectellus',
        '0988.Russula sanguinea',
        '0929.Ripartites tricholoma',
        '0720.Mycena capillaripes',
        '0445.Hygrocybe cantharellus',
        '0361.Fuligo septica',
        '0821.Phlebia rufa',
        '1056.Suillus luteus',
        '1226.Rhodocollybia butyracea f. asema',
        '0738.Mycena leptocephala',
        '0116.Cantharellus amethysteus',
        '1361.Paralepista flaccida',
        '0068.Auricularia mesenterica',
        '1088.Tricholoma basirubens',
        '0471.Hygrocybe spadicea',
        '0406.Gymnosporangium clavariiforme',
        '0168.Coltricia perennis',
        '0710.Mucronella flava',
        '1188.Macrolepiota fuliginosa',
        '0818.Phellodon tomentosus',
        '1315.Hydnum ellipsosporum',
        '1078.Trichaptum abietinum',
        '0769.Otidea alutacea',
        '1167.Cortinarius xanthocephalus',
        '1307.Melanoleuca exscissa',
        '0440.Hydnum rufescens',
        '1212.Taphrina amentorum',
        '1141.Evernia prunastri',
        '0646.Lepista sordida',
        '0967.Russula laeta',
        '1017.Serpula himantioides',
        '0911.Ramaria botrytis',
        '0481.Hygrophorus eburneus',
        '0544.Inocybe sindonia',
        '1309.Pluteus primus',
        '0859.Pluteus luctuosus',
        '0237.Cortinarius phoeniceus',
        '0360.Fuligo septica var. rufa',
        '0109.Calocybe gambosa',
        '0902.Deconica inquilina',
        '0993.Russula undulata',
        '1069.Trametes gibbosa',
        '0844.Physisporinus vitreus',
        '0650.Leucoagaricus leucothites',
        '1330.Hymenoscyphus fraxineus',
        '0437.Hydnellum ferrugineum',
        '0770.Otidea concinna',
        '1369.Cyclocybe erebia',
        '1251.Hygrocybe conicoides',
        '0732.Mycena galericulata',
        '1182.Kuehneromyces mutabilis',
        '0566.Lactarius acerrimus',
        '0130.Chlorociboria aeruginascens',
        '1168.Dermoloma pseudocuneifolium',
        '0031.Amanita excelsa',
        '0900.Pseudohydnum gelatinosum',
        '0402.Gymnopus fusipes',
        '1080.Trichoglossum hirsutum',
        '0323.Entoloma lividoalbum',
        '0199.Cortinarius casimiri',
        '0161.Clitocybe vibecina',
        '1283.Amanita vaginata',
        '1125.Verpa conica',
        '1370.Pseudoinonotus dryadeus',
        '0330.Entoloma prunuloides',
        '0979.Russula pectinatoides',
        '0048.Antrodia malicola',
        '1114.Tubaria conspersa',
        '0188.Cortinarius aprinus',
        '0155.Clitocybe odora var. odora ',
        '1236.Sarcomyxa serotina',
        '0617.Leccinum duriusculum',
        '1055.Suillus grevillei var. grevillei ',
        '0692.Melanoleuca cinereifolia',
        '0895.Psathyrella pygmaea',
        '0086.Xerocomellus chrysenteron',
        '0344.Exidia glandulosa',
        '0376.Geastrum fimbriatum',
        '1000.Russula violeipes',
        '0519.Inocybe agardhii',
        '0601.Lactarius rufus',
        '0238.Cortinarius quarciticus',
        '0084.Hemileccinum impolitum',
        '1065.Thelephora caryophyllea',
        '1052.Suillus collinitus',
        '1154.Xanthoria parietina',
        '0998.Russula veternosa',
        '1144.Lecanora chlarotera',
        '0863.Pluteus phlebophorus',
        '0548.Inonotus cuticularis',
        '0928.Rickenella swartzii',
        '1127.Volvariella hypopithys',
        '1090.Tricholoma cingulatum',
        '1243.Meottomyces dissimulans',
        '0281.Cystoderma carcharias',
        '1049.Leratiomyces squamosus',
        '1202.Propolis farinosa',
        '1147.Lobaria pulmonaria',
        '0446.Hygrocybe ceracea',
        '0125.Ceriporia reticulata',
        '1045.Leratiomyces percevalii',
        '1135.Xylaria carpophila',
        '1239.Hydnum umbilicatum ss auct. eur.',
        '0675.Marasmiellus vaillantii',
        '0841.Phyllotopsis nidulans',
        '0547.Inocybe whitei',
        '1041.Stropharia coronilla',
        '1254.Coprinellus flocculosus',
        '0059.Ascodichaena rugosa',
        '1172.Galerina uncialis',
        '1213.Tephrocybe rancida',
        '0660.Lycoperdon molle',
        '0508.Hypomyces aurantius',
        '1329.Tricholoma inocybeoides',
        '0443.Hygrocybe aurantiosplendens',
        '0612.Laetiporus sulphureus',
        '1271.Agrocybe rivulosa',
        '0456.Neohygrocybe ingrata',
        '0850.Pleurotus ostreatus',
        '0610.Lactarius vietus',
        '0177.Coprinopsis kubickae',
        '1119.Tulostoma brumale',
        '0233.Cortinarius pholideus',
        '0757.Mycena vulgaris',
        '0515.Annulohypoxylon multiforme',
        '1108.Tricholoma sulphureum',
        '1021.Skeletocutis amorpha',
        '0319.Entoloma hebes',
        '0511.Hypoxylon fragiforme',
        '1051.Suillus bovinus',
        '0057.Ascocoryne cylichnium',
        '0120.Capitotricha bicolor',
        '1324.Mycena tenerrima',
        '0340.Entoloma turbidum',
        '0569.Lactarius aspideus',
        '1222.Parasola schroeteri',
        '1073.Trechispora hymenocystis',
        '0473.Hygrocybe turunda',
        '0089.Xerocomellus pruinatus',
        '0852.Plicatura crispa',
        '0108.Calocera viscosa',
        '0924.Rhodocollybia maculata',
        '1001.Russula virescens',
        '0777.Oxyporus populinus',
        '0756.Mycena vitilis',
        '0230.Cortinarius olearioides',
        '0636.Lepiota erminea',
        '1171.Galerina sideroides',
        '1161.Chrysomphalina grossula',
        '0903.Deconica montana',
        '1101.Tricholoma psammopus',
        '1223.Gamundia striatula',
        '0447.Hygrocybe chlorophana',
        '0744.Mycena pseudocorticola',
        '0662.Lycoperdon perlatum',
        '0562.Lachnella alboviolascens',
        '0512.Hypoxylon fuscum',
        '0261.Cortinarius vernus',
        '0252.Cortinarius talus',
        '1229.Chlorophyllum rhacodes',
        '0183.Cortinarius acutus',
        '0107.Calocera furcata',
        '0157.Clitocybe phyllophila',
        '0687.Rhizomarasmius setosus',
        '0143.Clavulinopsis laeticolor',
        '0243.Cortinarius saniosus',
        '0214.Cortinarius elegantissimus',
        '0350.Exidiopsis grisea',
        '0257.Cortinarius trivialis',
        '1160.Calocybe ionides',
        '1272.Hygrocybe coccinea',
        '1351.Sidera vulgaris',
        '0113.Camarophyllopsis schulzeri',
        '0952.Russula emetica',
        '0693.Melanoleuca cognata',
        '1372.Lentinus substrictus',
        '0240.Cortinarius rubellus',
        '0673.Macrotyphula juncea',
        '0167.Coltricia confluens',
        '0364.Galerina hypnorum',
        '0571.Lactarius azonites f. azonites ',
        '0958.Russula gracillima',
        '0429.Hohenbuehelia auriscalpium',
        '0331.Entoloma rhodocalyx',
        '0749.Mycena rosea',
        '0173.Conocybe semiglobata',
        '0171.Conocybe albipes',
        '0664.Lyophyllum decastes',
        '0886.Psathyrella corrugis',
        '1102.Tricholoma saponaceum var. saponaceum ',
        '0076.Bjerkandera adusta',
        '0532.Inocybe fuscidula',
        '0822.Phlebia tremellosa',
        '0280.Cystoderma amianthinum',
        '0638.Lepiota grangei',
        '1035.Stereum subtomentosum',
        '0588.Lactarius mammosus',
        '0752.Mycena silvae-nigrae',
        '0595.Lactarius pubescens',
        '0043.Amanita submembranacea',
        '1207.Russula caerulea',
        '1038.Strobilurus stephanocystis',
        '0819.Phlebia fuscoatra',
        '0147.Infundibulicybe costata',
        '1208.Russula claroflava',
        '0035.Amanita lividopallescens',
        '0051.Armillaria mellea',
        '0930.Rosellinia corticium',
        '0164.Collybia cirrhata',
        '0941.Russula brunneoviolacea',
        '0365.Galerina graminea',
        '1081.Tricholoma aestuans',
        '0317.Entoloma formosum',
        '0870.Pluteus thomsonii',
        '0918.Ramariopsis kunzei',
        '0014.Agaricus litoralis',
        '0773.Otidea phlebophora',
        '0951.Russula clavipes',
        '0524.Inocybe cincinnata var. major',
        '1260.Coprinopsis acuminata',
        '0984.Russula risigallina',
        '1296.Psathyrella bipellis',
        '0594.Lactarius pterosporus',
        '1327.Ciboria amentacea',
        '0709.Morchella semilibera',
        '1365.Granulobasidium vellereum',
        '1174.Gyromitra gigas',
        '0526.Inocybe corydalina',
        '0461.Hygrocybe mucronella',
        '0908.Pucciniastrum areolatum',
        '1389.Xylodon nespori',
        '0388.Glyphium elatum',
        '0465.Cuphophyllus berkeleyi',
        '1377.Cerioporus squamosus',
        '1290.Entoloma chalybaeum',
        '1277.Cortinarius luhmannii',
        '1298.Leccinum cyaneobasileucum',
        '1004.Rutstroemia firma',
        '0797.Peziza badia',
        '1337.Rhodocybe gemina',
        '0813.Phellinus populicola',
        '0228.Cortinarius multiformis',
        '1269.Mycetinis scorodonius',
        '0324.Entoloma mougeotii',
        '0634.Lepiota cristata',
        '1039.Strobilurus tenacellus',
        '0448.Hygrocybe citrinovirens',
        '0311.Entoloma chalybaeum var. chalybaeum ',
        '0774.Otidea bufonia',
        '0504.Hypholoma udum',
        '0415.Hebeloma sacchariolens',
        '0146.Clitocybe alexandri',
        '1385.Phellodon connatus',
        '0367.Galerina paludosa',
        '0476.Gloioxanthomyces vitellinus',
        '1339.Paxillus ammoniavirescens',
        '0676.Mycetinis alliaceus',
        '0308.Entoloma asprellum',
        '0075.Bisporella pallescens',
        '0897.Psathyrella spadiceogrisea',
        '0275.Cudonia circinans',
        '0723.Mycena cinerella',
        '0202.Cortinarius cinnamomeus',
        '1267.Xerocomellus cisalpinus',
        '0748.Mycena renati',
        '0241.Cortinarius rufo-olivaceus',
        '0954.Russula farinipes',
        '0837.Pholiota tuberculosa',
        '0572.Lactarius blennius',
        '0809.Fuscoporia ferruginosa',
        '0551.Inonotus obliquus',
        '1215.Tricholoma apium',
        '0576.Lactarius controversus',
        '0916.Ramaria stricta',
        '0391.Gomphidius roseus',
        '1326.Ceratiomyxa fruticulosa var. porioides',
        '0754.Mycena stipata',
        '0785.Panellus stipticus',
        '0278.Cyathus striatus',
        '0707.Morchella conica',
        '0867.Pluteus roseipes',
        '0691.Melanamphora spinifera',
        '1393.Ombrophila pura',
        '0272.Crepidotus mollis',
        '1195.Mycena tintinnabulum',
        '0629.Echinoderma asperum',
        '0847.Pithya vulgaris',
        '0507.Hypocreopsis lichenoides',
        '0920.Resupinatus applicatus',
        '0343.Eutypa spinosa',
        '0579.Lactarius deterrimus',
        '0135.Clavaria argillacea',
        '0218.Cortinarius helvelloides',
        '1113.Trichopeziza leucophaea',
        '0846.Pithya cupressina',
        '0557.Laccaria amethystina',
        '0771.Otidea leporina',
        '0067.Auricularia auricula-judae',
        '1085.Tricholoma arvernense',
        '0060.Ascotremella faginea',
        '0832.Pholiota jahnii',
        '0348.Exidia thuretiana',
        '0510.Annulohypoxylon cohaerens',
        '1158.Byssomerulius corium',
        '0789.Peniophora incarnata',
        '0050.Antrodiella serpula',
        '1178.Hygrophoropsis pallida',
        '0810.Phellinus igniarius',
        '1286.Cortinarius lepistoides',
        '1384.Sutorius luridiformis',
        '1270.Roridomyces roridus',
        '0450.Cuphophyllus colemannianus',
        '0922.Rhizina undulata',
        '0839.Phragmidium mucronatum',
        '0879.Postia alni',
        '1013.Scleroderma verrucosum',
        '0438.Hydnum albidum',
        '0699.Deconica horizontalis',
        '0597.Lactarius quieticolor',
        '0150.Clitocybe fragrans',
        '1089.Tricholoma batschii',
        '0872.Polyporus tuberaster',
        '0358.Fomitopsis pinicola',
        '0065.Aurantiporus fissilis',
        '0200.Cortinarius bergeronii',
        '0201.Cortinarius cinnabarinus',
        '1118.Tulasnella violea',
        '0377.Geastrum pectinatum',
        '0485.Hygrophorus penarius',
        '1149.Peltigera didactyla',
        '0591.Lactarius pallidus',
        '1072.Trametes versicolor',
        '1166.Cortinarius puniceus',
        '0950.Russula densifolia',
        '1152.Physcia tenella',
        '0923.Rhizopogon obtextus',
        '0002.Agaricus altipes',
        '1148.Parmelia sulcata',
        '1345.Amanita citrina',
        '0455.Hygrocybe helobia',
        '0189.Cortinarius armillatus',
        '0506.Hypocrea pulvinata',
        '0931.Russula acrifolia',
        '1048.Protostropharia semiglobata',
        '0631.Lepiota brunneoincarnata',
        '0314.Entoloma euchroum',
        '0514.Hypoxylon macrocarpum',
        '1074.Tremella encephala',
        '0271.Crepidotus luteolus',
        '1159.Calocybe carnea',
        '0407.Gyrodon lividus',
        '0208.Cortinarius croceocaeruleus',
        '1103.Tricholoma scalpturatum',
        '0969.Russula mairei',
        '0944.Russula chloroides',
        '1316.Antrodiella faginea',
        '0890.Psathyrella microrrhiza',
        '0299.Diatrypella quercina',
        '1322.Arrhenia peltigerina',
        '0697.Melanoleuca polioleuca f. pusilla',
        '1304.Inocybe cincinnata',
        '0639.Lepiota griseovirens',
        '1079.Trichaptum fuscoviolaceum',
        '0881.Psathyrella ammophila',
        '0426.Hericium coralloides',
        '0098.Bovista nigrescens',
        '0823.Phlebia uda',
        '0868.Pluteus salicinus',
        '0198.Cortinarius caperatus',
        '0876.Oligoporus guttulatus',
        '1192.Mycena epipterygia var. epipterigioides',
        '0072.Basidioradulum radula',
        '0270.Crepidotus lundellii',
        '0681.Marasmius epiphylloides',
        '0468.Hygrocybe quieta',
        '0490.Hymenoscyphus fructigenus',
        '0012.Agaricus langei',
        '0939.Russula aurea',
        '0192.Cortinarius biformis',
        '1187.Lepista panaeolus',
        '0786.Paxillus filamentosus',
        '0025.Agrocybe pediades',
        '1219.Parasola kuehneri',
        '1023.Skeletocutis nivea',
        '0878.Oligoporus stipticus',
        '0972.Russula nigricans',
        '1011.Scleroderma bovista',
        '0874.Postia caesia',
        '0913.Ramaria formosa',
        '1059.Taphrina betulina',
        '0974.Russula ochroleuca',
        '0540.Inocybe mixtilis',
        '0568.Lactarius albocarneus',
        '1256.Coprinellus domesticus',
        '1070.Trametes hirsuta',
        '0830.Pholiota gummosa',
        '1268.Hypoxylon petriniae',
        '1037.Strobilurus esculentus',
        '0127.Cerrena unicolor',
        '0633.Lepiota clypeolaria',
        '1066.Thelephora palmata',
        '0082.Caloboletus calopus',
        '0556.Kretzschmaria deusta',
        '1064.Thelephora anthocephala',
        '0978.Russula parazurea',
        '1336.Rhodocybe caelata',
        '0656.Lycogala flavofuscum',
        '1191.Morchella vulgaris',
        '0436.Hydnellum cumulatum',
        '1302.Hygrocybe conica',
        '1288.Agaricus essettei',
        '0655.Lycogala epidendrum',
        '0478.Hygrophorus agathosmus',
        '1162.Clavaria falcata',
        '0669.Macrolepiota excoriata',
        '1364.Peniophorella pubera',
        '0495.Hyphodontia alutaria',
        '0106.Calocera cornea',
        '0606.Lactarius tabidus',
        '0806.Phallus impudicus',
        '0274.Crinipellis scabella',
        '1325.Pleurotus ostreatus var. praecox',
        '1334.Inocybe rimosa',
        '0333.Entoloma scabropellis',
        '1047.Stropharia rugosoannulata',
        '0423.Helvella leucomelaena',
        '1390.Xylodon paradoxus',
        '0985.Russula romellii',
        '0836.Pholiota squarrosa',
        '0559.Laccaria laccata',
        '0139.Claviceps purpurea',
        '1196.Neolentinus lepideus',
        '1331.Frantisekia mentschulensis',
        '0122.Ceratiomyxa fruticulosa',
        '0439.Hydnum repandum',
        '0384.Geoglossum umbratile',
        '1124.Lycoperdon pratense',
        '1342.Suillellus queletii',
        '0474.Cuphophyllus virgineus',
        '0300.Dichomitus campestris',
        '1005.Sarcodon imbricatus',
        '1029.Steccherinum ochraceum',
        '0442.Hygrocybe acutoconica',
        '1285.Cortinarius emollitoides',
        '0119.Cantharellus pallens',
        '0005.Agaricus bernardii',
        '0750.Mycena rubromarginata',
        '0251.Cortinarius tabularis',
        '0772.Otidea onotica',
        '0558.Laccaria bicolor',
        '0079.Butyriboletus appendiculatus',
        '0522.Inocybe bongardii',
        '1200.Phellinus pomaceus',
        '0714.Mycena adonis',
        '1220.Parasola auricoma',
        '0003.Agaricus arvensis',
        '0428.Heterobasidion annosum',
        '0320.Entoloma incanum',
        '0181.Elaphocordyceps ophioglossoides',
        '1044.Stropharia inuncta',
        '0010.Agaricus cupreobrunneus',
        '0115.Camarops tubulina',
        '0899.Pseudoclitocybe expallens',
        '0781.Panaeolus olivaceus',
        '1204.Deconica phyllogena',
        '0362.Galerina cephalotricha',
        '0100.Brevicellicium olivascens',
        '0964.Russula integra',
        '0521.Inocybe asterospora',
        '1091.Tricholoma columbetta',
        '0509.Hypomyces rosellus',
        '0488.Hymenochaete rubiginosa',
        '0981.Russula pseudointegra',
        '0593.Lactarius porninsis',
        '0394.Gymnopilus picreus',
        '0302.Cinereomyces lindbladii',
        '1300.Inocybe stellatospora',
        '1142.Hypogymnia physodes',
        '0293.Datronia mollis',
        '1186.Lepiota magnispora',
        '0346.Exidia recisa',
        '0203.Cortinarius citrinus',
        '0795.Perenniporia fraxinea',
        '0719.Mycena belliae',
        '0965.Russula ionochlora',
        '0011.Agaricus impudicus',
        '1109.Tricholoma terreum',
        '0235.Cortinarius psammocephalus',
        '0942.Russula carpini',
        '1391.Xylodon raduloides',
        '0654.Limacella guttata',
        '0975.Russula odorata',
        '0843.Physisporinus sanguinolentus',
        '0824.Phlebiopsis gigantea',
        '0545.Inocybe soluta',
        '0029.Amanita ceciliae',
        '1323.Gymnopus fuscopurpureus',
        '0196.Cortinarius calochrous',
        '0090.Cyanoboletus pulverulentus',
        '0683.Marasmius limosus',
        '1274.Loreleia marchantia',
        '1201.Pholiota adiposa',
        '0166.Colpoma quercinum',
        '0861.Pluteus pellitus',
        '0840.Phylloporia ribis',
        '1040.Stropharia aeruginosa',
        '0647.Leptosphaeria acuta',
        '0018.Agaricus sylvaticus',
        '0033.Amanita fulva',
        '0195.Cortinarius caesiocortinatus',
        '1177.Helvella macropus',
        '0883.Psathyrella candolleana',
        '0582.Lactarius fulvissimus',
        '1099.Tricholoma populinum',
        '1245.Hebeloma laterinum',
        '0392.Grifola frondosa',
        '1287.Cortinarius multiformium',
        '0368.Galerina pumila',
        '0137.Clavaria zollingeri',
        '0136.Clavaria fragilis',
        '1183.Lacrymaria lacrymabunda',
        '1248.Bolbitius reticulatus f. aleuriatus',
        '1155.Amanita citrina var. alba',
        '0896.Homophron spadiceum',
        '0891.Psathyrella multipedata',
        '0704.Gymnopus foetidus',
        '0400.Gymnopus erythropus',
        '0387.Meruliopsis taxicola',
        '0454.Hygrocybe glutinipes',
        '0626.Lentinellus ursinus',
        '0264.Craterellus cornucopioides',
        '0628.Leotia lubrica',
        '0112.Calyptella gibbosa',
        '0379.Geastrum schmidelii',
        '1295.Inocybe perlata',
        '0995.Russula velutipes',
        '0295.Dendrothele acerina',
        '0290.Daedalea quercina',
        '1343.Rubroboletus satanas',
        '1218.Parasola lactea',
        '1340.Mycena luteovariegata',
        '0679.Marasmius cohaerens',
        '0399.Gymnopus dryophilus',
        '0546.Inocybe subcarpta',
        '0351.Exobasidium vaccinii',
        '0711.Mutinus caninus',
        '0801.Peziza vesiculosa',
        '0097.Botryobasidium subcoronatum',
        '0322.Entoloma lanuginosipes',
        '0990.Russula solaris',
        '1146.Lecidella elaeochroma',
        '0613.Langermannia gigantea',
        '0263.Cortinarius vulpinus',
        '1371.Lentinus brumalis',
        '0403.Gymnopus hariolorum',
        '0337.Entoloma sericeum var. sericeum ',
        '0249.Cortinarius subpurpurascens',
        '0020.Agaricus subfloccosus',
        '1317.Cortinarius sphagnophilus',
        '1003.Russula xerampelina',
        '0023.Agrocybe arvalis',
        '1095.Tricholoma lascivum',
        '0222.Cortinarius limonius',
        '0543.Inocybe rimosa',
        '1359.Gymnopus perforans',
        '0170.Coniophora puteana',
        '0467.Hygrocybe punicea',
        '0578.Lactarius deliciosus',
        '1193.Mycena galopus var. nigra',
        '0254.Cortinarius torvus',
        '0614.Laxitextum bicolor',
        '0419.Helvella acetabulum',
        '0244.Cortinarius saturninus',
        '0762.Naucoria salicis',
        '1175.Lycoperdon excipuliforme',
        '0530.Inocybe flocculosa',
        '0242.Cortinarius sanguineus',
        '0131.Chondrostereum purpureum',
        '0632.Lepiota castanea',
        '1332.Callistosporium pinicola',
        '0670.Macrolepiota mastoidea',
        '0887.Psathyrella cotonea',
        '0776.Oxyporus obducens',
        '0194.Cortinarius foetens',
        '0500.Hypholoma lateritium',
        '0943.Russula cessans',
        '0798.Peziza fimeti',
        '0145.Clitocybe agrestis',
        '0312.Entoloma clypeatum',
        '0318.Entoloma griseocyaneum',
        '1276.Melanoleuca brevipes',
        '0623.Leccinum versipelle',
        '0001.Achroomyces disciformis',
        '0039.Amanita phalloides var. phalloides ',
        '1266.Coprinopsis nivea',
        '1009.Schizophyllum commune',
        '1067.Thelephora penicillata',
        '1360.Osmoporus odoratus',
        '0486.Hygrophorus persoonii',
        '0416.Hebeloma sinapizans',
        '0309.Entoloma atrocoeruleum',
        '1082.Tricholoma albobrunneum',
        '0796.Peziza arvernensis',
        '1112.Tricholomopsis rutilans',
        '0219.Cortinarius hemitrichus',
        '0083.Boletus edulis',
        '1387.Lyomyces crustosus',
        '1292.Flammulina velutipes',
        '1130.Vuilleminia coryli',
        '1303.Hygrocybe miniata',
        '0480.Hygrophorus discoxanthus',
        '1356.Volvopluteus gloiocephalus',
        '1319.Illosporiopsis christiansenii',
        '0577.Lactarius cyathuliformis',
        '1225.Pholiotina arrhenii',
        '0986.Russula lepida',
        '0717.Mycena arcangeliana',
        '1170.Galerina hybrida',
        '1311.Russula fragilis',
        '1381.Phaeoclavulina abietina',
        '0307.Reticularia lycoperdon',
        '0648.Leptosporomyces fuscostratus',
        '1347.Amyloporia xantha',
        '0007.Agaricus bitorquis',
        '0221.Cortinarius largus',
        '0306.Encoelia furfuracea',
        '0216.Cortinarius flexipes var. flabellus',
        '1063.Tarzetta cupularis',
        '0671.Macrolepiota procera',
        '0768.Omphalina pyxidata',
        '0906.Pterula multifida',
        '1386.Xylodon brevisetus',
        '0560.Laccaria proxima',
        '0074.Calycina citrina',
        '1042.Stropharia cyanea',
        '1007.Sarcoscypha austriaca',
        '1246.Lichenomphalia umbellifera',
        '0531.Inocybe fraudans',
        '0304.Dumontinia tuberosa',
        '0459.Gliophorus laetus',
        '0081.Hortiboletus bubalinus',
        '1060.Taphrina pruni',
        '0759.Myxarium nucleatum',
        '0615.Leccinum albostipitatum',
        '0197.Cortinarius caninus',
        '0956.Russula foetens',
        '1205.Radulomyces confluens',
        '1310.Pluteus umbrosus',
        '0831.Pholiota highlandensis',
        '0563.Lachnellula subtilissima',
        '0427.Hericium erinaceus',
        '0640.Echinoderma jacobi',
        '1211.Simocybe haustellaris',
        '0016.Agaricus porphyrizon',
        '0375.Ganoderma resinaceum',
        '1367.Hymenochaetopsis tabacina',
        '1358.Clitocella popinalis',
        '1116.Tubaria furfuracea',
        '1020.Sistotrema confluens',
        '0024.Agrocybe dura',
        '0700.Melastiza cornubiensis',
        '0882.Psathyrella fibrillosa',
        '1087.Tricholoma aurantium',
        '0581.Lactarius fluens',
        '0211.Cortinarius decipiens',
        '0590.Lactarius obscuratus',
        '0154.Clitocybe nebularis var. nebularis ',
        '0088.Xerocomellus porosporus',
        '1313.Suillus cavipes',
        '1002.Russula viscida',
        '0297.Diatrype bullata',
        '0380.Geastrum michelianum',
        '0808.Fuscoporia ferrea',
        '0282.Cystodermella granulosa',
        '0586.Lactarius lignyotus',
        '1357.Gliophorus irrigatus',
        '0398.Gymnopus confluens',
        '0207.Cortinarius cotoneus',
        '0021.Agaricus subperonatus',
        '0793.Peniophora quercina',
        '0690.Megacollybia platyphylla',
        '0927.Rickenella fibula',
        '0420.Helvella crispa',
        '0124.Ceriporia purpurea',
        '0716.Mycena amicta',
        '0518.Inocybe adaequata',
        '0825.Phleogena faginea',
        '0621.Leccinum scabrum',
        '0185.Cortinarius alcalinophilus',
        '0421.Helvella elastica',
        '1308.Melanoleuca polioleuca',
        '0725.Mycena crocata',
        '0564.Lachnum impudicum',
        '1046.Stropharia pseudocyanea',
        '0912.Ramaria fagetorum',
        '0287.Dacryobolus karstenii',
        '0047.Amylostereum laevigatum',
        '0204.Cortinarius caerulescens',
        '1164.Clavulinopsis umbrinella',
        '0926.Rhytisma salicinum',
        '0761.Naucoria escharioides',
        '1022.Skeletocutis carneogrisea',
        '1244.Inocybe erinaceomorpha',
        '0645.Lepista nuda',
        '0898.Pseudoclitocybe cyathiformis',
        '0933.Russula aeruginea',
        '0549.Inonotus hispidus',
        '0758.Mycena zephirus',
        '0782.Panaeolus papilionaceus',
        '0842.Physalacria cryptomeriae',
        '1121.Typhula erythropus',
        '0292.Daldinia concentrica',
        '0685.Gymnopus quercophilus',
        '1235.Phylloporus pelletieri',
        '0953.Russula faginea',
        '0489.Hymenoscyphus fagineus',
        '1301.Hemipholiota heteroclita',
        '0778.Panaeolus acuminatus',
        '0352.Fistulina hepatica',
        '0794.Peniophora rufomarginata',
        '1252.Gliophorus perplexus',
        '1353.Leucocybe candicans',
        '0349.Exidiopsis effusa',
        '0828.Pholiota conissans',
        '0570.Lactarius aurantiacus',
        '0303.Disciotis venosa',
        '1241.Gyromitra ancilis',
        '0851.Pleurotus pulmonarius',
        '0736.Mycena inclinata',
        '1289.Cortinarius uliginosus',
        '1210.Schizophyllum amplum',
        '0092.Boletus reticulatus',
        '0056.Arrhenia spathulata',
        '0250.Cortinarius subtortus',
        '1139.Cladonia diversa',
        '0528.Inocybe dulcamara',
        '0174.Conocybe subpubescens',
        '1131.Vuilleminia cystidiata',
        '0833.Pholiota lenta',
        '0763.Naucoria scolecina',
        '0642.Lepiota ochraceofulva',
        '0779.Panaeolus foenisecii',
        '1189.Mycetinis querceus',
        '1061.Tapinella atrotomentosa',
        '0550.Mensularia nodulosa',
        '0262.Cortinarius violaceus',
        '0835.Pholiota scamba',
        '0434.Hydnellum caeruleum',
        '0013.Agaricus lanipes',
        '0589.Lactarius musteus',
        '0386.Gloeoporus dichrous',
        '1163.Clavulina coralloides',
        '1093.Tricholoma fulvum',
        '1050.Suillus viscidus',
        '1209.Russula depallens',
        '0814.Fomitiporia punctata',
        '0225.Cortinarius malachius',
        '0493.Hyphoderma roseocremeum',
        '0193.Cortinarius bolaris',
        '0260.Cortinarius umbrinolens',
        '1232.Lyomyces sambuci',
        '0735.Mycena hiemalis',
        '0619.Leccinum schistophilum',
        '0877.Oligoporus ptychogaster',
        '0268.Crepidotus calolepis',
        '1083.Tricholoma album',
        '0608.Lactarius trivialis',
        '0921.Resupinatus trichotis',
        '0085.Pseudoboletus parasiticus',
        '0305.Elaphomyces muricatus',
        '0186.Cortinarius anomalus',
        '1176.Lycoperdon utriforme',
        '1138.Xylaria polymorpha',
        '0182.Corticium roseum',
        '0746.Mycena pura',
        '1297.Psathyrella rubiginosa',
        '0191.Cortinarius bataillei',
        '1227.Leucoagaricus nympharum',
        '0077.Bjerkandera fumosa',
        '1259.Coprinopsis atramentaria',
        '0128.Chalciporus piperatus',
        '0980.Russula pelargonia',
        '1180.Hymenoscyphus calyculus',
        '0598.Lactarius quietus',
        '0483.Hygrophorus mesotephrus',
        '1275.Melanoleuca friesii',
        '1136.Xylaria hypoxylon',
        '1216.Diatrype decorticata',
        '0449.Hygrocybe coccineocrenata',
        '0363.Galerina clavata',
        '0372.Ganoderma applanatum',
        '0246.Cortinarius fusisporus',
        '0665.Lyophyllum fumosum',
        '0747.Mycena purpureofusca',
        '0959.Russula grata',
        '0745.Mycena pterigena',
        '0740.Mycena olivaceomarginata',
        '0285.Cystolepiota bucknallii',
        '1197.Panaeolus subfirmus',
        '0017.Agaricus dulcidulus',
        '0783.Panaeolus semiovatus',
        '0523.Inocybe cincinnata var. cincinnata',
        '0239.Cortinarius raphanoides',
        '0529.Inocybe erubescens',
        '0405.Gymnopus peronatus',
        '0169.Coniophora arida',
        '0701.Meripilus giganteus',
        '0180.Cordyceps militaris',
        '0213.Cortinarius diasemospermus var. diasemospermus ',
        '0487.Hygrophorus pustulatus',
        '0686.Marasmius rotula',
        '1284.Clitocybe nebularis',
        '0054.Arrhenia lobata',
        '1281.Amanita phalloides',
        '0816.Phellinus tremulae',
        '1355.Butyriboletus fuscoroseus',
        '0961.Russula grisea',
        '1122.Typhula gyrans',
        '0383.Geoglossum fallax',
        '0046.Amylostereum chailletii',
        '1128.Volvariella surrecta',
        '1282.Amanita rubescens',
        '0855.Pluteus cinereofuscus',
        '1198.Panus conchatus',
        '0696.Melanoleuca polioleuca var. polioleuca',
        '0708.Morchella esculenta',
        '0728.Mycena erubescens',
        '0962.Russula heterophylla',
        '0441.Hydropus subalpinus',
        '1335.Hygrocybe acutoconica sensu lato',
        '0525.Inocybe cookei',
        '0026.Agrocybe praecox',
        '1291.Entoloma sericeum',
        '0766.Neottiella rutilans',
        '1179.Hygrophorus unicolor',
        '1068.Thelephora terrestris',
        '0503.Hypholoma radicosum',
        '0066.Aureoboletus gentilis',
        '0936.Russula anthracina',
        '1206.Resinomycena saccharifera',
        '0992.Russula subrubens',
        '0248.Cortinarius subporphyropus',
        '1380.Exidia nigricans',
        '0854.Pluteus cervinus',
        '0212.Cortinarius delibutus',
        '1374.Picipes melanopus',
        '0404.Gymnopus ocior',
        '0553.Inocutis rheades',
        '1333.Cortinarius anomalus',
        '1221.Parasola misera',
        '0412.Hebeloma crustuliniforme',
        '0345.Exidia pithya',
        '0310.Entoloma cetratum',
        '0052.Armillaria ostoyae',
        '0575.Lactarius circellatus',
        '0245.Cortinarius semisanguineus',
        '0516.Hypoxylon rubiginosum',
        '0657.Lycoperdon decipiens',
        '0957.Russula fragilis var. fragilis',
        '0158.Clitocybe rivulosa',
        '0259.Cortinarius uliginosus f. uliginosus ',
        '0339.Entoloma sordidulum',
        '0133.Ciboria caucus',
        '0475.Cuphophyllus ochraceopallidus',
        '0889.Coprinopsis marcescibilis',
        '0672.Macrotyphula fistulosa',
        '0335.Entoloma sericatum',
        '0034.Amanita gemmata',
        '0253.Cortinarius tophaceus',
        '1190.Melanophyllum haematospermum',
        '0123.Ceriporia excelsa',
        '0424.Hemimycena cucullata',
        '0739.Mycena metata',
        '1132.Xeromphalina campanella',
        '0946.Russula curtipes',
        '0338.Entoloma serrulatum',
        '1184.Lacrymaria pyrotricha',
        '0982.Russula puellaris',
        '0094.Xerocomus subtomentosus',
        '0983.Russula queletii',
        '0542.Inocybe petiginosa',
        '0644.Lepista irina',
        '0689.Marasmius wynnei',
        '0477.Hygrophoropsis aurantiaca',
        '0027.Aleuria aurantia',
        '1349.Sagaranella tylicolor',
        '1185.Lactarius necator',
        '0635.Echinoderma echinaceum',
        '1338.Microglossum griseoviride',
        '0172.Conocybe rickenii',
        '0666.Lyophyllum paelochroum',
        '0573.Lactarius camphoratus',
        '0554.Ischnoderma benzoinum',
        '0737.Mycena juniperina',
        '1293.Chlorophyllum brunneum',
        '0864.Pluteus plautus',
        '0288.Dacrymyces lacrymalis',
        '0945.Russula cuprea',
        '0527.Inocybe curvipes',
        '0682.Marasmius epiphyllus',
        '1015.Scutellinia scutellata',
        '1312.Russula globispora',
        '1100.Tricholoma portentosum',
        '0210.Cortinarius danicus',
        '1181.Junghuhnia nitida',
        '0663.Lycoperdon pyriforme',
        '0266.Hericium cirrhatum',
        '0869.Pluteus semibulbosus',
        '0603.Lactarius scoticus',
        '0469.Hygrocybe reidii',
        '0733.Mycena galopus',
        '0342.Entoloma vernum',
        '1346.Flammulaster limulatus',
        '0187.Cortinarius anserinus',
        '0432.Humaria hemisphaerica',
        '0301.Dictydiaethalium plumbeum',
        '0875.Oligoporus fragilis',
    ],
    'validation': [
        '0000.Abortiporus biennis',
        '0004.Agaricus augustus',
        '0006.Agaricus bisporus',
        '0015.Agaricus moelleri',
        '0022.Agaricus xanthodermus',
        '0028.Aleurodiscus amorphus',
        '0032.Amanita franchetii',
        '0036.Amanita citrina var. citrina',
        '0045.Amanita virosa',
        '0058.Ascocoryne sarcoides',
        '0061.Asterophora lycoperdoides',
        '0064.Aurantiporus croceus',
        '0071.Bankera violascens',
        '0073.Bispora antennata',
        '0080.Imleria badia',
        '0087.Boletus pinophilus',
        '0091.Caloboletus radicans',
        '0095.Botryobasidium aureum',
        '0101.Buglossoporus quercinus',
        '0102.Bulgaria inquinans',
        '0104.Byssonectria terrestris',
        '0110.Calyptella campanula',
        '0117.Cantharellus cibarius',
        '0121.Ceraceomyces serpens',
        '0126.Ceriporiopsis resinascens',
        '0134.Ciboria rufofusca',
        '0138.Clavariadelphus pistillaris',
        '0142.Clavulinopsis helvola',
        '0148.Clitocybe diatreta',
        '0149.Clitocybe ditopus',
        '0152.Infundibulicybe gibba',
        '0159.Infundibulicybe squamulosa',
        '0162.Clitopilus hobsonii',
        '0163.Clitopilus prunulus',
        '0209.Cortinarius croceus',
        '0217.Cortinarius flexipes var. inolens',
        '0224.Cortinarius lucorum',
        '0227.Cortinarius mucosus',
        '0229.Cortinarius obtusus',
        '0231.Cortinarius olivaceofuscus',
        '0236.Cortinarius purpurascens',
        '0258.Cortinarius turgidus',
        '0265.Craterellus tubaeformis',
        '0273.Crepidotus variabilis',
        '0276.Cudoniella acicularis',
        '0286.Cystolepiota seminuda',
        '0294.Delicatula integrella',
        '0296.Dermoloma cuneifolium',
        '0313.Entoloma conferendum',
        '0315.Entoloma sinuatum',
        '0325.Entoloma neglectum',
        '0327.Entoloma papillatum',
        '0328.Entoloma pleopodium',
        '0329.Entoloma politum',
        '0332.Entoloma rhodopolium',
        '0341.Entoloma undatum',
        '0353.Flammulaster granulosus',
        '0354.Flammulaster muricatus',
        '0355.Flammulaster subincarnatus',
        '0357.Fomes fomentarius',
        '0366.Galerina marginata',
        '0369.Galerina triscopa',
        '0370.Galerina vittiformis',
        '0371.Ganoderma adspersum',
        '0374.Ganoderma pfeifferi',
        '0378.Geastrum rufescens',
        '0389.Gomphidius glutinosus',
        '0396.Gymnopus acervatus',
        '0401.Gymnopus fagiphilus',
        '0409.Gyroporus castaneus',
        '0410.Gyroporus cyanescens',
        '0414.Hebeloma radicosum',
        '0417.Hebeloma theobrominum',
        '0418.Hebeloma velutipes',
        '0425.Hemimycena lactea',
        '0430.Hohenbuehelia fluxilis',
        '0435.Hydnellum concrescens',
        '0463.Hygrocybe phaeococcinea',
        '0464.Cuphophyllus pratensis',
        '0466.Gliophorus psittacinus',
        '0470.Cuphophyllus russocoriaceus',
        '0494.Hyphoderma setigerum',
        '0496.Hyphodontia pallidula',
        '0520.Inocybe assimilata',
        '0533.Inocybe geophylla',
        '0534.Inocybe godeyi',
        '0537.Inocybe lacera var. lacera',
        '0552.Mensularia radiata',
        '0555.Ischnoderma resinosum',
        '0561.Laccaria tortilis',
        '0565.Lachnum virgineum',
        '0567.Lactarius acris',
        '0583.Lactarius glyciosmus',
        '0592.Lactifluus piperatus',
        '0602.Lactarius ruginosus',
        '0605.Lactarius subdulcis',
        '0607.Lactarius torminosus',
        '0622.Leccinum variicolor',
        '0624.Leccinum vulpinum',
        '0641.Lepiota lilacea',
        '0649.Leucoagaricus leucothites var. carneifolius',
        '0653.Aspropaxillus giganteus',
        '0658.Lycoperdon echinatum',
        '0659.Lycoperdon lividum',
        '0667.Sphagnurus paluster',
        '0684.Marasmius oreades',
        '0694.Melanoleuca exscissa var. exscissa',
        '0698.Melanoleuca verrucipes',
        '0702.Merismodes anomalus',
        '0705.Mitrula paludosa',
        '0706.Mollisia cinerea',
        '0712.Mycena abramsii',
        '0715.Mycena aetites',
        '0718.Mycena aurantiomarginata',
        '0730.Mycena flavescens',
        '0731.Mycena flavoalba',
        '0741.Mycena pelianthina',
        '0743.Mycena polygramma',
        '0755.Mycena stylobates',
        '0780.Panaeolus fimicola',
        '0788.Peniophora cinerea',
        '0800.Peziza varia',
        '0802.Phaeolepiota aurea',
        '0815.Fomitiporia robusta',
        '0827.Pholiota astragalina',
        '0845.Piptoporus betulinus',
        '0848.Pleurocybella porrigens',
        '0862.Pluteus petasatus',
        '0865.Pluteus pouzarianus',
        '0866.Pluteus romellii',
        '0873.Porphyrellus porphyrosporus',
        '0880.Postia lactea',
        '0884.Homophron cernuum',
        '0888.Psathyrella impexa',
        '0892.Psathyrella obtusata',
        '0901.Pseudoplectania nigrella',
        '0905.Deconica subviscida',
        '0910.Pycnoporus cinnabarinus',
        '0932.Russula adusta',
        '0948.Russula decolorans',
        '0949.Russula delica',
        '0968.Russula luteotacta',
        '0970.Russula melliolens',
        '0973.Russula nitida',
        '0977.Russula paludosa',
        '0991.Russula sororia',
        '0994.Russula velenovskyi',
        '0996.Russula versicolor',
        '1006.Sarcodon squamosus',
        '1008.Sarcoscypha coccinea',
        '1014.Scopuloides rimosa',
        '1016.Scytinostroma hemidichophyticum',
        '1019.Sistotrema brinkmannii',
        '1026.Sphaerobolus stellatus',
        '1027.Spinellus fusiger',
        '1028.Steccherinum fimbriatum',
        '1036.Strobilomyces strobilaceus',
        '1043.Stropharia hornemannii',
        '1057.Suillus variegatus',
        '1075.Tremella foliacea',
        '1092.Tricholoma equestre',
        '1097.Tricholoma orirubens',
        '1106.Tricholoma stans',
        '1110.Tricholoma ustale',
        '1123.Tyromyces chioneus',
        '1126.Volvariella bombycina',
        '1134.Hymenopellis radicata',
        '1137.Xylaria longipes',
        '1143.Hypogymnia tubulosa',
        '1145.Protoparmeliopsis muralis',
        '1150.Peltigera hymenina',
        '1151.Physcia adscendens',
        '1153.Ramalina fastigiata',
        '1173.Gymnosporangium sabinae',
        '1199.Fomitiporia hippophaeicola',
        '1203.Deconica crobula',
        '1214.Tremella versicolor',
        '1224.Pholiotina aporos',
        '1228.Cortinarius balteatocumatilis',
        '1230.Paxillus obscurisporus',
        '1233.Craterellus undulatus',
        '1237.Cystodermella cinnabarina',
        '1240.Hypocrea minutispora',
        '1242.Hypocrea leucopus',
        '1247.Arrhenia gerardiana',
        '1255.Coprinellus xanthothrix',
        '1257.Coprinellus disseminatus',
        '1261.Coprinopsis lagopus',
        '1278.Tricholoma frondosae',
        '1279.Flammulina elastica',
        '1294.Chlorophyllum olivieri',
        '1320.Tricholoma saponaceum',
        '1321.Galerina vittiformis f. bispora',
        '1328.Phlebiella vaga',
        '1341.Suillellus luridus',
        '1350.Myochromella boudieri',
        '1354.Leucocybe connata',
        '1363.Peniophorella praetermissa',
        '1375.Picipes tubaeformis',
        '1379.Phellinopsis conchata',
    ],
    'test': [
        '0008.Agaricus campestris',
        '0009.Agaricus comtulus',
        '0019.Agaricus sylvicola',
        '0038.Amanita pantherina',
        '0044.Amanita vaginata var. vaginata',
        '0053.Arrhenia acerosa',
        '0055.Arrhenia retiruga',
        '0062.Asterophora parasitica',
        '0063.Aurantiporus alborubescens',
        '0069.Auriscalpium vulgare',
        '0070.Baeospora myosura',
        '0078.Bolbitius titubans',
        '0093.Hortiboletus rubellus',
        '0096.Botryobasidium vagum',
        '0099.Bovista plumbea',
        '0103.Byssocorticium atrovirens',
        '0105.Calloria neglecta',
        '0118.Craterellus cinereus',
        '0129.Chamaemyces fracidus',
        '0132.Chroogomphus rutilus',
        '0141.Clavulinopsis corniculata',
        '0151.Infundibulicybe geotropa',
        '0153.Clitocybe metachroa',
        '0156.Clitocybe phaeophthalma',
        '0165.Collybia cookei',
        '0175.Coprinus comatus',
        '0176.Coprinopsis ephemeroides',
        '0179.Ophiocordyceps entomorrhiza',
        '0184.Cortinarius alboviolaceus',
        '0190.Cortinarius barbatus',
        '0205.Cortinarius collinitus',
        '0220.Cortinarius hinnuleus',
        '0223.Cortinarius elatior',
        '0232.Cortinarius parvannulatus',
        '0234.Cortinarius pratensis',
        '0247.Cortinarius stillatitius',
        '0277.Cyathus olla',
        '0279.Cylindrobasidium laeve',
        '0289.Dacrymyces stillatus',
        '0291.Daedaleopsis confragosa',
        '0316.Entoloma exile',
        '0321.Entoloma incarnatofuscescens',
        '0326.Entoloma nitidum',
        '0356.Flammulina velutipes var. velutipes',
        '0359.Fuligo leviderma',
        '0381.Geoglossum cookeanum',
        '0382.Geoglossum elongatum',
        '0390.Gomphidius maculatus',
        '0393.Gymnopilus spectabilis',
        '0395.Gymnopilus penetrans',
        '0397.Gymnopus aquosus',
        '0408.Gyromitra esculenta',
        '0413.Hebeloma mesophaeum',
        '0431.Holwaya mucida',
        '0433.Hydnellum aurantiacum',
        '0444.Hygrocybe calciphila',
        '0453.Cuphophyllus fornicatus',
        '0457.Hygrocybe insipida',
        '0460.Hygrocybe miniata var. miniata',
        '0462.Neohygrocybe nitrata',
        '0472.Hygrocybe splendidissima',
        '0479.Hygrophorus chrysodon',
        '0482.Hygrophorus hypothejus f. hypothejus ',
        '0484.Hygrophorus olivaceoalbus',
        '0491.Hyphoderma argillaceum',
        '0492.Hyphoderma medioburiense',
        '0498.Hypholoma elongatum',
        '0501.Hypholoma marginatum',
        '0502.Phaeonematoloma myosotis',
        '0505.Hypocrea citrina',
        '0517.Inocybe acuta',
        '0535.Inocybe griseolilacina',
        '0536.Inocybe hirtella',
        '0539.Inocybe maculata',
        '0541.Inocybe napipes',
        '0574.Lactarius chrysorrheus',
        '0580.Lactarius evosmus',
        '0584.Lactarius helvus',
        '0587.Lactarius lilacinus',
        '0596.Lactarius pyrogalus',
        '0600.Lactarius rubrocinctus',
        '0604.Lactarius serifluus',
        '0609.Lactifluus vellereus',
        '0616.Leccinellum carpini',
        '0618.Leccinum niveum',
        '0620.Leccinum quercinum',
        '0627.Trametes betulina',
        '0637.Lepiota felina',
        '0651.Leucocoprinus brebissonii',
        '0661.Lycoperdon nigrescens',
        '0674.Marasmiellus ramealis',
        '0677.Gymnopus androsaceus',
        '0680.Marasmius curreyi',
        '0688.Marasmius torquescens',
        '0695.Melanoleuca grammopodia',
        '0703.Microglossum olivaceum',
        '0713.Mycena acicula',
        '0722.Mycena chlorantha',
        '0724.Mycena clavicularis',
        '0726.Mycena diosma',
        '0727.Mycena epipterygia',
        '0729.Mycena filopes',
        '0742.Mycena clavata',
        '0760.Naucoria bohemica',
        '0764.Naucoria subconspersa',
        '0765.Nectria cinnabarina',
        '0784.Panellus mitis',
        '0787.Paxillus involutus',
        '0790.Peniophora limitata',
        '0792.Peniophora polygonia',
        '0799.Peziza succosa',
        '0803.Phaeolus schweinitzii',
        '0805.Phallus hadriani',
        '0807.Phanerochaete velutina',
        '0812.Porodaedalea pini',
        '0817.Phellodon niger',
        '0820.Phlebia radiata',
        '0826.Flammula alnicola',
        '0834.Hemipholiota populnea',
        '0838.Pholiotina velata',
        '0849.Pleurotus dryinus',
        '0853.Pluteus atromarginatus',
        '0857.Pluteus hispidulus',
        '0871.Pluteus umbrosus var. umbrosus ',
        '0885.Parasola conopilus',
        '0893.Psathyrella piluliformis',
        '0904.Psilocybe semilanceata',
        '0907.Puccinia punctiformis',
        '0909.Pycnoporellus fulgens',
        '0914.Ramaria pallida',
        '0915.Ramaria sanguinea',
        '0917.Ramariopsis crocea',
        '0935.Russula amoenolens',
        '0937.Russula aquosa',
        '0938.Russula atrorubens',
        '0940.Russula betularum',
        '0947.Russula cyanoxantha',
        '0955.Russula fellea',
        '0960.Russula graveolens',
        '0963.Russula illota',
        '0966.Russula laccata',
        '0971.Russula nauseosa',
        '0976.Russula olivacea',
        '0987.Russula roseoaurantia',
        '0989.Russula sardonia',
        '0997.Russula vesca',
        '0999.Russula vinosa',
        '1012.Scleroderma citrinum',
        '1018.Simocybe sumptuosa',
        '1024.Sparassis crispa',
        '1030.Steccherinum oreophilum',
        '1031.Stereum gausapatum',
        '1032.Stereum hirsutum',
        '1033.Stereum rugosum',
        '1034.Stereum sanguinolentum',
        '1062.Tapinella panuoides',
        '1076.Tremella mesenterica',
        '1077.Tremiscus helvelloides',
        '1084.Tricholoma argyraceum',
        '1086.Tricholoma squarrulosum',
        '1094.Tricholoma imbricatum',
        '1096.Tricholoma matsutake',
        '1098.Tricholoma pessundatum',
        '1104.Tricholoma sciodes',
        '1105.Tricholoma sejunctum',
        '1107.Tricholoma stiparophyllum',
        '1111.Tricholomopsis decora',
        '1115.Tubaria dispersa',
        '1117.Tubifera ferruginosa',
        '1120.Tylopilus felleus',
        '1129.Vuilleminia comedens',
        '1133.Xerula pudens',
        '1157.Xerocomus ferrugineus',
        '1165.Cortinarius emollitus',
        '1169.Entoloma sericeum var. cinereo-opacum',
        '1194.Mycena olida',
        '1231.Hygrocybe konradii',
        '1234.Lepista saeva',
        '1238.Octaviania asterosperma',
        '1249.Pholiotina teneroides',
        '1250.Ampulloclitocybe clavipes',
        '1253.Coprinellus micaceus',
        '1262.Coprinopsis picacea',
        '1264.Coprinopsis martinii',
        '1265.Coprinopsis laanii',
        '1273.Crucibulum crucibuliforme',
        '1280.Inocybe lilacina',
        '1299.Tricholoma vaccinum',
        '1305.Inocybe lacera',
        '1314.Suillus grevillei',
        '1318.Crepidotus stenocystis',
        '1344.Rubroboletus legaliae',
        '1348.Glutinoglossum glutinosum',
        '1352.Hortiboletus engelii',
        '1362.Paralepista gilva',
        '1366.Cartilosoma ramentaceum',
        '1373.Picipes badius',
        '1376.Cladomeris umbellata',
        '1382.Phaeoclavulina eumorpha',
        '1388.Xylodon flaviporus (Berk. & M.A. Curtis) Riebesehl & E. Langer',
    ]
}


class FGVCFungi(Dataset):

    """
    [[Source]](https://github.com/learnables/learn2learn/blob/master/learn2learn/vision/datasets/fgvc_fungi.py)

    **Description**

    The FGVC Fungi dataset was originally introduced in the 5th Workshop on Fine-Grained Visual Categorization (FGVC)
    and then re-purposed for few-shot learning in Triantafillou et al., 2020.

    The dataset consists of 1,394 classes and 89,760 images of fungi.
    We provide the raw (unprocessed) images, and follow the train-validation-test splits of Triantafillou et al.

    *Important*
    You must agree to the original Terms of Use to use this dataset.
    More information here: [https://github.com/visipedia/fgvcx_fungi_comp](https://github.com/visipedia/fgvcx_fungi_comp)

    **References**

    1. [https://sites.google.com/view/fgvc5/home](https://sites.google.com/view/fgvc5/home)
    2. Triantafillou et al. 2020. "Meta-Dataset: A Dataset of Datasets for Learning to Learn from Few Examples." ICLR '20.
    3. [https://github.com/visipedia/fgvcx_fungi_comp](https://github.com/visipedia/fgvcx_fungi_comp)

    **Arguments**

    * **root** (str) - Path to download the data.
    * **mode** (str, *optional*, default='train') - Which split to use.
        Must be 'train', 'validation', or 'test'.
    * **transform** (Transform, *optional*, default=None) - Input pre-processing.
    * **target_transform** (Transform, *optional*, default=None) - Target pre-processing.
    * **download** (bool, *optional*, default=False) - Whether to download the dataset.

    **Example**

    ~~~python
    train_dataset = l2l.vision.datasets.FGVCFungi(root='./data', mode='train')
    train_dataset = l2l.data.MetaDataset(train_dataset)
    train_generator = l2l.data.Taskset(dataset=train_dataset, num_tasks=1000)
    ~~~

    """

    def __init__(self, root, mode='all', transform=None, target_transform=None, download=False):
        root = os.path.expanduser(root)
        self.root = os.path.expanduser(root)
        self.transform = transform
        self.target_transform = target_transform
        self._bookkeeping_path = os.path.join(self.root, 'fgvc-fungi-' + mode + '-bookkeeping.pkl')

        if not self._check_exists() and download:
            self.download()

        self.load_data(mode)

    def _check_exists(self):
        data_path = os.path.join(self.root, DATA_DIR, 'images')
        annotations_path = os.path.join(self.root, DATA_DIR, 'train.json')
        return os.path.exists(data_path)

    def download(self):
        data_path = os.path.join(self.root, DATA_DIR)
        os.makedirs(data_path, exist_ok=True)
        data_tar_path = os.path.join(data_path, os.path.basename(DATA_URL))
        annotations_tar_path = os.path.join(data_path, os.path.basename(ANNOTATIONS_URL))

        # Download data
        print('Downloading FGVC Fungi dataset (12.9Gb)')
        download_file(DATA_URL, data_tar_path, size=12_900_000_000)
        download_file(ANNOTATIONS_URL, annotations_tar_path)

        # Extract data
        tar_file = tarfile.open(data_tar_path)
        tar_file.extractall(data_path)
        tar_file.close()
        os.remove(data_tar_path)

        # Extract annotations
        tar_file = tarfile.open(annotations_tar_path)
        tar_file.extractall(data_path)
        tar_file.close()
        os.remove(annotations_tar_path)

    def load_data(self, mode='train'):
        if not os.path.exists(self._bookkeeping_path):
            # Load annotations
            data_path = os.path.join(self.root, DATA_DIR)
            train_path = os.path.join(data_path, 'train.json')
            with open(train_path, 'r') as f_train:
                train_annotations = json.load(f_train)
            valid_path = os.path.join(data_path, 'val.json')
            with open(valid_path, 'r') as f_valid:
                valid_annotations = json.load(f_valid)
            split_classes = sum(SPLITS.values(), []) if mode == 'all' else SPLITS[mode]
            split_classes = [int(cls[:4]) for cls in split_classes]

            # Create bookkeeping
            labels_to_indices = defaultdict(list)
            indices_to_labels = defaultdict(int)
            data_map = []

            # Process
            all_images = train_annotations['images'] + valid_annotations['images']
            all_annotations = train_annotations['annotations'] \
                + valid_annotations['annotations']
            counter = 0
            for image, annotation in zip(all_images, all_annotations):
                assert image['id'] == annotation['image_id']
                img_cat = annotation['category_id']
                if img_cat in split_classes:
                    img_path = os.path.join(data_path, image['file_name'])
                    label = split_classes.index(img_cat)
                    data_map.append((img_path, label))
                    labels_to_indices[label].append(counter)
                    indices_to_labels[counter] = label
                    counter += 1

            # Cache to disk
            bookkeeping = {
                'labels_to_indices': labels_to_indices,
                'indices_to_labels': indices_to_labels,
                'labels': list(labels_to_indices.keys()),
                'data_map': data_map,
            }
            with open(self._bookkeeping_path, 'wb') as f:
                pickle.dump(bookkeeping, f, protocol=-1)
        else:
            # Load bookkeeping
            with open(self._bookkeeping_path, 'rb') as f:
                bookkeeping = pickle.load(f)

        self._bookkeeping = bookkeeping
        self.labels_to_indices = bookkeeping['labels_to_indices']
        self.indices_to_labels = bookkeeping['indices_to_labels']
        self.labels = bookkeeping['labels']
        self.data_map = bookkeeping['data_map']
        self.length = len(self.indices_to_labels)

    def __getitem__(self, i):
        image, label = self.data_map[i]
        image = Image.open(image)
        if self.transform:
            image = self.transform(image)
        if self.target_transform:
            label = self.target_transform(label)
        return image, label

    def __len__(self):
        return self.length


if __name__ == '__main__':
    fungi = FGVCFungi('~/data', mode='all', download=True)
    fungi = FGVCFungi('~/data', mode='train', download=True)
    fungi = FGVCFungi('~/data', mode='validation', download=True)
    fungi = FGVCFungi('~/data', mode='test', download=True)
    print(len(fungi))
