# Introduction to PowerModelsITD

In this guide, we will go through a beginner guide that demonstrates the current capabilities of `PowerModelsITD`. This guide will showcase a _non-comprehensive_ list of things you can do with `PowerModelsITD`.

Each example case shown in this guide will be divided into **three** main parts.

- (1) Brief explanation of the test case and what we want to do/obtain.
- (2) The Julia code needed to perform the specific case study (_users can copy and paste this code directly_).
- (3) The console output obtained when the code is executed.

**Important**: All files used in this beginner guide are available in the `/test/data` folder of the repository.

## Example Case #1: Solving a simple Integrated T&D Optimal Power Flow (OPFITD)

### Test Case #1

In this example case, we want to solve the integrated T&D optimal power flow (opfitd) problem for a case where we have the **PJM 5-bus system** (`case5_withload`) as the _transmission system_ and the **IEEE 4 Node Test Feeder** (`case3_unbalanced.dss`) as the _distribution system_. The boundary information that specifies the respective boundary buses for both the transmission and distribution systems can be found in `case5_case3_unbal.json`.

The formulation used to solve this optimization problem is the **ACP-ACPU** formulation.

### Julia Code for Test Case #1

```julia
using PowerModelsITD
using Ipopt

silence!()  # Silences the console, so warning messages or information are not displayed

files_path = joinpath(dirname(pathof(PowerModelsITD)), "..")                    # file path for test cases files
pm_file = joinpath(files_path, "test/data/transmission/case5_withload.m")       # transmission system file
pmd_file = joinpath(files_path, "test/data/distribution/case3_unbalanced.dss")  # distribution system file
boundary_file = joinpath(files_path, "test/data/json/case5_case3_unbal.json")   # boundary file with info. about boundary buses
formulation = NLPowerModelITD{ACPPowerModel, ACPUPowerModel}                    # type of formulation to be used
data = parse_files(pm_file, pmd_file, boundary_file)                        # parse files into dictionary
result = solve_opfitd(data, formulation, Ipopt.Optimizer)                   # solve the opfitd

# Check the result
@info "Objective: $(result["objective"]) [\$/hr]"
@info "Termination Status of Solver: $(result["termination_status"])"
@info "Solve Time: $(result["solve_time"]) [seconds]"
@info "Solution Values for Transmission system: $(result["solution"]["it"]["pm"])"
@info "Solution Values for Distribution system(s): $(result["solution"]["it"]["pmd"])"
@info "Solution Values for Boundary(ies): $(result["solution"]["it"]["pmitd"])"
```

### Console Output for Test Case #1

After running the corresponding code, the user should get the following output displayed on the console.

```julia
[ Main | Info ] : Objective: 17953.746512039743 [$/hr]
[ Main | Info ] : Termination Status of Solver: LOCALLY_SOLVED
[ Main | Info ] : Solve Time: 0.05849599838256836 [seconds]

[ Main | Info ] : Solution Values for Transmission system: Dict{String, Any}("baseMVA" => 100.0, "branch" => Dict{String, Any}("4" => Dict{String, Any}("qf" => -215.4572384567855, "qt" => 220.01471251650577, "pt" => 72.72770254777281, "pf" => -72.10324334974787), "1" => Dict{String, Any}("qf" => -91.9457671031378, "qt" => 113.04924896169344, "pt" => -235.96795359006418, "pf" => 238.13985616928193), "5" => Dict{String, Any}("qf" => 71.37529049674947, "qt" => -69.00868855761938, "pt" => 57.04309684079022, "pf" => -56.75055132454856), "2" => Dict{String, Any}("qf" => 40.82747127060588, "qt" => -27.51897757380047, "pt" => -190.17342809935752, "pf" => 191.55864094141782), "6" => Dict{String, Any}("qf" => 19.289482716290323, "qt" => 1.0554040857229503, "pt" => 239.9976806084548, "pf" => -237.90830207159405), "7" => Dict{String, Any}("qf" => 3.7979894950920543, "qt" => -4.367184067738672, "pt" => -8.068452673591816, "pf" => 8.071196939812054), "3" => Dict{String, Any}("qf" => 208.61829769507162, "qt" => -204.19918131371665, "pt" => 220.40004611863847, "pf" => -219.6984944935753)), "gen" => Dict{String, Any}("4" => Dict{String, Any}("qg" => 54.23181658487047, "pg" => 18.96136666983862), "1" => Dict{String, Any}("qg" => 30.00000079376986, "pg" => 40.00000096678977), "5" => Dict{String, Any}("qg" => -203.1437772279937, "pg" => 460.3977267270933), "2" => Dict{String, Any}("qg" => 127.50000106876985, "pg" => 170.00000165033464), "3" => Dict{String, Any}("qg" => 390.00000301325525, "pg" => 315.97715122322427)), "multinetwork" => false, "bus" => Dict{String, Any}("4" => Dict{String, Any}("va" => -9.820934169803595e-29, "vm" => 0.8999999900918129), "1" => Dict{String, Any}("va" => 3.9557118095964663, "vm" => 0.9178154479809132), "5" => Dict{String, Any}("va" => -0.8036120919989638, "vm" => 0.9400883479210898), "2" => Dict{String, Any}("va" => -0.6563318572164789, "vm" => 0.9416319858451535), "10" => Dict{String, Any}("va" => 5.018509000535593, "vm" => 0.904864159526407), "3" => Dict{String, Any}("va" => -0.3122279603896953, "vm" => 0.9670939985734723)), "per_unit" => false)

[ Main | Info ] : Solution Values for Distribution system(s): Dict{String, Any}("line" => Dict{String, Any}("3bus_unbal.quad" => Dict{String, Any}("qf" => [1344.7785535214057, 1503.9392726830868, 1502.43791779139], "qt" => [-1333.3300000000002, -1500.0000000000002, -1500.0000000000002], "pt" => [-3333.3323371531164, -2333.3323371935803, -2333.3323371900296], "pf" => [3351.5305232165006, 2340.3516279866967, 2344.8964126191295]), "3bus_unbal.ohline" => Dict{String, Any}("qf" => [1354.6163225169614, 1507.2344513630765, 1504.4700979169072], "qt" => [-1344.7785535214057, -1503.9392726830868, -1502.43791779139], "pt" => [-3351.5305232165006, -2340.3516279866967, -2344.8964126191295], "pf" => [3367.110727286094, 2346.3597337552624, 2354.8245890269895])), "settings" => Dict{String, Any}("sbase" => 100000.0), "transformer" => Dict{String, Any}("3bus_unbal.subxf" => Dict{String, Any}("q" => [[1354.7952022418647, 1507.3400640138161, 1504.5761375753734], [-1354.6163225169614, -1507.2344513630765, -1504.4700979169072]], "p" => [[3367.1286152278885, 2346.3702946258413, 2354.835193007617], [-3367.110727286094, -2346.3597337552624, -2354.8245890269895]])), "generator" => Dict{String, Any}("3bus_unbal.gen1" => Dict{String, Any}("qg_bus" => [-0.0, -0.0, -0.0], "qg" => [-0.0, -0.0, -0.0], "pg" => [666.6676628468832, 666.6676628064197, 666.6676628099706], "pg_bus" => [666.6676628468832, 666.6676628064197, 666.6676628099706])), "load" => Dict{String, Any}("3bus_unbal.l2" => Dict{String, Any}("qd_bus" => [1500.0000000000002], "pd_bus" => [3000.0000000000005], "qd" => [1500.0000000000002], "pd" => [3000.0000000000005]), "3bus_unbal.l3" => Dict{String, Any}("qd_bus" => [1500.0000000000002], "pd_bus" => [3000.0000000000005], "qd" => [1500.0000000000002], "pd" => [3000.0000000000005]), "3bus_unbal.l1" => Dict{String, Any}("qd_bus" => [1333.3300000000002], "pd_bus" => [4000.0], "qd" => [1333.3300000000002], "pd" => [4000.0])), "bus" => Dict{String, Any}("3bus_unbal.loadbus" => Dict{String, Any}("va" => [-0.9542852429015141, -120.82463162589393, 119.31695673367102], "vm" => [7.406969380817659, 7.446613395025093, 7.431619824571949]), "3bus_unbal.substation" => Dict{String, Any}("va" => [-0.8089341499650244, -120.8072185671511, 119.19279637320398], "vm" => [7.489646348633082, 7.489651704574486, 7.489663202448464]), "3bus_unbal.primary" => Dict{String, Any}("va" => [-0.8772138344778904, -120.81603057758366, 119.24896754768255], "vm" => [7.4514218741879255, 7.4697354940559455, 7.462765580863505]), "3bus_unbal.sourcebus" => Dict{String, Any}("va" => [-0.8064197613818459, -120.80551025323088, 119.19451148306055], "vm" => [124.83030630100751, 124.83048117738736, 124.8306695534823])), "per_unit" => false)

[ Main | Info ] : Solution Values for Boundary(ies): Dict{String, Any}("multinetwork" => false, "boundary" => Dict{String, Any}("(100001, 5, voltage_source.3bus_unbal.source)" => Dict{String, Any}("pbound_fr" => [8068.452673591816], "qbound_fr" => [4367.184067738673]), "(100001, voltage_source.3bus_unbal.source, 5)" => Dict{String, Any}("pbound_to" => [-3367.1850486607445, -2346.4026604158908, -2354.8649645151804], "qbound_to" => [-1355.0096276913357, -1507.4706735124998, -1504.7037665348369])))
```

## Example Case #2: Solving a Multinetwork (Time-series) Multisystem (Multiple Distribution Systems) OPFITD

### Test Case #2

In this example case, we want to solve the integrated T&D optimal power flow (opfitd) problem for a **Multinetwork** case where we have the **PJM 5-bus system** (`case5_with2loads`) as the _transmission system_ and **two** copies of the **IEEE 4 Node Test Feeder** (`case3_unbalanced_withoutgen_mn.dss`) as the _distribution systems_, where each distribution system has load profiles for their respective loads for **four** timesteps. The boundary information that specifies the respective boundary buses for both the transmission and distribution systems can be found in `case5_case3x2.json`.

The formulation used to solve this optimization problem is the **ACP-ACPU** formulation.

### Julia Code for Test Case #2

```julia
using PowerModelsITD
using Ipopt

silence!()  # Silences the console, so warning messages or information are not displayed

files_path = joinpath(dirname(pathof(PowerModelsITD)), "..")                                    # file path for test cases files
pm_file = joinpath(files_path, "test/data/transmission/case5_with2loads.m")                     # transmission system file
pmd_file = joinpath(files_path, "test/data/distribution/case3_unbalanced_withoutgen_mn.dss")    # distribution system file
boundary_file = joinpath(files_path, "test/data/json/case5_case3x2.json")                       # boundary file with info. about boundary buses
pmd_files = [pmd_file, pmd_file]                                                                # vector of distrib. systems (in this case a copy of the same)
formulation = NLPowerModelITD{ACPPowerModel, ACPUPowerModel}                                    # type of formulation to be used
result = solve_mn_opfitd(pm_file, pmd_files, boundary_file, formulation, Ipopt.Optimizer; auto_rename=true)  # solve the multinetwork opfitd

# Check the result
@info "Objective: $(result["objective"]) [\$/hr]"
@info "Termination Status of Solver: $(result["termination_status"])"
@info "Solve Time: $(result["solve_time"]) [seconds]"
@info "Solution Values for Transmission system: $(result["solution"]["it"]["pm"])"
@info "Solution Values for Distribution system(s): $(result["solution"]["it"]["pmd"])"
@info "Solution Values for Boundary(ies): $(result["solution"]["it"]["pmitd"])"
```

### Console Output for Test Case #2

After running the corresponding code, the user should get the following output displayed on the console.

```julia
[ Main | Info ] : Objective: 71583.37023067646 [$/hr]
[ Main | Info ] : Termination Status of Solver: LOCALLY_SOLVED
[ Main | Info ] : Solve Time: 0.5699648857116699 [seconds]

[ Main | Info ] : Solution Values for Transmission system: Dict{String, Any}("multinetwork" => true, "nw" => Dict{String, Any}("4" => Dict{String, Any}("baseMVA" => 100.0, "branch" => Dict{String, Any}("3" => Dict{String, Any}("qf" => 209.7124512609962, "qt" => -205.25808038279263, "pt" => 220.4516503859562, "pf" => -219.74655219293345), "4" => Dict{String, Any}("qf" => -214.48170261865238, "qt" => 218.94452403011204, "pt" => 70.63553676891648, "pf" => -70.02040086637558), "1" => Dict{String, Any}("qf" => -93.26577989474518, "qt" => 114.45283104138902, "pt" => -235.99842638945864, "pf" => 238.17872276993856), "5" => Dict{String, Any}("qf" => 72.445478971592, "qt" => -70.00150166186677, "pt" => 57.688800117737046, "pf" => -57.388497698913255), "2" => Dict{String, Any}("qf" => 41.053330496054876, "qt" => -27.739129989384807, "pt" => -190.18204403801204, "pf" => 191.56783204014675), "6" => Dict{String, Any}("qf" => 19.29122822256867, "qt" => 1.0536848339377651, "pt" => 239.99768816280888, "pf" => -237.90830703516832), "7" => Dict{String, Any}("qf" => 1.418871577263358, "qt" => -2.003521254092567, "pt" => -6.017516284814005, "pf" => 6.018827255834227), "8" => Dict{String, Any}("qf" => 0.7049338049713338, "qt" => -1.2985975532566612, "pt" => -3.008593331044258, "pf" => 3.008930510206021)), "gen" => Dict{String, Any}("4" => Dict{String, Any}("qg" => 53.02059657131708, "pg" => 19.598449044556656), "1" => Dict{String, Any}("qg" => 30.00000079365296, "pg" => 40.00000096679825), "5" => Dict{String, Any}("qg" => -204.20439554885485, "pg" => 460.449338548765), "2" => Dict{String, Any}("qg" => 127.50000106865294, "pg" => 170.0000016503536), "3" => Dict{String, Any}("qg" => 390.00000300170404, "pg" => 313.2470390700032)), "bus" => Dict{String, Any}("4" => Dict{String, Any}("va" => -4.8091331437638894e-29, "vm" => 0.8999999900913073), "1" => Dict{String, Any}("va" => 3.955108495412184, "vm" => 0.9178896115768673), "5" => Dict{String, Any}("va" => -0.7695989397082574, "vm" => 0.9413800134626195), "2" => Dict{String, Any}("va" => -0.6574104691903474, "vm" => 0.9421095463639055), "6" => Dict{String, Any}("va" => -0.8254740984569308, "vm" => 0.9409689065117178), "10" => Dict{String, Any}("va" => 5.018515926892925, "vm" => 0.9048635910623162), "3" => Dict{String, Any}("va" => -0.3270759541168204, "vm" => 0.967421494353026))), "1" => Dict{String, Any}("baseMVA" => 100.0, "branch" => Dict{String, Any}("3" => Dict{String, Any}("qf" => 209.71245126093345, "qt" => -205.25808038273263, "pt" => 220.4516503861189, "pf" => -219.74655219309614), "4" => Dict{String, Any}("qf" => -214.4817026186947, "qt" => 218.94452403016115, "pt" => 70.6355367692654, "pf" => -70.02040086672375), "1" => Dict{String, Any}("qf" => -93.26577989466723, "qt" => 114.45283104133719, "pt" => -235.9984263896544, "pf" => 238.17872277013694), "5" => Dict{String, Any}("qf" => 72.44547897154284, "qt" => -70.001501661815, "pt" => 57.68880011784228, "pf" => -57.38849769901818), "2" => Dict{String, Any}("qf" => 41.05333049603968, "qt" => -27.73912998937486, "pt" => -190.1820440379769, "pf" => 191.56783204011106), "6" => Dict{String, Any}("qf" => 19.291228222568602, "qt" => 1.0536848339378524, "pt" => 239.99768816280888, "pf" => -237.90830703516832), "7" => Dict{String, Any}("qf" => 1.4188715773575376, "qt" => -2.003521254185469, "pt" => -6.017516285357827, "pf" => 6.0188272563781675), "8" => Dict{String, Any}("qf" => 0.7049338050415234, "qt" => -1.2985975533248926, "pt" => -3.0085933319508262, "pf" => 3.0089305111127547)), "gen" => Dict{String, Any}("4" => Dict{String, Any}("qg" => 53.02059657137874, "pg" => 19.598449044697038), "1" => Dict{String, Any}("qg" => 30.00000079365296, "pg" => 40.00000096679825), "5" => Dict{String, Any}("qg" => -204.2043955487948, "pg" => 460.44933854892776), "2" => Dict{String, Any}("qg" => 127.50000106865294, "pg" => 170.0000016503536), "3" => Dict{String, Any}("qg" => 390.00000300170404, "pg" => 313.24703907024724)), "bus" => Dict{String, Any}("4" => Dict{String, Any}("va" => 1.2906268155608688e-29, "vm" => 0.8999999900913073), "1" => Dict{String, Any}("va" => 3.9551084954114852, "vm" => 0.9178896115768619), "5" => Dict{String, Any}("va" => -0.7695989397229616, "vm" => 0.9413800134625423), "2" => Dict{String, Any}("va" => -0.6574104691947822, "vm" => 0.9421095463638751), "6" => Dict{String, Any}("va" => -0.8254740984889308, "vm" => 0.9409689065115902), "10" => Dict{String, Any}("va" => 5.018515926892924, "vm" => 0.9048635910623162), "3" => Dict{String, Any}("va" => -0.3270759541189024, "vm" => 0.9674214943530053))), "2" => Dict{String, Any}("baseMVA" => 100.0, "branch" => Dict{String, Any}("3" => Dict{String, Any}("qf" => 209.71245126094163, "qt" => -205.25808038273715, "pt" => 220.45165038612856, "pf" => -219.74655219310594), "4" => Dict{String, Any}("qf" => -214.4817026186913, "qt" => 218.94452403015868, "pt" => 70.63553676925657, "pf" => -70.02040086671512), "1" => Dict{String, Any}("qf" => -93.26577989467265, "qt" => 114.45283104134505, "pt" => -235.99842638966595, "pf" => 238.1787227701487), "5" => Dict{String, Any}("qf" => 72.44547897154538, "qt" => -70.00150166181722, "pt" => 57.688800117857696, "pf" => -57.38849769903358), "2" => Dict{String, Any}("qf" => 41.053330496036935, "qt" => -27.739129989371886, "pt" => -190.18204403797492, "pf" => 191.56783204010912), "6" => Dict{String, Any}("qf" => 19.291228222572368, "qt" => 1.0536848339335343, "pt" => 239.99768816280888, "pf" => -237.90830703516826), "7" => Dict{String, Any}("qf" => 1.4188715773462648, "qt" => -2.0035212541732794, "pt" => -6.017516285360724, "pf" => 6.0188272563810745), "8" => Dict{String, Any}("qf" => 0.7049338050293561, "qt" => -1.2985975533139513, "pt" => -3.008593331590926, "pf" => 3.008930510752742)), "gen" => Dict{String, Any}("4" => Dict{String, Any}("qg" => 53.02059657138327, "pg" => 19.59844904471449), "1" => Dict{String, Any}("qg" => 30.00000079365296, "pg" => 40.00000096679825), "5" => Dict{String, Any}("qg" => -204.2043955488037, "pg" => 460.4493385489374), "2" => Dict{String, Any}("qg" => 127.50000106865294, "pg" => 170.0000016503536), "3" => Dict{String, Any}("qg" => 390.00000300170404, "pg" => 313.24703907022297)), "bus" => Dict{String, Any}("4" => Dict{String, Any}("va" => -8.850496391421805e-29, "vm" => 0.8999999900913073), "1" => Dict{String, Any}("va" => 3.9551084954114537, "vm" => 0.9178896115768609), "5" => Dict{String, Any}("va" => -0.7695989397233032, "vm" => 0.9413800134625466), "2" => Dict{String, Any}("va" => -0.6574104691950458, "vm" => 0.9421095463638758), "6" => Dict{String, Any}("va" => -0.8254740984823784, "vm" => 0.9409689065116095), "10" => Dict{String, Any}("va" => 5.018515926892941, "vm" => 0.9048635910623148), "3" => Dict{String, Any}("va" => -0.3270759541192233, "vm" => 0.9674214943530055))), "3" => Dict{String, Any}("baseMVA" => 100.0, "branch" => Dict{String, Any}("3" => Dict{String, Any}("qf" => 209.7124512609906, "qt" => -205.2580803827864, "pt" => 220.45165038608707, "pf" => -219.7465521930642), "4" => Dict{String, Any}("qf" => -214.48170261865874, "qt" => 218.94452403012218, "pt" => 70.63553676895856, "pf" => -70.0204008664175), "1" => Dict{String, Any}("qf" => -93.26577989472547, "qt" => 114.45283104139405, "pt" => -235.9984263896151, "pf" => 238.17872277009747), "5" => Dict{String, Any}("qf" => 72.44547897158185, "qt" => -70.00150166185058, "pt" => 57.6888001178973, "pf" => -57.38849769907291), "2" => Dict{String, Any}("qf" => 41.05333049604079, "qt" => -27.739129989375172, "pt" => -190.1820440379844, "pf" => 191.56783204011862), "6" => Dict{String, Any}("qf" => 19.291228222579896, "qt" => 1.0536848339270362, "pt" => 239.99768816280894, "pf" => -237.90830703516818), "7" => Dict{String, Any}("qf" => 1.4188715772646756, "qt" => -2.0035212540928145, "pt" => -6.017516285012341, "pf" => 6.018827256032563), "8" => Dict{String, Any}("qf" => 0.7049338050059494, "qt" => -1.298597553290569, "pt" => -3.008593331239918, "pf" => 3.0089305104016915)), "gen" => Dict{String, Any}("4" => Dict{String, Any}("qg" => 53.02059657135414, "pg" => 19.598449044744708), "1" => Dict{String, Any}("qg" => 30.00000079365296, "pg" => 40.00000096679825), "5" => Dict{String, Any}("qg" => -204.20439554885937, "pg" => 460.44933854889604), "2" => Dict{String, Any}("qg" => 127.50000106865294, "pg" => 170.0000016503536), "3" => Dict{String, Any}("qg" => 390.00000300170404, "pg" => 313.24703906988566)), "bus" => Dict{String, Any}("4" => Dict{String, Any}("va" => 4.6922476467138066e-29, "vm" => 0.8999999900913073), "1" => Dict{String, Any}("va" => 3.9551084954116407, "vm" => 0.9178896115768624), "5" => Dict{String, Any}("va" => -0.7695989397156218, "vm" => 0.9413800134626014), "2" => Dict{String, Any}("va" => -0.6574104691939039, "vm" => 0.9421095463638939), "6" => Dict{String, Any}("va" => -0.8254740984679912, "vm" => 0.9409689065116826), "10" => Dict{String, Any}("va" => 5.0185159268929675, "vm" => 0.9048635910623125), "3" => Dict{String, Any}("va" => -0.3270759541200902, "vm" => 0.967421494353016)))), "per_unit" => false)

[ Main | Info ] : Solution Values for Distribution system(s): Dict{String, Any}("nw" => Dict{String, Dict{String, Any}}("4" => Dict("line" => Dict{String, Any}("3bus_unbal_nogen_mn.ohline" => Dict{String, Any}("qf" => [400.5777180879746, 449.1439873945076, 448.7567158126756], "qt" => [-400.3834881406001, -449.61753973455666, -449.40808066075476], "pt" => [-1202.2047813968609, -900.9716820211081, -901.437570378043], "pf" => [1204.092342226798, 901.8028241168121, 902.6707125959695]), "3bus_unbal_nogen_mn_2.quad" => Dict{String, Any}("qf" => [400.3857159633698, 449.61909326817045, 449.40944838966254], "qt" => [-399.999, -450.00000000000006, -450.00000000000006], "pt" => [-1200.0, -900.0000000000001, -900.0000000000001], "pf" => [1202.206724198158, 900.9725348138786, 901.4388339186033]), "3bus_unbal_nogen_mn.quad" => Dict{String, Any}("qf" => [400.3834881406001, 449.61753973455666, 449.40808066075476], "qt" => [-399.999, -450.00000000000006, -450.00000000000006], "pt" => [-1200.0, -900.0000000000001, -900.0000000000001], "pf" => [1202.2047813968609, 900.9716820211081, 901.437570378043]), "3bus_unbal_nogen_mn_2.ohline" => Dict{String, Any}("qf" => [400.5820103770306, 449.147016724407, 448.75940230925625], "qt" => [-400.3857159633698, -449.61909326817045, -449.40944838966254], "pt" => [-1202.206724198158, -900.9725348138786, -901.4388339186033], "pf" => [1204.0959489625075, 901.8044071741432, 902.673061143103])), "settings" => Dict{String, Any}("sbase" => 100000.0), "transformer" => Dict{String, Any}("3bus_unbal_nogen_mn_2.subxf" => Dict{String, Any}("q" => [[400.60383553276677, 449.1607731500957, 448.77317528774904], [-400.5820103770306, -449.147016724407, -448.75940230925625]], "p" => [[1204.0981313859945, 901.8057826939303, 902.6744388456775], [-1204.0959489625075, -901.8044071741432, -902.673061143103]]), "3bus_unbal_nogen_mn.subxf" => Dict{String, Any}("q" => [[400.59952401926625, 449.1577317466543, 448.77047663804154], [-400.5777180879746, -449.1439873945076, -448.7567158126756]], "p" => [[1204.0945224674979, 901.804198545205, 902.672088661453], [-1204.092342226798, -901.8028241168121, -902.6707125959695]])), "load" => Dict{String, Any}("3bus_unbal_nogen_mn_2.l1" => Dict{String, Any}("qd_bus" => [399.999], "pd_bus" => [1200.0], "qd" => [399.999], "pd" => [1200.0]), "3bus_unbal_nogen_mn.l1" => Dict{String, Any}("qd_bus" => [399.999], "pd_bus" => [1200.0], "qd" => [399.999], "pd" => [1200.0]), "3bus_unbal_nogen_mn_2.l3" => Dict{String, Any}("qd_bus" => [450.00000000000006], "pd_bus" => [900.0000000000001], "qd" => [450.00000000000006], "pd" => [900.0000000000001]), "3bus_unbal_nogen_mn.l2" => Dict{String, Any}("qd_bus" => [450.00000000000006], "pd_bus" => [900.0000000000001], "qd" => [450.00000000000006], "pd" => [900.0000000000001]), "3bus_unbal_nogen_mn_2.l2" => Dict{String, Any}("qd_bus" => [450.00000000000006], "pd_bus" => [900.0000000000001], "qd" => [450.00000000000006], "pd" => [900.0000000000001]), "3bus_unbal_nogen_mn.l3" => Dict{String, Any}("qd_bus" => [450.00000000000006], "pd_bus" => [900.0000000000001], "qd" => [450.00000000000006], "pd" => [900.0000000000001])), "bus" => Dict{String, Any}("3bus_unbal_nogen_mn.loadbus" => Dict{String, Any}("va" => [-0.8302295799057324, -120.79137859901346, 119.2504000096645], "vm" => [7.472201725143692, 7.48392238161073, 7.479449806375467]), "3bus_unbal_nogen_mn.sourcebus" => Dict{String, Any}("va" => [-0.7706229194665752, -120.77035368944988, 119.22965458272802], "vm" => [125.00492912857915, 125.0049742054994, 125.00503001738106]), "3bus_unbal_nogen_mn.primary" => Dict{String, Any}("va" => [-0.7991413115394987, -120.78074600616584, 119.23848685007212], "vm" => [7.487284000425346, 7.492696895897028, 7.490619086774545]), "3bus_unbal_nogen_mn_2.primary" => Dict{String, Any}("va" => [-0.855042232531755, -120.83663088144945, 119.18261888118052], "vm" => [7.484002753336548, 7.489418049928727, 7.487339328524334]), "3bus_unbal_nogen_mn.substation" => Dict{String, Any}("va" => [-0.7715260550666929, -120.77101852028557, 119.22898904907676], "vm" => [7.500242832893439, 7.500243675517293, 7.50024705477223]), "3bus_unbal_nogen_mn_2.sourcebus" => Dict{String, Any}("va" => [-0.8264989754781323, -120.82622950863666, 119.173778769923], "vm" => [124.95033736057859, 124.95038246062096, 124.95043829748143]), "3bus_unbal_nogen_mn_2.substation" => Dict{String, Any}("va" => [-0.8274029028977026, -120.8268949215367, 119.17311265297246], "vm" => [7.496967303220576, 7.496968146567504, 7.496971527361585]), "3bus_unbal_nogen_mn_2.loadbus" => Dict{String, Any}("va" => [-0.8861577649359917, -120.84727282112708, 119.19454256329932], "vm" => [7.468913830909152, 7.480639684348965, 7.476165145424161])), "per_unit" => false), "1" => Dict("line" => Dict{String, Any}("3bus_unbal_nogen_mn.ohline" => Dict{String, Any}("qf" => [400.57771808779745, 449.1439873947322, 448.7567158126303], "qt" => [-400.38348814059987, -449.6175397346913, -449.40808066071014], "pt" => [-1202.204781396862, -900.9716820213755, -901.4375703783082], "pf" => [1204.09234222689, 901.8028241171684, 902.6707125964111]), "3bus_unbal_nogen_mn_2.quad" => Dict{String, Any}("qf" => [400.3857159634153, 449.6190932679933, 449.40944838961826], "qt" => [-399.999, -450.00000000000006, -450.00000000000006], "pt" => [-1200.0, -900.0000000000001, -900.0000000000001], "pf" => [1202.2067241982484, 900.9725348139676, 901.4388339183369]), "3bus_unbal_nogen_mn.quad" => Dict{String, Any}("qf" => [400.38348814059987, 449.6175397346913, 449.40808066071014], "qt" => [-399.999, -450.00000000000006, -450.00000000000006], "pt" => [-1200.0, -900.0000000000001, -900.0000000000001], "pf" => [1202.204781396862, 900.9716820213755, 901.4375703783082]), "3bus_unbal_nogen_mn_2.ohline" => Dict{String, Any}("qf" => [400.5820103772543, 449.14701672423075, 448.75940230921145], "qt" => [-400.3857159634153, -449.6190932679933, -449.40944838961826], "pt" => [-1202.2067241982484, -900.9725348139676, -901.4388339183369], "pf" => [1204.095948962688, 901.8044071742323, 902.6730611427464])), "settings" => Dict{String, Any}("sbase" => 100000.0), "transformer" => Dict{String, Any}("3bus_unbal_nogen_mn_2.subxf" => Dict{String, Any}("q" => [[400.60383553298436, 449.16077317266615, 448.77317528770436], [-400.5820103772543, -449.14701672423075, -448.75940230921145]], "p" => [[1204.0981315680738, 901.8057830578177, 902.6744392091184], [-1204.095948962688, -901.8044071742323, -902.6730611427464]]), "3bus_unbal_nogen_mn.subxf" => Dict{String, Any}("q" => [[400.59952401908913, 449.15773174687035, 448.7704766493643], [-400.57771808779745, -449.1439873947322, -448.7567158126303]], "p" => [[1204.0945221037925, 901.8041985455613, 902.6720886618943], [-1204.09234222689, -901.8028241171684, -902.6707125964111]])), "load" => Dict{String, Any}("3bus_unbal_nogen_mn_2.l1" => Dict{String, Any}("qd_bus" => [399.999], "pd_bus" => [1200.0], "qd" => [399.999], "pd" => [1200.0]), "3bus_unbal_nogen_mn.l1" => Dict{String, Any}("qd_bus" => [399.999], "pd_bus" => [1200.0], "qd" => [399.999], "pd" => [1200.0]), "3bus_unbal_nogen_mn_2.l3" => Dict{String, Any}("qd_bus" => [450.00000000000006], "pd_bus" => [900.0000000000001], "qd" => [450.00000000000006], "pd" => [900.0000000000001]), "3bus_unbal_nogen_mn.l2" => Dict{String, Any}("qd_bus" => [450.00000000000006], "pd_bus" => [900.0000000000001], "qd" => [450.00000000000006], "pd" => [900.0000000000001]), "3bus_unbal_nogen_mn_2.l2" => Dict{String, Any}("qd_bus" => [450.00000000000006], "pd_bus" => [900.0000000000001], "qd" => [450.00000000000006], "pd" => [900.0000000000001]), "3bus_unbal_nogen_mn.l3" => Dict{String, Any}("qd_bus" => [450.00000000000006], "pd_bus" => [900.0000000000001], "qd" => [450.00000000000006], "pd" => [900.0000000000001])), "bus" => Dict{String, Any}("3bus_unbal_nogen_mn.loadbus" => Dict{String, Any}("va" => [-0.830229579920092, -120.79137859902801, 119.25040000964974], "vm" => [7.4722017251430914, 7.483922381610104, 7.479449806374842]), "3bus_unbal_nogen_mn.sourcebus" => Dict{String, Any}("va" => [-0.7706229194809523, -120.77035368946459, 119.22965458271331], "vm" => [125.0049291285691, 125.00497420548916, 125.00503001737073]), "3bus_unbal_nogen_mn.primary" => Dict{String, Any}("va" => [-0.7991413115538781, -120.78074600618037, 119.23848685005738], "vm" => [7.4872840004247445, 7.4926968958964055, 7.490619086773919]), "3bus_unbal_nogen_mn_2.primary" => Dict{String, Any}("va" => [-0.8550422325640687, -120.83663088148208, 119.18261888114806], "vm" => [7.484002753335518, 7.489418049927689, 7.487339328523306]), "3bus_unbal_nogen_mn.substation" => Dict{String, Any}("va" => [-0.7715260550810701, -120.77101852030012, 119.22898904906208], "vm" => [7.5002428328928366, 7.500243675516679, 7.5002470547716085]), "3bus_unbal_nogen_mn_2.sourcebus" => Dict{String, Any}("va" => [-0.8264989755103133, -120.826229508669, 119.17377876989066], "vm" => [124.95033736056153, 124.95038246060375, 124.95043829746425]), "3bus_unbal_nogen_mn_2.substation" => Dict{String, Any}("va" => [-0.827402902930025, -120.82689492156933, 119.17311265293998], "vm" => [7.4969673032195505, 7.496968146566467, 7.496971527360553]), "3bus_unbal_nogen_mn_2.loadbus" => Dict{String, Any}("va" => [-0.8861577649683284, -120.84727282115972, 119.19454256326688], "vm" => [7.468913830908116, 7.4806396843479295, 7.476165145423132])), "per_unit" => false), "2" => Dict("line" => Dict{String, Any}("3bus_unbal_nogen_mn.ohline" => Dict{String, Any}("qf" => [400.5777180878409, 449.1439873945529, 448.75671581271985], "qt" => [-400.3834881405547, -449.6175397345132, -449.40808066079904], "pt" => [-1202.2047813968604, -900.9716820211088, -901.4375703779534], "pf" => [1204.0923422269757, 901.8028241166346, 902.6707125959689]), "3bus_unbal_nogen_mn_2.quad" => Dict{String, Any}("qf" => [400.3857159633255, 449.6190932685255, 449.40944838966266], "qt" => [-399.999, -450.00000000000006, -450.00000000000006], "pt" => [-1200.0, -900.0000000000001, -900.0000000000001], "pf" => [1202.2067241982472, 900.9725348141451, 901.4388339188698]), "3bus_unbal_nogen_mn.quad" => Dict{String, Any}("qf" => [400.3834881405547, 449.6175397345132, 449.40808066079904], "qt" => [-399.999, -450.00000000000006, -450.00000000000006], "pt" => [-1200.0, -900.0000000000001, -900.0000000000001], "pf" => [1202.2047813968604, 900.9716820211088, 901.4375703779534]), "3bus_unbal_nogen_mn_2.ohline" => Dict{String, Any}("qf" => [400.58201037707516, 449.1470167248505, 448.75940230925687], "qt" => [-400.3857159633255, -449.6190932685255, -449.40944838966266], "pt" => [-1202.2067241982472, -900.9725348141451, -901.4388339188698], "pf" => [1204.095948962507, 901.8044071744985, 902.6730611437251])), "settings" => Dict{String, Any}("sbase" => 100000.0), "transformer" => Dict{String, Any}("3bus_unbal_nogen_mn_2.subxf" => Dict{String, Any}("q" => [[400.6038355441561, 449.16077316190854, 448.7731752991101], [-400.58201037707516, -449.1470167248505, -448.75940230925687]], "p" => [[1204.098131204093, 901.8057832399824, 902.6744390281976], [-1204.095948962507, -901.8044071744985, -902.6730611437251]]), "3bus_unbal_nogen_mn.subxf" => Dict{String, Any}("q" => [[400.5995240305012, 449.157731758069, 448.77047663808975], [-400.5777180878409, -449.1439873945529, -448.75671581271985]], "p" => [[1204.0945224676755, 901.8041985450279, 902.6720886614527], [-1204.0923422269757, -901.8028241166346, -902.6707125959689]])), "load" => Dict{String, Any}("3bus_unbal_nogen_mn_2.l1" => Dict{String, Any}("qd_bus" => [399.999], "pd_bus" => [1200.0], "qd" => [399.999], "pd" => [1200.0]), "3bus_unbal_nogen_mn.l1" => Dict{String, Any}("qd_bus" => [399.999], "pd_bus" => [1200.0], "qd" => [399.999], "pd" => [1200.0]), "3bus_unbal_nogen_mn_2.l3" => Dict{String, Any}("qd_bus" => [450.00000000000006], "pd_bus" => [900.0000000000001], "qd" => [450.00000000000006], "pd" => [900.0000000000001]), "3bus_unbal_nogen_mn.l2" => Dict{String, Any}("qd_bus" => [450.00000000000006], "pd_bus" => [900.0000000000001], "qd" => [450.00000000000006], "pd" => [900.0000000000001]), "3bus_unbal_nogen_mn_2.l2" => Dict{String, Any}("qd_bus" => [450.00000000000006], "pd_bus" => [900.0000000000001], "qd" => [450.00000000000006], "pd" => [900.0000000000001]), "3bus_unbal_nogen_mn.l3" => Dict{String, Any}("qd_bus" => [450.00000000000006], "pd_bus" => [900.0000000000001], "qd" => [450.00000000000006], "pd" => [900.0000000000001])), "bus" => Dict{String, Any}("3bus_unbal_nogen_mn.loadbus" => Dict{String, Any}("va" => [-0.8302295799209277, -120.79137859902852, 119.25040000964935], "vm" => [7.472201725143105, 7.483922381610151, 7.479449806374883]), "3bus_unbal_nogen_mn.sourcebus" => Dict{String, Any}("va" => [-0.7706229194816114, -120.77035368946491, 119.22965458271298], "vm" => [125.00492912856943, 125.00497420548972, 125.00503001737137]), "3bus_unbal_nogen_mn.primary" => Dict{String, Any}("va" => [-0.79914131155471, -120.78074600618085, 119.23848685005699], "vm" => [7.48728400042476, 7.492696895896448, 7.4906190867739575]), "3bus_unbal_nogen_mn_2.primary" => Dict{String, Any}("va" => [-0.85504223255677, -120.83663088147543, 119.18261888115482], "vm" => [7.4840027533357, 7.489418049927836, 7.487339328523451]), "3bus_unbal_nogen_mn.substation" => Dict{String, Any}("va" => [-0.77152605508187, -120.77101852030057, 119.2289890490616], "vm" => [7.500242832892855, 7.5002436755167095, 7.500247054771646]), "3bus_unbal_nogen_mn_2.sourcebus" => Dict{String, Any}("va" => [-0.826498975503429, -120.82622950866259, 119.1737787698974], "vm" => [124.95033736056432, 124.95038246060629, 124.95043829746685]), "3bus_unbal_nogen_mn_2.substation" => Dict{String, Any}("va" => [-0.8274029029227165, -120.82689492156277, 119.17311265294684], "vm" => [7.496967303219722, 7.496968146566622, 7.49697152736071]), "3bus_unbal_nogen_mn_2.loadbus" => Dict{String, Any}("va" => [-0.8861577649610355, -120.84727282115303, 119.19454256327364], "vm" => [7.468913830908303, 7.480639684348074, 7.476165145423272])), "per_unit" => false), "3" => Dict("line" => Dict{String, Any}("3bus_unbal_nogen_mn.ohline" => Dict{String, Any}("qf" => [400.5777180877519, 449.1439873945082, 448.75671581263083], "qt" => [-400.38348814055513, -449.6175397345571, -449.4080806607989], "pt" => [-1202.20478139695, -900.9716820211972, -901.4375703778643], "pf" => [1204.092342226887, 901.8028241168124, 902.6707125957021]), "3bus_unbal_nogen_mn_2.quad" => Dict{String, Any}("qf" => [400.38571596332577, 449.6190932680813, 449.40944838961764], "qt" => [-399.999, -450.00000000000006, -450.00000000000006], "pt" => [-1200.0, -900.0000000000001, -900.0000000000001], "pf" => [1202.2067241981586, 900.9725348138778, 901.4388339183367]), "3bus_unbal_nogen_mn.quad" => Dict{String, Any}("qf" => [400.38348814055513, 449.6175397345571, 449.4080806607989], "qt" => [-399.999, -450.00000000000006, -450.00000000000006], "pt" => [-1200.0, -900.0000000000001, -900.0000000000001], "pf" => [1202.20478139695, 900.9716820211972, 901.4375703778643]), "3bus_unbal_nogen_mn_2.ohline" => Dict{String, Any}("qf" => [400.5820103769862, 449.14701672440617, 448.759402309122], "qt" => [-400.38571596332577, -449.6190932680813, -449.40944838961764], "pt" => [-1202.2067241981586, -900.9725348138778, -901.4388339183367], "pf" => [1204.0959489624188, 901.8044071739638, 902.6730611429242])), "settings" => Dict{String, Any}("sbase" => 100000.0), "transformer" => Dict{String, Any}("3bus_unbal_nogen_mn_2.subxf" => Dict{String, Any}("q" => [[400.6038355327163, 449.16077315009466, 448.773175287606], [-400.5820103769862, -449.14701672440617, -448.759402309122]], "p" => [[1204.0981315678043, 901.8057826937508, 902.6744388454978], [-1204.0959489624188, -901.8044071739638, -902.6730611429242]]), "3bus_unbal_nogen_mn.subxf" => Dict{String, Any}("q" => [[400.5995240190437, 449.1577317239183, 448.7704766379962], [-400.5777180877519, -449.1439873945082, -448.75671581263083]], "p" => [[1204.0945224675868, 901.8041985452063, 902.6720886611851], [-1204.092342226887, -901.8028241168124, -902.6707125957021]])), "load" => Dict{String, Any}("3bus_unbal_nogen_mn_2.l1" => Dict{String, Any}("qd_bus" => [399.999], "pd_bus" => [1200.0], "qd" => [399.999], "pd" => [1200.0]), "3bus_unbal_nogen_mn.l1" => Dict{String, Any}("qd_bus" => [399.999], "pd_bus" => [1200.0], "qd" => [399.999], "pd" => [1200.0]), "3bus_unbal_nogen_mn_2.l3" => Dict{String, Any}("qd_bus" => [450.00000000000006], "pd_bus" => [900.0000000000001], "qd" => [450.00000000000006], "pd" => [900.0000000000001]), "3bus_unbal_nogen_mn.l2" => Dict{String, Any}("qd_bus" => [450.00000000000006], "pd_bus" => [900.0000000000001], "qd" => [450.00000000000006], "pd" => [900.0000000000001]), "3bus_unbal_nogen_mn_2.l2" => Dict{String, Any}("qd_bus" => [450.00000000000006], "pd_bus" => [900.0000000000001], "qd" => [450.00000000000006], "pd" => [900.0000000000001]), "3bus_unbal_nogen_mn.l3" => Dict{String, Any}("qd_bus" => [450.00000000000006], "pd_bus" => [900.0000000000001], "qd" => [450.00000000000006], "pd" => [900.0000000000001])), "bus" => Dict{String, Any}("3bus_unbal_nogen_mn.loadbus" => Dict{String, Any}("va" => [-0.8302295799131064, -120.79137859902089, 119.25040000965717], "vm" => [7.4722017251435435, 7.483922381610589, 7.479449806375325]), "3bus_unbal_nogen_mn.sourcebus" => Dict{String, Any}("va" => [-0.7706229194739376, -120.77035368945725, 119.22965458272064], "vm" => [125.00492912857676, 125.00497420549705, 125.00503001737864]), "3bus_unbal_nogen_mn.primary" => Dict{String, Any}("va" => [-0.7991413115468873, -120.78074600617322, 119.23848685006479], "vm" => [7.487284000425199, 7.492696895896888, 7.490619086774399]), "3bus_unbal_nogen_mn_2.primary" => Dict{String, Any}("va" => [-0.85504223254311, -120.8366308814605, 119.18261888116959], "vm" => [7.484002753336263, 7.489418049928443, 7.487339328524058]), "3bus_unbal_nogen_mn.substation" => Dict{String, Any}("va" => [-0.7715260550740553, -120.77101852029291, 119.2289890490694], "vm" => [7.500242832893295, 7.500243675517153, 7.500247054772084]), "3bus_unbal_nogen_mn_2.sourcebus" => Dict{String, Any}("va" => [-0.8264989754893535, -120.82622950864773, 119.17377876991193], "vm" => [124.95033736057383, 124.95038246061625, 124.95043829747675]), "3bus_unbal_nogen_mn_2.substation" => Dict{String, Any}("va" => [-0.8274029029090649, -120.82689492154778, 119.17311265296155], "vm" => [7.496967303220288, 7.496968146567221, 7.496971527361306]), "3bus_unbal_nogen_mn_2.loadbus" => Dict{String, Any}("va" => [-0.8861577649473606, -120.84727282113812, 119.19454256328841], "vm" => [7.468913830908865, 7.480639684348685, 7.4761651454238836])), "per_unit" => false)))

[ Main | Info ] : Solution Values for Boundary(ies): Dict{String, Any}("multinetwork" => true, "nw" => Dict{String, Any}("4" => Dict{String, Any}("boundary" => Dict{String, Any}("(100002, voltage_source.3bus_unbal_nogen_mn_2.source, 6)" => Dict{String, Any}("pbound_to" => [-1204.1049622880614, -901.8100181254906, -902.6783506307055], "qbound_to" => [-400.630030776433, -449.177722118188, -448.78980036204024]), "(100002, 6, voltage_source.3bus_unbal_nogen_mn_2.source)" => Dict{String, Any}("pbound_fr" => [3008.593331044258], "qbound_fr" => [1298.597553256661]), "(100001, 5, voltage_source.3bus_unbal_nogen_mn.source)" => Dict{String, Any}("pbound_fr" => [3008.5857746079846], "qbound_fr" => [1298.5874491212335]), "(100001, voltage_source.3bus_unbal_nogen_mn.source, 5)" => Dict{String, Any}("pbound_to" => [-1204.101347358377, -901.8084302478388, -902.6759970017687], "qbound_to" => [-400.6256962072555, -449.17466583314587, -448.7870870808322]))), "1" => Dict{String, Any}("boundary" => Dict{String, Any}("(100002, voltage_source.3bus_unbal_nogen_mn_2.source, 6)" => Dict{String, Any}("pbound_to" => [-1204.1049624672987, -901.8100184922217, -902.6783509913059], "qbound_to" => [-400.6300307993883, -449.17772215213427, -448.78980037337016]), "(100002, 6, voltage_source.3bus_unbal_nogen_mn_2.source)" => Dict{String, Any}("pbound_fr" => [3008.5933319508263], "qbound_fr" => [1298.5975533248927]), "(100001, 5, voltage_source.3bus_unbal_nogen_mn.source)" => Dict{String, Any}("pbound_fr" => [3008.5857742450717], "qbound_fr" => [1298.5874491439454]), "(100001, voltage_source.3bus_unbal_nogen_mn.source, 5)" => Dict{String, Any}("pbound_to" => [-1204.1013469861416, -901.8084302538781, -902.6759970050523], "qbound_to" => [-400.6256961956958, -449.17466584472527, -448.78708710352413]))), "2" => Dict{String, Any}("boundary" => Dict{String, Any}("(100002, voltage_source.3bus_unbal_nogen_mn_2.source, 6)" => Dict{String, Any}("pbound_to" => [-1204.1049621118384, -901.8100186687032, -902.6783508103842], "qbound_to" => [-400.63003079917036, -449.17772213001075, -448.7898003847704]), "(100002, 6, voltage_source.3bus_unbal_nogen_mn_2.source)" => Dict{String, Any}("pbound_fr" => [3008.593331590926], "qbound_fr" => [1298.5975533139513]), "(100001, 5, voltage_source.3bus_unbal_nogen_mn.source)" => Dict{String, Any}("pbound_fr" => [3008.585774607983], "qbound_fr" => [1298.5874491439235]), "(100001, voltage_source.3bus_unbal_nogen_mn.source, 5)" => Dict{String, Any}("pbound_to" => [-1204.1013473528665, -901.8084302476616, -902.6759970074544], "qbound_to" => [-400.6256962184772, -449.1746658331913, -448.7870870922549]))), "3" => Dict{String, Any}("boundary" => Dict{String, Any}("(100002, voltage_source.3bus_unbal_nogen_mn_2.source, 6)" => Dict{String, Any}("pbound_to" => [-1204.1049624727136, -901.8100181281534, -902.6783506390512], "qbound_to" => [-400.6300307763823, -449.1777221295564, -448.7898003846302]), "(100002, 6, voltage_source.3bus_unbal_nogen_mn_2.source)" => Dict{String, Any}("pbound_fr" => [3008.5933312399184], "qbound_fr" => [1298.5975532905688]), "(100001, 5, voltage_source.3bus_unbal_nogen_mn.source)" => Dict{String, Any}("pbound_fr" => [3008.58577461065], "qbound_fr" => [1298.587449086865]), "(100001, voltage_source.3bus_unbal_nogen_mn.source, 5)" => Dict{String, Any}("pbound_to" => [-1204.1013473527794, -901.8084302506836, -902.6759970071865], "qbound_to" => [-400.62569618428887, -449.1746658104148, -448.7870870921613])))))
```

As you can see on the output, the dictionary results have extra keys such as `nw`, which indicate that this is a multinetwork (multi-timestep) optimization problem. Below the `nw`, you can find the solutions of the respective multinetwork (i.e., timestep) as **"#"** (e.g., ```["nw"]["3"]``` represents the results for the **3th** timestep).

## Example Case #3: Solving OPFITD with voltage bounds (support for PowerModelsDistribution (PMD) Transformations)

### Test Case #3

In this example case, we want to solve the integrated T&D optimal power flow (opfitd) problem for a case where we have the **PJM 5-bus system** (`case5_withload`) as the _transmission system_ and the **IEEE 4 Node Test Feeder** (`case3_unbalanced.dss`) as the _distribution system_. The boundary information that specifies the respective boundary buses for both the transmission and distribution systems can be found in `case5_case3_unbal.json`.

The formulation used to solve this optimization problem is the **ACP-ACPU** formulation.

Additionally, we are specifying a `PMD` transformation for the **distribution system data**. This **transformation** applies **voltage bounds** to all the nodes in the distribution system(s). The function is called `apply_voltage_bounds!(...)`.

### Julia Code for Test Case #3

```julia
using PowerModelsITD
using Ipopt

silence!()  # Silences the console, so warning messages or information are not displayed

files_path = joinpath(dirname(pathof(PowerModelsITD)), "..")                    # file path for test cases files
pm_file = joinpath(files_path, "test/data/transmission/case5_withload.m")       # transmission system file
pmd_file = joinpath(files_path, "test/data/distribution/case3_unbalanced.dss")  # distribution system file
boundary_file = joinpath(files_path, "test/data/json/case5_case3_unbal.json")   # boundary file with info. about boundary buses
formulation = NLPowerModelITD{ACPPowerModel, ACPUPowerModel}                    # type of formulation to be used
data = parse_files(pm_file, pmd_file, boundary_file)                        # parse files into dictionary
# apply transformations
apply_voltage_bounds!(data; vm_lb=0.99, vm_ub=1.01)
# solve the optimization problem
result = solve_opfitd(data, formulation, Ipopt.Optimizer; make_si=false)                   # solve the opfitd

# Check the result
@info "Objective: $(result["objective"]) [\$/hr]"
@info "Termination Status of Solver: $(result["termination_status"])"
@info "Solve Time: $(result["solve_time"]) [seconds]"
@info "Solution Values for Distribution system(s): $(result["solution"]["it"]["pmd"]["bus"])"
```

### Console Output for Test Case #3

After running the corresponding code, the user should get the following output displayed on the console.

```julia
[ Main | Info ] : Objective: 18022.355809587236 [$/hr]
[ Main | Info ] : Termination Status of Solver: LOCALLY_SOLVED
[ Main | Info ] : Solve Time: 0.06421494483947754 [seconds]

[ Main | Info ] : Solution Values for Distribution system(s): Dict{String, Any}("3bus_unbal.loadbus" => Dict{String, Any}("va" => [-0.8977463887653532, -120.78309685941274, 119.34177620647625], "vm" => [0.9899999900302301, 0.9946671975187621, 0.9928991397184561]), "3bus_unbal.substation" => Dict{String, Any}("va" => [-0.7693363623113275, -120.76782278704773, 119.23219208890347], "vm" => [0.999742728740167, 0.999743306402737, 0.9997446568367797]), "3bus_unbal.primary" => Dict{String, Any}("va" => [-0.8297020935620707, -120.77556386286943, 119.28178890967312], "vm" => [0.9952384467648019, 0.9973943491437465, 0.9965725113327532]), "3bus_unbal.sourcebus" => Dict{String, Any}("va" => [-0.7671158858601453, -120.76631338622572, 119.23370678684667], "vm" => [0.9997629892456953, 0.9997641961144075, 0.9997655262819723]))
```

You can see in these results that the voltage magnitude values of the distribution system nodes is between the `0.99` and `1.01` bounds.

## Example Case #4: Solving OPFITD with Solution Processors (apply solution processor to convert voltage values solutions from `rectangular` to `polar` coordinates)

### Test Case #4

In this example case, we want to solve the integrated T&D optimal power flow (opfitd) problem for a case where we have the **PJM 5-bus system** (`case5_withload`) as the _transmission system_ and the **IEEE 4 Node Test Feeder** (`case3_unbalanced.dss`) as the _distribution system_. The boundary information that specifies the respective boundary buses for both the transmission and distribution systems can be found in `case5_case3_unbal.json`.

Additionally, we are specificying a `PMD` transformation for the distribution system data. This **transformation** applies **voltage bounds** to all the nodes in the distribution system(s). The function is called `apply_voltage_bounds!(...)`.

Finally, we also want to solve this problem using the **ACR-ACRU** formulation (i.e., AC OPF in **rectangular** coordinates) instead of the ACP-ACPU formulation used previously. We also want to show that by applying the `solution processor` to the `solve_opfitd(...)` function, we can convert the solution from **rectangular** to **polar** coordinates (after the problem has been solved in **rectangular** coordinates).

### Julia Code for Test Case #4

```julia
using PowerModelsITD
using Ipopt

silence!()  # Silences the console, so warning messages or information are not displayed

files_path = joinpath(dirname(pathof(PowerModelsITD)), "..")                    # file path for test cases files
pm_file = joinpath(files_path, "test/data/transmission/case5_withload.m")       # transmission system file
pmd_file = joinpath(files_path, "test/data/distribution/case3_unbalanced.dss")  # distribution system file
boundary_file = joinpath(files_path, "test/data/json/case5_case3_unbal.json")   # boundary file with info. about boundary buses
formulation = NLPowerModelITD{ACRPowerModel, ACRUPowerModel}                    # type of formulation to be used

data_rect = parse_files(pm_file, pmd_file, boundary_file)                       # parse files into dictionary - rectangular
data_polar = deepcopy(data_rect)                                                # parse files into dictionary - polar

# apply voltage bounds to both datasets
apply_voltage_bounds!(data_rect; vm_lb=0.99, vm_ub=1.01)
apply_voltage_bounds!(data_polar; vm_lb=0.99, vm_ub=1.01)

# solve the optimization problem
result_rect = solve_opfitd(data_rect, formulation, Ipopt.Optimizer; make_si=false)                                          # solve the opfitd, present result in rectangular
result_polar = solve_opfitd(data_polar, formulation, Ipopt.Optimizer; make_si=false, solution_processors=[sol_data_model!]) # solve the opfitd, present the result in polar

# Check the result
@info "Distribution system(s) voltages in rectangular coordinates: $(result_rect["solution"]["it"]["pmd"]["bus"])"
@info "Distribution system(s) voltages in polar coordinates: $(result_polar["solution"]["it"]["pmd"]["bus"])"
```

### Console Output for Test Case #4

After running the corresponding code, the user should get the following output displayed on the console.

```julia
[ Main | Info ] : Distribution system(s) voltages in rectangular coordinates: Dict{String, Any}("3bus_unbal.loadbus" => Dict{String, Any}("vi" => [-0.015511309363640641, -0.8545294554234174, 0.8655223138026301], "vr" => [0.9898784720063777, -0.5090601668099762, -0.4865386272048753]), "3bus_unbal.substation" => Dict{String, Any}("vi" => [-0.013423594907906103, -0.8590267647200966, 0.872425011363218], "vr" => [0.9996526099437699, -0.5114290820733716, -0.48822534573738224]), "3bus_unbal.primary" => Dict{String, Any}("vi" => [-0.014411577636549404, -0.8569394862376708, 0.8692352392326611], "vr" => [0.9951341025954522, -0.5103432319817417, -0.4874288450833038]), "3bus_unbal.sourcebus" => Dict{String, Any}("vi" => [-0.01338512498983875, -0.8590581872270858, 0.8724303155121654], "vr" => [0.9996733881089848, -0.5114171375591623, -0.4882586014767347]))

[ Main | Info ] : Distribution system(s) voltages in polar coordinates: Dict{String, Any}("3bus_unbal.loadbus" => Dict{String, Any}("va" => [-0.8977463956228972, -120.78309686739179, 119.34177619725007], "vm" => [0.989999994979725, 0.9946672024445378, 0.9928991446530021]), "3bus_unbal.substation" => Dict{String, Any}("va" => [-0.7693363704348831, -120.76782279518537, 119.23219208076581], "vm" => [0.9997427336408314, 0.9997433113034004, 0.9997446617374346]), "3bus_unbal.primary" => Dict{String, Any}("va" => [-0.8297021010938084, -120.77556387092758, 119.28178890104428], "vm" => [0.9952384516880516, 0.9973943540560338, 0.9965725162491212]), "3bus_unbal.sourcebus" => Dict{String, Any}("va" => [-0.767115894005601, -120.7663133943784, 119.23370677869384], "vm" => [0.9997629941462579, 0.9997642010149662, 0.9997655311825235]))
```

## Example Case #5: Solving OPFITD with Results in `MATH` Model (instead of default `ENG`)

### Test Case #5

In this example case, we want to solve the integrated T&D optimal power flow (opfitd) problem for a case where we have the **PJM 5-bus system** (`case5_withload`) as the _transmission system_ and the **IEEE 4 Node Test Feeder** (`case3_unbalanced.dss`) as the _distribution system_. The boundary information that specifies the respective boundary buses for both the transmission and distribution systems can be found in `case5_case3_unbal.json`.

The formulation used to solve this optimization problem is the **ACP-ACPU** formulation.

However, different from all the examples presented above, in this case, the result of the **distribution systems** will be obtained in the **MATH** model instead of the default **ENG** model. See [here](https://lanl-ansi.github.io/PowerModelsDistribution.jl/stable/manual/eng-data-model.html) for a more detailed explanation regarding the different models supported by `PMD/PMITD`.

### Julia Code for Test Case #5

```julia
using PowerModelsITD
using Ipopt

silence!()  # Silences the console, so warning messages or information are not displayed

files_path = joinpath(dirname(pathof(PowerModelsITD)), "..")                    # file path for test cases files
pm_file = joinpath(files_path, "test/data/transmission/case5_withload.m")       # transmission system file
pmd_file = joinpath(files_path, "test/data/distribution/case3_unbalanced.dss")  # distribution system file
boundary_file = joinpath(files_path, "test/data/json/case5_case3_unbal.json")   # boundary file with info. about boundary buses
formulation = NLPowerModelITD{ACPPowerModel, ACPUPowerModel}                    # type of formulation to be used
data = parse_files(pm_file, pmd_file, boundary_file)                            # parse files into dictionary
result = solve_opfitd(data, formulation, Ipopt.Optimizer; solution_model="MATH") # solve the opfitd, present result in MATH model instead of default ENG

# Check the result
@info "Objective: $(result["objective"]) [\$/hr]"
@info "Termination Status of Solver: $(result["termination_status"])"
@info "Solve Time: $(result["solve_time"]) [seconds]"
@info "Solution Values (MATH model) for Distribution system(s): $(result["solution"]["it"]["pmd"])"
@info "Solution Values (MATH model) for Boundary(ies): $(result["solution"]["it"]["pmitd"])"

```

### Console Output for Test Case #5

After running the corresponding code, the user should get the following output displayed on the console.

```julia
[ Main | Info ] : Objective: 17953.746512039743 [$/hr]
[ Main | Info ] : Termination Status of Solver: LOCALLY_SOLVED
[ Main | Info ] : Solve Time: 0.07705187797546387 [seconds]

[ Main | Info ] : Solution Values (MATH model) for Distribution system(s): Dict{String, Any}("branch" => Dict{String, Any}("4" => Dict{String, Any}("qf" => [1354.795202241865, 1507.3400640138161, 1504.5761375753734], "qt" => [-1354.6163225169614, -1507.2344513630767, -1504.4700979169072], "pt" => [-3367.119671256992, -2346.3650144634003, -2354.829891017304], "pf" => [3367.119671256992, 2346.3650144634003, 2354.8298910173044]), "1" => Dict{String, Any}("qf" => [1344.7785535214057, 1503.9392726830868, 1502.43791779139], "qt" => [-1333.3300000000002, -1500.0000000000002, -1500.0000000000002], "pt" => [-3333.3323371531164, -2333.3323371935803, -2333.3323371900296], "pf" => [3351.5305232165006, 2340.3516279866967, 2344.8964126191295]), "5" => Dict{String, Any}("qf" => [1354.7952022418647, 1507.3400640138161, 1504.5761375753734], "qt" => [-1354.795202241865, -1507.3400640138161, -1504.5761375753734], "pt" => [-3367.119671256992, -2346.3650144634003, -2354.8298910173044], "pf" => [3367.1286152278885, 2346.3702946258413, 2354.835193007617]), "2" => Dict{String, Any}("qf" => [1354.6163225169614, 1507.2344513630765, 1504.4700979169072], "qt" => [-1344.7785535214057, -1503.9392726830868, -1502.43791779139], "pt" => [-3351.5305232165006, -2340.3516279866967, -2344.8964126191295], "pf" => [3367.110727286094, 2346.3597337552624, 2354.8245890269895]), "6" => Dict{String, Any}("qf" => [1355.0096276913357, 1507.4706735124998, 1504.7037665348369], "qt" => [-1354.7952022418647, -1507.3400640138161, -1504.5761375753734], "pt" => [-3367.1286152278885, -2346.3702946258413, -2354.835193007617], "pf" => [3367.1850486607445, 2346.4026604158908, 2354.8649645151804]), "3" => Dict{String, Any}("qf" => [-1354.6163225169614, -1507.2344513630765, -1504.4700979169072], "qt" => [1354.6163225169614, 1507.2344513630767, 1504.4700979169072], "pt" => [3367.119671256992, 2346.3650144634003, 2354.829891017304], "pf" => [-3367.110727286094, -2346.3597337552624, -2354.8245890269895])), "gen" => Dict{String, Any}("1" => Dict{String, Any}("qg_bus" => [-0.0, -0.0, -0.0], "qg" => [-0.0, -0.0, -0.0], "pg" => [666.6676628468832, 666.6676628064197, 666.6676628099706], "pg_bus" => [666.6676628468832, 666.6676628064197, 666.6676628099706])), "transformer" => Dict{String, Any}("1" => Dict{String, Any}("qf" => [1354.7952022418647, 1507.3400640138161, 1504.5761375753734], "qt" => [-1354.7952022418647, -1507.3400640138161, -1504.5761375753734], "pt" => [-3367.1286152278885, -2346.3702946258413, -2354.835193007617], "pf" => [3367.1286152278885, 2346.3702946258413, 2354.835193007617]), "2" => Dict{String, Any}("qf" => [-1354.6163225169614, -1507.2344513630765, -1504.4700979169072], "qt" => [1354.6163225169614, 1507.2344513630765, 1504.4700979169072], "pt" => [3367.110727286094, 2346.3597337552624, 2354.8245890269895], "pf" => [-3367.110727286094, -2346.3597337552624, -2354.8245890269895])), "settings" => Dict{String, Any}("sbase_default" => 100000.0, "vbases_default" => Dict{String, Real}("4" => 132.79056191361394), "voltage_scale_factor" => 1000.0, "sbase" => 100000.0, "power_scale_factor" => 1000.0, "base_frequency" => 60.0), "multinetwork" => false, "load" => Dict{String, Any}("1" => Dict{String, Any}("qd_bus" => [1500.0000000000002], "pd_bus" => [3000.0000000000005], "qd" => [1500.0000000000002], "pd" => [3000.0000000000005]), "2" => Dict{String, Any}("qd_bus" => [1500.0000000000002], "pd_bus" => [3000.0000000000005], "qd" => [1500.0000000000002], "pd" => [3000.0000000000005]), "3" => Dict{String, Any}("qd_bus" => [1333.3300000000002], "pd_bus" => [4000.0], "qd" => [1333.3300000000002], "pd" => [4000.0])), "bus" => Dict{String, Any}("8" => Dict{String, Any}("va" => [-0.8064197613818459, -120.80551025323088, 119.19451148306055], "vm" => [0.5427404621782935, 0.5427412225103798, 0.5427420415368795]), "4" => Dict{String, Any}("va" => [-0.8064197613818459, -120.80551025323088, 119.19451148306055], "vm" => [124.83030630100751, 124.83048117738736, 124.8306695534823]), "1" => Dict{String, Any}("va" => [-0.9542852429015141, -120.82463162589393, 119.31695673367102], "vm" => [7.406969380817659, 7.446613395025093, 7.431619824571949]), "5" => Dict{String, Any}("va" => [-0.8089868491003656, -120.80727720360223, 119.19273784447546], "vm" => [0.5427292370876581, 0.542729249043147, 0.5427300853404582]), "2" => Dict{String, Any}("va" => [-0.8089341499650244, -120.8072185671511, 119.19279637320398], "vm" => [7.489646348633082, 7.489651704574486, 7.489663202448464]), "6" => Dict{String, Any}("va" => [-0.8089341499650244, -120.8072185671511, 119.19279637320398], "vm" => [0.5427279962777596, 0.5427283843894555, 0.5427292175687293]), "7" => Dict{String, Any}("va" => [-0.8063670574676239, -120.8054516152584, 119.19457001333551], "vm" => [0.5427392213907619, 0.5427403578738178, 0.5427411737823131]), "9" => Dict{String, Any}("va" => [-0.8036120919989638, -120.80361209199894, 119.19638790800104], "vm" => [124.83485996888251, 124.83485996888251, 124.83485996888251]), "3" => Dict{String, Any}("va" => [-0.8772138344778904, -120.81603057758366, 119.24896754768255], "vm" => [7.4514218741879255, 7.4697354940559455, 7.462765580863505])), "per_unit" => false)

[ Main | Info ] : Solution Values (MATH model) for Boundary(ies): Dict{String, Any}("multinetwork" => false, "boundary" => Dict{String, Any}("(100001, 9, 5)" => Dict{String, Any}("pbound_to" => [-3367.1850486607445, -2346.4026604158908, -2354.8649645151804], "qbound_to" => [-1355.0096276913357, -1507.4706735124998, -1504.7037665348369]), "(100001, 5, 9)" => Dict{String, Any}("pbound_fr" => [8068.452673591816], "qbound_fr" => [4367.184067738673])))
```

As seen in the console output, the nodes/buses are now referenced by their `MATH` names (i.e., numbers) and not their actual `ENG` name.
