import dd from "dd-trace";
import { NodeSDK } from "@opentelemetry/sdk-node";
import { TraceIdRatioBasedSampler } from "@opentelemetry/sdk-trace-base";
import { OTLPTraceExporter } from "@opentelemetry/exporter-trace-otlp-proto";
import { IORedisInstrumentation } from "@opentelemetry/instrumentation-ioredis";
import { HttpInstrumentation } from "@opentelemetry/instrumentation-http";
import { PrismaInstrumentation } from "@prisma/instrumentation";
import { WinstonInstrumentation } from "@opentelemetry/instrumentation-winston";
import { AwsInstrumentation } from "@opentelemetry/instrumentation-aws-sdk";
import { BullMQInstrumentation } from "@appsignal/opentelemetry-instrumentation-bullmq";
import {
  envDetector,
  processDetector,
  Resource,
} from "@opentelemetry/resources";
import { awsEcsDetectorSync } from "@opentelemetry/resource-detector-aws";
import { containerDetector } from "@opentelemetry/resource-detector-container";
import { env } from "@/src/env.mjs";

if (!process.env.VERCEL && process.env.NEXT_PUBLIC_LANGFUSE_CLOUD_REGION) {
  dd.init({
    runtimeMetrics: true,
    plugins: false,
  });

  const sdk = new NodeSDK({
    resource: new Resource({
      "service.name": env.OTEL_SERVICE_NAME,
    }),
    traceExporter: new OTLPTraceExporter({
      url: `${env.OTEL_EXPORTER_OTLP_ENDPOINT}/v1/traces`,
    }),
    instrumentations: [
      new IORedisInstrumentation(),
      new HttpInstrumentation({
        requireParentforOutgoingSpans: true,
        ignoreOutgoingRequestHook: (req) => {
          return req.host === "127.0.0.1";
        },
        requestHook: (span, req: any) => {
          const url = "path" in req ? req?.path : req?.url;
          let path = new URL(url, `http://${req?.host ?? "localhost"}`)
            .pathname;
          if (path.startsWith("/_next/static")) {
            path = "/_next/static/*";
          }
          span.updateName(`${req?.method} ${path}`);
          span.setAttribute("http.route", path);
        },
      }),
      new PrismaInstrumentation(),
      new AwsInstrumentation(),
      new WinstonInstrumentation({ disableLogSending: true }),
      new BullMQInstrumentation({ useProducerSpanAsConsumerParent: true }),
    ],
    resourceDetectors: [
      envDetector,
      processDetector,
      awsEcsDetectorSync,
      containerDetector,
    ],
    sampler: new TraceIdRatioBasedSampler(env.OTEL_TRACE_SAMPLING_RATIO),
  });

  sdk.start();
}
