<?php declare(strict_types=1);

/*
 * This file is part of PHP CS Fixer: custom fixers.
 *
 * (c) 2018 Kuba Werłos
 *
 * For the full copyright and license information, please view
 * the LICENSE file that was distributed with this source code.
 */

namespace PhpCsFixerCustomFixers\Fixer;

use PhpCsFixer\FixerDefinition\CodeSample;
use PhpCsFixer\FixerDefinition\FixerDefinition;
use PhpCsFixer\FixerDefinition\FixerDefinitionInterface;
use PhpCsFixer\Tokenizer\Tokens;
use PhpCsFixerCustomFixers\TokenRemover;

final class NoDoctrineMigrationsGeneratedCommentFixer extends AbstractFixer
{
    public function getDefinition(): FixerDefinitionInterface
    {
        return new FixerDefinition(
            'There must be no comments generated by Doctrine Migrations.',
            [new CodeSample('<?php
namespace Migrations;
use Doctrine\\DBAL\\Schema\\Schema;
/**
 * Auto-generated Migration: Please modify to your needs!
 */
final class Version20180609123456 extends AbstractMigration
{
    public function up(Schema $schema)
    {
        // this up() migration is auto-generated, please modify it to your needs
        $this->addSql("UPDATE t1 SET col1 = col1 + 1");
    }
    public function down(Schema $schema)
    {
        // this down() migration is auto-generated, please modify it to your needs
        $this->addSql("UPDATE t1 SET col1 = col1 - 1");
    }
}
')],
            '',
        );
    }

    /**
     * Must run before BracesFixer.
     */
    public function getPriority(): int
    {
        return 36;
    }

    public function isCandidate(Tokens $tokens): bool
    {
        return $tokens->isAnyTokenKindsFound([\T_COMMENT, \T_DOC_COMMENT]);
    }

    public function isRisky(): bool
    {
        return false;
    }

    public function fix(\SplFileInfo $file, Tokens $tokens): void
    {
        for ($index = $tokens->count() - 1; $index > 0; $index--) {
            if (!$tokens[$index]->isGivenKind([\T_COMMENT, \T_DOC_COMMENT])) {
                continue;
            }

            if (
                \strpos($tokens[$index]->getContent(), 'Auto-generated Migration: Please modify to your needs!') === false
                && \strpos($tokens[$index]->getContent(), 'this up() migration is auto-generated, please modify it to your needs') === false
                && \strpos($tokens[$index]->getContent(), 'this down() migration is auto-generated, please modify it to your needs') === false
            ) {
                continue;
            }

            TokenRemover::removeWithLinesIfPossible($tokens, $index);
        }
    }
}
