<?php

/**
 * This file is part of the Krystal Framework
 * 
 * Copyright (c) No Global State Lab
 * 
 * For the full copyright and license information, please view
 * the license file that was distributed with this source code.
 */

namespace Krystal\Validate;

use Krystal\Validate\Input\InputValidator;
use Krystal\Validate\File\FileValidator;
use Krystal\Validate\Renderer\RendererInterface;
use Krystal\I18n\TranslatorInterface;

final class ValidatorFactory implements FactoryInterface
{
    /**
     * Translator service
     * 
     * @var \Krystal\I18n\TranslatorInterface
     */
    private $translator;

    /**
     * Defines renderer for error messages
     * 
     * @var \Krystal\Validate\Renderer\RendererInterface
     */
    private $renderer;

    /**
     * State initialization
     * 
     * @param \Krystal\Validate\Renderer\RendererInterface $renderer
     * @param \Krystal\I18n\TranslatorInterface $translator Optional translator service
     * @return void
     */
    public function __construct(RendererInterface $renderer, TranslatorInterface $translator = null)
    {
        $this->setRenderer($renderer);
        $this->translator = $translator;
    }

    /**
     * Sets or overrides default renderer
     * 
     * @param \Krystal\Validate\Renderer\RendererInterface $renderer
     * @return void
     */
    public function setRenderer(RendererInterface $renderer)
    {
        $this->renderer = $renderer;
    }

    /**
     * Builds the instance
     * 
     * @param array $validators
     * @return \Krystal\Validate\ValidatorChain
     */
    public function build(array $validators)
    {
        $chain = array();

        foreach ($validators as $validator => $options) {
            switch (strtolower($validator)) {
                case 'input':
                    $chain[] = InputValidator::factory($options['source'], $options['definition'], $this->translator);
                break;

                case 'file':
                    $chain[] = FileValidator::factory($options['source'], $options['definition'], $this->translator);
                break;
            }
        }

        return new ValidatorChain($chain, $this->renderer);
    }
}
