from keras_hub.src.api_export import keras_hub_export
from keras_hub.src.models.phi3.phi3_backbone import Phi3Backbone
from keras_hub.src.tokenizers.sentence_piece_tokenizer import (
    SentencePieceTokenizer,
)


@keras_hub_export(
    [
        "keras_hub.tokenizers.Phi3Tokenizer",
        "keras_hub.models.Phi3Tokenizer",
    ]
)
class Phi3Tokenizer(SentencePieceTokenizer):
    """Phi3 tokenizer layer based on SentencePiece.

    This tokenizer class will tokenize raw strings into integer sequences and
    is based on `keras_hub.tokenizers.SentencePieceTokenizer`. Unlike the
    underlying tokenizer, it will check for all special tokens needed by
    Phi3 models and provides a `from_preset()` method to automatically
    download a matching vocabulary for a Phi3 preset.

    If input is a batch of strings (rank > 0), the layer will output a
    `tf.RaggedTensor` where the last dimension of the output is ragged.

    If input is a scalar string (rank == 0), the layer will output a dense
    `tf.Tensor` with static shape `[None]`.

    Args:
        proto: Either a `string` path to a SentencePiece proto file, or a
            `bytes` object with a serialized SentencePiece proto. See the
            [SentencePiece repository](https://github.com/google/sentencepiece)
            for more details on the format.

    Examples:
    ```python
    # Unbatched input.
    tokenizer = keras_hub.models.Phi3Tokenizer.from_preset(
        "phi3_mini_4k_instruct_en",
    )
    tokenizer("The quick brown fox jumped.")

    # Batched input.
    tokenizer(["The quick brown fox jumped.", "The fox slept."])

    # Detokenization.
    tokenizer.detokenize(tokenizer("The quick brown fox jumped."))
    ```
    """

    backbone_cls = Phi3Backbone

    def __init__(self, proto, **kwargs):
        self._add_special_token("<s>", "start_token")
        self._add_special_token("<|endoftext|>", "end_token")
        self.pad_token_id = 0
        super().__init__(proto=proto, **kwargs)
