﻿using Microsoft.Toolkit.Mvvm.Input;
using ProjectLex.InventoryManagement.Database.Models;
using ProjectLex.InventoryManagement.Desktop.DAL;
using ProjectLex.InventoryManagement.Desktop.Stores;
using ProjectLex.InventoryManagement.Desktop.Utilities;
using System;
using System.Collections.Generic;
using System.ComponentModel.DataAnnotations;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Input;
using static ProjectLex.InventoryManagement.Desktop.Utilities.Constants;

namespace ProjectLex.InventoryManagement.Desktop.ViewModels
{
    public class EditSupplierViewModel : ViewModelBase
    {
        private bool _isDisposed = false;

        private Supplier _supplier;

        public string _supplierName;

        [Required(ErrorMessage = "Name is Required")]
        [MinLength(2, ErrorMessage = "Name should be longer than 2 characters")]
        [MaxLength(50, ErrorMessage = "Name longer than 50 characters is Not Allowed")]
        public string SupplierName
        {
            get => _supplierName;
            set
            {
                SetProperty(ref _supplierName, value);
            }

        }


        private string _supplierAddress;

        [Required(ErrorMessage = "Address is Required")]
        [MinLength(10, ErrorMessage = "Address should be longer than 2 characters")]
        public string SupplierAddress
        {
            get => _supplierAddress;
            set
            {
                SetProperty(ref _supplierAddress, value);
            }
        }


        private string _supplierPhone;

        [Required(ErrorMessage = "Phone number is Required")]
        public string SupplierPhone
        {
            get => _supplierPhone;
            set
            {
                SetProperty(ref _supplierPhone, value);
            }
        }


        private string _supplierEmail;

        [Required(ErrorMessage = "Email is Required")]
        [RegularExpression("^[\\w-\\.]+@([\\w-]+\\.)+[\\w-]{2,4}$", ErrorMessage = "Invalid Email Format")]
        public string SupplierEmail
        {
            get => _supplierEmail;
            set
            {
                SetProperty(ref _supplierEmail, value);
            }
        }


        private string _supplierStatus;

        [Required(ErrorMessage = "Status is Required")]
        public string SupplierStatus
        {
            get { return _supplierStatus; }
            set
            {
                SetProperty(ref _supplierStatus, value);
            }
        }


        private readonly UnitOfWork _unitOfWork;
        private readonly NavigationStore _navigationStore;
        private readonly Action _closeDialogCallback;

        public RelayCommand SubmitCommand { get; }
        public RelayCommand CancelCommand { get; }

        public EditSupplierViewModel(NavigationStore navigationStore, UnitOfWork unitOfWork, Supplier supplier, Action closeDialogCallback)
        {
            _navigationStore = navigationStore;
            _unitOfWork = unitOfWork;
            _closeDialogCallback = closeDialogCallback;

            _supplier = supplier;
            SetInitialValues(_supplier);



            SubmitCommand = new RelayCommand(Submit);
            CancelCommand = new RelayCommand(Cancel);
        }

        private void SetInitialValues(Supplier supplier)
        {
            SupplierName = supplier.SupplierName;
            SupplierAddress = supplier.SupplierAddress;
            SupplierEmail = supplier.SupplierEmail;
            SupplierPhone = supplier.SupplierPhone;
            SupplierStatus = supplier.SupplierStatus;
        }


        private void Submit()
        {
            ValidateAllProperties();

            if (HasErrors)
            {
                return;
            }

            _supplier.SupplierName = _supplierName;
            _supplier.SupplierAddress = _supplierAddress;
            _supplier.SupplierEmail = _supplierEmail;
            _supplier.SupplierPhone = _supplierPhone;
            _supplier.SupplierStatus = _supplierStatus;

            _unitOfWork.SupplierRepository.Update(_supplier);
            _unitOfWork.LogRepository.Insert(LogUtil.CreateLog(LogCategory.SUPPLIERS, ActionType.UPDATE, $"Supplier updated; SupplierID: {_supplier.SupplierID};"));
            _unitOfWork.Save();
            _closeDialogCallback();
        }


        private void Cancel()
        {
            _closeDialogCallback();
        }

        public static EditSupplierViewModel LoadViewModel(NavigationStore navigationStore, UnitOfWork unitOfWork, Supplier supplier, Action closeDialogCallback)
        {
            EditSupplierViewModel viewModel = new EditSupplierViewModel(navigationStore, unitOfWork, supplier, closeDialogCallback);
            return viewModel;
        }

        

        protected override void Dispose(bool disposing)
        {
            if(!this._isDisposed)
            {
                if(disposing)
                {
                    // dispose managed resources
                }
                // dispose unmanaged resources
            }
            this._isDisposed = true;

            base.Dispose(disposing);
        }


        public bool CanModifySupplier(object obj)
        {
            return true;
        }
    }
}
