# MODULE: Typerig / IO / DTL KRN Parser (Objects)
# -----------------------------------------------------------
# (C) Vassil Kateliev, 2020 		(http://www.kateliev.com)
# (C) Karandash Type Foundry 		(http://www.karandash.eu)
#------------------------------------------------------------
# www.typerig.com

# - Dependencies -------------------------
from __future__ import print_function
import os, warnings
from datetime import datetime

from typerig.core.base import message

# - Init -----------------------------
__version__ = '0.1.0'

# - Classes --------------------------
class KRNparser(object):
	def __init__(self, file_path, file_mode='r'):
		# - File
		self.__extension = '.krn'
		self.__path = file_path
		self.__mode = file_mode
		self.__file_object = None
		
		if not file_path.endswith(self.__extension):
			raise NameError('ERROR:\t{} extension missing in file name: {}'.format(self.__extension, file_path))
		
		# - Vocabulary
		self.__head_info = 		'Comment DTL Kern file (KRN)\nComment Generated by TypeRig (www.typerig.com)\n'
		self.__comment_line = 	'Comment'
		self.__kern_line = 		'KPX'
		self.__head_start = 	'StartFontMetrics\n'
		self.__data_start = 	'StartKernData\n'
		self.__data_end = 		'EndKernData\n'
		self.__block_start = 	'StartKernPairs {}\n\n'
		self.__block_end =  	'EndKernPairs\n'
		
	def __enter__(self):
		self.__file_object = open(self.__path, self.__mode)
		return self

	def __exit__(self, type, val, tb):
		self.__file_object.close()

	def __iter__(self):
		return self

	def __next__(self):
		new_line = self.__file_object.readline().strip()

		if self.__file_object is None or new_line == b'':
			raise StopIteration
		else:
			if self.__kern_line in new_line:
				return tuple(new_line.strip(self.__kern_line).strip().split())
			else:
				return self.__next__()

	def next(self):
		# Python 2 fixup
		return self.__next__()

	def dump(self, pair_list, user_info=''):
		if self.__mode in ('a', 'w'):
			# - Prepare
			user_header = '\n'.join([self.__comment_line + ' ' + info for info in user_info.split('\n')])
			creation_date = self.__comment_line + ' Created on: ' + datetime.now().strftime('%d/%m/%Y %H:%M:%S') +'\n'
			pairs_start = self.__block_start.format(len(pair_list))

			file_head = self.__head_info + 		\
						user_header + '\n' +	\
						creation_date + 		\
						'\n' + 					\
						self.__head_start + 	\
						self.__data_start +		\
						pairs_start 

			file_tail = '\n' + self.__block_end + self.__data_end

			# - Dump
			self.__file_object.writelines(file_head)

			for first, second in pair_list:
				self.__file_object.write('{} {} {}\n'.format(self.__kern_line, first, second))

			self.__file_object.writelines(file_tail)

		else:
			warnings.warn('File not in writable mode! Aborting!', message.FileSaveWarning)


# - Test -----------------------------
if __name__ == '__main__':
	root_dir = os.path.dirname(os.path.dirname(__file__))
	krn_filename = '1.txt.krn'
	krn_filename_out = 'test.krn'
	krn_file = os.path.join(root_dir, krn_filename)
	krn_file_out = os.path.join(root_dir, krn_filename_out)

	append_glyphs = []
	with KRNparser(krn_file) as reader:
		for a in reader:
			append_glyphs.append(a)

	with KRNparser(krn_file_out, 'w') as writer:
		writer.dump(append_glyphs)
			