<?php

declare(strict_types=1);

/**
 * Highlighter
 *
 * Copyright (C) 2016, Some right reserved.
 *
 * @author Kacper "Kadet" Donat <kacper@kadet.net>
 *
 * Contact with author:
 * Xmpp: me@kadet.net
 * E-mail: contact@kadet.net
 *
 * From Kadet with love.
 */

namespace Kadet\Highlighter\Language;

use Kadet\Highlighter\Matcher\CommentMatcher;
use Kadet\Highlighter\Matcher\RegexMatcher;
use Kadet\Highlighter\Matcher\WordMatcher;
use Kadet\Highlighter\Parser\Rule;
use Kadet\Highlighter\Parser\Token\Token;
use Kadet\Highlighter\Parser\Validator\Validator;

class Latex extends GreedyLanguage
{
    protected static $mathEnvironments = ['align', 'equation', 'math'];

    /**
     * Tokenization rules
     */
    public function setupRules()
    {
        $this->rules->addMany([
            'call.symbol' => new Rule(new RegexMatcher('/(\\\[a-z@]+)/si'), [
                'context' => ['!delimiter.environment'],
                'priority' => -1
            ]),

            'expression.math' => [
                new Rule(new RegexMatcher('/((\${1,2}).*?\2)/s')),
                new Rule(new RegexMatcher('/(\\\\\(.*?\\\\\))/s')),
                new Rule(new RegexMatcher('/(\\\\\[.*?\\\\\])/s')),
                new Rule(
                    new RegexMatcher(
                        '/\\\begin{((?:' . implode('|', self::$mathEnvironments) . ')\*?)}(.*?)\\\end{\1}/s',
                        [2 => Token::NAME]
                    )
                )
            ],

            'delimiter.math' => new Rule(new WordMatcher(['$', '\[', '\]', '\(', '\)'], ['separated' => false]), [
                'context' => ['expression.math']
            ]),

            'symbol.annotation'    => new Rule(new RegexMatcher('/\[(.*?)\]/')), // make argument parsing?
            'symbol.environment' => new Rule(new RegexMatcher('/(\\\(?:begin|end)){(.*?)}/', [
                1 => 'delimiter.environment',
                2 => Token::NAME
            ]), ['context' => Validator::everywhere()]),

            'symbol.label' => new Rule(new RegexMatcher('/\\\(?:label|ref){(.*?)}/')),

            'operator' => [
                new Rule(new WordMatcher(['*', '&', '@', '\\\\'], ['separated' => false]), ['context' => Validator::everywhere()]),
                new Rule(new WordMatcher(['=', '-', '+', '/', '^', '_'], ['separated' => false]), [
                    'context'  => ['expression.math'],
                    'priority' => -1
                ]),
                'punctuation.brackets' => new Rule(new WordMatcher(['{', '}', '[', ']', '(', ')'], ['separated' => false]), [
                    'context' => ['expression.math']
                ]),
                'escape' => new Rule(new RegexMatcher('/(\\\[%@&])/si'), ['context' => Validator::everywhere(), 'priority' => -1])

            ],
            'comment' => new Rule(new CommentMatcher(['%'], [])),

            'format.bold' => new Rule(new RegexMatcher('/\\\textbf({((?>[^{}]+|(?1))+)})/si', [
                2 => Token::NAME
            ])),
            'format.italics' => new Rule(new RegexMatcher('/\\\textit({((?>[^{}]+|(?1))+)})/si', [
                2 => Token::NAME
            ])),

            'variable' => new Rule(new RegexMatcher('/(#\d+)/')),

            # math mode
            'number' => new Rule(new RegexMatcher('/(-?(?:0[0-7]+|0[xX][0-9a-fA-F]+|0b[01]+|[\d,.]+))/'), [
                'context'  => ['expression.math'],
                'priority' => -2
            ]),
        ]);
    }

    /** {@inheritdoc} */
    public function getIdentifier()
    {
        return 'latex';
    }

    public static function getMetadata()
    {
        return [
            'name'      => ['tex', 'latex'],
            'mime'      => ['application/x-tex', 'application/x-latex'],
            'extension' => ['*.tex', '*.aux', '*.toc']
        ];
    }
}
