# Administración y aseguramiento de la información
- Tipos de copias de seguridad:
  - Locales y Remotas.
  - Herramientas clásicas: copias de seguridad completas, diferenciales e incrementales.
  - Herramientas de sincronización con control de versiones.
- Planes de copias de seguridad y restablecimiento.
- Consideraciones:
    - Dónde realizar las copias de seguridad (backup), locales y remotas.
    - Programación de copias de seguridad: para decidir la programación habrá que tener en cuenta el tiempo asumible de pérdida de datos, si es necesario detener servicios para realizar las copias de seguridad, etc.
    - Espacio necesario, históricos, rotaciones de los medios, etc.
- Responsabilidad de las copias de seguridad. Personas encargadas de revisar las copias de seguridad y restablecerlas si es necesario. Asignar los permisos necesarios a estas personas. 
- Elección del software para realizar los backups. Uso de herramientas estándar o específicas para los datos/servicios a asegurar.
- Probar la integridad de las copias de seguridad:
    - Crear un plan de pruebas. Ventajas del uso de máquinas virtuales para las pruebas de integridad de los backups.
    - Comprobar si los tiempos de restauración son asumibles.
- Crear un plan de restauración:
    - Dónde se restaurarán los datos en caso de fallo físico de discos, servidores, etc.
    - Quien está autorizado para realizar la restauración.
    - Detallar los pasos a realizar.
- Recuperación en caso de fallo del sistema:
  - Responsables de la recuperación, mecanismos para localizar a estas personas.
  - Hardware crítico para la restauración.
  - Medidas a adoptar para mitigar el impacto del fallo.
  - Discos de arranque. Discos de recuperación.
  - Copias de seguridad del sistema. Recuperación del sistema mediante consola. Puntos de recuperación.
- Creación y recuperación de imágenes de servidores.
  - Cuotas de disco. Niveles de cuota y niveles de advertencia.
  - Compresión de datos.

---------------

# Ficheros en Linux

## Introducción a buscar ficheros
* https://github.com/jesnino/Bash/blob/master/Ficheros/EjerciciosFicherosBuscar.sh

## Conocer la extensión de un fichero en Linux
```Bash
file fichero
```

## Ejercicio calcular hash en Linux
```Bash
proceso=`ps -C nano -o cmd --no-headers`
ruta=`whereis $proceso`
sha512sum $ruta
```

## Introducción a bucles en Linux
* https://www.jesusninoc.com/2015/07/13/bucles-2/
* https://www.jesusninoc.com/2016/01/02/for-loop-examples/

## Bucle for en Linux
```Bash
for i in $(pgrep application.x); do ps -mo pid,tid,fname,user,psr -p $i;done
```

## Ver funcionamiento de awk
```Bash
echo "hola amigo" | awk '{print $1}'
echo "hola amigo" | awk '{print $2}'
echo "touch amigo" | awk '{print $1}'; echo "touch amigo" | awk '{print $2}'
```

## Calcular el hash SHA512 de todos los procesos que se están ejecutando en Linux
```Bash
for procesos in $(ps -aux | awk '{print $11}');do ruta=`whereis $procesos`; sha512sum $ruta;done
```

-------------

# Ficheros en PowerShell

## Listar ficheros cuya fecha de última escritura de ficheros sea de mayor que menos un día
```PowerShell
Get-ChildItem | where LastWriteTime -gt (Get-Date).AddDays(-1)
```
## Listar ficheros cuyo tamaño sea mayor de 1MB
```PowerShell
Get-ChildItem -Recurse | where Length -Gt 1MB
```
## Listar la ruta de los programas que se están ejecutando
```PowerShell
foreach($procesos in (Get-Process | select path))
{
    Get-FileHash $procesos.path
}
```
## Listar la ruta del programa Chrome que se está ejecutando
```PowerShell
foreach($id in Get-Process -Name chrome)
{
    Get-FileHash $id.Path
}
```

---------------

# Linux

## Copias de seguridad
* http://www.developandsys.es/copias-seguridad-linux/

## Programar la copia de seguridad
* https://github.com/jesusninoc/Bash/blob/master/Examenes/Examen-2016-02-11-Solucion.sh

## Ejercicios sobre copias de seguridad
- Listar archivos por fecha
  ```Bash
  ls -lt
  ls -ltr
  ls -l --sort=size
  ```
- Copia de seguridad para cada día de la semana y ser capaz de recuperar
  * https://www.youtube.com/watch?v=aDCfL9Pk3TQ&feature=youtu.be
  * https://github.com/jesnino/Bash/blob/master/Ficheros/EjercicioCalendario.sh
- Copiar cada fichero listado de un directorio en un nuevo directorio
  * https://github.com/jesusninoc/ClasesISO/blob/master/2019-01-22.md#utilizar-xargs
- Comprimir las copias de seguridad que se han realizado
  * https://linuxzone.es/metodos-de-compresion-y-descompresion-en-linux/
- Enviar un mail cuando se haya realizado la copia de seguridad
  * https://www.interserver.net/tips/kb/linux-mail-command-usage-examples/

## Copias de seguridas de bases de datos

### MySQL dump ignorando tablas
* https://github.com/jesusninoc/Scripts/blob/master/MySQL%20dump%20ignorando%20tablas.sh

### MySQL dump completo
* https://github.com/jesusninoc/Scripts/blob/master/MySQL%20dump%20completo.sh

### Borrar copias de seguridad de bases de datos en MySQL.sh
* https://github.com/jesusninoc/Scripts/blob/master/Borrar%20copias%20de%20seguridad%20de%20bases%20de%20datos%20en%20MySQL.sh

### Database Backup Program (examples)
* https://www.jesusninoc.com/11/26/database-backup-program-scripts-examples/

-----------------

# PowerShell

## Ejercicio sobre copias de seguridad con integridad: comprobar que los ficheros que tengo en una carpeta son correctos en la copia de seguridad mediante la integridad

```PowerShell
# Ideas

# Probar la integridad de las copias de seguridad:
# - Crear un plan de pruebas. 
# - Ventajas del uso de máquinas virtuales para las pruebas de integridad de los backups.
# - Comprobar si los tiempos de restauración son asumibles.

# Ficheros que se van a copiar
# fichero1.txt
# fichero2.txt
# fichero3.txt
# fichero4.txt

# Ficheros que tienen las integridades de los ficheros 1,2,3,4
# fichero1integridad.txt
# fichero2integridad.txt
# fichero3integridad.txt
# fichero4integridad.txt

# crear el zip 1,2,3,4

# -------------------- RECUPERACIÓN

# descomprimir zip 1,2,3,4

# ver integridad fichero que tiene los hashes 1,2,3,4

# fichero1 si abro si la integridad es correcta

"hola" > fichero1.txt
(Get-FileHash .\fichero1.txt).hash
(Get-FileHash .\fichero1.txt).hash > fichero1integridad.txt

# zip meter fichero1.txt más integridades.txt (se puede separar en dos archivos)

# hay fallo y tengo que recuperar la copia

# descomprimir y ver si el fichero1.txt lo puedo ver

# porque la integridad de fichero1 es correcta

# h(f1) == f1*

if((gc .\fichero1integridad.txt) -eq (Get-FileHash .\fichero1.txt).hash)
{
    "El hash coincide"
}
else
{
    "El hash no coincide"
}

# Ver si el hash coincide para todos los ficheros

mkdir ficheros
cd .\ficheros
"hola" > fichero1.txt
(Get-FileHash .\fichero1.txt).hash
(Get-FileHash .\fichero1.txt).hash > fichero1integridad.txt
"holas" > fichero2.txt
(Get-FileHash .\fichero2.txt).hash
(Get-FileHash .\fichero2.txt).hash > fichero2integridad.txt

# Se supone que se ha comprimido

# Hay fallo
 
# Verificar que todos los hashes son correctos

ls fichero?.txt | %{
    $ficherointeg = $_.FullName
    $ficherointeg2 = $ficherointeg.replace(".txt","")
    $ficherofinal = ([String]$ficherointeg2+"integridad.txt")

    if(((Get-FileHash $_).hash) -eq (gc $ficherofinal))
    {
        "El hash coincide"
    }
    else
    {
        "El hash no coincide"
    }
}
```

-------------
-------------

# Repaso evaluación
* https://github.com/jesusninoc/ClasesISO/blob/master/2019-12-05.md#resolver-las-siguientes-situaciones

----------

# Administración y aseguramiento de la información
- Tipos de copias de seguridad:
  - Locales y Remotas.
  - Herramientas clásicas: copias de seguridad completas, diferenciales e incrementales.
  - Herramientas de sincronización con control de versiones.
- Planes de copias de seguridad y restablecimiento.
- Consideraciones:
    - Dónde realizar las copias de seguridad (backup), locales y remotas.
    - Programación de copias de seguridad: para decidir la programación habrá que tener en cuenta el tiempo asumible de pérdida de datos, si es necesario detener servicios para realizar las copias de seguridad, etc.
    - Espacio necesario, históricos, rotaciones de los medios, etc.
- Responsabilidad de las copias de seguridad. Personas encargadas de revisar las copias de seguridad y restablecerlas si es necesario. Asignar los permisos necesarios a estas personas. 
- Elección del software para realizar los backups. Uso de herramientas estándar o específicas para los datos/servicios a asegurar.
- Probar la integridad de las copias de seguridad:
    - Crear un plan de pruebas. Ventajas del uso de máquinas virtuales para las pruebas de integridad de los backups.
    - Comprobar si los tiempos de restauración son asumibles.
- Crear un plan de restauración:
    - Dónde se restaurarán los datos en caso de fallo físico de discos, servidores, etc.
    - Quien está autorizado para realizar la restauración.
    - Detallar los pasos a realizar.
- Recuperación en caso de fallo del sistema:
  - Responsables de la recuperación, mecanismos para localizar a estas personas.
  - Hardware crítico para la restauración.
  - Medidas a adoptar para mitigar el impacto del fallo.
  - Discos de arranque. Discos de recuperación.
  - Copias de seguridad del sistema. Recuperación del sistema mediante consola. Puntos de recuperación.
- Creación y recuperación de imágenes de servidores.
  - Cuotas de disco. Niveles de cuota y niveles de advertencia.
  - Compresión de datos.

---------------

# PowerShell

## Ejercicio sobre copias de seguridad con integridad: comprobar que los ficheros que tengo en una carpeta son correctos en la copia de seguridad mediante la integridad

```PowerShell
# Ideas

# Probar la integridad de las copias de seguridad:
# - Crear un plan de pruebas. 
# - Ventajas del uso de máquinas virtuales para las pruebas de integridad de los backups.
# - Comprobar si los tiempos de restauración son asumibles.

# Ficheros que se van a copiar
# fichero1.txt
# fichero2.txt
# fichero3.txt
# fichero4.txt

# Ficheros que tienen las integridades de los ficheros 1,2,3,4
# fichero1integridad.txt
# fichero2integridad.txt
# fichero3integridad.txt
# fichero4integridad.txt

# crear el zip 1,2,3,4

# -------------------- RECUPERACIÓN

# descomprimir zip 1,2,3,4

# ver integridad fichero que tiene los hashes 1,2,3,4

# fichero1 si abro si la integridad es correcta

"hola" > fichero1.txt
(Get-FileHash .\fichero1.txt).hash
(Get-FileHash .\fichero1.txt).hash > fichero1integridad.txt

# zip meter fichero1.txt más integridades.txt (se puede separar en dos archivos)

# hay fallo y tengo que recuperar la copia

# descomprimir y ver si el fichero1.txt lo puedo ver

# porque la integridad de fichero1 es correcta

# h(f1) == f1*

if((gc .\fichero1integridad.txt) -eq (Get-FileHash .\fichero1.txt).hash)
{
    "El hash coincide"
}
else
{
    "El hash no coincide"
}

# Ver si el hash coincide para todos los ficheros

mkdir ficheros
cd .\ficheros
"hola" > fichero1.txt
(Get-FileHash .\fichero1.txt).hash
(Get-FileHash .\fichero1.txt).hash > fichero1integridad.txt
"holas" > fichero2.txt
(Get-FileHash .\fichero2.txt).hash
(Get-FileHash .\fichero2.txt).hash > fichero2integridad.txt

# Se supone que se ha comprimido

# Hay fallo
 
# Verificar que todos los hashes son correctos

ls fichero?.txt | %{
    $ficherointeg = $_.FullName
    $ficherointeg2 = $ficherointeg.replace(".txt","")
    $ficherofinal = ([String]$ficherointeg2+"integridad.txt")

    if(((Get-FileHash $_).hash) -eq (gc $ficherofinal))
    {
        "El hash coincide"
    }
    else
    {
        "El hash no coincide"
    }
}
```

## Programar la copia de seguridad
* https://www.jesusninoc.com/2017/03/01/tareas-programadas-en-powershell/
* https://github.com/jesusninoc/PowerShell/blob/master/TareasProgramadas/EjemploTareaProgramada.ps1

## Ejercicio: crear una tarea programada que ejecuta un script en PowerShell que realiza una copia de seguridad
### Crear una copia de seguridad
* https://github.com/jesusninoc/ClasesISO/blob/master/2019-11-26.md#ejercicio-sobre-copias-de-seguridad-con-integridad-comprobar-que-los-ficheros-que-tengo-en-una-carpeta-son-correctos-en-la-copia-de-seguridad-mediante-la-integridad
* https://www.jesusninoc.com/12/08/crear-una-tarea-programada-en-windows-que-ejecute-un-script-de-powershell-cada-10-minutos-de-forma-indefinida-aunque-el-porcentaje-de-bateria-sea-bajo/
### Tener en cuenta a la hora de programar la tarea que ejecuta el script
```PowerShell
# Cambiar la política de ejecución de scripts para poder ejecutar
Set-ExecutionPolicy -ExecutionPolicy Unrestricted -Scope CurrentUser

# Ecribir un mensaje por pantalla
echo "hola"

# Mostrar la ruta
pwd

# Parar el script
pause

# Almacenar en un fichero la hora a la que se ejecuta el script junto con el porcentaje de batería
echo (Get-Date).Hour >> C:\xampp\mysql\bin\fichero.txt
```
### Repetir cada 1 hora indefinidamente la ejecución de un script en PowerShell
```PowerShell
$Sta = New-ScheduledTaskAction -Execute "powershell"
$Stt = New-ScheduledTaskTrigger -Once -At 19pm -RepetitionInterval (New-TimeSpan -hour 1)
Register-ScheduledTask Task01 -Action $Sta -Trigger $Stt
```

--------------------
--------------------

# Resumen de todo lo que hemos visto

## Instalación de software libre y propietario

- Estructura de un sistema informático. Monolítica. Jerárquica. Capas o anillos (ring). Máquinas virtuales. Cliente-servidor.
  * https://github.com/jesusninoc/ClasesISO/blob/master/2019-09-17.md#n%C3%BAcleo
  ```PowerShell
  # Mostrar información sobre el núcleo
  [System.Environment]::OSVersion.Version
  wsl uname -r
  ```
  ```PowerShell
  # Mostrar información sobre el núcleo mediante una función
  function mostrarinformacion()
  {
      # Mostrar información sobre el núcleo
      [System.Environment]::OSVersion.Version
      wsl uname -r
  }
  mostrarinformacion
  ```
- Arquitectura de un sistema operativo. Sistemas por lotes (batch). Sistemas por lotes con multiprogramación. Sistemas de tiempo compartido. Sistemas distribuidos.
  * https://www.jesusninoc.com/07/03/3-gestion-del-hardware-en-powershell/#Procesador
- Funciones de un sistema operativo.
  - Controlar y gestionar el uso del hardware del ordenador: CPU, dispositivos de E/S, Memoria principal, tarjetas gráficas y el resto de periféricos.
    * https://www.jesusninoc.com/07/03/3-gestion-del-hardware-en-powershell
    ```PowerShell
    # Mostrar la carga del procesador
     Get-WmiObject win32_processor | Select-Object LoadPercentage
    ```
    ```PowerShell
    function mostrarcargaprocesador()
    {
        # Mostrar la carga del procesador
        Get-WmiObject win32_processor | Select-Object LoadPercentage
    }
    mostrarcargaprocesador
    ```
  - Administrar la ejecución de los procesos. Planificación.
    * https://www.jesusninoc.com/07/07/7-gestion-de-procesos-en-powershell/
    * https://github.com/jesusninoc/Scripts/blob/master/Ver%20si%20Apache%20est%C3%A1%20encendido.sh
    * https://github.com/jesusninoc/Scripts/blob/master/Ver%20si%20MySQL%20est%C3%A1%20encendido.sh
    * https://github.com/jesusninoc/Scripts/blob/master/Ver%20si%20Apache%20est%C3%A1%20encendido%20y%20enviar%20un%20mail.sh
    * https://github.com/jesnino/Bash/blob/master/Procesos/EjerciciosProcesos.sh
    
    ```PowerShell
    # Crear una carpeta con la fecha de hoy que contenga los procesos que se están ejecutando junto con información sobre los hilos (solución 1)

    $fecha = (Get-Date).ToString("yyyyMMdd")
    mkdir $fecha
    cd $fecha

    mkdir (Get-Process | select Name -Unique).name

    foreach($carpeta in (Get-ChildItem).Name)
    {
        $carpeta
        (Get-Process -Name $carpeta).Threads >> ($carpeta+"\informacion.txt")
    }
    ```
    ```PowerShell
    # Crear una carpeta con la fecha de hoy que contenga los procesos que se están ejecutando junto con información sobre los hilos (solución 2)

    $fecha = (Get-Date).ToString("yyyyMMdd")
    mkdir $fecha
    cd $fecha

    foreach($carpeta in (Get-Process | select Name -Unique).name)
    {
        $carpeta
        mkdir $carpeta
        (Get-Process -Name $carpeta).Threads >> ($carpeta+"\informacion.txt")
    }
    ```
    ```PowerShell
    # Crear una carpeta con la fecha de hoy que contenga los procesos que se están ejecutando junto con información sobre los hilos (solución mediante funciones)

    function infohilos ($proceso)
    {
        # Valores que necesita: proceso
        # Valores que devuelve: hilos
        (Get-Process -Name $proceso).Threads >> ($proceso+"\informacion.txt")
    }

    $fecha = (Get-Date).ToString("yyyyMMdd")
    mkdir $fecha
    cd $fecha

    foreach($proceso in (Get-Process | select Name -Unique).name)
    {
        $proceso
        mkdir $proceso -force
        infohilos $proceso
    }
    ```
    ```PowerShell
    # Crear una carpeta con la fecha de hoy que contenga los procesos que se están ejecutando junto el número de hilos (solución mediante funciones)

    function contarhilos ($proceso)
    {
        # Valores que necesita: proceso
        # Valores que devuelve: hilos
        (Get-Process -Name $proceso).Threads.Count
    }

    $fecha = (Get-Date).ToString("yyyyMMdd")
    mkdir $fecha
    cd $fecha

    foreach($proceso in (Get-Process | select Name -Unique).name)
    {
        $proceso
        mkdir $proceso -force
        contarhilos $proceso
    }
    ```
    ```PowerShell
    # Arrancar un servicio
    New-Service -Name "Tes2" -BinaryPathName '"C:\Program Files\MySQL\MySQL Server 5.1\bin\mysqld" --defaults-file="C:\Program Files\MySQL\MySQL Server 5.1\my.ini" MySQL'

    Start-Service tes2
    Stop-Service tes2
    ```
    ```Bash
    # Eliminar un proceso o un servicio
    kill -9 `pidof apache2`
    ```
      
  - Controlar el proceso de organización de la información. Creación, acceso (ubicación física) y borrado de archivos.
    * https://www.jesusninoc.com/07/04/4-gestion-del-sistema-de-archivos-en-powershell/#Archivos
  - Controlar el acceso de los programas o los usuarios a los recursos del sistema.
  ```Bash
  ps -U root -u root u
  ps -eo user,comm
  ```
  - Proporcionar interfaces de usuario: en modo texto y gráficos.
  - Servicios soporte: actualizaciones de software, controladores para nuevos periféricos, etc.
    * https://www.jesusninoc.com/07/05/5-gestion-del-software-en-powershell/#Actualizaciones
    * https://www.jesusninoc.com/07/07/7-gestion-de-procesos-en-powershell/
    * https://github.com/jesusninoc/ClasesISO/blob/master/2019-10-15.md#ver-actualizaciones
    * https://github.com/jesusninoc/ClasesISO/blob/master/2019-10-15.md#obtener-informaci%C3%B3n-sobre-las-actualizaciones
    ```PowerShell
    # Crear carpetas para cada tipo de ProviderName y dentro de cada carpeta meter los paquetes de cada ProviderName
    
    foreach($programa in Get-Package | select ProviderName, name)
    {
        mkdir $programa.ProviderName -Force
        $programa.Name >> ($programa.ProviderName+"\informacion.txt")
    }
    ```
    ```PowerShell
    # Crear carpetas para cada tipo de ProviderName y dentro de cada carpeta meter los paquetes de cada ProviderName (mediante funciones)

    function clasificarporProviderName($programa)
    {
        $programa.Name >> ($programa.ProviderName+"\informacion.txt")
    }

    foreach($programa in Get-Package | select ProviderName, name)
    {
        mkdir $programa.ProviderName -Force
        clasificarporProviderName $programa
    }
    ```
- Tipos de sistemas operativos.
  - Monousuario o multiusuario
  - Centralizado o distribuido
  - Monotarea o multitarea
  - Uniprocesador o multiprocesador
  - Instalables y/o autoarrancables.
- Sistemas operativos libres.
- Sistemas operativos propietarios.
- Tipos de aplicaciones. Software de sistema. Software de programación. Software de aplicación.
- Licencias y tipos de licencias. Según los derechos que cada autor se reserva sobre su obra. Según su destinatario.
- Máquinas virtuales (M.V.)
  - Concepto de virtualización del hardware y características de los principales productos software libre y propietario, para el uso de máquinas virtuales.
    * https://www.jesusninoc.com/07/06/6-virtualizacion-en-powershell/ 
  - Creación y personalización de M.V.
    * https://www.jesusninoc.com/07/06/6-virtualizacion-en-powershell/#Crear_host_virtual
  - Ventajas e inconvenientes de la virtualización.
- Consideraciones previas a la instalación de sistemas operativos libres y propietarios.
  - Particionado del disco duro.
    * https://github.com/jesusninoc/ClasesISO/blob/master/2018-04-04.md
    * https://github.com/jesusninoc/ClasesISO/blob/master/2019-10-17.md#comandos-para-trabajar-con-discos-en-linux
  - En sistemas Windows determinar la partición donde instalaremos el S.O.
  - En sistemas Linux determinar las particiones para los distintos puntos de montaje.
  - Controladores (drivers) de almacenamiento necesarios.
    * https://github.com/jesusninoc/PowerShell/blob/master/Procesos/EjerciciosProcesosAvanzado.ps1
    * https://github.com/jesusninoc/ClasesISO/blob/master/2019-10-15.md#ver-controladores 
- Instalación de sistemas operativos.
  - Requisitos hardware, versiones y licencias.
    * https://www.jesusninoc.com/07/03/3-gestion-del-hardware-en-powershell/
    * https://github.com/jesusninoc/ClasesISO/blob/master/2017-10-12.md
    * https://github.com/jesusninoc/PowerShell/blob/master/Hardware/EjemplosHardware.ps1
    * https://github.com/jesusninoc/PowerShell/blob/master/Hardware/EjemplosHardware2.ps1
    * https://github.com/jesusninoc/ClasesISO/blob/master/2019-11-20.md#1-realizar-un-inventario-de-tu-equipo-a-nivel-hardware-y-software-ten-en-cuenta-c%C3%B3mo-clasificar-la-informaci%C3%B3n-y-no-olvides-temas-importantes-como-por-ejemplo-controladores
  - Soporte utilizado para la instalación: CD/DVD, Pendrive, LAN.
  - Datos necesarios para la instalación: usuarios, contraseñas, nombre del equipo, direcciones IP, número de licencia, etc.
- Gestión de varios sistemas operativos en un ordenador.
  - Requisitos previos. Administración del espacio del disco. Particionado y redimensionado.
  - Problemas con el registro maestro de arranque (MBR). Elegir un gestor de arranque compatible con todos los sistemas operativos a instalar.
  - Preparar las particiones de los S.O. para permitir su arranque.
  - Analizar el orden en la instalación de los sistemas operativos.
- Gestores de arranque.
  - Código de arranque maestro (Master Boot Code).
    * https://github.com/jesusninoc/ClasesISO/blob/master/2018-10-08.md#arranque 
  - Formatos tabla de particiones. Master Boot Record (MBR) y Guid Partition Table (GPT).
  - Configuración de los gestores de arranque de los sistemas operativos libres y propietarios.
  - Reparación del gestor de arranque.
  - Sustitución del gestor de arranque estándar por otro más completo.
- Instalación/desinstalación de aplicaciones. Requisitos hardware y software, versiones y licencias.
  * https://github.com/jesusninoc/ClasesISO/blob/master/2019-10-15.md#instalar-un-paquete-msi-junto-con-informaci%C3%B3n-sobre-la-instalaci%C3%B3n
- Actualización de sistemas operativos y aplicaciones.
  - Actualizar a una versión superior (update).
    * https://www.jesusninoc.com/07/05/5-gestion-del-software-en-powershell/#Actualizaciones
  - Cambiar a una versión inferior (downgrade).
  - Instalación de parches: de seguridad, funcionales, opcionales, etc.
  - Automatizar las actualizaciones. Configurar la fuente de las actualizaciones.
- Ficheros necesarios para el arranque de los principales sistemas operativos.
- Registros (logs) del sistema.
  - Formato de los registros: fuente/origen, prioridades (informativos, advertencias, errores, etc.)
  - Herramientas para su monitorización en sistemas libres y propietarios.
    * https://www.jesusninoc.com/07/10/10-gestion-del-rendimiento-en-powershell/#Registros_del_sistema
    ```PowerShell
    # Almacenar el número de procesos que se están ejecutando y el número de hilos de cada proceso

    $procesos = gps
    $procesos.count | Out-File log.log -Append

    foreach($proceso in $procesos)
    {
        $proceso.Name | Out-File log.log -Append
        $proceso.threads.Count | Out-File log.log -Append
    }
    ```
    ```PowerShell
    # Almacenar el número de procesos que se están ejecutando y el número de hilos de cada proceso (mediante funciones, cutre)

    function guardarnombre()
    {
        $proceso.Name | Out-File log.log -Append
    }
    function guardarhilos()
    {
        $proceso.threads.Count | Out-File log.log -Append
    }

    $procesos = gps
    $procesos.count | Out-File log.log -Append

    foreach($proceso in $procesos)
    {
        guardarnombre
        guardarhilos
    }
    ```
    ```PowerShell
    # Almacenar el número de procesos que se están ejecutando y el número de hilos de cada proceso (mediante funciones)

    function guardarnombre($proceso)
    {
        $proceso.Name | Out-File log.log -Append
    }
    function guardarhilos($proceso)
    {
        $proceso.threads.Count | Out-File log.log -Append
    }

    $procesos = gps
    $procesos.count | Out-File log.log -Append

    foreach($proceso in $procesos)
    {
        guardarnombre $proceso
        guardarhilos $proceso
    }
    ```
  - Gestión: Aplicar filtros, asociar tareas en respuesta a ciertos eventos, etc.
- Actualización y mantenimiento de controladores de dispositivos.
  - Automatizar la actualización de controladores.
  - Volver a una versión anterior de un controlador.
  - Actualización manual de los controladores.

## Administración y aseguramiento de la información
- Sistemas de archivos:
  - Propietarios y libres.
  - Rutas y nombres de archivos. Estructura jerárquica.
    * https://www.jesusninoc.com/07/04/4-gestion-del-sistema-de-archivos-en-powershell/#La_ruta
- Gestión de sistemas de archivos mediante comandos y entornos gráficos.
- Gestión de enlaces.
- Estructura de directorios de sistemas operativos libres y propietarios.
- Búsqueda de información del sistema mediante comandos y herramientas gráficas.
  * https://github.com/jesusninoc/Bash/blob/master/Ficheros/EjerciciosFicherosBuscar.sh
- Identificación del software instalado mediante comandos y herramientas gráficas.
  * https://www.jesusninoc.com/07/05/5-gestion-del-software-en-powershell/
- Gestión de la información del sistema. Rendimiento. Estadísticas.
  * https://www.jesusninoc.com/07/10/10-gestion-del-rendimiento-en-powershell/#Rendimiento
- Montaje y desmontaje de dispositivos en sistemas operativos. Automatización.
  * https://www.jesusninoc.com/07/04/4-gestion-del-sistema-de-archivos-en-powershell/#Discos
- En sistemas Windows montar un volumen en una o más carpetas.
  * https://www.jesusninoc.com/07/04/4-gestion-del-sistema-de-archivos-en-powershell/#Montar_un_disco_virtual
- Herramientas de administración de discos. Particiones y volúmenes. Desfragmentación y chequeo.
  * https://www.jesusninoc.com/07/04/4-gestion-del-sistema-de-archivos-en-powershell/#Discos
- Permisos locales de acceso a ficheros y directorios.
  * https://github.com/jesusninoc/ClasesISO/blob/master/2019-11-05.md#permisos-en-linux
  * https://github.com/jesusninoc/ClasesISO/blob/master/2019-11-07.md#permisos-en-linux
  * https://www.jesusninoc.com/07/04/4-gestion-del-sistema-de-archivos-en-powershell/#Permisos-2
  * https://github.com/jesusninoc/ClasesSOM/blob/master/2018-11-26.md
  * https://github.com/jesusninoc/ClasesSOM/blob/master/2018-11-27.md
  * https://www.jesusninoc.com/03/30/eliminar-permisos-explicitos/

---------
---------

# Administración y aseguramiento de la información:
- Gestión de la información del sistema. Rendimiento. Estadísticas.
- Montaje y desmontaje de dispositivos en sistemas operativos. Automatización.
- En sistemas Windows montar un volumen en una o más carpetas.
- Herramientas de administración de discos. Particiones y volúmenes. Desfragmentación y chequeo.
- Permisos locales de acceso a ficheros y directorios.

---------------

# Soluciones examen simulacro (y ver funciones)
* https://github.com/jesusninoc/ClasesISO/blob/master/2020-11-19.md#simulacro-examen-liberatorio

# Soluciones con funciones

## Información sobre servicios (pero no se obtiene información sobre procesos)
```PowerShell
(Get-Service -Name Audiosrv,CryptSvc) | select *
```

## Información sobre servicios (buscar dos nombres de servicios, atentos al .name)
```PowerShell
foreach ($servicio in Get-WmiObject -Class win32_service)
{
    if($servicio.Name -eq "Audiosrv")
    {
        $servicio
    }
    if($servicio.Name -eq "CryptSvc")
    {
        $servicio
    }
}
```

---------
---------

# Administración y aseguramiento de la información:
- Gestión de la información del sistema. Rendimiento. Estadísticas.
- Montaje y desmontaje de dispositivos en sistemas operativos. Automatización.
- En sistemas Windows montar un volumen en una o más carpetas.
- Herramientas de administración de discos. Particiones y volúmenes. Desfragmentación y chequeo.
- Permisos locales de acceso a ficheros y directorios.

---------------

# Soluciones examen simulacro (y ver funciones)
* https://github.com/jesusninoc/ClasesISO/blob/master/2020-11-19.md#simulacro-examen-liberatorio

# Soluciones con funciones

## Información sobre servicios (pero no se obtiene información sobre procesos)
```PowerShell
(Get-Service -Name Audiosrv,CryptSvc) | select *
```

## Información sobre servicios (buscar dos nombres de servicios, atentos al .name)
```PowerShell
foreach ($servicio in Get-WmiObject -Class win32_service)
{
    if($servicio.Name -eq "Audiosrv")
    {
        $servicio
    }
    if($servicio.Name -eq "CryptSvc")
    {
        $servicio
    }
}
```

## Información sobre los procesos de los servicios
```PowerShell
foreach ($servicio in Get-WmiObject -Class win32_service)
{
    if($servicio.Name -eq "Audiosrv")
    {
        Get-Process -id $servicio.ProcessId
    }
    if($servicio.Name -eq "CryptSvc")
    {
        Get-Process -id $servicio.ProcessId
    }
}
```

## Información sobre los hilos de los procesos de los servicios
```PowerShell
foreach ($servicio in Get-WmiObject -Class win32_service)
{
    if($servicio.Name -eq "Audiosrv")
    {
        (Get-Process -id $servicio.ProcessId).Threads.id
        "----------------------------------"
    }
    if($servicio.Name -eq "CryptSvc")
    {
        (Get-Process -id $servicio.ProcessId).Threads.id
    }
}
```

## Juntar los dos if para obtener información sobre los servicios
```PowerShell
foreach ($servicio in Get-WmiObject -Class win32_service)
{
    if($servicio.Name -eq "Audiosrv" -or $servicio.Name -eq "CryptSvc")
    {
        $servicio
    }
}
```
 
## Preguntar al usuario mediante una función por la operación que quiera realizar:
- 1. Información sobre servicio
- 2. Información sobre procesos relacionados con los servicios
- 3. Información sobre hilos relacionados con los procesos de los servicios
```PowerShell
function comprobar()
{
    $opcion = Read-Host "¿Qué información quiere obtener sobre Audiosrv y CryptSvc?"
    switch($opcion)
    {
        "1"{
            foreach ($servicio in Get-WmiObject -Class win32_service)
            {
                if($servicio.Name -eq "Audiosrv" -or $servicio.Name -eq "CryptSvc")
                {
                    $servicio
                }
            }
        }
        "2"{
            foreach ($servicio in Get-WmiObject -Class win32_service)
            {
                if($servicio.Name -eq "Audiosrv" -or $servicio.Name -eq "CryptSvc")
                {
                    Get-Process -id $servicio.ProcessId
                }
            }
        }
        "3"{
            foreach ($servicio in Get-WmiObject -Class win32_service)
            {
                if($servicio.Name -eq "Audiosrv" -or $servicio.Name -eq "CryptSvc")
                {
                    (Get-Process -id $servicio.ProcessId).Threads.id
                }
            }
        }
    }
}

for(1)
{
    comprobar
}
```

## Función que se dedica a comprobar operaciones
```PowerShell
function comprobar($numero)
{
    foreach ($servicio in Get-WmiObject -Class win32_service)
    {
        if($servicio.Name -eq "Audiosrv" -or $servicio.Name -eq "CryptSvc")
        {
            switch($numero)
            {
                "1"{$servicio}
                "2"{Get-Process -id $servicio.ProcessId}
                "3"{(Get-Process -id $servicio.ProcessId).Threads.id}
            }
        }
    }
}
```

## Probar aleatoriamente operaciones
```PowerShell
for(1)
{
    comprobar (Get-Random(1..3))
    Start-Sleep -Seconds 5
}
```

## Probar operaciones leyendo de un fichero
```PowerShell
foreach ($operacionautomatica in gc .\operaciones.txt)
{
     comprobar $operacionautomatica
}
```

## Crear una función en la que pasas una fecha, te crea un directorio (año/mes/día) y también le pasas la palabra procesos, hilos o servicios y te crea ese contenido dentro de la carpeta que acabas de crear
```PowerShell
function pasarfecha($fecha,$palabra)
{
    mkdir $fecha.ToString("yyyyMMdd")
    cd $fecha.ToString("yyyyMMdd")
    switch($palabra)
    {
        "procesos"{Get-Process | Out-File procesos.txt}
        "servicios"{Get-Service | Out-File servicios.txt}
        "hilo"{(Get-Process).Threads.id | Out-File servicios.txt}
    }
    cd ..
}

pasarfecha (Get-Date) procesos
```

-------------

# Funciones en PowerShell
* https://www.jesusninoc.com/07/02/2-programacion-en-powershell/#Funciones
* https://www.jesusninoc.com/10/14/funciones-con-parametros-y-sin-parametros/

## Ejemplos sobre funciones

### Mediante una función mostrar un mensaje para cada usuario y pedirle un valor
```PowerShell
function pedirymostrar($i)
{
    $valor = Read-Host ("Introduzca valor" + "usuario" + $i)
    Write-Host ("El valor es " + $valor)
}

foreach($i in 1..10)
{
    pedirymostrar $i
}
```

### Mediante una función crear una carpeta para cada proceso que se está ejecutando
```PowerShell
# mkdir (ps).Name -WhatIf

function crearcarpeta($procesos)
{
    mkdir $procesos -WhatIf
}

crearcarpeta (ps).Name
```

### Crear una función que sume dos número que pedimos al usuario
```PowerShell
function suma([Int]$valor1,[Int]$valor2)
{
    ($valor1)+($valor2)
}

$valor1 = Read-Host "Introduzca número 1"
$valor2 = Read-Host "Introduzca número 2"
suma $valor1 $valor2
```

## Ejercicios sobre funciones: mediante la utilización de funciones en PowerShell resolver las siguientes situaciones:
- Mostrar todos los procesos.
- Mostrar los X primeros procesos.
- Mostrar los X últimos procesos.
- Ordenar todos los procesos por CPU de forma descendente.
- Mostrar los X procesos cuyo consumo (el parámetro que quieras utilizar) sea mayor que el valor que quieras monitorizar.
- Agrupar los procesos y muestra los que sean XXXXX.
- Abrir una página web las veces que quiera el usuario.
- Realizar un login.

### Función para mostrar todos los procesos
```PowerShell
function Procesos
{
Get-Process
}
Procesos
```

### Función para mostrar los X primeros procesos
```PowerShell
function ProcesosPrimeros($numero)
{
Get-Process | Select-Object -First $numero
}
ProcesosPrimeros 5
```

### Función para mostrar los X primeros procesos utilizando Mandatory
```PowerShell
function ProcesosUltimosM
{
param
(
[Parameter(Mandatory=$true)]
$numero
)
Get-Process | Select-Object -Last $numero
}
ProcesosUltimosM 5
```

### Función para mostrar los X últimos procesos
```PowerShell
function ProcesosUltimos($numero)
{
Get-Process | Select-Object -Last $numero
}
ProcesosUltimos 5
```

### Función para ordenar todos los procesos por CPU de forma descendente
```PowerShell
function ProcesosCPU
{
Get-Process | Sort-Object CPU -Descending
}
ProcesosCPU
```

### Función para mostrar los X procesos cuyo consumo (el parámetro que quieras utilizar) sea mayor que el valor que quieras monitorizar
```PowerShell
function ProcesosConsumo([String]$parametro,[Int]$valor,[Int]$numerovalores)
{
$parametro
$valor
$numerovalores
Get-Process | Where-Object $parametro -GT $valor | Select-Object -First $numerovalores
}
ProcesosConsumo CPU 30 5
```

### Función para agrupa los procesos y muestra los que sean XXXXX
```PowerShell
function ProcesoConcreto($nombreproceso)
{
Get-Process -Name $nombreproceso | Group-Object
}
ProcesoConcreto svchost
```

### Abrir una página web las veces que quiera el usuario
```PowerShell
function abrirweb ($numero,$url)
{
    1..$numero | %{
        start $url
        [System.Console]::Beep(261,500)
    }
}

abrirweb 3 http://www.jesusninoc.com
```

### Realizar un login mediante una función
```PowerShell
#Login de user y pass
#El user y pass correcto se escriben en el bloque BEGIN
#Ejecutar la función: login0
login0 -User jaime -Pass hola

function login0
{
  param([Parameter(Mandatory=$true, ValueFromPipeline=$true)][String[]][AllowEmptyString()]$User,$Pass)

  begin
  {
    Write-Host "Principio de login"
    $usercorrecto="user"
    $passcorrecto="pass"
  }

  process
  {
   if($User -eq $usercorrecto -and $Pass -eq $passcorrecto)
    {
        $ok=1
    }
    else
    {
        $ok=0
    }
  }
  end
  {
   if($ok)
    {
        Write-Host "User correcto"
    }
    else
    {
        Write-Host "User no correcto"
        $User,$Pass | Out-File log.txt -Append
    }
    Write-Host "Fin de login"
  }
}
```

---------------

# Ejercicio PowerShell

## Ejercicio con if UN NÚMERO ES PAR O IMPAR POWERSHELL

### Un número es par?, si el resto es 0. La operación del resto es %
```PowerShell
$numero=5
$numero%2
```

### Comprobar si un número es par
```PowerShell
$numero=5
if($numero%2 -eq 0)
{
"Número par"
}
else
{
"Número impar"
}
```

### Mover a una carpeta los ficheros pares y borrar los ficheros impares
```PowwerShell
-2..-8 | % {ni ((Get-Date).AddDays($_).ToString("yyyyMMdd")+".txt")}
```

## Eliminar el contenido de las carpetas impares de un servidor web
```PowerShell
# Crear un fichero dentro de todos los directorios
-2..-8 | % {mkdir (Get-Date).AddDays($_).ToString("yyyyMM-dd")}
ni -Path (ls).name -Name fichero -ItemType File

(ls | select name).name | %{
    $resultado = $_.split("-")[1] % 2
    if ($resultado -eq 1)
    {
        cd $_
        rm * -WhatIf
    }
    else
    {

    }
}
```

# Ejercicio Bash

## Eliminar el contenido de las carpetas impares de un servidor web
```Bash
echo "hola" > 201811-7
echo "hola" > 201811-8
echo "hola" > 201811-9

for i in $( ls *-? ); do
    echo "---------"
    resultado=`echo $i | cut -d "-" -f2`
    let result=resultado%2
    echo $result
    if [ $result -eq 1 ]; then
        echo "mover"
    fi
done
```

---------
---------

# Administración y aseguramiento de la información:
- Gestión de la información del sistema. Rendimiento. Estadísticas.
- Montaje y desmontaje de dispositivos en sistemas operativos. Automatización.
- En sistemas Windows montar un volumen en una o más carpetas.
- Herramientas de administración de discos. Particiones y volúmenes. Desfragmentación y chequeo.
- Permisos locales de acceso a ficheros y directorios.

---------------

# Linux

## Compressing files
* https://www.cyberciti.biz/howto/question/general/compress-file-unix-linux-cheat-sheet.php

## Sumar en Linux
* https://www.jesusninoc.com/11/28/como-realizar-operaciones-aritmeticas-en-linux/

## Crear calendario en Linux
* https://www.youtube.com/watch?v=aDCfL9Pk3TQ&feature=youtu.be
* https://github.com/jesnino/Bash/blob/master/Ficheros/EjercicioCalendario.sh

## Ejemplos en Bash (con repaso sobre comandos)

### Crea  directorios con los años desde 2020 hasta 2030
```Bash
mkdir {2020..2030}
```

### En cada uno de los directorios, crea un subdirectorio con los meses del año
```Bash
mkdir -p {2020..2030}/{Enero,Febrero,Marzo,Abril,Mayo,Junio,Julio,Agosto,Septiembre,Octubre,Noviembre,Diciembre}
```

### En cada mes, crear un directorio con cada día de la semana
```Bash
mkdir -p {2020..2030}/{Enero,Febrero,Marzo,Abril,Mayo,Junio,Julio,Agosto,Septiembre,Octubre,Noviembre,Diciembre}/{Lunes, Martes,Miercoles,Jueves,Viernes,Sabado,Domingo}
```

### Borrar todos los directorios Julio y Agosto de todos los años
```Bash
rm -r {2020..2030}/{Julio,Agosto}
```

### Crear un alias del comando "rm" que se llame borrar y que permita borrar el directorio Ejercicio1 (con su contenido)
```Bash
alias borrar='rm -r -f ejercicio1'
```

----------
----------

# Administración y aseguramiento de la información:
- Gestión de la información del sistema. Rendimiento. Estadísticas.
- Montaje y desmontaje de dispositivos en sistemas operativos. Automatización.
- En sistemas Windows montar un volumen en una o más carpetas.
- Herramientas de administración de discos. Particiones y volúmenes. Desfragmentación y chequeo.
- Permisos locales de acceso a ficheros y directorios.

---------------

# Repaso
## Eliminar el contenido de las carpetas impares de un servidor web
* https://github.com/jesusninoc/ClasesISO/blob/master/2019-11-05.md#eliminar-el-contenido-de-las-carpetas-impares-de-un-servidor-web-1

## Estamos viendo sistema de archivos
* https://github.com/jesnino/PowerShell/blob/master/Ficheros/EjemplosLeerOperacionesTareasBasicasII.ps1

## TAREAS BÁSICAS II (SISTEMA DE ARCHIVOS)
* https://www.jesusninoc.com/2017/07/04/4-gestion-del-sistema-de-archivos-en-powershell/

|Tarea|Cmdlet
|---|---
|Crear o modificar un archivo|New-item
|Crear un directorio|New-item
|Ver el contenido de un archivo|Get-Content
|Acceder al contenido de un directorio|Set-Location
|Listar el contenido de un directorio|Get-Childitem
|Eliminar un archivo|Remove-Item
|Eliminar un directorio|Remove-Item
|Copiar un archivo o un directorio|Copy-Item
|Mover un archivo o un directorio|Move-Item
|Renombrar un archivo o un directorio|Rename-Item
|Imprimir un archivo|Out-Printer

## Fechas
https://www.jesusninoc.com/07/02/2-programacion-en-powershell/#Fechas

## Tienen atributos que se pueden listar (show hidden files)
https://www.jesusninoc.com/2014/11/23/show-hidden-files/

## Creación de directorios
* https://github.com/jesnino/PowerShell/blob/master/Ficheros/SolucionEjerciosFicherosProcesos.ps1
* https://www.youtube.com/watch?v=CibNbK9sp8A&list=PL89qBrmFzAa34fM6OEMLV6ybDk3Y34qud&index=3

## If
* https://www.jesusninoc.com/2016/08/05/sentencia-condicional-if/
* https://www.jesusninoc.com/2015/04/26/sentencia-condicional-if-else/
* https://www.jesusninoc.com/2016/08/07/sentencia-condicional-if-elseif-else/

## BUCLE EN POWERSHELL
* https://www.jesusninoc.com/2017/10/08/listar-todos-los-caracteres-ascii-en-powershell/
* https://www.jesusninoc.com/2017/10/05/listar-todos-los-caracteres-ascii-en-bash/

---------------

# Ejercicios

## Repaso eliminar ficheros impares
```PowerShell
"hola" | out-file 2020-11-26.txt
"hola" | out-file 2020-11-25.txt

foreach($nombre in Get-ChildItem)
{
    if($nombre.Name.Split("-")[2].replace(".txt","") % 2 -eq 1)
    {
        Remove-Item $nombre -WhatIf
    }
}
```

## Repaso elminar ficheros impares (con alias de foreach: %)
```PowerShell
Get-ChildItem | %{
    if($_.Name.Split("-")[2].replace(".txt","") % 2 -eq 1)
    {
        Remove-Item $_ -WhatIf
    }
} 
```

## Crear para varios días de la semana (desde lunes pasado hasta hoy jueves) una estructura de directorios con los procesos que se están ejecutando en el sistema
```PowerShell
-3..0 | %{
    mkdir (Get-Date).AddDays($_).ToString("yyyyMMdd")
    cd (Get-Date).AddDays($_).ToString("yyyyMMdd")
    mkdir (Get-Process).Name
    cd ..
}
```

## Preguntar al usuario por la operación que quiere realizar y el nombre del fichero que sobre el que quiere realizar las siguientes operaciones que tenemos que programar son:
## - comprimir un archivo
## - descomprimir un archivo
```PowerShell
$operacion = Read-Host "Introduzca operación"
if ($operacion -eq "comprimir")
{
    $ficheroparacomprimir = Read-Host "Introduzca fichero para comprimir"
    Compress-Archive $ficheroparacomprimir -DestinationPath $ficheroparacomprimir.Replace(".txt",".zip")
}
elseif ($operacion -eq "descomprimir")
{
    $ficheroparadescomprimir = Read-Host "Introduzca fichero para descomprimir"
    Expand-Archive $ficheroparadescomprimir
}
```


## Obtener el hash de los programas que se están ejecutando en el sistema
## 1. Obtener la ruta de los programas que se están ejecutando
## 2. Hacer el hash a esos programas
```PowerShell
foreach($ruta in (Get-Process))
{
    $ruta.name
    Get-FileHash $ruta.Path
}
```

## Realizar tareas leyendo de un fichero
```PowerShell
#Añadir operaciones al fichero
"fecha" | Out-File operaciones.txt -Append
"hilos" | Out-File operaciones.txt -Append
"crear" | Out-File operaciones.txt -Append
"imprimir" | Out-File operaciones.txt -Append
"comprimir" | Out-File operaciones.txt -Append

gc .\operaciones.txt

foreach($operaciones in gc .\operaciones.txt)
{
    switch($operaciones)
    {
        "fecha"{get-date}
        "hilos"{Get-Process | select Threads}
        "crear"{New-Item -Name elque -ItemType File}
        "imprimir"{echo "hola" | Out-Printer "Microsoft Print to PDF"}
        "comprimir"{Compress-Archive -LiteralPath .\123-11.txt –CompressionLevel Optimal -DestinationPath 123.zip}
    }
}
```

---------------

# Preguntas examen liberatorio
- Opción 1
    - Realizar una función que monitoriza
    - Función que explica qué hace un proceso
    - Hacer una copia de seguridad de las dll del proceso Notepad junto con información de la integridad
- Opción 2
    - Crear una función que indique el proceso que más tiempo de CPU consume
    - Pasar un nombre de proceso a una función e indicar si se está ejecutando
    - Crear carpetas con nombres pares (5 carpetas)

--------------
--------------

# Posibles preguntas examen liberatorio
- Asignar varios permisos a una carpeta
  - https://github.com/jesusninoc/ClasesSOM/blob/master/2018-11-27.md#asignar-varios-permisos-a-una-carpeta
- Asignar varios permisos a distintas carpetas
  - https://github.com/jesusninoc/ClasesSOM/blob/master/2018-11-27.md#asignar-varios-permisos-a-distintas-carpetas
- Crear carpetas con información sobre los datos obtenidos del hardware.
  - https://www.jesusninoc.com/07/03/3-gestion-del-hardware-en-powershell/#Obtener_informacion_sobre_el_hardware_creando_un_objeto_con_todos_los_datos
```PowerShell
$objeto | Get-Member -MemberType NoteProperty | Select -ExpandProperty definition
$objeto | Get-Member -MemberType NoteProperty | Select -ExpandProperty name

$objeto | Get-Member -MemberType NoteProperty | %{
    $_ | Select -ExpandProperty name
    $_ | Select -ExpandProperty definition
}
```
- Carpeta sistemaoperativo con un fichero que tiene la partición donde se ha instalado.
- Carpeta placa_base y dentro de la carpeta un fichero con información.
- Analizar la información de un servicio con capturas de pantalla y comandos de PowerShell.
- Explica la relación entre servicios, procesos y algo más?
- Qué te permite realizar la virtualización en sistemas heterogéneos.
- Cómo harías un script para instalar un gestor virtual y una máquina virtual.
- Es lo mismo ordenar en PowerShell y luego seleccionar que seleccionar y ordenar.
- Qué cantidad de memoria tienes en GB en tu equipo?
- Analiza por qué un proceso tiene tanto consumo de CPU.
- Listar un proceso en Linux desde Powershell.
- Convierte el comando systeminfo a cmdlets en PowerShell.
- Guarda un fichero con la siguiente información: procesos que se están ejecutando, añade una línea con tu nombre al fichero y la hora en que has comprobado los procesos (hazlo tres veces y comprueba que se añaden los datos de las tres ejecuciones).
- Realizar un script que almacene información de cada proceso que se está ejecutando, piensa qué información puede ser interesante.
- Un equipo de la red comienza a realizar comportamientos extraños, realiza un script que pueda detectar qué ocurre y a qué hora.
- Mostrar los servicios que representan comportamientos extraños.
- ¿Cómo puedes darte cuenta de que hay un proceso que consume toda la memoria RAM del equipo?
- Realiza una función de login que almacene los intentos fallidos y correctos.
- Crea una función para hacer un login con dos factores de verificación (incluye comprobar el nombre de usuario).
- Crea una función que intente localizar los hilos que se crean para un proceso.
- Guardar un listado con los ficheros que tenemos en la unidad Q.
- Crear carpetas para cada extensión.
- Carpeta doc y dentro de la carpeta copiar ficheros con extensión doc.
- Añadir a cada carpeta doc, txt un listado con los ficheros y además añadir un listado de permisos.
- Como usuario pedro Crea un fichero con nombre secreto.txt al que únicamente él tenga acceso, tanto de lectura como de escritura.
- Crea otro fichero, también como usuario juan, con nombre datos.txt al que tengan acceso, tanto para leer como para escribir todos los usuarios que pertenezcan al mismo grupo.  Comprueba como usuario pablo que puedes modificar el fichero.
- Como usuario juan, crea un fichero con nombre datos.txt al que pueda acceder cualquier usuario para leer su contenido, y cualquier usuario del mismo grupo para leer o escribir.
- Como usuario pedro, copia un programa del directorio /usr/bin al directorio de trabajo con un nombre diferente. Mira los permisos de este programa. Comprueba que se puede ejecutar.
- Cambia los permisos de un fichero de tal forma que sólo lo pueda ejecutar el propietario del archivo.
- Aparecen unas capturas que se han realizado con la webcam, la webcam esta en estado activo, ¿qué ha ocurrido en el sistema?
- Un empleado sabe que va a ser despedido...
- Realizar la firma de todos los programas que se están ejecutando
- Verificación de contraseña utilizando ficheros en Linux

------------------------

# Examen liberatorio
- Las preguntas que implican respuestas por scripts se pueden hacer en PowerShell, Linux o WSL (o en todos a la vez).
- Utiliza bucles, condiciones, ficheros, funciones, etc.
 
## 1. Realizar un inventario de tu equipo a nivel hardware y software, ten en cuenta cómo clasificar la información y no olvides temas importantes como por ejemplo controladores.
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-10-12.md
* https://github.com/jesusninoc/PowerShell/blob/master/Hardware/EjemplosHardware.ps1
* https://github.com/jesusninoc/PowerShell/blob/master/Hardware/EjemplosHardware2.ps1

## 2. Crear una estructura de ficheros en Linux y establece permisos, los permisos tienen que estar establecidos con el mayor criterio posible.
* https://github.com/jesusninoc/Bash/blob/master/Permisos/EjercicioPermisos.sh

## 3. Explicar todo lo que sepas sobre virtualización y qué relación tiene con las clases de los jueves.
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-11-30.md
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-11-23.md
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-11-16.md
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-10-09.md
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-10-02.md
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-10-26.md
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-10-19.md
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-09-28.md
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-09-21.md

## 4. Analizar tu equipo desde el punto de vista de los programas, procesos, servicios e hilos.
* https://www.jesusninoc.com/2016/11/10/analizar-servicios-con-powershell/

## 5. Monitorizar un valor de tu equipo.
* https://github.com/jesusninoc/PowerShell/blob/master/Procesos/EjerciciosFuncionesProcesos2.ps1

## 6. Realizar un login implementando autenticación de segundo factor y almacena los intentos de login fallido.
* https://github.com/jesusninoc/PowerShell/blob/master/Funciones/EjerciciosFuncionesAvanzadasLoginIntentosFallidos.ps1
* https://github.com/jesusninoc/PowerShell/blob/master/Funciones/EjerciciosFuncionesAvanzadasLogin.ps1
* https://github.com/jesusninoc/PowerShell/blob/master/Funciones/EjerciciosFuncionesAvanzadasLoginHash.ps1

## 7. Crear carpetas con información sobre los datos obtenidos del hardware.
* https://www.jesusninoc.com/09/07/obtener-informacion-sobre-el-hardware-creando-un-objeto-con-todos-los-datos/
```PowerShell
$objeto | Get-Member -MemberType NoteProperty | %{
    [String]$dire=$_ | Select -ExpandProperty name
    mkdir $dire -Force
    cd $dire
    ni -name informacion.txt -Value ($_ | Select -ExpandProperty definition) -Force
    cd ..
}
```
## 8. Realizar la firma de todos los programas que se están ejecutando.
### Linux
* https://github.com/jesusninoc/ClasesISO/blob/master/2018-11-13.md#calcular-el-hash-sha512-de-todos-los-procesos-que-se-est%C3%A1n-ejecutando-en-linux
### PowerShell función simple
```PowerShell
function hashear($ruta){
    Get-FileHash $ruta
}

Get-Process | select Path | %{
    hashear $_.Path
}
```
### PowerShell función compleja
```PowerShell
function hasheo
{
  param
  (
    [Parameter(Mandatory=$true, ValueFromPipeline=$true)]
    [String[]]
    [AllowEmptyString()] 
    $ruta
  )

  begin
  {
    Write-Host "Principio de hasheo"
  }

  process
  {
    Get-FileHash $ruta
  }
  end
  {
    Write-Host "Final de hasheo"
  }
}

Get-Process | select Path | %{
    hashear $_.Path
}
```
## 9. Analiza por qué un proceso tiene tanto consumo de CPU.
### PowerShell función simple
```PowerShell
function consumo ($nombre,$valor)
{
   if((Get-Process -Name $nombre | select cpu).cpu -gt $valor)
    {
        "$nombre Consume mucho"
    }
    else
    {
        "$nombre No consume mucho"
    }
}

consumo -nombre powershell_ise -valor 10

Get-Process | %{
    consumo -nombre $_.Name -valor 10
}
```
### PowerShell función compleja
```PowerShell
function consumo
{
  param
  (
    [String[]]$nombre,$valor
  )

  begin
  {
    Write-Host "Principio de analizar"
    if($nombre -eq $null)
    {
        "Tiene que introducir un nombre de proceso, majo!"
        break
    }
  }

  process
  {
    if((Get-Process -Name $nombre | select cpu).cpu -gt $valor)
    {
        "$nombre Consume mucho"
    }
    else
    {
        "$nombre No consume mucho"
    }
  }
  end
  {
    Write-Host "Final de analizar"
  }
}

consumo -nombre notepad -valor 10
```
## 10. Simula un login, ten en cuenta que el password se encuentra en SHA512
```Powershell
[Reflection.Assembly]::LoadWithPartialName("System.Web")
[System.Web.Security.FormsAuthentication]::HashPasswordForStoringInConfigFile("Hola", "SHA512")
```
```Powershell
#################################################################################
#Función login de user y pass
#El user y pass (hash en SHA512) correcto se escriben en el bloque BEGIN

#Ejecutar la función
#PS C:\Users\usuario> loginH user pass

function loginH
{
  param
  (
    [Parameter(Mandatory=$true, ValueFromPipeline=$true)]
    [String[]]
    [AllowEmptyString()] 
    $User,$Pass
  )

  begin
  {
    Write-Host "Principio de login"
    $usercorrecto="user"
    #El pass en SHA512
    #$passcorrecto="pass"
    $passcorrecto="5B722B307FCE6C944905D132691D5E4A2214B7FE92B738920EB3FCE3A90420A19511C3010A0E7712B054DAEF5B57BAD
59ECBD93B3280F210578F547F4AED4D25"
  }

  process
  {
   #Realizar el SHA512 del pass introducido por el usuariO
   [Reflection.Assembly]::LoadWithPartialName("System.Web")
   $Pass
   if($User -eq $usercorrecto -and [System.Web.Security.FormsAuthentication]::HashPasswordForStoringInConfigFile($Pass, "SHA512") -eq $passcorrecto)
    {
        $ok=1
    }
    else
    {
        $ok=0
    }
  }
  end
  {
   if($ok)
    {
        Write-Host "User correcto"
    }
    else
    {
        Write-Host "User no correcto"
    }
    Write-Host "Fin de login"
  }
}
#################################################################################
```

------------
------------

# Examen liberatorio
- Las preguntas que implican respuestas por scripts se pueden hacer en PowerShell o Linux (o en los dos a la vez).
- Utiliza bucles, condiciones, ficheros, funciones, etc.
 
## 1. Realizar un inventario de tu equipo a nivel hardware y software, ten en cuenta cómo clasificar la información y no olvides temas importantes como por ejemplo controladores.
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-10-12.md
* https://github.com/jesusninoc/PowerShell/blob/master/Hardware/EjemplosHardware.ps1
* https://github.com/jesusninoc/PowerShell/blob/master/Hardware/EjemplosHardware2.ps1

```PowerShell
#Llamadas WMI
$ComputerSystem=Get-WmiObject Win32_ComputerSystem
$BaseBoard=Get-WmiObject Win32_BaseBoard
$BIOS=Get-WmiObject Win32_BIOS
$Processor=Get-WmiObject Win32_Processor
$Battery=Get-WmiObject Win32_Battery
$programas=Get-Package
$controladores=Get-Process -Module
 
#Crear un objeto con todos los datos sobre el hardware
$equipo=[PSCustomObject]@{
 Model = $ComputerSystem.Model
 ManufacturerBoard = $BaseBoard.Manufacturer
 BIOSVersion = $BIOS.SMbiosbiosversion
 BIOSSerialNumber = $BIOS.serialnumber
 ManufacturerProcessor=$Processor.Manufacturer
 MaxClockSpeed=$Processor.MaxClockSpeed
 Cargadelprcesador=$Processor.LoadPercentage
 programas=$programas.name
 conontroladores=$controladores
}

$equipo.programas
$equipo.conontroladores
```

## 2. Crear una estructura de ficheros en Linux y establece permisos, los permisos tienen que estar establecidos con el mayor criterio posible.
* https://github.com/jesusninoc/Bash/blob/master/Permisos/EjercicioPermisos.sh

## 3. Explicar todo lo que sepas sobre virtualización y qué relación tiene con las clases de los jueves.
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-11-30.md
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-11-23.md
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-11-16.md
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-10-09.md
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-10-02.md
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-10-26.md
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-10-19.md
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-09-28.md
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-09-21.md

## 4. Analizar tu equipo desde el punto de vista de los programas, procesos, servicios e hilos.
https://www.jesusninoc.com/2016/11/10/analizar-servicios-con-powershell/

## 5. Monitorizar un valor de tu equipo.
https://github.com/jesusninoc/PowerShell/blob/master/Procesos/EjerciciosFuncionesProcesos2.ps1
```PowerShell
Get-Process | Where-Object PM -GT 50MB
Get-Process | Where-Object CPU -GT 1 | Where-Object PM -GT 50MB
```

## 6. Realizar un login implementando autenticación de segundo factor y almacena los intentos de login fallido.
* https://github.com/jesusninoc/PowerShell/blob/master/Funciones/EjerciciosFuncionesAvanzadasLoginIntentosFallidos.ps1
* https://github.com/jesusninoc/PowerShell/blob/master/Funciones/EjerciciosFuncionesAvanzadasLogin.ps1
* https://github.com/jesusninoc/PowerShell/blob/master/Funciones/EjerciciosFuncionesAvanzadasLoginHash.ps1

```PowerShell
#Login de user y pass
#El user y pass correcto se escriben en el bloque BEGIN

function login0
{
  param([Parameter(Mandatory=$true, ValueFromPipeline=$true)][String[]][AllowEmptyString()]$User,$Pass,$Pass2)

  begin
  {
    Write-Host "Principio de login"
    $usercorrecto="jaime"
    $passcorrecto="hola"
    $passcorrecto2=gc .\pin.txt
  }

  process
  {
   if($User -eq $usercorrecto -and $Pass -eq $passcorrecto -and $Pass2 -eq $passcorrecto2)
    {
        $ok=1
    }
    else
    {
        $ok=0
    }
  }
  end
  {
   if($ok)
    {
        Write-Host "User correcto"
    }
    else
    {
        Write-Host "User no correcto"
        $User,$Pass,$Pass2 | Out-File log.txt -Append
    }
    Write-Host "Fin de login"
  }
}

#Ejecutar la función: login0
login0 -User jaime -Pass hola -Pass2 1234
```

-------------------------

# Examen liberatorio II

## Realizar una función de muestre el consumo de procesador analizando dos valores (consumo -cpu 20 -id 6000)

```PowerShell
# Sirve para ir de uno en uno
(Get-Process | select $proceso).cpu -gt $valor

# Varios valores
Get-Process | Where-Object cpu -gt 10 | Where-Object id -gt 6000
```

## Realizar una función que permita detectar que hay una firma igual a otra

```PowerShell
# Función simple para almacenar un hash de un programa
function hashear($ruta){
    Get-FileHash $ruta
}

(hashear 'C:\Windows\System32\notepad.exe' | select hash).hash | Out-File hash.txt

(gc hash.txt)

Get-Process | select Path | %{
    hashear $_.Path
}

Get-Process | select Path | %{
    (hashear $_.Path).hash
}

Get-Process | select Path | %{
    if ((hashear $_.Path).hash -match (gc hash.txt))
    {
        "Igual",$_.path
    }
}
```

## Realizar una función que muestre información sobre los hilos de un proceso

```PowerShell
function hilos($name){
    $processHandle = (Get-Process -Name $name).id
    $Threads = Get-WmiObject -Class Win32_Thread |
    Where-Object { $_.ProcessHandle -eq $processHandle }
    "The $name process has $($threads.count) threads"
    $threads | Format-Table -Property priority, thread*, User*Time, kernel*Time
}

hilos notepad
```

## Realizar una función que muestre los procesos que utilizan un controlador
```PowerShell
Get-Process -Module | select-string "OLEAUT32.dll"

Get-Process -Module | %{
    $_ | select-string "OLEAUT32.dll"
}

Get-Process | select name,Modules  | %{
    $_.name
    $_.Modules.ModuleName | select-string "OLEAUT32.dll"
}

Get-Process | select name,Modules | %{
    if($_.Modules.ModuleName -eq "OLEAUT32.dll")
    {
        $_.Name,$_.Modules.ModuleName
    }
}
```

------------------

# Examen de la primera evaluación

## Crear carpetas para cada extensión
```PowerShell
# Crear carpetas para cada extensión.

function extensiones
{
  param
  (
    [Parameter(Mandatory=$true, ValueFromPipeline=$true)]
    [String[]]
    $ruta
  )

  begin
  {
    Write-Host "Begin de extensión"
    $var = ls $ruta | select Extension
  }

  process
  {
    $var | %{
        mkdir $_.Extension
    }
  }
  end
  {
    Write-Host "Final de extensión"
    Get-ChildItem
  }
}

extensiones ../
```
## Carpeta txt y dentro de la carpeta copiar ficheros con extensión txt
```PowerShell
function txt
{
  param
  (
    [Parameter(Mandatory=$true, ValueFromPipeline=$true)]
    [String[]]
    $ruta
  )

  begin
  {
    Write-Host "Begin de txt"
    mkdir txt
  }

  process
  {
    Copy-Item -Path $ruta -Destination txt -Force -WhatIf
  }
  end
  {
    Write-Host "Final de txt"
    Get-ChildItem
  }
}

txt ../*.txt
```
## Realiza una función de login que almacene los intentos fallidos y correctos
* https://github.com/jesusninoc/PowerShell/blob/master/Funciones/EjerciciosFuncionesAvanzadasLoginIntentosFallidos.ps1
* https://github.com/jesusninoc/PowerShell/blob/master/Funciones/EjerciciosFuncionesAvanzadasLogin.ps1
* https://github.com/jesusninoc/PowerShell/blob/master/Funciones/EjerciciosFuncionesAvanzadasLoginHash.ps1

## Mostrar los servicios que representan comportamientos extraños
* https://www.jesusninoc.com/11/10/analizar-servicios-con-powershell/

------------------

# Repaso operaciones básicas en Bash y PowerShell
* https://github.com/jesnino/Bash/blob/master/Introduccion/EjemplosOperacionesCadenas.sh
* https://www.jesusninoc.com/2008/09/29/ejemplos-de-ejecucion-de-comandos-en-bash-y-powershell-parte-1/
* https://www.jesusninoc.com/2008/09/30/ejemplos-de-ejecucion-de-comandos-en-bash-y-powershell-parte-2/
* https://www.jesusninoc.com/2008/10/01/ejemplos-de-ejecucion-de-comandos-en-bash-y-powershell-parte-3/

# Hacer ejercicios en PowerShell
## Ejercicio de PowerShell | Crear diccionario
* https://www.youtube.com/watch?v=Pl2lK4urW3Q&list=PL89qBrmFzAa34fM6OEMLV6ybDk3Y34qud&index=2
* https://www.jesusninoc.com/2016/06/07/ejercicios-de-powershell-crear-un-diccionario/

# Enviar un mensaje entres varios equipos
### Cliente
```PowerShell
    ##Client
    $ip=[IPAddress]"127.0.0.1"
    $TcpClient=New-Object System.Net.Sockets.TcpClient($ip, "2050")
    $mensaje=New-Object System.IO.StreamWriter $TcpClient.GetStream()
    $mensaje.Write("echo hol2222a")
    $mensaje.Dispose()
```
### Servidor y cliente
```PowerShell
##Server
$ip=[IPAddress]"0.0.0.0"
$TcpListener=New-Object System.Net.Sockets.TcpListener (New-Object System.Net.IPEndPoint($ip,"2050"))
$TcpListener.Start()
 
while($true)
{
$mensaje2=(New-Object System.IO.StreamReader ($TcpListener.AcceptTcpClient().GetStream())).ReadLine()
$mensaje2
##Client
$ip=[IPAddress]"192.168.204.222"
$TcpClient=New-Object System.Net.Sockets.TcpClient($ip, "2051")
$mensaje=New-Object System.IO.StreamWriter $TcpClient.GetStream()
$mensaje.Write($mensaje2)
$mensaje.Dispose()
}
$TcpListener.Stop()
```
### Servidor que ejecuta
```PowerShell
##Server
$ip=[IPAddress]"0.0.0.0"
$TcpListener=New-Object System.Net.Sockets.TcpListener (New-Object System.Net.IPEndPoint($ip,"2051"))
$TcpListener.Start()
 
while($true)
{
$mensaje=(New-Object System.IO.StreamReader ($TcpListener.AcceptTcpClient().GetStream())).ReadLine()
$mensaje | iex
}
$TcpListener.Stop()
```

-------------
-------------

# Examen liberatorio
- Las preguntas que implican respuestas por scripts se pueden hacer en PowerShell o Linux (o en los dos a la vez).
- Utiliza bucles, condiciones, ficheros, funciones, etc.

# Preguntas
- Realizar una función que compare dos hashes.
- Obtener información sobre el sistema operativo.
- Guardar en un fichero comprimido información sobre el sistema operativo.
- Hacer un login registrando los intentos (el password se tiene que almacenar hash).
- ¿Cómo puedes detectar un problema en un sistema operativo relacionado con la ejecución de procesos?
- Mover carpetas impares
- Crear una copia de sguriad y restaurar

# Resolver las siguientes situaciones
- Hacer un login (registrar los intentos fallidos) que permita acceder a un sistema que realiza una copia de seguridad comprimida sobre información sobre el sistema (realizar la copia de seguridad).
```PowerShell
function copia()
{
    Get-Process > procesos.txt
    Compress-Archive -Path .\procesos.txt -DestinationPath aqui.zip -CompressionLevel Fastest -Force
}

$usuario = "juan"
$pass = "juan"
if($usuario -eq "juan" -and $pass -eq "juan")
{
    copia
}
```
- El sistema de copia de seguridad está fallando. ¿Cómo puedes detectar un problema en un sistema operativo relacionado con la ejecución de procesos? Haz una simulación de la situación.
```PowerShell
# Función que simula copia de seguridad
function copiaseguridad()
{
    Start-Sleep -Seconds 8
    Compress-Archive -Path .\123-10.txt -DestinationPath aqui.zip -CompressionLevel Fastest -Force
}

copiaseguridad

# En PowerShell distintos
(Get-Process -Name powershell_ise).Threads.Count
```

# Soluciones avanzadas

## Obtener información sobre el sistema operativo y enviarla por la red

### Servidor
```PowerShell
##Server
$port=2020
$endpoint = new-object System.Net.IPEndPoint ([IPAddress]::Any,$port)
$udpclient=new-Object System.Net.Sockets.UdpClient $port
$content=$udpclient.Receive([ref]$endpoint)
[Text.Encoding]::ASCII.GetString($content)
$udpclient.Dispose()
```

### Cliente
```PowerShell
Function menu {
    Clear-Host
    Write-Host "Iniciando el menu..."
    Write-Host "1. Información del hardware"
    Write-Host "2. Información de procesos"
    Write-Host "3. Ver registros de usuarios"
    Write-Host "4. Ver actualizaciones del sistema"
    Write-Host "5. Ver programas instalados"
    Write-Host "6. Salir"
}
1
menu

while($inp = Read-Host -Prompt "Escoge una accion"){

switch($inp){
        1{  $extension = Read-Host -Prompt "Pulse para analizar los componentes hardware de su pc"
            $port=2020
            $endpoint = new-object System.Net.IPEndPoint ([IPAddress]::Loopback,$port)
            $udpclient=new-Object System.Net.Sockets.UdpClient
            $b=[Text.Encoding]::ASCII.GetBytes('$ComputerSystem=Get-WmiObject Win32_ComputerSystem
            $BaseBoard=Get-WmiObject Win32_BaseBoard
            $BIOS=Get-WmiObject Win32_BIOS
            $Processor=Get-WmiObject Win32_Processor
            $Battery=Get-WmiObject Win32_Battery
            [PSCustomObject]@{
             Nombre=$Processor.Name
             Model = $ComputerSystem.Model
             Name = $ComputerSystem.Name
             PrimaryOwnerName = $ComputerSystem.primaryownername
             Totalphysicalmemory = $ComputerSystem.totalphysicalmemory
             ManufacturerBoard = $BaseBoard.Manufacturer
             BIOSVersion = $BIOS.SMbiosbiosversion
             BIOSSerialNumber = $BIOS.serialnumber
             version = $BaseBoard.version
             ManufacturerProcessor=$Processor.Manufacturer
             MaxClockSpeed=$Processor.MaxClockSpeed
             DeviceIDBattery=$Battery.DeviceID.trim()
             Designvoltaje=$Battery.designvoltage
             NumberOfCores=$Processor.NumberOfCores
             NumberOfLogicalProcessors=$Processor.NumberOfLogicalProcessors
             NumberOfEnabledCore=$Processor.NumberOfEnabledCore
             LoadPercentage=$Processor.LoadPercentage
             }')
             $bytesSent=$udpclient.Send($b,$b.length,$endpoint)
             $udpclient.Close()
            pause;
            break
        }
        2       
        {  
            $extension = Read-Host -Prompt "Pulse para analizar los procesos de su pc"
            $port=2020
            $endpoint = new-object System.Net.IPEndPoint ([IPAddress]::Loopback,$port)
            $udpclient=new-Object System.Net.Sockets.UdpClient
            $b=[Text.Encoding]::ASCII.GetBytes('Get-WmiObject -Class win32_process | select name, Path, ExecutablePath, CommandLine | Format-Custom')
            $bytesSent=$udpclient.Send($b,$b.length,$endpoint)
            $udpclient.Close()
            pause; 
            break
        }
        3
        {
         
            $extension = Read-Host -Prompt "Introduzca el nombre del equipo"
            $port=2020
            $endpoint = new-object System.Net.IPEndPoint ([IPAddress]::Loopback,$port)
            $udpclient=new-Object System.Net.Sockets.UdpClient
            $b=[Text.Encoding]::ASCII.GetBytes('$logs = get-eventlog system -ComputerName "$extension" -source Microsoft-Windows-Winlogon -After (Get-Date).AddDays(-7);
            $res = @(); ForEach ($log in $logs) {if($log.instanceid -eq 7001) {$type = "Logon"} Elseif ($log.instanceid -eq 7002){$type="Logoff"} Else {Continue} $res += New-Object PSObject -Property @{Time = $log.TimeWritten; "Event" = $type; User = (New-Object System.Security.Principal.SecurityIdentifier $Log.ReplacementStrings[1]).Translate([System.Security.Principal.NTAccount])}};
            $res')
            $bytesSent=$udpclient.Send($b,$b.length,$endpoint)
            $udpclient.Close()
        }

      
      
        6 {"Exit"; break}
        default {Write-Host -ForegroundColor red -BackgroundColor white "Invalid option. Please select another option";pause}
        
    }

menu
}
```

----------------
----------------

# Examen subida de nota

## Utilizando funciones, resolver la siguiente situación: 

- Solo puedes utilizar el script si eres un usuario reconocido
- Si eres un usuario reconocido preguntas al usuario qué quiere, opciones:
    - Opción 1. Te indica si se ha ejecutado una compresión en el sistema
    - Opción 2. Te cuenta el número de hilos que se están ejecutando en el sistema

## Solución
```PowerShell
$usuario = "juan"
$password = "1234"

if($usuario -eq (Read-Host "Intro usuario") -and $password -eq (Read-Host "Intro pass"))
{
    "Estás dentro"
    switch(Read-Host "- Opción 1. Te indica si se ha ejecutado una compresión en el sistema
- Opción 2. Te cuenta el número de hilos que se están ejecutando en el sistema" 
)
    {
        "1"{
            #while(1)
            #{
                Get-Process -Name WinRAR -OutVariable resultado
                if($resultado)
                {
                    "Se ha comprimido"
                }
                else
                {
                    "No se ha comprimido"
                }
            #}
        }
        "2"{(Get-Process).Threads.Count}
    }
}
```

------------------

# Cliente-Servidor

## Comunicación entre cliente y servidor
* https://www.jesusninoc.com/2015/02/25/creating-reverse-shell/
* https://www.jesusninoc.com/2015/02/26/creating-shell/
* https://www.jesusninoc.com/2017/10/18/crear-una-comunicacion-entre-un-cliente-en-bash-de-linux-y-un-servidor-en-powershell-de-windows-utilizando-tcpip/
* https://www.jesusninoc.com/2017/10/26/crear-una-comunicacion-entre-un-cliente-en-powershell-de-windows-y-un-servidor-en-bash-de-linux-utilizando-tcpip/
* https://www.jesusninoc.com/2016/04/30/simular-el-funcionamiento-de-un-servidor-web-utilizando-netcat-en-linux/
* https://www.jesusninoc.com/2009/06/06/ejecutar-nc-exe-cmd-exe-remotamente/

## Servidor desde PowerShell
https://www.jesusninoc.com/2017/05/06/crear-un-servidor-web-al-que-se-pueda-acceder-desde-cualquier-parte-de-la-red-privada-con-powershell/

## Utilizando la comunicación remota entre cliente y servidor simular una conexión a una shell y una reverse shell
* https://www.jesusninoc.com/01/27/ejecutar-un-cmdlet-remotamente-en-un-equipo-utilizando-sockets-udp/
* https://www.jesusninoc.com/2015/02/25/creating-reverse-shell/
* https://www.jesusninoc.com/2015/02/26/creating-shell/

## Ejecutar un comando remotamente utilizando un servidor web creado en PowerShell
* https://www.jesusninoc.com/12/18/ejercicios-de-powershell-ejecutar-un-comando-remotamente-utilizando-un-servidor-web-creado-en-powershell/

# Posibles preguntas
- Leer de fichero
- Foreach, if y funciones
- Procesos (hilos, procesos y servicios) y sistema de archivos (copias de seguridad y hash)

-------------------

# Examen liberatorio
- Las preguntas que implican respuestas por scripts se pueden hacer en PowerShell, Linux o WSL (o en todos a la vez).
- Utiliza bucles, condiciones, ficheros, funciones, etc.
 
## 1. Realizar un inventario de tu equipo a nivel hardware y software, ten en cuenta cómo clasificar la información y no olvides temas importantes como por ejemplo controladores.
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-10-12.md
* https://github.com/jesusninoc/PowerShell/blob/master/Hardware/EjemplosHardware.ps1
* https://github.com/jesusninoc/PowerShell/blob/master/Hardware/EjemplosHardware2.ps1

## 2. Crear una estructura de ficheros en Linux y establece permisos, los permisos tienen que estar establecidos con el mayor criterio posible.
* https://github.com/jesusninoc/Bash/blob/master/Permisos/EjercicioPermisos.sh

## 3. Explicar todo lo que sepas sobre virtualización y qué relación tiene con las clases de los jueves.
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-11-30.md
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-11-23.md
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-11-16.md
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-10-09.md
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-10-02.md
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-10-26.md
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-10-19.md
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-09-28.md
* https://github.com/jesusninoc/ClasesISO/blob/master/2017-09-21.md

## 4. Analizar tu equipo desde el punto de vista de los programas, procesos, servicios e hilos.
* https://www.jesusninoc.com/2016/11/10/analizar-servicios-con-powershell/

## 5. Monitorizar un valor de tu equipo.
* https://github.com/jesusninoc/PowerShell/blob/master/Procesos/EjerciciosFuncionesProcesos2.ps1

## 6. Realizar un login implementando autenticación de segundo factor y almacena los intentos de login fallido.
* https://github.com/jesusninoc/PowerShell/blob/master/Funciones/EjerciciosFuncionesAvanzadasLoginIntentosFallidos.ps1
* https://github.com/jesusninoc/PowerShell/blob/master/Funciones/EjerciciosFuncionesAvanzadasLogin.ps1
* https://github.com/jesusninoc/PowerShell/blob/master/Funciones/EjerciciosFuncionesAvanzadasLoginHash.ps1

## 7. Crear carpetas con información sobre los datos obtenidos del hardware.
* https://www.jesusninoc.com/09/07/obtener-informacion-sobre-el-hardware-creando-un-objeto-con-todos-los-datos/
```PowerShell
$objeto | Get-Member -MemberType NoteProperty | %{
    [String]$dire=$_ | Select -ExpandProperty name
    mkdir $dire -Force
    cd $dire
    ni -name informacion.txt -Value ($_ | Select -ExpandProperty definition) -Force
    cd ..
}
```
## 8. Realizar la firma de todos los programas que se están ejecutando.
### Linux
* https://github.com/jesusninoc/ClasesISO/blob/master/2018-11-13.md#calcular-el-hash-sha512-de-todos-los-procesos-que-se-est%C3%A1n-ejecutando-en-linux
### PowerShell función simple
```PowerShell
function hashear($ruta){
    Get-FileHash $ruta
}

Get-Process | select Path | %{
    hashear $_.Path
}
```
### PowerShell función compleja
```PowerShell
function hasheo
{
  param
  (
    [Parameter(Mandatory=$true, ValueFromPipeline=$true)]
    [String[]]
    [AllowEmptyString()] 
    $ruta
  )

  begin
  {
    Write-Host "Principio de hasheo"
  }

  process
  {
    Get-FileHash $ruta
  }
  end
  {
    Write-Host "Final de hasheo"
  }
}

Get-Process | select Path | %{
    hashear $_.Path
}
```
## 9. Analiza por qué un proceso tiene tanto consumo de CPU.
### PowerShell función simple
```PowerShell
function consumo ($nombre,$valor)
{
   if((Get-Process -Name $nombre | select cpu).cpu -gt $valor)
    {
        "$nombre Consume mucho"
    }
    else
    {
        "$nombre No consume mucho"
    }
}

consumo -nombre powershell_ise -valor 10

Get-Process | %{
    consumo -nombre $_.Name -valor 10
}
```
### PowerShell función compleja
```PowerShell
function consumo
{
  param
  (
    [String[]]$nombre,$valor
  )

  begin
  {
    Write-Host "Principio de analizar"
    if($nombre -eq $null)
    {
        "Tiene que introducir un nombre de proceso, majo!"
        break
    }
  }

  process
  {
    if((Get-Process -Name $nombre | select cpu).cpu -gt $valor)
    {
        "$nombre Consume mucho"
    }
    else
    {
        "$nombre No consume mucho"
    }
  }
  end
  {
    Write-Host "Final de analizar"
  }
}

consumo -nombre notepad -valor 10
```
## 10. Simula un login, ten en cuenta que el password se encuentra en SHA512
```Powershell
[Reflection.Assembly]::LoadWithPartialName("System.Web")
[System.Web.Security.FormsAuthentication]::HashPasswordForStoringInConfigFile("Hola", "SHA512")
```
```Powershell
#################################################################################
#Función login de user y pass
#El user y pass (hash en SHA512) correcto se escriben en el bloque BEGIN

#Ejecutar la función
#PS C:\Users\usuario> loginH user pass

function loginH
{
  param
  (
    [Parameter(Mandatory=$true, ValueFromPipeline=$true)]
    [String[]]
    [AllowEmptyString()] 
    $User,$Pass
  )

  begin
  {
    Write-Host "Principio de login"
    $usercorrecto="user"
    #El pass en SHA512
    #$passcorrecto="pass"
    $passcorrecto="5B722B307FCE6C944905D132691D5E4A2214B7FE92B738920EB3FCE3A90420A19511C3010A0E7712B054DAEF5B57BAD
59ECBD93B3280F210578F547F4AED4D25"
  }

  process
  {
   #Realizar el SHA512 del pass introducido por el usuariO
   [Reflection.Assembly]::LoadWithPartialName("System.Web")
   $Pass
   if($User -eq $usercorrecto -and [System.Web.Security.FormsAuthentication]::HashPasswordForStoringInConfigFile($Pass, "SHA512") -eq $passcorrecto)
    {
        $ok=1
    }
    else
    {
        $ok=0
    }
  }
  end
  {
   if($ok)
    {
        Write-Host "User correcto"
    }
    else
    {
        Write-Host "User no correcto"
    }
    Write-Host "Fin de login"
  }
}
#################################################################################
```
