// Copyright 2021 The IREE Authors
//
// Licensed under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception

#ifndef IREE_HAL_LOCAL_LOADERS_VMVX_MODULE_LOADER_H_
#define IREE_HAL_LOCAL_LOADERS_VMVX_MODULE_LOADER_H_

#include <stdbool.h>
#include <stdint.h>

#include "iree/base/api.h"
#include "iree/hal/local/executable_loader.h"
#include "iree/vm/api.h"

#ifdef __cplusplus
extern "C" {
#endif  // __cplusplus

// Creates an executable loader that can load compiled IREE VM bytecode modules
// using the VMVX module. |instance| will be used for all loaded contexts.
//
// Users may optionally provide additional VM modules that will be linked into
// the context after the VMVX module such that it can be referenced and before
// the generated bytecode modules such that they can be used to resolve imports.
// Modules are retained for the lifetime of the loader and may be instantiated
// in many contexts. Modules must be thread-safe and ideally have no thread
// synchronization requirements.
//
// Compiled VMVX programs that use the user-provided modules are encouraged to
// use optional imports so that they can still function when the specialized
// user modules are not available. It's also encouraged to create abstract
// modules to avoid combinatorial explosions in required modules: prefer to have
// modules focused around operations instead of implementations (no `avx512` and
// `arm-sve` modules, etc).
iree_status_t iree_hal_vmvx_module_loader_create(
    iree_vm_instance_t* instance, iree_host_size_t user_module_count,
    iree_vm_module_t** user_modules, iree_allocator_t host_allocator,
    iree_hal_executable_loader_t** out_executable_loader);

// Creates an executable loader that can load compiled IREE VM bytecode modules
// using the VMVX module. Uses an isolated VM instance.
iree_status_t iree_hal_vmvx_module_loader_create_isolated(
    iree_host_size_t user_module_count, iree_vm_module_t** user_modules,
    iree_allocator_t host_allocator,
    iree_hal_executable_loader_t** out_executable_loader);

#ifdef __cplusplus
}  // extern "C"
#endif  // __cplusplus

#endif  // IREE_HAL_LOCAL_LOADERS_VMVX_MODULE_LOADER_H_
