#!/bin/bash

# Copyright 2020 The IREE Authors
#
# Licensed under the Apache License v2.0 with LLVM Exceptions.
# See https://llvm.org/LICENSE.txt for license information.
# SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception

# Build and test, using CMake/CTest, with AddressSanitizer instrumentation.
#
# See https://clang.llvm.org/docs/AddressSanitizer.html. Components that don't
# work with ASan (e.g. Python bindings) are disabled.
#
# The desired build directory can be passed as the first argument. Otherwise, it
# uses the environment variable IREE_ASAN_BUILD_DIR, defaulting to "build-asan".
# Designed for CI, but can be run manually. It reuses the build directory if it
# already exists. Expects to be run from the root of the IREE repository.

set -xeuo pipefail

BUILD_DIR="${1:-${IREE_ASAN_BUILD_DIR:-build-asan}}"
IREE_ENABLE_ASSERTIONS="${IREE_ENABLE_ASSERTIONS:-ON}"
# Enable CUDA and HIP/ROCM compiler and runtime by default if not on Darwin.
OFF_IF_DARWIN="$(uname | awk '{print ($1 == "Darwin") ? "OFF" : "ON"}')"
IREE_HAL_DRIVER_CUDA="${IREE_HAL_DRIVER_CUDA:-${OFF_IF_DARWIN}}"
IREE_HAL_DRIVER_HIP="${IREE_HAL_DRIVER_HIP:-${OFF_IF_DARWIN}}"
IREE_TARGET_BACKEND_CUDA="${IREE_TARGET_BACKEND_CUDA:-${OFF_IF_DARWIN}}"
IREE_TARGET_BACKEND_ROCM="${IREE_TARGET_BACKEND_ROCM:-${OFF_IF_DARWIN}}"

source build_tools/cmake/setup_build.sh

CMAKE_ARGS=(
  "-G" "Ninja"
  "-DCMAKE_BUILD_TYPE=RelWithDebInfo"
  "-DIREE_ENABLE_ASAN=ON"
  "-B" "${BUILD_DIR?}"
  "-DPython3_EXECUTABLE=${IREE_PYTHON3_EXECUTABLE}"
  "-DPYTHON_EXECUTABLE=${IREE_PYTHON3_EXECUTABLE}"
  "-DIREE_ENABLE_ASSERTIONS=${IREE_ENABLE_ASSERTIONS}"
  "-DCMAKE_CXX_COMPILER_LAUNCHER=sccache"
  "-DCMAKE_C_COMPILER_LAUNCHER=sccache"
  "-DIREE_HAL_DRIVER_CUDA=${IREE_HAL_DRIVER_CUDA}"
  "-DIREE_HAL_DRIVER_HIP=${IREE_HAL_DRIVER_HIP}"
  "-DIREE_TARGET_BACKEND_CUDA=${IREE_TARGET_BACKEND_CUDA}"
  "-DIREE_TARGET_BACKEND_ROCM=${IREE_TARGET_BACKEND_ROCM}"
)

echo "::group::Configuring CMake"
"${CMAKE_BIN?}" "${CMAKE_ARGS[@]?}"
echo "::endgroup::"

echo "::group::Building all"
"${CMAKE_BIN?}" --build "${BUILD_DIR?}" -- -k 0
echo "::endgroup::"

echo "::group::Building test deps"
"${CMAKE_BIN?}" --build "${BUILD_DIR?}" --target iree-test-deps -- -k 0
echo "::endgroup::"

# Respect the user setting, but default to as many jobs as we have cores.
export CTEST_PARALLEL_LEVEL=${CTEST_PARALLEL_LEVEL:-$(nproc)}

# Respect the user setting, but default to turning off Vulkan.
export IREE_VULKAN_DISABLE=${IREE_VULKAN_DISABLE:-1}
# Respect the user setting, but default to turning off Metal.
export IREE_METAL_DISABLE="${IREE_METAL_DISABLE:-1}"
# Respect the user setting, but default to turning off CUDA.
export IREE_CUDA_DISABLE=${IREE_CUDA_DISABLE:-1}
# Respect the user setting, but default to turning off HIP.
export IREE_HIP_DISABLE=${IREE_HIP_DISABLE:-1}
# The VK_KHR_shader_float16_int8 extension is optional prior to Vulkan 1.2.
export IREE_VULKAN_F16_DISABLE=${IREE_VULKAN_F16_DISABLE:-1}
# Respect the user setting, but default to skipping tests that require Nvidia GPU.
export IREE_NVIDIA_GPU_TESTS_DISABLE=${IREE_NVIDIA_GPU_TESTS_DISABLE:-1}

# Tests to exclude by label. In addition to any custom labels (which are carried
# over from Bazel tags), every test should be labeled with its directory.
declare -a label_exclude_args=(
  # Exclude specific labels.
  # Put the whole label with anchors for exact matches.
  ^noasan$
  ^nodocker$

  # Exclude all tests in a directory.
  # Put the whole directory with anchors for exact matches.
  # For example:
  #   ^bindings/python/iree/runtime$

  # Exclude all tests in some subdirectories.
  # Put the whole parent directory with only a starting anchor.
  # Use a trailing slash to avoid prefix collisions.
  # For example:
  #   ^bindings/
)

if (( IREE_VULKAN_DISABLE == 1 )); then
  label_exclude_args+=("^driver=vulkan$")
fi
if (( IREE_CUDA_DISABLE == 1 )); then
  label_exclude_args+=("^driver=cuda$")
fi
if (( IREE_HIP_DISABLE == 1 )); then
  label_exclude_args+=("^driver=hip$")
fi
if (( IREE_VULKAN_F16_DISABLE == 1 )); then
  label_exclude_args+=("^vulkan_uses_vk_khr_shader_float16_int8$")
fi
if (( IREE_NVIDIA_GPU_TESTS_DISABLE == 1 )); then
  label_exclude_args+=("^requires-gpu")
fi

label_exclude_regex="($(IFS="|" ; echo "${label_exclude_args[*]?}"))"

pushd ${BUILD_DIR?}

echo "::group::Running main project ctests"
ctest \
  --timeout 900 \
  --output-on-failure \
  --no-tests=error \
  --label-exclude "${label_exclude_regex}"
echo "::endgroup::"

echo "::group::Running llvm-external-projects tests"
cmake --build . --target check-iree-dialects -- -k 0
echo "::endgroup::"

popd
